/*
 * Copyright (C) 2004 Zaheer Abbas Merali <zaheerabbas at merali.org>
 *                    Mahmood Rehemtulla  <mhr at blueyonder.co.uk>
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <gst/gst.h>
#include <glib.h>

#include "relay.h"
#include "gststuff.h"
#include "gui.h"

static void acast_relay_stopped(GstThread* thread, struct stream_details* sd)
{
	gchar* thread_name;
	struct acast_relay* ar;
	
	thread_name=g_strdup(gst_element_get_name(GST_ELEMENT(thread)));

	ar=(struct acast_relay*)g_hash_table_lookup(sd->relays,thread_name+13);

	if (ar==NULL) {
		g_print("Could not find relay thread %s that stopped\n",thread_name+13);
	}
	else {
		g_print("relay thread %s stopped\n",thread_name+13);
		if (ar->status==0) {
			g_print("It has already been noticed\n");
		}
		else {
			ar->status=0;
			if (sd->gui) update_stream_checkboxes(sd->gsgui);
		}
	}

	g_free(thread_name);
}

int acast_add_relay(struct stream_details* sd, gchar* name, gchar* ip, int port, gchar* mount, gchar* password, 
				gchar* servertype, gchar* webcasttype, int bitrate, int channels)
{
	struct acast_relay* ar;
	
	gchar* tmp;
	gchar* sinkplugin;
	
	/* for now limit it to same server type and same webcast type */
	if (g_strcasecmp(sd->webcastserver, servertype)==0 &&
		g_strcasecmp(sd->webcasttype, webcasttype)==0 &&
		sd->bitrate[1]==bitrate) {
	
		ar=g_new0(struct acast_relay,1);
		tmp=g_strdup_printf("relay_thread_%s",name);
		ar->thread=gst_thread_new(tmp);
		g_free(tmp);
		tmp=g_strdup_printf("relay_src_%s",name);
		if (!(ar->src=element_create(tmp,"gnomevfssrc"))) return;
		g_free(tmp);
		tmp=g_strdup_printf("http://%s:%d/%s",sd->iceip,sd->iceport,sd->icemount);
		g_object_set (G_OBJECT(ar->src), "location", tmp, NULL);
		tmp=g_strdup_printf("relay_sink_%s%d",name);
		
#ifdef HAVE_SHOUT2
		if (g_strcasecmp(webcasttype,"ogg")==0 || g_strcasecmp(servertype,"icecast2")==0) {
			if (!(ar->sink = element_create (tmp, "shout2send"))) return 1;
			g_object_set(G_OBJECT(ar->sink), "protocol",3,NULL);
		}
#endif		
		if (g_strcasecmp(webcasttype,"mp3")==0) {
#ifdef HAVE_SHOUT2
			if (g_strcasecmp(servertype,"icecast2")==0) {
				if (!(ar->sink = element_create (tmp, "shout2send"))) return 1;
				g_object_set(G_OBJECT(ar->sink), "protocol",3,NULL);
			}
			else if (g_strcasecmp(servertype,"icecast1")==0) {
				if (!(ar->sink = element_create (tmp, "shout2send"))) return 1;
				g_object_set(G_OBJECT(ar->sink), "protocol",1,NULL);
			}
			else if (g_strcasecmp(servertype,"shoutcast")==0) {
				if (!(ar->sink = element_create (tmp, "shout2send"))) return 1;
				g_object_set(G_OBJECT(ar->sink), "protocol",2,NULL);
			}
#else
			if (g_strcasecmp(servertype,"icecast1")==0) {
				if (!(ar->sink = element_create (tmp, "icecastsend"))) return 1;
				g_object_set(G_OBJECT(ar->sink), "icy",FALSE,NULL);
			}
			else if (g_strcasecmp(servertype,"shoutcast")==0) {
				if (!(ar->sink = element_create (tmp, "icecastsend"))) return 1;
				g_object_set(G_OBJECT(ar->sink), "icy",TRUE,NULL);
			}
#endif
		}
		
		g_object_set (G_OBJECT(ar->sink), "ip", ip, NULL);
		g_object_set (G_OBJECT(ar->sink), "port", port, NULL);
		g_object_set (G_OBJECT(ar->sink), "mount", mount, NULL);
		g_object_set (G_OBJECT(ar->sink), "password", password, NULL);
			
		/* now connect the src and sink */
		/* once we allow decoding and encoding maybe we can use autoplugger..rather than doing an if else if job */
		gst_bin_add(GST_BIN(ar->thread),ar->src);
		gst_bin_add(GST_BIN(ar->thread),ar->sink);
		g_signal_connect(G_OBJECT(ar->thread),"shutdown",G_CALLBACK(acast_relay_stopped),sd);	
		gst_pad_connect(gst_element_get_pad(ar->src,"src"),
						gst_element_get_pad(ar->sink,"sink"));
		g_hash_table_insert(sd->relays,name,ar);
		ar->status=0;		
		return 1;
	}
	return 0;
}	
	
int acast_start_relay(struct stream_details* sd, const gchar* name)
{
	struct acast_relay* ar;
	
	ar=(struct acast_relay*)g_hash_table_lookup(sd->relays,name);
	
	if (ar!=NULL) {
		if (ar->status==0) {
			gst_element_set_state(GST_ELEMENT(ar->thread),GST_STATE_PLAYING);
			ar->status=1;
			return 1;
		}
	}
	return 0;
}

int acast_stop_relay(struct stream_details* sd, const gchar* name)
{
	struct acast_relay* ar;
	
	ar=(struct acast_relay*)g_hash_table_lookup(sd->relays,name);
	
	if (ar!=NULL) {
		if (ar->status==1) {
			gst_element_set_state(GST_ELEMENT(ar->thread),GST_STATE_NULL);
			ar->status=0;
			return 1;
		}
	}
	return 0;
}

int acast_delete_relay(struct stream_details* sd, const gchar* name)
{
	struct acast_relay* ar;
	
	ar=(struct acast_relay*)g_hash_table_lookup(sd->relays,name);
	
	if (ar!=NULL) {
		if (ar->status==0) {
			gst_element_set_state(GST_ELEMENT(ar->thread),GST_STATE_NULL);
			gst_pad_disconnect(gst_element_get_pad(ar->src,"src"),
								gst_element_get_pad(ar->sink,"sink"));
			gst_bin_remove(GST_BIN(ar->thread),ar->src);
			gst_bin_remove(GST_BIN(ar->thread),ar->sink);
			
			g_hash_table_remove(sd->relays,name);
			g_free(ar);
			return 1;
		}
	}
	
	return 0;
}

gboolean acast_relay_get_active(struct stream_details* sd, const gchar* name)
{
	struct acast_relay* ar;
	
	ar=(struct acast_relay*)g_hash_table_lookup(sd->relays,name);

	if (ar!=NULL) {
		return (ar->status==1);
	}
	return FALSE;
}
