/*
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * VIA, S3 GRAPHICS, AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 
#ifndef __HW_H__
#define __HW_H__

#include <linux/ioport.h>
#include <asm/io.h>

#include "share.h"


//***************************************************//
//* Definition IGA1 Design Method of CRTC Registers *//
//***************************************************//
#define IGA1_HOR_TOTAL_FORMULA(x)           ((x)/8)-5
#define IGA1_HOR_ADDR_FORMULA(x)            ((x)/8)-1
#define IGA1_HOR_BLANK_START_FORMULA(x)     ((x)/8)-1
#define IGA1_HOR_BLANK_END_FORMULA(x,y)     ((x+y)/8)-1
#define IGA1_HOR_SYNC_START_FORMULA(x)      ((x)/8)-1
#define IGA1_HOR_SYNC_END_FORMULA(x,y)      ((x+y)/8)-1

#define IGA1_VER_TOTAL_FORMULA(x)           (x)-2
#define IGA1_VER_ADDR_FORMULA(x)            (x)-1
#define IGA1_VER_BLANK_START_FORMULA(x)     (x)-1
#define IGA1_VER_BLANK_END_FORMULA(x,y)     (x+y)-1
#define IGA1_VER_SYNC_START_FORMULA(x)      (x)-1
#define IGA1_VER_SYNC_END_FORMULA(x,y)      (x+y)-1

//***************************************************//
//* Definition IGA2 Design Method of CRTC Registers *//
//***************************************************//
#define IGA2_HOR_TOTAL_FORMULA(x)           (x)-1
#define IGA2_HOR_ADDR_FORMULA(x)            (x)-1
#define IGA2_HOR_BLANK_START_FORMULA(x)     (x)-1
#define IGA2_HOR_BLANK_END_FORMULA(x,y)     (x+y)-1
#define IGA2_HOR_SYNC_START_FORMULA(x)      (x)-1
#define IGA2_HOR_SYNC_END_FORMULA(x,y)      (x+y)-1

#define IGA2_VER_TOTAL_FORMULA(x)           (x)-1
#define IGA2_VER_ADDR_FORMULA(x)            (x)-1
#define IGA2_VER_BLANK_START_FORMULA(x)     (x)-1
#define IGA2_VER_BLANK_END_FORMULA(x,y)     (x+y)-1
#define IGA2_VER_SYNC_START_FORMULA(x)      (x)-1
#define IGA2_VER_SYNC_END_FORMULA(x,y)      (x+y)-1

/* Define Register Number for IGA1 CRTC Timing */
#define IGA1_HOR_TOTAL_REG_NUM          2           // location: {CR00,0,7},{CR36,3,3} 
#define IGA1_HOR_ADDR_REG_NUM           1           // location: {CR01,0,7}
#define IGA1_HOR_BLANK_START_REG_NUM    1           // location: {CR02,0,7}
#define IGA1_HOR_BLANK_END_REG_NUM      3           // location: {CR03,0,4},{CR05,7,7},{CR33,5,5}
#define IGA1_HOR_SYNC_START_REG_NUM     2           // location: {CR04,0,7},{CR33,4,4}
#define IGA1_HOR_SYNC_END_REG_NUM       1           // location: {CR05,0,4}
#define IGA1_VER_TOTAL_REG_NUM          4           // location: {CR06,0,7},{CR07,0,0},{CR07,5,5},{CR35,0,0}
#define IGA1_VER_ADDR_REG_NUM           4           // location: {CR12,0,7},{CR07,1,1},{CR07,6,6},{CR35,2,2}
#define IGA1_VER_BLANK_START_REG_NUM    4           // location: {CR15,0,7},{CR07,3,3},{CR09,5,5},{CR35,3,3}
#define IGA1_VER_BLANK_END_REG_NUM      1           // location: {CR16,0,7}
#define IGA1_VER_SYNC_START_REG_NUM     4           // location: {CR10,0,7},{CR07,2,2},{CR07,7,7},{CR35,1,1}
#define IGA1_VER_SYNC_END_REG_NUM       1           // location: {CR11,0,3}

/* Define Register Number for IGA1 Shadow CRTC Timing */
#define IGA1_SHADOW_HOR_TOTAL_REG_NUM       2       // location: {CR6D,0,7},{CR71,3,3}
#define IGA1_SHADOW_HOR_BLANK_END_REG_NUM   1       // location: {CR6E,0,7}
#define IGA1_SHADOW_VER_TOTAL_REG_NUM       2       // location: {CR6F,0,7},{CR71,0,2}
#define IGA1_SHADOW_VER_ADDR_REG_NUM        2       // location: {CR70,0,7},{CR71,4,6}
#define IGA1_SHADOW_VER_BLANK_START_REG_NUM 2       // location: {CR72,0,7},{CR74,4,6}
#define IGA1_SHADOW_VER_BLANK_END_REG_NUM   2       // location: {CR73,0,7},{CR74,0,2}
#define IGA1_SHADOW_VER_SYNC_START_REG_NUM  2       // location: {CR75,0,7},{CR76,4,6}
#define IGA1_SHADOW_VER_SYNC_END_REG_NUM    1       // location: {CR76,0,3}



/* Define Register Number for IGA2 CRTC Timing */
#define IGA2_HOR_TOTAL_REG_NUM          2           // location: {CR50,0,7},{CR55,0,3}
#define IGA2_HOR_ADDR_REG_NUM           2           // location: {CR51,0,7},{CR55,4,6}
#define IGA2_HOR_BLANK_START_REG_NUM    2           // location: {CR52,0,7},{CR54,0,2}
#define IGA2_HOR_BLANK_END_REG_NUM      3           // location: {CR53,0,7},{CR54,3,5},{CR5D,6,6}
#define IGA2_HOR_SYNC_START_REG_NUM     3           // location: {CR56,0,7},{CR54,6,7},{CR5C,7,7}
#define IGA2_HOR_SYNC_END_REG_NUM       2           // location: {CR57,0,7},{CR5C,6,6}
#define IGA2_VER_TOTAL_REG_NUM          2           // location: {CR58,0,7},{CR5D,0,2}
#define IGA2_VER_ADDR_REG_NUM           2           // location: {CR59,0,7},{CR5D,3,5}
#define IGA2_VER_BLANK_START_REG_NUM    2           // location: {CR5A,0,7},{CR5C,0,2}
#define IGA2_VER_BLANK_END_REG_NUM      2           // location: {CR5E,0,7},{CR5C,3,5}
#define IGA2_VER_SYNC_START_REG_NUM     2           // location: {CR5E,0,7},{CR5F,5,7}
#define IGA2_VER_SYNC_END_REG_NUM       1           // location: {CR5F,0,4}

/* Define Offset and Fetch Count Register*/ 
#define IGA1_OFFSET_REG_NUM             2                                 // location: {CR13,0,7},{CR35,5,7}
#define IGA1_OFFSER_ALIGN_BYTE          8                                 // 8 bytes alignment.
#define IGA1_OFFSET_FORMULA(x,y)        (x*y)/IGA1_OFFSER_ALIGN_BYTE      // x: H resolution, y: color depth

#define IGA1_FETCH_COUNT_REG_NUM        2                                 // location: {SR1C,0,7},{SR1D,0,1}
#define IGA1_FETCH_COUNT_ALIGN_BYTE     16                                // 16 bytes alignment.
#define IGA1_FETCH_COUNT_PATCH_VALUE    4                                 // x: H resolution, y: color depth  
#define IGA1_FETCH_COUNT_FORMULA(x,y)   ((x*y)/IGA1_FETCH_COUNT_ALIGN_BYTE)+ IGA1_FETCH_COUNT_PATCH_VALUE                       




#define IGA2_OFFSET_REG_NUM             2           // location: {CR66,0,7},{CR67,0,1}
#define IGA2_OFFSET_ALIGN_BYTE		    8
#define IGA2_OFFSET_FORMULA(x,y)	    (x*y)/IGA2_OFFSET_ALIGN_BYTE	 // x: H resolution, y: color depth

#define IGA2_FETCH_COUNT_REG_NUM        2           // location: {CR65,0,7},{CR67,2,3}
#define IGA2_FETCH_COUNT_ALIGN_BYTE	    16
#define IGA2_FETCH_COUNT_PATCH_VALUE	0
#define IGA2_FETCH_COUNT_FORMULA(x,y)	((x*y)/IGA2_FETCH_COUNT_ALIGN_BYTE)+ IGA2_FETCH_COUNT_PATCH_VALUE

// Staring Address
#define IGA1_STARTING_ADDR_REG_NUM      4           // location: {CR0C,0,7},{CR0D,0,7},{CR34,0,7},{CR48,0,1}
#define IGA2_STARTING_ADDR_REG_NUM      3           // location: {CR62,1,7},{CR63,0,7},{CR64,0,7}

// Define Display OFFSET
// These value are by HW suggested value
#define K800_IGA1_FIFO_MAX_DEPTH                384     // location: {SR17,0,7} 
#define K800_IGA1_FIFO_THRESHOLD                328     // location: {SR16,0,5},{SR16,7,7} 
#define K800_IGA1_GFX_PREQ_THRESHOLD            74      // location: {SR18,0,5},{SR18,7,7}
#define K800_IGA1_DISPLAY_QUEUE_EXPIRE_NUM      0       // location: {SR22,0,4}. (128/4) =64, K800 must be set zero,
                                                        // because HW only 5 bits 
                                                        
#define K800_IGA2_FIFO_MAX_DEPTH                384     // location: {CR68,4,7},{CR94,7,7},{CR95,7,7} 
#define K800_IGA2_FIFO_THRESHOLD                328     // location: {CR68,0,3},{CR95,4,6}
#define K800_IGA2_GFX_PREQ_THRESHOLD            74      // location: {CR92,0,3},{CR95,0,2}
#define K800_IGA2_DISPLAY_QUEUE_EXPIRE_NUM      128     // location: {CR94,0,6}

#define P880_IGA1_FIFO_MAX_DEPTH                192     // location: {SR17,0,7} 
#define P880_IGA1_FIFO_THRESHOLD                128     // location: {SR16,0,5},{SR16,7,7} 
#define P880_IGA1_GFX_PREQ_THRESHOLD            16      // location: {SR18,0,5},{SR18,7,7}
#define P880_IGA1_DISPLAY_QUEUE_EXPIRE_NUM      0       // location: {SR22,0,4}. (128/4) =64, K800 must be set zero,
                                                        // because HW only 5 bits 
                                                        
#define P880_IGA2_FIFO_MAX_DEPTH                96      // location: {CR68,4,7},{CR94,7,7},{CR95,7,7} 
#define P880_IGA2_FIFO_THRESHOLD                64      // location: {CR68,0,3},{CR95,4,6}
#define P880_IGA2_GFX_PREQ_THRESHOLD            8       // location: {CR92,0,3},{CR95,0,2}
#define P880_IGA2_DISPLAY_QUEUE_EXPIRE_NUM      128     // location: {CR94,0,6}

#define IGA1_FIFO_DEPTH_SELECT_REG_NUM          1
#define IGA1_FIFO_THRESHOLD_REG_NUM             2
#define IGA1_GFX_PREQ_THRESHOLD_REG_NUM         2
#define IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM   1

#define IGA2_FIFO_DEPTH_SELECT_REG_NUM          3
#define IGA2_FIFO_THRESHOLD_REG_NUM             2
#define IGA2_GFX_PREQ_THRESHOLD_REG_NUM         2
#define IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM   1


#define IGA1_FIFO_DEPTH_SELECT_FORMULA(x)                   (x/2)-1
#define IGA1_FIFO_THRESHOLD_FORMULA(x)                      (x/4)    
#define IGA1_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(x)            (x/4)  
#define IGA2_FIFO_DEPTH_SELECT_FORMULA(x)                   ((x/2)/4)-1
#define IGA2_FIFO_THRESHOLD_FORMULA(x)                      (x/4)
#define IGA2_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(x)            (x/4) 

/************************************************************************/
/*  LCD Timing                                                          */
/************************************************************************/
#define LCD_POWER_SEQ_TD0               500000         // 500 ms = 500000 us 
#define LCD_POWER_SEQ_TD1               50000          // 50 ms = 50000 us
#define LCD_POWER_SEQ_TD2               0              // 0 us
#define LCD_POWER_SEQ_TD3               210000         // 210 ms = 210000 us

#define CLE266_POWER_SEQ_UNIT           71             // 2^10 * (1/14.31818M) = 71.475 us (K400.revA)
#define K800_POWER_SEQ_UNIT             142            // 2^11 * (1/14.31818M) = 142.95 us (K400.revB)   
#define P880_POWER_SEQ_UNIT             572            // 2^13 * (1/14.31818M) = 572.1 us

#define CLE266_POWER_SEQ_FORMULA(x)     (x)/CLE266_POWER_SEQ_UNIT
#define K800_POWER_SEQ_FORMULA(x)       (x)/K800_POWER_SEQ_UNIT
#define P880_POWER_SEQ_FORMULA(x)       (x)/P880_POWER_SEQ_UNIT        


#define LCD_POWER_SEQ_TD0_REG_NUM       2   // location: {CR8B,0,7},{CR8F,0,3}
#define LCD_POWER_SEQ_TD1_REG_NUM       2   // location: {CR8C,0,7},{CR8F,4,7}       
#define LCD_POWER_SEQ_TD2_REG_NUM       2   // location: {CR8D,0,7},{CR90,0,3}
#define LCD_POWER_SEQ_TD3_REG_NUM       2   // location: {CR8E,0,7},{CR90,4,7}


// LCD Scaling factor
// x: indicate setting horizontal size
// y: indicate panel horizontal size

#define CLE266_LCD_HOR_SCF_FORMULA(x,y)   (x/y)*1024    // Horizontal scaling factor 10 bits (2^10)                                                                    
#define CLE266_LCD_VER_SCF_FORMULA(x,y)   (x/y)*1024    // Vertical scaling factor 10 bits (2^10)                                                        
#define K800_LCD_HOR_SCF_FORMULA(x,y)     (x/y)*4096    // Horizontal scaling factor 10 bits (2^12)            
#define K800_LCD_VER_SCF_FORMULA(x,y)     (x/y)*2048    // Vertical scaling factor 10 bits (2^11)  

#define LCD_HOR_SCALING_FACTOR_REG_NUM  3   // location: {CR9F,0,1},{CR77,0,7},{CR79,4,5}
#define LCD_VER_SCALING_FACTOR_REG_NUM  3   // location: {CR79,3,3},{CR78,0,7},{CR79,6,7}       

        

//************************************************//
//*      Define IGA1 Display Timing              *//
//************************************************//
struct io_register {
    u8      io_addr;
    u8      start_bit;
    u8      end_bit;
};


/* IGA1 Horizontal Total */
struct iga1_hor_total
{
    int     reg_num;
    struct  io_register reg[IGA1_HOR_TOTAL_REG_NUM];
};

/* IGA1 Horizontal Addressable Video */
struct iga1_hor_addr {
    int     reg_num;
    struct  io_register reg[IGA1_HOR_ADDR_REG_NUM];
};

/* IGA1 Horizontal Blank Start */
struct iga1_hor_blank_start {
    int     reg_num;
    struct  io_register reg[IGA1_HOR_BLANK_START_REG_NUM];
};

/* IGA1 Horizontal Blank End */
struct iga1_hor_blank_end {
    int     reg_num;
    struct  io_register reg[IGA1_HOR_BLANK_END_REG_NUM];
};

/* IGA1 Horizontal Sync Start */
struct iga1_hor_sync_start {
    int     reg_num;
    struct  io_register reg[IGA1_HOR_SYNC_START_REG_NUM];
};

/* IGA1 Horizontal Sync End */
struct iga1_hor_sync_end {
    int     reg_num;
    struct  io_register reg[IGA1_HOR_SYNC_END_REG_NUM];
};

/* IGA1 Vertical Total */
struct iga1_ver_total {
    int     reg_num;
    struct  io_register reg[IGA1_VER_TOTAL_REG_NUM];
};

/* IGA1 Vertical Addressable Video */
struct iga1_ver_addr {
    int     reg_num;
    struct  io_register reg[IGA1_VER_ADDR_REG_NUM];
};

/* IGA1 Vertical Blank Start */
struct iga1_ver_blank_start {
    int     reg_num;
    struct  io_register reg[IGA1_VER_BLANK_START_REG_NUM];
};

/* IGA1 Vertical Blank End */
struct iga1_ver_blank_end {
    int     reg_num;
    struct  io_register reg[IGA1_VER_BLANK_END_REG_NUM];
};

/* IGA1 Vertical Sync Start */
struct iga1_ver_sync_start {
    int     reg_num;
    struct  io_register reg[IGA1_VER_SYNC_START_REG_NUM];
};

/* IGA1 Vertical Sync End */
struct iga1_ver_sync_end {
    int     reg_num;
    struct  io_register reg[IGA1_VER_SYNC_END_REG_NUM];
};

//************************************************//
//      Define IGA1 Shadow Display Timing         //
//************************************************//

/* IGA1 Shadow Horizontal Total */
struct iga1_shadow_hor_total
{
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_HOR_TOTAL_REG_NUM];
};

/* IGA1 Shadow Horizontal Blank End */
struct iga1_shadow_hor_blank_end {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_HOR_BLANK_END_REG_NUM];
};


/* IGA1 Shadow Vertical Total */
struct iga1_shadow_ver_total {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_TOTAL_REG_NUM];
};

/* IGA1 Shadow Vertical Addressable Video */
struct iga1_shadow_ver_addr {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_ADDR_REG_NUM];
};

/* IGA1 Shadow Vertical Blank Start */
struct iga1_shadow_ver_blank_start {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_BLANK_START_REG_NUM];
};

/* IGA1 Shadow Vertical Blank End */
struct iga1_shadow_ver_blank_end {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_BLANK_END_REG_NUM];
};

/* IGA1 Shadow Vertical Sync Start */
struct iga1_shadow_ver_sync_start {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_SYNC_START_REG_NUM];
};

/* IGA1 Shadow Vertical Sync End */
struct iga1_shadow_ver_sync_end {
    int     reg_num;
    struct  io_register reg[IGA1_SHADOW_VER_SYNC_END_REG_NUM];
};

//************************************************//
//      Define IGA2 Display Timing                //
//************************************************//

/* IGA2 Horizontal Total */
struct iga2_hor_total {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_TOTAL_REG_NUM];
};

/* IGA2 Horizontal Addressable Video */
struct iga2_hor_addr {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_ADDR_REG_NUM];
};

/* IGA2 Horizontal Blank Start */
struct iga2_hor_blank_start {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_BLANK_START_REG_NUM];
};

/* IGA2 Horizontal Blank End */
struct iga2_hor_blank_end {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_BLANK_END_REG_NUM];
};

/* IGA2 Horizontal Sync Start */
struct iga2_hor_sync_start {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_SYNC_START_REG_NUM];
};

/* IGA2 Horizontal Sync End */
struct iga2_hor_sync_end {
    int     reg_num;
    struct  io_register reg[IGA2_HOR_SYNC_END_REG_NUM];
};

/* IGA2 Vertical Total */
struct iga2_ver_total {
    int     reg_num;
    struct  io_register reg[IGA2_VER_TOTAL_REG_NUM];
};

/* IGA2 Vertical Addressable Video */
struct iga2_ver_addr {
    int     reg_num;
    struct  io_register reg[IGA2_VER_ADDR_REG_NUM];
};

/* IGA2 Vertical Blank Start */
struct iga2_ver_blank_start {
    int     reg_num;
    struct  io_register reg[IGA2_VER_BLANK_START_REG_NUM];
};

/* IGA2 Vertical Blank End */
struct iga2_ver_blank_end {
    int     reg_num;
    struct  io_register reg[IGA2_VER_BLANK_END_REG_NUM];
};

/* IGA2 Vertical Sync Start */
struct iga2_ver_sync_start {
    int     reg_num;
    struct  io_register reg[IGA2_VER_SYNC_START_REG_NUM];
};

/* IGA2 Vertical Sync End */
struct iga2_ver_sync_end {
    int     reg_num;
    struct  io_register reg[IGA2_VER_SYNC_END_REG_NUM];
};

/* IGA1 Offset Register */
struct iga1_offset {
    int     reg_num;
    struct  io_register reg[IGA1_OFFSET_REG_NUM];
}; 

/* IGA2 Offset Register */
struct iga2_offset {
    int     reg_num;
    struct  io_register reg[IGA2_OFFSET_REG_NUM];
}; 

struct offset{
    struct iga1_offset            iga1_offset_reg;
    struct iga2_offset            iga2_offset_reg;
};

/* IGA1 Fetch Count Register */
struct iga1_fetch_count {
    int     reg_num;
    struct  io_register reg[IGA1_FETCH_COUNT_REG_NUM];
}; 

/* IGA2 Fetch Count Register */
struct iga2_fetch_count {
    int     reg_num;
    struct  io_register reg[IGA2_FETCH_COUNT_REG_NUM];
}; 

struct fetch_count{
    struct iga1_fetch_count       iga1_fetch_count_reg;
    struct iga2_fetch_count       iga2_fetch_count_reg;
};

/* Starting Address Register */
struct iga1_starting_addr {
    int     reg_num;
    struct  io_register reg[IGA1_STARTING_ADDR_REG_NUM];
}; 

struct iga2_starting_addr {
    int     reg_num;
    struct  io_register reg[IGA2_STARTING_ADDR_REG_NUM];
}; 

struct starting_addr {
    struct iga1_starting_addr       iga1_starting_addr_reg;
    struct iga2_starting_addr       iga2_starting_addr_reg;
};

/* LCD Power Sequence Timer */
struct lcd_pwd_seq_td0{
    int     reg_num;
    struct  io_register reg[LCD_POWER_SEQ_TD0_REG_NUM];
};

struct lcd_pwd_seq_td1{
    int     reg_num;
    struct  io_register reg[LCD_POWER_SEQ_TD1_REG_NUM];
};

struct lcd_pwd_seq_td2{
    int     reg_num;
    struct  io_register reg[LCD_POWER_SEQ_TD2_REG_NUM];
};

struct lcd_pwd_seq_td3{
    int     reg_num;
    struct  io_register reg[LCD_POWER_SEQ_TD3_REG_NUM];
};

struct _lcd_pwd_seq_timer{
    struct lcd_pwd_seq_td0       td0;
    struct lcd_pwd_seq_td1       td1;
    struct lcd_pwd_seq_td2       td2;
    struct lcd_pwd_seq_td3       td3;
};

/* LCD Scaling Factor */
struct _lcd_hor_scaling_factor{
    int     reg_num;
    struct  io_register reg[LCD_HOR_SCALING_FACTOR_REG_NUM];
};

struct _lcd_ver_scaling_factor{
    int     reg_num;
    struct  io_register reg[LCD_VER_SCALING_FACTOR_REG_NUM];
};


struct _lcd_scaling_factor{
    struct _lcd_hor_scaling_factor  lcd_hor_scaling_factor;
    struct _lcd_ver_scaling_factor  lcd_ver_scaling_factor;
};

struct pll_map {
    u32     clk;
    u32     cle266_pll;
    u32     k800_pll;
}; 

struct rgbLUT {
    u8     red;
    u8     green;
    u8     blue;
};

struct lcd_pwd_seq_timer {
    u16     td0;
    u16     td1;
    u16     td2;
    u16     td3;    
};


// Display FIFO Relation Registers
struct iga1_fifo_depth_select {
    int     reg_num;
    struct  io_register reg[IGA1_FIFO_DEPTH_SELECT_REG_NUM];
};

struct iga1_fifo_threshold_select {
    int     reg_num;
    struct  io_register reg[IGA1_FIFO_THRESHOLD_REG_NUM];    
};

struct iga1_gfx_preq_thresold_select {
    int     reg_num;
    struct  io_register reg[IGA1_GFX_PREQ_THRESHOLD_REG_NUM];    
};

struct iga1_display_queue_expire_num {
    int     reg_num;
    struct  io_register reg[IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM];    
};

struct iga2_fifo_depth_select {
    int     reg_num;
    struct  io_register reg[IGA2_FIFO_DEPTH_SELECT_REG_NUM];
};

struct iga2_fifo_threshold_select {
    int     reg_num;
    struct  io_register reg[IGA2_FIFO_THRESHOLD_REG_NUM];
};

struct iga2_gfx_preq_thresold_select {
    int     reg_num;
    struct  io_register reg[IGA2_GFX_PREQ_THRESHOLD_REG_NUM];
};

struct iga2_display_queue_expire_num {
    int     reg_num;
    struct  io_register reg[IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM];    
};

struct fifo_depth_select {
    struct  iga1_fifo_depth_select iga1_fifo_depth_select_reg;
    struct  iga2_fifo_depth_select iga2_fifo_depth_select_reg;
};

struct fifo_threshold_select {
    struct  iga1_fifo_threshold_select iga1_fifo_threshold_select_reg;
    struct  iga2_fifo_threshold_select iga2_fifo_threshold_select_reg;
};

struct gfx_preq_thresold_select {
    struct  iga1_gfx_preq_thresold_select iga1_gfx_preq_thresold_select_reg;
    struct  iga2_gfx_preq_thresold_select iga2_gfx_preq_thresold_select_reg;
};

struct display_queue_expire_num {
    struct  iga1_display_queue_expire_num iga1_display_queue_expire_num_reg;
    struct  iga2_display_queue_expire_num iga2_display_queue_expire_num_reg;
};

 

struct iga1_crtc_timing {
    struct iga1_hor_total         hor_total;
    struct iga1_hor_addr          hor_addr;
    struct iga1_hor_blank_start   hor_blank_start;
    struct iga1_hor_blank_end     hor_blank_end;
    struct iga1_hor_sync_start    hor_sync_start;
    struct iga1_hor_sync_end      hor_sync_end;
    struct iga1_ver_total         ver_total;
    struct iga1_ver_addr          ver_addr;
    struct iga1_ver_blank_start   ver_blank_start;
    struct iga1_ver_blank_end     ver_blank_end;
    struct iga1_ver_sync_start    ver_sync_start;
    struct iga1_ver_sync_end      ver_sync_end;
};

struct iga1_shadow_crtc_timing {
    struct iga1_shadow_hor_total        hor_total;
    struct iga1_hor_addr                hor_addr;
    struct iga1_hor_blank_start         hor_blank_start;
    struct iga1_shadow_hor_blank_end    hor_blank_end;
    struct iga1_hor_sync_start          hor_sync_start;
    struct iga1_hor_sync_end            hor_sync_end;
    struct iga1_shadow_ver_total        ver_total;
    struct iga1_shadow_ver_addr         ver_addr;
    struct iga1_shadow_ver_blank_start  ver_blank_start;
    struct iga1_shadow_ver_blank_end    ver_blank_end;
    struct iga1_shadow_ver_sync_start   ver_sync_start;
    struct iga1_shadow_ver_sync_end     ver_sync_end;
};

struct iga2_crtc_timing {
    struct iga2_hor_total         hor_total;
    struct iga2_hor_addr          hor_addr;
    struct iga2_hor_blank_start   hor_blank_start;
    struct iga2_hor_blank_end     hor_blank_end;
    struct iga2_hor_sync_start    hor_sync_start;
    struct iga2_hor_sync_end      hor_sync_end;
    struct iga2_ver_total         ver_total;
    struct iga2_ver_addr          ver_addr;
    struct iga2_ver_blank_start   ver_blank_start;
    struct iga2_ver_blank_end     ver_blank_end;
    struct iga2_ver_sync_start    ver_sync_start;
    struct iga2_ver_sync_end      ver_sync_end;
};


static struct pll_map pll_value[] = {
    {CLK_25_175M,  CLE266_PLL_25_175M,  K800_PLL_25_175M},
    {CLK_26_880M,  CLE266_PLL_26_880M,  K800_PLL_26_880M},
    {CLK_31_490M,  CLE266_PLL_31_490M,  K800_PLL_31_490M},
    {CLK_31_500M,  CLE266_PLL_31_500M,  K800_PLL_31_500M},
    {CLK_31_728M,  CLE266_PLL_31_728M,  K800_PLL_31_728M},
    {CLK_32_668M,  CLE266_PLL_32_668M,  K800_PLL_32_668M},
    {CLK_36_000M,  CLE266_PLL_36_000M,  K800_PLL_36_000M},
    {CLK_40_000M,  CLE266_PLL_40_000M,  K800_PLL_40_000M},
    {CLK_41_291M,  CLE266_PLL_41_291M,  K800_PLL_41_291M},
    {CLK_43_163M,  CLE266_PLL_43_163M,  K800_PLL_43_163M},
    {CLK_49_500M,  CLE266_PLL_49_500M,  K800_PLL_49_500M},
    {CLK_52_406M,  CLE266_PLL_52_406M,  K800_PLL_52_406M},
    {CLK_56_250M,  CLE266_PLL_56_250M,  K800_PLL_56_250M},
    {CLK_65_000M,  CLE266_PLL_65_000M,  K800_PLL_65_000M},
    {CLK_68_179M,  CLE266_PLL_68_179M,  K800_PLL_68_179M},
    {CLK_78_750M,  CLE266_PLL_78_750M,  K800_PLL_78_750M},
    {CLK_80_136M,  CLE266_PLL_80_136M,  K800_PLL_80_136M},
    {CLK_83_950M,  CLE266_PLL_83_950M,  K800_PLL_83_950M},
    {CLK_94_500M,  CLE266_PLL_94_500M,  K800_PLL_94_500M},
    {CLK_108_000M, CLE266_PLL_108_000M, K800_PLL_108_000M},
    {CLK_125_104M, CLE266_PLL_125_104M, K800_PLL_125_104M},
    {CLK_133_308M, CLE266_PLL_133_308M, K800_PLL_133_308M},
    {CLK_135_000M, CLE266_PLL_135_000M, K800_PLL_135_000M},
    {CLK_157_500M, CLE266_PLL_157_500M, K800_PLL_157_500M},
    {CLK_162_000M, CLE266_PLL_162_000M, K800_PLL_162_000M},
    {CLK_202_500M, CLE266_PLL_202_500M, K800_PLL_202_500M},
    {CLK_234_000M, CLE266_PLL_234_000M, K800_PLL_234_000M},
    {CLK_297_500M, CLE266_PLL_297_500M, K800_PLL_297_500M}
};

#define NUM_TOTAL_PLL_TABLE ARRAY_SIZE(pll_value)

static struct fifo_depth_select display_fifo_depth_reg= {
    // IGA1 FIFO Depth_Select
    {IGA1_FIFO_DEPTH_SELECT_REG_NUM, {{SR17,0,7}}},
    // IGA2 FIFO Depth_Select
    {IGA2_FIFO_DEPTH_SELECT_REG_NUM, {{CR68,4,7}, {CR94,7,7}, {CR95,7,7}}}
};

static struct fifo_threshold_select fifo_threshold_select_reg= {
    // IGA1 FIFO Threshold Select
    {IGA1_FIFO_THRESHOLD_REG_NUM, {{SR16,0,5},{SR16,7,7}}},
    // IGA2 FIFO Threshold Select
    {IGA2_FIFO_THRESHOLD_REG_NUM, {{CR68,0,3}, {CR95,4,6}}}
};

static struct gfx_preq_thresold_select gfx_preq_thresold_select_reg= {
    // IGA1 GFX PREG Thresold Select
    {IGA1_GFX_PREQ_THRESHOLD_REG_NUM, {{SR18,0,5},{SR18,7,7}}},
    // IGA2 GFX PREG Thresold Select
    {IGA2_GFX_PREQ_THRESHOLD_REG_NUM, {{CR92,0,3}, {CR95,0,2}}}
};

static struct display_queue_expire_num display_queue_expire_num_reg= {
    // IGA1 Display Queue Expire Num
    {IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM, {{SR22,0,4}}},
    // IGA2 Display Queue Expire Num
    {IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM, {{CR94,0,6}}}
};

// Definition Offset Registers 
static struct offset offset_reg = {
    // IGA1 Offset Register
    {IGA1_OFFSET_REG_NUM, {{CR13,0,7},{CR35,5,7}}},
    // IGA2 Offset Register
    {IGA2_OFFSET_REG_NUM, {{CR66,0,7},{CR67,0,1}}}
};

// Definition Fetch Count Registers 
static struct fetch_count fetch_count_reg = {
    // IGA1 Fetch Count Register
    {IGA1_FETCH_COUNT_REG_NUM, {{SR1C,0,7},{SR1D,0,1}}},
    // IGA2 Fetch Count Register
    {IGA2_FETCH_COUNT_REG_NUM, {{CR65,0,7},{CR67,2,3}}}
};

// Definition Starting Address Registers 
/*static struct starting_addr starting_addr_reg = {
    // IGA1 Starting Address Register
    {IGA1_STARTING_ADDR_REG_NUM, {{CR0C,0,7},{CR0D,0,7},{CR34,0,7},{CR48,0,1}}},
    // IGA2 Starting Address Register
    {IGA2_STARTING_ADDR_REG_NUM, {{CR62,1,7},{CR63,0,7},{CR64,0,7}}}
};*/

#define IGA1_STARTING_ADDR_REG_NUM      4           // location: {CR0C,0,7},{CR0D,0,7},{CR34,0,7},{CR48,0,1}
#define IGA2_STARTING_ADDR_REG_NUM      3           // location: {CR62,1,7},{CR63,0,7},{CR64,0,7}

static struct iga1_crtc_timing iga1_crtc_reg = {
    // IGA1 Horizontal Total
    {IGA1_HOR_TOTAL_REG_NUM, {{CR00,0,7}, {CR36,3,3}}},
    // IGA1 Horizontal Addressable Video
    {IGA1_HOR_ADDR_REG_NUM, {{CR01,0,7}}},
    // IGA1 Horizontal Blank Start
    {IGA1_HOR_BLANK_START_REG_NUM, {{CR02,0,7}}},
    // IGA1 Horizontal Blank End
    {IGA1_HOR_BLANK_END_REG_NUM, {{CR03,0,4}, {CR05,7,7}, {CR33,5,5}}},
    // IGA1 Horizontal Sync Start
    {IGA1_HOR_SYNC_START_REG_NUM, {{CR04,0,7}, {CR33,4,4}}},
    // IGA1 Horizontal Sync End
    {IGA1_HOR_SYNC_END_REG_NUM, {{CR05,0,4}}},
    // IGA1 Vertical Total
    {IGA1_VER_TOTAL_REG_NUM, {{CR06,0,7}, {CR07,0,0}, {CR07,5,5}, {CR35,0,0}}},
    // IGA1 Vertical Addressable Video
    {IGA1_VER_ADDR_REG_NUM, {{CR12,0,7}, {CR07,1,1}, {CR07,6,6}, {CR35,2,2}}},
    // IGA1 Vertical Blank Start
    {IGA1_VER_BLANK_START_REG_NUM, {{CR15,0,7}, {CR07,3,3}, {CR09,5,5}, {CR35,3,3}}},
    // IGA1 Vertical Blank End
    {IGA1_VER_BLANK_END_REG_NUM, {{CR16,0,7}}},
    // IGA1 Vertical Sync Start
    {IGA1_VER_SYNC_START_REG_NUM, {{CR10,0,7}, {CR07,2,2}, {CR07,7,7}, {CR35,1,1}}},
    // IGA1 Vertical Sync End
    {IGA1_VER_SYNC_END_REG_NUM, {{CR11,0,3}}}
};

static struct iga1_shadow_crtc_timing iga1_shadow_crtc_reg = {
    // IGA1 Shadow Horizontal Total
    {IGA1_SHADOW_HOR_TOTAL_REG_NUM, {{CR6D,0,7}, {CR71,3,3}}},
    // IGA1 Horizontal Addressable Video
    {IGA1_HOR_ADDR_REG_NUM, {{CR01,0,7}}},
    // IGA1 Horizontal Blank Start
    {IGA1_HOR_BLANK_START_REG_NUM, {{CR02,0,7}}},
    // IGA1 Shadow Horizontal Blank End
    {IGA1_SHADOW_HOR_BLANK_END_REG_NUM, {{CR6E,0,7}}},
    // IGA1 Horizontal Sync Start
    {IGA1_HOR_SYNC_START_REG_NUM, {{CR04,0,7}, {CR33,4,4}}},
    // IGA1 Horizontal Sync End
    {IGA1_HOR_SYNC_END_REG_NUM, {{CR05,0,4}}},
    // IGA1 Shadow Vertical Total
    {IGA1_SHADOW_VER_TOTAL_REG_NUM, {{CR6F,0,7}, {CR71,0,2}}},
    // IGA1 Shadow Vertical Addressable Video
    {IGA1_SHADOW_VER_ADDR_REG_NUM, {{CR70,0,7}, {CR71,4,6}}},
    // IGA1 Shadow Vertical Blank Start
    {IGA1_SHADOW_VER_BLANK_START_REG_NUM, {{CR72,0,7}, {CR74,4,6}}},
    // IGA1 Shadow Vertical Blank End
    {IGA1_SHADOW_VER_BLANK_END_REG_NUM, {{CR73,0,7}, {CR74,0,2}}},
    // IGA1 Shadow Vertical Sync Start
    {IGA1_SHADOW_VER_SYNC_START_REG_NUM, {{CR75,0,7}, {CR76,4,6}}},
    // IGA1 Shadow Vertical Sync End
    {IGA1_SHADOW_VER_SYNC_END_REG_NUM, {{CR76,0,3}}}
    
};

static struct iga2_crtc_timing iga2_crtc_reg = {
    // IGA2 Horizontal Total
    {IGA2_HOR_TOTAL_REG_NUM, {{CR50,0,7}, {CR55,0,3}}},
    // IGA2 Horizontal Addressable Video
    {IGA2_HOR_ADDR_REG_NUM, {{CR51,0,7}, {CR55,4,6}}},
    // IGA2 Horizontal Blank Start
    {IGA2_HOR_BLANK_START_REG_NUM, {{CR52,0,7}, {CR54,0,2}}},
    // IGA2 Horizontal Blank End
    {IGA2_HOR_BLANK_END_REG_NUM, {{CR53,0,7}, {CR54,3,5}, {CR5D,6,6}}},
    // IGA2 Horizontal Sync Start
    {IGA2_HOR_SYNC_START_REG_NUM, {{CR56,0,7}, {CR54,6,7}, {CR5C,7,7}}},
    // IGA2 Horizontal Sync End
    {IGA2_HOR_SYNC_END_REG_NUM, {{CR57,0,7}, {CR5C,6,6}}},
    // IGA2 Vertical Total
    {IGA2_VER_TOTAL_REG_NUM, {{CR58,0,7}, {CR5D,0,2}}},
    // IGA2 Vertical Addressable Video
    {IGA2_VER_ADDR_REG_NUM, {{CR59,0,7}, {CR5D,3,5}}},
    // IGA2 Vertical Blank Start
    {IGA2_VER_BLANK_START_REG_NUM, {{CR5A,0,7}, {CR5C,0,2}}},
    // IGA2 Vertical Blank End
    {IGA2_VER_BLANK_END_REG_NUM, {{CR5B,0,7}, {CR5C,3,5}}},
    // IGA2 Vertical Sync Start
    {IGA2_VER_SYNC_START_REG_NUM, {{CR5E,0,7}, {CR5F,5,7}}},
    // IGA2 Vertical Sync End
    {IGA2_VER_SYNC_END_REG_NUM, {{CR5F,0,4}}}
};

/*static struct _lcd_pwd_seq_timer lcd_pwd_seq_timer = {
    // LCD Power Sequence TD0
    {LCD_POWER_SEQ_TD0_REG_NUM, {{CR8B,0,7}, {CR8F,0,3}}},
    // LCD Power Sequence TD1
    {LCD_POWER_SEQ_TD1_REG_NUM, {{CR8C,0,7}, {CR8F,4,7}}},
    // LCD Power Sequence TD2
    {LCD_POWER_SEQ_TD2_REG_NUM, {{CR8D,0,7}, {CR90,0,3}}},
    // LCD Power Sequence TD3
    {LCD_POWER_SEQ_TD3_REG_NUM, {{CR8E,0,7}, {CR90,4,7}}}
    
};*/

/*static struct _lcd_scaling_factor lcd_scaling_factor = {
    // LCD Horizontal Scaling Factor Register
    {LCD_HOR_SCALING_FACTOR_REG_NUM, {{CR9F,0,1}, {CR77,0,7}, {CR79,4,5}}},
    // LCD Vertical Scaling Factor Register
    {LCD_VER_SCALING_FACTOR_REG_NUM, {{CR79,3,3}, {CR78,0,7}, {CR79,6,7}}}
};*/

static struct rgbLUT palLUT_table[]= {
    // {R,G,B}
    // Index 0x00~0x03
    {0x00, 0x00, 0x00}, {0x00, 0x00, 0x2A}, {0x00, 0x2A, 0x00}, {0x00, 0x2A, 0x2A},
    // Index 0x04~0x07
    {0x2A, 0x00, 0x00}, {0x2A, 0x00, 0x2A}, {0x2A, 0x15, 0x00}, {0x2A, 0x2A, 0x2A},
    // Index 0x08~0x0B
    {0x15, 0x15, 0x15}, {0x15, 0x15, 0x3F}, {0x15, 0x3F, 0x15}, {0x15, 0x3F, 0x3F},
    // Index 0x0C~0x0F
    {0x3F, 0x15, 0x15}, {0x3F, 0x15, 0x3F}, {0x3F, 0x3F, 0x15}, {0x3F, 0x3F, 0x3F},
    // Index 0x10~0x13
    {0x00, 0x00, 0x00}, {0x05, 0x05, 0x05}, {0x08, 0x08, 0x08}, {0x0B, 0x0B, 0x0B},
    // Index 0x14~0x17
    {0x0E, 0x0E, 0x0E}, {0x11, 0x11, 0x11}, {0x14, 0x14, 0x14}, {0x18, 0x18, 0x18},
    // Index 0x18~0x1B
    {0x1C, 0x1C, 0x1C}, {0x20, 0x20, 0x20}, {0x24, 0x24, 0x24}, {0x28, 0x28, 0x28},
    // Index 0x1C~0x1F
    {0x2D, 0x2D, 0x2D}, {0x32, 0x32, 0x32}, {0x38, 0x38, 0x38}, {0x3F, 0x3F, 0x3F},
    // Index 0x20~0x23
    {0x00, 0x00, 0x3F}, {0x10, 0x00, 0x3F}, {0x1F, 0x00, 0x3F}, {0x2F, 0x00, 0x3F},
    // Index 0x24~0x27
    {0x3F, 0x00, 0x3F}, {0x3F, 0x00, 0x2F}, {0x3F, 0x00, 0x1F}, {0x3F, 0x00, 0x10},
    // Index 0x28~0x2B
    {0x3F, 0x00, 0x00}, {0x3F, 0x10, 0x00}, {0x3F, 0x1F, 0x00}, {0x3F, 0x2F, 0x00},
    // Index 0x2C~0x2F
    {0x3F, 0x3F, 0x00}, {0x2F, 0x3F, 0x00}, {0x1F, 0x3F, 0x00}, {0x10, 0x3F, 0x00},
    // Index 0x30~0x33
    {0x00, 0x3F, 0x00}, {0x00, 0x3F, 0x10}, {0x00, 0x3F, 0x1F}, {0x00, 0x3F, 0x2F},
    // Index 0x34~0x37
    {0x00, 0x3F, 0x3F}, {0x00, 0x2F, 0x3F}, {0x00, 0x1F, 0x3F}, {0x00, 0x10, 0x3F},
    // Index 0x38~0x3B
    {0x1F, 0x1F, 0x3F}, {0x27, 0x1F, 0x3F}, {0x2F, 0x1F, 0x3F}, {0x37, 0x1F, 0x3F},
    // Index 0x3C~0x3F
    {0x3F, 0x1F, 0x3F}, {0x3F, 0x1F, 0x37}, {0x3F, 0x1F, 0x2F}, {0x3F, 0x1F, 0x27},
    // Index 0x40~0x43
    {0x3F, 0x1F, 0x1F}, {0x3F, 0x27, 0x1F}, {0x3F, 0x2F, 0x1F}, {0x3F, 0x3F, 0x1F},
    // Index 0x44~0x47
    {0x3F, 0x3F, 0x1F}, {0x37, 0x3F, 0x1F}, {0x2F, 0x3F, 0x1F}, {0x27, 0x3F, 0x1F},
    // Index 0x48~0x4B
    {0x1F, 0x3F, 0x1F}, {0x1F, 0x3F, 0x27}, {0x1F, 0x3F, 0x2F}, {0x1F, 0x3F, 0x37},
    // Index 0x4C~0x4F
    {0x1F, 0x3F, 0x3F}, {0x1F, 0x37, 0x3F}, {0x1F, 0x2F, 0x3F}, {0x1F, 0x27, 0x3F},
    // Index 0x50~0x53
    {0x2D, 0x2D, 0x3F}, {0x31, 0x2D, 0x3F}, {0x36, 0x2D, 0x3F}, {0x3A, 0x2D, 0x3F},
    // Index 0x54~0x57
    {0x3F, 0x2D, 0x3F}, {0x3F, 0x2D, 0x3A}, {0x3F, 0x2D, 0x36}, {0x3F, 0x2D, 0x31},
    // Index 0x58~0x5B
    {0x3F, 0x2D, 0x2D}, {0x3F, 0x31, 0x2D}, {0x3F, 0x36, 0x2D}, {0x3F, 0x3A, 0x2D},
    // Index 0x5C~0x5F
    {0x3F, 0x3F, 0x2D}, {0x3A, 0x3F, 0x2D}, {0x36, 0x3F, 0x2D}, {0x31, 0x3F, 0x2D},
    // Index 0x60~0x63
    {0x2D, 0x3F, 0x2D}, {0x2D, 0x3F, 0x31}, {0x2D, 0x3F, 0x36}, {0x2D, 0x3F, 0x3A},
    // Index 0x64~0x67
    {0x2D, 0x3F, 0x3F}, {0x2D, 0x3A, 0x3F}, {0x2D, 0x36, 0x3F}, {0x2D, 0x31, 0x3F},
    // Index 0x68~0x6B
    {0x00, 0x00, 0x1C}, {0x07, 0x00, 0x1C}, {0x0E, 0x00, 0x1C}, {0x15, 0x00, 0x1C},
    // Index 0x6C~0x6F
    {0x1C, 0x00, 0x1C}, {0x1C, 0x00, 0x15}, {0x1C, 0x00, 0x0E}, {0x1C, 0x00, 0x07},
    // Index 0x70~0x73
    {0x1C, 0x00, 0x00}, {0x1C, 0x07, 0x00}, {0x1C, 0x0E, 0x00}, {0x1C, 0x15, 0x00},
    // Index 0x74~0x77
    {0x1C, 0x1C, 0x00}, {0x15, 0x1C, 0x00}, {0x0E, 0x1C, 0x00}, {0x07, 0x1C, 0x00},
    // Index 0x78~0x7B
    {0x00, 0x1C, 0x00}, {0x00, 0x1C, 0x07}, {0x00, 0x1C, 0x0E}, {0x00, 0x1C, 0x15},
    // Index 0x7C~0x7F
    {0x00, 0x1C, 0x1C}, {0x00, 0x15, 0x1C}, {0x00, 0x0E, 0x1C}, {0x00, 0x07, 0x1C},
    // Index 0x80~0x83
    {0x0E, 0x0E, 0x1C}, {0x11, 0x0E, 0x1C}, {0x15, 0x0E, 0x1C}, {0x18, 0x0E, 0x1C},
    // Index 0x84~0x87
    {0x1C, 0x0E, 0x1C}, {0x1C, 0x0E, 0x18}, {0x1C, 0x0E, 0x15}, {0x1C, 0x0E, 0x11},
    // Index 0x88~0x8B
    {0x1C, 0x0E, 0x0E}, {0x1C, 0x11, 0x0E}, {0x1C, 0x15, 0x0E}, {0x1C, 0x18, 0x0E},
    // Index 0x8C~0x8F
    {0x1C, 0x1C, 0x0E}, {0x18, 0x1C, 0x0E}, {0x15, 0x1C, 0x0E}, {0x11, 0x1C, 0x0E},
    // Index 0x90~0x93
    {0x0E, 0x1C, 0x0E}, {0x0E, 0x1C, 0x11}, {0x0E, 0x1C, 0x15}, {0x0E, 0x1C, 0x18},
    // Index 0x94~0x97
    {0x0E, 0x1C, 0x1C}, {0x0E, 0x18, 0x1C}, {0x0E, 0x15, 0x1C}, {0x0E, 0x11, 0x1C},
    // Index 0x98~0x9B
    {0x14, 0x14, 0x1C}, {0x16, 0x14, 0x1C}, {0x18, 0x14, 0x1C}, {0x1A, 0x14, 0x1C},
    // Index 0x9C~0x9F
    {0x1C, 0x14, 0x1C}, {0x1C, 0x14, 0x1A}, {0x1C, 0x14, 0x18}, {0x1C, 0x14, 0x16},
    // Index 0xA0~0xA3
    {0x1C, 0x14, 0x14}, {0x1C, 0x16, 0x14}, {0x1C, 0x18, 0x14}, {0x1C, 0x1A, 0x14},
    // Index 0xA4~0xA7
    {0x1C, 0x1C, 0x14}, {0x1A, 0x1C, 0x14}, {0x18, 0x1C, 0x14}, {0x16, 0x1C, 0x14},
    // Index 0xA8~0xAB
    {0x14, 0x1C, 0x14}, {0x14, 0x1C, 0x16}, {0x14, 0x1C, 0x18}, {0x14, 0x1C, 0x1A},
    // Index 0xAC~0xAF
    {0x14, 0x1C, 0x1C}, {0x14, 0x1A, 0x1C}, {0x14, 0x18, 0x1C}, {0x14, 0x16, 0x1C},
    // Index 0xB0~0xB3
    {0x00, 0x00, 0x10}, {0x04, 0x00, 0x10}, {0x08, 0x00, 0x10}, {0x0C, 0x00, 0x10},
    // Index 0xB4~0xB7
    {0x10, 0x00, 0x10}, {0x10, 0x00, 0x0C}, {0x10, 0x00, 0x08}, {0x10, 0x00, 0x04},
    // Index 0xB8~0xBB
    {0x10, 0x00, 0x00}, {0x10, 0x04, 0x00}, {0x10, 0x08, 0x00}, {0x10, 0x0C, 0x00},
    // Index 0xBC~0xBF
    {0x10, 0x10, 0x00}, {0x0C, 0x10, 0x00}, {0x08, 0x10, 0x00}, {0x04, 0x10, 0x00},    
    // Index 0xC0~0xC3
    {0x00, 0x10, 0x00}, {0x00, 0x10, 0x04}, {0x00, 0x10, 0x08}, {0x00, 0x10, 0x0C},
    // Index 0xC4~0xC7
    {0x00, 0x10, 0x10}, {0x00, 0x0C, 0x10}, {0x00, 0x08, 0x10}, {0x00, 0x04, 0x10},
    // Index 0xC8~0xCB
    {0x08, 0x08, 0x10}, {0x0A, 0x08, 0x10}, {0x0C, 0x08, 0x10}, {0x0E, 0x08, 0x10},
    // Index 0xCC~0xCF
    {0x10, 0x08, 0x10}, {0x10, 0x08, 0x0E}, {0x10, 0x08, 0x0C}, {0x10, 0x08, 0x0A},  
    // Index 0xD0~0xD3
    {0x10, 0x08, 0x08}, {0x10, 0x0A, 0x08}, {0x10, 0x0C, 0x08}, {0x10, 0x0E, 0x08},
    // Index 0xD4~0xD7
    {0x10, 0x10, 0x08}, {0x0E, 0x10, 0x08}, {0x0C, 0x10, 0x08}, {0x0A, 0x10, 0x08},
    // Index 0xD8~0xDB
    {0x08, 0x10, 0x08}, {0x08, 0x10, 0x0A}, {0x08, 0x10, 0x0C}, {0x08, 0x10, 0x0E},
    // Index 0xDC~0xDF
    {0x08, 0x10, 0x10}, {0x08, 0x0E, 0x10}, {0x08, 0x0C, 0x10}, {0x08, 0x0A, 0x10},  
    // Index 0xE0~0xE3
    {0x0B, 0x0B, 0x10}, {0x0C, 0x0B, 0x10}, {0x0D, 0x0B, 0x10}, {0x0F, 0x0B, 0x10},
    // Index 0xE4~0xE7
    {0x10, 0x0B, 0x10}, {0x10, 0x0B, 0x0F}, {0x10, 0x0B, 0x0D}, {0x10, 0x0B, 0x0C},
    // Index 0xE8~0xEB
    {0x10, 0x0B, 0x0B}, {0x10, 0x0C, 0x0B}, {0x10, 0x0D, 0x0B}, {0x10, 0x0F, 0x0B},
    // Index 0xEC~0xEF
    {0x10, 0x10, 0x0B}, {0x0F, 0x10, 0x0B}, {0x0D, 0x10, 0x0B}, {0x0C, 0x10, 0x0B},   
    // Index 0xF0~0xF3
    {0x0B, 0x10, 0x0B}, {0x0B, 0x10, 0x0C}, {0x0B, 0x10, 0x0D}, {0x0B, 0x10, 0x0F},
    // Index 0xF4~0xF7
    {0x0B, 0x10, 0x10}, {0x0B, 0x0F, 0x10}, {0x0B, 0x0D, 0x10}, {0x0B, 0x0C, 0x10},
    // Index 0xF8~0xFB
    {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00},
    // Index 0xFC~0xFF
    {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00}, {0x00, 0x00, 0x00}     
};

#endif /* __HW_H__ */
  



