/*
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * VIA, S3 GRAPHICS, AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 
#include <linux/config.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <asm/io.h>


#include "hw.h"
#include "share.h"
#include "debug.h"
#include "chip.h"
#include "viamode.h"


// External Function
extern int tmds_trasmitter_identify(void);
extern void init_dvi_size(void);
extern int lvds_trasmitter_identify(void);
extern void init_lcd_size(void);
extern void tv_set_mode(int video_index);
extern void lcd_set_mode(struct crt_mode_table *lcd_crt_timing, int video_index, int mode_bpp, int set_iga);
extern void dvi_set_mode(struct crt_mode_table *dvi_crt_timing, int video_index, int mode_bpp, int set_iga);


// External struct

extern struct chip_information  	chip_info;
extern struct crt_setting_information   crt_setting_info;
extern struct tv_setting_information	tv_setting_info;
extern struct tmds_setting_information	tmds_setting_info; 	
extern struct lvds_setting_information	lvds_setting_info; 


extern int refresh;
extern int tv_level;
extern int tv_system;
extern int tv_out_signal;
extern int tv_dedotcrawl;

extern int CRT_ON, TV_ON, DVI_ON, LCD_ON;

/* Access I/O Function */ 
inline void write_reg(u8 index, u16 io_port, u8 data);
void write_reg_mask(u8 index, int io_port, u8 data, u8 mask);
inline u8 read_reg(int io_port, u8 index);
inline void lock_crt(void);
inline void unlock_crt(void);

void set_crt_output_path(int set_iga);
void set_tv_output_path(int set_iga, int output_interface);
void set_dvi_output_path(int set_iga, int output_interface);
void set_lcd_output_path(int set_iga, int output_interface);
void SetVCLK( u32 CLK, int set_iga );
void init_gfx_chip_info(void);
void init_tv_chip_info(void);
void init_tmds_chip_info(void);
void init_lvds_chip_info(void);
void load_reg(int timing_value, int load_reg_num, struct io_register *reg, int io_type);



inline void write_reg(u8 index, u16 io_port, u8 data)
{
    outb(index, io_port);
    outb(data, io_port+1);
    //DEBUG_MSG(KERN_INFO "\nIndex=%2d Value=%2d", index, data);
}

void write_reg_mask(u8 index, int io_port, u8 data, u8 mask)
{
    u8 tmp;
    
    outb(index, io_port);
    tmp = inb(io_port+1);
    outb((data & mask) | (tmp & (~mask)), io_port+1);
    //DEBUG_MSG(KERN_INFO "\nIndex=%2d Value=%2d", index, tmp);  
}

void write_dac_reg(u8 index, u8 r, u8 g, u8 b)
{
    outb(index, LUT_INDEX_WRITE);
    outb(r, LUT_DATA);      
    outb(g, LUT_DATA);
    outb(b, LUT_DATA); 
}

inline u8 read_reg(int io_port, u8 index)
{
    outb(index, io_port);
    return(inb(io_port+1));
}

inline void lock_crt(void)
{
    write_reg_mask(CR11, VIACR, BIT7, BIT7);       
}

inline void unlock_crt(void)
{
    write_reg_mask(CR11, VIACR, 0, BIT7);
    write_reg_mask(CR47, VIACR, 0, BIT0);
}

void set_output_path(int device, int set_iga, int output_interface)
{
    switch(device) {
        case DEVICE_CRT:
            set_crt_output_path(set_iga);
            break;    
        case DEVICE_TV:
            set_tv_output_path(set_iga, output_interface); 
            break;
        case DEVICE_DVI:
            set_dvi_output_path(set_iga, output_interface);
            break;
        case DEVICE_LCD:
            set_lcd_output_path(set_iga, output_interface);
            break;           
    }
}

void set_crt_output_path(int set_iga)
{
    write_reg_mask(CR36, VIACR, 0x00, BIT4+BIT5);
    
    switch(set_iga) {
        case IGA1:
            write_reg_mask(SR16, VIASR, 0x00, BIT6);
            break;    
        case IGA2:           
        case IGA1_IGA2:
            write_reg_mask(CR6A, VIACR, 0xC0, BIT6+BIT7);
            write_reg_mask(SR16, VIASR, 0x40, BIT6);
            if (set_iga == IGA1_IGA2)
                write_reg_mask(CR6B, VIACR, 0x08, BIT3);
            break;           
    }
}

void set_tv_output_path(int set_iga, int output_interface)
{
    switch(set_iga) 
    {
        case IGA1:
            // Set IGA1 display channel to normal case beside CLE266.AX chip 
            // and disable simultaneous display.
            if ((chip_info.gfx_chip_name == UNICHROME_CLE266) && (chip_info.gfx_chip_revision == CLE266_REVISION_AX))
            {
                SetVCLK(0x471c, IGA1);
                write_reg_mask(CR6B, VIACR, 0xC0, BIT3+BIT6+BIT7);
            } 
            else
                write_reg_mask(CR6B, VIACR, 0x00, BIT3+BIT6+BIT7);
            
            break;    
        case IGA2:
            // Set IGA2 display channel to normal case beside CLE266.AX chip 
            // and disable simultaneous display.
            write_reg_mask(CR6A, VIACR, 0xC0, BIT6+BIT7);
            if (chip_info.gfx_chip_name == UNICHROME_CLE266 && chip_info.gfx_chip_revision == CLE266_REVISION_AX)
                write_reg_mask(CR6B, VIACR, 0x30, BIT3+BIT4+BIT5); 
            else
                write_reg_mask(CR6B, VIACR, 0x00, BIT3+BIT6+BIT7);
            
            // Disable LCD Scaling
            write_reg_mask(CR79, VIACR, 0x00, BIT0);
            break;
        case IGA1_IGA2:
            // Set IGA1_IGA2 display channel to normal case beside CLE266.AX chip 
            // and enable simultaneous display.
            write_reg_mask(CR6A, VIACR, 0xC0, BIT6+BIT7);
            if (chip_info.gfx_chip_name == UNICHROME_CLE266 && chip_info.gfx_chip_revision == CLE266_REVISION_AX)
                write_reg_mask(CR6B, VIACR, 0xC0, BIT3+BIT6+BIT7); 
            else
                write_reg_mask(CR6B, VIACR, 0x00, BIT3+BIT6+BIT7);
            break;           
    }
    
    switch(output_interface) 
    {
        case INTERFACE_DVP0:
            if (set_iga== IGA1)
                write_reg_mask(CR6C, VIACR, 0x00, BIT7);
            else
                write_reg_mask(CR6C, VIACR, 0x80, BIT7);
                
            if (chip_info.gfx_chip_name == UNICHROME_CLE266 && chip_info.gfx_chip_revision == CLE266_REVISION_AX)
            {
                write_reg_mask(CR6B, VIACR, 0x01, BIT0);
                write_reg_mask(CR6C, VIACR, 0x01, BIT0);
            }
            else
            {
                write_reg_mask(CR6B, VIACR, 0x00, BIT0);
                write_reg_mask(CR6C, VIACR, 0x21, BIT0+BIT5);
            }
            write_reg_mask(SR1E, VIASR, 0xC0, BIT6+BIT7);               
            break;
            
        case INTERFACE_DVP1:
        
            if (set_iga== IGA1)
            {
                write_reg_mask(CR6C, VIACR, 0x00, BIT7);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR6C, VIACR, 0x80, BIT7); 
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            write_reg_mask(CR6C, VIACR, 0x21, BIT0+BIT5);
            write_reg_mask(SR1E, VIASR, 0x30, BIT4+BIT5);
            break;
        case INTERFACE_DFP_HIGH:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR6C, VIACR, 0x00, BIT7);
                write_reg_mask(CR96, VIACR, 0x00, BIT4);
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR6C, VIACR, 0x80, BIT7);
                write_reg_mask(CR96, VIACR, 0x10, BIT4); 
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
            }
            write_reg_mask(CR6C, VIACR, 0x21, BIT0+BIT5);
            break;
        case INTERFACE_DFP_LOW:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR6C, VIACR, 0x00, BIT7);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR6C, VIACR, 0x80, BIT7); 
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
            }
            write_reg_mask(CR6C, VIACR, 0x21, BIT0+BIT5);
            break;
    } 
}

void set_dvi_output_path(int set_iga, int output_interface)
{
   
    switch(output_interface) 
    {
        case INTERFACE_DVP0:
            
            break;
        case INTERFACE_DVP1:
            if (chip_info.gfx_chip_name == UNICHROME_CLE266)
            {
                if (set_iga== IGA1)
                    write_reg_mask(CR93, VIACR, 0x21, BIT0+BIT5+BIT7);
                else
                    write_reg_mask(CR93, VIACR, 0xA1, BIT0+BIT5+BIT7);    
            }
            else
            {
                if (set_iga== IGA1)
                    write_reg_mask(CR9B, VIACR, 0x00, BIT4);
                else
                    write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            write_reg_mask(SR1E, VIASR, 0x30, BIT4+BIT5);
            break;
        case INTERFACE_DFP_HIGH:
            break;
        case INTERFACE_DFP_LOW:
            break;
    } 
}

void set_lcd_output_path(int set_iga, int output_interface)
{
    
    switch(set_iga) 
    {
        case IGA1:
            write_reg_mask(CR6B, VIACR, 0x00, BIT3+BIT6+BIT7);
            break;    
        case IGA2:
            write_reg_mask(CR6A, VIACR, 0xC0, BIT6+BIT7);
            break;
        case IGA1_IGA2:
            write_reg_mask(CR6B, VIACR, 0x00, BIT3+BIT6+BIT7);
            break;           
    }
    
    switch(output_interface) 
    {
        case INTERFACE_DVP0:
            
            break;
        case INTERFACE_DVP1:
            if (set_iga== IGA1)
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            else
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
                
            write_reg_mask(CR6A, VIACR, 0x01, BIT0);       
            break;
        case INTERFACE_DFP_HIGH:
            if (set_iga== IGA1)
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
            else
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
            write_reg_mask(CR6A, VIACR, 0x01, BIT0);        
            break;
        case INTERFACE_DFP_LOW:
            if (set_iga== IGA1)
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            else
                write_reg_mask(CR99, VIACR, 0x10, BIT4);    
            write_reg_mask(CR6A, VIACR, 0x01, BIT0);    
            break;
        case INTERFACE_DFP:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
                write_reg_mask(CR99, VIACR, 0x10, BIT4);    
            }
            write_reg_mask(CR6A, VIACR, 0x01, BIT0);
            break;               
    } 
}

/* Search Mode Index */
static int SearchModeSetting( int ModeInfoIndex )
{
   int i=0;

   while ( ( i<NUM_TOTAL_MODETABLE ) && ( ModeInfoIndex != CLE266Modes[i].ModeIndex) ){i++;}
   if( ( i >= NUM_TOTAL_MODETABLE ) ) i=0;
   return i;

}

void load_fix_bit_crtc_reg(void)
{
     write_reg_mask(CR03, VIACR, 0x80, BIT7);   // always set to 1
     write_reg(CR18, VIACR, 0xff);                              // line compare should set all bits = 1 (extend modes)
     write_reg_mask(CR07, VIACR, 0x10, BIT4);                   // line compare should set all bits = 1 (extend modes)
     write_reg_mask(CR09, VIACR, 0x40, BIT6);                   // line compare should set all bits = 1 (extend modes)
     write_reg_mask(CR35, VIACR, 0x10, BIT4);                   // line compare should set all bits = 1 (extend modes)
     write_reg_mask(CR33, VIACR, 0x05, BIT0+BIT1+BIT2);    	// line compare should set all bits = 1 (extend modes)
     //write_reg_mask(CR32, VIACR, 0x01, BIT0);
     write_reg(CR17, VIACR, 0xe3);                              // extend mode always set to e3h
     write_reg(CR08, VIACR, 0x00);                              // extend mode always set to 0h
     write_reg(CR14, VIACR, 0x00);                              // extend mode always set to 0h
     
     if ((chip_info.gfx_chip_name == UNICHROME_CLE266) && (chip_info.gfx_chip_revision == CLE266_REVISION_AX))
         write_reg_mask(SR1A, VIASR, 0x02, BIT1);
    
}

void load_reg(int timing_value, int load_reg_num, struct io_register *reg, int io_type)
{
    int reg_mask;
    int bit_num=0;
    int data;
    int i,j;
    int shift_next_reg;
    int start_index, end_index, cr_index;
    u16 get_bit;



    for (i=0; i<load_reg_num; i++)
    {
	    reg_mask = 0;
	    data=0;
	    start_index = reg[i].start_bit;
	    end_index = reg[i].end_bit;
	    cr_index = reg[i].io_addr;

	    shift_next_reg = bit_num;
	    for(j=start_index;j<=end_index;j++)
	    {
		    //if (bit_num==8) timing_value = timing_value >>8;
		    reg_mask = reg_mask | (BIT0 << j);
		    get_bit = (timing_value & (BIT0 << bit_num));
		    data = data | ((get_bit >> shift_next_reg)<< start_index);
		    bit_num ++;
	    }
	    if (io_type == VIACR)
	    {
		    //DEBUG_MSG(KERN_INFO "i=%2d %2d %2d %2d\n", i, cr_index, data, reg_mask);    
	        write_reg_mask(cr_index, VIACR, data, reg_mask);
		
	    }
	    else
	    {
		    //DEBUG_MSG(KERN_INFO "\nSR = %2d", cr_index);    
		    //DEBUG_MSG(KERN_INFO "\ni=%2d %2d %2d %2d", i, cr_index, data, reg_mask);    
	        write_reg_mask(cr_index, VIASR, data, reg_mask);
	    }    

    }

}

/* Write Registers */
static void WriteRegX(struct io_reg RegTable[],int ItemNum)
{
   int               i;
   unsigned char     RegTemp;

   //DEBUG_MSG(KERN_INFO "Table Size : %x!!\n",ItemNum );


   for( i=0; i< ItemNum; i++ ){
       outb( RegTable[i].index, RegTable[i].port );
       RegTemp = inb(RegTable[i].port+1 );
       RegTemp = ( RegTemp & ( ~RegTable[i].mask ) ) | RegTable[i].value ;
       outb( RegTemp, RegTable[i].port+1 );
   }
}

void load_offset_reg(int h_addr, int bpp_byte, int set_iga)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg;
    
    
    switch(set_iga) {
        case IGA1_IGA2:
        case IGA1:
            reg_value = IGA1_OFFSET_FORMULA(h_addr, bpp_byte);
            load_reg_num = offset_reg.iga1_offset_reg.reg_num;
            reg = offset_reg.iga1_offset_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
            if (set_iga == IGA1)
                break;
        case IGA2:
            reg_value = IGA2_OFFSET_FORMULA(h_addr, bpp_byte);
            load_reg_num = offset_reg.iga2_offset_reg.reg_num;
            reg = offset_reg.iga2_offset_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);    
            break;
    }
}  

void load_fetch_count_reg(int h_addr, int bpp_byte, int set_iga)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg=NULL;
    
    switch(set_iga) {
        case IGA1_IGA2:
        case IGA1:    
            reg_value = IGA1_FETCH_COUNT_FORMULA(h_addr, bpp_byte);
            load_reg_num = fetch_count_reg.iga1_fetch_count_reg.reg_num;
            reg = fetch_count_reg.iga1_fetch_count_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIASR);
            if (set_iga == IGA1)            
                break;
        case IGA2:
            reg_value = IGA2_FETCH_COUNT_FORMULA(h_addr, bpp_byte);
            load_reg_num = fetch_count_reg.iga2_fetch_count_reg.reg_num;
            reg = fetch_count_reg.iga2_fetch_count_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
            break;
    }

}


void load_FIFO_reg(int set_iga, int hor_active, int ver_active)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg=NULL;
    int iga1_fifo_max_depth=0, iga1_fifo_threshold=0, iga1_gfx_preg_threshold=0, iga1_display_gueue_expire_num=0;
    int iga2_fifo_max_depth=0, iga2_fifo_threshold=0, iga2_gfx_preg_threshold=0, iga2_display_gueue_expire_num=0;
    
    
    if (set_iga == IGA1)
    {
        if (chip_info.gfx_chip_name == UNICHROME_K800)
        {
            iga1_fifo_max_depth = K800_IGA1_FIFO_MAX_DEPTH;
            iga1_fifo_threshold = K800_IGA1_FIFO_THRESHOLD;
            iga1_gfx_preg_threshold = K800_IGA1_GFX_PREQ_THRESHOLD;
            // If resolution > 1280x1024, expire length = 64, else  expire length = 128
            if ((hor_active > 1280) && (ver_active > 1024))
                iga1_display_gueue_expire_num = 16;
            else
                iga1_display_gueue_expire_num = K800_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
            
        }
        
        if (chip_info.gfx_chip_name == UNICHROME_P880)
        {
            iga1_fifo_max_depth = P880_IGA1_FIFO_MAX_DEPTH;
            iga1_fifo_threshold = P880_IGA1_FIFO_THRESHOLD;
            iga1_gfx_preg_threshold = P880_IGA1_GFX_PREQ_THRESHOLD;
            iga1_display_gueue_expire_num = P880_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
            
            // If resolution > 1280x1024, expire length = 64, else  expire length = 128
            if ((hor_active > 1280) && (ver_active > 1024))
                iga1_display_gueue_expire_num = 16;
            else
                iga1_display_gueue_expire_num = P880_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
        }
                        
        // Set Display FIFO Depath Select
        reg_value = IGA1_FIFO_DEPTH_SELECT_FORMULA(iga1_fifo_max_depth);
        load_reg_num = display_fifo_depth_reg.iga1_fifo_depth_select_reg.reg_num;
        reg = display_fifo_depth_reg.iga1_fifo_depth_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);
        
        // Set Display FIFO Threshold Select
        reg_value = IGA1_FIFO_THRESHOLD_FORMULA(iga1_fifo_threshold);
        load_reg_num = fifo_threshold_select_reg.iga1_fifo_threshold_select_reg.reg_num;
        reg = fifo_threshold_select_reg.iga1_fifo_threshold_select_reg.reg;
	    //DEBUG_MSG(KERN_INFO "\n Display FIFO Threhold Select Value=%2d REG_NUM=%2d",reg_value,load_reg_num); 
        load_reg(reg_value, load_reg_num, reg, VIASR);  
        
        // Set GFX PREG Threshold Select
        reg_value = iga1_gfx_preg_threshold;
        load_reg_num = gfx_preq_thresold_select_reg.iga1_gfx_preq_thresold_select_reg.reg_num;
        reg = gfx_preq_thresold_select_reg.iga1_gfx_preq_thresold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);
        
        // Set Display Gueue Expire Num
        reg_value = IGA1_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga1_display_gueue_expire_num);
        load_reg_num = display_queue_expire_num_reg.iga1_display_queue_expire_num_reg.reg_num;
        reg = display_queue_expire_num_reg.iga1_display_queue_expire_num_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);
         
    }
    else
    {
        if (chip_info.gfx_chip_name == UNICHROME_K800)
        {
            iga2_fifo_max_depth = K800_IGA2_FIFO_MAX_DEPTH;
            iga2_fifo_threshold = K800_IGA2_FIFO_THRESHOLD;
            iga2_gfx_preg_threshold = K800_IGA2_GFX_PREQ_THRESHOLD;
            
            // If resolution > 1280x1024, expire length = 64, else  expire length = 128
            if ((hor_active > 1280) && (ver_active > 1024))
                iga2_display_gueue_expire_num = 16;
            else
                iga2_display_gueue_expire_num = K800_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
        }
        
        if (chip_info.gfx_chip_name == UNICHROME_P880)
        {
            iga2_fifo_max_depth = P880_IGA2_FIFO_MAX_DEPTH;
            iga2_fifo_threshold = P880_IGA2_FIFO_THRESHOLD;
            iga2_gfx_preg_threshold = P880_IGA2_GFX_PREQ_THRESHOLD;
                        
            // If resolution > 1280x1024, expire length = 64, else  expire length = 128
            if ((hor_active > 1280) && (ver_active > 1024))
                iga2_display_gueue_expire_num = 16;
            else
                iga2_display_gueue_expire_num = P880_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
        }
        
        if (chip_info.gfx_chip_name == UNICHROME_K800)
        {
            // Set Display FIFO Depath Select
            reg_value = IGA2_FIFO_DEPTH_SELECT_FORMULA(iga2_fifo_max_depth)-1; // Patch LCD in IGA2 case 
            load_reg_num = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg_num;
            reg = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
        }
        else
        {
        
            // Set Display FIFO Depath Select
            reg_value = IGA2_FIFO_DEPTH_SELECT_FORMULA(iga2_fifo_max_depth);
            load_reg_num = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg_num;
            reg = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
        }    
        
        // Set Display FIFO Threshold Select
        reg_value = IGA2_FIFO_THRESHOLD_FORMULA(iga2_fifo_threshold);
        load_reg_num = fifo_threshold_select_reg.iga2_fifo_threshold_select_reg.reg_num;
        reg = fifo_threshold_select_reg.iga2_fifo_threshold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);  
        
        // Set GFX PREG Threshold Select
        reg_value = iga2_gfx_preg_threshold;
        load_reg_num = gfx_preq_thresold_select_reg.iga2_gfx_preq_thresold_select_reg.reg_num;
        reg = gfx_preq_thresold_select_reg.iga2_gfx_preq_thresold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);
        
        // Set Display Gueue Expire Num
        reg_value = IGA2_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga2_display_gueue_expire_num);
        load_reg_num = display_queue_expire_num_reg.iga2_display_queue_expire_num_reg.reg_num;
        reg = display_queue_expire_num_reg.iga2_display_queue_expire_num_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);
      
    }
    
}

u32 get_clk_value(int clk)
{
   int i;
   
   for(i=0; i<NUM_TOTAL_PLL_TABLE; i++)
   {
        if(clk==pll_value[i].clk)
        {
            switch(chip_info.gfx_chip_name) {    
                case UNICHROME_CLE266:
                case UNICHROME_K400:
                    return(pll_value[i].cle266_pll);    
                break;
                case UNICHROME_K800:
                case UNICHROME_P880:
                    return(pll_value[i].k800_pll);
                break;
             }
        }        
    }            
   DEBUG_MSG(KERN_INFO "Can't find match PLL value\n\n");    
   return(0);
}

// Set VCLK
void SetVCLK( u32 CLK, int set_iga )
{
    unsigned char RegTemp;

    // H.W. Reset : ON
    write_reg_mask(CR17, VIACR, 0x00, BIT7);

    if ((set_iga==IGA1) || (set_iga==IGA1_IGA2))
    {  

        // Change D,N FOR VCLK
	    switch(chip_info.gfx_chip_name) {    
                case UNICHROME_CLE266:
                case UNICHROME_K400:
                    write_reg(SR46, VIASR, CLK/0x100);        
                    write_reg(SR47, VIASR, CLK%0x100);        
                    break;
                case UNICHROME_K800:
                case UNICHROME_P880:
                    write_reg(SR44, VIASR, CLK/0x10000);
		    DEBUG_MSG(KERN_INFO "\nSR44=%x", CLK/0x10000);
                    write_reg(SR45, VIASR, (CLK&0xFFFF)/0x100);
		    DEBUG_MSG(KERN_INFO "\nSR45=%x", (CLK&0xFFFF)/0x100);
                    write_reg(SR46, VIASR, CLK%0x100);
		    DEBUG_MSG(KERN_INFO "\nSR46=%x", CLK%0x100);
                    break;
        }
        

    }
    
    if ((set_iga==IGA2) || (set_iga==IGA1_IGA2))
    {
        // Change D,N FOR LCK
	    switch(chip_info.gfx_chip_name) {    
                case UNICHROME_CLE266:
                case UNICHROME_K400:
                    write_reg(SR44, VIASR, CLK/0x100);        
                    write_reg(SR45, VIASR, CLK%0x100);        
                    break;
                case UNICHROME_K800:
                case UNICHROME_P880:
                    write_reg(SR4A, VIASR, CLK/0x10000);
                    write_reg(SR4B, VIASR, (CLK&0xFFFF)/0x100);
                    write_reg(SR4C, VIASR, CLK%0x100);
                break;
        }
        
    }

    // H.W. Reset : OFF
    write_reg_mask(CR17, VIACR, 0x80, BIT7);

    // Reset PLL
    if ((set_iga==IGA1) || (set_iga==IGA1_IGA2))
    {
        write_reg_mask(SR40, VIASR, 0x02, BIT1);
        write_reg_mask(SR40, VIASR, 0x00, BIT1);
    }
    
     if ((set_iga==IGA2) || (set_iga==IGA1_IGA2))
    {
        write_reg_mask(SR40, VIASR, 0x01, BIT0); 
        write_reg_mask(SR40, VIASR, 0x00, BIT0);
    }

    // Fire!
    RegTemp = inb(VIARMisc);
    outb(RegTemp | (BIT2+BIT3),VIAWMisc);
}

void load_crtc_timing(struct display_timing device_timing, int set_iga)
{
    int i;
    int load_reg_num=0;
    int reg_value=0;
    struct io_register *reg=NULL;
    
    for (i=0;i<12;i++)
    {
        if (set_iga == IGA1)
        {
	        switch(i) {
		        case H_TOTAL_INDEX:
		            reg_value = IGA1_HOR_TOTAL_FORMULA(device_timing.hor_total);
			        load_reg_num = iga1_crtc_reg.hor_total.reg_num;
			        reg = iga1_crtc_reg.hor_total.reg;
			        break;
		        case H_ADDR_INDEX:
		            reg_value = IGA1_HOR_ADDR_FORMULA(device_timing.hor_addr);
			        load_reg_num = iga1_crtc_reg.hor_addr.reg_num;
			        reg = iga1_crtc_reg.hor_addr.reg;
			        break;
		        case H_BLANK_SATRT_INDEX:
		            reg_value = IGA1_HOR_BLANK_START_FORMULA(device_timing.hor_blank_start);
			        load_reg_num = iga1_crtc_reg.hor_blank_start.reg_num;
			        reg = iga1_crtc_reg.hor_blank_start.reg;
			        break;
		        case H_BLANK_END_INDEX:
		            reg_value = IGA1_HOR_BLANK_END_FORMULA(device_timing.hor_blank_start, device_timing.hor_blank_end);
			        load_reg_num = iga1_crtc_reg.hor_blank_end.reg_num;
			        reg = iga1_crtc_reg.hor_blank_end.reg;
			        break;
		        case H_SYNC_SATRT_INDEX:
		            reg_value = IGA1_HOR_SYNC_START_FORMULA(device_timing.hor_sync_start);
			        load_reg_num = iga1_crtc_reg.hor_sync_start.reg_num;
			        reg = iga1_crtc_reg.hor_sync_start.reg;
			        break;
		        case H_SYNC_END_INDEX:
		            reg_value = IGA1_HOR_SYNC_END_FORMULA(device_timing.hor_sync_start, device_timing.hor_sync_end);
			        load_reg_num = iga1_crtc_reg.hor_sync_end.reg_num;
			        reg = iga1_crtc_reg.hor_sync_end.reg;
			        break;
		        case V_TOTAL_INDEX:
		            reg_value = IGA1_VER_TOTAL_FORMULA(device_timing.ver_total);
			        load_reg_num = iga1_crtc_reg.ver_total.reg_num;
			        reg = iga1_crtc_reg.ver_total.reg;
			        break;
		        case V_ADDR_INDEX:
		            reg_value = IGA1_VER_ADDR_FORMULA(device_timing.ver_addr);
			        load_reg_num = iga1_crtc_reg.ver_addr.reg_num;
			        reg = iga1_crtc_reg.ver_addr.reg;
			        break;
		        case V_BLANK_SATRT_INDEX:
		            reg_value = IGA1_VER_BLANK_START_FORMULA(device_timing.ver_blank_start);
			        load_reg_num = iga1_crtc_reg.ver_blank_start.reg_num;
			        reg = iga1_crtc_reg.ver_blank_start.reg;
			        break;
		        case V_BLANK_END_INDEX:
		            reg_value = IGA1_VER_BLANK_END_FORMULA(device_timing.ver_blank_start, device_timing.ver_blank_end);
			        load_reg_num = iga1_crtc_reg.ver_blank_end.reg_num;
			        reg = iga1_crtc_reg.ver_blank_end.reg;
			        break;
		        case V_SYNC_SATRT_INDEX:
		            reg_value = IGA1_VER_SYNC_START_FORMULA(device_timing.ver_sync_start);
			        load_reg_num = iga1_crtc_reg.ver_sync_start.reg_num;
			        reg = iga1_crtc_reg.ver_sync_start.reg;
			        break;
		        case V_SYNC_END_INDEX:
			        reg_value = IGA1_VER_SYNC_END_FORMULA(device_timing.ver_sync_start, device_timing.ver_sync_end);
			        load_reg_num = iga1_crtc_reg.ver_sync_end.reg_num;
			        reg = iga1_crtc_reg.ver_sync_end.reg;
			        break;

	        }
	    }
	    
	    if (set_iga == IGA1_IGA2)
	    {
	        switch(i) {
		        case H_TOTAL_INDEX:
		            reg_value = IGA1_HOR_TOTAL_FORMULA(device_timing.hor_total);
			        load_reg_num = iga1_shadow_crtc_reg.hor_total.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_total.reg;
			        break;
		        case H_ADDR_INDEX:
		            reg_value = IGA1_HOR_ADDR_FORMULA(device_timing.hor_addr);
			        load_reg_num = iga1_shadow_crtc_reg.hor_addr.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_addr.reg;
			        break;
		        case H_BLANK_SATRT_INDEX:
		            reg_value = IGA1_HOR_BLANK_START_FORMULA(device_timing.hor_blank_start);
			        load_reg_num = iga1_shadow_crtc_reg.hor_blank_start.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_blank_start.reg;
			        break;
		        case H_BLANK_END_INDEX:
		            reg_value = IGA1_HOR_BLANK_END_FORMULA(device_timing.hor_blank_start, device_timing.hor_blank_end);
			        load_reg_num = iga1_shadow_crtc_reg.hor_blank_end.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_blank_end.reg;
			        break;
		        case H_SYNC_SATRT_INDEX:
		            reg_value = IGA1_HOR_SYNC_START_FORMULA(device_timing.hor_sync_start);
			        load_reg_num = iga1_shadow_crtc_reg.hor_sync_start.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_sync_start.reg;
			        break;
		        case H_SYNC_END_INDEX:
		            reg_value = IGA1_HOR_SYNC_END_FORMULA(device_timing.hor_sync_start, device_timing.hor_sync_end);
			        load_reg_num = iga1_shadow_crtc_reg.hor_sync_end.reg_num;
			        reg = iga1_shadow_crtc_reg.hor_sync_end.reg;
			        break;
		        case V_TOTAL_INDEX:
		            reg_value = IGA1_VER_TOTAL_FORMULA(device_timing.ver_total);
			        load_reg_num = iga1_shadow_crtc_reg.ver_total.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_total.reg;
			        break;
		        case V_ADDR_INDEX:
		            reg_value = IGA1_VER_ADDR_FORMULA(device_timing.ver_addr);
			        load_reg_num = iga1_shadow_crtc_reg.ver_addr.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_addr.reg;
			        break;
		        case V_BLANK_SATRT_INDEX:
		            reg_value = IGA1_VER_BLANK_START_FORMULA(device_timing.ver_blank_start);
			        load_reg_num = iga1_shadow_crtc_reg.ver_blank_start.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_blank_start.reg;
			        break;
		        case V_BLANK_END_INDEX:
		            reg_value = IGA1_VER_BLANK_END_FORMULA(device_timing.ver_blank_start, device_timing.ver_blank_end);
			        load_reg_num = iga1_shadow_crtc_reg.ver_blank_end.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_blank_end.reg;
			        break;
		        case V_SYNC_SATRT_INDEX:
		            reg_value = IGA1_VER_SYNC_START_FORMULA(device_timing.ver_sync_start);
			        load_reg_num = iga1_shadow_crtc_reg.ver_sync_start.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_sync_start.reg;
			        break;
		        case V_SYNC_END_INDEX:
			        reg_value = IGA1_VER_SYNC_END_FORMULA(device_timing.ver_sync_start, device_timing.ver_sync_end);
			        load_reg_num = iga1_shadow_crtc_reg.ver_sync_end.reg_num;
			        reg = iga1_shadow_crtc_reg.ver_sync_end.reg;
			        break;

	        }       
	    }
	    
	    if (set_iga == IGA2)
	    {
	        switch(i) {
		        case H_TOTAL_INDEX:
		            reg_value = IGA2_HOR_TOTAL_FORMULA(device_timing.hor_total);
			        load_reg_num = iga2_crtc_reg.hor_total.reg_num;
			        reg = iga2_crtc_reg.hor_total.reg;
			        break;
		        case H_ADDR_INDEX:
		            reg_value = IGA2_HOR_ADDR_FORMULA(device_timing.hor_addr);
			        load_reg_num = iga2_crtc_reg.hor_addr.reg_num;
			        reg = iga2_crtc_reg.hor_addr.reg;
			        break;
		        case H_BLANK_SATRT_INDEX:
		            reg_value = IGA2_HOR_BLANK_START_FORMULA(device_timing.hor_blank_start);
			        load_reg_num = iga2_crtc_reg.hor_blank_start.reg_num;
			        reg = iga2_crtc_reg.hor_blank_start.reg;
			        break;
		        case H_BLANK_END_INDEX:
		            reg_value = IGA2_HOR_BLANK_END_FORMULA(device_timing.hor_blank_start, device_timing.hor_blank_end);
			        load_reg_num = iga2_crtc_reg.hor_blank_end.reg_num;
			        reg = iga2_crtc_reg.hor_blank_end.reg;
			        break;
		        case H_SYNC_SATRT_INDEX:
		            reg_value = IGA2_HOR_SYNC_START_FORMULA(device_timing.hor_sync_start);
			        load_reg_num = iga2_crtc_reg.hor_sync_start.reg_num;
			        reg = iga2_crtc_reg.hor_sync_start.reg;
			        break;
		        case H_SYNC_END_INDEX:
		            reg_value = IGA2_HOR_SYNC_END_FORMULA(device_timing.hor_sync_start, device_timing.hor_sync_end);
			        load_reg_num = iga2_crtc_reg.hor_sync_end.reg_num;
			        reg = iga2_crtc_reg.hor_sync_end.reg;
			        break;
		        case V_TOTAL_INDEX:
		            reg_value = IGA2_VER_TOTAL_FORMULA(device_timing.ver_total);
			        load_reg_num = iga2_crtc_reg.ver_total.reg_num;
			        reg = iga2_crtc_reg.ver_total.reg;
			        break;
		        case V_ADDR_INDEX:
		            reg_value = IGA2_VER_ADDR_FORMULA(device_timing.ver_addr);
			        load_reg_num = iga2_crtc_reg.ver_addr.reg_num;
			        reg = iga2_crtc_reg.ver_addr.reg;
			        break;
		        case V_BLANK_SATRT_INDEX:
		            reg_value = IGA2_VER_BLANK_START_FORMULA(device_timing.ver_blank_start);
			        load_reg_num = iga2_crtc_reg.ver_blank_start.reg_num;
			        reg = iga2_crtc_reg.ver_blank_start.reg;
			        break;
		        case V_BLANK_END_INDEX:
		            reg_value = IGA2_VER_BLANK_END_FORMULA(device_timing.ver_blank_start, device_timing.ver_blank_end);
			        load_reg_num = iga2_crtc_reg.ver_blank_end.reg_num;
			        reg = iga2_crtc_reg.ver_blank_end.reg;
			        break;
		        case V_SYNC_SATRT_INDEX:
		            reg_value = IGA2_VER_SYNC_START_FORMULA(device_timing.ver_sync_start);
			        load_reg_num = iga2_crtc_reg.ver_sync_start.reg_num;
			        reg = iga2_crtc_reg.ver_sync_start.reg;
			        break;
		        case V_SYNC_END_INDEX:
			        reg_value = IGA2_VER_SYNC_END_FORMULA(device_timing.ver_sync_start, device_timing.ver_sync_end);
			        load_reg_num = iga2_crtc_reg.ver_sync_end.reg_num;
			        reg = iga2_crtc_reg.ver_sync_end.reg;
			        break;

	        }
	     }
	     load_reg(reg_value, load_reg_num, reg, VIACR);
	}    
}
         
void fill_crtc_timing(struct crt_mode_table *crt_table, int mode_index, int bpp_byte, int set_iga)
{
    struct VideoModeTable *video_mode;
    struct display_timing crt_reg;
    int i;
    int index=0;
    int h_addr,v_addr;
    u32 pll_D_N;


    video_mode = &CLE266Modes[SearchModeSetting(mode_index)];
    
    for(i=0;i<video_mode->mode_array;i++)
    {
       index = i;

       if (crt_table[i].refresh_rate == crt_setting_info.refresh_rate)
       break;
    }
	
    crt_reg = crt_table[index].crtc;
    h_addr = crt_reg.hor_addr;
    v_addr = crt_reg.ver_addr;

   /* DEBUG_MSG(KERN_INFO "===Debug===");
    DEBUG_MSG(KERN_INFO "Index=%d\n", index);
    DEBUG_MSG(KERN_INFO "Refresh_rate=%d\n", crt_table[index].refresh_rate);
    DEBUG_MSG(KERN_INFO "HT=%d\n",crt_reg.hor_total);
    DEBUG_MSG(KERN_INFO "HAdr=%d\n",crt_reg.hor_addr);	
    DEBUG_MSG(KERN_INFO "HBS=%d\n",crt_reg.hor_blank_start);
    DEBUG_MSG(KERN_INFO "HBE=%d\n",crt_reg.hor_blank_end);	
    DEBUG_MSG(KERN_INFO "HSS=%d\n",crt_reg.hor_sync_start);
    DEBUG_MSG(KERN_INFO "HSE=%d\n",crt_reg.hor_sync_end);	

    DEBUG_MSG(KERN_INFO "VT=%d\n",crt_reg.ver_total);
    DEBUG_MSG(KERN_INFO "VAdr=%d\n",crt_reg.ver_addr);	
    DEBUG_MSG(KERN_INFO "VBS=%d\n",crt_reg.ver_blank_start);
    DEBUG_MSG(KERN_INFO "VBE=%d\n",crt_reg.ver_blank_end);	
    DEBUG_MSG(KERN_INFO "VSS=%d\n",crt_reg.ver_sync_start);
    DEBUG_MSG(KERN_INFO "VSE=%d\n",crt_reg.ver_sync_end);	*/
    
    if (set_iga == IGA1)
    {
        unlock_crt(); 	
        write_reg(CR09, VIACR, 0x00); //initial CR09=0 
        // update starting address
        write_reg(CR0C, VIACR, 0x00); //initial starting address
        write_reg(CR0D, VIACR, 0x00);
	    write_reg(CR34, VIACR, 0x00);
        write_reg_mask(CR48, VIACR, 0x00, BIT0+BIT1);
        write_reg_mask(CR11, VIACR, 0x00, BIT4+BIT5+BIT6);
        write_reg_mask(CR17, VIACR, 0x00, BIT7);
    }
    else
    {
        // update starting address
        write_reg(CR62, VIACR, 0x00); 
        write_reg(CR63, VIACR, 0x00);
        write_reg(CR64, VIACR, 0x00);
    }
    
    switch(set_iga) {
        case IGA1:
            load_crtc_timing(crt_reg, IGA1);
            break;
        case IGA2:
            load_crtc_timing(crt_reg, IGA2);
            break;
        case IGA1_IGA2:
            load_crtc_timing(crt_reg, IGA1_IGA2); 
            load_crtc_timing(crt_reg, IGA2);       
            break;
    }    
        
    load_fix_bit_crtc_reg();
    lock_crt();
    write_reg_mask(CR17, VIACR, 0x80, BIT7);
    load_offset_reg(h_addr, bpp_byte, set_iga);
    load_fetch_count_reg(h_addr, bpp_byte, set_iga);
    
    // load FIFO
    if ( (chip_info.gfx_chip_name != UNICHROME_CLE266) && (chip_info.gfx_chip_name != UNICHROME_K400))
        load_FIFO_reg(set_iga, h_addr, v_addr); 
        
    // load SR Register About Memory and Color part
    switch(bpp_byte) {
        case MODE_8BPP:
            write_reg_mask(SR15, VIASR, 0x22, 0xFE);
            break;
        case MODE_16BPP:
            write_reg_mask(SR15, VIASR, 0xB6, 0xFE);
            break;
        case MODE_32BPP:
            write_reg_mask(SR15, VIASR, 0xAE, 0xFE);
            break;
        
    }
    pll_D_N = get_clk_value(crt_table[index].clk);
    DEBUG_MSG( KERN_INFO "PLL=%x", pll_D_N);
    SetVCLK(pll_D_N, set_iga);		

}

void init_chip_info(void)
{
    init_gfx_chip_info(); 
    init_tv_chip_info();
    init_tmds_chip_info();
    init_lvds_chip_info(); 
    
    crt_setting_info.iga_path = IGA1; 
    //crt_setting_info.h_active = 1024;
    //crt_setting_info.v_active = 768;
    //crt_setting_info.bpp = bpp;
    crt_setting_info.refresh_rate = refresh;
    
    
    tv_setting_info.iga_path = IGA1; 
    tv_setting_info.level = tv_level;
    tv_setting_info.system = tv_system;
    tv_setting_info.out_signal = tv_out_signal;
    tv_setting_info.dedotcrawl = tv_dedotcrawl;
    
    tmds_setting_info.iga_path = IGA1; 
    //tmds_setting_info.h_active = 1024;
    //tmds_setting_info.v_active = 768;
    //tmds_setting_info.bpp = bpp;
    //tmds_setting_info.refresh_rate = refresh;
    
    lvds_setting_info.iga_path = IGA1; 
    //tmds_setting_info.h_active = 1024;
    //tmds_setting_info.v_active = 768;
    //tmds_setting_info.bpp = bpp;
    //tmds_setting_info.refresh_rate = refresh;
    
    DEBUG_MSG(KERN_INFO "TV= %2d\n", chip_info.tv_chip_info.tv_chip_name); 
    DEBUG_MSG(KERN_INFO "TV_SLAVE_ADDR= %2d\n", chip_info.tv_chip_info.tv_chip_slave_addr);
}

void update_device_setting(int hres, int vres, int bpp, int vmode_refresh)
{
    crt_setting_info.h_active = hres;
    crt_setting_info.v_active = vres;
    crt_setting_info.bpp = bpp;
    crt_setting_info.refresh_rate = vmode_refresh;
    
    tv_setting_info.h_active = hres;
    tv_setting_info.v_active = vres;
    tv_setting_info.bpp = bpp;    
       
    tmds_setting_info.h_active = hres;
    tmds_setting_info.v_active = vres;
    tmds_setting_info.bpp = bpp;
    tmds_setting_info.refresh_rate = vmode_refresh;
    
    lvds_setting_info.h_active = hres;
    lvds_setting_info.v_active = vres;
    lvds_setting_info.bpp = bpp;
    lvds_setting_info.refresh_rate = vmode_refresh;
}

void init_gfx_chip_info(void)
{
    struct pci_dev *pdev=NULL;
    u8     tmp; 
 
    // Indentify GFX Chip Name
    pdev = (struct pci_dev *)pci_find_device(UNICHROME_CLE266_VID, UNICHROME_CLE266_DID, NULL);
    if (pdev != NULL)
        chip_info.gfx_chip_name = UNICHROME_CLE266;
 
    pdev = (struct pci_dev *)pci_find_device(UNICHROME_K400_VID, UNICHROME_K400_DID, NULL);
    if (pdev != NULL)
        chip_info.gfx_chip_name = UNICHROME_K400;
           
    pdev = (struct pci_dev *)pci_find_device(UNICHROME_K800_VID, UNICHROME_K800_DID, NULL);
    if (pdev != NULL)
        chip_info.gfx_chip_name = UNICHROME_K800; 
    
    pdev = (struct pci_dev *)pci_find_device(UNICHROME_P880_VID, UNICHROME_P880_DID, NULL);
    if (pdev != NULL)
        chip_info.gfx_chip_name = UNICHROME_P880;
    
    
    // Check revision of CLE266 Chip 
    if (chip_info.gfx_chip_name == UNICHROME_CLE266)
    {
        // CR4F only define in CLE266.CX chip    
        tmp = read_reg(VIACR, CR4F);
        write_reg(CR4F, VIACR, 0x55);           
        if (read_reg(VIACR,CR4F)!=0x55)
            chip_info.gfx_chip_revision = CLE266_REVISION_AX;
        else
            chip_info.gfx_chip_revision = CLE266_REVISION_CX;

        write_reg(CR4F, VIACR, tmp);           // restore orignal CR4F value    
    }                      
}

void init_tv_chip_info(void)
{
    u8  data;
    
    tv_encoder_identify();
    data = read_reg(VIASR, SR12) & BIT5; // HW strapping: 0: TV on AMR 1: TV on AGP
    if ((chip_info.gfx_chip_name == UNICHROME_CLE266) || (data == 0x20))
    {
        chip_info.tv_chip_info.tv_chip_on_slot = TV_ON_AMR;
        chip_info.tv_chip_info.output_interface = INTERFACE_DVP0;    
    }
    else
    {
        chip_info.tv_chip_info.tv_chip_on_slot = TV_ON_AGP;
        data = read_reg(VIASR, SR12) & BIT4; // HW strapping: 0: 12-bits DVI 1: 24-bit Panel 
        if (data == 0x10)
            chip_info.tv_chip_info.output_interface = INTERFACE_DFP_LOW;
        else
            chip_info.tv_chip_info.output_interface = INTERFACE_DFP_HIGH;    
    }        
            
}

void init_tmds_chip_info(void)
{
    // set DVP1 default for DVI
    chip_info.tmds_chip_info.output_interface = INTERFACE_DVP1; 
    tmds_trasmitter_identify();
    DEBUG_MSG(KERN_INFO "TMDS Chip = %d\n", chip_info.tmds_chip_info.tmds_chip_name);
    init_dvi_size();
}

void init_lvds_chip_info(void)
{
    lvds_trasmitter_identify();
    DEBUG_MSG(KERN_INFO "TMDS Chip = %d\n", chip_info.tmds_chip_info.tmds_chip_name);
    init_lcd_size();
}

void init_dac(int set_iga)
{
    int i;
    u8 tmp;

    if (set_iga == IGA1)
    {
        write_reg_mask(SR1A, VIASR, 0x00, BIT0);        // access Primary Display's LUT 
        write_reg_mask(SR1B, VIASR, 0x00, BIT7+BIT6);   // turn off LCK
        for(i=0; i<256; i++)
        {
            write_dac_reg(i, palLUT_table[i].red, palLUT_table[i].green, palLUT_table[i].blue);     
        }
        
        write_reg_mask(SR1B, VIASR, 0xC0, BIT7+BIT6);   // turn on LCK
    }
    else
    {
	    tmp=read_reg(VIACR, CR6A);
        write_reg_mask(CR6A, VIACR, 0x40, BIT6); 	
        write_reg_mask(SR1A, VIASR, 0x01, BIT0);        // access Secondary Display's LUT 
        for(i=0; i<256; i++)
        {
            write_dac_reg(i, palLUT_table[i].red, palLUT_table[i].green, palLUT_table[i].blue);     
        }               
       write_reg_mask(SR1A, VIASR, 0x00, BIT0);        // set IGA1 DAC for default
       write_reg(CR6A, VIACR, tmp);	       
    }
}

void device_screen_off(void)
{
    if (CRT_ON==1){
        write_reg_mask(SR01, VIASR, 0x20, BIT5);   // turn off CRT screen (IGA1)    
    }
    if (TV_ON==1){
    }
    if (DVI_ON==1){
    }
    if (LCD_ON==1){
    }
}

void device_screen_on(void)
{
    if (CRT_ON==1){
        write_reg_mask(SR01, VIASR, 0x00, BIT5);   // turn on CRT screen (IGA1)    
    }
    if (TV_ON==1){
    }
    if (DVI_ON==1){
    }
    if (LCD_ON==1){
    }
}

int setmode(int vmode_index,int hor_res,int ver_res,int video_bpp)
{
     int i,j;
     int port;
     u8 value,index,mask;
     struct VideoModeTable      *vmode_tbl;
     struct crt_mode_table	*crt_timing;		

     DEBUG_MSG(KERN_INFO "Set Mode!!\n");
     DEBUG_MSG(KERN_INFO "vmode_index=%d hor_res=%d ver_res=%d video_bpp=%d\n", vmode_index, hor_res, ver_res, video_bpp);
     
     device_screen_off();
     vmode_tbl = &CLE266Modes[SearchModeSetting(vmode_index)];
     crt_timing = vmode_tbl->crtc;  
     inb(VIAStatus);
     outb(0x00,VIAAR);
  /* Write Common Setting for Video Mode */
     WriteRegX( CLE266_ModeXregs,NUM_TOTAL_ModeXregs );

  /* Fill VPIT Parameters */
  // Write Misc Register
     outb(VPIT.Misc,VIAWMisc);
     
  // Write Sequencer
     for ( i=1; i<StdSR ; i++ ) {
       outb(i,VIASR);
       outb(VPIT.SR[i-1],VIASR+1);
     }
  // Write CRTC
     fill_crtc_timing(crt_timing, vmode_index, video_bpp/8,IGA1);
    
  // Write Graphic Controller
     for ( i=0; i<StdGR ; i++ ) {
       outb(i,VIAGR);
       outb(VPIT.GR[i],VIAGR+1);
     }

  // Write Attribute Controller
     for ( i=0; i<StdAR ; i++ ) {
       inb(VIAStatus);
       outb(i,VIAAR);
       outb(VPIT.AR[i],VIAAR);
     }

     inb(VIAStatus);
     outb(0x20,VIAAR);

  /* Update Patch Register */
   
    if ((chip_info.gfx_chip_name == UNICHROME_CLE266) || (chip_info.gfx_chip_name == UNICHROME_K400))
    { 		    
        for( i=0; i< NUM_TOTAL_PATCH_MODE; i++ )
        {
            if (res_patch_table[i].mode_index==vmode_index)
            {
                for(j=0; j<res_patch_table[i].table_length; j++)
                {     
                   index = res_patch_table[i].io_reg_table[j].index;
                   port = res_patch_table[i].io_reg_table[j].port;
                   value = res_patch_table[i].io_reg_table[j].value;
                   mask = res_patch_table[i].io_reg_table[j].mask ; 
                   write_reg_mask(index, port, value, mask);
                }
            }
        }
   }
  
  /* Update Refresh Rate Setting */

  /* Clear On Screen */
  if (TV_ON)
  {
      tv_set_mode(vmode_index);
  } 
        
  if (DVI_ON)
  {
      dvi_set_mode(crt_timing, vmode_index, video_bpp, IGA1);
     
  } 
  
  if (LCD_ON)
  {
      lcd_set_mode(crt_timing, vmode_index, video_bpp, IGA1_IGA2);
  } 

  device_screen_on();
  
  return 1;
}

int get_pixclock(int hres, int vres, int vmode_refresh)
{
  int i;
  
  for(i=0; i<NUM_TOTAL_RES_MAP_REFRESH; i++)
  {
      if((hres==res_map_refresh_tbl[i].hres) && (vres==res_map_refresh_tbl[i].vres) && (vmode_refresh == res_map_refresh_tbl[i].vmode_refresh))
          return(res_map_refresh_tbl[i].pixclock);
  }
  return(RES_640X480_60HZ_PIXCLOCK); 		  
 	
}

int get_refresh(int hres, int vres, int pixclock)
{
  int i;
  for(i=0; i<NUM_TOTAL_RES_MAP_REFRESH; i++)
  {
      if((hres==res_map_refresh_tbl[i].hres) && (vres==res_map_refresh_tbl[i].vres) && (pixclock == res_map_refresh_tbl[i].pixclock))
          return(res_map_refresh_tbl[i].vmode_refresh);
  }
  return(60); 
}


    


