/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/locks.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <asm/uaccess.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_superblock.h>
#include <linux/jfs/jfs_dmap.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_logmgr.h>
#include <linux/jfs/jfs_debug.h>

pid_t jfsIOthread;
pid_t jfsCommitThread;
struct task_struct *jfsIOtask;
struct task_struct *jfsCommitTask;
DECLARE_MUTEX_LOCKED(jfsIOsem);

#ifdef _JFS_DEBUG
int32 jfsFYI=0;
int32 jfsERROR=1;
#endif

/*
 * External declarations
 */
int32 jfs_mount(struct super_block *, char *);
int32 jfs_umount(struct super_block *);

extern int jfsIOWait(void *);
extern int jfs_lazycommit(void *);
extern void jfs_put_inode(struct inode *inode);
extern void jfs_read_inode(struct inode * inode);
extern void jfs_delete_inode(struct inode * inode);
#ifdef kern22
extern void jfs_write_inode(struct inode * inode);
#else
extern void jfs_write_inode(struct inode * inode, int wait);
#endif

#ifndef kern22
#if CONFIG_PROC_FS
void jfs_proc_init(void);
void jfs_proc_clean(void);
#endif
#endif
#ifdef kern22
static int jfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
	struct statfs tmp;
	imap_t *imap = sb->s_jfs_ipimap->i_jfs_imap;

	jFYI(1,("In jfs_statfs\n"));
	tmp.f_type = JFS_SUPER_MAGIC;
	tmp.f_bsize = sb->s_jfs_bsize;
	tmp.f_blocks = sb->s_jfs_bmap->db_mapsize;
	tmp.f_bfree = sb->s_jfs_bmap->db_nfree;
	tmp.f_bavail = sb->s_jfs_bmap->db_nfree;
	tmp.f_files = atomic_read(&imap->im_numinos);
	tmp.f_ffree = atomic_read(&imap->im_numfree);
	tmp.f_namelen = JFS_NAME_MAX;
	return copy_to_user(buf, &tmp, bufsiz) ? -EFAULT : 0;
}
#else /* ! kern22 */
static int jfs_statfs(struct super_block *sb, struct statfs *buf)
{
	imap_t *imap = sb->s_jfs_ipimap->i_jfs_imap;

	jFYI(1,("In jfs_statfs\n"));
	buf->f_type = JFS_SUPER_MAGIC;
	buf->f_bsize = sb->s_jfs_bsize;
	buf->f_blocks = sb->s_jfs_bmap->db_mapsize;
	buf->f_bfree = sb->s_jfs_bmap->db_nfree;
	buf->f_bavail = sb->s_jfs_bmap->db_nfree;
	buf->f_files = atomic_read(&imap->im_numinos);
	buf->f_ffree = atomic_read(&imap->im_numfree);
	buf->f_namelen = JFS_NAME_MAX;
	return 0;
}
#endif /* kern22 */

static void jfs_put_super(struct super_block *sb)
{
	int32 rc;

	jFYI(1,("In jfs_put_super\n"));
	rc = jfs_umount(sb);
	if (rc)
	{
		jERROR(1,("jfs_umount failed with return code %d\n", rc));
	}
	unload_nls(sb->s_jfs_nls_tab);
	sb->s_jfs_nls_tab = NULL;

#ifdef kern22
	truncate_inode_pages(sb->s_jfs_direct_inode, 0);
	iput(sb->s_jfs_direct_inode);
	sb->s_jfs_direct_inode = NULL;
	MOD_DEC_USE_COUNT;
#else
	/*
	 * We need to clean out the direct_inode pages since this inode
	 * is not in the inode hash.
	 */
	 fsync_inode_buffers(sb->s_jfs_direct_inode);
#ifdef TRUNC_NO_TOSS
	/* We are building on top of XFS changes */
	truncate_inode_pages(sb->s_jfs_direct_mapping, 0, TRUNC_NO_TOSS);
#else
	truncate_inode_pages(sb->s_jfs_direct_mapping, 0);
#endif
	iput(sb->s_jfs_direct_inode);
	sb->s_jfs_direct_inode = NULL;
	sb->s_jfs_direct_mapping = NULL;
#endif
}

static struct super_operations jfs_sops = {
	read_inode:	jfs_read_inode,
	write_inode:	jfs_write_inode,
	put_inode:	jfs_put_inode,
	delete_inode:	jfs_delete_inode,
	put_super:	jfs_put_super,
	statfs:		jfs_statfs,
	clear_inode:	diClearExtension,
};

#ifndef kern22
static
#endif
struct super_block *jfs_read_super(struct super_block *sb, void *data, 
					  int silent)
{
	struct inode	*inode;
	int32		rc;

jFYI(1,("In jfs_read_super s_dev=0x%x s_flags=0x%lx\n", sb->s_dev, sb->s_flags));
#ifdef kern22
	MOD_INC_USE_COUNT;
	lock_super(sb);
#endif
	/*
	 * Initialize direct-mapping inode/address-space
	 */
	inode = get_empty_inode();
	if (inode == NULL)
		return NULL;
	inode->i_sb = sb;
	inode->i_ino = 0;
	inode->i_nlink = 1;
	inode->i_dev = sb->s_dev;
#ifdef kern22
	inode->i_size = 0xff000000L;
	sb->s_jfs_direct_inode = inode;
#else
	inode->i_size = 0x0000010000000000L;
	inode->i_mapping->a_ops = &direct_aops;
	sb->s_jfs_direct_inode = inode;
	sb->s_jfs_direct_mapping = inode->i_mapping;
#endif
	rc = jfs_mount(sb, data);
	if (rc)
	{
		if (!silent)
			jERROR(1,("jfs_mount failed w/return code = %d\n", rc));
		goto out_mount_failed;
	}

	sb->s_magic = JFS_SUPER_MAGIC;
	sb->s_op = &jfs_sops;

	inode = iget(sb, ROOT_I);
	if (!inode)
		goto out_no_root;
#ifdef kern22
	sb->s_root = d_alloc_root(inode, 0);
#else
	sb->s_root = d_alloc_root(inode);
#endif
	if (!sb->s_root)
		goto out_no_root;

	sb->s_jfs_nls_tab = load_nls_default();

#ifdef kern22
	unlock_super(sb);
#endif
	return sb;

out_no_root:
	jEVENT(1,("jfs_read_super: get root inode failed\n"));
	if (inode)
		iput(inode);

	rc = jfs_umount(sb);
	if (rc)
	{
		jERROR(1,("jfs_umount failed with return code %d\n", rc));
	}
out_mount_failed:
#ifdef kern22
	truncate_inode_pages(sb->s_jfs_direct_inode, 0);
	iput(sb->s_jfs_direct_inode);
	sb->s_jfs_direct_inode = NULL;
	unlock_super(sb);
	MOD_DEC_USE_COUNT;
	sb->s_dev = 0;
#else
	 fsync_inode_buffers(sb->s_jfs_direct_inode);
#ifdef TRUNC_NO_TOSS
	/* We are building on top of XFS changes */
	truncate_inode_pages(sb->s_jfs_direct_mapping, 0, TRUNC_NO_TOSS);
#else
	truncate_inode_pages(sb->s_jfs_direct_mapping, 0);
#endif
	iput(sb->s_jfs_direct_inode);
	sb->s_jfs_direct_inode = NULL;
	sb->s_jfs_direct_mapping = NULL;
#endif
	return 0;
}

#ifdef kern22
static struct file_system_type jfs_fs_type = {
	"jfs", FS_REQUIRES_DEV, jfs_read_super, 0
};
#else
static DECLARE_FSTYPE_DEV(jfs_fs_type, "jfs", jfs_read_super);

#endif

spinlock_t jfs_rwlock_lock;
spinlock_t jfs_mutex_lock;
extern int metapage_init(void);
extern int txInit(void);
extern void txExit(void);
extern void metapage_exit(void);

#ifdef kern22
int init_jfs_fs(void)
#else
static int __init init_jfs_fs(void)
#endif
{
	struct inode dummy_inode;
	struct super_block dummy_super;
	int rc;

	printk("JFS development version: $Name: v0_2_0 $\n");

	jFYI(1,("fs dependent super_block size : %d bytes\n",
		sizeof(dummy_super.u)));
	jFYI(1,("jfs specific super_block size : %d bytes\n",
		sizeof(dummy_super.u.jfs_sb)));
	jFYI(1,("ext2 specific super_block size : %d bytes\n",
		sizeof(dummy_super.u.ext2_sb)));
	jFYI(1,("fs dependent inode size : %d bytes\n", sizeof(dummy_inode.u)));
	jFYI(1,("jfs specific inode size : %d bytes\n",
		sizeof(dummy_inode.u.jfs_i)));
	jFYI(1,("ext2 specific inode size : %d bytes\n",
		sizeof(dummy_inode.u.ext2_i)));

	/*
	 * initialize global variables
	 */
	spin_lock_init(&jfs_mutex_lock);
	spin_lock_init(&jfs_rwlock_lock);

	/*
	 * Metapage initialization
	 */
	rc = metapage_init();
	if (rc) {
		jERROR(1,("metapage_init failed w/rc = %d\n", rc));
		return -rc;
	}

	/*
	 * Log Manager initialization
	 */
	rc = lmInit();
	if (rc) {
		jERROR(1,("lmInit failed w/rc = %d\n", rc));
		goto unwind1;
	}

	/*
	 * Transaction Manager initialization
	 */
	rc = txInit();
	if (rc) {
		jERROR(1,("txInit failed w/rc = %d\n", rc));
		goto unwind1;
	}

	/*
	 * I/O completion thread (endio)
	 */
	jfsIOthread = kernel_thread(jfsIOWait, 0,
				    CLONE_FS|CLONE_FILES|CLONE_SIGHAND);
	if (jfsIOthread < 0) {
		jERROR(1,("init_jfs_fs: fork failed w/rc = %d\n", jfsIOthread));
		goto unwind2;
	}
	down(&jfsIOsem);	/* Wait until IO thread starts */

#ifdef _JFS_LAZYCOMMIT
	jfsCommitThread = kernel_thread(jfs_lazycommit, 0,
					CLONE_FS|CLONE_FILES|CLONE_SIGHAND);
	if (jfsCommitThread < 0) {
		jERROR(1,("init_jfs_fs: fork failed w/rc = %d\n", jfsCommitThread));
		goto unwind3;
	}
	down(&jfsIOsem);	/* Wait until IO thread starts */
#endif

#ifndef kern22
#if CONFIG_PROC_FS
	jfs_proc_init();
#endif
#endif
	return register_filesystem(&jfs_fs_type);

#ifdef _JFS_LAZYCOMMIT
unwind3:
	send_sig(SIGKILL, jfsIOtask, 1);
	down(&jfsIOsem);	/* Wait until IO thread exits */
#endif
unwind2:
	txExit();
unwind1:
	metapage_exit();
	return -rc;
}

#ifdef kern22

#ifdef MODULE
int init_module(void)
{
	return init_jfs_fs();
}

void cleanup_module(void)
{
	jFYI(1,("cleanup_module called\n"));
	txExit();
	metapage_exit();
	send_sig(SIGKILL, jfsIOtask, 1);
	down(&jfsIOsem);	/* Wait until IO thread exits */
#ifdef _JFS_LAZYCOMMIT
	send_sig(SIGKILL, jfsCommitTask, 1);
	down(&jfsIOsem);	/* Wait until Commit thread exits */
#endif
        unregister_filesystem(&jfs_fs_type);
}
#endif /* MODULE */

#else /*  > kern22 */

static void __exit exit_jfs_fs(void)
{
	jFYI(1,("exit_jfs_fs called\n"));
	txExit();
	metapage_exit();
	send_sig(SIGKILL, jfsIOtask, 1);
	down(&jfsIOsem);	/* Wait until IO thread exits */
#ifdef _JFS_LAZYCOMMIT
	send_sig(SIGKILL, jfsCommitTask, 1);
	down(&jfsIOsem);	/* Wait until Commit thread exits */
#endif
#if CONFIG_PROC_FS
	jfs_proc_clean();
#endif
        unregister_filesystem(&jfs_fs_type);
}

/* EXPORT_NO_SYMBOLS; */

module_init(init_jfs_fs)
module_exit(exit_jfs_fs)

#endif /* kern22 */
