/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: jfs/namei.c
 *
 */

/*
 * Change History :
 *
 */

#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/malloc.h>
#include <linux/jfs/jfs_dtree.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_inode.h>
#include <linux/jfs/jfs_dinode.h>
#include <linux/jfs/jfs_dmap.h>
#include <linux/jfs/jfs_unicode.h>
#include <linux/jfs/jfs_metapage.h>
#include <linux/jfs/jfs_debug.h>

#ifdef _JFS_OS2
#include "jfs_ea.h"

int jfs_xrmdir(struct inode *, struct inode *, UniChar *);
int jfs_CopyACL(inode_t *, inode_t *, int32);

#endif				/* _JFS_OS2 */

extern struct inode_operations jfs_file_inode_operations;
extern struct inode_operations jfs_dir_inode_operations;
extern struct inode_operations jfs_symlink_inode_operations;
extern struct file_operations jfs_file_operations;
extern struct file_operations jfs_dir_operations;
extern struct address_space_operations jfs_aops;

/*
 * forward references
 */
int commitZeroLink(int32, struct inode *);

/*
 * NAME:	jfs_create(dip, dentry, mode)
 *
 * FUNCTION:	create a regular file in the parent directory <dip>
 *		with name = <from dentry> and mode = <mode>
 *
 * PARAMETER:	dip 	- parent directory vnode
 *		dentry	- dentry of new file
 *		mode	- create mode (rwxrwxrwx).
 *
 * RETURN:	Errors from subroutines
 *
 */
int jfs_create(struct inode *dip, struct dentry *dentry, int mode)
{
	const char *name = (char *) dentry->d_name.name;
	int32 rc = 0;
	int32 tid = 0;			/* transaction id */
	struct inode *ip = NULL;	/* child directory inode */
	ino_t ino;
	component_t dname;		/* child directory name */
	btstack_t btstack;
	struct inode *iplist[2];
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

	jFYI(1, ("jfs_create: dip:0x%p name:%s\n", dip, name));

	/*
	 * the named file exists for "." or ".."
	 */
	if (name[0] == '.') {
		if ((name[1] == '.' && name[2] == '\0') || name[1] == '\0')
			return EEXIST;
	}

	IWRITE_LOCK(dip);

	if (dip->i_nlink == 0) {
		rc = ENOENT;
		goto out1;
	}

	/*
	 * search parent directory for entry/freespace
	 * (dtSearch() returns parent directory page pinned)
	 */
	if ((rc = get_UCSname(&dname, dentry, dip->i_sb->s_jfs_nls_tab)))
		goto out1;

#ifdef _I_DONT_THINK_WE_NEED_THIS
	if (isReadOnly(dip)) {
		rc = EROFS;
		goto out2;
	}
#endif

	/*
	 * Either iAlloc() or txBegin() may block.  Deadlock can occur if we
	 * block there while holding dtree page, so we allocate the inode &
	 * begin the transaction before we search the directory.
	 */
	ip = ialloc(dip, mode);
	if (ip == NULL)
		goto out2;

	IWRITE_LOCK(ip);

	txBegin(dip->i_sb, &tid, 0);

	if ((rc = dtSearch(dip, &dname, &ino, &btstack, JFS_CREATE))) {
		jERROR(1, ("jfs_create: dtSearch returned %d\n", rc));
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);
		txEnd(tid);
		goto out2;
	}

#ifdef _JFS_LAZYCOMMIT
	tblk = &TxBlock[tid];
	tblk->xflag |= COMMIT_CREATE;
	tblk->ip = ip;
#endif

	iplist[0] = dip;
	iplist[1] = ip;

	/*
	 * initialize the child XAD tree root in-line in inode
	 */
	xtInitRoot(tid, ip);

	/*
	 * create entry in parent directory for child directory
	 * (dtInsert() releases parent directory page)
	 */
	ino = ip->i_ino;
	if ((rc = dtInsert(tid, dip, &dname, &ino, &btstack))) {
		jERROR(1, ("jfs_create: dtInsert returned %d\n", rc));
		/* discard new inode */
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);

		if (rc == EIO)
			txAbort(tid, 1);	/* Marks Filesystem dirty */
		else
			txAbort(tid, 0);	/* Filesystem full */
		txEnd(tid);
		goto out2;
	}

	ip->i_op = &jfs_file_inode_operations;
#ifndef kern22
	ip->i_fop = &jfs_file_operations;
	ip->i_mapping->a_ops = &jfs_aops;
#endif
#ifdef _STILL_TO_PORT
	imark(ip, IACC | ICHG | IUPD | IFSYNC);
#endif				/*  _STILL_TO_PORT */
	mark_inode_dirty(ip);
	d_instantiate(dentry, ip);

	dip->i_version = ++event;
	dip->i_ctime = dip->i_mtime = CURRENT_TIME;
#ifdef _STILL_TO_PORT
	imark(dip, ICHG | IUPD | IFSYNC);
#endif				/*  _STILL_TO_PORT */
	mark_inode_dirty(dip);

	rc = txCommit(tid, 2, &iplist[0], 0);
	txEnd(tid);

	IWRITE_UNLOCK(ip);
      out2:
	free_UCSname(&dname);

      out1:

	IWRITE_UNLOCK(dip);
	jFYI(1, ("jfs_create: rc:%d\n", -rc));
	return -rc;
}


/*
 * NAME:	jfs_mkdir(dip, dentry, mode)
 *
 * FUNCTION:	create a child directory in the parent directory <dip>
 *		with name = <from dentry> and mode = <mode>
 *
 * PARAMETER:	dip 	- parent directory vnode
 *		dentry	- dentry of child directory
 *		mode	- create mode (rwxrwxrwx).
 *
 * RETURN:	Errors from subroutines
 *
 * note:
 * EACCESS: user needs search+write permission on the parent directory
 */
int jfs_mkdir(struct inode *dip, struct dentry *dentry, int mode)
{
	const char *name = (char *) dentry->d_name.name;
	int32 rc = 0;
	int32 tid = 0;		/* transaction id */
#ifdef _JFS_OS2
	int32 rc1 = 0;
#endif
	struct inode *ip = NULL;	/* child directory inode */
	ino_t ino;
	component_t dname;	/* child directory name */
	btstack_t btstack;
	struct inode *iplist[2];
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

	jFYI(1, ("jfs_mkdir: dip:0x%p name:%s\n", dip, name));

	/*
	 * the named file exists for "." or ".."
	 */
	if (name[0] == '.') {
		if ((name[1] == '.' && name[2] == '\0') || name[1] == '\0')
			return EEXIST;
	}

	IWRITE_LOCK(dip);

	if (dip->i_nlink == 0) {
		rc = ENOENT;
		goto out1;
	}
#ifdef _I_DONT_THINK_WE_NEED_A_LIMIT
	/* link count overflow on parent directory ? */
	if (dip->i_nlink >= LINK_MAX) {
		rc = EMLINK;
		goto out1;
	}
#endif				/*  _I_DONT_THINK_WE_NEED_A_LIMIT */

	/*
	 * search parent directory for entry/freespace
	 * (dtSearch() returns parent directory page pinned)
	 */
	if ((rc = get_UCSname(&dname, dentry, dip->i_sb->s_jfs_nls_tab)))
		goto out1;

#ifdef _I_DONT_THINK_WE_NEED_THIS
	if (isReadOnly(dip)) {
		rc = EROFS;
		goto out2;
	}
#endif

	/*
	 * Either iAlloc() or txBegin() may block.  Deadlock can occur if we
	 * block there while holding dtree page, so we allocate the inode &
	 * begin the transaction before we search the directory.
	 */
	ip = ialloc(dip, S_IFDIR | mode);
	if (ip == NULL)
		goto out2;

	IWRITE_LOCK(ip);

	txBegin(dip->i_sb, &tid, 0);

	if ((rc = dtSearch(dip, &dname, &ino, &btstack, JFS_CREATE))) {
		jERROR(1, ("jfs_mkdir: dtSearch returned %d\n", rc));
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);
		txEnd(tid);
		goto out2;
	}

#ifdef _JFS_LAZYCOMMIT
	tblk = &TxBlock[tid];
	tblk->xflag |= COMMIT_CREATE;
	tblk->ip = ip;
#endif

	iplist[0] = dip;
	iplist[1] = ip;

	/*
	 * initialize the child directory in-line in inode
	 */
	dtInitRoot(tid, ip, dip->i_ino);

	/*
	 * create entry in parent directory for child directory
	 * (dtInsert() releases parent directory page)
	 */
	ino = ip->i_ino;
	if ((rc = dtInsert(tid, dip, &dname, &ino, &btstack))) {
		jERROR(1, ("jfs_mkdir: dtInsert returned %d\n", rc));
		/* discard new directory inode */
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);

		if (rc == EIO)
			txAbort(tid, 1);	/* Marks Filesystem dirty */
		else
			txAbort(tid, 0);	/* Filesystem full */
		txEnd(tid);
		goto out2;
	}

	ip->i_nlink = 2;	/* for '.' */
	ip->i_op = &jfs_dir_inode_operations;
#ifndef kern22
	ip->i_fop = &jfs_dir_operations;
#endif
	ip->i_jfs_flag |= IFSYNC;
	mark_inode_dirty(ip);
	d_instantiate(dentry, ip);

	/* update parent directory inode */
	dip->i_nlink++;		/* for '..' from child directory */
	dip->i_version = ++event;
	dip->i_ctime = dip->i_mtime = CURRENT_TIME;
	dip->i_jfs_flag |= IFSYNC;
	mark_inode_dirty(dip);

#ifdef _JFS_OS2

	if ((rc1 == 0) && (dip->i_acl.flag) && (dip->i_acl.size))
		rc1 = jfs_CopyACL(ip, dip, tid);

#endif				/* _JFS_OS2 */

	rc = txCommit(tid, 2, &iplist[0], 0);
	txEnd(tid);

	IWRITE_UNLOCK(ip);

#ifdef _JFS_OS2
	/* If we successfully added the directory, but failed adding the EA or
	 * ACL, we must cleanup the created directory entry and return the
	 * error.
	 */
	if (rc1) {
		jfs_xrmdir(dip, ip, name);
		rc = rc1;
	}
#endif				/* _JFS_OS2 */

      out2:
	free_UCSname(&dname);

      out1:

	IWRITE_UNLOCK(dip);

	jFYI(1, ("jfs_mkdir: rc:%d\n", -rc));
	return -rc;
}

/*
 * NAME:	jfs_rmdir(dip, dentry)
 *
 * FUNCTION:	remove a link to child directory
 *
 * PARAMETER:	dip 	- parent inode
 *		dentry	- child directory dentry
 *
 * RETURN:	EINVAL	- if name is . or ..
 *		EINVAL  - if . or .. exist but are invalid.
 *		errors from subroutines
 *
 * note:
 * if other threads have the directory open when the last link 
 * is removed, the "." and ".." entries, if present, are removed before 
 * rmdir() returns and no new entries may be created in the directory, 
 * but the directory is not removed until the last reference to 
 * the directory is released (cf.unlink() of regular file).
 */
int jfs_rmdir(struct inode *dip, struct dentry *dentry)
{
	int32 rc;
	int32 tid = 0;		/* transaction id */
	struct inode *ip = dentry->d_inode;
	ino_t ino;
	const char *name = dentry->d_name.name;
	component_t dname;
	struct inode *iplist[2];
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

	jFYI(1, ("jfs_rmdir: dip:0x%p name:%s\n", dip, name));

	if (!S_ISDIR(ip->i_mode))
		return -ENOTDIR;

	/* prevent suicide and parenticide */
	if (name[0] == '.') {
		/* removing ".." */
		if (name[1] == '.' && name[2] == '\0')
			return -EEXIST;
		/* removing "." */
		else if (name[1] == '\0')
			return -EINVAL;
	}
#ifdef _STILL_TO_PORT
	if (isReadOnly(ip))
		return -EROFS;
#endif				/*  _STILL_TO_PORT */

	IWRITE_LOCK_LIST(2, dip, ip);

	/* directory must be empty to be removed */
	if (!dtEmpty(ip)) {
		IWRITE_UNLOCK(ip);
		IWRITE_UNLOCK(dip);
		rc = ENOTEMPTY;
		goto out;
	}

	if ((rc = get_UCSname(&dname, dentry, dip->i_sb->s_jfs_nls_tab))) {
		IWRITE_UNLOCK(ip);
		IWRITE_UNLOCK(dip);
		goto out;
	}

	txBegin(dip->i_sb, &tid, 0);

	iplist[0] = dip;
	iplist[1] = ip;

#ifdef _JFS_LAZYCOMMIT
	tblk = &TxBlock[tid];
	tblk->xflag |= COMMIT_DELETE;
	tblk->ip = ip;
#endif

	/*
	 * delete the entry of target directory from parent directory
	 */
	ino = ip->i_ino;
	if ((rc = dtDelete(tid, dip, &dname, &ino, JFS_REMOVE))) {
		jERROR(1, ("jfs_rmdir: dtDelete returned %d\n", rc));
		if (rc == EIO)
			txAbort(tid, 1);
		txEnd(tid);

		IWRITE_UNLOCK(ip);
		IWRITE_UNLOCK(dip);

		goto out2;
	}

	/* update parent directory's link count corresponding
	 * to ".." entry of the target directory deleted
	 */
	dip->i_nlink--;
	dip->i_ctime = dip->i_mtime = CURRENT_TIME;
	dip->i_version = ++event;
	mark_inode_dirty(dip);
#ifdef _STILL_TO_PORT
	imark(dip, ICHG | IUPD | IFSYNC);
#endif				/*  _STILL_TO_PORT */

	/*
	 * OS/2 could have created EA and/or ACL
	 */
	/* free EA from both persistent and working map */
	if (ip->i_jfs_ea.flag & DXD_EXTENT) {
		/* free EA pages */
		txEA(tid, ip, &ip->i_jfs_ea, NULL);
	}
	ip->i_jfs_ea.flag = 0;

	/* free ACL from both persistent and working map */
	if (ip->i_jfs_acl.flag & DXD_EXTENT) {
		/* free ACL pages */
		txEA(tid, ip, &ip->i_jfs_acl, NULL);
	}
	ip->i_jfs_acl.flag = 0;

	/* mark the target directory as deleted */
	ip->i_nlink = 0;
	mark_inode_dirty(ip);
#ifdef _STILL_TO_PORT
	imark(ip, ICHG | IFSYNC);
#endif				/*  _STILL_TO_PORT */

	rc = txCommit(tid, 2, &iplist[0], 0);

	txEnd(tid);

	IWRITE_UNLOCK(ip);
	IWRITE_UNLOCK(dip);

	d_delete(dentry);

      out2:
	free_UCSname(&dname);

      out:
	jFYI(1, ("jfs_rmdir: rc:%d\n", rc));
	return -rc;
}

/*
 * NAME:	jfs_unlink(dip, dentry)
 *
 * FUNCTION:	remove a link to object <vp> named by <name> 
 *		from parent directory <dvp>
 *
 * PARAMETER:	dip 	- inode of parent directory
 *		dentry 	- dentry of object to be removed
 *
 * RETURN:	errors from subroutines
 *
 * note:
 * temporary file: if one or more processes have the file open
 * when the last link is removed, the link will be removed before
 * unlink() returns, but the removal of the file contents will be
 * postponed until all references to the files are closed.
 *
 * JFS does NOT support unlink() on directories.
 *
 */
int jfs_unlink(struct inode *dip, struct dentry *dentry)
{
	const char *name = dentry->d_name.name;
	int32 rc;
	int32 tid = 0;		/* transaction id */
	struct inode *ip = dentry->d_inode;
	ino_t ino;
	component_t dname;	/* object name */
	struct inode *iplist[2];
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

	jFYI(1, ("jfs_unlink: dip:0x%p name:%s\n", dip, name));

	/* JFS does NOT support unlink() for directory */
	if (S_ISDIR(ip->i_mode))
		return EPERM;

#ifdef _STILL_TO_PORT
	/* Must be able to write to device */
	if (isReadOnly(ip))
		return EROFS;
#endif				/*  _STILL_TO_PORT */

	if ((rc = get_UCSname(&dname, dentry, dip->i_sb->s_jfs_nls_tab)))
		goto out;

	IWRITE_LOCK_LIST(2, ip, dip);

	txBegin(dip->i_sb, &tid, 0);

	iplist[0] = dip;
	iplist[1] = ip;

	/*
	 * delete the entry of target file from parent directory
	 */
	ino = ip->i_ino;
	if ((rc = dtDelete(tid, dip, &dname, &ino, JFS_REMOVE))) {
		jERROR(1, ("jfs_unlink: dtDelete returned %d\n", rc));
		if (rc == EIO)
			txAbort(tid, 1);	/* Marks FS Dirty */
		txEnd(tid);
		IWRITE_UNLOCK(ip);
		IWRITE_UNLOCK(dip);
		goto out1;
	}

	ASSERT(ip->i_nlink);

	ip->i_ctime = dip->i_ctime = dip->i_mtime = CURRENT_TIME;
	dip->i_version = ++event;
	mark_inode_dirty(dip);

	/* update target's inode */
	ip->i_nlink--;
	mark_inode_dirty(ip);

	/*
	 *      commit zero link count object
	 */
	if (ip->i_nlink == 0) {
		assert(!(ip->i_jfs_cflag & COMMIT_NOLINK));
		/* free block resources */
		if ((rc = commitZeroLink(tid, ip))) {
			txAbort(tid, 1);	/* Marks FS Dirty */
			txEnd(tid);
			IWRITE_UNLOCK(ip);
			IWRITE_UNLOCK(dip);
			goto out1;
		}
#ifdef _JFS_LAZYCOMMIT
		tblk = &TxBlock[tid];
		tblk->xflag |= COMMIT_DELETE;
		tblk->ip = ip;
#endif
	}

	rc = txCommit(tid, 2, &iplist[0], 0);
	txEnd(tid);
	IWRITE_UNLOCK(ip);
	IWRITE_UNLOCK(dip);

	d_delete(dentry);

      out1:
	free_UCSname(&dname);
      out:
	jFYI(1, ("jfs_unlink: rc:%d\n", rc));
	return rc;
}

/*
 * NAME:	commitZeroLink()
 *
 * FUNCTION:    for non-directory, called by jfs_remove(),
 *		truncate a regular file, directory or symbolic
 *		link to zero length. return 0 if type is not 
 *		one of these.
 *
 *		if the file is currently associated with a VM segment
 *		only permanent disk and inode map resources are freed,
 *		and neither the inode nor indirect blocks are modified
 *		so that the resources can be later freed in the work
 *		map by ctrunc1.
 *		if there is no VM segment on entry, the resources are
 *		freed in both work and permanent map.
 *		(? for temporary file - memory object is cached even 
 *		after no reference:
 *		reference count > 0 -   )
 *
 * PARAMETERS:	cd	- pointer to commit data structure.
 *			  current inode is the one to truncate.
 *
 * RETURN :	Errors from subroutines
 */
int commitZeroLink(int32 tid, struct inode * ip)
{
	int rc = 0;
	int filetype, committype;
	tblock_t *tblk;

	jFYI(1,("commitZeroLink: tid = %d, ip = 0x%p\n", tid, ip));
	/* set the IFSYNC flag to force iwrite() to update inode */
	ip->i_jfs_flag |= IFSYNC;

	filetype = ip->i_mode & S_IFMT;
	switch (filetype) {
	case S_IFREG:
		break;
	case S_IFLNK:
		/* fast symbolic link */
		if (ip->i_size <= 256) {
			ip->i_size = 0;
			return 0;
		}
		break;
	default:
		assert(filetype != S_IFDIR);
		return 0;
	}

#ifdef _STILL_TO_PORT
	/*
	 *      free from block allocation map:
	 *
	 * if there is no cache control element associated with 
	 * the file, free resources in both persistent and work map;
	 * otherwise just persistent map. 
	 */
	if (ip->i_cacheid) {
		committype = COMMIT_PMAP;

		/* mark for iClose() to free from working map */
		ip->i_cflag |= COMMIT_FREEWMAP;
	} else
		committype = COMMIT_PWMAP;
#else /* _STILL_TO_PORT */
	ip->i_jfs_cflag |= COMMIT_FREEWMAP;
	committype = COMMIT_PMAP;
#endif /* _STILL_TO_PORT */

	/* mark transaction of block map update type */
	tblk = &TxBlock[tid];
	tblk->xflag |= committype;

	/*
	 * free EA
	 */
	if (ip->i_jfs_ea.flag & DXD_EXTENT) {
#ifdef _STILL_TO_PORT
		/* free EA pages from cache */
		if (committype == COMMIT_PWMAP)
			bmExtentInvalidate(ip, addressDXD(&ip->i_ea),
					   lengthDXD(&ip->i_ea));
#endif /* _STILL_TO_PORT */

		/* acquire maplock on EA to be freed from block map */
		txEA(tid, ip, &ip->i_jfs_ea, NULL);

		if (committype == COMMIT_PWMAP)
			ip->i_jfs_ea.flag = 0;
	}

	/*
	 * free ACL
	 */
	if (ip->i_jfs_acl.flag & DXD_EXTENT) {
#ifdef _STILL_TO_PORT
		/* free ACL pages from cache */
		if (committype == COMMIT_PWMAP)
			bmExtentInvalidate(ip, addressDXD(&ip->i_acl),
					   lengthDXD(&ip->i_acl));
#endif /* _STILL_TO_PORT */

		/* acquire maplock on EA to be freed from block map */
		txEA(tid, ip, &ip->i_jfs_acl, NULL);

		if (committype == COMMIT_PWMAP)
			ip->i_jfs_acl.flag = 0;
	}

	/*
	 * free xtree/data (truncate to zero length):
	 * free xtree/data pages from cache if COMMIT_PWMAP, 
	 * free xtree/data blocks from persistent block map, and
	 * free xtree/data blocks from working block map if COMMIT_PWMAP;
	 */
	if (ip->i_size)
		rc = xtTruncate(tid, ip, 0, committype);

	return rc;
}


/*
 * NAME:	freeZeroLink()
 *
 * FUNCTION:    for non-directory, called by iClose(),
 *		free resources of a file from cache and WORKING map 
 *		for a file previously committed with zero link count
 *		while associated with a pager object,
 *
 * PARAMETER:	ip	- pointer to inode of file.
 *
 * RETURN:	0 -ok
 */
int freeZeroLink(struct inode * ip)
{
	int32 rc = 0;
	int32 type;

	jFYI(1,("freeZeroLink: ip = 0x%p\n", ip));

	/* return if not reg or symbolic link or if size is
	 * already ok.
	 */
	type = ip->i_mode & S_IFMT;

	switch (type) {
	case S_IFREG:
		break;
	case S_IFLNK:
		/* if its contained in inode nothing to do */
		if (ip->i_size <= 256)
			return 0;
		break;
	default:
		return 0;
	}

	/*
	 * free EA
	 */
	if (ip->i_jfs_ea.flag & DXD_EXTENT) {
		int64 xaddr;
		int32 xlen;
		maplock_t maplock;	/* maplock for COMMIT_WMAP */
		pxdlock_t *pxdlock;	/* maplock for COMMIT_WMAP */

		/* free EA pages from cache */
		xaddr = addressDXD(&ip->i_jfs_ea);
		xlen = lengthDXD(&ip->i_jfs_ea);
#ifdef _STILL_TO_PORT
		bmExtentInvalidate(ip, xaddr, xlen);
#endif

		/* free EA extent from working block map */
		maplock.index = 1;
		pxdlock = (pxdlock_t *) & maplock;
		pxdlock->flag = mlckFREEPXD;
		PXDaddress(&pxdlock->pxd, xaddr);
		PXDlength(&pxdlock->pxd, xlen);
		txFreeMap(ip, pxdlock, 0, COMMIT_WMAP);
	}

	/*
	 * free ACL
	 */
	if (ip->i_jfs_acl.flag & DXD_EXTENT) {
		int64 xaddr;
		int32 xlen;
		maplock_t maplock;	/* maplock for COMMIT_WMAP */
		pxdlock_t *pxdlock;	/* maplock for COMMIT_WMAP */

		/* free ACL pages from cache */
		xaddr = addressDXD(&ip->i_jfs_acl);
		xlen = lengthDXD(&ip->i_jfs_acl);
#ifdef _STILL_TO_PORT
		bmExtentInvalidate(ip, xaddr, xlen);
#endif

		/* free ACL extent from working block map */
		maplock.index = 1;
		pxdlock = (pxdlock_t *) & maplock;
		pxdlock->flag = mlckFREEPXD;
		PXDaddress(&pxdlock->pxd, xaddr);
		PXDlength(&pxdlock->pxd, xlen);
		txFreeMap(ip, pxdlock, 0, COMMIT_WMAP);
	}

	/*
	 * free xtree/data (truncate to zero length):
	 * free xtree/data pages from cache, and
	 * free xtree/data blocks from working block map;
	 */
	if (ip->i_size)
		rc = xtTruncate(0, ip, 0, COMMIT_WMAP);

	return rc;
}

/*
 * NAME:	jfs_link(vp, dvp, name, crp)
 *
 * FUNCTION:	create a link to <vp> by the name = <name>
 *		in the parent directory <dvp>
 *
 * PARAMETER:	vp 	- target object
 *		dvp	- parent directory of new link
 *		name	- name of new link to target object
 *		crp	- credential
 *
 * RETURN:	Errors from subroutines
 *
 * note:
 * JFS does NOT support link() on directories (to prevent circular
 * path in the directory hierarchy);
 * EPERM: the target object is a directory, and either the caller
 * does not have appropriate privileges or the implementation prohibits
 * using link() on directories [XPG4.2].
 *
 * JFS does NOT support links between file systems:
 * EXDEV: target object and new link are on different file systems and
 * implementation does not support links between file systems [XPG4.2].
 */
int jfs_link(struct dentry *old_dentry,
	     struct inode *dir, struct dentry *dentry)
{
	int32 rc;
	int32 tid = 0;
	struct inode *ip = old_dentry->d_inode;
	ino_t ino;
	component_t dname;
	btstack_t btstack;
	struct inode *iplist[2];

	jFYI(1,("jfs_link: %s %s\n", old_dentry->d_name.name, dentry->d_name.name));
/* The checks for links between filesystems and permissions are
   handled by the VFS layer                                     */

	/* JFS does NOT support link() on directories */
	if (S_ISDIR(ip->i_mode))
		return -EPERM;

	IWRITE_LOCK_LIST(2, dir, ip);

	txBegin(ip->i_sb, &tid, 0);

	/* link count overflow on target object ? */
	if (ip->i_nlink >= LINK_MAX) {
		rc = EMLINK;
		goto out;
	}

	/* validate directory search/write permission */
	/* Normally not needed. This will be checked by
	   the VFS layer */
	/*if (rc = iAccess(dir, IEXEC|IWRITE, crp))
	   goto out;
	 */

	/*
	 * scan parent directory for entry/freespace
	 */
	if ((rc = get_UCSname(&dname, dentry, ip->i_sb->s_jfs_nls_tab)))
		goto out;

	if ((rc = dtSearch(dir, &dname, &ino, &btstack, JFS_CREATE)))
		goto out;

	/*
	 * create entry for new link in parent directory
	 */
	ino = ip->i_ino;
	if ((rc = dtInsert(tid, dir, &dname, &ino, &btstack)))
		goto out;

	dir->i_version = ++event;

	/* update object inode */
	ip->i_nlink++;		/* for new link */
	ip->i_ctime = CURRENT_TIME;
	mark_inode_dirty(dir);
#ifdef kern22
	ip->i_count++;
#else
	atomic_inc(&ip->i_count);
#endif
	d_instantiate(dentry, ip);

	iplist[0] = ip;
	iplist[1] = dir;
	rc = txCommit(tid, 2, &iplist[0], 0);

      out:
	IWRITE_UNLOCK(dir);
	IWRITE_UNLOCK(ip);

	txEnd(tid);

	jFYI(1, ("jfs_link: rc:%d\n", rc));
	return -rc;
}

/*
 * NAME:	jfs_symlink(dip, dentry, name)
 *
 * FUNCTION:	creates a symbolic link to <symlink> by name <name>
 *		        in directory <dip>
 *
 * PARAMETER:	dip	    - parent directory vnode
 *		        dentry 	- dentry of symbolic link
 *		        name    - the path name of the existing object 
 *			              that will be the source of the link
 *
 * RETURN:	errors from subroutines
 *
 * note:
 * ENAMETOOLONG: pathname resolution of a symbolic link produced
 * an intermediate result whose length exceeds PATH_MAX [XPG4.2]
*/

int jfs_symlink(struct inode *dip, struct dentry *dentry, const char *name)
{
	int32 rc;
	int32 tid = 0;
	ino_t ino = 0;
	component_t dname;
	int ssize;		/* source pathname size */
	btstack_t btstack;
	struct inode *ip = dentry->d_inode;
	uint8 *i_fastsymlink;
	int64 xlen = 0;
	int32 bmask = 0, xsize;
	int64 xaddr;
	metapage_t *mp;
	struct super_block *sb;
	tlock_t *tlck;
	xtlock_t *xtlck;
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

#ifdef _STILL_TO_PORT
	dxd_t *i_dxd;
#endif				/*  _STILL_TO_PORT */
	struct inode *iplist[2];

	jFYI(1, ("jfs_symlink: dip:0x%p name:%s\n", dip, name));

	IWRITE_LOCK(dip);

	ssize = strlen(name) + 1; /* Account for null terminator */

	txBegin(dip->i_sb, &tid, 0);

#ifdef _I_DONT_THINK_WE_NEED_TO_CHECK
	/* validate search+write permission on parent directory */
	if (rc = iAccess(dip, IEXEC | IWRITE, crp))
		goto out1;
#endif				/* _I_DONT_THINK_WE_NEED_TO_CHECK */

	/*
	 * search parent directory for entry/freespace
	 * (dtSearch() returns parent directory page pinned)
	 */

	if ((rc = get_UCSname(&dname, dentry, dip->i_sb->s_jfs_nls_tab)))
		goto out1;

	if ((rc = dtSearch(dip, &dname, &ino, &btstack, JFS_CREATE)))


		goto out2;



	/*
	 * allocate on-disk/in-memory inode for symbolic link:
	 * (iAlloc() returns new, locked inode)
	 */

	ip = ialloc(dip, S_IFLNK | 0777);
	if (ip == NULL) {
		BT_PUTSEARCH(&btstack);
		rc = ENOSPC;
		goto out2;
	}
	IWRITE_LOCK(ip);

#ifdef _JFS_LAZYCOMMIT
	tblk = &TxBlock[tid];
	tblk->xflag |= COMMIT_CREATE;
	tblk->ip = ip;
#endif

	/*
	 * create entry for symbolic link in parent directory
	 */

	ino = ip->i_ino;



	if ((rc = dtInsert(tid, dip, &dname, &ino, &btstack))) {
		jERROR(1, ("jfs_symlink: dtInsert returned %d\n", rc));
		/* discard ne inode */
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);
		goto out2;

	}

	/* fix symlink access permission
	 * (dir_create() ANDs in the u.u_cmask, 
	 * but symlinks really need to be 777 access)
	 */
	ip->i_mode |= 0777;

	ip->i_op = &jfs_symlink_inode_operations;
#ifndef kern22
	ip->i_mapping->a_ops = &jfs_aops;
#endif

#ifdef _STILL_TO_PORT
	imark(ip, ICHG | IUPD | IFSYNC);

	/* update parent directory inode */
	imark(dip, ICHG | IUPD | IFSYNC);
#endif				/* _STILL_TO_PORT */



	/*
	   *       write symbolic link target path name
	 */
	xtInitRoot(tid, ip);

	/*
	 * write source path name inline in on-disk inode (fast symbolic link)
	 */

	if (ssize <= IDATASIZE) {
#ifdef _STILL_TO_PORT
		ip->i_dxd.flag = DXD_INLINE;
#endif				/* _STILL_TO_PORT */

		i_fastsymlink = &(((char *) (ip->i_jfs_inode_ext))
				  [sizeof(dasd_t) + sizeof(dxd_t)]);
		memcpy(i_fastsymlink, name, ssize);
		ip->i_size = ssize;
		/*
		 * This could probably be done cleaner, but for now, treat
		 * the transaction like an xtree update.
		 */
		tlck = txLock(tid, ip, (metapage_t *)&ip->i_jfs_bxflag,
			      tlckXTREE);
		xtlck = (xtlock_t *)&tlck->lock;
		xtlck->lwm.length = (ssize + XTSLOTSIZE -1) >> L2XTSLOTSIZE;
		jFYI(1,
		     ("jfs_symlink: fast symlink added  ssize:%d name:%s \n",
		      ssize, name));
	}
	/*
	 * write source path name in a single extent
	 */
	else {
		jFYI(1,
		     ("jfs_symlink: allocate extent ip:0x%p\n", ip));

		/*
		 * even though the data of symlink object (source 
		 * path name) is treated as non-journaled user data,
		 * it is read/written thru buffer cache for performance.
		 */
		sb = ip->i_sb;
		bmask = sb->s_jfs_bsize - 1;
		xsize = (ssize + bmask) & ~bmask;
		xaddr = 0;
		xlen = xsize >> sb->s_jfs_l2bsize;
		if ((rc = xtInsert(tid, ip, 0, 0, xlen, &xaddr, 0)) == 0) {
			ip->i_size = ssize;
			while (ssize) {
				int copy_size = MIN(ssize, PSIZE);
				
				mp = get_metapage(ip, xaddr, PSIZE,	
						  META_ABSOLUTE);

				if (mp == NULL) {
					dtDelete(tid, dip, &dname, &ino,
						 JFS_REMOVE);
					ip->i_nlink = 0;
					IWRITE_UNLOCK(ip);
					iput(ip);
					rc = EIO;
					goto out2;
				}
				memcpy(mp->data, name, copy_size);
				flush_metapage(mp);
#if 0
				mark_buffer_uptodate(bp, 1);
				mark_buffer_dirty(bp, 1);
				if (IS_SYNC(dip)) {
					ll_rw_block(WRITE, 1, &bp);
					wait_on_buffer(bp);
				}
				brelse(bp);
#endif /* 0 */
				ssize -= copy_size;
				xaddr += sb->s_jfs_nbperpage;
			}
			ip->i_blocks = LBLK2PBLK(sb, xlen);
		} else {
			dtDelete(tid, dip, &dname, &ino, JFS_REMOVE);
			ip->i_nlink = 0;
			IWRITE_UNLOCK(ip);
			iput(ip);
			rc = ENOSPC;
			goto out2;
		}
	}
	dip->i_version = ++event;

	mark_inode_dirty(ip);
	d_instantiate(dentry, ip);

	/*
	 * commit update of parent directory and link object
	 *
	 * if extent allocation failed (ENOSPC),
	 * the parent inode is committed regardless to avoid
	 * backing out parent directory update (by dtInsert())
	 * and subsequent dtDelete() which is harmless wrt 
	 * integrity concern.  
	 * the symlink inode will be freed by iput() at exit
	 * as it has a zero link count (by dtDelete()) and 
	 * no permanant resources. 
	 */

	iplist[0] = dip;
	if (rc == 0) {
		iplist[1] = ip;
		rc = txCommit(tid, 2, &iplist[0], 0);
	} else
		rc = txCommit(tid, 1, &iplist[0], 0);

	IWRITE_UNLOCK(ip);

      out2:

	free_UCSname(&dname);
      out1:
	IWRITE_UNLOCK(dip);

	txEnd(tid);

	jFYI(1, ("jfs_symlink: rc:%d\n", -rc));
	return -rc;
}


/*
 * NAME:        jfs_rename
 *
 * FUNCTION:    rename a file or directory
 */
int jfs_rename(struct inode *old_dir, struct dentry *old_dentry,
	       struct inode *new_dir, struct dentry *new_dentry)
{
	btstack_t btstack;
	ino_t ino;
	component_t new_dname;
	struct inode *new_ip;
	component_t old_dname;
	struct inode *old_ip;
	int rc;
	int tid = 0;
	tlock_t *tlck;
	dtlock_t *dtlck;
	lv_t *lv;
	int ipcount;
	struct inode *iplist[4];
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif


	jFYI(1,("jfs_rename: %s %s\n", old_dentry->d_name.name, new_dentry->d_name.name));

	old_ip = old_dentry->d_inode;
	new_ip = new_dentry->d_inode;

	if (old_dir == new_dir) {
		if (new_ip)
			IWRITE_LOCK_LIST(3, old_dir, old_ip, new_ip);
		else
			IWRITE_LOCK_LIST(2, old_dir, old_ip);
	} else {
		if (new_ip)
			IWRITE_LOCK_LIST(4, old_dir, new_dir, old_ip, new_ip);
		else
			IWRITE_LOCK_LIST(3, old_dir, new_dir, old_ip);
	}

	if ((rc = get_UCSname(&old_dname, old_dentry,
			      old_dir->i_sb->s_jfs_nls_tab)))
		goto out1;

	if ((rc = get_UCSname(&new_dname, new_dentry,
			      old_dir->i_sb->s_jfs_nls_tab)))
		goto out2;

	/*
	 * Make sure source inode number is what we think it is
	 */
	rc = dtSearch(old_dir, &old_dname, &ino, &btstack, JFS_LOOKUP);
	if (rc || (ino != old_ip->i_ino)) {
		rc = ENOENT;
		goto out3;
	}

	/*
	 * Make sure dest inode number (if any) is what we think it is
	 */
	rc = dtSearch(new_dir, &new_dname, &ino, &btstack, JFS_LOOKUP);
	if (rc == 0) {
		if ((new_ip == 0) || (ino != new_ip->i_ino)) {
			rc = ESTALE;
			goto out3;
		}
	}
	else if (rc != ENOENT)
		goto out3;
	else if (new_ip)
	{
		/* no entry exists, but one was expected */
		rc = ESTALE;
		goto out3;
	}

	if (S_ISDIR(old_ip->i_mode)) {
		if (new_ip) {
			if (! dtEmpty(new_ip)) {
				rc = ENOTEMPTY;
				goto out3;
			}
		}
		else if ((new_dir != old_dir) &&
			 (new_dir->i_nlink >= JFS_LINK_MAX)) {
			rc = EMLINK;
			goto out3;
		}
	}

	/*
	 * The real work starts here
	 */
	txBegin(new_dir->i_sb, &tid, 0);

	if (new_ip) {
		/*
		 * Change existing directory entry to new inode number
		 */
		ino = new_ip->i_ino;
		rc = dtModify(tid, new_dir, &new_dname, &ino, old_ip->i_ino,
			      JFS_RENAME);
		if (rc)
			goto out4;
		new_ip->i_nlink--;
		if (S_ISDIR(new_ip->i_mode)) {
			new_ip->i_nlink--;
			assert (new_ip->i_nlink == 0);
#ifdef _JFS_LAZYCOMMIT
			tblk = &TxBlock[tid];
			tblk->xflag |= COMMIT_DELETE;
			tblk->ip = new_ip;
#endif
		} else if (new_ip->i_nlink == 0) {
			assert(!(new_ip->i_jfs_cflag & COMMIT_NOLINK));
			/* free block resources */
			if ((rc = commitZeroLink(tid, new_ip))) {
				txAbort(tid, 1);	/* Marks FS Dirty */
				goto out4;
			}
#ifdef _JFS_LAZYCOMMIT
			tblk = &TxBlock[tid];
			tblk->xflag |= COMMIT_DELETE;
			tblk->ip = new_ip;
#endif
		} else {
			new_ip->i_ctime = CURRENT_TIME;
			mark_inode_dirty(new_ip);
		}
	}
	else
	{
		/*
		 * Add new directory entry
		 */
		rc = dtSearch(new_dir, &new_dname, &ino, &btstack, JFS_CREATE);
		if (rc) {
			jERROR(1,("jfs_rename didn't expect dtSearch to fail w/rc = %d\n", rc));
			goto out4;
		}

		ino = old_ip->i_ino;
		rc = dtInsert(tid, new_dir, &new_dname, &ino, &btstack);
		if (rc) {
			jERROR(1,("jfs_rename: dtInsert failed w/rc = %d\n",
				  rc));
			goto out4;
		}
		if (S_ISDIR(old_ip->i_mode))
			new_dir->i_nlink++;
	}
	/*
	 * Remove old directory entry
	 */

	ino = old_ip->i_ino;
	rc = dtDelete(tid, old_dir, &old_dname, &ino, JFS_REMOVE);
	if (rc) {
		jERROR(1,("jfs_rename did not expect dtDelete to return rc = %d\n",
			  rc));
		txAbort(tid, 1);	/* Marks Filesystem dirty */
		goto out4;
	}
	if (S_ISDIR(old_ip->i_mode)) {
		old_dir->i_nlink--;
 		if (old_dir != new_dir) {
			/*
		         * Change inode number of parent for moved directory
			 */

			((dtroot_t *)old_ip->i_jfs_inode_ext)->header.idotdot =
				new_dir->i_ino;

			/* Linelock header of dtree */
			tlck = txLock(tid, old_ip,
				      (metapage_t *)&old_ip->i_jfs_bxflag,
				      tlckDTREE|tlckBTROOT);
			dtlck = (dtlock_t *)&tlck->lock;
			ASSERT(dtlck->index == 0);
			lv = (lv_t *)&dtlck->lv[0];
			lv->offset = 0;
			lv->length = 1;
			dtlck->index++;
		}
	}

	/*
	 * Update ctime on changed/moved inodes & mark dirty
	 */
	old_ip->i_ctime = CURRENT_TIME;
	mark_inode_dirty(old_ip);

	new_dir->i_version = ++event;
	new_dir->i_ctime= CURRENT_TIME;	
	mark_inode_dirty(new_dir);

	/* Build list of inodes modified by this transaction */
	ipcount = 0;
	iplist[ipcount++] = old_ip;
	if (new_ip)
		iplist[ipcount++] = new_ip;
	iplist[ipcount++] = old_dir;

	if (old_dir != new_dir) {
		iplist[ipcount++] = new_dir;
		old_dir->i_version = ++event;
		old_dir->i_ctime= CURRENT_TIME;	
		mark_inode_dirty(old_dir);
	}

	rc = txCommit(tid, ipcount, iplist, 0);

out4:
	txEnd(tid);
out3:
	free_UCSname(&new_dname);
out2:
	free_UCSname(&old_dname);
out1:
	IWRITE_UNLOCK(old_dir);
	IWRITE_UNLOCK(old_ip);
	if (old_dir != new_dir)
		IWRITE_UNLOCK(new_dir);
	if (new_ip)
		IWRITE_UNLOCK(new_ip);

	jFYI(1,("jfs_rename: returning %d\n", rc));
	return -rc;
}


/*
 * NAME:        jfs_mknod
 *
 * FUNCTION:    Create a special file (device)
 */
int jfs_mknod(struct inode *dir, struct dentry *dentry, int mode, int rdev)
{
	btstack_t btstack;
	component_t dname;
	ino_t ino;
	struct inode *ip;
	struct inode *iplist[2];
	int rc;
	int tid;
#ifdef _JFS_LAZYCOMMIT
	tblock_t *tblk;
#endif

	jFYI(1,("jfs_mknod: %s\n", dentry->d_name.name));

	if ((rc = get_UCSname(&dname, dentry, dir->i_sb->s_jfs_nls_tab)))
		return rc;

	IWRITE_LOCK(dir);

	ip = ialloc(dir, mode);
	if (ip == NULL) {
		rc = ENOSPC;
		goto out1;
	}

	IWRITE_LOCK(ip);

	txBegin(dir->i_sb, &tid, 0);

	if ((rc = dtSearch(dir, &dname, &ino, &btstack, JFS_CREATE))) {
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);
		txEnd(tid);
		goto out1;
	}

#ifdef _JFS_LAZYCOMMIT
	tblk = &TxBlock[tid];
	tblk->xflag |= COMMIT_CREATE;
	tblk->ip = ip;
#endif

	ino = ip->i_ino;
	if ((rc = dtInsert(tid, dir, &dname, &ino, &btstack))) {
		ip->i_nlink = 0;
		IWRITE_UNLOCK(ip);
		iput(ip);
		txEnd(tid);
		goto out1;
	}

	if (S_ISREG(ip->i_mode)) {
		ip->i_op = &jfs_file_inode_operations;
#ifdef kern22
	} else if (S_ISCHR(ip->i_mode))
		ip->i_op = &chrdev_inode_operations;
	else if (S_ISBLK(ip->i_mode))
		ip->i_op = &blkdev_inode_operations;
	else if (S_ISFIFO(ip->i_mode))
		init_fifo(ip);

	if (S_ISBLK(ip->i_mode) || S_ISCHR(ip->i_mode))
		ip->i_rdev = to_kdev_t(rdev);
#else
		ip->i_fop = &jfs_file_operations;
		ip->i_mapping->a_ops = &jfs_aops;
	} else
		init_special_inode(ip, ip->i_mode, rdev);
#endif

	mark_inode_dirty(ip);
	d_instantiate(dentry, ip);

	dir->i_version = ++event;
	dir->i_ctime = dir->i_mtime = CURRENT_TIME;

	mark_inode_dirty(dir);

	iplist[0] = dir;
	iplist[1] = ip;
	rc = txCommit(tid, 2, iplist, 0);
	txEnd(tid);

	IWRITE_UNLOCK(ip);
out1:
	IWRITE_UNLOCK(dir);
	free_UCSname(&dname);

	jFYI(1,("jfs_mknod: returning %d\n", rc));
	return -rc;
}
