/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/locks.h>
#include <linux/errno.h>
#include <asm/uaccess.h>
#include <linux/jfs_fs.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_extent.h>
#include <linux/jfs/jfs_txnmgr.h>
#include <linux/jfs/jfs_debug.h>

#ifdef kern22
/*
 * Forward declarations
 */
#ifndef _JFS_4K
static int PromoteI(struct inode *, loff_t);
#endif				/*  _JFS_4K */
static void WRClean(struct inode *, int64, boolean_t, boolean_t);

ssize_t jfs_file_write(struct file *filp,
		       const char *buf, size_t count, loff_t * ppos)
{
	loff_t ablks;
	loff_t asize;
	struct buffer_head *bp;
	uint coff;
#ifndef _JFS_4K
	uint32 endblks;
	uint32 endbytes;
#endif				/* _JFS_4K */
	struct inode *ip = filp->f_dentry->d_inode;
	int l2bsize;
	unsigned long limit;
	int nb;
	loff_t nbc;
	loff_t nblks;
	int nbperpage;
	size_t nbytes;
	loff_t newsize;
	loff_t off;
	loff_t oldsize;
	int partial;
	const char *pdata;
	loff_t pos;
	int rc = 0;
	int rem;
	loff_t reqblks;
	struct super_block *sb = ip->i_sb;
	xad_t xad;
	int64 xaddr;
	uint8 xflag;
	int32 xlen;

	jFYI(1,
	     ("jfs_write: count = 0x%lx, pos = 0x%lx\n", (ulong)count, (ulong)* ppos));

	if (count == 0)
		return 0;

	IWRITE_LOCK(ip);

	if (ip->i_mode & (S_ISGID | S_ISUID)) {
		ip->i_mode &= ~(S_ISGID | S_ISUID);
		mark_inode_dirty(ip);
	}

	if (filp->f_flags & O_APPEND)
		pos = ip->i_size;
	else
		pos = *ppos;

	limit = current->rlim[RLIMIT_FSIZE].rlim_cur;
	if (limit < RLIM_INFINITY) {
		if ((pos + count) > limit) {
			count = limit - pos;
			if (((signed) count) <= 0) {
				IWRITE_UNLOCK(ip);
				send_sig(SIGXFSZ, current, 0);
				return -EFBIG;
			}
		}
	}

	pdata = buf;

	/* remember the current file size for backout */
	oldsize = ip->i_size;

#ifndef _JFS_4K
	/* if the write starts beyond the last byte
	 * of the file + 1 (i.e. lseek beyond end of file
	 * then write), extend the file size to the starting
	 * offset of the write.
	 */
	if (pos > ip->i_size) {
		if (rc = PromoteI(ip, pos)) {
			IWRITE_UNLOCK(ip);
			return (-rc);
		}
	}
#endif				/*  _JFS_4K */

	/* get the log2 block size and number of blocks per page */
	l2bsize = sb->s_jfs_l2bsize;
	nbperpage = sb->s_jfs_nbperpage;

	/* initialize the allocation hint/return value.  the xad
	 * pointed to by xp will hold the input allocation hint
	 * provided to the extent allocator as well as the results
	 * (set by the allocator) of the allocation.
	 */
	XADaddress(&xad, 0);

	/* compute the ending size of the write */
	newsize = pos + count;
	off = pos;
	nbytes = count;

#ifdef _JFS_4K
	asize = (ip->i_size + POFFSET) & ~POFFSET;
	partial = 0;
#else				/* ! _JFS_4K */
	/* get the number of blocks backing the last page of the
	 * file, the number of bytes backed by these blocks, and
	 * indicator of whether the last page is fully backed.
	 */
/*      endblks = CM_BTOBLKS(ip->i_size,ip->i_ipmnt->i_bsize,l2bsize); */
	endblks = (((ip->i_size + ip->i_blksize - 1) >> l2bsize) &
		   (nbperpage - 1)) + 1;
	endbytes = endblks << l2bsize;
	partial = (endblks != nbperpage);

	/* get the file's current allocation size.
	 * this size is defined as part (pages) of the file
	 * which can handle the current write request without
	 * any new disk allocation required (excluding disk
	 * allocation to fill file holes for sparse files).
	 */
	asize =
	    (ip->i_size) ? ((ip->i_size - 1) & ~POFFSET) + endbytes : 0;

	/* force allocated size to page boundary to defer
	 * extending last partial page in step II.
	 */
	if (partial && asize < newsize)
		asize &= ~POFFSET;
#endif				/* _JFS_4K */

	/*
	 * step I: if any of the current write request is within
	 * the file's current allocation size, handle this part
	 * of the write request.
	 */
	if ((nbc = asize - off) > 0) {
		/* loop over the pages (cbufs) for this portion of
		 * the write, allocating disk space (file holes within
		 * sparse file) and copying data as neccessary.
		 */
		for (nbc = MIN(nbc, nbytes); nbc > 0; nbc -= nb,
		     nbytes -= nb, off += nb, pdata += nb) {
			/* get the starting offset within the cbuf at
			 * which data will be copied and the number of
			 * bytes to copy.
			 */
			coff = (uint) off & POFFSET;
			nb = MIN(nbc, PSIZE - coff);

			/* get the cbuf for the current page */
//                        if (rc = cmReadWR(ip,off,nb,&cp))
//                                goto out;
			if (xtLookup(ip, off >> sb->s_jfs_l2bsize,
				     nbperpage, &xflag, &xaddr, &xlen, 0) ||
			    (xlen == 0)) {
				/* current page represents a file hole */
				/* must be a sparse file to have a hole.
				 */
//                                assert(ISSPARSE(ip));

				/* allocate disk space for the hole */
				XADoffset(&xad, off >> sb->s_jfs_l2bsize);
				XADlength(&xad, nbperpage);
/* ??? block allocation per page at a time ? */
				if ((rc = extFill(ip, &xad)))
					goto out;
				xaddr = addressXAD(&xad);

				bp = getblk(ip->i_dev,
					    xaddr >> sb->s_jfs_l2nbperpage,
					    PSIZE);
			}

			else if (xflag & XAD_NOTRECORDED) {
				/* Current page allocated but not recorded.
				 * Change the page to allocated and recorded.
				 */
				XADoffset(&xad, off >> sb->s_jfs_l2bsize);
				XADlength(&xad, xlen);
				XADaddress(&xad, xaddr);
				if ((rc = extRecord(ip, &xad)))
					goto out;

				bp = getblk(ip->i_dev,
					    xaddr >> sb->s_jfs_l2nbperpage,
					    PSIZE);

			} else
				bp = bread(ip->i_dev,
					   xaddr >> sb->s_jfs_l2nbperpage,
					   PSIZE);

			if (bp == NULL) {
				rc = EIO;
				goto out;
			}

			lock_buffer(bp);

			if (!buffer_uptodate(bp)) {
				/* new page */
				if (nb < PSIZE)
					memset(bp->b_data, 0, PSIZE);
				mark_buffer_uptodate(bp, 1);
			}

			/* copy in the data to the cbuf */
			if (copy_from_user(bp->b_data + coff, pdata, nb)) {
jERROR(1,("jfs_file_write: copy_from_user failed\n"));
				rc = EFAULT;
				unlock_buffer(bp);
				brelse(bp);
				goto out;
			}

			unlock_buffer(bp);
			mark_buffer_dirty(bp, 0);
			update_vm_cache_conditional(ip, off, bp->b_data + coff,
						    nb, (unsigned long)pdata);
			brelse(bp);
		}

		/* adjust the file size if this portion of the
		 * write has extended it.
		 */
		ip->i_size = MAX(ip->i_size, off);

		/* are we done copying data for this write ? */
		if (nbytes == 0)
			goto out;

		/* more data to write. if the last page of the
		 * file is NOT partial (i.e. new extent allocation
		 * is required), construct the allocation hint
		 * that will be used for the extent allocation.
		 */
		if (!partial) {
			XADaddress(&xad, xaddr);
			XADoffset(&xad, ((off - 1) & ~POFFSET) >> l2bsize);
			XADlength(&xad, nbperpage);
			xad.flag = 0;
		}
	}

	/* compute number of blocks required to hold the remaining
	 * portion of the write.
	 */
	reqblks =
	    ((newsize - (off & ~POFFSET) + sb->s_jfs_bsize - 1) >>
	     l2bsize);

#ifndef _JFS_4K
	/*
	 * step II: if the last page of the file is partially backed,
	 * the allocation for this partial page must be extended
	 * as part of the write and we'll handle that here
	 * along with the data movement for the page.
	 */
	if (partial) {
		/* get the starting offset within the cbuf at
		 * which data will be copied and the number of
		 * bytes to copy.
		 */
		coff = (uint) off & POFFSET;
		nb = MIN(nbytes, PSIZE - coff);

		/* get the cbuf for the current page */
		if ((rc = xtLookup(ip, off >> sb->s_jfs_l2bsize,
				   nbperpage, &xflag, &xaddr, &xlen, 0)))
			goto out;
		if (coff) {
			bp =
			    bread(ip->i_dev,
				  xaddr >> sb->s_jfs_l2nbperpage, PSIZE);
			if (bp == NULL) {
				rc = EIO;
				goto out;
			}
		} else {
			bp =
			    getblk(ip->i_dev,
				   xaddr >> sb->s_jfs_l2nbperpage, PSIZE);
			if (bp == NULL) {
				rc = EIO;
				goto out;
			}

			mark_buffer_uptodate(bp, 1);
		}

		/* copy the data to the cbuf */
		if (copy_from_user(bp->b_data + coff, pdata, nb)) {
			rc = EFAULT;
			brelse(bp);
			goto out;
		}

		/* zero any bytes between the end of the file
		 * and the start of the write.
		 */
		if ((rem = coff - (endbytes)) > 0)
			memset(bp->b_data + endbytes, 0, rem);

		/* extend the allocation.
		 * in extending the allocation for the current
		 * (partial) page, we attempt to extend the allocation
		 * such that it is sufficient to handle the remaining
		 * allocation requirements of the current write
		 * request (i.e. we ask extRealloc() to allocate
		 * an extent of reqblks blocks).
		 * extRealloc() may provide us with an extent with
		 * less blocks than we asked for but the number
		 * of block provided should always be sufficient to
		 * cover new requirements of the partial page.
		 */
		if ((rc = extRealloc(ip, cp, reqblks, &xad, FALSE))) {
			coff = (oldsize - 1) & POFFSET;
			bzero(cp->cm_cdata + coff, PSIZE - coff);
			cmPut(cp, FALSE);
			goto out;
		}

		/* get the size of the extent allocated and
		 * its starting disk block address.
		 */
		xlen = lengthXAD(&xad);
		xaddr = addressXAD(&xad);

		/* get the number of blocks required for the current
		 * page.
		 */
		nblks = MIN(nbperpage, xlen);

		/* zero any bytes beyond the end of the write and
		 * up to the end of the backed portion of the cbuf.
		 */
		if (rem = (nblks << l2bsize) - (coff + nb))
			bzero(cp->cm_cdata + coff + nb, rem);

		/* update the cbuf with the starting address and
		 * number of blocks for the page.
		 */
		PXDaddress(&cp->cm_pxd, xaddr);
		PXDlength(&cp->cm_pxd, nblks);

		cmPut(cp, TRUE);

		/* adjust the file offset, user write buffer address,
		 * and number of bytes written.
		 */
		off += nb;
		pdata += nb;
		nbytes -= nb;

		/* adjust the file size to reflect the bytes written */
		ip->i_size = off;

		/* adjust the number of disk blocks required with
		 * the number of block contained within the newly
		 * allocated extent.
		 */
		reqblks -= xlen;

		/* adjust the newly allocated extents starting address
		 * and number of block remaining to reflect the blocks
		 * used by the current page.
		 */
		xlen -= nblks;
		xaddr += nblks;
	} else
#endif				/* ! _JFS_4K */
		xlen = 0;

	/* get allocation hint if we currently do not have one */
	if (addressXAD(&xad) == 0) {
		/* get a hint */
		if ((rc = extHint(ip, off, &xad))) {
			WRClean(ip, oldsize, count != nbytes, FALSE);
			goto out;
		}
	}

	/*
	 * step III: handle the portion of the write that is beyond
	 * the last page of the file, a page (cbuf) at a time.
	 */
	for (ablks = reqblks; nbytes > 0; nbytes -= nb,
	     off += nb, pdata += nb, xlen -= nblks, xaddr += nblks) {
		/* get the starting offset within the cbuf at
		 * which data will be copied and the number of
		 * bytes to copy.
		 */
		coff = (uint) off & POFFSET;
		nb = MIN(nbytes, PSIZE - coff);

		/* check if we have an extent in hand with free space in
		 * it. if not, allocate a new extent. extAlloc() is used
		 * to allocate the extent and we ask it allocate ablks
		 * blocks.  however, it may allocate an extent with less
		 * than ablks (this many contigious free blocks are not
		 * avaliable) but the extent will always be large enough
		 * to cover the current page.  initially, we ask this
		 * routine to allocate reqblks blocks (i.e. ablks ==
		 * reqblks).  however, if less than reqblks are provided
		 * next allocation request will be for the minimum
		 * of the number of blocks still required and the number
		 * of block (last) provided.
		 */
		if (xlen == 0) {
			/* try to allocate an extent containing ablks blocks.
			 */
			jFYI(0,
			     ("jfs_write calling extAlloc: ablks = 0x%lx, off = 0x%lx\n",
			      (ulong)ablks, (ulong)off));
			if (
			    (rc =
			     extAlloc(ip, ablks, off >> L2PSIZE, &xad,
				      FALSE))) {
				jERROR(1,
				       ("jfs_write: extAlloc returned %d\n",
					rc));
				WRClean(ip, oldsize, count != nbytes,
					FALSE);
				break;
			}

			/* get the size of the extent allocated and its
			 * starting address.
			 */
			xlen = lengthXAD(&xad);
			xaddr = addressXAD(&xad);

			/* update the number of required blocks to reflect
			 * the number of blocks just allocated.
			 */
			reqblks -= xlen;

			/* update ablks for the next go round */
			ablks = MIN(reqblks, xlen);
		}

		bp = getblk(ip->i_dev, xaddr, PSIZE);
		if (bp == NULL) {
			jERROR(1,
			       ("jfs_write: getblk returned NULL, xaddr = 0x%lx\n",
				(ulong)xaddr));
			rc = EIO;
			WRClean(ip, oldsize, count != nbytes, TRUE);
			break;
		}
		lock_buffer(bp);

		/* copy in the data */
		if (copy_from_user(bp->b_data + coff, pdata, nb)) {
			rc = EFAULT;
			jERROR(1, ("jfs_write: copy_from_user failed\n"));
			unlock_buffer(bp);
			brelse(bp);
			WRClean(ip, oldsize, count != nbytes, TRUE);
			break;
		}

		/* determine how many blocks are required for this page.
		 */
		nblks = MIN(nbperpage, xlen);

		/* zero any bytes between the end of the file
		 * and the start of the write.
		 */
		if (coff)
			memset(bp->b_data, 0, coff);

		/* zero any bytes beyond the end of the write and
		 * up to the end of the backed portion of the cbuf.
		 */
		if ((rem = (nblks << l2bsize) - (coff + nb)))
			memset(bp->b_data + coff + nb, 0, rem);

		mark_buffer_uptodate(bp, 1);
		unlock_buffer(bp);
		mark_buffer_dirty(bp, 0);
		brelse(bp);

		/* adjust the file size to reflect the data written */
		ip->i_size = off + nb;
	}

      out:
	IWRITE_UNLOCK(ip);
	if (count > nbytes) {
		*ppos = off;
		ip->i_ctime = ip->i_mtime = CURRENT_TIME;
		mark_inode_dirty(ip);

		return (count - nbytes);
	} else
		return -rc;

#ifdef _STILL_TO_PORT
	/*
	 * if file was open O_SYNC and if we actually modified the file,
	 * sync the modifications.
	 */
	if (flags & FSYNC) {
		if (nbytes || ip->i_size != oldsize)
			rc = SyncDataI(ip, offset, count);
		return rc;
	}

	if (nbytes == 0)
		return rc;

	/*
	 *      write-behind:
	 *
	 * initiate async write of all pages outside of last cluster
	 * of current write;
	 */
	first = offset >> CM_L2BSIZE;	/* first page of current write */
	last = (offset + nbytes - 1) >> L2PSIZE;	/* last page of current write */
	cllast = last & ~(CM_WRCLNBLKS - 1);

	/* If Lazywrite is turned off, asynchronously write the data
	 */
	if (LazyOff) {
		cmAsyncWrite(ip->i_cacheid, first, last - first + 1);
		ip->i_wrbehind = FALSE;
		ip->i_wbpage = last;
		return rc;
	}

	/*
	 * sequential write:
	 *
	 * either the current write starts in the same block
	 * as the previous write and continues into another block,
	 * or the write starts in the contiguous block to
	 * the previous write;
	 */
	if ((first == ip->i_wbpage && last != ip->i_wbpage) ||
	    (first == ip->i_wbpage + 1)) {
		wbclust = ip->i_wbpage & ~(CM_WRCLNBLKS - 1);

		if ((ip->i_wrbehind) &&	/* prior behavior was sequential */
		    (cllast > wbclust)) {	/* write extends past last cluster */
			jEVENT(0,
			       ("wrbehind: s start:%d%d n:%d%d\n",
				ip->i_wbpage, cllast - wbclust));
			/* init async write of preceding cluster(s) */
			cmAsyncWrite(ip->i_cacheid, wbclust,
				     (int64) (cllast - wbclust));
		}
		ip->i_wrbehind = TRUE;
	} else if (first == ip->i_wbpage)	/* write within same page */
		ip->i_wrbehind = TRUE;
	else
		ip->i_wrbehind = FALSE;

	ip->i_wbpage = last;

	return (rc);
#endif				/*  _STILL_TO_PORT */
}


#ifndef _JFS_4K
/*
 * NAME:         PromoteI()
 *
 * FUNCTION:     extend the size of a file in preparation for
 *               a write operation that starts beyond the last fully
 *               allocated page of a file.
 *
 * PARAMETERS:
 *      ip      - the inode of the file.
 *      offset  - starting offset of the write.
 *
 * RETURN VALUES:
 *      0       - success.
 *      EIO     - i/o error.
 *      ENOSPC  - insufficient disk resources.
 */
static int PromoteI(struct inode *ip, loff_t offset)
{
	int64 pnosz, pnooff;

	/* compute the page numbers of the last page of
	 * the file and the page containing the starting
	 * offset of the write.
	 */
	pnosz = ((ip->i_size + POFFSET) >> L2PSIZE);
	pnooff = offset >> L2PSIZE;

	/* check if the pages are the same.
	 */
	if (pnooff != pnosz) {
#ifdef _STILL_TO_PORT
		/* for a dense file: if there are pages between
		 * the last page of the file and the page containing
		 * the offset, extend the file to the end of the
		 * page prior to the starting offset.
		 */
		if (ISSPARSE(ip) == FALSE && pnooff - 1 > pnosz)
			return (ExtendI(ip, CM_CBLKTOB(pnooff - 1), TRUE));
#endif				/*  _STILL_TO_PORT */

		/* if the last page of the file is a partially
		 * backed page, extend the file to the end
		 * of the last page.
		 */
		ipmnt = ip->i_ipmnt;
		if (CM_BTOBLKS
		    (ip->i_size, ipmnt->i_bsize,
		     ipmnt->i_l2bsize) !=
		    ipmnt->i_nbperpage) return (ExtendI(ip,
							CM_CBLKTOB(pnosz),
							TRUE));
	}

	return (0);
}
#endif				/* ! _JFS_4K */


#ifndef _JFS_4K
/*
 * NAME:        ExtendI()
 *
 * FUNCTION:    extend the size of a file.
 *
 * PARAMETERS:
 *      ip      - the inode of the file.
 *      newsize - new file size.
 *      abnr    - boolean_t indicating whether the extents allocated
 *                as part of the extension should be allocated but
 *                not recorded.
 *
 * RETURN VALUES:
 *      0       - success
 *      EIO     - i/o error.
 *      ENOSPC  - insufficient disk resources.
 */
ExtendI(struct inode * ip, int64 newsize, boolean_t abnr)
{
	int32 nbperpage, l2bsize, endblks, rc, sparse, endbytes, rc2;
	int64 asize, reqblks, xlen, pnoold, pnonew, oldsize;
	xad_t xad, *xp = &xad;
	cbuf_t *cp;

	/* new size should be greater than current size */
	assert(newsize > ip->i_size);

	/* get the log2 block size and number of blocks per page */
	l2bsize = ip->i_ipmnt->i_l2bsize;
	nbperpage = ip->i_ipmnt->i_nbperpage;

	if (newsize > ((int64) 1 << (40 + l2bsize)))
		return ENOSPC;

	asize = (ip->i_size) ? ((ip->i_size - 1) & ~POFFSET) : 0;

	/* get the number of blocks backing the last page of the
	 * file and the number of bytes backed by these blocks.
	 */
	endblks = CM_BTOBLKS(ip->i_size, ip->i_ipmnt->i_bsize, l2bsize);
	endbytes = endblks << l2bsize;

	/* get the file's current allocation size */
	asize = (ip->i_size) ? ((ip->i_size - 1) & ~POFFSET) + endbytes
	    : 0;

	/* if the new size is within the allocated size,
	 * set the new size and return.
	 */
	if (newsize <= asize) {
		ip->i_size = newsize;
		return (0);
	}

	/* remember the current file size for backout */
	oldsize = ip->i_size;

	/* initialize the allocation hint/return value.
	 * the xad pointed to by xp will hold the input allocation
	 * hint provided to the extent allocator as well as
	 * the results (set by the allocator) of the allocation.
	 */
	XADaddress(xp, 0);

	/* get an indicator of whether the file is sparse */
	sparse = ISSPARSE(ip);

	/* determine how many additional blocks must be added to
	 * the file, excluding partial page requirements, if any.
	 */
	reqblks =
	    (sparse) ? CM_BTOBLKS(newsize, ip->i_ipmnt->i_bsize,
				  l2bsize) : CM_BTOFBLKS(newsize - asize,
							 l2bsize);

	/*
	 * if the last page of the file partially backed,
	 * handle the extension of this page.
	 */
	if (endblks != nbperpage) {
		/* get the cbuf for the last (partial) page of the file.
		 */
		if (rc = cmRead(ip, ip->i_size - 1, -1, &cp))
			return (rc);

		/* determine how many additional blocks are required
		 * for the partial page and how many blocks will be
		 * allocated.
		 */
		if (sparse) {
			/* get the page numbers for the pages containing
			 * the current size and newsize.
			 */
			pnoold = CM_BTOCBLK(ip->i_size);
			pnonew = CM_BTOCBLK(newsize);

			/* if they are the same than no additional blocks
			 * are required and will try to allocate a single
			 * extent for the last page's new requirements.
			 */
			if (pnoold == pnonew) {
				xlen = reqblks;
			} else {
				/* pages are not the same, so we have to
				 * account for all of the last partial
				 * page. if the page containing newsize
				 * is the page after the last page we
				 * will try to allocate a single extent
				 * for both.  otherwise, we'll only allocate
				 * an extent for the partial page's new
				 * requirements and let the code below
				 * handle the new size page.
				 */
				reqblks += nbperpage;
				xlen = (pnoold + 1 == pnonew) ? reqblks :
				    nbperpage;
			}
		} else {
			/* for dense files, we have to take into
			 * account the existing blocks of the last
			 * page. and we'll try to allocate a single
			 * extent for all of the new blocks required
			 * by the file extension.
			 */
			reqblks += endblks;
			xlen = reqblks;
		}

		/* extend the allocation. in extending the allocation for
		 * the current (partial) page, we attempt to extend the
		 * allocation such that it maybe sufficient to handle the
		 * remaining allocation requirements of the current extend
		 * request. extRealloc() may provide us with an extent with
		 * less blocks than we asked for but the number of block
		 * provided should always be sufficient to cover new
		 * requirements of the partial page.
		 */
		if (rc = extRealloc(ip, cp, xlen, xp, abnr)) {
			cmPut(cp, FALSE);
			return (rc);
		}

		/* get the number of blocks in the new allocated
		 * extent.
		 */
		xlen = lengthXAD(xp);

		/* update the cbuf with the starting address and
		 * number of blocks for the page.
		 */
		PXDlength(&cp->cm_pxd, MIN(nbperpage, xlen));
		PXDaddress(&cp->cm_pxd, addressXAD(xp));

		/* zero the portion of the cbuf that reflects the
		 * additional allocation.
		 */
		bzero(cp->cm_cdata + endbytes, PSIZE - endbytes);

		/* put the cbuf.
		 */
		cmPut(cp, TRUE);

		/* update the file's current allocated size to reflect
		 * the new extent.
		 */
		asize += ((xlen - endblks) << l2bsize);

		/* adjust the file size to reflect this size.
		 */
		ip->i_size = MIN(newsize, asize);

		/* adjust the number of blocks required to reflect
		 * the newly allocated extent.
		 */
		reqblks -= xlen;
	}

	/* for sparse file: recompute the file's current allocation
	 * size as the start of the page containing the new size.
	 */
	if (sparse)
		asize = (newsize - 1) & ~POFFSET;

	/* if no allocation hint, call extHint() to get one */
	if (addressXAD(xp) == 0) {
		if (rc = extHint(ip, asize, xp))
			return (rc);
	}

	/*
	 * add extents to the end of the file until we have extended
	 * the file such that it reflects the new size.
	 */
	for (xlen = reqblks; reqblks > 0; reqblks -= xlen) {
		/* extAlloc() is used to allocate the extents and we ask
		 * it allocate xlen blocks.  however, it may allocate an
		 * extent with less than xlen (this many contigious free
		 * blocks are not avaliable) but the extent will always
		 * be large enough to cover a page.  initially, we ask
		 * ask this routine to allocate reqblks blocks (i.e. xlen ==
		 * reqblks).  however, if less than reqblks are provided,
		 * the next allocation request will be for the minimum
		 * of the number of blocks still required and the number
		 * of block (last) provided.
		 */
		xlen = MIN(reqblks, xlen);
		if (
		    (rc =
		     extAlloc(ip, xlen, CM_OFFTOCBLK(asize), xp,
			      abnr))) goto error;

		/* get the number of blocks in the new allocated
		 * extent.
		 */
		xlen = lengthXAD(xp);

		/* update the file's current allocated size to reflect
		 * the new extent.
		 */
		asize += (xlen << l2bsize);

		/* adjust the file size to reflect this size.
		 */
		ip->i_size = MIN(newsize, asize);
	}

	return (0);

      error:
	/* if we have done some extension of the file size, move
	 * it back to the old size.
	 */
	if (ip->i_size != oldsize) {
		rc2 = xtTruncate(0, ip, oldsize, COMMIT_WMAP);
		assert(rc2 == 0);
	}

	return (rc);
}
#endif				/* ! _JFS_4K */


/*
 * NAME:        WRClean()
 *
 * FUNCTION:    clean up after a write request exception.
 *
 * PARAMETERS:
 *      ip        - the inode of the file.
 *      oldsize   - starting file size.
 *      bmoved    - TRUE is some of the write has completed; otherwise,
 *                  FALSE.
 *      overalloc - TRUE if file is overallocated; otherwise, FALSE.
 *
 * RETURN VALUES: none.
 */
static void
WRClean(struct inode *ip, int64 oldsize, boolean_t bmoved,
	boolean_t overalloc)
{
	int32 rc;

	jFYI(0, ("WRClean called!\n"));

	/* truncate the file back to its old size if no
	 * data was moved but the file was extended (i.e.
	 * PromoteI).
	 */
	if (bmoved == FALSE && ip->i_size != oldsize) {
		rc = xtTruncate(0, ip, oldsize, COMMIT_WMAP);
		assert(rc == 0);
		return;
	}

	/* truncate the file's allocation to i_size if
	 * it is over allocated.
	 */
	if (overalloc) {
		rc = xtTruncate(0, ip, ip->i_size, COMMIT_WMAP);
		assert(rc == 0);
	}

	return;
}
#endif /* kern22 */
