/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_unicode.h>
#include <linux/jfs/jfs_uniupr.h>
#include <linux/jfs/jfs_debug.h>

/*
 * NAME:	jfs_strfromUCS()
 *
 * FUNCTION:	Convert unicode string to character string
 *
 */
int32	jfs_strfromUCS(
char			*to,
const UniChar		*from,
int32			len,
struct nls_table	*codepage)
{
	int	i;
	int	outlen = 0;

	for (i = 0; (i < len) && from[i]; i++) {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,2,16))
		/* 2.2.14 */
		unsigned char *up = codepage->page_uni2charset[from[i] >> 8];
		unsigned char c = 0;
		if (up)
			c = up[from[i] & 0xff];
		if (c)
			to[i] = c;
		else
			to[i] = '?';	/* substitution character */
	}
	outlen = i;
#else
		int charlen;
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,3,0))
		/* 2.4.0 or greater */
		charlen = codepage->uni2char(from[i], &to[outlen],
					     NLS_MAX_CHARSET_SIZE);
#else
		/* 2.2.16 or greater */
		codepage->uni2char(from[i] >> 8, from[i] & 0xff, &to[outlen], 3,
				   &charlen);
#endif

		if(charlen > 0) {
		    outlen += charlen;
		} else {
		    to[outlen++] = '?';
		}
	}
#endif
	to[outlen] = 0;
	jEVENT(0,("jfs_strfromUCS returning %d - '%s'\n", outlen, to));
	return outlen;
}

/*
 * NAME:	jfs_strtoUCS()
 *
 * FUNCTION:	Convert character string to unicode string
 *
 */
int32	jfs_strtoUCS(
UniChar			*to,
const char		*from,
int32			len,
struct nls_table	*codepage)
{
	int	charlen;
	int 	i;

	jEVENT(0,("jfs_strtoUCS - '%s'\n", from));

	for (i = 0; len && *from; i++, from += charlen, len -= charlen) {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,2,16))
	/* 2.2.14 */
		struct nls_unicode uc;

		uc = codepage->charset2uni[(uint)*from];
		to[i] = uc.uni1 | (uc.uni2 << 8);
		charlen = 1;
#else
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0))
	/* 2.2.16 or later */
		unsigned char	uni1, uni2;
		codepage->char2uni((unsigned char *)from, &charlen,
				   &uni1, &uni2);
		to[i] = uni1 | (uni2 << 8);
#else
	/* 2.4.0 or later */
		charlen = codepage->char2uni(from, len, &to[i]);
#endif
		if (charlen < 1)
		{
			jERROR(1,("jfs_strtoUCS: char2uni returned %d.\n",
				  charlen));
			to[i] = 0x003f; /* a question mark */
			charlen = 1;
		}
#endif
	}
	
jEVENT(0,(" returning %d\n", i));

	to[i] = 0;
	return i;
}

/*
 * NAME:	get_UCSname()
 *
 * FUNCTION:	Allocate and translate to unicode string
 *
 */
int get_UCSname (component_t * uniName, struct dentry *dentry,
		 struct nls_table *nls_tab)
{
	int length = dentry->d_name.len;

	if (length> JFS_NAME_MAX)
		return ENAMETOOLONG;

	uniName->name = kmalloc((length + 1) * sizeof(UniChar), GFP_KERNEL);

	if (uniName->name == NULL)
		return ENOSPC;

	uniName->namlen = jfs_strtoUCS(uniName->name, dentry->d_name.name,
				       length, nls_tab);

	return 0;
}
