/*
 *
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: jfs_ftruncate.c
 */

/*
 * Change History :
 *
 */

#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_inode.h>
#include <linux/jfs/jfs_dmap.h>
#include <linux/jfs/jfs_debug.h>

/* forward reference */
int32 iTruncate(int32, struct inode *, int64);

/*
 * NAME:	jfs_ftruncate(ip, length)
 *
 * FUNCTION:	truncate regular file <vp> to the specified size <length>.
 *
 * PARAMETER:	vp 	_ file to be truncated.
 *		length	- new length
 *
 * RETURN:	
 *
 * note: EINVAL: JFS does NOT support ftruncate() on file types
 * other than regular filei: the effect of ftruncate()/truncate()
 * on file types other than regular file is unspecified. [XPG4.2].
 *
 * Linux does not call truncate() to grow a file.  So we'll always try
 * to shrink the file.
 */
int jfs_ftruncate( struct inode *ip,
		  loff_t length)
{
	int32 rc = 0;
	int32 tid = 0;

	if (!S_ISREG(ip->i_mode))
		return EINVAL;

	IWRITE_LOCK(ip);

	ASSERT(length >= 0);
	/*
	 *      truncate down
	 */
	txBegin(ip->i_sb, &tid, 0);

	if ((rc = iTruncate(tid, ip, (int64) length))) {
		txEnd(tid);
		goto out;
	}
	ip->i_jfs_flag |= IFSYNC;

	rc = txCommit(tid, 1, &ip, 0);

	txEnd(tid);

	ip->i_mtime = ip->i_ctime = CURRENT_TIME;
	mark_inode_dirty(ip);

      out:
	IWRITE_UNLOCK(ip);

	return rc;
}


/*
 * NAME:	iTruncate(tid, ip, newsize)
 *
 * FUNCTION:    truncate up/down a regular file to specified size, or 
 *		truncate down directory or symbolic link to zero length 
 *		(length is ignored and assumed to be zero if the object
 * 		is a directory or symbolic link). 
 *		if length is > 0, the file must be open (i.e.bound to a 
 *		VM segment). 
 *		return 0 if type is not one of these.
 *
 * PARAMETER:	ip	- inode to truncate
 *		newsize	- new size of the file
 *
 * RETURN:
 *			
 * SERIALIZATION: the IWRITE_LOCK is held on entry/exit.
 */
int32 iTruncate(int32 tid, struct inode * ip, int64 newsize)
{
	int32 rc;
/*
printf("iTruncate: ip:0x%p eof:0x%llx\n", ip, newsize); 
*/
	if (!S_ISREG(ip->i_mode))
		return EINVAL;

	/*
	 * if the newsize is not an integral number of pages,
	 * the file between newsize and next page boundary will 
	 * be cleared.
	 * if truncating into a file hole, it will cause
	 * a full block to be allocated for the logical block.
	 */

	/*
	 * if the file was commited with zero link count before
	 * (temporary file), its persistent resources were already
	 * freed at that time: just truncate/free working resources;
	 */
	if (ip->i_jfs_cflag & COMMIT_NOLINK) {
		rc = xtTruncate(0, ip, newsize, COMMIT_WMAP);
		return rc;
	}

	/*
	 * delete pages and set new size.
	 */
	rc = xtTruncate(tid, ip, newsize, COMMIT_TRUNCATE | COMMIT_PWMAP);

	return rc;
}
