/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/locks.h>
#include <linux/errno.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_debug.h>

ssize_t jfs_file_write(struct file *, const char *, size_t, loff_t *);
int jfs_ftruncate(struct inode *, loff_t);

#ifndef kern22
int jfs_fsync_inode(struct inode *, int);

int jfs_fsync_file(struct file *file, struct dentry *dentry, int datasync)
{
	return jfs_fsync_inode(dentry->d_inode, datasync);
}
#endif

#ifdef kern22
static struct file_operations jfs_file_operations = {
	write:		jfs_file_write,	
	fsync:		file_fsync,
#else
struct file_operations jfs_file_operations = {
	write:		generic_file_write,	
	fsync:		jfs_fsync_file,
#endif
	read:		generic_file_read,
	mmap:		generic_file_mmap,
};

#ifdef kern22
static int jfs_bmap(
struct inode	*ip,
int		block)
{
	int64	xaddr;
	uint8	xflag;
	int32	xlen;

	if (xtLookup(ip, block << ip->i_sb->s_jfs_l2nbperpage,
		     ip->i_sb->s_jfs_nbperpage, &xflag, &xaddr, &xlen, 0) ||
	    (xlen == 0))
		return 0;

	jFYI(1,("jfs_bmap mapped block 0x%x to 0x%lx\n", block, (ulong)xaddr));

	/*
	 * If the block is "allocated but not recorded", it behaves the same
	 * as an unallocated hole
	 */
	if (xflag & XAD_NOTRECORDED)
		return 0;

	return ((int)xaddr >> ip->i_sb->s_jfs_l2nbperpage);
}
#endif /* kern22 */

static void jfs_truncate(struct inode *ip)
{
	jFYI(1,("jfs_truncate: size = 0x%lx\n", (ulong)ip->i_size));

	(void)jfs_ftruncate(ip, ip->i_size);
}

struct inode_operations jfs_file_inode_operations = {
#ifdef kern22
        default_file_ops:	&jfs_file_operations,
	readpage:		generic_readpage,
	bmap:			jfs_bmap,
#endif
	truncate:		jfs_truncate,
};
