// -*- c++ -*-
// Copyright (C) 2001 Dr.-Ing. Martin Froehlich <Martin.Froehlich@mediaWays.net>
//  
// PURPOSE OF THIS FILE: 
//   Provides the LDAP search functions based on the RFC 1823 LDAP-API
//  
// - Automatic Version Information via CVS/RCS:
//   $Id: ldaplink.h,v 1.12.2.3 2004/05/12 14:00:46 zvision Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/Attic/ldaplink.h,v $
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#if !defined(LDAPLINK_H)	/* make idempotent */
#define LDAPLINK_H "@(#) $Id: ldaplink.h,v 1.12.2.3 2004/05/12 14:00:46 zvision Exp $"

#include <sys/time.h>

#include <ldap.h>		// std RFC 1823 LDAP-API

#include <ptlib.h>		// the PWlib

#ifdef P_SOLARIS
#define map stl_map
#endif

#include <map>			// STL map
#include <vector>               // STL vector
#include "gk_ldap_interface.h"

/** Class that holds the current names of the attribute names used for the
    LDAP access 
*/
using std::map;
typedef std::map<PString, PString> LDAPAttributeNamesClass;
typedef LDAPAttributeNamesClass::value_type LDAPANValuePair;

// LDAP config tags and names
namespace lctn {
	/// tags named after config file tags, used as indices to LDAPAttrTags
enum LDAPAttributeNamesEnum {
	DN=0, H323ID, TelephonNo, H235PassWord, IPAddress, SubscriberNo, 
	LocalAccessCode, NationalAccessCode, InternationalAccessCode, 
	CallingLineIdRestriction, SpecialDial, PrefixBlacklist, PrefixWhitelist, MAX_ATTR_NO };

	/// list of names (keys) as used in config file
extern const char * LDAPAttrTags[MAX_ATTR_NO];
}


typedef std::map<PString, PStringList> LDAPAttributeValueClass;
typedef LDAPAttributeValueClass::value_type LDAPAVValuePair;

/** Class that contains search queries
 */
class LDAPQuery {
public:
	enum LDAPQueryOp {
		LDAPand, LDAPor, LDAPnot, LDAPNONE };
	unsigned LDAPOperator;
	LDAPAttributeValueClass LDAPAttributeValues;
	LDAPQuery(): LDAPOperator(LDAPor) {}; // Default is "or"
};

typedef std::map<PString, LDAPAttributeValueClass> LDAPEntryClass;
typedef LDAPEntryClass::value_type LDAPECValuePair;
/** Class that contains search answers
 */
class LDAPAnswer {
public:
	LDAPAnswer();
	virtual ~LDAPAnswer();
	int status;			// as LDAP.ld_errno
	LDAPEntryClass LDAPec;	// the attributes and their values
	virtual bool complete(void);	// test if this is all we need
};

class LDAPCtrl {
public:
	LDAPCtrl(LDAPAttributeNamesClass *, // the Attribute names
		 struct timeval ,	// the devault timeout for *_st operations
		 PString &,		// Name of the LDAP Server
		 PString &,		// Distinguished Name (DN) from where to search
		 PString &,		// UserDN of acting user
		 PString &,		// Pasword for simple auth. of BindUserDN
		 unsigned int,	// 0 for no cache (default 0)
		 unsigned int,	// timeout in seconds (default 10)
		 int			// Port of the LDAP Server (default IANA port)
		); 
	virtual ~LDAPCtrl();

	// searching for user accreditation
	virtual LDAPAnswer * DirectoryUserLookup(const PString &); 
	virtual LDAPAnswer * DirectoryLookup(LDAPQuery &); // general lookup
	virtual LDAPAnswer * collectAttributes(LDAPQuery &, PStringList);


protected:
	// Some of this data might look superflous, but experience teaches to
	// keep the connection details. At least they come handy during a
	// debugging session
	LDAPAttributeNamesClass * AttributeNames; // names of the LDAP attributes
	struct timeval timeout;	// timeout for *_st operations
	PString ServerName;		// Name of the LDAP Server
	int ServerPort;		// Port of the LDAP Server
	PString SearchBaseDN;		// Distinguished Name (DN) from where to search
	PString BindUserDN;		// UserDN of acting user
	PString BindUserPW;		// Pasword for simple auth. of BindUserDN
	unsigned int sizelimit;	// size of local cache in bytes
	unsigned int timelimit;	// timeout for operations in seconds
private:
	GK_LDAP * ldap;			// The ldap connection
	bool known_to_be_bound;	// _known_ status of binding
	void Initialize(void);	// initializer, called from constructors
	void Destroy(void);		// actual destructor called from formal one
	int Bind(bool);		// binding, may be enforced by passing true
	int Unbind(bool);		// unbinding, may be enforced by passing true

};

#endif /* defined(LDAPLINK_H) */

//
// End of ldaplink.h
//
