/*
 * gkacct.cxx
 *
 * Accounting modules for GNU Gatekeeper. Provides generic
 * support for accounting to the gatekeeper.
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: gkacct.cxx,v $
 * Revision 1.1.2.13.2.1  2004/05/31 22:38:42  zvision
 * Fixed CDR file rotation per number of lines
 *
 * Revision 1.1.2.13  2004/05/22 12:12:42  zvision
 * Parametrized FileAcct CDR format
 *
 * Revision 1.1.2.12  2004/05/20 12:37:51  zvision
 * Timer expiration time adjustment, if it is in the past
 *
 * Revision 1.1.2.11  2004/05/12 14:00:42  zvision
 * Header file usage more consistent. Solaris std::map problems fixed.
 * Compilation warnings removed. VSNET2003 project files added. ANSI.h removed.
 *
 * Revision 1.1.2.10  2004/05/10 16:45:05  zvision
 * New flexible CDR file rotation for FileAcct module
 *
 * Revision 1.1.2.9  2003/12/26 14:00:54  zvision
 * Fixed VC6 warning about too long browser/debug symbols
 *
 * Revision 1.1.2.8  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.7  2003/11/01 10:35:47  zvision
 * Fixed missing semicolon. Thanks to Hu Yuxin
 *
 * Revision 1.1.2.6  2003/10/27 20:27:52  zvision
 * Improved handling of multiple accounting modules and better tracing
 *
 * Revision 1.1.2.5  2003/10/07 15:22:28  zvision
 * Added support for accounting session updates
 *
 * Revision 1.1.2.4  2003/09/28 14:09:48  zvision
 * Microsecond field added back to h323-xxx-time attributes
 *
 * Revision 1.1.2.3  2003/09/18 01:18:24  zvision
 * Merged accounting code parts from 2.2
 *
 * Revision 1.1.2.2  2003/09/14 21:13:06  zvision
 * Added new FileAcct logger from Tamas Jalsovszky. Thanks!
 * Fixed module stacking. API redesigned.
 *
 * Revision 1.1.2.1  2003/06/19 15:36:04  zvision
 * Initial generic accounting support for GNU GK.
 *
 */
#ifdef HAS_ACCT

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <h225.h>
#include <map>
#include "gk_const.h"
#include "h323util.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "gktimer.h"
#include "gkacct.h"

namespace {
/// name of the config file section for accounting configuration
const char* const GkAcctSectionName = "Gatekeeper::Acct";
}

PString GkAcctLogger::AsString(
	const PTime& tm
	)
{
	struct tm _tm;
	struct tm* tmptr = &_tm;
	time_t t;
	
	if( (time(&t) != (time_t)(-1))
#ifndef WIN32
		&& (localtime_r(&t,tmptr) == tmptr) )
#else
		&& ((tmptr = localtime(&t)) != NULL) )
#endif
	{
		char buf[10];
		
		buf[0] = 0;
		if( strftime(buf,sizeof(buf),"%Z",tmptr) < 10 )
		{
			buf[9] = 0;
			const PString tzname(buf);
			if( !tzname.IsEmpty() )
			{
				PString s = tm.AsString( "hh:mm:ss.uuu @@@ www MMM d yyyy" );
				s.Replace("@@@",tzname);
				return s;
			}
		}
	}
	
	return tm.AsString( "hh:mm:ss.uuu z www MMM d yyyy" );
}

PString GkAcctLogger::AsString(
	const time_t& tm
	)
{
	struct tm _tm;
	struct tm* tmptr = &_tm;
	time_t t = tm;
	
#ifndef WIN32
	if( localtime_r(&t,tmptr) == tmptr ) {
		char buf[48];
		size_t sz = strftime(buf,sizeof(buf),"%T.000 %Z %a %b %d %Y",tmptr);
#else
	if( (tmptr = localtime(&t)) != NULL ) {
		char buf[96];
		size_t sz = strftime(buf,sizeof(buf),"%H:%M:%S.000 %Z %a %b %d %Y",tmptr);
#endif
		if( sz < sizeof(buf) && sz > 0 )
			return buf;
	}
	
	return PTime(tm).AsString( "hh:mm:ss.uuu z www MMM d yyyy" );
}

GkAcctLogger::GkAcctLogger(
	PConfig& cfg, 
	const PString& moduleName,
	const char* cfgSecName
	) 
	: 
	controlFlag(Required),
	defaultStatus(Fail),
	enabledEvents(AcctAll),
	supportedEvents(AcctNone),
	name(moduleName),
	configSectionName(cfgSecName)
{
	if( configSectionName.IsEmpty() )
		configSectionName = moduleName;
		
	const PStringArray control( 
		cfg.GetString( GkAcctSectionName, moduleName, "" ).Tokenise(";,")
		);

	if( control.GetSize() < 1 )
		PTRACE(1,"GKACCT\tEmpty config entry for module "<<moduleName);
	else if( moduleName *= "default" ) {
		controlFlag = Required;
		defaultStatus = Toolkit::AsBool(control[0]) ? Ok : Fail;
		supportedEvents = AcctAll;
	} else if (control[0] *= "optional")
		controlFlag = Optional;
	else if (control[0] *= "sufficient")
		controlFlag = Sufficient;
	else if (control[0] *= "alternative")
		controlFlag = Alternative;
	
	if( control.GetSize() > 1 )
		enabledEvents = GetEvents(control);
	
	PTRACE(1,"GKACCT\tCreated module "<<moduleName<<" with event mask "
		<<PString(PString::Unsigned,(long)enabledEvents,16)
		);
}

GkAcctLogger::~GkAcctLogger()
{
	PTRACE(1,"GKACCT\tDestroyed module "<<name);
}

int GkAcctLogger::GetEvents(
	const PStringArray& tokens
	) const
{
	int mask = 0;
	
	for( PINDEX i = 1; i < tokens.GetSize(); i++ )
	{
		const PString& token = tokens[i];
		if( token *= "start" )
			mask |= AcctStart;
		else if( token *= "stop" )
			mask |= AcctStop;
		else if( token *= "update" )
			mask |= AcctUpdate;
		else if( token *= "on" )
			mask |= AcctOn;
		else if( token *= "off" )
			mask |= AcctOff;
	}
	
	return mask;
}

GkAcctLogger::Status GkAcctLogger::LogAcctEvent(
	AcctEvent evt, /// accounting event to log
	callptr& call /// additional data for the event
	)
{
	return (evt & enabledEvents & supportedEvents) ? defaultStatus : Next;
}

const char* const FileAcct::m_intervalNames[] =
{
	"Hourly", "Daily", "Weekly", "Monthly"
};

FileAcct::FileAcct( 
	PConfig& cfg,
	const PString& moduleName,
	const char* cfgSecName
	)
	:
	GkAcctLogger(cfg, moduleName, cfgSecName),
	m_cdrFile(NULL), m_rotateLines(-1), m_rotateSize(-1), m_rotateInterval(-1),
	m_rotateMinute(-1), m_rotateHour(-1), m_rotateDay(-1), 
	m_rotateTimer(GkTimerManager::INVALID_HANDLE), m_cdrLines(0),
	m_standardCDRFormat(true), m_gkName(Toolkit::Instance()->GKName())
{
	SetSupportedEvents(FileAcctEvents);

	m_cdrString = cfg.GetString(GetConfigSectionName(), "CDRString", "");
	m_standardCDRFormat = Toolkit::AsBool(
		cfg.GetString(GetConfigSectionName(), "StandardCDRFormat", 
			m_cdrString.IsEmpty() ? "1" : "0"
			));

	// determine rotation type (by lines, by size, by time)	
	const PString rotateCondition = cfg.GetString(
		GetConfigSectionName(), "Rotate", ""
		).Trim();
	if (!rotateCondition) {
		const char suffix = rotateCondition[rotateCondition.GetLength()-1];
		if (rotateCondition[0] == 'L' || rotateCondition[0] == 'l') {
			// rotate per number of lines
			m_rotateLines = rotateCondition.Mid(1).AsInteger();
			if (suffix == 'k' || suffix == 'K')
				m_rotateLines *= 1000;
			else if (suffix == 'm' || suffix == 'M')
				m_rotateLines *= 1000*1000;
		} else if (rotateCondition[0] == 'S' || rotateCondition[0] == 's') {
			// rotate per CDR file size
			m_rotateSize = rotateCondition.Mid(1).AsInteger(); 
			if (suffix == 'k' || suffix == 'K')
				m_rotateSize *= 1024;
			else if (suffix == 'm' || suffix == 'M')
				m_rotateSize *= 1024*1024;
		} else {
			for (int i = 0; i < RotationIntervalMax; i++)
				if (strcasecmp(rotateCondition, m_intervalNames[i]) == 0)
					m_rotateInterval = i;

			if (m_rotateInterval < 0 || m_rotateInterval >= RotationIntervalMax)
				PTRACE(1, "GKACCT\t" << GetName() << " unsupported rotation "
					"method: " << rotateCondition << " - rotation disabled"
					);
			else {
				// time based rotation
				GetRotateInterval(cfg, GetConfigSectionName());
			}
		}
	}

	m_cdrFilename = cfg.GetString(GetConfigSectionName(), "DetailFile", "");
	m_cdrFile = OpenCDRFile(m_cdrFilename);
	if (m_cdrFile && m_cdrFile->IsOpen()) {
		PTRACE(2, "GKACCT\t" << GetName() << " CDR file: "
			<< m_cdrFile->GetFilePath()
			);
		// count an initial number of CDR lines
		if (m_rotateLines > 0) {
			PString s;
			m_cdrFile->SetPosition(0);
			while (m_cdrFile->ReadLine(s))
				m_cdrLines++;
			m_cdrFile->SetPosition(m_cdrFile->GetLength());
		}
	}

	// setup rotation timer in case of time based rotation
	PTime now, rotateTime;
		
	switch (m_rotateInterval) 
	{
	case Hourly:
		rotateTime = PTime(0, m_rotateMinute, now.GetHour(), now.GetDay(), 
			now.GetMonth(), now.GetYear(), now.GetTimeZone()
			);
		if (rotateTime <= now)
			rotateTime += PTimeInterval(0, 0, 0, 1); // 1 hour
		m_rotateTimer = Toolkit::Instance()->GetTimerManager()->RegisterTimer(
			this, &FileAcct::RotateOnTimer, rotateTime, 60*60
			);
		PTRACE(5, "GKACCT\t" << GetName() << " hourly rotation enabled (first "
			"rotation sheduled at " << rotateTime
			);
		break;
		
	case Daily:
		rotateTime = PTime(0, m_rotateMinute, m_rotateHour, now.GetDay(), 
			now.GetMonth(), now.GetYear(), now.GetTimeZone()
			);
		if (rotateTime <= now)
			rotateTime += PTimeInterval(0, 0, 0, 0, 1); // 1 day
		m_rotateTimer = Toolkit::Instance()->GetTimerManager()->RegisterTimer(
			this, &FileAcct::RotateOnTimer, rotateTime, 60*60*24
			);
		PTRACE(5, "GKACCT\t" << GetName() << " daily rotation enabled (first "
			"rotation sheduled at " << rotateTime
			);
		break;
		
	case Weekly:
		rotateTime = PTime(0, m_rotateMinute, m_rotateHour, now.GetDay(), 
			now.GetMonth(), now.GetYear(), now.GetTimeZone()
			);
		if (rotateTime.GetDayOfWeek() < m_rotateDay)
			rotateTime += PTimeInterval(0, 0, 0, 0,
				m_rotateDay - rotateTime.GetDayOfWeek() /* days */
				);
		else if (rotateTime.GetDayOfWeek() > m_rotateDay)
			rotateTime -= PTimeInterval(0, 0, 0, 0,
				rotateTime.GetDayOfWeek() - m_rotateDay /* days */
				);
		if (rotateTime <= now)
			rotateTime += PTimeInterval(0, 0, 0, 0, 7); // 1 week
		m_rotateTimer = Toolkit::Instance()->GetTimerManager()->RegisterTimer(
			this, &FileAcct::RotateOnTimer, rotateTime, 60*60*24*7
			);
		PTRACE(5, "GKACCT\t" << GetName() << " weekly rotation enabled (first "
			"rotation sheduled at " << rotateTime
			);
		break;
		
	case Monthly:
		rotateTime = PTime(0, m_rotateMinute, m_rotateHour, 1, 
			now.GetMonth(), now.GetYear(), now.GetTimeZone()
			);
		rotateTime += PTimeInterval(0, 0, 0, 0, m_rotateDay - 1);
		while (rotateTime.GetMonth() != now.GetMonth())				
			rotateTime -= PTimeInterval(0, 0, 0, 0, 1); // 1 day
		m_rotateTimer = Toolkit::Instance()->GetTimerManager()->RegisterTimer(
			this, &FileAcct::RotateOnTimer, rotateTime
			);
		PTRACE(5, "GKACCT\t" << GetName() << " monthly rotation enabled (first "
			"rotation sheduled at " << rotateTime
			);
		break;
	}
}

FileAcct::~FileAcct()
{
	if (m_rotateTimer != GkTimerManager::INVALID_HANDLE)
		Toolkit::Instance()->GetTimerManager()->UnregisterTimer(m_rotateTimer);
		
	PWaitAndSignal lock(m_cdrFileMutex);
	if (m_cdrFile) {
		m_cdrFile->Close();
		delete m_cdrFile;
	}
}

void FileAcct::GetRotateInterval(
	PConfig& cfg,
	const PString& section
	)
{
	PString s;
	
	if (m_rotateInterval == Hourly)
		m_rotateMinute = cfg.GetInteger(section, "RotateTime", 59);
	else {
		s = cfg.GetString(section, "RotateTime", "00:59");
		m_rotateHour = s.AsInteger();
		m_rotateMinute = 0;
		if (s.Find(':') != P_MAX_INDEX)
			m_rotateMinute = s.Mid(s.Find(':') + 1).AsInteger();
			
		if (m_rotateHour < 0 || m_rotateHour > 23 || m_rotateMinute < 0
			|| m_rotateMinute > 59) {
			PTRACE(1, "GKACCT\t" << GetName() << " invalid "
				"RotateTime specified: " << s
				);
			m_rotateMinute = 59;
			m_rotateHour = 0;
		}
	}
			
	if (m_rotateInterval == Weekly)	{
		s = cfg.GetString(section, "RotateDay", "Sun");
		if (strspn(s, "0123456") == (size_t)s.GetLength()) {
			m_rotateDay = s.AsInteger();
		} else {
			std::map<PCaselessString, int> dayNames;
			dayNames["sun"] = 0; dayNames["sunday"] = 0;
			dayNames["mon"] = 1; dayNames["monday"] = 1;
			dayNames["tue"] = 2; dayNames["tuesday"] = 2;
			dayNames["wed"] = 3; dayNames["wednesday"] = 3;
			dayNames["thu"] = 4; dayNames["thursday"] = 4;
			dayNames["fri"] = 5; dayNames["friday"] = 5;
			dayNames["sat"] = 6; dayNames["saturday"] = 6;
			std::map<PCaselessString, int>::const_iterator i = dayNames.find(s);
			m_rotateDay = (i != dayNames.end()) ? i->second : -1;
		}
		if (m_rotateDay < 0 || m_rotateDay > 6) {
			PTRACE(1, "GKACCT\t" << GetName() << " invalid "
				"RotateDay specified: " << s
				);
			m_rotateDay = 0;
		}
	} else if (m_rotateInterval == Monthly) {
		m_rotateDay = cfg.GetInteger(section, "RotateDay", 1);
		if (m_rotateDay < 1 || m_rotateDay > 31) {
			PTRACE(1, "GKACCT\t" << GetName() << " invalid "
				"RotateDay specified: " << cfg.GetString(section, "RotateDay", "")
				);
			m_rotateDay = 1;
		}
	}
}

GkAcctLogger::Status FileAcct::LogAcctEvent(
	GkAcctLogger::AcctEvent evt, 
	callptr& call
	)
{
	if ((evt & GetEnabledEvents() & GetSupportedEvents()) == 0)
		return Next;
		
	if (!call) {
		PTRACE(1, "GKACCT\t" << GetName() << " - missing call info for event"
			<< evt
			);
		return Fail;
	}
	
	PString cdrString;
	
	if (!GetCDRText(cdrString, evt, call)) {
		PTRACE(2, "GKACCT\t" << GetName() << " - unable to get CDR text for "
			"event " << evt << ", call no. " << call->GetCallNumber()
			);
		return Fail;
	}

	PWaitAndSignal lock(m_cdrFileMutex);
	
	if (m_cdrFile && m_cdrFile->IsOpen()) {
		if (m_cdrFile->WriteLine(PString(cdrString))) {
			PTRACE(5, "GKACCT\t" << GetName() << " - CDR string for event "
				<< evt << ", call no. " << call->GetCallNumber() 
				<< ": " << cdrString
				);
			m_cdrLines++;
			if (IsRotationNeeded())
				Rotate();
			return Ok;
		} else
			PTRACE(1, "GKACCT\t" << GetName() << " - write CDR text for event "
				<< evt << ", call no. " << call->GetCallNumber()
				<< " failed: " << m_cdrFile->GetErrorText()
				);
	} else
		PTRACE(1, "GKACCT\t" << GetName() << " - write CDR text for event "
			<< evt << ", for call no. " << call->GetCallNumber()
			<< " failed: CDR file is closed"
			);
		
	return Fail;
}

bool FileAcct::GetCDRText(
	PString& cdrString,
	AcctEvent evt,
	callptr& call
	)
{
	if ((evt & AcctStop) != AcctStop || !call)
		return false;
	
	if (m_standardCDRFormat)	
		cdrString = call->GenerateCDR();
	else {
		std::map<PString, PString> params;

		SetupCDRParams(params, call);
		cdrString = ReplaceCDRParams(m_cdrString, params);
	}	
	
	return !cdrString;
}

void FileAcct::SetupCDRParams(
	/// CDR parameters (name => value) associations
	std::map<PString, PString>& params,
	/// call (if any) associated with an accounting event being logged
	callptr& call
	) const
{
	PIPSocket::Address addr;
	WORD port = 0;
	time_t t;
	
	params["g"] = m_gkName;
	params["n"] = PString(call->GetCallNumber());
	params["d"] = call->GetDuration();
	params["c"] = call->GetDisconnectCause();
	params["s"] = call->GetAcctSessionId();
	params["CallId"] = ::AsString(call->GetCallIdentifier().m_guid);
	params["ConfId"] = ::AsString(call->GetConferenceIdentifier());
	
	t = call->GetSetupTime();
	if (t)
		params["setup-time"] = AsString(t);
	t = call->GetConnectTime();
	if (t)
		params["connect-time"] = AsString(t);
	t = call->GetDisconnectTime();
	if (t)
		params["disconnect-time"] = AsString(t);
	
	if (call->GetSrcSignalAddr(addr, port)) {
		params["caller-ip"] = addr.AsString();
		params["caller-port"] = port;
	}
	
	PString srcInfo = call->GetSourceInfo();
	params["src-info"] = srcInfo;

	// Get User-name
	if (!(srcInfo.IsEmpty() || srcInfo == "unknown")) {
		const PINDEX index = srcInfo.FindOneOf(":");
		if( index != P_MAX_INDEX )
			srcInfo = srcInfo.Left(index);
	}
	
	endptr callingEP = call->GetCallingParty();
	PString userName;
		
	if (callingEP && callingEP->GetAliases().GetSize() > 0)
		userName = GetBestAliasAddressString(
			callingEP->GetAliases(),
			H225_AliasAddress::e_h323_ID
			);
	else if (!srcInfo)
		userName = srcInfo;
	else if (addr.IsValid())
		userName = addr.AsString();
		
	if (!userName)
		params["u"] = userName;

	PString stationId = call->GetCallingStationId();
		
	if (stationId.IsEmpty() && callingEP && callingEP->GetAliases().GetSize() > 0)
		stationId = GetBestAliasAddressString(
			callingEP->GetAliases(),
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
					
	if (stationId.IsEmpty())
		stationId = srcInfo;
			
	if (stationId.IsEmpty() && addr.IsValid() && port)
		stationId = ::AsString(addr, port);

	if (!stationId)
		params["Calling-Station-Id"] = stationId;
		
	addr = (DWORD)0;
	port = 0;
		
	if (call->GetDestSignalAddr(addr, port)) {
		params["callee-ip"] = addr.AsString();
		params["callee-port"] = port;
	}

	PString destInfo = call->GetDestInfo();
	params["dest-info"] = destInfo;

	stationId = call->GetCalledStationId();
	
	if (stationId.IsEmpty() && !(destInfo.IsEmpty() || destInfo == "unknown")) {
		const PINDEX index = destInfo.FindOneOf(":");
		stationId = (index == P_MAX_INDEX) ? destInfo : destInfo.Left(index);
	}
		
	if (stationId.IsEmpty()) {
		endptr calledEP = call->GetCalledParty();
		if (calledEP && calledEP->GetAliases().GetSize() > 0)
			stationId = GetBestAliasAddressString(
				calledEP->GetAliases(),
				H225_AliasAddress::e_dialedDigits,
				H225_AliasAddress::e_partyNumber,
				H225_AliasAddress::e_h323_ID
				);
	}
	
	if (stationId.IsEmpty() && addr.IsValid() && port)
		stationId = ::AsString(addr, port);
		
	if (!stationId)
		params["Called-Station-Id"] = stationId;
}

PString FileAcct::ReplaceCDRParams(
	/// parametrized CDR string
	const PString& cdrStr,
	/// parameter values
	const std::map<PString, PString>& params
	)
{
	PString finalCDR((const char*)cdrStr);
	PINDEX len = finalCDR.GetLength();
	PINDEX pos = 0;

	while (pos != P_MAX_INDEX && pos < len) {
		pos = finalCDR.Find('%', pos);
		if (pos++ == P_MAX_INDEX)
			break;
		if (pos >= len) // strings ending with '%' - special case
			break;
		const char c = finalCDR[pos]; // char next after '%'
		if (c == '%') { // replace %% with %
			finalCDR.Delete(pos, 1);
			len--;
		} else if (c == '{') { // escaped syntax (%{Name})
			const PINDEX closingBrace = finalCDR.Find('}', ++pos);
			if (closingBrace != P_MAX_INDEX) {
				const PINDEX paramLen = closingBrace - pos;
				std::map<PString, PString>::const_iterator i = params.find(
					finalCDR.Mid(pos, paramLen)
					);
				if (i != params.end()) {
					const PINDEX escapedLen = i->second.GetLength();
					finalCDR.Splice(i->second, pos - 2, paramLen + 3);
					len = len + escapedLen - paramLen - 3;
					pos = pos - 2 + escapedLen;
				} else {
					// replace out of range parameter with an empty string
					finalCDR.Delete(pos - 2, paramLen + 3);
					len -= paramLen + 3;
					pos -= 2;
				}
			}
		} else { // simple syntax (%c)
			std::map<PString, PString>::const_iterator i = params.find(c);
			if (i != params.end()) {
				const PINDEX escapedLen = i->second.GetLength();
				finalCDR.Splice(i->second, pos - 1, 2);
				len = len + escapedLen - 2;
				pos = pos - 1 + escapedLen;
			} else {
				// replace out of range parameter with an empty string
				finalCDR.Delete(pos - 1, 2);
				len -= 2;
				pos--;
			}
		}
	}

	return finalCDR;
}

bool FileAcct::IsRotationNeeded()
{
	if (m_rotateLines > 0 && m_cdrLines >= m_rotateLines)
		return true;
	if (m_rotateSize > 0 && m_cdrFile && m_cdrFile->GetLength() >= m_rotateSize)
		return true;
	return false;
}

void FileAcct::RotateOnTimer(
	GkTimer* timer
	)
{
	if (m_rotateInterval == Monthly) {
		// setup next time for one-shot timer
		const PTime& rotateTime = timer->GetExpirationTime();
		PTime newRotateTime(rotateTime.GetSecond(), rotateTime.GetMinute(),
			rotateTime.GetHour(), 1, 
			rotateTime.GetMonth() < 12 ? rotateTime.GetMonth() + 1 : 1, 
			rotateTime.GetMonth() < 12 ? rotateTime.GetYear() : rotateTime.GetYear() + 1,
			rotateTime.GetTimeZone()
			);
	
		newRotateTime += PTimeInterval(0, 0, 0, 0, m_rotateDay - 1);
		
		const int month = newRotateTime.GetMonth();
		while (newRotateTime.GetMonth() != month)				
			newRotateTime -= PTimeInterval(0, 0, 0, 0, 1); // 1 day
		
		timer->SetExpirationTime(newRotateTime);
		timer->SetFired(false);
	}	
	PWaitAndSignal lock(m_cdrFileMutex);
	Rotate();
}

void FileAcct::Rotate()
{
	if (m_cdrFile) {
		if (m_cdrFile->IsOpen())
			m_cdrFile->Close();
		delete m_cdrFile;
		m_cdrFile = NULL;
	}
	
	const PFilePath fn = m_cdrFilename;
	
	if (PFile::Exists(fn))
		if (!PFile::Rename(fn, fn.GetFileName() + PTime().AsString(".yyyyMMdd-hhmmss")))
			PTRACE(1, "GKACCT\t" << GetName() << " rotate failed - could not "
				"rename the log file"
				);
	
	m_cdrFile = OpenCDRFile(fn);
	m_cdrLines = 0;
}

PTextFile* FileAcct::OpenCDRFile(
	const PFilePath& fn
	)
{
	PTextFile* cdrFile = new PTextFile(fn, PFile::ReadWrite, 
		PFile::Create | PFile::DenySharedWrite
		);
	if (!cdrFile->IsOpen()) {
   	    PTRACE(1, "GKACCT\t" << GetName() << " could not open file"
			" required for plain text accounting \""
			<< fn << "\" :" << cdrFile->GetErrorText()
			);
		delete cdrFile;
	    return NULL;
	}
	cdrFile->SetPermissions(PFileInfo::UserRead | PFileInfo::UserWrite);
	cdrFile->SetPosition(cdrFile->GetLength());
	return cdrFile;
}


GkAcctFactoryBase::GkAcctFactoryArray* GkAcctFactoryBase::acctLoggerFactories = NULL;

GkAcctFactoryBase::GkAcctFactoryBase(
	const PString& moduleName
	) 
	: 
	name(moduleName)
{
	static GkAcctFactoryArray factories;
	acctLoggerFactories = &factories;
	
	PINDEX i;
	
	for( i = 0; i < acctLoggerFactories->GetSize(); i++ )
	{
		GkAcctFactoryBase *const factory = acctLoggerFactories->GetAt(i);
		if( factory && ((factory == this) || (factory->GetName() == moduleName)) )
		{
			PTRACE(1,"GKACCT\tGlobal list of modules already contains"
				" a factory for "<<name
				);
			break;
		}
	}
	
	if( i >= acctLoggerFactories->GetSize() )
	{
		acctLoggerFactories->SetAt(i,this);
		PTRACE(3,"GKACCT\tRegistered factory for "<<name);
	}
}

GkAcctFactoryBase::~GkAcctFactoryBase()
{
}

PObject::Comparison GkAcctFactoryBase::Compare( 
	const PObject& obj
	) const
{
	return name.Compare(((const GkAcctFactoryBase&)obj).name);
}

GkAcctFactoryBase* GkAcctFactoryBase::FindFactoryFor(
	const PString& moduleName
	)
{
	if( acctLoggerFactories != NULL )
		for( PINDEX i = 0; i < acctLoggerFactories->GetSize(); i++ )
		{
			GkAcctFactoryBase *const factory = acctLoggerFactories->GetAt(i);
			if( factory && (factory->GetName() == moduleName) )
				return factory;
		}
	return NULL;
}

GkAcctLoggers::GkAcctLoggers(
	PConfig& cfg
	)
{
	const PStringArray acctLoggerNames( 
		cfg.GetKeys(GkAcctSectionName) 
		);

	for( PINDEX i = 0; i < acctLoggerNames.GetSize(); i++ ) 
	{
		GkAcctFactoryBase* acctFactory 
			= GkAcctFactoryBase::FindFactoryFor(acctLoggerNames[i]);

		if( acctFactory != NULL )
			Append(acctFactory->CreateAcctLogger(cfg));
		else
			PTRACE(1,"GKACCT\tUnknown accounting logger module name "
				<<acctLoggerNames[i]<< " - not created"
				);
	}
	
	acctUpdateInterval = cfg.GetInteger(CallTableSection,"AcctUpdateInterval",0);
}

GkAcctLoggers::~GkAcctLoggers()
{
}

BOOL GkAcctLoggers::LogAcctEvent( 
	GkAcctLogger::AcctEvent evt, /// accounting event to log
	callptr& call, /// additional data for the event
	const time_t now
	)
{
	// check if the acct update logging is enabled 
	// and update interval has passed
	if( evt & GkAcctLogger::AcctUpdate ) {
		if( !call )
			return FALSE;
		if( acctUpdateInterval == 0 
			|| (now - call->GetLastAcctUpdateTime()) < acctUpdateInterval )
			return TRUE;
		else
			call->SetLastAcctUpdateTime(now);
	}
	
	BOOL finalResult = TRUE;
	GkAcctLogger::Status status = GkAcctLogger::Ok;

	// log the event with all configured modules
	for( PINDEX i = 0; i < GetSize(); i++ ) {
		GkAcctLogger* logger = (GkAcctLogger*)GetAt(i);
		if( logger == NULL ) {
			PTRACE(1,"GKACCT\tNULL logger on the list");
			return FALSE;
		}
		
		if( (evt & logger->GetEnabledEvents() & logger->GetSupportedEvents()) == 0 )
			continue;
			
		switch( status = logger->LogAcctEvent( evt, call ) )
		{
		case GkAcctLogger::Ok:
#if PTRACING
			if( PTrace::CanTrace(3) ) {
				ostream& strm = PTrace::Begin(3,__FILE__,__LINE__);
				strm<<"GKACCT\t"<<logger->GetName()<<" logged event "<<evt;
				if( call )
					strm<<" for call no. "<<call->GetCallNumber();
				PTrace::End(strm);
			}
#endif
			break;
			
		default:
			// required and sufficient rules always determine 
			// status of the request
			if( logger->GetControlFlag() == GkAcctLogger::Required
				|| logger->GetControlFlag() == GkAcctLogger::Sufficient ) {
#if PTRACING
				if( PTrace::CanTrace(3) ) {
					ostream& strm = PTrace::Begin(3,__FILE__,__LINE__);
					strm<<"GKACCT\t"<<logger->GetName()<<" failed to log event "<<evt;
					if( call )
						strm<<" for call no. "<<call->GetCallNumber();
					PTrace::End(strm);
				}
#endif
				finalResult = FALSE;
			}
		}
		
		// sufficient and alternative are terminal rules (on log success)
		if( status == GkAcctLogger::Ok
			&& (logger->GetControlFlag() == GkAcctLogger::Sufficient
			|| logger->GetControlFlag() == GkAcctLogger::Alternative) )
			break;
	}

	// a last rule determine status of the the request
	if( finalResult && status != GkAcctLogger::Ok )
		finalResult = FALSE;
		
#if PTRACING
	if( PTrace::CanTrace(2) ) {
		ostream& strm = PTrace::Begin(2,__FILE__,__LINE__);
		strm<<"GKACCT\t"<<(finalResult?"Successfully logged event ":"Failed to log event ")
			<<evt;
		if( call )
			strm<<" for call no. "<<call->GetCallNumber();
		PTrace::End(strm);
	}
#endif
	return finalResult;
}

namespace {
// append the "default" rule for accounting
GkAcctFactory<GkAcctLogger> DefaultAcctFactory("default");
// append file based accounting logger to the global list of loggers
GkAcctFactory<FileAcct> FileAcctFactory("FileAcct");
}

#endif /* HAS_ACCT */
