/* FreeTDS - Library of routines accessing Sybase and Microsoft databases
 * Copyright (C) 1998-1999  Brian Bruns
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <limits.h>
#include <assert.h>
#include <ctype.h>
#ifdef __FreeBSD__
#include <sys/time.h>
#endif

#include "tdsutil.h"


static char  software_version[]   = "$Id: util.c,v 1.7 1999/06/13 00:07:19 camber Exp $";
static void *no_unused_var_warn[] = {software_version,
                                     no_unused_var_warn};


static char interf_file[MAXPATH];

void tds_set_parent(TDSSOCKET* tds, void* the_parent)
{
	if (tds)
      		tds->parent = the_parent;
}

void* tds_get_parent(TDSSOCKET* tds)
{
      return( tds->parent);
}

int set_interfaces_file_loc(char *interf)
{
	if (strlen(interf)>MAXPATH) return 0;

	strcpy(interf_file,interf);
	return 1; /* SUCCEED */
}



/* ============================== lookup_host() ==============================
 *
 * Def:  Given a servername and port name or number, lookup the
 *       hostname and service.  The server ip will be stored in the
 *       string 'servername' in dotted-decimal notation.  The service port
 *       number will be stored in string form in the 'port' parameter.
 *
 *       If we can't determine both the IP address and port number then
 *       'ip' and 'port' will be set to empty strings.
 *
 * Ret:  void
 *
 * ===========================================================================
 */
static void lookup_host(
   const char  *servername,  /* (I) name of the server                  */
   const char  *portname,    /* (I) name or number of the port          */
   char        *ip,          /* (O) dotted-decimal ip address of server */
   char        *port)        /* (O) port number of the service          */
{
   struct hostent   *host    = gethostbyname(servername);
   struct servent   *service = getservbyname(portname, "tcp");
   int               num     = 0;

   if (service==NULL)
   {
      num = atoi(portname);
   }
   else
   {
      num = ntohs(service->s_port);
   }

   if (host==NULL || num==0)
   {
      ip[0]   = '\0';
      port[0] = '\0';
   }
   else
   {
      struct in_addr *ptr = (struct in_addr *) host->h_addr;
      strncpy(ip, inet_ntoa(*ptr), 17);
      sprintf(port, "%d", num);
   }
} /* lookup_host()  */



/* ========================= search_interface_file() =========================
 *
 * Def:  Open and read the file 'file' searching for a logical server
 *       by the name of 'host'.  If one is found then lookup
 *       the IP address and port number and store them in 'ip_addr', and
 *       'ip_port'.
 *
 * Ret:  void
 *
 * ===========================================================================
 */
static void search_interface_file(
   const char *dir,     /* (I) Name of base directory for interface file */
   const char *file,    /* (I) Name of the interface file                */
   const char *host,    /* (I) Logical host to search for                */
   char       *ip_addr, /* (O) dotted-decimal IP address                 */
   char       *ip_port) /* (O) Port number for database server           */
{
   char  pathname[PATH_MAX+1];
   char  line[255];
   char  tmp_ip[sizeof(line)];
   char  tmp_port[sizeof(line)];
   FILE *in;
   char *field;
   int   found=0;

   ip_addr[0]  = '\0';
   ip_port[0]  = '\0';
   line[0]     = '\0';
   tmp_ip[0]   = '\0';
   tmp_port[0] = '\0';

   /*
    * create the full pathname to the interface file
    */
   if (file==NULL || file[0]=='\0')
   {
      pathname[0] = '\0';
   }
   else
   {
      if (dir==NULL || dir[0]=='\0')
      {
         pathname[0] = '\0';
      }
      else
      {
         strncpy(pathname, dir, sizeof(pathname));
         strncat(pathname, "/", sizeof(pathname));
      }
      strncat(pathname, file, sizeof(pathname));
      pathname[sizeof(pathname)-1] = '\0';
   }


   /*
    *  parse the interfaces file and find the server and port
    */
   if ((in = fopen(pathname,"r"))==NULL)
   {
      return;
   }

   while (fgets(line,sizeof(line)-1,in))
   {
      if (line[0]=='#') continue; /* comment */

      if (!isspace(line[0]))
      {
         field = strtok(line,"\n\t ");
         if (!strcmp(field,host)) found=1;
         else found=0;
      }
      else if (found && isspace(line[0]))
      {
         field = strtok(line,"\n\t ");
         if (field!=NULL && !strcmp(field,"query"))
         {
            field = strtok(NULL,"\n\t "); /* tcp */
            field = strtok(NULL,"\n\t "); /* ether */
            field = strtok(NULL,"\n\t "); /* host */

            strcpy(tmp_ip,field);
            field = strtok(NULL,"\n\t "); /* port */
            strcpy(tmp_port,field);
         }
      }
   }
   fclose(in);


   /*
    * Look up the host and service
    */
   lookup_host(tmp_ip, tmp_port, ip_addr, ip_port);
} /* search_interface_file()  */


/* ============================ get_server_info() ============================
 *
 * Def:  Try to find the IP number and port for a (possibly) logical server
 *       name.
 *
 * Note: It is the callers responsibility to supply large enough buffers
 *       to hold the ip and port numbers.  ip_addr should be at least 17
 *       bytes long and ip_port should be at least 6 bytes long.
 *
 * Ret:  True if it found the server, false otherwise.
 *
 * ===========================================================================
 */
int get_server_info(
   char *server,   /* (I) logical or physical server name      */
   char *ip_addr,  /* (O) string representation of IP address  */
   char *ip_port)  /* (O) string representation of port number */
{
   ip_addr[0] = '\0';
   ip_port[0] = '\0';

   /*
    * Look for the server in the interf_file iff interf_file has been set.
    */
   if (ip_addr[0]=='\0' && interf_file[0]!='\0')
   {
      search_interface_file("", interf_file, ip_addr, server, ip_port);
   }

   /*
    * if we haven't found the server yet then look for a $HOME/.interfaces file
    */
   if (ip_addr[0]=='\0')
   {
      char  *home = getenv("HOME");
      if (home!=NULL && home[0]!='\0')
      {
         search_interface_file(home, ".interfaces", server, ip_addr, ip_port);
      }
   }

   /*
    * if we haven't found the server yet then look in $SYBBASE/interfaces file
    */
   if (ip_addr[0]=='\0')
   {
      char  *sybase = getenv("SYBASE");
      if (sybase!=NULL && sybase[0]!='\0')
      {
         search_interface_file(sybase, "interfaces", server, ip_addr, ip_port);
      }
   }

   /*
    * If we still don't have the server and port then assume the user
    * typed an actual server name.
    */
   if (ip_addr[0]=='\0')
   {
      char  *tmp_port;

      /*
       * Make a guess about the port number
       */
#ifdef TDS42
      tmp_port = "1433";
#else
      tmp_port = "4000";
#endif
      /* FIX ME -- Need a symbolic constant for the environment variable */
      if (getenv("DBLIB_PORT")!=NULL && getenv("DBLIB_PORT")[0]!='\0')
      {
         tmp_port = getenv("DBLIB_PORT");
      }

      /*
       * lookup the host and service
       */
      lookup_host(server, tmp_port, ip_addr, ip_port);
   }

   return ip_addr[0]!='\0' && ip_port[0]!='\0';
} /* get_server_info()  */



static int   write_dump = 0;      /* is TDS stream debug log turned on? */
static FILE *dumpfile   = NULL;   /* file pointer for dump log          */



/* ============================== tdsdump_off() ==============================
 *
 * Def:  temporarily turn off logging.  Note- 
 *
 * Ret:  void
 *
 * ===========================================================================
 */
void tdsdump_off()
{
   write_dump = 0;
} /* tdsdump_off()  */


/* ============================== tdsdump_on() ===============================
 *
 * Def:  turn logging back on.  Note-  You must call tdsdump_open() before 
 *       calling this routine.
 *
 * Ret:  void
 *
 * ===========================================================================
 */
void tdsdump_on()
{
   if (dumpfile != NULL)
   {
      write_dump = 1;
   }
} /* tdsdump_on()  */


/* ============================= tdsdump_open() ==============================
 *
 * Def:  This creates and truncates a human readable dump file for the TDS
 *       traffic.  The name of the file is specified by the filename
 *       parameter.  If that is given as NULL, it opens a
 *       file named "tdsdump.out" in the current directory.
 *
 * Ret:  true iff the file was opened, false if it couldn't be opened.
 *
 * ===========================================================================
 */
int tdsdump_open(const char *filename)
{
   int   result;   /* really should be a boolean, not an int */
   if (filename == NULL || filename[0]=='\0')
   {
      filename = "tdsdump.out";
   }

   if (NULL == (dumpfile = fopen(filename, "w")))
   {
      tdsdump_off();
      result = 0;
   }
   else
   {
      tdsdump_on();
      result = 1;
   }
   return result;
} /* tdsdump_open()  */



/* ============================= tdsdump_close() =============================
 *
 * Def:  Close the TDS dump log file.
 *
 * Ret:  void
 *
 * ===========================================================================
 */
void tdsdump_close()
{
   if (dumpfile!=NULL)
   {
      fclose(dumpfile);
   }
   tdsdump_off();
} /* tdsdump_close()  */


/* =========================== tdsdump_dump_buf() ============================
 *
 * Def:  Dump the contents of data into the log file in a human readable
 *       format.
 *
 * Ret:  void
 *
 * ===========================================================================
 */
static void tdsdump_dump_buf(
   const void    *buf,     /* (I) buffer to dump                      */
   int            length)  /* (I) number of bytes in the buffer       */
{
   int                   i;
   int                   j;
   const int             bytesPerLine = 16;
   const unsigned char  *data         = buf;

   if (write_dump && dumpfile!=NULL)
   {
      for(i=0; i<length; i+=bytesPerLine)
      {
         /*
          * print the offset as a 4 digit hex number
          */
         fprintf(dumpfile, "%04x  ", i);

         /*
          * print each byte in hex
          */
         for(j=i; j<length && (j-i)<bytesPerLine; j++)
         {
            fprintf(dumpfile, "%02x ", data[j]);
         }
         
         /*
          * skip over to the ascii dump column
          */
         for(; 0!=(j % bytesPerLine); j++)
         {
            fprintf(dumpfile, "   ");
         }
         fprintf(dumpfile, "  |");

         /*
          * print each byte in ascii
          */
         for(j=i; j<length && (j-i)<bytesPerLine; j++)
         {
            fprintf(dumpfile, "%c", (isprint(data[j])) ? data[j] : '.');
         }
         fprintf(dumpfile, "|\n");
      }
      fprintf(dumpfile, "\n");
   }
} /* tdsdump_dump_buf()  */


/* ============================== tdsdump_log() ==============================
 * 
 * Def:  This function write a message to the debug log.  fmt is a printf-like
 *       format string.  It recognizes the following format characters:
 *          d     The next argument is printed a decimal number
 *          s     The next argument is printed as a character string
 *          L     This doesn't consume any arguments, it simply
 *                prints the current local time.
 *          D     This dumps a buffer in hexadecimal and ascii.
 *                The next argument is a pointer to the buffer
 *                and the argument after that is the number 
 *                of bytes in the buffer.
 * 
 * Ret:  void
 * 
 * ===========================================================================
 */
void tdsdump_log(const char *fmt, ...)
{
   if (write_dump && dumpfile!=NULL)
   {
      const char     *ptr;

      va_list   ap;
      va_start(ap, fmt);
      
      for(ptr = fmt; *ptr != '\0'; ptr++)
      {
         if (*ptr == '%')
         {
            ptr++;
            switch(*ptr)
            {
               case 's':
               {
                  char   *s = va_arg(ap, char *);
                  fputs(s, dumpfile);
                  break;
               }
               case 'd':
               {
                  int i = va_arg(ap, int);
                  fprintf(dumpfile, "%d", i);
                  break;
               }
               case 'x':
               {
                  int i = va_arg(ap, int);
                  fprintf(dumpfile, "%x", i);
                  break;
               }
               case 'D':
               {
                  char  *buf = va_arg(ap, char *);
                  int    len = va_arg(ap, int);
                  tdsdump_dump_buf(buf, len);
                  break;
               }
               case 'L': /* current local time */
               {
                  char        buf[1024];
                  struct tm  *tm;
                  time_t      t;

#ifdef __FreeBSD__
                  struct timeval  tv;
#endif

#ifdef __FreeBSD__
                  gettimeofday(&tv, NULL);
                  t = tv.tv_sec;
#else
                  /* 
                   * XXX Need to get a better time resolution for 
                   * non-FreeBSD systems.
                   */
                  time(&t);
#endif
                  tm = localtime(&t);
                  strftime(buf, sizeof(buf)-1, "%Y-%m-%d %H:%M:%S", tm);
                  fputs(buf, dumpfile);
#ifdef __FreeBSD__
                  fprintf(dumpfile, ".%06lu", tv.tv_usec);
#endif
               }
               default:
               {
                  break;
               }
            }
         }
         else
         {
            fputc(*ptr, dumpfile);
         }
      }
   }
} /* tdsdump_log()  */


