/* FreeTDS - Library of routines accessing Sybase and Microsoft databases
 * Copyright (C) 1998-1999  Brian Bruns
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "tds.h"
#include "tdsutil.h"
#include <stdlib.h>
#include <unistd.h>


static char  software_version[]   = "$Id: token.c,v 1.29 1999/08/13 02:21:14 camber Exp $";
static void *no_unused_var_warn[] = {software_version,
                                     no_unused_var_warn};

/* 
** All these functions processes the input from the wire so it 
** may be used by the upper level functions.
*/

static unsigned char seq=0;
int (*g_tds_msg_handler)() = NULL;
int (*g_tds_err_handler)() = NULL;


static int tds_process_env_chg(TDSSOCKET *tds);
static int tds_process_msg(TDSSOCKET *tds,int marker);
static int tds_process_compute_result(TDSSOCKET *tds);
static int tds_process_result(TDSSOCKET *tds);
static int tds_process_col_name(TDSSOCKET *tds);
static int tds_process_col_info(TDSSOCKET *tds);
static int tds_process_compute(TDSSOCKET *tds);
static int tds_process_row(TDSSOCKET *tds);


int tds_process_default_tokens(TDSSOCKET *tds, int marker)
{
TDSRESULTINFO *info;
int order_len;
int tok_size;
int   more_results;
int   cancelled;

   tdsdump_log("%L inside tds_process_default_tokens() marker is %x\n", marker);
   switch(marker) {
      case TDS_ENV_CHG_TOKEN:
      {
         tds_process_env_chg(tds);
         break;
      }
      case TDS_END_TOKEN:
      case TDS_END2_TOKEN:
      case TDS_END3_TOKEN:
      {
         tds_process_end(tds, marker, &more_results, &cancelled);
         break;
      }
      case TDS_124_TOKEN:
         tds_get_n(tds,NULL,8);
         break;
      case TDS_RET_STAT_TOKEN:
	tds->res_info->has_status=1;
	tds->res_info->ret_status=tds_get_int(tds);
        break;
      case TDS_ERR_TOKEN:
      case TDS_MSG_TOKEN:
      case TDS_EED_TOKEN:
         tds_process_msg(tds,marker);
         break;
      case TDS_CAP_TOKEN:
	 tok_size = tds_get_smallint(tds);
         tds_get_n(tds,tds->capabilities,tok_size > TDS_MAX_CAPABILITY ? TDS_MAX_CAPABILITY : tok_size);
         break;
      case TDS_LOGIN_ACK_TOKEN:
         tds_get_n(tds,NULL,tds_get_smallint(tds));
         break;
      case TDS_ORDER_BY_TOKEN:
         order_len = tds_get_smallint(tds);
         tds_get_n(tds, NULL, order_len);
         /* get the next token which is ROW_TOKEN (209)
			tds_get_byte(tds);
			if(orderLen > 1)
				tds_process_column_row(tds); */
         break;
      case TDS_168_TOKEN:
         tds_process_compute_result(tds); 
         break;
         /* 167 is somehow related to compute columns */
      case TDS_167_TOKEN:
      case TDS_174_TOKEN: 
         tds_get_n(tds, NULL, tds_get_smallint(tds));
         break;
      case TDS_RESULT_TOKEN:
         tds_process_result(tds); 
         break;
      case TDS_COL_NAME_TOKEN:
         tds_process_col_name(tds); 
         break;
      case TDS_COL_INFO_TOKEN:
         tds_process_col_info(tds); 
         break;
      case TDS_CMP_ROW_TOKEN:
         tds_process_compute(tds); 
         break;
      case TDS_ROW_TOKEN:
         tds_process_row(tds); 
         break;
      default:
         fprintf(stderr,"Unknown marker: %d!!\n",marker); 
         return 0;
   }	
}	
int tds_process_login_tokens(TDSSOCKET *tds)
{
int succeed=0;
int marker;
int len;
unsigned char ack;
char *tmpbuf;

	tdsdump_log("%L inside tds_process_login_tokens()\n");
	/* get_incoming(tds->s); */
	do {
		marker=tds_get_byte(tds);
		switch(marker) {
			case TDS_LOGIN_ACK_TOKEN:
				len = tds_get_smallint(tds);
				ack = tds_get_byte(tds);
				tds_get_n(tds, NULL, len-1);
/*
				tmpbuf = (char *) malloc(len);
				tds_get_n(tds, tmpbuf, len);
				tdsdump_log("%L login ack marker = %d\n%D\n", marker, tmpbuf, len);
				free(tmpbuf);
*/
				/* TDS 5.0 reports 5 on success 6 on failure
				** TDS 4.2 reports 1 on success and is not
				** present on failure */
				if (ack==5 || ack==1) succeed=1;
				break;
			default:
				tds_process_default_tokens(tds,marker);
				break;
		}
	} while (marker!=TDS_END_TOKEN);
	tdsdump_log("%L leaving tds_process_login_tokens() returning %d\n",succeed);
	return succeed;
}
int tds_process_result_tokens(TDSSOCKET *tds)
{
int result=0;
int marker;
int more_results = 0;
int cancelled;

	if (tds->res_info && tds->res_info->state==TDS_COMPLETED) {
		/* tds_process_row() now eats the end marker and sets
		** state to TDS_COMPLETED
		*/
		return TDS_NO_MORE_RESULTS;
	}

	/* get_incoming(tds->s); */
	do {
		marker=tds_get_byte(tds);
tdsdump_log("%L processing result tokens.  marker is  %x\n", marker);

		switch(marker) {
      			case TDS_ERR_TOKEN:
      			case TDS_MSG_TOKEN:
      			case TDS_EED_TOKEN:
				tds_process_msg(tds,marker);
				break;
			case TDS_RESULT_TOKEN:
				if (!result) {
					tds_process_result(tds);
					result++;
				} else {
					tds_unget_byte(tds);
					return TDS_SUCCEED;
				}
				break;
			case TDS_COL_NAME_TOKEN:
				if (!result) {
					tds_process_col_name(tds);
					result++;
				} else {
					tds_unget_byte(tds);
					return TDS_SUCCEED;
				}
				break;
			case TDS_ROW_TOKEN:
				if (!result) {
				} else {
					tds->res_info->rows_exist=1;
					tds_unget_byte(tds);
					return TDS_SUCCEED;
				}
      			case TDS_RET_STAT_TOKEN:
				tds->res_info->has_status=1;
				tds->res_info->ret_status=tds_get_int(tds);
				/* return TDS_SUCCEED; */
				break;
			case TDS_END_TOKEN:
			case TDS_END2_TOKEN:
			case TDS_END3_TOKEN:
				tds_process_end(tds, marker, 
						&more_results, 
						&cancelled);
				break;
			default:
				tds_process_default_tokens(tds, marker);
				break;
		}
	} while (!is_end_token(marker) || more_results);

	if (tds->res_info) 
		tds->res_info->state=TDS_COMPLETED;

	if (tds->msg_info && tds->msg_info->priv_msg_type==1)
		return TDS_FAIL;

	return TDS_NO_MORE_RESULTS;
}
int tds_process_row_tokens(TDSSOCKET *tds)
{
int marker;
int   more_results;
int   cancelled;

	while (1) {
		marker=tds_get_byte(tds);
tdsdump_log("%L processing row tokens.  marker is  %x\n", marker);
		switch(marker) {
			case TDS_RESULT_TOKEN:
				tds_unget_byte(tds);
				return TDS_NO_MORE_ROWS;
			case TDS_ROW_TOKEN:
				tds_process_row(tds);
				return TDS_SUCCEED;
			case TDS_END_TOKEN:
			case TDS_END2_TOKEN:
			case TDS_END3_TOKEN:
				tds_process_end(tds, marker, 
						&more_results, 
						&cancelled);
				tds->res_info->more_results = more_results;
				return TDS_NO_MORE_ROWS;
			default:
				tds_process_default_tokens(tds,marker);
				break;
		}
	} 
}
static int tds_process_col_name(TDSSOCKET *tds)
{
int hdrsize, len=0;
int col,num_cols=0;
int colnamelen;
struct tmp_col_struct {
	char *column_name;
	struct tmp_col_struct *next;
};
struct tmp_col_struct *head=NULL, *cur=NULL, *prev;
TDSCOLINFO *curcol;
TDSRESULTINFO *info;

	hdrsize = tds_get_smallint(tds);

	/* this is a little messy...TDS 5.0 gives the number of columns
	** upfront, while in TDS 4.2, you're expected to figure it out
	** by the size of the message. So, I use a link list to get the
	** colum names and then allocate the result structure, copy
	** and delete the linked list */
	while (len<hdrsize) {
		prev = cur;
		cur = (struct tmp_col_struct *) 
			malloc(sizeof (struct tmp_col_struct));
		if (prev) prev->next=cur;
		if (!head) head = cur;

		colnamelen = tds_get_byte(tds);
		cur->column_name = (char *) malloc(colnamelen+1);
		tds_get_n(tds,cur->column_name, colnamelen);
		cur->column_name[colnamelen]='\0';
		cur->next=NULL;

		len += colnamelen + 1;
		num_cols++;
	}

	if (tds->res_info) tds_free_results(tds->res_info);
	tds->res_info = tds_alloc_results(num_cols);
	info = tds->res_info;
	/* tell the upper layers we are processing results */
	info->state = TDS_PENDING; 
	cur=head;
	for (col=0;col<info->num_cols;col++) 
	{
		curcol=info->columns[col];
		strncpy(curcol->column_name, cur->column_name, 
			sizeof(curcol->column_name));
		prev=cur; cur=cur->next;
		free(prev->column_name);
		free(prev);
	}

} 
static int tds_process_col_info(TDSSOCKET *tds)
{
int col,hdrsize;
TDSCOLINFO *curcol;
TDSRESULTINFO *info;
int text_size = 0;
int tmp_rest;
int remainder;


	hdrsize = tds_get_smallint(tds);

	info = tds->res_info;
	for (col=0;col<info->num_cols;col++) 
	{
		curcol=info->columns[col];
		tds_get_n(tds, NULL, 4);
		curcol->column_type = tds_get_byte(tds);
		if(curcol->column_type == SYBTEXT) {
			tds_get_n(tds,NULL,16);
			text_size = 16;
		} else if (!is_fixed_type(curcol->column_type)) {
			curcol->column_size = tds_get_byte(tds);
		} else { 
			curcol->column_size = get_size_by_type(curcol->column_type);
		}
		curcol->column_offset = info->row_size;
		info->row_size += curcol->column_size + 1;
		if(IS_TDS42(tds)) {
			/* actually this 4 should be a machine dependent #define */
			remainder = info->row_size % 4; 
			if (remainder) info->row_size += (4 - remainder);
		}
	}
	/* get the rest of the bytes */
	tmp_rest = hdrsize - ((6 * info->num_cols) + text_size);
	if(tmp_rest > 0) {
		fprintf(stderr,"NOTE:tds_process_col_info: draining %d bytes\n", tmp_rest);
		tds_get_n(tds, NULL, tmp_rest);
	}

	/* tmp_rest = hdrsize - (6 * info->num_cols); */
	/* the_rest = tds_get_byte(tds); */
	if(tmp_rest > 0)
		tds_get_n(tds, NULL, tmp_rest);

	info->current_row = tds_alloc_row(info);

	return 1; /* SUCCEED */
}
static int tds_process_compute_result(TDSSOCKET *tds)
{
int hdrsize;
int col, num_cols;
TDSCOLINFO *curcol;
TDSCOMPUTEINFO *info;
int remainder;

	info = tds->comp_info;
	if (info) tds_free_compute_results(info);

	hdrsize = tds_get_smallint(tds);
	/* unknown always 1 ? */
	tds_get_smallint(tds); 
	num_cols = tds_get_byte(tds);
	tds->comp_info = tds_alloc_compute_results(num_cols);
	info = tds->comp_info;

	for (col=0;col<num_cols;col++) 
	{
		curcol=info->columns[col];
		/* user type and some other stuff? */
		tds_get_n(tds,NULL,6);
		/* column type */
		curcol->column_type = tds_get_byte(tds);
		/* column size */
		if (!is_fixed_type(curcol->column_type)) {
			curcol->column_size = tds_get_byte(tds);
		} else { 
			curcol->column_size = get_size_by_type(curcol->column_type);
		}
		curcol->column_offset = info->row_size;
		info->row_size += curcol->column_size + 1;
		/* actually this 4 should be a machine dependent #define */
		remainder = info->row_size % 4; 
		if (remainder) info->row_size += (4 - remainder);

		tds_get_byte(tds);
	}
	tds_get_smallint(tds);

}
static int tds_process_result(TDSSOCKET *tds)
{
int hdrsize;
int colnamelen;
int savepos;
int col, num_cols;
TDSCOLINFO *curcol;
TDSRESULTINFO *info;
int remainder;

	info = tds->res_info;

	if (info) tds_free_results(info);

	hdrsize = tds_get_smallint(tds);

	savepos = tds->in_pos;
	num_cols = tds_get_smallint(tds);

	tds->res_info = tds_alloc_results(num_cols);
	info = tds->res_info;
	/* tell the upper layers we are processing results */
	info->state = TDS_PENDING; 

	for (col=0;col<info->num_cols;col++) 
	{
		curcol=info->columns[col];
		colnamelen = tds_get_byte(tds);
		tds_get_n(tds,curcol->column_name, colnamelen);
		curcol->column_name[colnamelen]='\0';
		tds_get_byte(tds); /* skip null */
		curcol->column_usertype = tds_get_smallint(tds);
		tds_get_smallint(tds);  /* another unknown */
		curcol->column_type = tds_get_byte(tds);
		if (!is_fixed_type(curcol->column_type)) {
			curcol->column_size = tds_get_byte(tds);
		} else { 
			curcol->column_size = get_size_by_type(curcol->column_type);
		}
		if (curcol->column_type==SYBNUMERIC || curcol->column_type==SYBDECIMAL) {
			curcol->column_prec = tds_get_byte(tds); /* precision */
			curcol->column_scale = tds_get_byte(tds); /* scale */
		}

		curcol->column_offset = info->row_size;

		if (curcol->column_type==SYBNUMERIC) {
			info->row_size += sizeof(TDS_NUMERIC) + 1;
		} else {
			info->row_size += curcol->column_size + 1;
		}
		if(IS_TDS50(tds)){
			/* actually this 4 should be a machine dependent #define */
			remainder = info->row_size % 4; 
			if (remainder) info->row_size += (4 - remainder);
		}

		tds_get_byte(tds);
	}
	info->current_row = tds_alloc_row(info);

	return 1;  /* SUCCEED */
}
static int tds_process_compute(TDSSOCKET *tds)
{
int colsize, i;
TDSCOLINFO *curcol;
TDSCOMPUTEINFO *info;
int compid;
unsigned char *dest;

	info = tds->comp_info;
	
	compid = tds_get_smallint(tds); /* compute id? */
	for (i=0;i<info->num_cols;i++)
	{
		curcol = info->columns[i];
		if (!is_fixed_type(curcol->column_type)) {
			colsize = tds_get_byte(tds);
		} else {
			colsize = get_size_by_type(curcol->column_type);
		}
		dest = &(info->current_row[curcol->column_offset]);
		tds_get_n(tds,dest,colsize);
		dest[colsize]='\0';
	}
	return TDS_SUCCEED;
}
static int tds_process_row(TDSSOCKET *tds)
{
int colsize, i;
TDSCOLINFO *curcol;
TDSRESULTINFO *info;
unsigned char *dest;
TDS_NUMERIC *num;

	info = tds->res_info;
	info->row_count++;
	for (i=0;i<info->num_cols;i++)
	{
		curcol = info->columns[i];
		if(curcol->column_type == SYBTEXT) {
			tds_get_n(tds,NULL,25);
			colsize = tds_get_byte(tds);
			tds_get_n(tds,NULL,3);
		} else if (!is_fixed_type(curcol->column_type)) {
			colsize = tds_get_byte(tds);
		} else {
			colsize = get_size_by_type(curcol->column_type);
		}

		/* set NULL flag in the row buffer */
		if (colsize==0) {
			tds_set_null(info->current_row, i);
		} else {
			tds_clr_null(info->current_row, i);
		}

		if (curcol->column_type == SYBNUMERIC) {
			/* handling NUMERIC datatypes: 
			** since these can be passed around independent
			** of the original column they were from, I decided
			** to embed the TDS_NUMERIC datatype in the row buffer
			** instead of using the wire representation even though
			** it uses a few more bytes
			*/
			num = (TDS_NUMERIC *) &(info->current_row[curcol->column_offset]);
			num->precision = curcol->column_prec;
			num->scale = curcol->column_scale;
			tds_get_n(tds,num->array,colsize);
			info->current_row[curcol->column_offset + sizeof(TDS_NUMERIC)]='\0';
		} else {
			dest = &(info->current_row[curcol->column_offset]);
			tds_get_n(tds,dest,colsize);
			dest[colsize]='\0';
		}
		if (curcol->column_type == SYBDATETIME4) {
			tdsdump_log("%L datetime4 %d %d %d %d\n", dest[0], dest[1], dest[2], dest[3]);
		}
		/* printf("%s\n",curcol->column_value); */
	}
	return TDS_SUCCEED;
}
void tds_process_end(
   TDSSOCKET     *tds,
   int            marker,
   int           *more_results,
   int           *was_cancelled)
{
    int tmp = tds_get_smallint(tds);

   *more_results = (tmp & 0x1) != 0;
   *was_cancelled = (tmp & 0x20) != 0;
   if (tds->res_info)  {
      tds->res_info->more_results=*more_results;
      if (*was_cancelled || !(*more_results)) {
          tds->res_info->state = TDS_COMPLETED;
      }
   }
   tds_get_n(tds,NULL,6);
}
int tds_client_msg(TDSSOCKET *tds, int msgnum, int level, int state, int line, char *message)
{
        if(tds->parent) {
		tds->msg_info->msg_number=msgnum;
        	tds->msg_info->msg_level=level; /* severity? */
        	tds->msg_info->msg_state=state;
        	tds->msg_info->server=strdup("OpenClient");
        	tds->msg_info->line_number=line;
        	tds->msg_info->message=strdup(message);
        	g_tds_err_handler(tds->parent);
	}
}
int tds_submit_query(TDSSOCKET *tds, char *query)
{
unsigned char *buf;
int	bufsize;
int	marker;
int   more_results;
int   cancelled;


	if (!query) return TDS_FAIL;

	if (tds->res_info && (tds->res_info->state==TDS_PENDING)) {
#if 0
		/* read until:
		** 1) a row: return result pending error 
		** 2) an end token: continue with select
		*/
		do {
			marker=tds_get_byte(tds);
			switch(marker) {
				case TDS_ROW_TOKEN:
				case TDS_CMP_ROW_TOKEN:
      				case TDS_RESULT_TOKEN:
      				case TDS_COL_NAME_TOKEN:
      				case TDS_COL_INFO_TOKEN:
					tds_unget_byte(tds);
					tds_client_msg(tds,10000,7,0,1,
        "Attempt to initiate a new SQL Server operation with results pending.");
					return TDS_FAIL;
					break;
				case TDS_END_TOKEN:
				case TDS_END2_TOKEN:
         				tds_process_end(tds, marker, 
							&more_results, 
							&cancelled);
					if (more_results) {
						tds_client_msg(tds,10000,7,0,1,
        "Attempt to initiate a new SQL Server operation with results pending.");
						return TDS_FAIL;
					}	
					break;
				default:
					/* eat anything else, who knows where we are */
					tds_process_default_tokens(tds,marker);
					break;
			}
		} while (marker!=TDS_END_TOKEN && marker!=TDS_END2_TOKEN);

#endif
		/* FIX ME -- get real message number et al. 
		** if memory serves the servername is 
		** OpenClient for locally generated messages,
		** but this needs to be verified too.
		*/
		tds_client_msg(tds,10000,7,0,1,
        "Attempt to initiate a new SQL Server operation with results pending.");
		return TDS_FAIL;
	}

	if (tds->res_info)
		tds->res_info->state = TDS_QUERYING;
	if (IS_TDS42(tds)) {
		bufsize = strlen(query);
		buf = (unsigned char *) malloc(bufsize);
		memset(buf,'\0',bufsize);
		memcpy(&buf[0],query,strlen(query));
		tds->out_flag=0x01;
	} else {
		bufsize = strlen(query)+6;
		buf = (unsigned char *) malloc(bufsize);
		memset(buf,'\0',bufsize);
		buf[0]=33; /* this number stays the same even if I change the query */
		buf[1]=(strlen(query)+1)%256;
		buf[2]=(strlen(query)+1)/256;

		memcpy(&buf[6],query,strlen(query));
		tds->out_flag=0x0F;
	}
	tds_put_n(tds, buf, bufsize);
	tds_flush_packet(tds);
	
	free(buf);

	/* get_incoming(tds->s); */
	return TDS_SUCCEED;
}
static int tds_process_env_chg(TDSSOCKET *tds)
{
int size, type;

	size = tds_get_smallint(tds);
	/* this came in a patch, apparently someone saw an env message
	** that was different from what we are handling? -- brian
	*/
	tds_get_n(tds,NULL,size);
/*
	type = tds_get_byte(tds);
	tds_get_n(tds,NULL,tds_get_byte(tds));
	tds_get_n(tds,NULL,tds_get_byte(tds));
*/

	return TDS_SUCCEED;
}

int tds_process_column_row(TDSSOCKET *tds)
{
int colsize, i;
TDSCOLINFO *curcol;
TDSRESULTINFO *info;
unsigned char *dest;


      info = tds->res_info;
      info->row_count++;
#if 0
                      while(tds->in_pos < tds->in_len)
                      {
                              aByte = tds_get_byte(tds);
                              fprintf(stderr, "[%d]=>%d<=",tds->in_pos, aByte);
                              if (aByte>=' ' && aByte<'z')
                                      fprintf(stderr,"=>%c<=",aByte);
                              fprintf(stderr, "\n");
                      }
#endif
      for (i=0;i<(info->num_cols -1);i++)
      {
              curcol = info->columns[i];
              if (!is_fixed_type(curcol->column_type)) {
                      colsize = tds_get_byte(tds);
              } else {
                      colsize = get_size_by_type(curcol->column_type);
              }
		dest = &(info->current_row[curcol->column_offset]);
		tds_get_n(tds,dest,colsize);
		dest[colsize]='\0';
              /* printf("%s\n",curcol->column_value); */
	}

	/* now skip over some stuff and get the rest of the columns */
	tds_get_n(tds,NULL,25);
	colsize = tds_get_byte(tds);
	tds_get_n(tds,NULL,3);
	curcol = info->columns[i];
	dest = &(info->current_row[curcol->column_offset]);
	tds_get_n(tds,dest,colsize);
	dest[colsize]='\0';

	return TDS_SUCCEED;
}
int (*tds_err_handle(TDSSOCKET *tds, int (*handler)() ))
{
      /* tds->msg_info->err_fun = *handler; */
}

int (*tds_msg_handle(TDSSOCKET *tds, int (*handler)() ))
{
      /* tds->msg_info->msg_fun = *handler; */
}

static int tds_process_msg(TDSSOCKET *tds,int marker)
{
int rc;
int len;
int len_msg;
int len_svr;
int len_sqlstate;
int msg_type;
char *tmpbuf;


	/* packet length */
	len = tds_get_smallint(tds);

/*
	tmpbuf = (char *) malloc(len);
	tds_get_n(tds, tmpbuf, len);
	tdsdump_log("%L msg marker = %d\n%D\n", marker, tmpbuf, len);
	free(tmpbuf);
	return 0;
*/
	/* message number */
	tds->msg_info->msg_number = tds_get_int(tds);

	/* msg state */
	tds->msg_info->msg_state = tds_get_byte(tds);

	/* msg level */
	tds->msg_info->msg_level = tds_get_byte(tds);

	/* determine if msg or error */
	if (marker==TDS_EED_TOKEN) {
		if (tds->msg_info->msg_level<=10) 
                    tds->msg_info->priv_msg_type = 0;
		else 
                    tds->msg_info->priv_msg_type = 1;
		/* junk this info for now */
		len_sqlstate = tds_get_byte(tds);
		tds->msg_info->sql_state = (char*)malloc(len_sqlstate+1);
		tds_get_n(tds, tds->msg_info->sql_state,len_sqlstate);
		tds->msg_info->sql_state[len_sqlstate] = '\0';

		/* unknown values */
		tds_get_byte(tds);
		tds_get_smallint(tds);
	} 
	else if (marker==TDS_MSG_TOKEN) {
		tds->msg_info->priv_msg_type = 0;
	} else if (marker==TDS_ERR_TOKEN) {
		tds->msg_info->priv_msg_type = 1;
	} else {
		fprintf(stderr,"tds_process_msg() called with unknown marker!\n");
		return(1);
	}

	/* the message */
	len_msg = tds_get_smallint(tds);
	tds->msg_info->message = (char*)malloc(len_msg+1);
	tds_get_n(tds, tds->msg_info->message, len_msg);
	tds->msg_info->message[len_msg] = '\0';

	/* server name */
	len_svr = tds_get_byte(tds);
	tds->msg_info->server = (char*)malloc(len_svr+1);
	tds_get_n(tds, tds->msg_info->server, len_svr);
	tds->msg_info->server[len_svr] = '\0';

	/* stored proc name if available */
	rc = tds_get_byte(tds);
	if (rc) {
		tds_unget_byte(tds);
		tds->msg_info->proc_name=tds_msg_get_proc_name(tds);
	} else {
		tds->msg_info->proc_name=NULL;
	}

	/* line number in the sql statement where the problem occured */
	tds->msg_info->line_number = tds_get_smallint(tds);

	/* call the global_tds_msg_handler that was set by an upper layer 
	** (dblib, ctlib or some other one).  Call it with the pointer to 
	** the "parent" structure.
	*/

	if(tds->parent) {
		if (tds->msg_info->priv_msg_type) 
			g_tds_err_handler(tds->parent);
		else
			g_tds_msg_handler(tds->parent);
	} else {
		/*
		if(tds->msg_info->msg_number)
			fprintf(stderr, "Msg %d, Level %d, State %d, Server %s, Line %d\n%s\n",
			tds->msg_info->msg_number,
			tds->msg_info->msg_level,
			tds->msg_info->msg_state,
			tds->msg_info->server,
			tds->msg_info->line_number,
			tds->msg_info->message);
		*/
		tds_free_msg(tds->msg_info);
	}
}

char *tds_msg_get_proc_name(TDSSOCKET *tds)
{
int len_proc;
char *proc_name;

	len_proc = tds_get_byte(tds);

	if (len_proc > 0) {
              /* its the length of a stored procedure name */
              proc_name = (char*)malloc(len_proc+1);
              tds_get_n(tds, proc_name, len_proc);
              proc_name[len_proc] = '\0';
	} else {

              /* set the procname to null since there isn't one in the stream */
              proc_name = (char*)NULL;
	}

	return proc_name;

}

int tds_reset_msg_info(TDSSOCKET *tds)
{
      tds->msg_info->priv_msg_type = 0;
      tds->msg_info->msg_number = 0;
      tds->msg_info->msg_state = 0;
      tds->msg_info->msg_level = 0;
      tds->msg_info->line_number = 0;

      if( tds->msg_info->message)
              free(tds->msg_info->message);

      if(tds->msg_info->server)
              free(tds->msg_info->server);
      if(tds->msg_info->proc_name)
              free(tds->msg_info->proc_name);
}

/* =========================== tds_is_result_row() ===========================
 * 
 * Def:  does the next token in stream signify a result row?
 * 
 * Ret:  true if stream is positioned at a row, false otherwise
 * 
 * ===========================================================================
 */
int tds_is_result_row(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   if (marker==TDS_ROW_TOKEN)
   {
      result = 1;
   }
   return result;
} /* tds_is_result_row()  */


int tds_is_result_set(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = (marker==TDS_COL_NAME_TOKEN 
             || marker==TDS_COL_INFO_TOKEN 
             || marker==TDS_RESULT_TOKEN);
   return result;
} /* tds_is_result_set()  */


int tds_is_end_of_results(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = marker==TDS_END_TOKEN || marker==TDS_DONEPROC;
   return result;
} /* tds_is_end_of_results()  */


int tds_is_doneinproc(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = marker==TDS_DONEINPROC;
   return result;
} /* tds_is_end_of_results()  */


int tds_is_error(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = marker==TDS_ERR_TOKEN;
   return result;
} /* tds_is_error()  */

int tds_is_message(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = marker==TDS_MSG_TOKEN;
   return result;
} /* tds_is_message()  */
int tds_is_control(TDSSOCKET *tds)
{
   const int  marker = tds_peek(tds);
   int        result = 0;

   result = (marker==TDS_174_TOKEN || marker==TDS_167_TOKEN);
   return result;
}
/*
** set the null bit for the given column in the row buffer
*/
void tds_set_null(unsigned char *current_row, int column)
{
int bytenum = column  / 8;
int bit = column % 8;
unsigned char mask = 1 << bit;

	tdsdump_log("%L setting column %d to NULL\n", column);
	current_row[bytenum] |= mask;
}
void tds_clr_null(unsigned char *current_row, int column)
{
int bytenum = column  / 8;
int bit = column % 8;
unsigned char mask = ~(1 << bit);

	tdsdump_log("%L setting column %d to NULL\n", column);
	current_row[bytenum] &= mask;
}
int tds_get_null(unsigned char *current_row, int column)
{
int bytenum = column  / 8;
int bit = column % 8;
unsigned char mask = 1 << bit;

	return (current_row[bytenum] & mask) ? 1 : 0;
}

