/* FreeTDS - Library of routines accessing Sybase and Microsoft databases
 * Copyright (C) 1998-1999  Brian Bruns
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "tdsutil.h"
#include "tds.h"
#include "sybfront.h"
#include "sybdb.h"
#include "dblib.h"
#include "tdsconvert.h"
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>

static char  software_version[]   = "$Id: dblib.c,v 1.37 1999/08/11 01:53:58 camber Exp $";
static void *no_unused_var_warn[] = {software_version,
                                     no_unused_var_warn};


static int _db_get_server_type(int bindtype);
static int _get_printable_size(TDSCOLINFO *colinfo);

/* info/err message handler functions (or rather pointers to them) */
int (*g_dblib_msg_handler)() = NULL;
int (*g_dblib_err_handler)() = NULL;
extern int (*g_tds_msg_handler)();
extern int (*g_tds_err_handler)();
#ifdef TDS42
int g_dblib_version = DBVERSION_42;
#endif
#ifdef TDS50
int g_dblib_version = DBVERSION_100;
#endif
#ifdef TDS46
int g_dblib_version = DBVERSION_46;
#endif


static void buffer_init(DBPROC_ROWBUF *buf)
{
   memset(buf, 0xad, sizeof(*buf));

   buf->buffering_on     = 0;
   buf->first_in_buf     = 0;
   buf->newest           = -1;
   buf->oldest           = 0;
   buf->elcount          = 0;
   buf->element_size     = 0;
   buf->rows_in_buf      = 0;
   buf->rows             = NULL;
   buf->next_row         = 1;
} /* buffer_init()  */

static void buffer_clear(DBPROC_ROWBUF *buf)
{
   buf->next_row         = 1;
   buf->first_in_buf     = 0;
   buf->newest           = -1;
   buf->oldest           = 0;
   buf->rows_in_buf      = 0;
   buf->rows             = NULL;
} /* buffer_clear()  */


static void buffer_free(DBPROC_ROWBUF *buf)
{
   if (buf->rows != NULL)
   {
      free(buf->rows);
   }
   buf->rows = NULL;
} /* clear_buffer()  */

static void buffer_delete_rows(
   DBPROC_ROWBUF *buf,    /* (U) buffer to clear             */
   int            count)  /* (I) number of elements to purge */
{
   assert(count <= buf->elcount); /* possibly a little to pedantic */

   if (count > buf->rows_in_buf)
   {
      count = buf->rows_in_buf;
   }

   
   buf->oldest        = (buf->oldest + count) % buf->elcount;
   buf->rows_in_buf  -= count;
   buf->first_in_buf  = count==buf->rows_in_buf ? buf->next_row-1 : buf->first_in_buf + count;

   
   assert(buf->first_in_buf >= 0);
} /* buffer_delete_rows() */


static int buffer_start_resultset(
   DBPROC_ROWBUF *buf,          /* (U) buffer to clear */
   int            element_size) /*                     */
{
   int    space_needed = -1;

   assert(element_size > 0);

   if (buf->rows != NULL)
   {
      memset(buf->rows, 0xad, buf->element_size*buf->rows_in_buf);
      free(buf->rows);
   }

   buf->first_in_buf     = 0;
   buf->next_row         = 1;
   buf->newest           = -1;
   buf->oldest           = 0;
   buf->elcount          = buf->buffering_on ? buf->elcount : 1;
   buf->element_size     = element_size;
   buf->rows_in_buf      = 0;
   space_needed          = element_size * buf->elcount;   
   buf->rows             = malloc(space_needed);

   return buf->rows==NULL ? FAIL : SUCCEED;
} /* buffer_start_resultset()  */


static void buffer_delete_all_rows(
   DBPROC_ROWBUF *buf)    /* (U) buffer to clear */
{
   buffer_delete_rows(buf, buf->rows_in_buf);
} /* delete_all_buffer_rows() */

static int buffer_index_of_resultset_row(
   DBPROC_ROWBUF *buf,         /* (U) buffer to clear                 */
   int            row_number)  /* (I)                                 */
{
   int   result = -1;

   if (row_number < buf->first_in_buf)
   {
      result = -1;
   }
   else if (row_number > ((buf->rows_in_buf + buf->first_in_buf) -1))
   {
      result = -1;
   }
   else
   {
      result = ((row_number - buf->first_in_buf) 
                + buf->oldest) % buf->elcount;
   }
   return result;
} /* buffer_index_of_resultset_row()  */


static void *buffer_row_address(
   DBPROC_ROWBUF *buf,    /* (U) buffer to clear                 */
   int            index)  /* (I) raw index of row to return      */
{
   void   *result = NULL;

   assert(index >= 0);
   assert(index < buf->elcount);

   if (index>=buf->elcount || index<0)
   {
      result = NULL;
   }
   else
   {
      int   offset = buf->element_size * (index % buf->elcount);
      result = (char *)buf->rows + offset;
   }
   return result;
} /* buffer_row_address()  */


static void buffer_add_row(
   DBPROC_ROWBUF *buf,       /* (U) buffer to add row into  */
   void          *row,       /* (I) pointer to the row data */
   int            row_size)
{
   void   *dest = NULL;

   assert(row_size > 0);
   assert(row_size == buf->element_size);
   
   assert(buf->elcount >= 1);
   
   buf->newest = (buf->newest + 1) % buf->elcount;
   if (buf->rows_in_buf==0 && buf->first_in_buf==0)
   {
      buf->first_in_buf = 1;
   }
   buf->rows_in_buf++;

   /* 
    * if we have wrapped around we need to adjust oldest
    * and rows_in_buf
    */
   if (buf->rows_in_buf > buf->elcount)
   {
      buf->oldest = (buf->oldest + 1) % buf->elcount;
      buf->first_in_buf++;
      buf->rows_in_buf--;
   }
   
   assert(buf->elcount >= buf->rows_in_buf);
   assert( buf->rows_in_buf==0
           || (((buf->oldest+buf->rows_in_buf) - 1)%buf->elcount)==buf->newest);
   assert(buf->rows_in_buf>0 || (buf->first_in_buf==buf->next_row-1));
   assert(buf->rows_in_buf==0 || 
          (buf->first_in_buf<=buf->next_row));
   assert(buf->next_row-1 <= (buf->first_in_buf + buf->rows_in_buf));

   dest = buffer_row_address(buf, buf->newest);
   memcpy(dest, row, row_size);
} /* buffer_add_row()  */


static int buffer_is_full(DBPROC_ROWBUF *buf)
{
   return buf->rows_in_buf==buf->elcount;
} /* buffer_is_full()  */

static void buffer_set_buffering(
   DBPROC_ROWBUF *buf,      /* (U)                                         */
   int            buf_size) /* (I) number of rows to buffer, 0 to turn off */
{
   /* XXX If the user calls this routine in the middle of 
    * a result set and changes the size of the buffering
    * they are pretty much toast.
    *
    * We need to figure out what to do if the user shrinks the 
    * size of the row buffer.  What rows should be thrown out, 
    * what should happen to the current row, etc?
    */

   assert(buf_size >= 0);

   if (buf_size < 0)
   {
      /* XXX is it okay to ignore this? */
   }
   else if (buf_size == 0)
   {
      buf->buffering_on = 0;
      buf->elcount = 1;
      buffer_delete_all_rows(buf);
   }
   else
   {
      buf->buffering_on = 1;
      buffer_clear(buf);
      buffer_free(buf);
      buf->elcount = buf_size;
      if (buf->element_size > 0)
      {
         buf->rows = malloc(buf->element_size * buf->elcount);
      }
      else
      {
         buf->rows = NULL;
      }
   }
} /* buffer_set_buffering()  */

static void buffer_transfer_bound_data(
   DBPROC_ROWBUF *buf,      /* (U)                                         */
   DBPROCESS     *dbproc,   /* (I)                                         */
   int            row_num)  /* (I) resultset row number                    */
{
   int            i;
   TDSCOLINFO    *curcol;
   TDSRESULTINFO *resinfo;
   TDSSOCKET     *tds;
   int            srctype;
   int            desttype;
   /* this should probably go somewhere else */
   TDS_NUMERIC	numeric;
   
   tds     = (TDSSOCKET *) dbproc->tds_socket;
   resinfo = tds->res_info;
   
   for (i=0;i<resinfo->num_cols;i++)
   {
      curcol = resinfo->columns[i];
      if (curcol->column_nullbind) 
      {
         if (tds_get_null(resinfo->current_row,i)) {
            *((DBINT *)curcol->column_nullbind)=-1;
         } else {
            *((DBINT *)curcol->column_nullbind)=0;
         }
      }
      if (curcol->varaddr)
      {
         int   index = buffer_index_of_resultset_row(buf, row_num);
         
         if (index<0)
         {
            assert(1==0);
            /* XXX now what? */
         }
         else
         {
            char   *src = ((char*)buffer_row_address(buf, index)) 
               + curcol->column_offset;
            desttype = _db_get_server_type(curcol->column_bindtype);
            srctype = tds_get_conversion_type(curcol->column_type,
                                           curcol->column_size);

            dbconvert(dbproc,
                      srctype,                      /* srctype  */
                      src,                          /* src      */
                      -1,                           /* srclen   */
                      desttype,                     /* desttype */
                      curcol->varaddr,              /* dest     */
                      curcol->column_bindlen);      /* destlen  */
         }
         
      }
   }
} /* buffer_transfer_bound_data()  */

RETCODE dbinit()
{
   /* set the functions in the TDS layer to point to the correct info/err
    * message handler functions */
   g_tds_msg_handler = dblib_handle_info_message;
   g_tds_err_handler = dblib_handle_err_message;
   
   
   return SUCCEED;
}
LOGINREC *dblogin()
{
LOGINREC * loginrec;

	loginrec = (LOGINREC *) malloc(sizeof(LOGINREC));
	loginrec->tds_login = (void *) tds_alloc_login();

	/* set default values for loginrec */
	tds_set_library(loginrec->tds_login,"DB-Library");
	tds_set_charset(loginrec->tds_login,"iso_1");
	tds_set_packet(loginrec->tds_login,512);

	return loginrec;
}
void dbloginfree(LOGINREC *login)
{
	if (login) free(login);
}
RETCODE DBSETLPACKET(LOGINREC *login, short packet_size)
{
	tds_set_packet(login->tds_login,packet_size);
	return SUCCEED;
}
RETCODE DBSETLPWD(LOGINREC *login, char *password)
{
	tds_set_passwd(login->tds_login,password);
	return SUCCEED;
}
RETCODE DBSETLUSER(LOGINREC *login, char *username)
{
	tds_set_user(login->tds_login,username);
	return SUCCEED;
}
RETCODE DBSETLHOST(LOGINREC *login, char *hostname)
{
	tds_set_host(login->tds_login,hostname);
	return SUCCEED;
}
RETCODE DBSETLAPP(LOGINREC *login, char *application)
{
	tds_set_app(login->tds_login,application);
	return SUCCEED;
}
DBPROCESS *dbopen(LOGINREC *login,char *server)
{
   DBPROCESS *dbproc;
   char	*envbuf;
   int version_value;
   
   dbproc = (DBPROCESS *) malloc(sizeof(DBPROCESS));
   memset(dbproc,'\0',sizeof(DBPROCESS));
   tds_set_server(login->tds_login,server);
   
   envbuf = (char*)getenv("DBLIB_VERSION");
	if(envbuf)
	{
		version_value = atoi(envbuf);
		dblib_setTDS_version(login->tds_login, version_value);
	}
	else
	{
		dblib_setTDS_version(login->tds_login, g_dblib_version );
	}
   
   dbproc->tds_socket = (void *) tds_connect(login->tds_login);
   dbproc->dbbuf[0]='\0'; /* initialize query buffer */
   
   if(dbproc->tds_socket) {
      tds_set_parent( dbproc->tds_socket, dbproc);
   } else {
      fprintf(stderr,"DB-Library: Login incorrect.\n");
      return NULL;
   }
   
   buffer_init(&(dbproc->row_buf));
   
   return dbproc;
}
RETCODE	dbcmd(DBPROCESS *dbproc, char *cmdstring)
{
	/* here is one of those large static buffers that should be malloced
	** but in the interest of time i kludged :)   */
	strcat(dbproc->dbbuf,cmdstring);
	return SUCCEED;
}
RETCODE dbsqlexec(DBPROCESS *dbproc)
{
RETCODE   rc = FAIL;
TDSSOCKET *tds;

   tds = (TDSSOCKET *) dbproc->tds_socket;
   if (tds->res_info && tds->res_info->more_results) 
   /* if (dbproc->more_results && tds_is_end_of_results(dbproc->tds_socket)) */
   {
      dbresults(dbproc);
   }
      
   if (SUCCEED == (rc = dbsqlsend(dbproc)))
   {
      /* 
       * XXX We need to observe the timeout value and abort 
       * if this times out.
       */
      rc = dbsqlok(dbproc);
   }
   return rc;
}
RETCODE dbuse(DBPROCESS *dbproc,char *dbname)
{
char query[255];
/* int retval; */

	sprintf(query,"use %s",dbname);
	dbcmd(dbproc,query);
	dbsqlexec(dbproc);
	while (dbresults(dbproc)!=NO_MORE_RESULTS)
		while (dbnextrow(dbproc)!=NO_MORE_ROWS);
	return SUCCEED;
}
RETCODE dbclose(DBPROCESS *dbproc)
{
TDSRESULTINFO *resinfo;
TDSSOCKET *tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	if (tds) {
		resinfo = tds->res_info;
		if (resinfo) tds_free_results(resinfo);
                buffer_free(&(dbproc->row_buf));
		tds_free_socket(tds);
	}

	free(dbproc);
        return SUCCEED;
}
void dbexit()
{
}


RETCODE dbresults_r(DBPROCESS *dbproc, int recursive)
{
RETCODE       retcode = FAIL;
unsigned char marker; 
TDSSOCKET *tds;

   
   /* 
    * For now let's assume we have only 5 possible classes of tokens 
    * at the next byte in the TDS stream
    *   1) The start of a result set, either TDS_RESULT_TOKEN (tds ver 5.0)
    *      or TDS_COL_NAME_TOKEN (tds ver 4.2).
    *   2) A row token (TDS_ROW_TOKEN)
    *   3) An end token (either 0xFD or 0xFE)
    *   4) A done in proc token (0xFF)
    *   5) A message or error token
    */

   buffer_clear(&(dbproc->row_buf));

/* all of this should be handled in the tds layer, that way all
 * CLIs benefit. 
 */

#if 0
   tds = (TDSSOCKET *) dbproc->tds_socket;
   if (tds->res_info && !tds->res_info->more_results)
   {
      return NO_MORE_RESULTS;
   }

   while (tds_is_message(dbproc->tds_socket)
      || tds_is_error(dbproc->tds_socket))
   {
      if (tds_is_error(dbproc->tds_socket))
      {
         retcode = FAIL;
      }
      marker = tds_get_byte(dbproc->tds_socket);
      tds_process_default_tokens(dbproc->tds_socket, marker);
      /* XXX What else do we need to do on messages and errors? */
   }

   if (tds_is_result_set(dbproc->tds_socket))
   {
      while(tds_is_result_set(dbproc->tds_socket))
      {
         marker = tds_get_byte(dbproc->tds_socket);
         tds_process_default_tokens(dbproc->tds_socket, marker);
      }
      if(dbproc->tds_socket->res_info)
      {
         assert(dbproc->tds_socket->res_info->row_size > 0);
         retcode = buffer_start_resultset(&(dbproc->row_buf), 
                                      dbproc->tds_socket->res_info->row_size);
      }
   }
   else if (tds_is_result_row(dbproc->tds_socket))
   {
      retcode = FAIL;
   }
   else if (tds_is_end_of_results(dbproc->tds_socket))
   {
      int   more_results;
      int   canceled; 

      marker = tds_get_byte(dbproc->tds_socket);
      tds_process_end(dbproc->tds_socket, marker, &more_results, &canceled);
      dbproc->more_results = more_results;
      if (dbproc->more_results)
      {
         dbresults_r(dbproc, 1);
      }
      else
      {
         retcode = NO_MORE_RESULTS;
      }
   }
   else if (tds_is_doneinproc(dbproc->tds_socket))
   {
      abort();
   }
#endif
   retcode = tds_process_result_tokens(dbproc->tds_socket);
   if (retcode == TDS_SUCCEED) {
   	retcode = buffer_start_resultset(&(dbproc->row_buf), 
                                      dbproc->tds_socket->res_info->row_size);
   }
   return retcode;
}
   
/* =============================== dbresults() ===============================
 * 
 * Def: 
 * 
 * Ret:  SUCCEED, FAIL, NO_MORE_RESULTS, or NO_MORE_RPC_RESULTS
 * 
 * ===========================================================================
 */
RETCODE dbresults(DBPROCESS *dbproc)
{
   return dbresults_r(dbproc, 0);
}


int dbnumcols(DBPROCESS *dbproc)
{
TDSRESULTINFO *resinfo;
TDSSOCKET *tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	return resinfo->num_cols;
}
char *dbcolname(DBPROCESS *dbproc, int column)
{
static char buf[255];
TDSRESULTINFO *resinfo;
TDSSOCKET *tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	if (column < 1 || column > resinfo->num_cols) return NULL;
	strcpy (buf,resinfo->columns[column-1]->column_name);
	return buf;
}

RETCODE dbgetrow(
   DBPROCESS *dbproc, 
   DBINT row)
{
   RETCODE   result = FAIL;
   int       index = buffer_index_of_resultset_row(&(dbproc->row_buf), row);
   if (-1 == index)
   {
      result = NO_MORE_ROWS;
   }
   else
   {
      dbproc->row_buf.next_row = row;
      buffer_transfer_bound_data(&(dbproc->row_buf), dbproc, row);
      dbproc->row_buf.next_row++;
      result = REG_ROW;
   }

   return result;
}

RETCODE dbnextrow(DBPROCESS *dbproc)
{
   TDSRESULTINFO *resinfo;
   TDSSOCKET     *tds;
   int            rc;
   RETCODE        result = FAIL;
   int            marker;
   

   tds = (TDSSOCKET *) dbproc->tds_socket;
   resinfo = tds->res_info;
   
   if (dbproc->row_buf.buffering_on && buffer_is_full(&(dbproc->row_buf))
       && (-1 == buffer_index_of_resultset_row(&(dbproc->row_buf), 
                                               dbproc->row_buf.next_row)))
   {
      result = BUF_FULL;
   }
   else 
   {
      /*
       * Now try to get the dbproc->row_buf.next_row item into the row
       * buffer
       */
      if (-1 != buffer_index_of_resultset_row(&(dbproc->row_buf), 
                                              dbproc->row_buf.next_row))
      {
         /*
          * Cool, the item we want is already there
          */
         rc     = TDS_SUCCEED;
         result = REG_ROW;
      }
      else
      {
#if 0
         /*
          * Check to see if we have a row available to read
          */
         marker = tds_peek(dbproc->tds_socket);
        
         if (marker==TDS_END_TOKEN || marker==TDS_END2_TOKEN 
             || marker==TDS_END3_TOKEN)
         {
            /*
             * Bummer, the row doesn't exist.
             */
            rc     = TDS_NO_MORE_ROWS;
            result = NO_MORE_ROWS;
         }
         else
         {
#endif
            /* 
             * XXX Note- we need to handle "compute" results as well.
             * I don't believe the current src/tds/token.c handles those
             * so we don't handle them yet either.
             */

            /* 
             * Get the row from the TDS stream.
             */
            rc = tds_process_row_tokens(dbproc->tds_socket);
            if (rc == TDS_SUCCEED)
            {
               /*
                * Add the row to the row buffer
                */
               buffer_add_row(&(dbproc->row_buf), resinfo->current_row, 
                              resinfo->row_size);
               result = REG_ROW;
            }
            else if (rc == TDS_NO_MORE_ROWS)
            {
               result = NO_MORE_ROWS;
            }
            else 
            {
               result = FAIL;
            }
#if 0
         }
#endif
      }
   
      if (result == REG_ROW)
      {
         /*
          * The data is in the row buffer, now transfer it to the 
          * bound variables
          */
         buffer_transfer_bound_data(&(dbproc->row_buf), dbproc, 
                                    dbproc->row_buf.next_row);
         dbproc->row_buf.next_row++;
      }
   }
   return result;
} /* dbnextrow()  */

static int _db_get_server_type(int bindtype)
{
	switch (bindtype) {
		case CHARBIND:
		case STRINGBIND:
		case NTBSTRINGBIND:
			return SYBCHAR;
			break;
		case FLT8BIND:
			return SYBFLT8;
			break;
		case REALBIND:
			return SYBREAL;
			break;
		case INTBIND:
			return SYBINT4;
			break;
		case SMALLBIND:
			return SYBINT2;
			break;
		case TINYBIND:
			return SYBINT1;
			break;
                default:
                        return -1;
                        break;
	}
}
/* Notes on changes:
** conversion functions are now handled in the TDS layer.
** The main reason for this is that ctlib and ODBC (and presumably DBI) need
** to be able to do conversions between datatypes. This is possible because
** the format of complex data (dates, money, numeric, decimal) is defined by
** its representation on the wire, thus what we call DBMONEY is exactly its
** format on the wire. CLIs that need a differtent representation (ODBC?) 
** need to convert from this format anyway, so the code would already be in
** place.
** The datatypes are also defined by its Server-type so all CLIs should be 
** able to map native types to server types as well.
*/
DBINT dbconvert(DBPROCESS *dbproc,
		int srctype,
		BYTE *src,
		DBINT srclen,
		int desttype,
		BYTE *dest,
		DBINT destlen)
{
	return tds_convert(srctype, src, srclen, desttype, dest, destlen);
}
RETCODE dbbind(
   DBPROCESS *dbproc,
   int        column,
   int        vartype,
   DBINT      varlen,
   BYTE      *varaddr)
{
/* this is really fucked up:
1) it ignores most of the arguements
2) it assumes everything is a string -- fixed
3) it uses a fixed length (255 byte) array in dbproc->resinfo->columns to store the varaddrs -- fixed 
MEANING: don't consider this implemented by any means. This is a total kludge
*/
   TDSCOLINFO    *colinfo = NULL;
   TDSRESULTINFO *resinfo = NULL;
   TDSSOCKET     *tds     = NULL;
   int            srctype = -1;   
   int            okay    = TRUE; /* so far, so good */

   /* 
    * Note on logic-  I'm using a boolean variable 'okay' to tell me if
    * everything that has happened so far has gone okay.  Basically if 
    * something happened that wasn't okay we don't want to keep doing 
    * things, but I also don't want to have a half dozen exit points from 
    * this function.  So basically I've wrapped each set of operation in a 
    * "if (okay)" statement.  Once okay becomes false we skip everything 
    * else.
    */
   okay = (dbproc!=NULL && dbproc->tds_socket!=NULL && varaddr!=NULL);

   if (okay)
   {
      tds     = (TDSSOCKET *) dbproc->tds_socket;
      resinfo = tds->res_info;
   }
   
   okay = okay && ((column >= 1) && (column <= resinfo->num_cols));

   if (okay)
   {
      colinfo  = resinfo->columns[column-1];
      srctype = tds_get_conversion_type(colinfo->column_type,
                                     colinfo->column_size);
      okay = okay && dbwillconvert(srctype, vartype);
   }

   if (okay)
   {   
      colinfo->varaddr         = (char *)varaddr;
      colinfo->column_bindtype = vartype;
      colinfo->column_bindlen  = varlen;
   }

   return okay ? SUCCEED : FAIL;
} /* dbbind()  */

void dbsetifile(char *filename)
{
	set_interfaces_file_loc(filename);
}
RETCODE dbnullbind(DBPROCESS *dbproc, int column, DBINT *indicator)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

        /*
         *  XXX Need to check for possibly problems before assuming
         *  everything is okay
         */
	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	colinfo = resinfo->columns[column-1];
	colinfo->column_nullbind = (TDS_CHAR *) indicator;

        return SUCCEED;
}
DBINT DBCOUNT(DBPROCESS *dbproc)
{
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	return resinfo->row_count;
}
void dbclrbuf(DBPROCESS *dbproc, DBINT n)
{
   if (n <= 0) return;

   if (dbproc->row_buf.buffering_on)
   {
      if (n >= dbproc->row_buf.rows_in_buf) {
      	buffer_delete_rows(&(dbproc->row_buf), dbproc->row_buf.rows_in_buf - 1);
      } else {
      	buffer_delete_rows(&(dbproc->row_buf), n);
      }
   }
}
DBBOOL dbwillconvert(int srctype, int desttype)
{
   /* XXX */
   return TRUE;
}
int dbcoltype(DBPROCESS *dbproc,int column)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	colinfo = resinfo->columns[column-1];
	switch (colinfo->column_type) {
		case SYBVARCHAR:
			return SYBCHAR; 
		case SYBVARBINARY:
			return SYBBINARY;
		case SYBDATETIMN:
			return SYBDATETIME;
		case SYBMONEYN:
			return SYBMONEY;
		case SYBFLTN:
			return SYBFLT8;
		case SYBINTN:
			if (colinfo->column_size==4)
				return SYBINT4;
			else if (colinfo->column_size==2)
				return SYBINT2; 
			else if (colinfo->column_size==1)
				return SYBINT1; 
		default:
			return colinfo->column_type;
	}
	return 0; /* something went wrong */
}
DBTYPEINFO *dbcoltypeinfo(DBPROCESS *dbproc, int column)
{
/* FIXME -- this is not thread safe, we would need to move typeinfo into
** the dbproc structure, however that seems a waste for such a little used
** function...maybe malloc it when used, cleanup on dbclose() */
static DBTYPEINFO typeinfo;
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	colinfo = resinfo->columns[column-1];
	typeinfo.precision = colinfo->column_prec;
	typeinfo.scale = colinfo->column_scale;
	return &typeinfo;
}
char *dbcolsource(DBPROCESS *dbproc,int colnum)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	colinfo = resinfo->columns[colnum-1];
	return colinfo->column_name;
}
DBINT dbcollen(DBPROCESS *dbproc, int column)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	if (column<1 || column>resinfo->num_cols) return -1;
	colinfo = resinfo->columns[column-1];
	return colinfo->column_size;
}
DBINT dbdatlen(DBPROCESS *dbproc, int column)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	/* FIX ME -- this is the columns info, need per row info */
	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	if (column<1 || column>resinfo->num_cols) return -1;
	colinfo = resinfo->columns[column-1];
	return colinfo->column_size;
}
BYTE *dbdata(DBPROCESS *dbproc, int column)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	if (column<1 || column>resinfo->num_cols) return NULL;

	colinfo = resinfo->columns[column-1];
	/* FIX ME -- check for null value */
	return &resinfo->current_row[colinfo->column_offset];
}
RETCODE dbcancel(DBPROCESS *dbproc)
{
   /* 
    * XXX This doesn't do enough to really cancel. 
    * To really cancel you need to send a CANCEL packet,
    * (packet type 0x06) and then discard all input 
    * until you receive acknowledgment of the cancel.
    */
   tds_process_default_tokens(dbproc->tds_socket,CANCEL);
   return SUCCEED;
}
void dbprrow(DBPROCESS *dbproc)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;
int i,col,collen,namlen,len;
char dest[256];
int desttype, srctype;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;

	while(dbnextrow(dbproc)==REG_ROW) {
		for (col=0;col<resinfo->num_cols;col++)
		{
			colinfo = resinfo->columns[col];
			if (tds_get_null(resinfo->current_row,col)) {
				strcpy(dest,"NULL");
			} else {
				desttype = _db_get_server_type(STRINGBIND);
				srctype = tds_get_conversion_type(colinfo->column_type,colinfo->column_size);
				dbconvert(dbproc, srctype ,&resinfo->current_row[colinfo->column_offset], -1, desttype, dest, 255);

				/* printf ("some data\t"); */
			}
			printf("%s",dest);
			collen = _get_printable_size(colinfo);
			namlen = strlen(colinfo->column_name);
			len = collen > namlen ? collen : namlen;
			for (i=strlen(dest);i<len;i++)
				printf(" ");
			printf(" ");
		}
		printf ("\n");
	}
}
static int _get_printable_size(TDSCOLINFO *colinfo)
{
	switch (colinfo->column_type) {
		case SYBINTN:
			switch(colinfo->column_size) {
				case 1: return 3;
				case 2: return 6;
				case 4: return 11;
			}
		case SYBINT1:
			return 3;
		case SYBINT2:
			return 6;
		case SYBINT4:
        		return 11;
		case SYBVARCHAR:
		case SYBCHAR:
			return colinfo->column_size;
		case SYBFLT8:
			return 11; /* FIX ME -- we do not track precision */
		case SYBREAL:
			return 11; /* FIX ME -- we do not track precision */
		case SYBMONEY:
        		return 12; /* FIX ME */
		case SYBMONEY4:
        		return 12; /* FIX ME */
		case SYBDATETIME:
        		return 26; /* FIX ME */
		case SYBDATETIME4:
        		return 26; /* FIX ME */
		case SYBBIT:
			return 1;
		/* FIX ME -- not all types present */
		default:
			return 0;
	}

}
void dbprhead(DBPROCESS *dbproc)
{
TDSCOLINFO * colinfo;
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;
int i,col, len, collen, namlen;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	for (col=0;col<resinfo->num_cols;col++)
	{
		colinfo = resinfo->columns[col];
		collen = _get_printable_size(colinfo);
		namlen = strlen(colinfo->column_name);
		len = collen > namlen ? collen : namlen;
		printf("%s",colinfo->column_name);
		for (i=strlen(colinfo->column_name);i<len;i++)
			printf(" ");
		printf(" ");
	}
	printf ("\n");
	for (col=0;col<resinfo->num_cols;col++)
	{
		colinfo = resinfo->columns[col];
		collen = _get_printable_size(colinfo);
		namlen = strlen(colinfo->column_name);
		len = collen > namlen ? collen : namlen;
		for (i=0;i<len;i++)
			printf("-");
		printf(" ");
	}
	printf ("\n");

}
DBBOOL DBROWS(DBPROCESS *dbproc)
{
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	if (resinfo->rows_exist) return TRUE;
	else return FALSE;
}
/* STUBS */
RETCODE dbsetdeflang(char *language)
{
	return SUCCEED;
}
int dbgetpacket(DBPROCESS *dbproc)
{
	return 512;
}
RETCODE dbsettime(int seconds)
{
	return SUCCEED;
}
RETCODE dbsetlogintime(int seconds)
{
	return SUCCEED;
}
DBBOOL dbhasretstat(DBPROCESS *dbproc)
{
TDSSOCKET *tds = (TDSSOCKET *) dbproc->tds_socket;
	if (tds->res_info->has_status) {
		return TRUE;
	} else {
		return FALSE;
	}
}
DBINT dbretstatus(DBPROCESS *dbproc)
{
TDSSOCKET *tds = (TDSSOCKET *) dbproc->tds_socket;
	return tds->res_info->ret_status;
}
RETCODE DBCMDROW(DBPROCESS *dbproc)
{
	return SUCCEED;
}
int dbaltcolid(DBPROCESS *dbproc, int computeid, int column)
{
	return -1;
}
DBINT dbadlen(DBPROCESS *dbproc,int computeid, int column)
{
	return 0;
}
int dbalttype(DBPROCESS *dbproc, int computeid, int column)
{
	return 0;
}
BYTE *dbadata(DBPROCESS *dbproc, int computeid, int column)
{
	return "";
}
int dbaltop(DBPROCESS *dbproc, int computeid, int column)
{
	return -1;
}
RETCODE dbsetopt(DBPROCESS *dbproc, int option, char *char_param, int int_param)
{
   switch (option) {
      case DBBUFFER:
      {
         /* XXX should be more robust than just a atoi() */
         buffer_set_buffering(&(dbproc->row_buf), atoi(char_param));
         break;
      }
      default:
      {
         break;
      }
   }
   return SUCCEED;
}
void dbsetinterrupt(DBPROCESS *dbproc, int (*ckintr)(),int (*hndlintr)())
{
}
int dbnumrets(DBPROCESS *dbproc)
{
	return 0;
}
char *dbretname(DBPROCESS *dbproc, int retnum)
{
	return NULL;
}
BYTE *dbretdata(DBPROCESS *dbproc, int retnum)
{
	return NULL;
}
int dbretlen(DBPROCESS *dbproc, int retnum)
{
	return -1;
}
RETCODE dbsqlok(DBPROCESS *dbproc)
{
   unsigned char   marker;
/* this will fail under TDS 5.0, you must read the entire message to 
 * determine if it is a msg or an error 
 */
#if 0
   /*
    * See what the next packet from the server is.  If it is an error
    * then we should return FAIL
    */
   marker = tds_peek(dbproc->tds_socket);
   
   return marker == TDS_ERR_TOKEN ? FAIL : SUCCEED;
#endif 
	/* for now */
	return SUCCEED;
}
int dbnumalts(DBPROCESS *dbproc,int computeid)
{
	return 0;
}
BYTE *dbbylist(DBPROCESS *dbproc, int computeid, int size)
{
	return NULL;
}
DBBOOL DBDEAD(DBPROCESS *dbproc)
{
	if(dbproc)
		return FALSE;
	else
		return TRUE;
}

int (*dberrhandle(int (*handler)())) ()
{
   int (*retFun)() = g_dblib_err_handler;

   g_dblib_err_handler = handler;
   return retFun;
}

int (*dbmsghandle(int (*handler)()))()
{
   int (*retFun)() = g_dblib_msg_handler;

   g_dblib_msg_handler = handler;
   return retFun;
}

RETCODE dbmnyadd(DBPROCESS *dbproc, DBMONEY *m1, DBMONEY *m2, DBMONEY *sum)
{
	return SUCCEED;
}
RETCODE dbmnysub(DBPROCESS *dbproc, DBMONEY *m1, DBMONEY *m2, DBMONEY *diff)
{
	return SUCCEED;
}
RETCODE dbmnymul(DBPROCESS *dbproc, DBMONEY *m1, DBMONEY *m2, DBMONEY *prod)
{
	return SUCCEED;
}
RETCODE dbmnydivide(DBPROCESS *dbproc, DBMONEY *m1, DBMONEY *m2, DBMONEY *quotient)
{
	return SUCCEED;
}
RETCODE dbmnycmp(DBPROCESS *dbproc, DBMONEY *m1, DBMONEY *m2)
{
	return SUCCEED;
}
RETCODE dbmnyscale(DBPROCESS *dbproc, DBMONEY *dest, int multiplier, int addend)
{
	return SUCCEED;
}
RETCODE dbmnyzero(DBPROCESS *dbproc, DBMONEY *dest)
{
	return SUCCEED;
}
RETCODE dbmnymaxpos(DBPROCESS *dbproc, DBMONEY *dest)
{
	return SUCCEED;
}
RETCODE dbmnymaxneg(DBPROCESS *dbproc, DBMONEY *dest)
{
	return SUCCEED;
}
RETCODE dbmnyndigit(DBPROCESS *dbproc, DBMONEY *mnyptr,DBCHAR *value, DBBOOL *zero)
{
	return SUCCEED;
}
RETCODE dbmnyinit(DBPROCESS *dbproc,DBMONEY *mnyptr, int trim, DBBOOL *negative)
{
	return SUCCEED;
}
RETCODE dbmnydown(DBPROCESS *dbproc,DBMONEY *mnyptr, int divisor, int *remainder)
{
	return SUCCEED;
}
RETCODE dbmnyinc(DBPROCESS *dbproc,DBMONEY *mnyptr)
{
	return SUCCEED;
}
RETCODE dbmnydec(DBPROCESS *dbproc,DBMONEY *mnyptr)
{
	return SUCCEED;
}
RETCODE dbmnyminus(DBPROCESS *dbproc,DBMONEY *src, DBMONEY *dest)
{
	return SUCCEED;
}
RETCODE dbmny4minus(DBPROCESS *dbproc, DBMONEY4 *src, DBMONEY4 *dest)
{
	return SUCCEED;
}
RETCODE dbmny4zero(DBPROCESS *dbproc, DBMONEY4 *dest)
{
	return SUCCEED;
}
RETCODE dbmny4add(DBPROCESS *dbproc, DBMONEY4 *m1, DBMONEY4 *m2, DBMONEY4 *sum)
{
	return SUCCEED;
}
RETCODE dbmny4sub(DBPROCESS *dbproc, DBMONEY4 *m1, DBMONEY4 *m2, DBMONEY4 *diff)
{
	return SUCCEED;
}
RETCODE dbmny4mul(DBPROCESS *dbproc, DBMONEY4 *m1, DBMONEY4 *m2, DBMONEY4 *prod)
{
	return SUCCEED;
}
RETCODE dbmny4divide(DBPROCESS *dbproc, DBMONEY4 *m1, DBMONEY4 *m2, DBMONEY4 *quotient)
{
	return SUCCEED;
}
RETCODE dbmny4cmp(DBPROCESS *dbproc, DBMONEY4 *m1, DBMONEY4 *m2)
{
	return SUCCEED;
}
RETCODE dbdatecmp(DBPROCESS *dbproc, DBDATETIME *d1, DBDATETIME *d2)
{
	return SUCCEED;
}
RETCODE dbdatecrack(DBPROCESS *dbproc, DBDATEREC *dateinfo, DBDATETIME *datetime)
{
	return SUCCEED;
}
void dbrpwclr(LOGINREC *login)
{
}
RETCODE dbrpwset(LOGINREC *login, char *srvname, char *password, int pwlen)
{
	return SUCCEED;
}
int dbspid(DBPROCESS *dbproc)
{
	return 0;
}
RETCODE dbsetuserdata(DBPROCESS *dbproc, BYTE *ptr)
{
	dbproc->user_data = ptr;
	return SUCCEED;
}
BYTE *dbgetuserdata(DBPROCESS *dbproc)
{
	return dbproc->user_data;
}
RETCODE dbsetversion(DBINT version)
{
	g_dblib_version  = version;
	return SUCCEED;
}
RETCODE dbmnycopy(DBPROCESS *dbproc, DBMONEY *src, DBMONEY *dest)
{
	return SUCCEED;
}
RETCODE dbcanquery(DBPROCESS *dbproc)
{
	return SUCCEED;
}

void dbfreebuf(DBPROCESS *dbproc)
{
   dbproc->dbbuf[0] = '\0';
} /* dbfreebuf()  */

RETCODE dbclropt(DBPROCESS *dbproc,int option, char *param)
{
	return SUCCEED;
}
DBBOOL dbisopt(DBPROCESS *dbproc,int option, char *param)
{
	return TRUE;
}
DBINT DBCURROW(DBPROCESS *dbproc)
{
	return 0;
}
int DBCURCMD(DBPROCESS *dbproc)
{
	return 0;
}
RETCODE DBMORECMDS(DBPROCESS *dbproc)
{
	return SUCCEED;
}
int dbrettype(DBPROCESS *dbproc,int retnum)
{
	return 0;
}
int dbstrlen(DBPROCESS *dbproc)
{
	return 0;
}
RETCODE dbstrcpy(DBPROCESS *dbproc, int start, int numbytes, char *dest)
{
	return SUCCEED;
}
RETCODE dbsafestr(DBPROCESS *dbproc,char *src, DBINT srclen, char *dest, DBINT destlen, int quotetype)
{
	return SUCCEED;
}
char *dbprtype(int token)
{
   char  *result = NULL;


   switch (token)
   {
      case SYBINT1:         result = "tinyint";         break;
      case SYBINT2:         result = "smallint";        break;
      case SYBINT4:         result = "int";             break;
      case SYBMONEY:        result = "money";           break;
      case SYBFLT8:         result = "float";           break;
      case SYBDATETIME:     result = "datetime";        break;
      case SYBBIT:          result = "bit";             break;
      case SYBCHAR:         result = "char";            break;
      case SYBVARCHAR:      result = "varchar";         break;
      case SYBTEXT:         result = "text";            break;
      case SYBBINARY:       result = "binary";          break;
      case SYBVARBINARY:    result = "varbinary";       break;
      case SYBIMAGE:        result = "image";           break;
      case SYBDECIMAL:      result = "decimal";         break;
      case SYBNUMERIC:      result = "numeric";         break;
      case SYBINTN:         result = "integer-null";    break;
      case SYBDATETIMN:     result = "datetime-null";   break;
      case SYBMONEYN:       result = "money-null";      break;
      case SYBFLTN:         result = "float-null";      break;
      case SYBAOPSUM:       result = "sum";             break;
      case SYBAOPAVG:       result = "avg";             break;
      case SYBAOPCNT:       result = "count";           break;
      case SYBAOPMIN:       result = "min";             break;
      case SYBAOPMAX:       result = "max";             break;
      case SYBDATETIME4:    result = "smalldatetime";   break;
      case SYBMONEY4:       result = "smallmoney";      break;
      case SYBREAL:         result = "real";            break;
      default:              result = "";                break;
   }
   return result;
} /* dbprtype()  */
DBBINARY *dbtxtimestamp(DBPROCESS *dbproc, int column)
{
	return NULL;
}
DBBINARY *dbtxptr(DBPROCESS *dbproc,int column)
{
	return NULL;
}
RETCODE dbwritetext(DBPROCESS *dbproc,char *objname, DBBINARY *textptr,DBTINYINT textptrlen, DBBINARY *timestamp, DBBOOL log, DBINT size, BYTE *text)
{
	return SUCCEED;
}
STATUS dbreadtext(DBPROCESS *dbproc, void *buf, DBINT bufsize)
{
	return SUCCEED;
}
RETCODE dbmoretext(DBPROCESS *dbproc, DBINT size, BYTE *text)
{
	return SUCCEED;
}
void dbrecftos(char *filename)
{
}
char *dbversion()
{
	return NULL;
}
RETCODE dbsetdefcharset(char *charset)
{
	return SUCCEED;
}
RETCODE dbreginit(
      DBPROCESS *dbproc,
      DBCHAR *procedure_name,
      DBSMALLINT namelen )
{
      return SUCCEED;
}
RETCODE dbreglist(DBPROCESS *dbproc)
{
      return SUCCEED;
}
RETCODE dbregparam(
      DBPROCESS    *dbproc,
      char         *param_name,
      int          type,
      DBINT        datalen,
      BYTE         *data )
{
      return SUCCEED;
}
RETCODE dbregexec(
      DBPROCESS      *dbproc,
      DBSMALLINT    options)
{
	return SUCCEED;
}
char      *dbmonthname(DBPROCESS *dbproc,char *language,int monthnum,DBBOOL shortform)
{
char *shortmon[] = {"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
char *longmon[] = {"January","February","March","April","May","June","July","August","September","October","November","December"};

	if (shortform)
		return shortmon[monthnum-1];
	else
		return longmon[monthnum-1];
}
char      *dbname(DBPROCESS *dbproc)
{
	return NULL;
}
RETCODE dbsqlsend(DBPROCESS *dbproc)
{
int   result = FAIL;
TDSSOCKET *tds;

   tds = (TDSSOCKET *) dbproc->tds_socket;
   if (tds->res_info && tds->res_info->more_results)
   {
      /* 
       * XXX If I read the documentation correctly it gets a
       * bit more complicated than this.
       *
       * You see if the person did a query and retrieved all 
       * the rows but didn't call dbresults() and if the query 
       * didn't return multiple results then this routine should
       * just end the TDS_END_TOKEN packet and be done with it.
       *
       * Unfortunately the only way we can know that is by peeking 
       * ahead to the next byte.  Peeking could block and this is supposed
       * to be a non-blocking call.  
       *
       */
 
      result = FAIL;
   }
   else
   {
      dbproc->more_results = TRUE;
      if (tds_submit_query(dbproc->tds_socket, dbproc->dbbuf)!=TDS_SUCCEED) {
	return FAIL;
      }
      if (! dbproc->noautofree)
      {
         dbproc->dbbuf[0]='\0';
      }
      result = SUCCEED;
   }
   return result;
}
RETCODE dbaltutype(DBPROCESS *dbproc, int computeid, int column)
{
	return SUCCEED;
}
RETCODE dbaltlen(DBPROCESS *dbproc, int computeid, int column)
{
	return SUCCEED;
}
RETCODE dbpoll(DBPROCESS *dbproc, long milliseconds, DBPROCESS **ready_dbproc, int *return_reason)
{
	return SUCCEED;
}

DBINT DBLASTROW(DBPROCESS *dbproc)
{
TDSRESULTINFO * resinfo;
TDSSOCKET * tds;

	tds = (TDSSOCKET *) dbproc->tds_socket;
	resinfo = tds->res_info;
	return resinfo->row_count;
#if 0
  DBINT   result;

   if (dbproc->row_buf.rows_in_buf == 0)
   {
      result = 0;
   }
   else
   {
      /* rows returned from the row buffer start with 1 instead of 0
      ** newest is 0 based. */
      result = dbproc->row_buf.newest + 1;
   }
   return result;
#endif
}

DBINT DBFIRSTROW(DBPROCESS *dbproc)
{
   DBINT   result;

   if (dbproc->row_buf.rows_in_buf == 0)
   {
      result = 0;
   }
   else
   {
      result = dbproc->row_buf.oldest;
   }
   return result;
}
int DBIORDESC(DBPROCESS *dbproc)
{
   return dbproc->tds_socket->s;
}
int DBIOWDESC(DBPROCESS *dbproc)
{
   return dbproc->tds_socket->s;
}
 
