/*
 *  apcsetup.c  -- configuration parts for apcupsd package
 *
 *  apcupsd.c   -- Simple Daemon to catch power failure signals from a
 *                 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *              -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

/*********************************************************************/
char *setup_ups_model_V_codes (char *s, UPSINFO *ups)
{
	char *cp;

	switch(s[0]) {
		case '0': return(cp = "APC Matrix-UPS 3000");
		case '2': return(cp = "APC Smart-UPS 250");
		case '3':
		case '4': return(cp = "APC Smart-UPS 400");
		case '5': return(cp = "APC Matrix-UPS 5000");
		case '6': return(cp = "APC Smart-UPS 600");
		case '7': return(cp = "APC Smart-UPS 900");
		case '8': return(cp = "APC Smart-UPS 1250");
		case '9': return(cp = "APC Smart-UPS 2000");
		case 'A': return(cp = "APC Smart-UPS 1400");
		case 'B': return(cp = "APC Smart-UPS 1000");
		case 'C': return(cp = "APC Smart-UPS 650");
		case 'D': return(cp = "APC Smart-UPS 420");
		case 'E': return(cp = "APC Smart-UPS 280");
		case 'F': return(cp = "APC Smart-UPS 450");
		case 'G': return(cp = "APC Smart-UPS 700");
		case 'H': return(cp = "APC Smart-UPS 700 XL");
		case 'I': return(cp = "APC Smart-UPS 1000");
		case 'J': return(cp = "APC Smart-UPS 1000 XL");
		case 'K': return(cp = "APC Smart-UPS 1400");
		case 'L': return(cp = "APC Smart-UPS 1400 XL");
		case 'M': return(cp = "APC Smart-UPS 2200");
		case 'N': return(cp = "APC Smart-UPS 2200 XL");
		case 'P': return(cp = "APC Smart-UPS 3000");
		case 'O': return(cp = "APC Smart-UPS 5000");
		default:
			break;
	}
	return(cp = ups->mode.long_name);
}

/*********************************************************************/
char *setup_ups_model_b_codes (char *s, UPSINFO *ups)
{
	char *cp;

/*
 *      New Firmware revison and model ID String in NN.M.D is the format
 *         This returns two numbers and a character "D"
 *           where NN == UPS ID Code.
 *               12 == Back-UPS Pro 650
 *               13 == Back-UPS Pro 1000
 *               52 == Smart-UPS 700
 *               72 == Smart-UPS 1400
 *           where M == Possible Case Style, unknown???
 *               1 == Stand Alone
 *               8 == Rack Mount
 *               9 == Rack Mount
 *           where D == unknown???
 *
 */

	fprintf(stderr, "\n%s: 'b' %s", ups->argvalue, s);

	return(cp = ups->mode.long_name);
}

/*********************************************************************/
void setup_ups_model (UPSINFO *ups)
{
	char response[32];
	char *cp;
	unsigned char b;
	int i;
	response[0] = '\0';

	for (i=0; i<4; i++) {
		b=0x01;
		write(ups->fd, &b, 1);
		sleep(1);
	}
	getline(response, ups);

	if (strlen(response)) {
		ups->mode.long_name[0] = '\0';
		sprintf(ups->mode.long_name, "%s", response);
		return;
	}

	response[0] = '\0';
	strcpy(response, smart_poll('V', ups));

	if (strlen(response)) {
		cp = setup_ups_model_V_codes(response, ups);
		if (cp != ups->mode.long_name)
			sprintf(ups->mode.long_name, "%s", cp);
		return;
	}

	response[0] = '\0';
	strcpy(response, smart_poll('b', ups));

	if (strlen(response))
		fprintf(stderr, "\n%s: 'b' %s", ups->argvalue,
		        setup_ups_model_b_codes(response, ups));

}

#undef __DEBUG_NAME

/*********************************************************************/
void setup_ups_name (UPSINFO *ups)
{
	char response[32];
	char name[8];
	char a = '-';
	char b = '\n';
	char c = 'c';
	char n;
	int i;
	int j = strlen(ups->name);
	response[0] = '\0';
	name[0] = '\0';

#ifdef __DEBUG_NAME
	fprintf(stderr, "\n%s: NAME %s ", ups->argvalue, smart_poll('c', ups));
	fprintf(stderr, "%d NEW_NAME ", j);
#endif /* __DEBUG_NAME */

	if (j >= 8)
		j = 8;
	strncpy(name, ups->name, j);

	write(ups->fd, &c, 1);	/* c = 'c' */
	sleep(1);
	write(ups->fd, &a, 1);	/* a = '-' */
	sleep(1);

	for (i=0; i<8; i++) {
		if (i<=j) {
			n = name[i];
#ifdef __DEBUG_NAME
			fprintf(stderr, "%c", n);
#endif /* __DEBUG_NAME */
			write(ups->fd, &n, 1);
			sleep(1);
		} else {
			write(ups->fd, &b, 1);	/* b = '\n' */
			sleep(1);
		}
	}

	(void) smart_poll('c', ups);
	(void) smart_poll('c', ups);
	ups->name[0] = '\0';

	/* UPS_NAME */
	strcpy(ups->name, smart_poll('c', ups));

#ifdef __DEBUG_NAME
	fprintf(stderr, " CHANGED_NAME %s\n", ups->name);
#endif /* __DEBUG_NAME */
}

/*********************************************************************/
char *setup_ups_string (UPSINFO *ups, char cmd, char *setting)
{
	char *cp;
	char response[32];
	char tempstring[8];
	int i;
	char a = '-';
	response[0] = '\0';
	tempstring[0] = '\0';

	strcpy(tempstring, smart_poll(cmd, ups));

	/*
	 * This two loop set is for 
	 */
	for (i=0; i<2; i++) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		tempstring[0] = '\0';
		strcpy(tempstring, smart_poll(cmd, ups));
	}

	while(strcmp(tempstring, setting) != 0) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		getline(response, ups);
		tempstring[0] = '\0';
		strcpy(tempstring, smart_poll(cmd, ups));
	}
	cp = (char *)&tempstring[0];

	return(cp);
}

/*********************************************************************/
int setup_ups_single (UPSINFO *ups, char cmd, int single)
{
	char response[32];
	char a = '-';
	response[0] = '\0';

	while (single != atof(smart_poll(cmd, ups))) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
                write(ups->fd, &cmd, 1);
                getline(response, ups);
        }
	smart_poll(cmd, ups);
	return((int) atof(smart_poll(cmd, ups)));
}

/*********************************************************************/
int setup_ups_bubble (UPSINFO *ups, char cmd, int setting)
{
	char response[32];
	int temp[4];
	int lopoint[5];
	int range_temp, i;
	char a = '-';
	response[0] = '\0';

	for (i=0; i<5; i++)
		lopoint[i] = 0;

	range_temp = 1000;

	for (i=0; i<4; i++) {
		temp[i] = atof(smart_poll(cmd, ups));
		if (temp[i] < range_temp)
			range_temp = temp[i];
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		getline(response, ups);
	}
	while(range_temp != atof(smart_poll(cmd, ups))) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		getline(response, ups);
	}
        for (i=0; i<4; i++) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		getline(response, ups);
		lopoint[i] = atof(smart_poll(cmd, ups));
	}
	while(lopoint[setting] != atof(smart_poll(cmd, ups))) {
		write(ups->fd, &a, 1);
		getline(response, ups);
		if (strcmp(response, "NO") == 0)
			break;
		write(ups->fd, &cmd, 1);
		getline(response, ups);
	}
	return(atoi(smart_poll(cmd, ups)));
}

/*********************************************************************/
void setup_ups_sensitivity (UPSINFO *ups)
{
	/* SENSITIVITY */
	strcpy(ups->sensitivity, setup_ups_string(ups, 's', ups->sensitivity));
}

/*********************************************************************/
void setup_ups_wakeup (UPSINFO *ups)
{
	/* WAKEUP_DELAY */
	ups->wakeup = setup_ups_single(ups, 'r', ups->wakeup);
}

/*********************************************************************/
void setup_ups_sleep (UPSINFO *ups)
{
	/* SLEEP_DELAY */
	ups->sleep = setup_ups_single(ups, 'p', ups->sleep);
}

/*********************************************************************/
void setup_ups_lo_xfer (UPSINFO *ups)
{
	/* LOW_TRANSFER_LEVEL */
	ups->lotrans = setup_ups_bubble(ups, 'l', ups->lotrans);
}

/*********************************************************************/
void setup_ups_hi_xfer (UPSINFO *ups)
{
	/* HIGH_TRANSFER_LEVEL */
	ups->hitrans = setup_ups_bubble(ups, 'u', ups->hitrans);
}

/*********************************************************************/
void setup_ups_chargepoint (UPSINFO *ups)
{
	/* UPS_BATT_CAP_RETURN */
	ups->minon = setup_ups_bubble(ups, 'e', ups->minon);
}

/*********************************************************************/
void setup_ups_alarm (UPSINFO *ups)
{
	/* ALARM_STATUS */
	strcpy(ups->beepstate, setup_ups_string(ups, 'k', ups->beepstate));
}

/*********************************************************************/
void setup_ups_lowbatt (UPSINFO *ups)
{
	/* LOWBATT_SHUTDOWN_LEVEL */
	ups->lowbatt = setup_ups_single(ups, 'q', ups->lowbatt);
}

/*********************************************************************/
void setup_ups_selftest (UPSINFO *ups)
{
	/* UPS_SELFTEST */
	strcpy(ups->selftest, setup_ups_string(ups, 'E', ups->selftest));
}

/*********************************************************************/
void setup_UPSlink_flags (UPSINFO *ups)
{
	char response[32];
	response[0] = '\0';

	ups->UPSlink_flags = 0;

	/* UPS_STATUS */
	strcpy(response, smart_poll('Q', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000001;

	/* ONBATT_STATUS_FLAG */
	strcpy(response, smart_poll('9', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000002;

	/* POWER_FAIL_HISTORY */
	strcpy(response, smart_poll('G', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000004;

	/* UPS Self-Test Status and return on-line? */
	strcpy(response, smart_poll('X', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000008;

	/* LINE_VOLTAGE */
	strcpy(response, smart_poll('L', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000010;

	/* UPS_LINE_MAX */
	strcpy(response, smart_poll('M', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000020;

	/* UPS_LINE_MIN */
	strcpy(response, smart_poll('N', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000040;

	/* OUTPUT_VOLTAGE */
	strcpy(response, smart_poll('O', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000080;

	/* BATT_FULL */
	strcpy(response, smart_poll('f', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000100;

	/* BATT_VOLTAGE */
	strcpy(response, smart_poll('B', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000200;

	/* UPS_LOAD */
	strcpy(response, smart_poll('P', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000400;

	/* LINE_FREQ */
	strcpy(response, smart_poll('F', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00000800;

	/* UPS_RUNTIME_LEFT */
	strcpy(response, smart_poll('j', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00001000;

	/* UPS_TEMP */
	strcpy(response, smart_poll('C', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00002000;

	/* DIP_SWITCH_SETTINGS */
	strcpy(response, smart_poll('7', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00004000;

	/* UNKNOWN */
	strcpy(response, smart_poll('8', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00008000;

	/* SENSITIVITY */
	strcpy(response, smart_poll('s', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00010000;

	/* WAKEUP_DELAY */
	strcpy(response, smart_poll('r', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00020000;

	/* SLEEP_DELAY */
	strcpy(response, smart_poll('p', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00040000;

	/* LOW_TRANSFER_LEVEL */
	strcpy(response, smart_poll('l', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00080000;

	/* HIGH_TRANSFER_LEVEL */
	strcpy(response, smart_poll('u', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00100000;

	/* UPS_BATT_CAP_RETURN */
	strcpy(response, smart_poll('e', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00200000;

	/* ALARM_STATUS */
	strcpy(response, smart_poll('k', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00400000;

	/* LOWBATT_SHUTDOWN_LEVEL */
	strcpy(response, smart_poll('q', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x00800000;

	/* UPS_NAME */
	strcpy(response, smart_poll('c', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x01000000;

	/* UPS_SELFTEST */
	strcpy(response, smart_poll('E', ups));
	if (strlen(response) && (strcmp(response,"NA") != 0))
		ups->UPSlink_flags |= 0x02000000;
}

/*********************************************************************/
void setup_extended (UPSINFO *ups)
{
	setup_UPSlink_flags(ups);

	if (ups->UPSlink_flags & 0x00010000)
		setup_ups_sensitivity(ups);

	if (ups->UPSlink_flags & 0x00020000)
		setup_ups_wakeup(ups);

	if (ups->UPSlink_flags & 0x00040000)
		setup_ups_sleep(ups);

	if (ups->UPSlink_flags & 0x00080000)
		setup_ups_lo_xfer(ups);

	if (ups->UPSlink_flags & 0x00100000)
		setup_ups_hi_xfer(ups);

	if (ups->UPSlink_flags & 0x00200000)
		setup_ups_chargepoint(ups);

	if (ups->UPSlink_flags & 0x00400000)
		setup_ups_alarm(ups);

	if (ups->UPSlink_flags & 0x00800000)
		setup_ups_lowbatt(ups);

	if (ups->UPSlink_flags & 0x01000000)
		setup_ups_name(ups);

	if (ups->UPSlink_flags & 0x02000000)
		setup_ups_selftest(ups);

	setup_ups_model(ups);
}
