/*
 *  apcipc.c  -- IPC functions
 *
 *  Copyright (C) 1999 Jonathan H N Chin <jc254@newton.cam.ac.uk>
 *                1999 Facchetti Riccardo <fizban@tin.it>
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *               BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *            -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 *  Contributed by Facchetti Riccardo <fizban@tin.it>
 *
 *  Rewritten locking to use sem.h
 *  -RF
 *
 *  Thanks to Jonathan for his locking algoritm.
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#ifdef NEW_THREADS

UPSINFO *shmUPS;
int idshmUPS;
int idsemUPS;

#define READ_CNT	0
#define WRITE_LCK	1
#define NUM_SEM		2
#define NUM_SEM_OPER	3 /* 3 for write sem that need 1 operation more */

struct sembuf semUPS[NUM_SEM_OPER];
#define SEMBUF ((struct sembuf *)(&semUPS))

int init_ipc (void) {
	/*
	 * If we fail initializing IPC structures it is better destroy any
	 * semi-initialized IPC.
	 */
	if (create_shmarea() == FAILURE)
		return FAILURE;
	if (create_semaphore() == FAILURE) {
		destroy_shmarea();
		return FAILURE;
	}
	return SUCCESS;
}

int attach_ipc (void) {
	/*
	 * If we fail attaching IPC structures, the problem is local so return
	 * failure but don't touch alredy initialized IPC structures.
	 */
	if (attach_shmarea() == FAILURE)
		return FAILURE;
	if (attach_semaphore() == FAILURE)
		return FAILURE;
	return SUCCESS;
}

int detach_ipc (void) {
	if (detach_shmarea() == FAILURE) {
		return FAILURE;
	}
	return SUCCESS;
}

int destroy_ipc (void) {
	if (destroy_shmarea() == FAILURE) {
		return FAILURE;
	}
	if (destroy_semaphore() == FAILURE) {
		return FAILURE;
	}
	return SUCCESS;
}

int read_lock (void) {
	/*
	 * Acquire writer lock to be sure no other write is in progress.
	 */
	semUPS[0].sem_num	= WRITE_LCK;
	semUPS[0].sem_op	= -1;
	semUPS[0].sem_flg	= SEM_UNDO;
	/*
	 * Increment reader counter.
	 */
	semUPS[1].sem_num	= READ_CNT;
	semUPS[1].sem_op	= +1;
	semUPS[1].sem_flg	= SEM_UNDO;
	/*
	 * Release writer lock: we don't need it any more.
	 */
	semUPS[2].sem_num	= WRITE_LCK;
	semUPS[2].sem_op	= +1;
	semUPS[2].sem_flg	= SEM_UNDO;

	/*
	 * Note that the above three operations are atomic.
	 */
	if (semop(idsemUPS, SEMBUF, 3) == -1) {
		fprintf(stderr, "read_lock: can not increment read cnt.\n");
		return FAILURE;
	}
	return SUCCESS;
}

int read_unlock (void) {
	/*
	 * Reader counter must be decremented
	 */
	semUPS[0].sem_num	= READ_CNT;
	semUPS[0].sem_op	= -1;
	semUPS[0].sem_flg	= SEM_UNDO;

	if (semop(idsemUPS, SEMBUF, 1) == -1) {
		fprintf(stderr, "read_unlock: can not unlock read sem.\n");
		return FAILURE;
	}
	return SUCCESS;
}

int write_lock (void) {
	/*
	 * Acquire writer lock to be sure no other write is in progress.
	 */
	semUPS[0].sem_num	= WRITE_LCK;
	semUPS[0].sem_op	= -1;
	semUPS[0].sem_flg	= SEM_UNDO;
	if (semop(idsemUPS, SEMBUF, 1) == -1) {
		fprintf(stderr, "write_lock: can not acquire write lock.\n");
		return FAILURE;
	}
	/*
	 * Reader counter must be zero.
	 */
	semUPS[0].sem_num	= READ_CNT;
	semUPS[0].sem_op	= 0;
	semUPS[0].sem_flg	= 0;

	if (semop(idsemUPS, SEMBUF, 1) == -1) {
		fprintf(stderr, "write_lock: can not assert write sem.\n");
		return FAILURE;
	}

	/*
	 * Note that the two operations above are _not_ atomic since when we
	 * get the write lock, we can still have readers and threads that
	 * may want to increment reader counter and we can not allow new
	 * readers when we are trying to acquire writer lock. After locking
	 * for write, we can wait for reader counter to become zero because
	 * no other reader increment is possible.
	 *
	 * Jonathan, can you explain better ?
	 *
	 * -RF
	 */

	return SUCCESS;
}

int write_unlock (void) {
	/*
	 * Release writer lock.
	 */
	semUPS[0].sem_num	= WRITE_LCK;
	semUPS[0].sem_op	= +1;
	semUPS[0].sem_flg	= SEM_UNDO;

	if (semop(idsemUPS, SEMBUF, 1) == -1) {
		fprintf(stderr, "write_unlock: can not unlock write sem.\n");
		return FAILURE;
	}
	return SUCCESS;
}

int create_semaphore (void) {
	union semun sun;
	unsigned short int vals[NUM_SEM];
	
	vals[READ_CNT]	= 0;
	vals[WRITE_LCK]	= 1;

	/*
	 * Get the semaphores.
	 */
	if ((idsemUPS = semget(SEM_ID, NUM_SEM, IPC_CREAT|0644)) == -1) {
		fprintf(stderr, "create_semaphore: can not create sem.\n");
		return FAILURE;
	}

	/*
	 * Now set them up.
	 */
	sun.array = vals;
	if (semctl(idsemUPS, 0, SETALL, sun) == -1) {
		fprintf(stderr, "create_semaphore: can not set up sem.\n");
		return FAILURE;
	}

	return SUCCESS;
}

int attach_semaphore (void) {
	if ((idsemUPS = semget(SEM_ID, NUM_SEM, 0)) == -1) {
		fprintf(stderr, "attach_semaphore: can not attach sem.\n");
		return FAILURE;
	}
	return SUCCESS;
}

int destroy_semaphore (void) {
	union semun arg;

	if (semctl(idsemUPS, 0, IPC_RMID, arg) == -1) {
		fprintf(stderr, "destory_semaphore: can not destroy sem.\n");
		return FAILURE;
	}
	return SUCCESS;
}

/*
 * Theory of operation:
 *
 * we create an shm for the size of myUPS. This area will be used by all the
 * threads to get the data from serial polling thread.
 * To make sure to have a valid UPSINFO struct we use a semaphore than will be
 * asserted only when the serial polling thread is writing to the shm area so
 * that no other thread may read from that area until the write is done.
 * Multiple read lock is permitted.
 */
int create_shmarea (void) {
	if ((idshmUPS = shmget(SHM_ID, sizeof(UPSINFO), IPC_CREAT|0644))
								== -1) {
		fprintf(stderr, "create_shmarea: can not create shm area.\n");
		return FAILURE;
	}
	if ((shmUPS = (UPSINFO *)shmat(idshmUPS, 0, 0)) == (UPSINFO *)-1) { 
		fprintf(stderr, "create_shmarea: can not attach shm area.\n");
		destroy_shmarea();
		return FAILURE;
	}
	
	return SUCCESS;
}

int attach_shmarea (void) {
	if ((idshmUPS = shmget(SHM_ID, sizeof(UPSINFO), 0)) == -1) {
		fprintf(stderr, "attach_shmarea: can not get shm area.\n");
		return FAILURE;
	}
	if ((shmUPS = (UPSINFO *)shmat(idshmUPS, 0, 0)) == (UPSINFO *)-1) { 
		fprintf(stderr, "attach_shmarea: can not attach shm area.\n");
		return FAILURE;
	}

	return SUCCESS;
}

int detach_shmarea (void) {
	if (shmdt((char *)shmUPS) != 0)
		return FAILURE;


	return SUCCESS;
}

int destroy_shmarea (void) {
	if (shmctl(idshmUPS, IPC_RMID, NULL) == -1) {
		fprintf(stderr, "destroy_shmarea: can not destroy shm area.\n");
		return FAILURE;
	}

	return SUCCESS;
}

int read_shmarea (UPSINFO *ups) {
	if (read_lock() == FAILURE)
		return FAILURE;
	memcpy(ups, shmUPS, sizeof(UPSINFO));
	return read_unlock();
}

int write_shmarea (UPSINFO *ups) {
	if (write_lock() == FAILURE)
		return FAILURE;
	memcpy(shmUPS, ups, sizeof(UPSINFO));
	return write_unlock();
}

/*
 * To do at the very start of a thread, to sync data from the common data
 * structure.
 */
int read_andlock_shmarea (UPSINFO *ups) {
	if (write_lock() == FAILURE)
		return FAILURE;
	memcpy(ups, shmUPS, sizeof(UPSINFO));
	return SUCCESS;
}

/*
 * To do at the very end of a thread, to sync data to the common data
 * structure.
 */
int write_andunlock_shmarea (UPSINFO *ups) {
	memcpy(shmUPS, ups, sizeof(UPSINFO));
	return write_unlock();
}

/*
 * For read-only requests.  No copying is done.
 * Danger! Use with caution:
 *
 * 1) Don't modify anything in the shmUPS-structure while holding this lock.
 * 2) Don't hold the lock for too long.
 */
UPSINFO * lock_shmups (void) {
	if (read_lock() == FAILURE)
		return NULL;
	return shmUPS;
}

void unlock_shmups (void) {
	read_unlock();
}
#endif /* NEW_THREADS */
