/*
 *  apcconfig.c  -- configuration parts for apcupsd package
 *
 *  apcupsd.c    -- Simple Daemon to catch power failure signals from a
 *                  BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *               -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 * Rewritten by:
 *      Riccardo Facchetti <fizban@tin.it>
 *      Jonathan H N Chin  <jc254@newton.cam.ac.uk>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

/* ---------------------------------------------------------------------- */

#define WHERE(MEMBER) ((size_t) &((UPSINFO *)0)->MEMBER)
#define AT(UPS,OFFSET) ((size_t)UPS + OFFSET)

typedef int (HANDLER) (UPSINFO *, size_t, GENINFO *, const char *);

static HANDLER match_cmd, match_int, match_log, match_range,
               match_slave, match_str;
static HANDLER obsolete;

/* ---------------------------------------------------------------------- */

static GENINFO accesses[] = {
	{ "false",  "false",                 FALSE }, /* must come first */
	{ "true",   "true",                  TRUE },
	{ NULL,     "*invalid-access-mode*", FALSE },
};

static GENINFO cables[] = {
	{ "simple",    "Custom Cable Simple", CUSTOM_SIMPLE },
	{ "smart",     "Custom Cable Smart",  CUSTOM_SMART },
	{ "ether",     "Ethernet Link",       APC_NET },
	{ "940-0020B", "APC Cable 940-0020B", APC_940_0020B },
	{ "940-0023A", "APC Cable 940-0023A", APC_940_0023A },
	{ "940-0024B", "APC Cable 940-0024B", APC_940_0024B },
	{ "940-0024C", "APC Cable 940-0024C", APC_940_0024C },
	{ "940-0024G", "APC Cable 940-0024G", APC_940_0024G },
	{ "940-0095A", "APC Cable 940-0095A", APC_940_0095A },
	{ "940-0095C", "APC Cable 940-0095C", APC_940_0095C },
	{ NULL,        "*invalid-cable*",     NO_CABLE  },
};

static GENINFO classes[] = {
	{ "standalone",     "Stand Alone",           STANDALONE },
	{ "shareslave",     "ShareUPS Slave",        SHARESLAVE },
	{ "sharemaster",    "ShareUPS Master",       SHAREMASTER },
	{ "netslave",       "Net Slave",             NETSLAVE },
	{ "netmaster",      "Net Master",            NETMASTER },
	{ "sharenetmaster", "ShareUPS & Net Master", SHARENETMASTER },
	{ NULL,             "*invalid-ups-class*",   NO_CLASS },
};

static GENINFO logins[] = {
	{ "always",  "always",               ALWAYS }, /* must come first */
	{ "disable", "disable",              NEVER },
	{ "timeout", "timeout",              TIMEOUT },
	{ "percent", "percent",              PERCENT },
	{ "minutes", "minutes",              MINUTES },
	{ NULL,      "*invalid-login-mode*", NO_LOGON },
};

static GENINFO modes[] = {
	{ "disable",  "Network & ShareUPS Disabled", DISABLE },
	{ "share",    "ShareUPS",                    SHARE },
	{ "net",      "NetworkUPS",                  NET },
	{ "sharenet", "Network & ShareUPS",          SHARENET },
	{ NULL,       "*invalid-ups-mode*",          NO_SHARE_NET },
};

static GENINFO types[] = {
	{ "backups",       "BackUPS",                   BK },
	{ "sharebasic",    "ShareUPS Basic Port",       SHAREBASIC },
	{ "netups",        "NetUPS Virtual Basic Port", NETUPS },
	{ "backupspro",    "BackUPS Pro",               BKPRO },
	{ "smartvsups",    "SmartUPS VS",               VS },
	{ "newbackupspro", "Smarter BackUPS Pro",       NBKPRO },
	{ "backupspropnp", "Smarter BackUPS Pro",       NBKPRO },
	{ "smartups",      "SmartUPS",                  SMART },
	{ "matrixups",     "MatrixUPS",                 MATRIX },
	{ "sharesmart",    "ShareUPS Advanced Port",    SHARESMART },
	{ NULL,            "*invalid-ups-type*",        NO_UPS },
};

static struct pairs {
	const char *key;
	HANDLER *handler;
	size_t offset;
	GENINFO *values;
} table[] = {

	{ "ANNOY",        match_int, WHERE(annoy),      0 },
	{ "BATTERYLEVEL", match_int, WHERE(percent),    0 },
	{ "DELAY",        match_int, WHERE(delay),      0 },
	{ "HITRANSFER",   match_int, WHERE(hitrans),    0 },
	{ "HTTPPORT",     match_int, WHERE(httpport),   0 },
	{ "LOGGING",      match_int, WHERE(logtime),    0 },
	{ "LOTRANSFER",   match_int, WHERE(lotrans),    0 },
	{ "LOWBATT",      match_int, WHERE(lowbatt),    0 },
	{ "MINUTES",      match_int, WHERE(runtime),    0 },
	{ "NETPORT",      match_int, WHERE(NetUpsPort), 0 },
	{ "NETTIME",      match_int, WHERE(nettime),    0 },
	{ "PROCFS",       match_int, WHERE(proctime),   0 },
	{ "RETURNCHARGE", match_int, WHERE(minon),      0 },
	{ "SLEEP",        match_int, WHERE(sleep),      0 },
	{ "TIMEOUT",      match_int, WHERE(maxtime),    0 },
	{ "WAKEUP",       match_int, WHERE(wakeup),     0 },

	{ "CONTROL",   match_cmd, WHERE(control_path), 0 },
	{ "BATTCMD",   match_cmd, WHERE(batt_cmd),     0 },
	{ "LIMITCMD",  match_cmd, WHERE(limit_cmd),    0 },
	{ "LOADCMD",   match_cmd, WHERE(load_cmd),     0 },
	{ "PWRCMD",    match_cmd, WHERE(pwr_cmd),      0 },
	{ "REBOOTCMD", match_cmd, WHERE(reboot_cmd),   0 },
	{ "REMOTECMD", match_cmd, WHERE(remote_cmd),   0 },
	{ "RETCMD",    match_cmd, WHERE(ret_cmd),      0 },
	{ "TIMECMD",   match_cmd, WHERE(time_cmd),     0 },

	{ "LOGGING", match_log,    -1, 0 },
	{ "SLAVE",   match_slave,  -1, 0 },

	{ "BEEPSTATE",   match_str, WHERE(beepstate),   0 },
	{ "DEVICE",      match_str, WHERE(device),      0 },
	{ "LOCKFILE",    match_str, WHERE(lockpath),    0 },
	{ "MASTER",      match_str, WHERE(master_name), 0 },
	{ "SELFTEST",    match_str, WHERE(selftest),    0 },
	{ "SENSITIVITY", match_str, WHERE(sensitivity), 0 },
	{ "UPSNAME",     match_str, WHERE(name),        0 },
	{ "USERMAGIC",   match_str, WHERE(usermagic),   0 },

	{ "ACCESS",     match_range, WHERE(enable_access), accesses },
	{ "HTTPACCESS", match_range, WHERE(enable_http),   accesses },
	{ "NOLOGON",    match_range, WHERE(nologin),       logins },
	{ "UPSCABLE",   match_range, WHERE(cable),         cables },
	{ "UPSCLASS",   match_range, WHERE(class),         classes },
	{ "UPSMODE",    match_range, WHERE(sharenet),      modes },
	{ "UPSTYPE",    match_range, WHERE(mode),          types },

	/*
	 * Obsolete configuration options: to be removed in the future.
	 * Made obsolete with 3.9.1.
	 * The warning string is passed into the GENINFO * field since it is
	 * not used any more for obsoleted options: we are only interested in
	 * printing the message.
	 * There is a new meaning for offset field too. If TRUE will bail out,
	 * if FALSE it will continue to run apcupsd. This way we can bail out
	 * if an obsolete option is too sensible to continue running apcupsd.
	 * -RF
	 *
	 * An example entry may be:

	{ "CONTROL",   obsolete, TRUE, (GENINFO *)
		"CONTROL config option is obsolete, use /usr/lib/apcupsd/ "
		"scripts instead." },
	 *
	 *
	 */
	{ "JUSTOSHUTUPCOMPILER",   obsolete, TRUE, (GENINFO *)
		"This message should never be seen. It is intended to "
		"shut up compiler." },

	/* must be last */
	{ NULL, 0, -1, 0 },
};

static int obsolete (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	char *msg = (char *)junk;
	char flag = (int)offset;
	fprintf(stderr, "%s\n", msg);

	return ((flag == TRUE) ? FAILURE : SUCCESS);
}

/* ---------------------------------------------------------------------- */

/* JHNC-FIXME: any particular reason not to do case-insensitive matching? */

/* ---------------------------------------------------------------------- */

static int match_cmd (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	struct stat statbuf;
	char x[MAXSTRING];

	if (!sscanf(v, "%s", x))
		return FAILURE;

	/* check existence and exec bit */
/*
 * JHNC-FIXME (remove/replace this comment once fixed)
 *
 * Why are we checking S_IEXEC?
 * We want the program to be executable by *us*, not just the owner, surely?
 * And why was it |S_IEXEC anyhow?
 *
 * Also, my stat(2) manpage says S_IXUSR is the POSIX version
 */
	if (!((stat(x, &statbuf) == 0) && (S_ISREG(statbuf.st_mode))
	      && (statbuf.st_mode & S_IXUSR))) {
		fprintf(stderr, "%s: Invalid Command: %s\n", ups->argvalue, x);
		return FAILURE;
	}
	strcpy((char *)AT(ups,offset), x);
	return SUCCESS;
}

static int match_int (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	int x;
	if (sscanf(v, "%d", &x)) {
		*(int *)AT(ups,offset) = x;
		return SUCCESS;
	}
	return FAILURE;
}

/*
 * JHNC-FIXME (remove/replace this comment once fixed)
 *
 * Just what is "__NEW_LOGS" anyway?
 * If it doesn't exist, this special case match_log can go away
 * and we just use match_int.
 *
 * Some silly person wanted to move the location of the logs.
 * Something about running a static version of the OS on a cdrom.
 * Seemed reasonable at the time............
 * AMH Wed Dec  9 19:46:46 CST 1998
 */
#ifdef __NEW_LOGS
static int match_log (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	char x[MAXSTRING];
	int time;

	if (!sscanf(v, "%d %s", &time, x))
		return FAILURE;

	strcpy(ups->logfile, x);
	ups->logtime = time;
	return SUCCESS;
}
#else
static int match_log (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	int time;

	if (!sscanf(v, "%d", &time))
		return FAILURE;

	ups->logtime = time;
	return SUCCESS;
}
#endif

static int match_range (UPSINFO *ups, size_t offset,
                        GENINFO *vs, const char *v) {
	char x[MAXSTRING];
	INTERNALGENINFO *t;

	if (!vs) {
		/*
		 * Shouldn't ever happen so exit if it ever does.
		 */
		fprintf(stderr,
		        "%s: Bogus configuration table! Fix and recompile.\n",
			ups->argvalue);
		exit(1);
	}

	if (!sscanf(v, "%s", x))
		return FAILURE;

	for (; vs->name; vs++)
		if (!strcmp(x, vs->name))
			break;

	t = (INTERNALGENINFO *)AT(ups,offset);

	/* Copy the structure */
	if (vs->name == NULL) {
		/*
		 * A NULL here means that the config value is bogus, so
		 * print an error message and return error.
		 */
		fprintf(stderr,
		        "%s: Bogus configuration value (%s)\n",
			ups->argvalue, vs->long_name);
		return FAILURE;
	}

	t->type = vs->type;
	strcpy(t->name, vs->name);
	strcpy(t->long_name, vs->long_name);
	return SUCCESS;
}

static int match_slave (UPSINFO *ups, size_t offset,
                        GENINFO *junk, const char *v) {
	char x[MAXSTRING];

	if (!sscanf(v, "%s", x))
		return FAILURE;

	if (ups->slave_count >= MAXSLAVES) {
		fprintf(stderr, "%s: Exceeded max slaves number (%d)\n",
		        ups->argvalue, MAXSLAVES);
		return FAILURE;
	}

	strcpy(ups->slaves[ups->slave_count].name, x);
	ups->slave_count++;

	return SUCCESS;
}

/*
 * JHNC-FIXME (remove/replace this comment once fixed)
 *
 * Do we ever want str to contain whitespace?
 * If so, we can't use sscanf(3)
 */
static int match_str (UPSINFO *ups, size_t offset,
                      GENINFO *junk, const char *v) {
	char x[MAXSTRING];

	if (!sscanf(v, "%s", x))
		return FAILURE;

	strcpy((char *)AT(ups,offset), x);
	return SUCCESS;
}

/* ---------------------------------------------------------------------- */

/**
 ** JHNC:
 **
 ** This function has been so buggy, I thought commentary was needed.
 ** Be careful about changing anything. Make sure you understand what is
 ** going on first. Even the people who wrote it kept messing it up... 8^)
 **
 **/
static int ParseConfig (UPSINFO *ups, char *line) {
	struct pairs *p;
	char *key, *value;
/**
 ** Hopefully my notation is obvious enough:
 **   (a) : a
 **   a|b : a or b
 **    a? : zero or one a
 **    a* : zero or more a
 **    a+ : one or more a
 **
 ** On entry, situation is expected to be:
 **
 ** line  = WS* ((KEY (WS+ VALUE)* WS* (WS+ COMMENT)?) | COMMENT?) (EOL|EOF)
 ** key   = undef
 ** value = undef
 **
 ** if line is not of this form we will eventually return an error
 ** line, key, value point to null-terminated strings
 ** EOF may be end of file, or if it occurs alone signifies null string
 **
 **/
	/* skip initial whitespace */
	for (key = line; *key && isspace(*key); key++)
		;
/**
 ** key   = ((KEY (WS+ VALUE)* WS* (WS+ COMMENT)?) | COMMENT?) (EOL|EOF)
 ** value = undef
 **/
	/* catch EOF (might actually be only an empty line) and comments */
	if (!*key || (*key == '#'))
		return(SUCCESS);
/**
 ** key   = (KEY (WS+ VALUE)* WS* (WS+ COMMENT)? (EOL|EOF)
 ** value = undef
 **/
	/* convert key to UPPERCASE */
	for (value = key; *value && !isspace(*value); value++)
		*value = toupper(*value);
/**
 ** key   = KEY (WS+ VALUE)* WS* (WS+ COMMENT)? (EOL|EOF)
 ** value = (WS+ VALUE)* WS* (WS+ COMMENT)? (EOL|EOF)
 **/
	/* null out whitespace in the string */
	for (; *value && isspace(*value); value++)
		*value = '\0';
/**
 ** key   = KEY
 ** value = (VALUE (WS+ VALUE)* WS* (WS+ COMMENT)? (EOL|EOF))
 **       | (COMMENT (EOL|EOF))
 **       | EOF
 **
 ** key is now fully `normalised' (no leading or trailing garbage)
 ** value contains zero or more whitespace delimited elements and there
 ** may be a trailing comment.
 **/
	/* look for key in table */
	for (p = table; p->key && strcmp(p->key, key); p++)
		;
/**
 ** It is *not* the responsibility of a dispatcher (ie. ParseConfig())
 ** to parse `value'.
 ** Parsing `value' is the responsibility of the handler (ie. p->handler()).
 ** (Although one could conceive of the case where a ParseConfig()-alike
 ** function were invoked recursively as the handler...!)
 **
 ** Currently all the handler functions (match_int(), match_str(), etc)
 ** just use sscanf() to pull out the bits they want out of `value'.
 **
 ** In particular, if "%s" is used, leading/trailing whitespace
 ** is automatically removed.
 **
 ** In addition, unused bits of value are simply discarded if the handler
 ** does not use them. So trailing garbage on lines is unimportant.
 **/
	if (p->key) {
		if (p->handler)
			return(p->handler(ups, p->offset, p->values, value));
		else
			return(SUCCESS);
	} else {
		return(FAILURE);
	}
}

/* ---------------------------------------------------------------------- */

void zero_ups_struct (UPSINFO *ups) {

	memset(ups, 0, sizeof(UPSINFO));
	ups->fd			= -1;

	ups->enable_access.type	= FALSE;
	ups->enable_http.type	= FALSE;

        strcpy(ups->enable_access.name, accesses[0].name);
        strcpy(ups->enable_access.long_name, accesses[0].long_name);
        ups->enable_access.type	= accesses[0].type;

	strcpy(ups->enable_http.name, accesses[0].name);
	strcpy(ups->enable_http.long_name, accesses[0].long_name);
	ups->enable_http.type	= accesses[0].type;	

        strcpy(ups->nologin.name, logins[0].name);
        strcpy(ups->nologin.long_name, logins[0].long_name);
        ups->nologin.type	= logins[0].type;

	ups->annoy		= 60;
	ups->maxtime		= 0;
	ups->delay		= 30;
	ups->nologin_time	= 0;
	ups->nologin_file	= FALSE;

	ups->logtime		= 0;
	ups->proctime		= 0;
	ups->reports		= FALSE;
	ups->nettime		= 60;
	ups->percent		= 10;
	ups->runtime		= 5;

	strcpy(ups->sensitivity, "H");
	ups->wakeup		= 0;
	ups->sleep		= 20;

	strcpy(ups->selftest,    "336");
	ups->lotrans		= 2;
	ups->hitrans		= 2;
	ups->minon		= 3;
	ups->lowbatt		= 2;

	strcpy(ups->beepstate,   "0");

/*
 * JHNC-FIXME (remove/replace this comment once fixed)
 *
 * On a badly configured system (root account with badly setup path, say)
 * "reboot" will not find the reboot command.
 *
 * Also, for security, shouldn't ups->*_cmd be the full path to executables?
 * Invoking system with an unsanitized path/environment is dodgy.
 */
	strcpy(ups->reboot_cmd, "reboot");

	ups->httpport		= (ups->enable_http.type) ? 1999 : 0 ;

	ups->NetUpsPort		= 0;
	ups->slave_bool		= FALSE;
	ups->slave_count	= 0;
	ups->send_slaves_urg	= FALSE;

	ups->lockfile		= -1;

	ups->load		= FALSE;
	ups->timedout		= FALSE;
	ups->timelout		= FALSE;
	ups->emergencydown	= FALSE;
	ups->remotedown		= FALSE;
	ups->remote_state	= TRUE;
}

int check_for_config (char *argvalue, UPSINFO *ups)
{
	FILE *apcconf;
	char line[MAXTOKENLEN];
	int errors = 0;
	int erpos = 0;

	zero_ups_struct(ups);
	sprintf(ups->argvalue, "%s", argvalue);

	if ((apcconf = fopen(APCCONF, "r")) == NULL) return(1);

	/* JHNC: check configuration file format is a suitable version */
	if (fgets(line, sizeof(line), apcconf) != NULL) {
		/*
		 * The -1 in sizeof is there because of the last character
		 * to be checked. In line is '\n' and in APC... is '\0'.
		 * They never match so don't even compare them.
		 * -RF
		 */
		if (strncmp(line, APC_CONFIG_MAGIC, sizeof(APC_CONFIG_MAGIC)-1)
					!= 0) {
			fprintf(stderr,
				"%s: Warning: old configuration file found.\n\n"
				"%s: Expected: \"%s\"\n"
				"%s: Found:    \"%s\"\n\n"
				"%s: Please check new file format and\n"
				"%s: modify accordingly the first line\n"
				"%s: of config file.\n\n"
				"%s: Processing config file anyway.\n",
				ups->argvalue,
				ups->argvalue, APC_CONFIG_MAGIC,
				ups->argvalue, line,
				ups->argvalue, ups->argvalue,
				ups->argvalue, ups->argvalue);
		}
		/*
		 * Here we have read alredy the first line of configuration
		 * so there may be the case where the first line is a config
		 * parameter and we must not discard it reading another line.
		 * Jump into the reading configuration loop.
		 * -RF
		 */
		goto jump_into_the_loop;
	}

	while (fgets(line, sizeof(line), apcconf) != NULL) {
jump_into_the_loop:
		erpos++;

		if (ParseConfig(ups, line)) {
			errors++;
			printf("Error at line [%d] reading config file %s.\n",
				erpos, APCCONF);
		}
	}
	fclose(apcconf);

	/*
	 * The next step will need a good ups struct.
	 * Of course if here we have errors, the apc struct is not good
	 * so don't bother to post-process it.
	 *
	 * -RF
	 */
	if (errors)
		goto bail_out;

	/*
	 * post-process the configuration stored in the ups structure
	 */

	if (ups->class.type != NETSLAVE)
		ups->usermagic[0] = '\0';

	if (ups->annoy >= ups->delay)
		ups->delay = 0;

	if ((ups->sharenet.type == SHAREMASTER) ||
	    (ups->sharenet.type == SHARENETMASTER)) {
		ups->maxtime = 0;
		ups->percent = 10;
		ups->runtime = 5;
	}

	if ((ups->cable.type < CUSTOM_SMART) && ups->mode.type >= BKPRO) {
		fprintf(stderr, "%s: Error :: Changing UPSTYPE from %s ",
		        ups->argvalue, ups->mode.long_name);

		/*
		 * No errors expected from this operation.
		 * -RF
		 */
		errors += match_range(ups, WHERE(mode), types, "backups");

		fprintf(stderr, "to %s due wrong Cable of Smart Signals.\n\a",
		        ups->mode.long_name);

		if (errors) {
			fprintf(stderr, "Lookup operation failed:"
					"bad 'types' table in apcconfig.c\n");
			goto bail_out;
		}
	}

	if (ups->control_path[0]) {

	sprintf(ups->annoyme,         "%s ANNOYME",    ups->control_path);
	sprintf(ups->changeme,        "%s CHANGEME",   ups->control_path);
	sprintf(ups->doshutdown,      "%s DOSHUTDOWN", ups->control_path);
	sprintf(ups->emergency,       "%s EMERGENCY",  ups->control_path);
	sprintf(ups->failing,         "%s FAILING",    ups->control_path);
	sprintf(ups->loadlimit,       "%s LOADLIMIT",  ups->control_path);
	sprintf(ups->mainsback,       "%s MAINSBACK",  ups->control_path);
	sprintf(ups->onbatt,          "%s ONBATTERY",  ups->control_path);
	sprintf(ups->powerout,        "%s POWEROUT",   ups->control_path);
	sprintf(ups->remote_shutdown, "%s REMOTE",     ups->control_path);
	sprintf(ups->restartme,       "%s RESTARTME",  ups->control_path);
	sprintf(ups->runlimit,        "%s RUNLIMIT",   ups->control_path);
	sprintf(ups->timeout,         "%s TIMEOUT",    ups->control_path);
	sprintf(ups->waitasec,        "%s WAITASEC",   ups->control_path);

	}

	/*
	 * apcupsd _must_ have a lock file, mainly for security reasons.
	 * If apcupsd is running and a lock is not there the admin could
	 * mistakenly open a serial device with minicom for using it as a
	 * modem or other device. Think about the implications of sending
	 * extraneous characters to the UPS: a wrong char and the machine
	 * can be powered down.
	 * No thanks.
	 * On the other hand if a lock file is there, minicom (like any
	 * other serious serial program) will refuse to open the device.
	 *
	 * About "/var/lock//LCK..." nicety ... the right word IMHO is
	 * simplify, so don't bother.
	 *
	 * -RF
	 */
	if (ups->lockpath[0] == '\0')
		strcpy(ups->lockpath, LOCK_DEFAULT);

	/*
	 * If APC_NET, the lockfile is not needed.
	 */
	if (ups->cable.type != APC_NET) {
		char *dev = strrchr(ups->device, '/');

		strcat(ups->lockpath, APC_LOCK_PREFIX);
		strcat(ups->lockpath, dev?++dev:ups->device);
	} else {
		ups->lockpath[0] = 0;
		ups->lockfile = -1;
	}

	if ((ups->slave_count > 0) && ups->master_name[0]) {
		fprintf(stderr, "I can't be both MASTER and SLAVE\n");
		errors++;
		goto bail_out;
	}

	switch (ups->nologin.type) {
		case TIMEOUT:
			if (ups->maxtime != 0)
				ups->nologin_time = (int)(ups->maxtime * 0.9);
			break;
		case PERCENT:
			ups->nologin_time = (int)(ups->percent * 1.1);
			if (ups->nologin_time == ups->percent)
				ups->nologin_time++;
			break;
		case MINUTES:
			ups->nologin_time = (int)(ups->runtime * 1.1);
			if (ups->nologin_time == ups->runtime)
				ups->nologin_time++;
			break;
		default:
			break;
	}

	if ((ups->logtime) || (ups->proctime))
		ups->reports = TRUE;

bail_out:
	return (errors ? FAILURE : SUCCESS);
}
