/*

Name:
DRV_SUN.C

Description:
Mikmod driver for output on the Sun audio device (/dev/audio).
Based on the linux drv_vox.c.

Author:
Valtteri Vuorikoski <vuori@sci.fi>

Portability:
SunOS, Solaris. 

*/
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef SUNOS
#include <sun/audioio.h>
#else
#include <sys/audioio.h>
#endif /* SUNOS */

#include <sys/ioctl.h>
#include <sys/wait.h>

#include "mikmod.h"

static int sndfd;
static int play_encoding = -1, play_precision; /* we need these in a few places */

/* increase this to get better cpu efficiency, decrease for better
   interactive performance */
#define FRAGSIZE 512

/*

This is from/based on the SOX package (libst.h and raw.c) by Lance Norskog.
Parts of this..
** Copyright (C) 1989 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.

*/
#define st_linear_to_ulaw(linearword) ulaw_comp_table[(linearword / 4) & 0x3fff]

extern unsigned char ulaw_comp_table[];

/* this converts unsigned linear data from VC_WriteBytes() to ulaw or signed linear */
long unsignedtoulaw(char *buf, int nsamp)
{
   register long datum;
   long done = 0;

   if( play_encoding != AUDIO_ENCODING_ULAW && play_precision != 8 )
       return done;

   while(done < nsamp)
   {
	if( play_precision == 8 )
	{
		datum = (long) *((unsigned char *) buf);
		datum ^= 128; /* convert to signed */
	}
	if( play_encoding == AUDIO_ENCODING_ULAW )
	{
		datum = datum << 8;
        	/* round up to 12 bits of data */
        	datum += 0x8;  /* + 0b1000 */
		datum = st_linear_to_ulaw(datum);
	}
	*buf++ = (char) datum;
	done++;
    }

    return done;
}
/* end of soxism */

BOOL Sun_IsThere(void)
{
	return (access("/dev/audio",W_OK)==0);
}


BOOL Sun_Init(void)
{
	int play_stereo, play_rate, audiotype;
	struct audio_info audioinfo;

	if( (sndfd=open("/dev/audio",O_WRONLY)) < 0 ) 
	{
		myerr="Cannot open audio device";
		return 0;
	}

#ifdef SUNOS
	if( ioctl( sndfd, AUDIO_GETDEV, &audiotype ) < 0 ) {
		printf( "warning: could not determine audio device type\n" );
		/* attempt to guess the encoding */
		play_encoding = -1;
	}
	else {
		switch(audiotype) {
		case AUDIO_DEV_AMD:
		printf( "You have the amd audio device\n" );
		play_encoding = AUDIO_ENCODING_ULAW;
		break;

		case AUDIO_DEV_SPEAKERBOX:
		case AUDIO_DEV_CODEC:
		printf( "You have the dbri audio device. warning: this is untested\n" );
		play_encoding = AUDIO_ENCODING_LINEAR;
		break;

		case AUDIO_DEV_UNKNOWN:
		default:
		printf( "You have an unrecognized audio device. warning: this is untested\n" );
		play_encoding = AUDIO_ENCODING_ULAW;
		break;
		}
	}
#endif /* SUNOS */

	play_precision = (md_mode & DMODE_16BITS) ? 16 : 8;
	play_stereo = (md_mode & DMODE_STEREO) ? 2 : 1;
	play_rate = md_mixfreq;

/* Solaris supports AUDIO_GETDEV, but it returns an entirely different thing
 * (audio_device_t) and the values for it do not appear to be documented
 * anywhere, contrary to what audio(7) says. Therefore, we attempt a guess
 * at what to use based on what the user tells us to use. We also
 * attempt a guess when we can't determine the type of audio device on
 * SunOS */
#ifdef SUNOS
	if( play_encoding == -1 ) {
#endif /* SUNOS */
		if( play_precision == 8 && play_stereo == 1 && play_rate <= 8000 )
			play_encoding = AUDIO_ENCODING_ULAW;
		else
			play_encoding = AUDIO_ENCODING_LINEAR;
#ifdef SUNOS
	}
#endif /* SUNOS */

	if( play_precision != 8 && play_encoding == AUDIO_ENCODING_ULAW )
	{
		myerr = "16-bit samples not supported with ulaw encoding";
		close( sndfd );
		return 0;
	}
	if( (play_stereo != 1 || play_rate != 8000) && AUDIO_ENCODING_ULAW )
		printf( "Warning: not using 8khz mono with ulaw. This might not sound good\n" );

	AUDIO_INITINFO( &audioinfo );
	audioinfo.play.precision = play_precision;
	audioinfo.play.channels = play_stereo;
	audioinfo.play.sample_rate = play_rate;
	audioinfo.play.encoding = play_encoding;
	
	if( ioctl( sndfd, AUDIO_SETINFO, &audioinfo ) < 0 )
	{
		myerr = "Initialization of the audio device failed";
		close( sndfd );
		return 0;
	}

	if( !VC_Init() ) 
	{
		close(sndfd);
		return 0;
	}

	return 1;
}


void Sun_Exit(void)
{
	VC_Exit();
	close(sndfd);
}


void Sun_PlayStart(void)
{
	VC_PlayStart();
}


void Sun_PlayStop(void)
{
	VC_PlayStop();
}


char audiobuffer[32768];

void Sun_Update(void)
{

  fd_set fds;
  struct timeval tv = {0,0};
  FD_ZERO(&fds);
  FD_SET(sndfd,&fds);
  if( select( sndfd+1, 0, &fds, 0, &tv ) > 0 ) {
  	int buf_size = md_mixfreq;
  	if( md_mode & DMODE_16BITS ) buf_size <<= 1;
  	if( md_mode & DMODE_STEREO ) buf_size <<= 1;
  	buf_size /= 10;
	VC_WriteBytes(audiobuffer,buf_size);
	unsignedtoulaw(audiobuffer,buf_size);
	write(sndfd,audiobuffer,buf_size);
  }
}


DRIVER drv_sun={
	NULL,
	"Sun Audio",
	"MikMod Sun Audio Driver v1.1",
	Sun_IsThere,
	VC_SampleLoad,
	VC_SampleUnload,
	Sun_Init,
	Sun_Exit,
	Sun_PlayStart,
	Sun_PlayStop,
	Sun_Update,
	VC_VoiceSetVolume,
	VC_VoiceSetFrequency,
	VC_VoiceSetPanning,
	VC_VoicePlay
};

/*

this table is used to convert signed linear data to ulaw

this is part of libst.c from the SOX package by Lance Norskog.

*/
unsigned char ulaw_comp_table[16384] = {
	0xff,0xfe,0xfe,0xfd,0xfd,0xfc,0xfc,0xfb,
	0xfb,0xfa,0xfa,0xf9,0xf9,0xf8,0xf8,0xf7,
	0xf7,0xf6,0xf6,0xf5,0xf5,0xf4,0xf4,0xf3,
	0xf3,0xf2,0xf2,0xf1,0xf1,0xf0,0xf0,0xef,
	0xef,0xef,0xef,0xee,0xee,0xee,0xee,0xed,
	0xed,0xed,0xed,0xec,0xec,0xec,0xec,0xeb,
	0xeb,0xeb,0xeb,0xea,0xea,0xea,0xea,0xe9,
	0xe9,0xe9,0xe9,0xe8,0xe8,0xe8,0xe8,0xe7,
	0xe7,0xe7,0xe7,0xe6,0xe6,0xe6,0xe6,0xe5,
	0xe5,0xe5,0xe5,0xe4,0xe4,0xe4,0xe4,0xe3,
	0xe3,0xe3,0xe3,0xe2,0xe2,0xe2,0xe2,0xe1,
	0xe1,0xe1,0xe1,0xe0,0xe0,0xe0,0xe0,0xdf,
	0xdf,0xdf,0xdf,0xdf,0xdf,0xdf,0xdf,0xde,
	0xde,0xde,0xde,0xde,0xde,0xde,0xde,0xdd,
	0xdd,0xdd,0xdd,0xdd,0xdd,0xdd,0xdd,0xdc,
	0xdc,0xdc,0xdc,0xdc,0xdc,0xdc,0xdc,0xdb,
	0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xda,
	0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xd9,
	0xd9,0xd9,0xd9,0xd9,0xd9,0xd9,0xd9,0xd8,
	0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd7,
	0xd7,0xd7,0xd7,0xd7,0xd7,0xd7,0xd7,0xd6,
	0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0xd5,
	0xd5,0xd5,0xd5,0xd5,0xd5,0xd5,0xd5,0xd4,
	0xd4,0xd4,0xd4,0xd4,0xd4,0xd4,0xd4,0xd3,
	0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0xd2,
	0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xd1,
	0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd0,
	0xd0,0xd0,0xd0,0xd0,0xd0,0xd0,0xd0,0xcf,
	0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,
	0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xce,
	0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xce,
	0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xcd,
	0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,
	0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcc,
	0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,
	0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,0xcb,
	0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,
	0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xca,
	0xca,0xca,0xca,0xca,0xca,0xca,0xca,0xca,
	0xca,0xca,0xca,0xca,0xca,0xca,0xca,0xc9,
	0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,
	0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc8,
	0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,
	0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc7,
	0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,
	0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc6,
	0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,
	0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,0xc5,
	0xc5,0xc5,0xc5,0xc5,0xc5,0xc5,0xc5,0xc5,
	0xc5,0xc5,0xc5,0xc5,0xc5,0xc5,0xc5,0xc4,
	0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,
	0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc3,
	0xc3,0xc3,0xc3,0xc3,0xc3,0xc3,0xc3,0xc3,
	0xc3,0xc3,0xc3,0xc3,0xc3,0xc3,0xc3,0xc2,
	0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
	0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc1,
	0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,
	0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc0,
	0xc0,0xc0,0xc0,0xc0,0xc0,0xc0,0xc0,0xc0,
	0xc0,0xc0,0xc0,0xc0,0xc0,0xc0,0xc0,0xbf,
	0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
	0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
	0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
	0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbe,
	0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,
	0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,
	0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,
	0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0xbd,
	0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,
	0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,
	0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,
	0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbd,0xbc,
	0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,
	0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,
	0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,
	0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbc,0xbb,
	0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,
	0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,
	0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,
	0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xba,
	0xba,0xba,0xba,0xba,0xba,0xba,0xba,0xba,
	0xba,0xba,0xba,0xba,0xba,0xba,0xba,0xba,
	0xba,0xba,0xba,0xba,0xba,0xba,0xba,0xba,
	0xba,0xba,0xba,0xba,0xba,0xba,0xba,0xb9,
	0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,
	0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,
	0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,
	0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb9,0xb8,
	0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,
	0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,
	0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,
	0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb8,0xb7,
	0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,
	0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,
	0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,
	0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb6,
	0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,
	0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,
	0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,
	0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb6,0xb5,
	0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,
	0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,
	0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,
	0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb5,0xb4,
	0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,
	0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,
	0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,
	0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb4,0xb3,
	0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,
	0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,
	0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,
	0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb2,
	0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
	0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
	0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
	0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb1,
	0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,
	0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,
	0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,
	0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb1,0xb0,
	0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,
	0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,
	0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,
	0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,
	0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xae,
	0xae,0xae,0xae,0xae,0xae,0xae,0xae,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
	0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,
	0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xab,
	0xab,0xab,0xab,0xab,0xab,0xab,0xab,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
	0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,
	0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa9,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,
	0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa8,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,
	0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,
	0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa6,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,
	0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
	0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,
	0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa3,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
	0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,
	0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa1,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,
	0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0xa0,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,
	0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,
	0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9e,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
	0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,
	0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,
	0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,
	0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x9a,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x99,
	0x99,0x99,0x99,0x99,0x99,0x99,0x99,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
	0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
	0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x96,
	0x96,0x96,0x96,0x96,0x96,0x96,0x96,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x95,
	0x95,0x95,0x95,0x95,0x95,0x95,0x95,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,
	0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,
	0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,
	0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,
	0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x90,
	0x90,0x90,0x90,0x90,0x90,0x90,0x90,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,
	0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8f,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
	0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,
	0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8d,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,
	0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,
	0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8b,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,
	0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x8a,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x89,
	0x89,0x89,0x89,0x89,0x89,0x89,0x89,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x87,
	0x87,0x87,0x87,0x87,0x87,0x87,0x87,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x86,
	0x86,0x86,0x86,0x86,0x86,0x86,0x86,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x85,
	0x85,0x85,0x85,0x85,0x85,0x85,0x85,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
	0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,
	0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,
	0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
	0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
	0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
	0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
	0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
	0x05,0x05,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
	0x06,0x06,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
	0x07,0x07,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
	0x08,0x08,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,
	0x09,0x09,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,0x0a,
	0x0a,0x0a,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
	0x0b,0x0b,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
	0x0c,0x0c,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,
	0x0d,0x0d,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	0x0e,0x0e,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
	0x10,0x10,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
	0x11,0x11,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x12,0x12,0x12,0x12,0x12,0x12,
	0x12,0x12,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x13,0x13,0x13,0x13,0x13,0x13,
	0x13,0x13,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,
	0x14,0x14,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
	0x15,0x15,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x16,0x16,0x16,0x16,0x16,0x16,
	0x16,0x16,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,
	0x17,0x17,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x19,0x19,0x19,0x19,0x19,0x19,
	0x19,0x19,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,
	0x1a,0x1a,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,0x1b,
	0x1b,0x1b,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,
	0x1c,0x1c,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,0x1d,
	0x1d,0x1d,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,
	0x1e,0x1e,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,
	0x1f,0x1f,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
	0x20,0x20,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
	0x21,0x21,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22,
	0x22,0x22,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x23,0x23,0x23,0x23,0x23,0x23,
	0x23,0x23,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x24,0x24,0x24,0x24,0x24,0x24,
	0x24,0x24,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x25,0x25,0x25,0x25,0x25,0x25,
	0x25,0x25,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,
	0x26,0x26,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x27,0x27,0x27,0x27,0x27,0x27,
	0x27,0x27,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,
	0x28,0x28,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x29,0x29,0x29,0x29,0x29,0x29,
	0x29,0x29,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,0x2a,
	0x2a,0x2a,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,0x2b,
	0x2b,0x2b,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
	0x2c,0x2c,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,0x2d,
	0x2d,0x2d,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,0x2e,
	0x2e,0x2e,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,0x2f,
	0x2f,0x2f,0x30,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x31,0x31,0x31,0x31,0x31,0x31,
	0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,
	0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,
	0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,
	0x31,0x31,0x32,0x32,0x32,0x32,0x32,0x32,
	0x32,0x32,0x32,0x32,0x32,0x32,0x32,0x32,
	0x32,0x32,0x32,0x32,0x32,0x32,0x32,0x32,
	0x32,0x32,0x32,0x32,0x32,0x32,0x32,0x32,
	0x32,0x32,0x33,0x33,0x33,0x33,0x33,0x33,
	0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
	0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
	0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
	0x33,0x33,0x34,0x34,0x34,0x34,0x34,0x34,
	0x34,0x34,0x34,0x34,0x34,0x34,0x34,0x34,
	0x34,0x34,0x34,0x34,0x34,0x34,0x34,0x34,
	0x34,0x34,0x34,0x34,0x34,0x34,0x34,0x34,
	0x34,0x34,0x35,0x35,0x35,0x35,0x35,0x35,
	0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
	0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
	0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
	0x35,0x35,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x37,0x37,0x37,0x37,0x37,0x37,
	0x37,0x37,0x37,0x37,0x37,0x37,0x37,0x37,
	0x37,0x37,0x37,0x37,0x37,0x37,0x37,0x37,
	0x37,0x37,0x37,0x37,0x37,0x37,0x37,0x37,
	0x37,0x37,0x38,0x38,0x38,0x38,0x38,0x38,
	0x38,0x38,0x38,0x38,0x38,0x38,0x38,0x38,
	0x38,0x38,0x38,0x38,0x38,0x38,0x38,0x38,
	0x38,0x38,0x38,0x38,0x38,0x38,0x38,0x38,
	0x38,0x38,0x39,0x39,0x39,0x39,0x39,0x39,
	0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,
	0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,
	0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,
	0x39,0x39,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,
	0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,
	0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,
	0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,
	0x3a,0x3a,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,
	0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,
	0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,
	0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,
	0x3b,0x3b,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,
	0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,
	0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,
	0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,
	0x3c,0x3c,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,
	0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,
	0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,
	0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,
	0x3d,0x3d,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,
	0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,
	0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,
	0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,0x3e,
	0x3e,0x3e,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,
	0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,
	0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,
	0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,
	0x3f,0x3f,0x40,0x40,0x40,0x40,0x40,0x40,
	0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
	0x40,0x40,0x41,0x41,0x41,0x41,0x41,0x41,
	0x41,0x41,0x41,0x41,0x41,0x41,0x41,0x41,
	0x41,0x41,0x42,0x42,0x42,0x42,0x42,0x42,
	0x42,0x42,0x42,0x42,0x42,0x42,0x42,0x42,
	0x42,0x42,0x43,0x43,0x43,0x43,0x43,0x43,
	0x43,0x43,0x43,0x43,0x43,0x43,0x43,0x43,
	0x43,0x43,0x44,0x44,0x44,0x44,0x44,0x44,
	0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
	0x44,0x44,0x45,0x45,0x45,0x45,0x45,0x45,
	0x45,0x45,0x45,0x45,0x45,0x45,0x45,0x45,
	0x45,0x45,0x46,0x46,0x46,0x46,0x46,0x46,
	0x46,0x46,0x46,0x46,0x46,0x46,0x46,0x46,
	0x46,0x46,0x47,0x47,0x47,0x47,0x47,0x47,
	0x47,0x47,0x47,0x47,0x47,0x47,0x47,0x47,
	0x47,0x47,0x48,0x48,0x48,0x48,0x48,0x48,
	0x48,0x48,0x48,0x48,0x48,0x48,0x48,0x48,
	0x48,0x48,0x49,0x49,0x49,0x49,0x49,0x49,
	0x49,0x49,0x49,0x49,0x49,0x49,0x49,0x49,
	0x49,0x49,0x4a,0x4a,0x4a,0x4a,0x4a,0x4a,
	0x4a,0x4a,0x4a,0x4a,0x4a,0x4a,0x4a,0x4a,
	0x4a,0x4a,0x4b,0x4b,0x4b,0x4b,0x4b,0x4b,
	0x4b,0x4b,0x4b,0x4b,0x4b,0x4b,0x4b,0x4b,
	0x4b,0x4b,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,
	0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,
	0x4c,0x4c,0x4d,0x4d,0x4d,0x4d,0x4d,0x4d,
	0x4d,0x4d,0x4d,0x4d,0x4d,0x4d,0x4d,0x4d,
	0x4d,0x4d,0x4e,0x4e,0x4e,0x4e,0x4e,0x4e,
	0x4e,0x4e,0x4e,0x4e,0x4e,0x4e,0x4e,0x4e,
	0x4e,0x4e,0x4f,0x4f,0x4f,0x4f,0x4f,0x4f,
	0x4f,0x4f,0x4f,0x4f,0x4f,0x4f,0x4f,0x4f,
	0x4f,0x4f,0x50,0x50,0x50,0x50,0x50,0x50,
	0x50,0x50,0x51,0x51,0x51,0x51,0x51,0x51,
	0x51,0x51,0x52,0x52,0x52,0x52,0x52,0x52,
	0x52,0x52,0x53,0x53,0x53,0x53,0x53,0x53,
	0x53,0x53,0x54,0x54,0x54,0x54,0x54,0x54,
	0x54,0x54,0x55,0x55,0x55,0x55,0x55,0x55,
	0x55,0x55,0x56,0x56,0x56,0x56,0x56,0x56,
	0x56,0x56,0x57,0x57,0x57,0x57,0x57,0x57,
	0x57,0x57,0x58,0x58,0x58,0x58,0x58,0x58,
	0x58,0x58,0x59,0x59,0x59,0x59,0x59,0x59,
	0x59,0x59,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,
	0x5a,0x5a,0x5b,0x5b,0x5b,0x5b,0x5b,0x5b,
	0x5b,0x5b,0x5c,0x5c,0x5c,0x5c,0x5c,0x5c,
	0x5c,0x5c,0x5d,0x5d,0x5d,0x5d,0x5d,0x5d,
	0x5d,0x5d,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
	0x5e,0x5e,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,
	0x5f,0x5f,0x60,0x60,0x60,0x60,0x61,0x61,
	0x61,0x61,0x62,0x62,0x62,0x62,0x63,0x63,
	0x63,0x63,0x64,0x64,0x64,0x64,0x65,0x65,
	0x65,0x65,0x66,0x66,0x66,0x66,0x67,0x67,
	0x67,0x67,0x68,0x68,0x68,0x68,0x69,0x69,
	0x69,0x69,0x6a,0x6a,0x6a,0x6a,0x6b,0x6b,
	0x6b,0x6b,0x6c,0x6c,0x6c,0x6c,0x6d,0x6d,
	0x6d,0x6d,0x6e,0x6e,0x6e,0x6e,0x6f,0x6f,
	0x6f,0x6f,0x70,0x70,0x71,0x71,0x72,0x72,
	0x73,0x73,0x74,0x74,0x75,0x75,0x76,0x76,
	0x77,0x77,0x78,0x78,0x79,0x79,0x7a,0x7a,
	0x7b,0x7b,0x7c,0x7c,0x7d,0x7d,0x7e,0x7e};
