/*
 *
 *   qrash: the second portable demo in the world
 *
 *   Copyright (C) 1997  Queue Members Group Art Division
 *   Coded by Mad Max / Queue Members Group (Mike Shirobokov)
 *   <mad_max@qmg.rising.ru>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include "image.h"
#include "video.h"
#include "resource.h"
#include "misc.h"
#include "smooth.h"

struct rgb { uchar r,g,b; };

Image::Image( Image& image )
{
  *this = image;
  data = new uchar[ image.bytesPerLine*image.sizeY ];
  memcpy( data, image.data, image.bytesPerLine*image.sizeY );
  palette = image.palette;
  loaded = false;
}

Image::Image( char* name, Stretch stretch )
{
  uchar* buf = (uchar*)resGetResource(name);
  bytesPerLine = sizeX = *(buf+8)*256 + *(buf+9);
  sizeY = *(buf+10)*256 + *(buf+11);
  rgb* tmp = (rgb*)(buf+32);
  palette = (vidRGB*)cmalloc(sizeof(vidRGB)*256);
  for( int i=0; i<256; i++ ) {
    palette[i].r = tmp[i].r;
    palette[i].g = tmp[i].g;
    palette[i].b = tmp[i].b;
  }
  data = buf+800; loaded = true;
  switch(stretch) {
    case STRETCH_ALIAS:
      Smooth();
    case STRETCH: {
      StretchTo( vidScaleX(sizeX), vidScaleY(sizeY), vidScaleX(bytesPerLine) );
      break;
    }
    case STRETCH_BIT0: {
      StretchToBit( vidScaleX(sizeX), vidScaleY(sizeY), false );
      break;
    }
    case STRETCH_BIT1: {
      StretchToBit( vidScaleX(sizeX), vidScaleY(sizeY), true );
      break;
    }
    case STRETCH_SCREEN: {
      StretchTo( vidSizeX, vidSizeY, vidBytesPerLine);
      break;
    }
  }
}

Image::Image( int sizex, int sizey, int bytesperline, vidRGB pal[256], uchar* buf )
{
  sizeX = sizex;
  sizeY = sizey;
  bytesPerLine = bytesperline;
  palette = pal;
  data = buf;
  loaded = false;
}

Image::~Image()
{
  if( loaded ) {
    cfree(data-800);
    cfree(palette);
 }
  else
    cfree(data);
}

void Image::Show( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    memcpy( to, from, sizex );
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowT( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    for( int j=0; j<sizex; j++ ) {
      if( from[j] ) to[j]=from[j];
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowTC( int x, int y, uchar color, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    for( int j=0; j<sizex; j++ ) {
      if( from[j] ) to[j]=color;
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowAdd( int x, int y, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

  uchar* from = data + offsety*bytesPerLine + offsetx;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    for( int j=0; j<sizex; j++ ) {
      to[j] = from[j]+to[j] < VID_MAX_BRIGHT ? from[j]+to[j] : VID_MAX_BRIGHT-1;
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowTB( int x, int y, uchar color, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int sizex = min( sizeX - offsetx, vidSizeX-x );
  if( sizex <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int sizey = min( sizeY - offsety, vidSizeY-y );
  if( sizey <= 0 ) return;

  uchar* from = data + offsety*bytesPerLine + offsetx/8;
  uchar* to = page + y*vidBytesPerLine + x;
  for( int i=0; i<sizey; i++ ) {
    register mask = 0x80 >> (offsetx&7);
    for( int j=0,k=0; j<sizex; j++ ) {
      if( from[k] & mask ) to[j]=color;
      mask >>= 1; if( !mask ) { mask = 0x80; k++; }
    }
    to += vidBytesPerLine;
    from += bytesPerLine;
  }
}

void Image::ShowR( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
		       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    int stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      to[j] = from1[x>>16];
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRT( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
		       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      if( from1[x>>16] ) to[j] = from1[x>>16];
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRTC( int x, int y, int w, int h, uchar color, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
		       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      if( from1[x>>16] ) to[j] = color;
    }
    to += vidBytesPerLine;
  }
}

void Image::ShowRTAdd( int x, int y, int w, int h, PAGE page )
{
  if( x > vidSizeX || y > vidSizeY ) return;
  int offsetx=0, offsety=0, offsetpage=0;
  if( x < 0 ) {
    offsetx = -x;
    x = 0;
  }
  int size_x = min( w - offsetx, vidSizeX-x );
  if( size_x <= 0 ) return;
  if( y < 0 ) {
    offsety = -y;
    y = 0;
  }
  int size_y = min( h - offsety, vidSizeY-y );
  if( size_y <= 0 ) return;

  uchar* from = data + (offsety*(sizeY-1)/h)*bytesPerLine +
		       offsetx*(sizeX-1)/w, *from1=from;
  uchar* to = page + y*vidBytesPerLine + x;
  int stepy = (sizeY<<16)/h; y=0;
  for( int i=0; i<size_y; i++, y+=stepy ) {
    from1 = from+(y>>16)*bytesPerLine;
    register stepx = (sizeX<<16)/w;
    for( int j=0, x=0; j<size_x; j++, x+=stepx ) {
      to[j] = from1[x>>16]+to[j] <= 255 ? from1[x>>16]+to[j] : 255;
    }
    to += vidBytesPerLine;
  }
}

void Image::StretchTo( int newx, int newy, int newbytesperline )
{
  if( newbytesperline >= bytesPerLine && newy >= sizeY ) return;
  uchar* newdata = new uchar[ newy*newbytesperline ];
  for( int y=0; y<newy; y++ ) {
    for( int x=0; x<newx; x++ ) {
      newdata[ y*newbytesperline+x ] =
	data[ y*(sizeY-1)/newy*bytesPerLine+x*(sizeX-1)/newx ];
    }
  }
  sizeX = newx; sizeY = newy; bytesPerLine = newbytesperline;
  if( loaded )
    cfree(data-800);
  else
    cfree(data);
  data = newdata;
  loaded = false;
}

void Image::StretchToBit( int newx, int newy, bool neg )
{
  int newbytesperline = (newx+7)/8;
  uchar* newdata=(uchar*)cmalloc( 800+newy*newbytesperline );
  memcpy( newdata, data-800, 800 );
  newdata+=800;
  if( newx > sizeX && newy > sizeY ) return;
  for( int y=0; y<newy; y++ ) {
    memset( newdata+y*newbytesperline, 0, newbytesperline );
    uchar mask=0x1;
    for( int x=0; x<newx; x++ ) {
      newdata[ y*newbytesperline+x/8 ] <<= 1;
      if( (data[ y*(sizeY-1)/newy*bytesPerLine+x*(sizeX-1)/newx ]==0) == neg ) {
	newdata[ y*newbytesperline+x/8 ] |= 1;
      }
    }
  }
  cfree(data-800); data=newdata;
  sizeX = newx; sizeY = newy; bytesPerLine = newbytesperline;
}

static int _du, _dv, _length, *y_offset;

#ifdef __WATCOMC__

// eax - ul, ebx - vl, ebp - y_offset, edi - data, esi - page
void map_line( PAGE page, uchar* data, int vl, int ul, int du, int dv );
#pragma aux map_line =\
"mov _du,ecx"\
"mov _dv,edx"\
"mov edx,vidSizeX"\
"add edx,esi"\
"mov _length,edx"\
"push ebp"\
"mov ebp,y_offset"\
"_loop:"\
"mov edx,eax"\
"mov ecx,ebx"\
"shr ecx,16"\
"add eax,_du"\
"shr edx,16"\
"add ebx,_dv"\
"add edx,[ecx*4+ebp]"\
"inc esi"\
"mov dl,[edi+edx]"\
"mov [esi-1],dl"\
"cmp esi,_length"\
"jl _loop"\
"pop ebp"\
parm [esi] [edi] [eax] [ebx] [ecx] [edx]\
modify [eax ebx ecx edx esi edi];

void map_line2( PAGE page, uchar* data, int vl, int ul, int du, int dv );
#pragma aux map_line2 =\
"mov _du,ecx"\
"mov _dv,edx"\
"mov edx,vidSizeX"\
"add edx,esi"\
"mov _length,edx"\
"push ebp"\
"mov ebp,y_offset"\
"_loop:"\
"mov edx,eax"\
"mov ecx,ebx"\
"shr ecx,16"\
"add eax,_du"\
"shr edx,16"\
"add ebx,_dv"\
"add edx,[ecx*4+ebp]"\
"add esi,2"\
"mov dl,[edi+edx]"\
"mov [esi-2],dl"\
"cmp esi,_length"\
"jl _loop"\
"pop ebp"\
parm [esi] [edi] [eax] [ebx] [ecx] [edx]\
modify [eax ebx ecx edx esi edi];

#else

inline void map_line( PAGE page, uchar* data, int vl, int ul, int du, int dv )
{
  int n = vidSizeX;
  for( int x=0; x<n; x++ ) {
    *page++ = data[ y_offset[vl>>16]+(ul>>16) ];
    ul += du; vl += dv;
  }
}

inline void map_line2( PAGE page, uchar* data, int ul, int vl, int du, int dv )
{
  int n = vidSizeX/2;
  for( int x=0; x<n; x++ ) {
    *(page+=2) = data[ y_offset[vl>>16]+(ul>>16) ];
    ul += du; vl += dv;
  }
}

#endif

void Image::MapToPage( int u1, int v1,		//  1	2
		       int u2, int v2,		//
		       int u3, int v3,		//  3	4
		       int u4, int v4,		//
		       PAGE page )
{
  y_offset = new int[sizeY];
  int i;
  y_offset[0] = 0;
  for( i=1; i<sizeY; i++ ) y_offset[i] = y_offset[i-1]+bytesPerLine;
  for( int y=0; y<vidSizeY; y++ ) {
    int ul = u1+(u3-u1)*y/vidSizeY, vl = v1+(v3-v1)*y/vidSizeY,
	ur = u2+(u4-u2)*y/vidSizeY, vr = v2+(v4-v2)*y/vidSizeY;
    int du = ((ur-ul)<<16) / vidSizeX,
	dv = ((vr-vl)<<16) / vidSizeX;
    map_line( page, data, ul<<16, vl<<16, du, dv );
    page += vidBytesPerLine;
  }
  delete y_offset;
}

void Image::MapToPage2( int u1, int v1, 	 //  1	 2
		       int u2, int v2,		//
		       int u3, int v3,		//  3	4
		       int u4, int v4,		//
		       PAGE page )
{
  y_offset = new int[sizeY];
  int i;
  y_offset[0] = 0;
  for( i=1; i<sizeY; i++ ) y_offset[i] = y_offset[i-1]+bytesPerLine;
  for( int y=0; y<vidSizeY; y++ ) {
    int ul = u1+(u3-u1)*y/vidSizeY, vl = v1+(v3-v1)*y/vidSizeY,
	ur = u2+(u4-u2)*y/vidSizeY, vr = v2+(v4-v2)*y/vidSizeY;
    int du = ((ur-ul)<<17) / vidSizeX,
	dv = ((vr-vl)<<17) / vidSizeX;
    map_line2( page, data, ul<<16, vl<<16, du, dv );
    page += y%2 ? vidBytesPerLine+1 : vidBytesPerLine-1;
  }
  delete y_offset;
}

void Image::Smooth()
{
  int bpl = vidBytesPerLine, x = vidSizeX, y = vidSizeY, ps = vidPageSize;
  vidBytesPerLine = bytesPerLine;
  vidSizeX = sizeX; vidSizeY = sizeY;
  vidPageSize = bytesPerLine*sizeY;
  SmoothPage(data,data);
  vidBytesPerLine = bpl;
  vidSizeX = x; vidSizeY = y;
  vidPageSize = ps;
}
