/*
 *
 *   qrash: the second portable demo in the world
 *
 *   Copyright (C) 1997  Queue Members Group Art Division
 *   Coded by Mad Max / Queue Members Group (Mike Shirobokov)
 *   <mad_max@qmg.rising.ru>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 */
#include <math.h>
#include "3d.h"
#include "misc.h"
#include "video.h"
#include "resource.h"
#include "lines.h"


extern "C" {
  uchar phongTable[PHONG_STEPS][PHONG_STEPS];
}

#define FLOAT_ROTATE

#ifdef FLOAT_ROTATE
float cosTable3D[COS_STEPS_3D], sinTable3D[COS_STEPS_3D];
#else
int cosTable3D[COS_STEPS_3D], sinTable3D[COS_STEPS_3D];
#endif
int acosTable[ACOS_STEPS];

void DrawFaceZGTS();
void DrawFaceZGT();
void DrawFaceZPT();
void DrawFaceZG();
void DrawFaceZP();
void DrawFaceSPT();
void DrawFaceSGT();
void DrawFaceSP();

typedef void (DrawFaceProc)();

extern "C" {
typedef void (DrawLineProc)( int x, int z, shorts l, shorts t );
#ifdef __WATCOMC__
#pragma aux DrawLineProc parm [esi] [edi] [ebx] [eax] modify [edx ecx];
#endif
}

extern "C" {

DrawLineProc DrawLineZGTS;	 // zbuffer + gouraud + texture + shadow
DrawLineProc DrawLineZGT;	 // zbuffer + gouraud + texture
DrawLineProc DrawLineSGT;	 // sort + gouraud + texture
DrawLineProc DrawLineZPT;	 // zbuffer + phong + texture
DrawLineProc DrawLineSPT;	 // sort + phong + texture
DrawLineProc DrawLineZG;	 // zbuffer + gouraud
DrawLineProc DrawLineZP;	 // zbuffer + phong
DrawLineProc DrawLineSP;	 // sort + phong

}

DrawFaceProc* DrawFace;

extern "C" {

static int *zs;
static shorts *ls, *ts;
static uchar** texture;
static bool need_clipping;
static int flags;

extern PAGE shadow;

// drawLineZGTS patches
extern short* zbufferOffsetZGTS;
extern uchar* textureZGTS;
extern PAGE colorZGTS, bwZGTS;
extern int x2ZGTS, zsZGTS;
extern int x2ZGTS, zsZGTS;
extern shorts tsZGTS, lsZGTS;

// drawLineZGT patches
extern uchar** textureZGT;
extern PAGE *colorZGT, *bwZGT;
extern int *x2ZGT, *zsZGT;
extern shorts *tsZGT, *lsZGT;
extern short **zbuffer0ZGT, **zbuffer1ZGT;

// drawLineZB patches
extern PAGE bwZG;
extern int x2ZG, zsZG;
extern shorts lsZG;
extern short *zbuffer0ZG, *zbuffer1ZG;

// drawLineZP patches
extern PAGE *bwZP;
extern int *x2ZP, *zsZP;
extern shorts *lsZP;
extern uchar** phongZP;
extern short **zbuffer0ZP, **zbuffer1ZP;

// drawLineSGT patches
extern uchar* textureSGT;
extern PAGE colorSGT, bwSGT;
extern int x2SGT;
extern shorts tsSGT, lsSGT;

// drawLineZPT patches
extern uchar** textureZPT;
extern PAGE *colorZPT, *bwZPT;
extern int *x2ZPT, *zsZPT;
extern shorts *tsZPT, *lsZPT;
extern uchar** phongZPT;
extern short **zbuffer0ZPT, **zbuffer1ZPT;

// drawLineSPT patches
extern uchar** textureSPT;
extern PAGE *colorSPT, *bwSPT;
extern int *x2SPT;
extern shorts *tsSPT, *lsSPT;
extern uchar** phongSPT;

// drawLineSP patches
extern PAGE *colorSP, *bwSP;
extern int *x2SP;
extern shorts *lsSP;
extern uchar** phongSP;

}

int foo;
shorts foos;
uchar* foot;

void FacedObject::setDrawLineZGTS( PAGE color, PAGE bw, short* zbuffer,
				   int shadow_offset )
{
  DrawFace = &DrawFaceZGTS;
  shadow = bw - shadow_offset;
  zbufferOffsetZGTS = zbuffer - shadow_offset;
  texture = &textureZGTS;
  colorZGTS = color; bwZGTS = bw;
  zs = &zsZGTS;
  ls = &lsZGTS; ts = &tsZGTS;
}

void FacedObject::setDrawLineZGT( PAGE color, PAGE bw, short* zbuffer )
{
  DrawFace = &DrawFaceZGT;
  *colorZGT = color; *bwZGT = bw;
  texture = textureZGT;
  zs = zsZGT;
  ls = lsZGT; ts = tsZGT;
  *zbuffer0ZGT = *zbuffer1ZGT = zbuffer;
}

void FacedObject::setDrawLineZPT( PAGE color, PAGE bw, short* zbuffer )
{
  DrawFace = &DrawFaceZPT;
  *phongZPT = (uchar*)phongTable;
  *colorZPT = color; *bwZPT = bw;
  texture = textureZPT;
  zs = zsZPT;
  ls = lsZPT; ts = tsZPT;
  *zbuffer0ZPT = *zbuffer1ZPT = zbuffer;
}

void FacedObject::setDrawLineSPT( PAGE color, PAGE bw )
{
  *phongSPT = (uchar*)phongTable;
  DrawFace = &DrawFaceSPT;
  *colorSPT = color; *bwSPT = bw;
  texture = textureSPT;
  ls = lsSPT;
  zs = &foo; ts = tsSPT;
}

void FacedObject::setDrawLineSGT( PAGE color, PAGE bw )
{
  DrawFace = &DrawFaceSGT;
  colorSGT = color; bwSGT = bw;
  texture = &textureSGT;
  zs = &foo;
  ls = &lsSGT; ts = &tsSGT;
}

void FacedObject::setDrawLineZG( PAGE bw, short* zbuffer )
{
  DrawFace = &DrawFaceZG;
  bwZG= bw; zbuffer0ZG = zbuffer1ZG = zbuffer;
  zs = &zsZG;
  ls = &lsZG; ts = &foos;
}

void FacedObject::setDrawLineZP( PAGE bw, short* zbuffer )
{
  DrawFace = &DrawFaceZP;
  *phongZP = (uchar*)phongTable;
  *bwZP= bw;
  *zbuffer0ZP = *zbuffer1ZP = zbuffer;
  zs = zsZP;
  ls = lsZP; ts = &foos;
}

void FacedObject::setDrawLineSP( PAGE bw )
{
  DrawFace = &DrawFaceSP;
  *phongSP = (uchar*)phongTable;
  *bwSP = bw;
  ls = lsSP;
  ts = &foos; zs = &foo;
  texture = &foot;
}

static int leftx, rightx, leftxs, rightxs, leftz, leftzs;
static shorts leftt, leftl, leftts, leftls;
static int face_height, offset;

void DrawFaceZGTS()
{
  for( int y=0; y<face_height; y++ ) {
    int x1 = offset+(leftx>>16);
    x2ZGTS = offset+(rightx>>16);
    if(x2ZGTS > offset && x1<offset+vidSizeX) {
      if(x2ZGTS > offset+vidSizeX-1) x2ZGTS = offset+vidSizeX-1;
      if(x1<offset) {
	register tmp = offset-x1;
	DrawLineZGTS( offset, leftz+zsZGTS*tmp, leftl+lsZGTS*tmp,
		      shorts( leftt.s.u+tsZGTS.s.u*tmp,
			      leftt.s.v+tsZGTS.s.v*tmp ) );
      }
      else {
	DrawLineZGTS( x1, leftz, leftl, leftt );
      }
    }
    leftx += leftxs; leftz += leftzs;
    leftt += leftts; leftl += leftls;
    rightx += rightxs;
    offset += vidBytesPerLine;
  }
}

void DrawFaceZGT()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2ZGT = offset+(rightx>>16);
      if(*x2ZGT > offset && x1<offset+vidSizeX) {
	if(*x2ZGT > offset+vidSizeX)
	  *x2ZGT = offset+vidSizeX;
	if( x1 < offset ) {
	  register tmp = offset-x1;
	  DrawLineZGT( offset, leftz+*zsZGT*tmp, leftl+*lsZGT*tmp,
		      shorts( leftt.s.u+tsZGT->s.u*tmp,
			      leftt.s.v+tsZGT->s.v*tmp ) );
	}
	else {
	  DrawLineZGT( x1, leftz, leftl, leftt );
	}
      }
      leftx += leftxs; leftz += leftzs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2ZGT = offset+(rightx>>16);
      DrawLineZGT( x1, leftz, leftl, leftt );
      leftx += leftxs; leftz += leftzs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceZPT()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      if( offset >= 0 ) {
	int x1 = offset+(leftx>>16);
	*x2ZPT = offset+(rightx>>16);
	if(*x2ZPT > offset && x1<offset+vidSizeX) {
	  if(*x2ZPT > offset+vidSizeX)
	    *x2ZPT = offset+vidSizeX;
	  if( x1 < offset ) {
	    register short tmp = offset-x1;
	    DrawLineZPT( offset, leftz+*zsZPT*tmp,
			shorts( leftl.s.u+lsZPT->s.u*tmp,
				leftl.s.v+lsZPT->s.v*tmp ),
			shorts( leftt.s.u+tsZPT->s.u*tmp,
				leftt.s.v+tsZPT->s.v*tmp ) );
	  }
	  else {
	    DrawLineZPT( x1, leftz, leftl, leftt );
	  }
	}
      }
      leftx += leftxs; leftz += leftzs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2ZPT = offset+(rightx>>16);
      DrawLineZPT( x1, leftz, leftl, leftt );
      leftx += leftxs; leftz += leftzs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceZG()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      x2ZG = offset+(rightx>>16);
      if(x2ZG> offset && x1<offset+vidSizeX) {
	if(x2ZG> offset+vidSizeX) x2ZG= offset+vidSizeX;
	if(x1<offset) {
	  register tmp = offset-x1;
	  DrawLineZG( offset, leftz+zsZG*tmp, leftl+lsZG*tmp, 0 );
	}
	else {
	  DrawLineZG( x1, leftz, leftl, 0 );
	}
      }
      leftx += leftxs; leftz += leftzs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      x2ZG= offset+(rightx>>16);
      DrawLineZG( x1, leftz, leftl, 0 );
      leftx += leftxs; leftz += leftzs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceSGT()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      x2SGT = offset+(rightx>>16);
      if(x2SGT > offset && x1<offset+vidSizeX) {
	if(x2SGT > offset+vidSizeX)
	  x2SGT = offset+vidSizeX;
	if( x1 < offset ) {
	  register tmp = offset-x1;
	  DrawLineSGT( offset, 0, leftl+lsSGT*tmp,
		      shorts( leftt.s.u+tsSGT.s.u*tmp,
			      leftt.s.v+tsSGT.s.v*tmp ) );
	}
	else {
	  DrawLineSGT( x1, 0, leftl, leftt );
	}
      }
      leftx += leftxs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      x2SGT = offset+(rightx>>16);
      DrawLineSGT( x1, 0, leftl, leftt );
      leftx += leftxs;
      leftt += leftts; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceSPT()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      if( offset >= 0 ) {
	int x1 = offset+(leftx>>16);
	*x2SPT = offset+(rightx>>16);
	if(*x2SPT > offset && x1<offset+vidSizeX) {
	  if(*x2SPT > offset+vidSizeX)
	    *x2SPT = offset+vidSizeX;
	  if( x1 < offset ) {
	    register short tmp = offset-x1;
	    DrawLineSPT( offset, 0,
			shorts( leftl.s.u+lsSPT->s.u*tmp,
				leftl.s.v+lsSPT->s.v*tmp ),
			shorts( leftt.s.u+tsSPT->s.u*tmp,
				leftt.s.v+tsSPT->s.v*tmp ) );
	  }
	  else {
	    DrawLineSPT( x1, 0, leftl, leftt );
	  }
	}
      }
      leftx += leftxs;
      leftt += leftts;
      leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2SPT = offset+(rightx>>16);
      DrawLineSPT( x1, 0, leftl, leftt );
      leftx += leftxs;
      leftt += leftts;
      leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceZP()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      if( offset >= 0 ) {
	int x1 = offset+(leftx>>16);
	*x2ZP = offset+(rightx>>16);
	if(*x2ZP > offset && x1<offset+vidSizeX) {
	  if(*x2ZP > offset+vidSizeX) *x2ZP = offset+vidSizeX;
	  if(x1<offset) {
	    register tmp = offset-x1;
	    DrawLineZP( offset, leftz+*zsZP*tmp,
			shorts( leftl.s.u+lsZP->s.u*tmp,
				leftl.s.v+lsZP->s.v*tmp ), 0 );
	  }
	  else {
	    DrawLineZP( x1, leftz, leftl, 0 );
	  }
	}
      }
      leftx += leftxs; leftz += leftzs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2ZP = offset+(rightx>>16);
      DrawLineZP( x1, leftz, leftl, 0 );
      leftx += leftxs; leftz += leftzs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

void DrawFaceSP()
{
  if( need_clipping ) {
    for( int y=0; y<face_height; y++ ) {
      if( offset >= 0 ) {
	int x1 = offset+(leftx>>16);
	*x2SP = offset+(rightx>>16);
	if(*x2SP> offset && x1<offset+vidSizeX) {
	  if(*x2SP> offset+vidSizeX) *x2SP = offset+vidSizeX;
	  if(x1<offset) {
	    register tmp = offset-x1;
	    DrawLineSP( offset, 0,
			shorts( leftl.s.u+lsSP->s.u*tmp,
				leftl.s.v+lsSP->s.v*tmp ), 0 );
	  }
	  else {
	    DrawLineSP( x1, 0, leftl, 0 );
	  }
	}
      }
      leftx += leftxs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
  else {
    for( int y=0; y<face_height; y++ ) {
      int x1 = offset+(leftx>>16);
      *x2SP= offset+(rightx>>16);
      DrawLineSP( x1, 0, leftl, 0 );
      leftx += leftxs; leftl += leftls;
      rightx += rightxs;
      offset += vidBytesPerLine;
    }
  }
}

#ifdef __WATCOMC__
int div64(int x, int y);
#pragma aux div64 = \
    "mov   edx, eax"\
    "shl   eax, 8  "\
    "sar   edx, 24 "\
    "idiv  ebx	   "\
    parm [eax] [ebx] \
    modify exact [eax edx] \
    value [eax]
#else
#ifdef i386
inline int div64(int x, int y)
{
  unsigned long rv;
__asm__(
"movl %1,%%edx	\n"
"shll $8,%%eax \n"
"sarl $24,%%edx \n"
"idivl %2	\n"
: "=a"(rv)
: "a" (x), "b" (y) );
  return rv;
}
#else
inline int div64(int x, int y)
{
  return (float)x*256.0/y;
}
#endif
#endif

inline int right_steps( Point* v1, Point *v2 )
{
  int height = v2->y - v1->y;
  if( !height ) return 0;

  rightxs = ((v2->x - v1->x)) / height;
  rightx = v1->x;

  return height;
}

inline int left_steps( Point* v1, Point *v2 )
{
  int height = v2->y - v1->y;
  if( !height ) return 0;

  leftx = v1->x;
  leftxs = ((v2->x - v1->x)) / height;

  if( !(flags & SORT) ) {
    leftz = v1->z << 16;
    leftzs = ((v2->z - v1->z) << 16) / height;
  }

  if( !(flags & NO_TEXTURE) ) {
    leftt = v1->t << 8;
    leftts = shorts( ((v2->t.s.u - v1->t.s.u) << 8) / height,
		     ((v2->t.s.v - v1->t.s.v) << 8) / height );
  }

  leftl = v1->l << 8;
  leftls = shorts( ((v2->l.s.u - v1->l.s.u) << 8) / height,
		   ((v2->l.s.v - v1->l.s.v) << 8) / height );

  return height;
}

void Face::Draw( int flags )
{
  Point* v1 = vert[0];
  Point* v2 = vert[1];
  Point* v3 = vert[2];

  if(v1->y > v2->y) { Point* v = v1; v1 = v2; v2 = v; }
  if(v1->y > v3->y) { Point* v = v1; v1 = v3; v3 = v; }
  if(v2->y > v3->y) { Point* v = v2; v2 = v3; v3 = v; }

  int v1v3 = v1->y-v3->y, v1v2 = v1->y-v2->y, v2v3 = v2->y-v3->y;
  if( !v1v3 || v1->y >= vidSizeY ) return;

  int denom = ((v1->x-v3->x)>>16)*v2v3 - ((v2->x-v3->x)>>16)*v1v3;
  if( !denom ) return;

  ::flags = flags;

  if( !(flags & NO_TEXTURE) ) {
    v1->t.s.u -= du; v1->t.s.v -= dv;
    v2->t.s.u -= du; v2->t.s.v -= dv;
    v3->t.s.u -= du; v3->t.s.v -= dv;
  }

  if( denom > 0 ) {
    left_steps(v1,v3);
    face_height = right_steps(v1,v2);
  }
  else {
    right_steps(v1,v3);
    face_height = left_steps(v1,v2);
  }

  if( !(flags & NO_TEXTURE) ) {
    *ts = shorts(
      div64( (v1->t.s.u-v3->t.s.u)*v2v3 -
	     (v2->t.s.u-v3->t.s.u)*v1v3, denom),
      div64( (v1->t.s.v-v3->t.s.v)*v2v3 -
	     (v2->t.s.v-v3->t.s.v)*v1v3, denom ) );
    *::texture = texture;
  }

  if( !foo ) foo = 1;
  *ls = shorts(
    div64( (v1->l.s.u-v3->l.s.u)*v2v3 -
	   (v2->l.s.u-v3->l.s.u)*v1v3, denom),
    div64( (v1->l.s.v-v3->l.s.v)*v2v3 -
	   (v2->l.s.v-v3->l.s.v)*v1v3, denom) );

  if( !(flags & SORT) ) {
    *zs = div64( (v1->z-v3->z)*v2v3 -
		 (v2->z-v3->z)*v1v3, denom ) << 8;
  }

  offset = v1->y * vidBytesPerLine;
  need_clipping = v1->clipped || v2->clipped || v3->clipped;

  if( v2->y >= vidSizeY ) {
    face_height -= v2->y-vidSizeY;
    if( face_height ) DrawFace();
    goto exit;
  }

  if( face_height ) DrawFace();

  if(  denom > 0 ) {
    face_height = right_steps(v2,v3);
  }
  else {
    face_height = left_steps(v2,v3);
  }

  if( v3->y >= vidSizeY ) face_height -= v3->y-vidSizeY;

  if( face_height ) DrawFace();

exit:

  if( !(flags & NO_TEXTURE) ) {
    v1->t.s.u += du; v1->t.s.v += dv;
    v2->t.s.u += du; v2->t.s.v += dv;
    v3->t.s.u += du; v3->t.s.v += dv;
  }

}

inline void Vector::Rotate( int cx, int cy, int cz, int rx, int ry, int rz )
{
#ifdef FLOAT_ROTATE
  float TX = x-cx, TY = y-cy, TZ = z-cz;
  if(ry) {
    if( ry < 0 ) ry = ry % COS_STEPS_3D + COS_STEPS_3D;
    else ry %= COS_STEPS_3D;
    float T1 = TX * cosTable3D[ ry ],
	  T2 = TZ * cosTable3D[ ry ],
	  T3 = TZ * sinTable3D[ ry ],
	  T4 = TX * sinTable3D[ ry ];
    TX = T1 - T3;
    TZ = T4 + T2;
  }
  if(rx) {
    if( rx < 0 ) rx = rx % COS_STEPS_3D + COS_STEPS_3D;
    else rx %= COS_STEPS_3D;
    float T1 = TY * cosTable3D[ rx ],
	  T2 = TZ * sinTable3D[ rx ],
	  T3 = TY * sinTable3D[ rx ],
	  T4 = TZ * cosTable3D[ rx ];
    TY = T1 - T2;
    TZ = T3 + T4;
  }
  if(rz) {
    if( rz < 0 ) rz = rz % COS_STEPS_3D + COS_STEPS_3D;
    else rz %= COS_STEPS_3D;
    float T1 = TY * cosTable3D[ rz ],
	  T2 = TX * sinTable3D[ rz ],
	  T3 = TY * sinTable3D[ rz ],
	  T4 = TX * cosTable3D[ rz ];
    TY = T1 - T2;
    TX = T3 + T4;
  }
  x = (int)TX+cx; y = TY+cy; z = TZ+cz;
#else
  int TX = x-cx, TY = y-cy, TZ = z-cz;
  if(ry) {
    if( ry < 0 ) ry = ry % COS_STEPS_3D + COS_STEPS_3D;
    else ry %= COS_STEPS_3D;
    int T1 = TX * cosTable3D[ ry ];
    int T2 = TZ * cosTable3D[ ry ];
    int T3 = TZ * sinTable3D[ ry ];
    int T4 = TX * sinTable3D[ ry ];
    TX = ( T1 - T3 ) >> 8;
    TZ = ( T4 + T2 ) >> 8;
  }
  if(rx) {
    if( rx < 0 ) rx = rx % COS_STEPS_3D + COS_STEPS_3D;
    else rx %= COS_STEPS_3D;
    int T1 = TY * cosTable3D[ rx ];
    int T2 = TZ * sinTable3D[ rx ];
    int T3 = TY * sinTable3D[ rx ];
    int T4 = TZ * cosTable3D[ rx ];
    TY = ( T1 - T2 ) >> 8;
    TZ = ( T3 + T4 ) >> 8;
  }
  if(rz) {
    if( rz < 0 ) rz = rz % COS_STEPS_3D + COS_STEPS_3D;
    else rz %= COS_STEPS_3D;
    int T1 = TY * cosTable3D[ rz ];
    int T2 = TX * sinTable3D[ rz ];
    int T3 = TY * sinTable3D[ rz ];
    int T4 = TX * cosTable3D[ rz ];
    TY = ( T1 - T2 ) >> 8;
    TX = ( T3 + T4 ) >> 8;
  }
  x = TX+cx; y = TY+cy; z = TZ+cz;
#endif
}

Face::Face( Point* p1, Point* p2, Point* p3, uchar* t ) :
texture(t), du(0), dv(0)
{
  if( p1 == p2 || p2 == p3 || p1 == p3 ) {
    error( "Equal vertices in Face::Face" );
  }
  vert[0]=p1; vert[1]=p2; vert[2]=p3;
  float x1 = vert[0]->x, y1 = vert[0]->y,
	z1 = vert[0]->z, x2 = vert[1]->x,
	y2 = vert[1]->y, z2 = vert[1]->z,
	x3 = vert[2]->x, y3 = vert[2]->y,
	z3 = vert[2]->z;
  normal.x = ((y2-y1)*(z3-z1)-(z2-z1)*(y3-y1));
  normal.y = ((z2-z1)*(x3-x1)-(x2-x1)*(z3-z1));
  normal.z = ((x2-x1)*(y3-y1)-(y2-y1)*(x3-x1));
  normal.Normalize(ACOS_STEPS);
  size_u = du = vert[0]->t.s.u;
  size_v = dv = vert[0]->t.s.v;
  for( int i=1; i<3; i++ ) {
    if( vert[i]->t.s.u < du ) du = vert[i]->t.s.u;
    if( vert[i]->t.s.v < dv ) dv = vert[i]->t.s.v;
    if( vert[i]->t.s.u > size_u ) size_u = vert[i]->t.s.u;
    if( vert[i]->t.s.v > size_v ) size_v = vert[i]->t.s.v;
  }
  size_u -= du-1;
  size_v -= dv-1;
}

FacedObject::FacedObject( FacedObject& o ) :
 Object3D(o.dx,o.dy,o.dz,o.rx,o.ry,o.rz,o.lx,o.ly,o.lv,o.lo,o.sx,o.sy,o.sz),
 points(), faces(), tmp_points(), flags(o.flags),
 faceCallback(0), pointCallback(0),
 perspect(1)
{
  int i;
  for( i=0; i<o.points.Count; i++ ) {
    Point* p = (Point*)o.points.At(i);
    points.Insert( new Point(*p) );
    tmp_points.Insert( new Point(*p) );
  }
  for( i=0; i<o.faces.Count; i++ ) {
    Face* f = (Face*)o.faces.At(i);
    int n[3];
    for( int j=0; j<3; j++ ) {
      n[j] = o.tmp_points.IndexOf( f->vert[j] );
    }
    faces.Insert( new Face( tmp_points[n[0]],
			    tmp_points[n[1]],
			    tmp_points[n[2]],
			    f->texture, f->normal, f->du, f->dv ) );
  }
}

inline char* GetToken(Parser& fp)
{
  const char Spaces[] = {" \t\r\n:,\"" };
  return fp.getToken(Spaces);
}

/*inline */bool WaitToken( const char* token, Parser& fp )
{
  char* s;
  do {
    s = GetToken(fp);
    if( !s ) return false;
  } while ( strcmp( token, s ) );
  return true;
}

/*inline */float GetNumber( const char* token, Parser& fp )
{
  if( WaitToken( token, fp ) )
    return atof( GetToken(fp) );
  else
    return 0;
}

void FacedObject::ImportASC( char* filename, char* name, float Scale,
			     Image* texture, int mapping )
{
  if( !texture ) mapping = 0;
  Parser fp(filename);
  float MinX = 999999, MinY = 999999, MinZ = 999999,
	MaxX = -999999, MaxY = -999999, MaxZ = -999999,
	MinU = 999999, MaxU = -999999,
	MinV = 999999, MaxV = -999999;
  int NVertices;
  while( NVertices = GetNumber("Vertices",fp) ) {
//    if( NVertices > 2000 ) error( "Too large ASC" );
    int foo = GetNumber("Faces",fp);
    if( mapping & MAP_ASC && strcmp( GetToken(fp), "Mapped" ) ) {
      error( "Texture mapping information not found in .ASC" );
    }
    int i;
    for( i=0; i<NVertices; i++ ) {
//	WaitToken( "Vertex", fp );
      float X = GetNumber("X",fp);
      MinX = min( X, MinX );
      MaxX = max( X, MaxX );
      float Y = GetNumber("Y",fp);
      MinY = min( Y, MinY );
      MaxY = max( Y, MaxY );
      float Z = GetNumber("Z",fp);
      MinZ = min( Z, MinZ );
      MaxZ = max( Z, MaxZ );
      if( mapping & MAP_ASC ) {
	float U = GetNumber("U",fp);
	MinU = min( U, MinU );
	MaxU = max( U, MaxU );
	float V = GetNumber("V",fp);
	MinV = min( V, MinV );
	MaxV = max( V, MaxV );
      }
    }
  };
  float Max = max( MaxX-MinX, MaxZ-MinZ ); Max = max( Max, MaxY-MinY );
  maxx = (MaxX-(MaxX+MinX)/2)*Scale / Max;
  minx = (MinX-(MaxX+MinX)/2)*Scale / Max;
  maxz = (MaxY-(MaxY+MinY)/2)*Scale / Max;
  minz = (MinY-(MaxY+MinY)/2)*Scale / Max;
  miny = -(MaxZ-(MaxZ+MinZ)/2)*Scale / Max;
  maxy = -(MinZ-(MaxZ+MinZ)/2)*Scale / Max;

  fp.restart();
  char* Name;
  bool eof = false;
  do {
    do {
      eof = !WaitToken( "Named", fp ) || !WaitToken( "object", fp );
      if( eof ) break;
      Name = GetToken(fp);
    } while( name && strcmp(Name,name) && !eof );
    if( eof ) break;
    if( strcmp( GetToken(fp), "Tri-mesh" ) ) continue;
    int NVertices = GetNumber("Vertices",fp);
    int NFaces = GetNumber("Faces",fp);
    int werePoints = points.Count;
    int i;
    for( i=0; i<NVertices; i++ ) {
      int X = (GetNumber("X",fp)-(MaxX+MinX)/2)*Scale / (Max);
      int Y = (GetNumber("Y",fp)-(MaxY+MinY)/2)*Scale / (Max);
      int Z = (GetNumber("Z",fp)-(MaxZ+MinZ)/2)*Scale / (Max);
      float u = 0, v = 0;
      if( texture ) {
	if( mapping & MAP_ASC ) {
	  if( mapping & MAP_TILE ) {
	    u = int( (GetNumber("U",fp)-MinU)*Scale/(MaxU-MinU) )
		% texture->sizeX;
	    v = int( (GetNumber("V",fp)-MinV)*Scale/(MaxV-MinV) )
		% texture->sizeY;
	  }
	  else {
	    u = (GetNumber("U",fp)-MinU)*(texture->sizeX-1)/(MaxU-MinU);
	    v = (GetNumber("V",fp)-MinV)*(texture->sizeY-1)/(MaxV-MinV);
	  }
	}
	else {
	  if( mapping & MAP_NOSCALE ) {
	    if( mapping & MAP_XY ) {
	      u=X+VID_MAX_SIZE_X/2;
	      v=-Z+VID_MAX_SIZE_Y/2;
	    }
	    if( mapping & MAP_XZ ) {
	      u=X+VID_MAX_SIZE_X/2;
	      v=Y+VID_MAX_SIZE_Y/2;
	    }
	    if( mapping & MAP_YZ ) {
	      u=-Z+VID_MAX_SIZE_X/2;
	      v=Y+VID_MAX_SIZE_Y/2;
	    }
	  }
	  else {
	    if( mapping & MAP_XY ) {
	      u = X, v = -Z;
	    }
	    if( mapping & MAP_XZ ) {
	      u = X, v = Y;
	    }
	    if( mapping & MAP_YZ ) {
	      u = -Z, v = Y;
	    }
	    if( mapping & MAP_TILE ) {
	      u = int(u+Scale/2) % texture->sizeX;
	      v = int(v+Scale/2) % texture->sizeY;
	    }
	    else {
	      u = (u+Scale/2)*(texture->sizeX-1)/Scale;
	      v = (v+Scale/2)*(texture->sizeY-1)/Scale;
	    }
	  }
	  if( mapping & MAP_SPHERE ) {
	    u = Y ? (atan2(Y,X*2)+M_PI)/M_PI/2*(texture->sizeX-1) : texture->sizeX-1,
	    v = ( atan2(Z,sqrt(sqr(X)+sqr(Y)))+M_PI )/M_PI/2*(texture->sizeY-1);
	  }
	}
      }
      points.Insert( new Point( X, -Z, Y, shorts(u,v) ) );
      tmp_points.Insert( new Point( X, -Z, Y, shorts(u,v) ) );
    }
    for( i=0; i<NFaces; i++ ) {
      int n[3];
      char token[2] = " ";
      *token='A';
      for( int j=0; j<3; j++ ) {
	n[j] = GetNumber( token, fp)+werePoints;
	(*token)++;
      }
      faces.Insert( new Face( tmp_points[n[0]],
			      tmp_points[n[1]],
			      tmp_points[n[2]] ) );
    }
    if( name ) break;
  } while (!eof);
  texture_image = texture;
}

void FacedObject::Prepare()
{
  int i;
  if( !faces.Count ) {
    error( "empty obect in FacedObject::Prepare" );
  }
  if( texture_image ) {
    SplitFaces();
    SplitTexture();
  }
  for( i=0; i<faces.Count; i++ ) {
    Face* f = faces[i];
    for( int j=0; j<3; j++ ) {
      Vector* n = &points[ tmp_points.IndexOf( f->vert[j] ) ]->normal;
      n->x += f->normal.x;
      n->y += f->normal.y;
      n->z += f->normal.z;
    }
  }
  for( i=0; i<points.Count; i++ ) {
    points[i]->normal.Normalize(ACOS_STEPS);
  }
}

int compareFaces( const Face* f1, const Face* f2 )
{
  int z1 = f1->vert[0]->z+f1->vert[1]->z+f1->vert[2]->z,
      z2 = f2->vert[0]->z+f2->vert[1]->z+f2->vert[2]->z;
  if( z1 > z2 ) return 1;
  if( z1 < z2 ) return -1;
  return 0;
};

TCollection<Face> tmp_faces;

void FacedObject::Draw( short* zbuffer, PAGE color, PAGE bw )
{
  int i;
  for( i=0; i<points.Count; i++ ) {
    *tmp_points[i] = *points[i];
  }
  for( int j=0; j<faces.Count; j++ ) {
    Face* f = faces[j];
    if( ! (flags & TWO_SIDES) ) {
      Vector n = f->normal;
      n.Rotate(0,0,0,rx,ry,rz);
      if( n.z < -10 ) continue;
    }
    for( int i=0; i<3; i++ ) {
      Point* p = f->vert[i];
      if( p->processed ) continue;
      p->processed = true;
      p->normal.Rotate(0,0,0,rx+lx,ry+ly,rz+lz);
      p->x *= sx; p->y *= sy; p->z *= sz;
      p->Rotate(0,0,0,rx,ry,rz);
      p->Move( dx, dy, dz );
      if( p->z > VID_MAX_SIZE_X ) {
	p->clipped = true;
      }
      if( flags & PERSPECT ) {
	float scale_z = p->z/perspect;
	if( scale_z > VID_MAX_SIZE_X ) scale_z = VID_MAX_SIZE_X;
	float scale = (0.5+0.5*(VID_MAX_SIZE_X-scale_z)/VID_MAX_SIZE_X);
	p->x = vidScaleX( p->x/scale + VID_MAX_SIZE_X/2 );
	p->y = vidScaleY( p->y/scale + VID_MAX_SIZE_Y/2 );
      }
      else {
	p->x = vidScaleX( p->x + VID_MAX_SIZE_X/2 );
	p->y = vidScaleY( p->y + VID_MAX_SIZE_Y/2 );
      }
      if( /*p->z > 0 || p->z < MIN_Z ||*/
	  p->x >= vidSizeX || p->x < 0 || p->y < 0 ) {
	p->clipped = true;
      }
      else {
	p->clipped = false;
      }
      if( flags & PHONG ) {
	p->l.s.u = acosTable[ (p->normal.x+ACOS_STEPS)/2 ];
	p->l.s.v = acosTable[ (p->normal.y+ACOS_STEPS)/2 ];
      }
      else {
	if( flags & TWO_SIDES ) {
	  p->l = abs(p->normal.z) * (1<<8) * VID_MAX_BRIGHT / ACOS_STEPS;
	}
	else {
	  p->l = p->normal.z < 0 ?
	    0 : (p->normal.z) * (1<<8) * VID_MAX_BRIGHT / ACOS_STEPS;
	}
	p->l = ((VID_MAX_BRIGHT-1)<<8)*lo + p->l*lv;
	if( p->l > ((VID_MAX_BRIGHT-1)<<8) )
	  p->l = ((VID_MAX_BRIGHT-1)<<8);
      }
      if( pointCallback ) pointCallback(p);
      p->x <<= 16;
    }
    if(f->vert[0]->clipped && f->vert[1]->clipped && f->vert[2]->clipped) {
      if( (f->vert[0]->x<0 && f->vert[1]->x<0 && f->vert[2]->x<0) ||
	  (f->vert[0]->x > vidSizeX && f->vert[1]->x > vidSizeX &&
	   f->vert[2]->x > vidSizeX ) )
	continue;
    }
    tmp_faces.Insert(f);
  }
  if( flags & NO_TEXTURE ) {
    if( flags & PHONG ) {
      if( flags & SORT	) {
	setDrawLineSP(bw);
      }
      else {
	setDrawLineZP(bw, zbuffer);
      }
    }
    else {
      setDrawLineZG(bw, zbuffer);
    }
  }
  else {
    if( flags & SHADOW ) {
      setDrawLineZGTS( color, bw, zbuffer,
	int( vidScaleY(ly/2.5-dy)*sy)*vidBytesPerLine -
	     vidScaleX(lx/2.5-dx)*sx );
    }
    else {
      if( flags & SORT	) {
	if( flags & PHONG ) {
	  setDrawLineSPT( color, bw );
	}
	else {
	  setDrawLineSGT( color, bw );
	}
      }
      else {
	if( flags & PHONG ) {
	  setDrawLineZPT( color, bw, zbuffer );
	}
	else {
	  setDrawLineZGT( color, bw, zbuffer );
	}
      }
    }
  }
  if( flags & SORT ) {
    tmp_faces.Sort( compareFaces );
  }
  for( i=0; i<tmp_faces.Count; i++ ) {
    if( faceCallback && !faceCallback(tmp_faces[i]) ) continue;
    tmp_faces[i]->Draw( flags );
  }
  tmp_faces.DeleteAll();
}

inline int morph( int from, int to, int step )
{
  return from+( ((to-from)*step) >> 10 );
}

void FacedObject::Morph( FacedObject& o1, FacedObject& o2,
			 int step, int steps )
{
  if( points.Count != o1.points.Count || points.Count != o2.points.Count ||
      faces.Count != o1.faces.Count || faces.Count != o2.faces.Count ) {
    error( "Trying to morph different objects" );
  }
//  if( step<0 || step>=steps ) error( "Morph step is out of range" );
  if( step<0 ) step=0;
  if( step>steps ) step=steps;
  step = step*1024.0/steps;
  int i;
  for( i=0; i<points.Count; i++ ) {
    Point* p = points[i];
    Point* p1 = o1.points[i];
    Point* p2 = o2.points[i];
    p->x = morph( p1->x, p2->x, step );
    p->y = morph( p1->y, p2->y, step );
    p->z = morph( p1->z, p2->z, step );
    Vector* n = &points[i]->normal;
    Vector* n1 = &o1.points[i]->normal;
    Vector* n2 = &o2.points[i]->normal;
    n->x = morph( n1->x, n2->x, step );
    n->y = morph( n1->y, n2->y, step );
    n->z = morph( n1->z, n2->z, step );
  }
  for( i=0; i<faces.Count; i++ ) {
    Vector* n = &faces[i]->normal;
    Vector* n1 = &o1.faces[i]->normal;
    Vector* n2 = &o2.faces[i]->normal;
    n->x = morph( n1->x, n2->x, step );
    n->y = morph( n1->y, n2->y, step );
    n->z = morph( n1->z, n2->z, step );
  }
}

void FacedObject::MoveAbs( int dx, int dy, int dz )
{
  for( int i=0; i<points.Count; i++ ) {
    points[i]->Move( dx, dy, dz );
  }
}

void FacedObject::RotateAbs( int rx, int ry, int rz )
{
  int i;
  for( i=0; i<points.Count; i++ ) {
    points[i]->Rotate( 0,0,0, rx, ry, rz );
  }
  for( i=0; i<faces.Count; i++ ) {
    faces[i]->normal.Rotate( 0,0,0, rx, ry, rz );
  }
}

void FacedObject::Store( char* filename )
{
  FILE* fp = fopen( filename, "ab+" );
  fwrite( &maxx, sizeof(int), 1, fp );
  fwrite( &maxy, sizeof(int), 1, fp );
  fwrite( &maxz, sizeof(int), 1, fp );
  fwrite( &minx, sizeof(int), 1, fp );
  fwrite( &miny, sizeof(int), 1, fp );
  fwrite( &minz, sizeof(int), 1, fp );
  fwrite( &dx, sizeof(int), 1, fp );
  fwrite( &dy, sizeof(int), 1, fp );
  fwrite( &dz, sizeof(int), 1, fp );
  fwrite( &rx, sizeof(int), 1, fp );
  fwrite( &ry, sizeof(int), 1, fp );
  fwrite( &rz, sizeof(int), 1, fp );
  fwrite( &lv, sizeof(float), 1, fp );
  fwrite( &lo, sizeof(float), 1, fp );
  fwrite( &sx, sizeof(float), 1, fp );
  fwrite( &sy, sizeof(float), 1, fp );
  fwrite( &sz, sizeof(float), 1, fp );
  fwrite( &flags, sizeof(int), 1, fp );
  fwrite( &perspect, sizeof(float), 1, fp );
  int i;
  fwrite( &points.Count, sizeof(int), 1, fp );
  for( i=0; i<points.Count; i++ ) {
    fwrite( points.At(i), sizeof(Point), 1, fp );
  }
  fwrite( &faces.Count, sizeof(int), 1, fp );
  for( i=0; i<faces.Count; i++ ) {
    Face* f = (Face*)faces.At(i);
    for( int j=0; j<3; j++ ) {
      int tmp = tmp_points.IndexOf( f->vert[j] );
      fwrite( &tmp, sizeof(int), 1, fp );
    }
    fwrite( &f->normal, sizeof(Vector), 1, fp );
  }
  resClose(fp);
}

void FacedObject::Load( char* filename, Image* texture )
{
  FILE* fp = resOpenFile( filename );
  fread( &maxx, sizeof(int), 1, fp );
  fread( &maxy, sizeof(int), 1, fp );
  fread( &maxz, sizeof(int), 1, fp );
  fread( &minx, sizeof(int), 1, fp );
  fread( &miny, sizeof(int), 1, fp );
  fread( &minz, sizeof(int), 1, fp );
  fread( &dx, sizeof(int), 1, fp );
  fread( &dy, sizeof(int), 1, fp );
  fread( &dz, sizeof(int), 1, fp );
  fread( &rx, sizeof(int), 1, fp );
  fread( &ry, sizeof(int), 1, fp );
  fread( &rz, sizeof(int), 1, fp );
  fread( &lv, sizeof(float), 1, fp );
  fread( &lo, sizeof(float), 1, fp );
  fread( &sx, sizeof(float), 1, fp );
  fread( &sy, sizeof(float), 1, fp );
  fread( &sz, sizeof(float), 1, fp );
  fread( &flags, sizeof(int), 1, fp );
  fread( &perspect, sizeof(float), 1, fp );
  int oldPointsCount = points.Count;
  int pointsCount;
  fread( &pointsCount, sizeof(int), 1, fp );
  int i;
  for( i=0; i<pointsCount; i++ ) {
    Point* p = new Point;
    fread( p, sizeof(Point), 1, fp );
    points.Insert(p);
    tmp_points.Insert(new Point(*p));
  }
  int facesCount;
  fread( &facesCount, sizeof(int), 1, fp );
  for( i=0; i<facesCount; i++ ) {
    int n[3];
    for( int j=0; j<3; j++ ) {
      fread( &(n[j]), sizeof(int), 1, fp );
    }
    Face* f = new Face( tmp_points.At(n[0]+oldPointsCount),
			tmp_points.At(n[1]+oldPointsCount),
			tmp_points.At(n[2]+oldPointsCount) );
    fread( &f->normal, sizeof(Vector), 1, fp );
    faces.Insert(f);
  }
  resClose(fp);
  texture_image = texture;
}

#ifdef __WATCOMC__
void memsetw( short* mem, int size, short value );
#pragma aux memsetw = \
"cld"\
"push ax"\
"shl eax,16"\
"pop ax"\
"shr ecx,1"\
"rep stosd"\
parm [edi] [ecx] [eax];
#else
#ifdef i386
void memsetw( short* mem, int size, short value )
{
__asm__(
"pushl %%edi		\n"
"movl %0,%%edi		\n"
"cld			\n"
"pushw %%ax		\n"
"shl $16,%%eax		\n"
"popw %%ax		\n"
"shr $1,%%ecx		\n"
"repne			\n"
"stosl			\n"
"popl %%edi		\n"
:
: "ri" (mem), "c" (size), "a"(value) );
}
#else
void memsetw( short* mem, int size, short value )
{
  for( int i=0; i<size; i++ ) mem[i] = value;
}
#endif
#endif

void Object3D::ClearZBuffer( short* zbuffer, int value )
{
  for( int i=0; i<vidSizeY; i++ )
    memsetw( zbuffer+i*vidBytesPerLine, vidSizeX, value );
}

void Init3D()
{
  int i;
  for( i=0; i<COS_STEPS_3D; i++ ) {
#ifdef FLOAT_ROTATE
    cosTable3D[i] = cos(2*M_PI*i/COS_STEPS_3D);
    sinTable3D[i] = sin(2*M_PI*i/COS_STEPS_3D);
#else
    cosTable3D[i] = 256*cos(2*M_PI*i/COS_STEPS_3D);
    sinTable3D[i] = 256*sin(2*M_PI*i/COS_STEPS_3D);
#endif
  }
  for( i=0; i<ACOS_STEPS; i++ ) {
    acosTable[i] = acos((float)(i-ACOS_STEPS/2)/ACOS_STEPS*2) /
		   M_PI*(PHONG_STEPS-1);
  }
  for( i=0; i<PHONG_STEPS; i++ ) {
    for( int j=0; j<PHONG_STEPS; j++ ) {
      float dist = sqrt( (float)sqr(i+1-PHONG_STEPS/2) +
			 (float)sqr(j+1-PHONG_STEPS/2) )
		   / (PHONG_STEPS/2);
      if( dist > 1 ) dist = 1;
      phongTable[i][j] = cos( acos(1-dist) )*(VID_MAX_BRIGHT-1);
    }
  }
  for( i=0; i<vidSizeY; i++ ) {
//    y_table[i] = vidBytesPerLine*i;
  }
}

void Cluster::Insert( Face* f )
{
  TCollection<Face>::Insert(f);
  UpdateBlock();
}

void Cluster::UpdateBlock()
{
  du = dv = (1<<16);
  size_u = size_v = 0;
  for( int i=0; i<Count; i++ ) {
    Face* f = At(i);
    if( !f ) continue;
    if( f->du < du ) du = f->du;
    if( f->dv < dv ) dv = f->dv;
    if( f->du+f->size_u > size_u )
      size_u = f->du+f->size_u;
    if( f->dv+f->size_v > size_v )
      size_v = f->dv+f->size_v;
  }
  size_u -= du;
  size_v -= dv;
}

Cluster* Cluster::Split()
{
  Cluster* c = new Cluster;
  if( size_u > size_v ) {
    for( int i=0; i<Count; i++ ) {
      Face* f = At(i);
      if( !f ) continue;
      if( f->du+f->size_u/2 >= du+size_u/2 ) {
	Delete(i);
	c->Insert(f);
      }
    }
  }
  else {
    for( int i=0; i<Count; i++ ) {
      Face* f = At(i);
      if( !f ) continue;
      if( f->dv+f->size_v/2 >= dv+size_v/2 ) {
	Delete(i);
	c->Insert(f);
      }
    }
  }
  UpdateBlock();
  if( c->Count )
    return c;
  else {
    error( "cluster splitting failed" );
    return 0;
  }
}

void FacedObject::SplitTexture()
{
  TCollection<Cluster> clusters;
  Cluster* first = new Cluster;
  int i;
  for( i=0; i<faces.Count; i++ ) {
    first->Insert( faces[i] );
  }
  clusters.Insert( first );
  for( i=0; i<clusters.Count; i++ ) {
    Cluster* c = clusters[i];
    if( (c->size_u > TEXTURE_SIZE  || c->size_v > TEXTURE_SIZE ) ) {
      clusters.Insert( c->Split() );
      clusters.Insert( c );
      clusters.Delete( i );
    }
  }
  clusters.Pack();
  for( i=0; i<clusters.Count; i++ ) {
    Cluster* cluster = clusters[i];
/*
    DrawLinePix( vidVideoMemory,
		 cluster->du, cluster->dv,
		 cluster->du, cluster->dv+cluster->size_v, 255 );
    DrawLinePix( vidVideoMemory,
		 cluster->du, cluster->dv,
		 cluster->du+cluster->size_u, cluster->dv, 255 );
    DrawLinePix( vidVideoMemory,
		 cluster->du, cluster->dv+cluster->size_v,
		 cluster->du+cluster->size_u,
		 cluster->dv+cluster->size_v, 255 );
    DrawLinePix( vidVideoMemory,
		 cluster->du+cluster->size_u, cluster->dv,
		 cluster->du+cluster->size_u,
		 cluster->dv+cluster->size_v, 255 );
*/
//    getchar();
    cluster->Pack();
    if( !cluster->Count ) continue;
    cluster->UpdateBlock();
    uchar* cluster_texture = new uchar[TEXTURE_SIZE*TEXTURE_SIZE];
    int v;
    for( v=0; v < cluster->size_v; v++ ) {
      memcpy( cluster_texture + v*TEXTURE_SIZE,
	      texture_image->data +
	      (cluster->dv+v)*texture_image->bytesPerLine + cluster->du,
	      TEXTURE_SIZE );
    }
    for( ; v < TEXTURE_SIZE; v++ ) {
      memcpy( cluster_texture + v*TEXTURE_SIZE,
	      texture_image->data +
	      (cluster->dv+cluster->size_v)*texture_image->bytesPerLine +
	      cluster->du, TEXTURE_SIZE );
    }
    for( int j=0; j<cluster->Count; j++ ) {
      Face* f = (*cluster)[j];
      f->texture = cluster_texture;
      f->du = cluster->du;
      f->dv = cluster->dv;
    }
    cluster->DeleteAll();
    clusters.Free(i);
  }
}

void FacedObject::SplitFace( Face* f, bool vertical )
{
  static color = 255;
  Point *p1, *p2;
  int n1 ,n2;
  bool s01 = false;
  int split;
  int i;

  if( vertical ) {
    split = f->dv+f->size_v/2;
    if( ((split - f->vert[0]->t.s.v) ^ (split - f->vert[1]->t.s.v)) < 0 ) {
      n1 = 2; s01 = true;
    }
    if( ((split - f->vert[1]->t.s.v) ^ (split - f->vert[2]->t.s.v)) < 0 ) {
      if( s01 ) n2 = 0; else n1 = 0;
    }
    if( ((split - f->vert[2]->t.s.v) ^ (split - f->vert[0]->t.s.v)) < 0 ) {
      n2 = 1;
    }
  }
  else {
    split = f->du+f->size_u/2;
    if( ((split - f->vert[0]->t.s.u) ^ (split - f->vert[1]->t.s.u)) < 0 ) {
      n1 = 2; s01 = true;
    }
    if( ((split - f->vert[1]->t.s.u) ^ (split - f->vert[2]->t.s.u)) < 0 ) {
      if( s01 ) n2 = 0; else n1 = 0;
    }
    if( ((split - f->vert[2]->t.s.u) ^ (split - f->vert[0]->t.s.u)) < 0 ) {
      n2 = 1;
    }
  }
  p1 = new Point ( f->vert[(n2+1)%3]->x +
		   (f->vert[(n2+2)%3]->x - f->vert[(n2+1)%3]->x)*0.5,
		   f->vert[(n2+1)%3]->y +
		   (f->vert[(n2+2)%3]->y - f->vert[(n2+1)%3]->y)*0.5,
		   f->vert[(n2+1)%3]->z +
		   (f->vert[(n2+2)%3]->z - f->vert[(n2+1)%3]->z)*0.5,
		   shorts(
		     f->vert[(n2+1)%3]->t.s.u +
		     (f->vert[(n2+2)%3]->t.s.u - f->vert[(n2+1)%3]->t.s.u)*0.5,
		     f->vert[(n2+1)%3]->t.s.v +
		     (f->vert[(n2+2)%3]->t.s.v - f->vert[(n2+1)%3]->t.s.v)*0.5
		   ) );
  p2 = new Point ( f->vert[(n1+1)%3]->x +
		   (f->vert[(n1+2)%3]->x - f->vert[(n1+1)%3]->x)*0.5,
		   f->vert[(n1+1)%3]->y +
		   (f->vert[(n1+2)%3]->y - f->vert[(n1+1)%3]->y)*0.5,
		   f->vert[(n1+1)%3]->z +
		   (f->vert[(n1+2)%3]->z - f->vert[(n1+1)%3]->z)*0.5,
		   shorts(
		     f->vert[(n1+1)%3]->t.s.u +
		     (f->vert[(n1+2)%3]->t.s.u - f->vert[(n1+1)%3]->t.s.u)*0.5,
		     f->vert[(n1+1)%3]->t.s.v +
		     (f->vert[(n1+2)%3]->t.s.v - f->vert[(n1+1)%3]->t.s.v)*0.5
		   ) );
  for( i=0; i<points.Count; i++ ) {
    Point* p = points[i];
    if( p->x==p1->x && p->y==p1->y && p->z==p1->z ) {
      delete p1;
      p1 = tmp_points[ points.IndexOf(p) ];
      break;
    }
  }
  if( i == points.Count ) {
    tmp_points.Insert(p1);
    points.Insert( new Point(*p1) );
  }
  for( i=0; i<points.Count; i++ ) {
    Point* p = points[i];
    if( p->x==p2->x && p->y==p2->y && p->z==p2->z ) {
      delete p2;
      p2 = tmp_points[ points.IndexOf(p) ];
      break;
    }
  }
  if( i == points.Count ) {
    tmp_points.Insert(p2);
    points.Insert( new Point(*p2) );
  }
  int n3 = 3-n1-n2;
  if( !n3 ) {
    Point* p = p1; p1 = p2; p2 = p;
  }
  faces.Delete(f);
  if( p1 != p2 )
    faces.Insert( new Face( f->vert[n3], p1, p2 ) );
/*
  Face* ff = faces[faces.Count-1];
  for( i=0; i<3; i++ ) {
    DrawLinePix( vidVideoMemory,
		 vidScaleX((ff->vert[i]->x>>16)+VID_MAX_SIZE_X/2),
		 vidScaleY((ff->vert[i]->y>>16)+VID_MAX_SIZE_Y/2),
		 vidScaleX((ff->vert[(i+1)%3]->x>>16)+VID_MAX_SIZE_X/2),
		 vidScaleY((ff->vert[(i+1)%3]->y>>16)+VID_MAX_SIZE_Y/2),
		 color );
  }
//  getch();
  color--;
*/
  if( sqrt( sqr((int)p1->t.s.u-(int)f->vert[(n3+2)%3]->t.s.u)+
	    sqr((int)p1->t.s.v-(int)f->vert[(n3+2)%3]->t.s.v) ) <
      sqrt( sqr((int)p2->t.s.u-(int)f->vert[(n3+1)%3]->t.s.u)+
	    sqr((int)p2->t.s.v-(int)f->vert[(n3+1)%3]->t.s.v) ) ) {

    if( p1 != p2 )
      faces.Insert( new Face ( p2, p1, f->vert[(n3+2)%3] ) );
/*
    ff = faces[faces.Count-1];
    for( i=0; i<3; i++ ) {
      DrawLinePix( vidVideoMemory,
		   vidScaleX((ff->vert[i]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[i]->y>>16)+VID_MAX_SIZE_Y/2),
		   vidScaleX((ff->vert[(i+1)%3]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[(i+1)%3]->y>>16)+VID_MAX_SIZE_Y/2),
		   color );
    }
  //  getch();
    color--;
*/
    faces.Insert( new Face( p1, f->vert[(n3+1)%3], f->vert[(n3+2)%3] ) );
/*
    ff = faces[faces.Count-1];
    for( i=0; i<3; i++ ) {
      DrawLinePix( vidVideoMemory,
		   vidScaleX((ff->vert[i]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[i]->y>>16)+VID_MAX_SIZE_Y/2),
		   vidScaleX((ff->vert[(i+1)%3]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[(i+1)%3]->y>>16)+VID_MAX_SIZE_Y/2),
		   color );
    }
  //  getch();
    color--;
*/
  }
  else {
    if( p1 != p2 )
      faces.Insert( new Face ( p2, p1, f->vert[(n3+1)%3] ) );
/*
    ff = faces[faces.Count-1];
    for( i=0; i<3; i++ ) {
      DrawLinePix( vidVideoMemory,
		   vidScaleX((ff->vert[i]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[i]->y>>16)+VID_MAX_SIZE_Y/2),
		   vidScaleX((ff->vert[(i+1)%3]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[(i+1)%3]->y>>16)+VID_MAX_SIZE_Y/2),
		   color );
    }
  //  getch();
    color--;
*/
    faces.Insert( new Face( p2, f->vert[(n3+1)%3], f->vert[(n3+2)%3] ) );
/*
    ff = faces[faces.Count-1];
    for( i=0; i<3; i++ ) {
      DrawLinePix( vidVideoMemory,
		   vidScaleX((ff->vert[i]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[i]->y>>16)+VID_MAX_SIZE_Y/2),
		   vidScaleX((ff->vert[(i+1)%3]->x>>16)+VID_MAX_SIZE_X/2),
		   vidScaleY((ff->vert[(i+1)%3]->y>>16)+VID_MAX_SIZE_Y/2),
		   color );
    }
  //  getch();
    color--;
*/
  }
  delete f;
}

void FacedObject::SplitFaces()
{
  for( int i=0; i<faces.Count; i++ ) {
    Face* f = faces[i];
    if( f->size_u > TEXTURE_SIZE ) {
      SplitFace( f, false );
      continue;
    }
    if( f->size_v > TEXTURE_SIZE ) {
      SplitFace( f, true );
    }
  }
  faces.Pack();
}

void FacedObject::Attach( FacedObject* o )
{
  int i;
  for( i=0; i<o->faces.Count; i++ ) {
    faces.Insert( o->faces[i] );
  }
  for( i=0; i<o->points.Count; i++ ) {
    points.Insert( o->points[i] );
    tmp_points.Insert( o->tmp_points[i] );
  }
}

void FaceCollection::Insert( Face* f )
{
  if( (f->vert[0]->x != f->vert[1]->x || f->vert[0]->y != f->vert[1]->y ||
       f->vert[0]->z != f->vert[1]->z ) &&
      (f->vert[1]->x != f->vert[2]->x || f->vert[1]->y != f->vert[2]->y ||
       f->vert[1]->z != f->vert[2]->z ) &&
      (f->vert[0]->x != f->vert[2]->x || f->vert[0]->y != f->vert[2]->y ||
       f->vert[0]->z != f->vert[2]->z ) ) {
    TCollection<Face>::Insert(f);
  }
}
