{ *******************************************************************
  *			    This file is part of the WMFH package				*
  ******************************************************************* }


{

	 	Mn_unit.pas	   A Turbo Pascal unit for menus and menu bars


			 Copyright (c) 1997   Gianfranco Boggio-Togna                              *

							C.P. 14021                                   
					  I-20140 Milano (Italy)                             

					 e-mail: gbt@computer.org                                                     *


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

}


{$R+}
{$S+}

{ Defining 'UseHelp' allows a help section identifier to be associated
  with a menu entry, so that pressing F1 while the entry is selected 
  causes the help section to be displayed.
}

{$DEFINE  UseHelp}

UNIT   MN_unit ;

INTERFACE

USES   Dos, Crt, WND_Unit, KB_Unit, MS_Unit ;

TYPE
	   Bar_entry_type = ( Command_entry, Repeating_Command_entry, Menu_entry) ; 	



{ =========================================================================== 
							 M E N U S  
  =========================================================================== }


{ -------------------------- 
  Start definition of a menu
  -------------------------- }

FUNCTION  MN_new_menu ( menu_title: String ;       
						frame:      WND_Frame_type ;
					 	shadow:     WND_Shadow_type ;
						line_width,
					 	background_color,
					 	foreground_color,
					 	high_foreground_color,
					 	low_foreground_color,
					 	select_background_color,
					 	select_foreground_color,
					 	select_high_foreground_color : Byte)
						: Integer ;


{ menu_title                    title for menu window
  frame                         frame type for menu window
  shadow                        shadow type for menu window
  line_width                    max line width for a menu entry
  background_color              for menu window
  foreground_color              for menu window
  high_foreground_color         for highlighted character
  low_foreground_color          for inactive (non-selectable) entries
  select_background_color       for selection bar
  select_foreground_color       for selection bar

  MN_new_menu                   an integer identifying the menu
}


{ ---------------------- 
  Add an entry to a menu 
  ---------------------- }

PROCEDURE  MN_menu_line ( menu:   		 Integer;
						  result: 		 Word ;
{$IFDEF USEHELP}
					   	  help_section: Integer ;
{$ENDIF}
					      left, right:  String );


{ menu                  the menu identifier
  result                the value to be returned when the entry is selected

  help_section 			the identifier of the help section to be displayed
						when F1 is pressed, or 0 if no help is available

  left                  text string to be left justified within entry
  right                 text string to be right justified within entry
						
  SPECIAL CASES

  left = right = ''   	insert a separator line
  right = '#16'         the entry has a submenu (identifier in 'result')

  Within the 'left' string, the escape sequence '\e' will cause the next
  character (and that character only) to be displayed using the color
  defined by the 'high_foreground_color' parameter in the call to
  'MN_new_menu'.  When the menu is displayed, entering the letter following
  the escape sequence activates the menu entry.
}


{ ---------------------------------
  Make an entry selectable/inactive 
  --------------------------------- }

PROCEDURE  MN_menu_line_selectable (menu: Integer;
									line_no: Integer;
									sel: Boolean) ;

{ menu          the menu identifier
  line_no       the entry number (depends on order of calls to menu_line)
  sel           whether the entry is to be selectable
}

{ -------------------------------
  Change the text in a menu entry 
  ------------------------------- }

PROCEDURE  MN_menu_line_new_text (menu: Integer;
								  line_no: Integer;
								  left, right: String) ;
{ menu          the menu identifier
  line_no       the entry number (depends on order of calls to menu_line)
  left,right    the new text strings
}


{ -----------------------------
  Remove (permanently) an entry 
  ----------------------------- }

PROCEDURE  MN_menu_line_delete (menu: Integer; line_no: Integer) ;

{ menu          the menu identifier
  line_no       the entry number (depends on order of calls to menu_line)
}


{ --------------------------------------------
  Display a menu and return the user selection 
  -------------------------------------------- }

FUNCTION   MN_show_menu  (menu: Integer; y, x: Integer) : Word ;

{ menu          the menu identifier
  y, x          the top left corner coordinates

  MN_show_menu  the value of the 'result' parameter in the call to menu_line
				for the entry selected, or 0 if no entry selected
}



{ =========================================================================== 
							 M E N U   B A R S  
  =========================================================================== }



{ ------------------------------ 
  Start definition of a menu bar
  ------------------------------ }

FUNCTION  MN_new_menu_bar ( background_color,
							foreground_color,
						 	high_foreground_color,
						 	select_background_color,
						 	select_foreground_color,
						 	select_high_foreground_color: Byte)
							: Integer ;

{ background_color              
  foreground_color
  high_foreground_color         highlighted letter

  select_background_color       colors for selected entry       
  select_foreground_color
  select_high_foreground_color          

  MN_new_menu_bar               an integer identifying the menu bar
}


{ -------------------------- 
  Add an entry to a menu bar 
  -------------------------- }

PROCEDURE  	MN_menu_bar_entry ( menu_bar: Integer;
								name: String;
							    menu_or_value: Word;
								entry_type: Bar_entry_type ) ;

{ menu_bar              the integer identifying the menu bar
  name                  the entry
  menu_or_value         either a menu identifier or a value to be
						returned when the entry is selected
  entry_type            menu, command, repeating command

  Within the 'name' string, the escape sequence '\e' will cause the next
  character (and that character only) to be displayed using the color
  defined by the 'high_foreground_color' parameter in the call to
  'MN_new_menu_bar'.  Entering Alt + <the letter following the escape
  sequence> will activate the entry.
}


{ ------------------
  Display a menu bar
  ------------------ }

PROCEDURE   MN_show_menu_bar  (menu_bar: Integer; y, x, xlast: Integer) ;

{ menu_bar              the integer identifying the menu bar
  y, x                  bar coordinates
  xlast                 x coordinate of last char of menu bar                           

  SPECIAL CASE
  y = 0
  x = n                 show menu with n-th item highlighted
  xlast = ?             ignored
}


{ -----------------------------------------
  Check whether menu bar has been activated
  ----------------------------------------- }

FUNCTION   MN_check_menu_bar  (menu_bar: Integer; key: Word) : Word ;

{ menu_bar              the integer identifying the menu bar
  key                   the scan code for the last key pressed.
						If 0, check for mouse event.
  MN_check_menu_bar     the value associated to a menu bar entry, or 0
}



IMPLEMENTATION
{$IFDEF UseHelp}
USES	HLP_Unit ;	
{$ENDIF}		

CONST
		Copyright:	String =
					'WMFH 1.0 - Copyright 1997 Gianfranco Boggio-Togna' ;


CONST
		Alt_letter: Array [0..25]  OF  Word =
					(   KB_AltA, KB_AltB, KB_AltC, KB_AltD, KB_AltE,
						KB_AltF, KB_AltG, KB_AltH, KB_AltI, KB_AltJ,
						KB_AltK, KB_AltL, KB_AltM, KB_AltN, KB_AltO,
						KB_AltP, KB_AltQ, KB_AltR, KB_AltS, KB_AltT,
						KB_AltU, KB_AltV, KB_AltW, KB_AltX, KB_AltY,
						KB_AltZ ) ;

		max_items = 16 ; 		{ on a menu bar }


TYPE
		Text_string  = String [80] ;

		Menu_ptr     = ^ Menu ;
		Line_ptr     = ^ Line ;

		Menu =   RECORD
						width:                  Byte ;
						lines:                  Byte ;
						frame_t:                WND_Frame_type ;
						shadow_t:               WND_Shadow_type ;
						bg_color:               Byte ;
						fg_color:               Byte ;
						high_fg_color:          Byte ;
						low_fg_color:           Byte ;
						select_bg_color:        Byte ;
						select_fg_color:        Byte ;
						select_high_fg_color:   Byte ;
						lines_list:             Line_ptr ;
				 END ;


		Line =   RECORD
						next_line:              Line_ptr ;
						prev_line:              Line_ptr ;
						result:                 Word ;
						help:					Integer ;
						letter:                 Char ;
						flags:                  Byte ;
						entry:                  Text_string ;
				 END ;

		Menu_bar_ptr     = ^ Menu_bar ;

		Menu_bar_item =  RECORD
								item_menu_or_value: Word ;
								item_type:       	Bar_entry_type ;   
								item_y:             Byte ;      
								item_x:             Byte ;
								item_name_length:   Byte ;
								item_name:          String [32] ;
								item_letter:        Char ;

						 END ;

		Menu_bar =  RECORD
						bg_color:               Byte ;
						fg_color:               Byte ;
						high_fg_color:          Byte ;
						select_bg_color:        Byte ;
						select_fg_color:        Byte ;
						select_high_fg_color:   Byte ;
						n_entries:              Byte ;
						y_coord:                Byte ;
						x_coord:                Byte ;
						x_last_coord:           Byte ;
						entries:        ARRAY [1..max_items] OF Menu_bar_item ;
					END ;


CONST
		max_menus = 50 ;

		Line_record_length = SizeOf(Line_ptr)*2 + SizeOf(Word) +
							 + SizeOf(Char) + SizeOf(Byte) ;

		flag_select  = 1 ;
		flag_line    = 2 ;
		flag_submenu = 4 ;
		flag_head    = 128 ;

		max_menu_bars = 10 ;

VAR
		menu_table:     ARRAY [1..max_menus]     OF Menu_ptr ;  
		last_menu :     Integer ;

		menu_bar_table: ARRAY [1..max_menu_bars] OF Menu_bar_ptr ;      
		last_menu_bar : Integer ;


{ -----------------------------------------------------------------------
							 M E N U S 
  ----------------------------------------------------------------------- }

{ Insert a new node AFTER an existing one }

PROCEDURE  link_after (new, old: Line_ptr) ;
VAR
		f:      Line_ptr ;
BEGIN
		f := old^.next_line ;  
		old^.next_line := new ;
		new^.prev_line := old ; 
		new^.next_line := f ;
		f^.prev_line   := new ;
END ;

{ Insert a new node BEFORE an existing one }

PROCEDURE  link_before (new, old: Line_ptr) ;
VAR
		f:      Line_ptr ;
BEGIN
		f := old^.prev_line ;  
		old^.prev_line := new ;
		new^.prev_line := f ; 
		new^.next_line := old ;
		f^.next_line   := new ;
END ;

{ Remove a node }

PROCEDURE  unlink_line (l: Line_ptr) ;
VAR
		p, n:   Line_ptr ;
BEGIN
		p := l^.prev_line ;
		n := l^.next_line ;
		n^.prev_line := p;
		p^.next_line := n ;
END ;

{ ------------------------------------------------------------- }

FUNCTION  MN_new_menu ( menu_title: String ;       
						frame:      WND_Frame_type ;
					    shadow:     WND_Shadow_type ;
					 	line_width,
					 	background_color,
					 	foreground_color,
					 	high_foreground_color,
					 	low_foreground_color,
					 	select_background_color,
					 	select_foreground_color,
					 	select_high_foreground_color: Byte)
						: Integer ;
VAR
		lp:         Line_ptr ;  
BEGIN
		Inc(last_menu) ;
		new (menu_table [last_menu]) ;
		WITH  menu_table [last_menu]^  DO
		  BEGIN
				width           :=    line_width ;              
				frame_t         :=    frame ;           
				shadow_t        :=    shadow ;          
				bg_color        :=    background_color ;        
				fg_color        :=    foreground_color ;        
				high_fg_color   :=    high_foreground_color ;   
				low_fg_color    :=    low_foreground_color ;    
				select_bg_color :=    select_background_color ;
				select_fg_color :=    select_foreground_color ;
				select_high_fg_color :=    select_high_foreground_color ;

				GetMem (lp, Line_record_length + Length(menu_title) + 1) ;
				lp^.flags := flag_head ;
				lp^.next_line := lp ;
				lp^.prev_line := lp ;
				lp^.entry := menu_title ;
				lines_list := lp ;

				lines := 0 ;
		  END ; 
		MN_new_menu := last_menu ;
END ;


{ ------------------------------------------------------------- }

PROCEDURE  MN_menu_line ( menu:   		 Integer;
					      result: 		 Word ;
{$IFDEF USEHELP}
					   	  help_section: Integer ;
{$ENDIF}
					   	  left, right:  String );
VAR
		i, n:   Integer ;
		lp:     Line_ptr ;      
		mp:     Menu_ptr ;
		l:      Char ;
BEGIN
		IF  (menu > 0)  AND  (menu <= last_menu)         THEN
		  BEGIN
			l := ' ' ;
			mp := menu_table [menu] ;           
			n := 0 ;
			i := 1 ;
			WHILE  i <=  Length (left)  DO
			  BEGIN
				IF  left [i] = '\'  THEN
				  BEGIN
					Inc (i) ;
					IF  (i > Length (left))  OR (left[i] = '\')  THEN
					  Inc(n) 
					ELSE
					  IF  (left[i] = 'e')  OR  (left[i] = 'E')  THEN
						IF  i < Length(left)  THEN
						  l := left [i+1] ;
				  END 
				ELSE
				  Inc(n) ;
				Inc(i) ;

			  END ;     

			GetMem (lp, Line_record_length + mp^.width + 4 + 1) ;
			link_before (lp, mp^.lines_list) ;          
			lp^.result := result ;
			lp^.help := help_section ;
			lp^.letter := UpCase(l) ;
			IF  (left = '')  AND  (right = '')  THEN 
			  BEGIN
				lp^.flags := flag_line ;
				lp^.entry [0] := Chr(mp^.width + 3) ;
				CASE  mp^.frame_t  OF
				WND_Single_frame:  BEGIN
									 lp^.entry [1] := Chr(195) ;
									 lp^.entry [mp^.width+3] := Chr(180) ;
								   END ;    
				WND_Double_frame:  BEGIN
									 lp^.entry [1] := Chr(199) ;
									 lp^.entry [mp^.width+3] := Chr(182) ;
								   END ;    
				WND_No_frame:      BEGIN
									 lp^.entry [1] := ' ' ;
									 lp^.entry [mp^.width+3] := ' ' ;
								   END ;    
				END ;

				FOR  i := 2  TO mp^.width+2  DO
				  lp^.entry[i] := Chr(196) ;
			  END 
			ELSE
			  BEGIN
				lp^.flags := flag_select ;
				IF  right = #16  THEN
				  lp^.flags := lp^.flags OR flag_submenu ;
				lp^.entry := left ;
			   	n :=  mp^.width - n - Length(right) - 1;
			   	FOR  i := 1  TO n  DO
			      lp^.entry := lp^.entry + ' ' ;
				lp^.entry := lp^.entry + right ;
			  END ;
			Inc(mp^.lines) ;

		  END ;
END ;

{ ------------------------------------------------------------- }

PROCEDURE  MN_menu_line_selectable (menu: Integer;
									line_no: Integer;
									sel: Boolean) ;
VAR 
		i:      Integer ;
		lp:         Line_ptr ;  
BEGIN
		IF  (menu > 0)  AND  (menu <= last_menu)  THEN
		  WITH  menu_table [menu]^  DO
			BEGIN
			  i := 1 ;
			  lp := lines_list^.next_line ;
			  WHILE  NOT ((lp^.flags  AND  flag_head) = flag_head) AND
				     (i <> line_no)  DO
				BEGIN
				  lp := lp^.next_line ;
				  Inc(i) ;
				END ;
			  IF  NOT ((lp^.flags  AND  flag_head) = flag_head) THEN
				IF  sel  THEN
				  lp^.flags := lp^.flags  OR flag_select 
				ELSE
				  lp^.flags := lp^.flags  AND (NOT flag_select) ;
			END ;
END ;


{ ------------------------------------------------------------- }

PROCEDURE  MN_menu_line_new_text (menu: Integer;
								  line_no: Integer;
								  left, right: String) ;
VAR 
		i, n:  Integer ;
		l:     Char ;
		lp:    Line_ptr ;  
BEGIN
		IF  (menu > 0)  AND  (menu <= last_menu)  THEN
		  WITH  menu_table [menu]^  DO
			BEGIN
				i := 1 ;
				lp := lines_list^.next_line ;
				WHILE  NOT ((lp^.flags  AND  flag_head) = flag_head) AND
					   (i <> line_no)  DO
				  BEGIN
					lp := lp^.next_line ;
					Inc(i) ;
				  END ;
				IF  NOT ((lp^.flags  AND  flag_head) = flag_head) THEN
				  BEGIN
					l := ' ' ;
					n := 0 ;
					i := 1 ;
					WHILE  i <=  Length (left)  DO
			  		  BEGIN
						IF  left [i] = '\'  THEN
				  		  BEGIN
							Inc (i) ;
							IF  (i > Length (left))  OR (left[i] = '\')  THEN
					  			Inc(n) 
							ELSE
						  	  IF  (left[i] = 'e')  OR  (left[i] = 'E')  THEN
								IF  i < Length(left)  THEN
								  l := left [i+1] ;
					  	  END 
						ELSE
				  		  Inc(n) ;
						Inc(i) ;

			  		  END ;     
					lp^.entry := left ;
					lp^.letter := UpCase(l) ;
					n :=  width - n - Length(right) - 1;
					FOR  i := 1  TO n  DO
						lp^.entry := lp^.entry + ' ' ;
					lp^.entry := lp^.entry + right ;
				  END ;
			END ;
END ;


{ ------------------------------------------------------------- }

PROCEDURE  MN_menu_line_delete (menu: Integer; line_no: Integer) ;
VAR 
		i:      Integer ;
		lp:         Line_ptr ;  
BEGIN
		IF  (menu > 0)  AND  (menu <= last_menu)  THEN
		  WITH  menu_table [menu]^  DO
			BEGIN
				i := 1 ;
				lp := lines_list^.next_line ;
				WHILE  NOT ((lp^.flags  AND  flag_head) = flag_head) AND
					   (i <> line_no)  DO
				  BEGIN
					lp := lp^.next_line ;
					Inc(i) ;
				  END ;
				IF  NOT ((lp^.flags  AND  flag_head) = flag_head) THEN
				  BEGIN
						Dec(lines) ;
						unlink_line (lp) ;
						FreeMem (lp, Line_record_length + width + 4 + 1) ;
				  END ;
			END ;
END ;


{ ------------------------------------------------------------- }

FUNCTION   MN_show_menu  (menu: Integer; y, x: Integer) : Word ;

VAR
		screen_save:       Integer ;
		no_screen_save:    Integer ;
		lp:                Line_ptr ;   
		y_value:           Integer ;
		sel_ptr:           Line_ptr ;   
		key:               Word ;
		ch:                Char ;
		sub:               Word ;
		finished:          Boolean ;
		xm, ym:            Integer ;

  PROCEDURE  list_entries (selected: Line_ptr) ;
  VAR
		i:      Integer ;
  BEGIN
		WITH  menu_table [menu]^  DO
		  BEGIN

			y_value := y + 1 ;
			lp := lines_list^.next_line ;

			WHILE  NOT ((lp^.flags AND flag_head) = flag_head) DO
			  BEGIN
				IF  lp = selected THEN
				  BEGIN
					TextColor (select_fg_color) ;
					TextBackground (select_bg_color) ;
				  END
				ELSE
				  BEGIN
					TextBackground (bg_color) ;
					IF (lp^.flags  AND  flag_select) = 0 THEN
						TextColor (low_fg_color) 
					ELSE
						TextColor (fg_color) ;
				  END ;
				IF  (lp^.flags  AND  flag_line) = flag_line  THEN
				  BEGIN
					TextColor (fg_color) ;
					gotoXY (x , y_value) ;
					Write (lp^.entry) ;
				  END
				ELSE
				  BEGIN
					gotoXY (x+1 , y_value) ;
					write (' ') ;

					i := 1 ;
					WHILE  i <= Length(lp^.entry)  DO
					  BEGIN
						IF  lp = selected  THEN
						  TextColor (select_fg_color) 
						ELSE
						  IF  (lp^.flags AND flag_select) = 0  THEN
							TextColor (low_fg_color) 
						  ELSE
							TextColor (fg_color) ;
						IF  lp^.entry[i] = '\'  THEN
						  BEGIN  
							Inc(i) ;
							IF  lp^.entry[i] <> '\'  THEN
							  IF  (lp^.entry[i] = 'e')  OR
								  (lp^.entry[i] = 'E') THEN
								BEGIN
								  Inc(i) ;
								  IF  (lp^.flags  AND flag_select) = 0 THEN
									TextColor (low_fg_color) 
								  ELSE
									IF  lp = selected  THEN
									  TextColor (select_high_fg_color) 
									ELSE
									  TextColor (high_fg_color) ;
								END ;
						  END ;
						Write (lp^.entry[i]) ;
						Inc(i) ;
					  END ;

					gotoXY (x+width+1 , y_value) ;
					write (' ') ;
				  END ;

				lp := lp^.next_line ;
				Inc (y_value) ;
			  END ;
		  END ;

  END ;


BEGIN
		MN_show_menu := 0 ;
		no_screen_save := 0 ;
		IF  (menu < 0)  OR  (menu > last_menu)  THEN
				Exit ;
		WITH  menu_table [menu]^  DO
		  BEGIN 
			screen_save := 1 ;
			IF  (frame_t <> WND_No_frame)  AND  (x > 1)  THEN
			  BEGIN
				WND_Open_Window ( y, x-1, y + lines + 1 , x + width + 2 + 1 ,
								  fg_color, bg_color,
								  shadow_t, WND_No_frame,
								  '',
								  screen_save
								) ;
				WND_Open_Window ( y, x, y + lines + 1 , x + width + 2,
								  fg_color, bg_color,
								  WND_No_shadow, frame_t,
								  lines_list^.entry,
								  no_screen_save
								)  ;
			  END
			ELSE
				WND_Open_Window ( y, x, y + lines + 1 , x + width + 2,
								  fg_color, bg_color,
								  shadow_t, frame_t,
								  lines_list^.entry,
								  screen_save
								)  ;

			sel_ptr := lines_list^.next_line ;
			WHILE  (sel_ptr^.flags  AND  flag_select) = 0  DO
					sel_ptr := sel_ptr^.next_line ;


			finished := False ;

			REPEAT
			    list_entries (sel_ptr) ;

				REPEAT UNTIL  KeyPressed  OR  MS_LeftPressed  OR  MS_RightPressed ;
				IF  MS_LeftPressed  THEN
				  BEGIN
						xm := MS_WhereX ;
						ym := MS_WhereY ;
						y_value := ym - y - 1 ;
						IF  (ym < y)           OR
							(y_value > lines)  OR
							(xm < x)           OR
							(xm > x + width)  THEN
								key := KB_Esc 
						ELSE
						  BEGIN
								REPEAT UNTIL NOT MS_LeftDown ;
								lp := lines_list^.next_line ;
								WHILE  y_value > 0  DO
								  BEGIN
									lp := lp^.next_line ;
									Dec(y_value) ;
								  END ;
								IF  (lp^.flags AND flag_select) = 0  THEN
								  key := KB_Space  
								ELSE
								  BEGIN
									sel_ptr := lp ;
									key := KB_Enter ;
								  END ;
						  END ;
				  END
				ELSE
				  IF  MS_RightPressed  THEN
					BEGIN
						key := KB_Esc ;
                        REPEAT UNTIL NOT MS_RightDown ;
                    END
				  ELSE
					  key := KB_read ;

			  ch := Chr(Lo(key)) ;
			  IF  NOT (ch IN [#27, #13, #0])  THEN
				BEGIN
				  ch := UpCase(ch) ;
				  lp := lines_list^.next_line ;
				  WHILE  NOT ((lp^.flags AND flag_head) = flag_head)  DO
					BEGIN
					  IF  ((lp^.flags  AND  flag_select) = flag_select) AND
						  (lp^.letter = ch)  THEN
						BEGIN
						  sel_ptr := lp ;
						  key := KB_Enter ;
						END ;
					  lp := lp^.next_line ;
					END ;
				END ;

			  CASE  key  OF

			  KB_Esc:
					BEGIN
						finished := True ;
						MN_show_menu := 0 ;
					END ;

{$IFDEF UseHelp}
			  KB_F1:
					BEGIN
						IF  sel_ptr^.help <> 0  THEN
							HLP_display_section (sel_ptr^.help, '') ;
					END ;
{$ENDIF}		

			  KB_Enter:  
					BEGIN
					  IF (sel_ptr^.flags AND flag_submenu) = flag_submenu THEN
						BEGIN
						  y_value := y + 2 ;
						  lp := lines_list^.next_line ;
						  WHILE  lp <> sel_ptr  DO
							BEGIN
							  lp := lp^.next_line ;
							  Inc(y_value) ;
							END ;
						  sub := MN_show_menu (sel_ptr^.result, y_value, x + 2) ;
						  IF  sub <> 0  THEN
							MN_show_menu := sub 
						  ELSE
							key := KB_Space ;
						END
					  ELSE
						MN_show_menu := sel_ptr^.result ;
					  IF  key <> KB_Space  THEN
							  finished := True ;
					END ;

			  KB_Up:
					REPEAT
					  sel_ptr := sel_ptr^.prev_line ;
					UNTIL  (sel_ptr^.flags AND flag_select) = flag_select ; 

			  KB_Down:
					REPEAT
					  sel_ptr := sel_ptr^.next_line ;
					UNTIL  (sel_ptr^.flags AND flag_select) = flag_select ;  

			  KB_Left,
			  KB_Right:
					BEGIN
					  finished := True ;
					  MN_show_menu := key ;
					END ;

			  ELSE  IF  Lo(key) = 0  THEN
					  BEGIN
						finished := True ;
						MN_show_menu := key ;
					  END ;

			  END ;

			UNTIL  finished ;

		  END ;         
		  WND_Close_Window (screen_save) ;
END ;


{ -----------------------------------------------------------------------
							 M E N U   B A R S 
  ----------------------------------------------------------------------- }

FUNCTION  MN_new_menu_bar ( background_color,
							foreground_color,
							high_foreground_color,
							select_background_color,
							select_foreground_color,
							select_high_foreground_color: Byte)
							: Integer ;

BEGIN
		Inc(last_menu_bar) ;
		new (menu_bar_table [last_menu_bar]) ;
		WITH  menu_bar_table [last_menu_bar]^  DO
		  BEGIN
				bg_color                :=  background_color ;  
				fg_color                :=  foreground_color ;  
				high_fg_color           :=  high_foreground_color ;     
				select_bg_color         :=  select_background_color ;   
				select_fg_color         :=  select_foreground_color ;   
				select_high_fg_color    :=  select_high_foreground_color ;      
				n_entries               :=  0 ; 
		  END ; 
		MN_new_menu_bar := last_menu_bar ;
END ;


{ ------------------------------------------------------------- }

PROCEDURE  MN_menu_bar_entry (menu_bar: 	 Integer;
							  name: 		 String;
							  menu_or_value: Word;
							  entry_type: 	 Bar_entry_type ) ;
VAR
		i, n:   Integer ;
		mp:     Menu_bar_ptr ;
		l:      Char ;
BEGIN
		IF  (menu_bar > 0)  AND  (menu_bar <= last_menu_bar)     THEN
		  BEGIN
			l := ' ' ;
			mp := menu_bar_table [menu_bar] ;           
			n := 0 ;
			i := 1 ;
			WHILE  i <=  Length (name)  DO
			  BEGIN
				IF  name [i] = '\'  THEN
				  BEGIN
					Inc (i) ;
					IF  (i > Length (name))  OR (name[i] = '\')  THEN
					  Inc(n) 
					ELSE
					  IF  (name[i] = 'e')  OR  (name[i] = 'E')  THEN
						IF  i < Length(name)  THEN
								l := name [i+1] ;
				  END 
				ELSE
				  Inc(n) ;
				Inc(i) ;

			  END ;     

			Inc(mp^.n_entries) ;
			WITH  mp^.entries[mp^.n_entries]  DO
			  BEGIN
				item_letter := UpCase(l) ;
				item_name := name ;
				item_name_length := n ;
				item_type := entry_type ;
				item_menu_or_value := menu_or_value ;
			  END ;
		  END ;
END ;


{ ------------------------------------------------------------- }

PROCEDURE   MN_show_menu_bar  (menu_bar: Integer; y, x, xlast: Integer) ;
VAR
		i, j, n, spaces, extra: Integer ;
		mp:                     Menu_bar_ptr ;
BEGIN
		WND_Save_cursor (False) ;
		MS_Hide ;
		IF  (menu_bar > 0)  AND  (menu_bar <= last_menu_bar)  THEN
		  BEGIN
			mp := menu_bar_table [menu_bar] ;           
			WITH  mp^  DO
			  BEGIN
				IF  y <> 0  THEN
				  BEGIN
					y_coord := y ;
					x_coord := x ;
					x_last_coord := xlast ;
				  END ;
				n := 0 ;
				FOR  i := 1 TO  n_entries  DO
				  n := n + entries[i].item_name_length ;
				spaces := 1 ;
				extra := 0 ;
				IF n_entries > 1  THEN 
				  BEGIN
					spaces := (x_last_coord-x_coord+1-n) DIV n_entries ;
					extra :=  (x_last_coord-x_coord+1-n) MOD n_entries ;
					extra := (extra - spaces) DIV 2 ;
					IF  extra < 0  THEN
					  extra := 0 ;
				  END ;
				IF  spaces > 5  THEN
				  spaces := 5 ;
				gotoXY (x_coord, y_coord) ;
				n := x_coord ;
				TextBackground (bg_color) ;
				TextColor (fg_color) ;
				FOR  j := 1 TO extra  DO
				  BEGIN
					Write (' ') ;
					Inc(n) ;
				  END ;
				FOR  i := 1 TO n_entries  DO
				  BEGIN
					IF  spaces <> 0  THEN
					  BEGIN
						IF  (y = 0)  AND  (x <> 0)  AND (x = i-1)  THEN
						  TextBackground (select_bg_color) 
						ELSE
						  TextBackground (bg_color) ;
						Write (' ') ;
						Inc(n) ;
						TextBackground (bg_color) ;
						FOR  j := 1 TO spaces-2  DO
						  BEGIN
							Write (' ') ;
							Inc(n) ;
						  END ;
						IF  spaces > 1  THEN
						  BEGIN
							IF  (y = 0)  AND  (x = i)  THEN
							  TextBackground (select_bg_color) 
							ELSE
							  TextBackground (bg_color) ;
							Write (' ') ;
							Inc(n) ;
						  END ;
					  END ;
					j := 1 ;
					WITH  entries[i]    DO
					  BEGIN
						IF  y <> 0  THEN
						  BEGIN
							item_x := n ;
							item_y := y ;
						  END ;
						WHILE  j <= Length(item_name)  DO
						  BEGIN
							IF  (y = 0)  AND  (x = i)  THEN
							  BEGIN
								TextColor (select_fg_color) ;
								TextBackground (select_bg_color) ;
							  END
							ELSE
							  BEGIN
								TextColor (fg_color) ;
								TextBackground (bg_color) ;
							  END ;
							IF  item_name[j] = '\'  THEN
							  BEGIN  
								Inc(j) ; 
								IF  item_name[j] <> '\'  THEN
								  IF  (item_name[j] = 'e')  OR
									  (item_name[j] = 'E') THEN
									BEGIN
									  Inc(j) ; 
									  IF  (y = 0)  AND  (x = i)  THEN
										TextColor (select_high_fg_color) 
									  ELSE
										TextColor (high_fg_color) ;
									END ;
							  END ;
							Write (item_name[j]) ;
							Inc(j) ; Inc(n) ;
						  END ;
					  END ;
				  END ;
				TextBackground (bg_color) ;
				FOR  i := n TO  x_last_coord  DO
				  Write (' ') ;
			  END ;
		  END ;
		MS_Show ;
		WND_Restore_Cursor ;
END ;


{ ------------------------------------------------------------- }

FUNCTION   MN_check_menu_bar  (menu_bar: Integer; key: Word) : Word ;
VAR
		i:          Integer ;
		mp:         Menu_bar_ptr ;
		left_down:  Boolean ;
		right_down: Boolean ;
		result:     Word ;
		cursor_showing:	Boolean ;

  FUNCTION  do_menus (i: Integer) : Word ;

	FUNCTION  display : Word ;
	VAR
		v :     Word ;
	BEGIN
		   MN_show_menu_bar (menu_bar, 0, i, 0) ;

		   v := MN_show_menu (mp^.entries[i].item_menu_or_value,
						   	  mp^.entries[i].item_y + 1,
						   	  mp^.entries[i].item_x - 1 ) ;

		   MN_show_menu_bar (menu_bar, 0, 0, 0) ;

		   display := v ;
	END ;

  VAR
	   v:       Word ;
  BEGIN
		   v := display ;
		   WHILE  (v = KB_Left)  OR  (v = KB_Right)  DO
			 IF  v = KB_Left  THEN
			   BEGIN
				 Dec(i) ;
				 IF  i = 0  THEN
						i := mp^.n_entries ;
				 IF  mp^.entries[i].item_type = Menu_entry  THEN
						 v := display ;
			   END 
			 ELSE
			   BEGIN
				 Inc(i) ;
				 IF  i > mp^.n_entries THEN
						i := 1 ;
				 IF  mp^.entries[i].item_type = Menu_entry  THEN
						 v := display ;
			   END ;
		   do_menus := v ;
  END ;



  {	The number of the menu bar entry selected or 0 if no entry selected }

  FUNCTION  selected_menu_bar_entry (key: Word) : Integer ;

  VAR
		i:      Integer ;
		found:  Boolean ;
		x, y:   Integer ;
  BEGIN

	  IF  key = 0  THEN
		BEGIN
		  x := MS_WhereX ;
		  y := MS_WhereY ;
		END ;

	  i := 1 ;
	  found := False ;
	  WHILE  (i <= mp^.n_entries)  AND  (NOT found) DO
		WITH  mp^.entries[i]  DO
		  IF  key <> 0  THEN
			BEGIN
			  IF  (item_letter IN ['A'..'Z'])  AND
				  (key = Alt_letter [Ord(item_letter) - Ord('A')] ) THEN
				  found := True 
			  ELSE
				  Inc(i) ;
			END
		  ELSE
			BEGIN
			  IF  MS_RightDown  AND  NOT  right_down  THEN
				BEGIN
				  selected_menu_bar_entry := 0 ;
				  REPEAT UNTIL NOT MS_RightDown ;
				  Exit ;
				END ;
			  IF  MS_LeftDown AND  (y = item_y)  THEN
				BEGIN
				  IF  item_name_length > 1  THEN
					BEGIN
					  IF  (x >= item_x)  AND
						  (x < item_x + item_name_length) THEN
						found := True 
					ELSE
						Inc(i) ;
					END
				  ELSE
					BEGIN
					  IF  (x >= item_x-1)  AND
						  (x < item_x + item_name_length + 1) THEN
						found := True 
					ELSE
						Inc(i) ;
					END
				END
			  ELSE
				  Inc(i) ;
			END ;

	  IF found  THEN
		BEGIN
		  IF  left_down  THEN
		  	IF  mp^.entries[i].item_type = Repeating_Command_entry  THEN
				MS_Delay 
			ELSE
				REPEAT UNTIL NOT MS_LeftDown ;
		END 
	  ELSE
		  i := 0 ;

	  selected_menu_bar_entry := i ;
  END ;


BEGIN
		WND_Save_cursor (False) ;
		left_down :=  MS_LeftDown ;
		right_down := MS_RightDown ;

		IF  (menu_bar > 0)  AND  (menu_bar <= last_menu_bar)  THEN
		  BEGIN
		    mp := menu_bar_table [menu_bar] ;         
			result := 0 ;
			i := selected_menu_bar_entry (key) ;
			WHILE  i <> 0  DO
			  BEGIN
				WITH  mp^.entries[i]  DO
				  IF  item_type = Menu_entry  THEN
					BEGIN
					  result := do_menus (i) ;
					  i := selected_menu_bar_entry (result) ;
					END
				  ELSE  
					BEGIN
					  result := item_menu_or_value ;
					  i := 0 ;
					END ;
			  END ;
			MN_check_menu_bar := result ; 
		  END ;
	   	WND_Restore_cursor;
END ;

{ -----------------------------------------------------------------------
					U N I T    I N I T I A L I Z A T I O N           
  ----------------------------------------------------------------------- }

BEGIN
		last_menu := 0 ;
		last_menu_bar := 0 ;
END.
