typedef enum {FALSE, TRUE} boolean;

/*
A quick note on naming schemes used for constants
For the screen and the field the suffixes used are:
_C = columns
_R = rows
_W = width    (in pixels)
_H = height   (in pixels)
_X = X offset (in pixels)
_Y = Y offset (in pixels)

For the balls the suffixes used are:
_D = diameter (widthwise, in pixels)
_W = width    (the true width, modified by the aspect ratio, in pixels)
_H = height   (the true height, modified by the aspect ratio, in pixels)
*/

//Ascii codes
const int KEY_ESC = 27;

//Screen dimensions
const int SCREEN_W = 320;
const int SCREEN_H = 200;
//Screen segment
const int SCREEN_SEG = 0xA000;

//Circle dimensions
const int BALL_D = 15; //Diameter
const int BALL_W = 15; //BALL_D*(320/320); //Modified due to aspect ratio
const int BALL_H = 12; //BALL_D*(200/240); //Modified due to aspect ratio

//Playing field dimensions
const int FIELD_C = 15;
const int FIELD_R = 16;
const int FIELD_W = FIELD_C*BALL_W;
const int FIELD_H = FIELD_R*BALL_H;
const int FIELD_X = (SCREEN_W - FIELD_W)/2;
const int FIELD_Y = (SCREEN_H - FIELD_H - BALL_H);

//Colors
const int BALL_COLOR       = 1;
const int PLAYER_COLOR     = 3;
const int BORDER_COLOR     = 120;
const int BACKGROUND_COLOR = 0;

//Some gamemplay constants
const int MAX_TIME_IN_AIR = 15; //the amount of time game ticks before the player falls one space
const int WIN_REQUIREMENT =  FIELD_R - 4; //how high the player has to make it to win
const int INITIAL_DELAY = 5;


//Text strings
char TEXT_INTRO[] =
"Welcome to Ballz!\r\n"
"\r\n"
"Created by:\r\n"
"           Shaun Jackman\r\n"
"\r\n"
"The object of this game is to guide your character, (the light blue coloured\r\n"
"little devil), to the very top of the playing field. This would be easy if\r\n"
"it were not for your arch-nemesis GRAVITY! Yes, gravity is out to get you\r\n"
"again. This time he is throwing large dark blue balls at you. Your strategy\r\n"
"should be fairly simple: do not get squished by his balls. Heck, while you're\r\n"
"at it, mock him by using his own balls against him! Climb those balls to reach\r\n"
"your goal at the top of the screen!     Have fun!!!\r\n"
"\r\n"
"There are 4 Difficulty levels.\r\n"
"They are:\r\n"
"  Complete Wuss, Your Average Joe,\r\n"
"  This is Pretty Tough, and You Gotta Be Kidding\r\n"
"The difficulty can be changed at any time with the numbers 0-3 on your keypad.\r\n"
"\r\n"
"Controlling your character, \"Little blue ball\", could not be easier.\r\n"
"Pressing 4 moves \"Little blue ball\" left.\r\n"
"Pressing 6 moves \"Little blue ball\" right.\r\n"
"Pressing 8 makes \"Little blue ball\" jump.\r\n"
"Press escape at any time to quit.\r\n"
"                                            (press any key to start the game)"
"$";

const char TEXT_WON[] =
"Congratulations! You won!\r\n"
"You beat the evil Lord Gravitron down to a bloody pulp!\r\n"
"\r\n"
"$";

const char TEXT_LOST[] =
"Ah... what a shame. You got squished.\r\n"
"Maybe next time you'll do better, eh?\r\n"
"\r\n"
"$";

const char TEXT_QUIT[] =
"How are you ever going to beat the game if you keeep on quitting?\r\n"
"I better see you back here some time soon...\r\n"
"\r\n"
"$";

const char TEXT_SCORE1[] =
"You survived long enough to see "
"$";
const char TEXT_SCORE2[] =
" balls hit the ground.\r\n"
"$";


//Some standard functions

//Randomizes the random number generator with BIOS timer
int randomNumber;  //last random number generated

void randomize() {
  asm {
    mov     ax, 0
    mov     es, ax
    mov     ax, [es:0x046C]
    mov     randomNumber, ax
  }
}

int random(int max) {
  int returnValue;
  asm {
    mov     ax, max
    xchg    bx,ax
    imul    ax,randomNumber, 9421
    inc     ax
    mov     randomNumber, ax
    xor     dx,dx
    div     bx
    mov     returnValue, dx
  }
  return returnValue;
}

int kbhit() {
  int returnValue;
  asm {
    mov ah, 0xb
    int 0x21
    cbw
    mov returnValue, ax
  }
  return returnValue;
}

int getch() {
  int returnValue;
  asm {
    mov ax, 0x0700
    int 0x21
    mov ah, 0
    mov returnValue, ax
  }
  return returnValue;
}

void writeString(const char* s) {
  asm {
    lea dx, s
    mov ah, 9
    int 0x21
  }
}

// Waits for the nth vertical retrace, used as a delay in the main loop
void waitVRetrace(int n) {
  asm mov dx, 0x3DA
  for(int i = 0; i < n; i++) {
    l1:
    asm {
      in al, dx
      and al, 0x08
      jnz l1
    }
    l2:
    asm {
      in al, dx
      and al, 0x08
      jz l2
    }
  }
}


//------------------------------
void vline(int x, int color) {
  asm {
    push es
    mov ax, SCREEN_SEG
    mov es, ax
    mov ax, color
    mov di, x
    mov cx, SCREEN_H
  }
  loop1:
  asm {
    stosb
    add di, SCREEN_W - 1
    loop loop1
    pop es
  }
}


void drawCircle(int X, int Y, short int color) {
  asm {
  push ds
  //Put screen segment in DS
  mov ax, SCREEN_SEG
  mov ds, ax

  //Put screen offset in BX  (BX = DX * 320 + CX)
  mov cx, X
  mov dx, Y
  xor bx, bx
  mov bh, dl
  shl dx, 6
  add bx, dx
  add bx, cx

  mov ax, color

  mov [bx+SCREEN_W* 0+ 4], al
  mov [bx+SCREEN_W* 0+ 5], al
  mov [bx+SCREEN_W* 0+ 6], al
  mov [bx+SCREEN_W* 0+ 7], al
  mov [bx+SCREEN_W* 0+ 8], al
  mov [bx+SCREEN_W* 0+ 9], al
  mov [bx+SCREEN_W* 0+10], al
  mov [bx+SCREEN_W* 1+ 2], al
  mov [bx+SCREEN_W* 1+ 3], al
  mov [bx+SCREEN_W* 1+11], al
  mov [bx+SCREEN_W* 1+12], al
  mov [bx+SCREEN_W* 2+ 1], al
  mov [bx+SCREEN_W* 2+13], al
  mov [bx+SCREEN_W* 3+ 1], al
  mov [bx+SCREEN_W* 3+13], al
  mov [bx+SCREEN_W* 4+ 0], al
  mov [bx+SCREEN_W* 4+14], al
  mov [bx+SCREEN_W* 5+ 0], al
  mov [bx+SCREEN_W* 5+14], al
  mov [bx+SCREEN_W* 6+ 0], al
  mov [bx+SCREEN_W* 6+14], al
  mov [bx+SCREEN_W* 7+ 0], al
  mov [bx+SCREEN_W* 7+14], al
  mov [bx+SCREEN_W* 8+ 1], al
  mov [bx+SCREEN_W* 8+13], al
  mov [bx+SCREEN_W* 9+ 1], al
  mov [bx+SCREEN_W* 9+13], al
  mov [bx+SCREEN_W*10+ 2], al
  mov [bx+SCREEN_W*10+ 3], al
  mov [bx+SCREEN_W*10+11], al
  mov [bx+SCREEN_W*10+12], al
  mov [bx+SCREEN_W*11+ 4], al
  mov [bx+SCREEN_W*11+ 5], al
  mov [bx+SCREEN_W*11+ 6], al
  mov [bx+SCREEN_W*11+ 7], al
  mov [bx+SCREEN_W*11+ 8], al
  mov [bx+SCREEN_W*11+ 9], al
  mov [bx+SCREEN_W*11+10], al
  pop ds
  }
}

//------------------------------
extern "C" void playGame() {
  asm {
    mov ax, 0x03
    int 0x10
    lea dx, TEXT_INTRO
    mov ah, 0x9
    int 0x21
  }
  getch();

  int field[FIELD_C];
  for(int i = 0; i<FIELD_C; i++) {
    field[i] = 0;
  }

  randomize();

  asm {
    mov ax, 0x13
    int 0x10
  }
  vline(FIELD_X - 1, BORDER_COLOR);
  vline(FIELD_X + FIELD_W, BORDER_COLOR);

  int delay = INITIAL_DELAY - 1;

  int nBallsDown = 0;

  int ball_C = random(FIELD_C);
  int ball_R = FIELD_R;
  int ball_X = 0;
  int ball_Y = 0;
  boolean ballDown = FALSE;

  int player_R = 0;
  int player_C = FIELD_C/2;
  int player_X = 0;
  int player_Y = 0;
  int timeInAir = 0;

  boolean dead = FALSE;
  boolean won  = FALSE;
  boolean quit = FALSE;
  while(dead != TRUE && won != TRUE && quit != TRUE) {
    //Wait for screen retrace to delay main loop
    waitVRetrace(delay);

    if(!ballDown) {//if the current ball is still moving
      drawCircle(ball_X, ball_Y, BACKGROUND_COLOR);
    }
    ball_X = ball_C * BALL_W + FIELD_X;
    ball_Y = (FIELD_R - ball_R) * BALL_H + FIELD_Y;
    drawCircle(ball_X, ball_Y, BALL_COLOR);

    drawCircle(player_X, player_Y, BACKGROUND_COLOR);
    player_X = player_C * BALL_W + FIELD_X;
    player_Y = (FIELD_R - player_R) * BALL_H + FIELD_Y;
    drawCircle(player_X, player_Y, PLAYER_COLOR);

    if(kbhit()) {
      char key = getch();
      if(key == 0) {
        getch();
      }
      if(key == '8') {
        if(timeInAir == 0) {// if player's on the ground
          player_R++;
        }
      }
      if(key == '4') {
        if(player_C > 0 && player_R >= field[player_C-1]) {// if there is no ball blocking the player
          player_C--;
        }
      }
      if(key == '6') {
        if(player_C < FIELD_C && player_R >= field[player_C+1]) {// if there is no ball blocking the player
          player_C++;
        }
      }
      if(key >= '0' && key <= '3') {
        delay = INITIAL_DELAY - (key - '0');
      }
      if(key == KEY_ESC) {
        quit = TRUE;
      }
    }

    ballDown = FALSE;
    if(ball_R == field[ball_C]) {
      ballDown = TRUE;
      nBallsDown++;
      field[ball_C]++;
      ball_C = random(FIELD_C);
      ball_R = FIELD_R;
    }
    ball_R--;

    if(player_R > field[player_C]) {//if player is in mid-air
      timeInAir++;
    } else {
      timeInAir = 0;
    }
    if(timeInAir == MAX_TIME_IN_AIR) {//if player is due to fall
      player_R--;
      timeInAir = 0;
    }

    if(player_C == ball_C  &&  player_R == ball_R) {//if the player is currently in the same position as the ball.
      dead = TRUE;
    }
    if(player_R == WIN_REQUIREMENT) {//if player has climbed the screen
      won = TRUE;
    }
  } // while(won != TRUE && dead != TRUE && quit != TRUE)
  asm {
    mov ax, 0x3
    int 0x10
  }
  if(won) {
    asm {
      lea dx, TEXT_WON
      mov ah, 0x9
      int 0x21
    }
  }
  if(dead) {
    asm {
      lea dx, TEXT_LOST
      mov ah, 0x9
      int 0x21
    }
  }
  if(quit) {
    asm {
      lea dx, TEXT_QUIT
      mov ah, 0x9
      int 0x21
    }
  }
  asm {
    lea dx, TEXT_SCORE1
    mov ah, 0x9
    int 0x21
  }
  asm {
    mov ax, nBallsDown
    aam
    mov bl, al
    mov al, ah
    xor ah, ah
    aam
    mov cx, ax
    mov ah, 2
    mov dl, ch
    add dl, '0'
    cmp dl, '0'
    db 0x74, 2 // only print if digit is not zero; jz $+2
    int 0x21
    mov dl, cl
    add dl, '0'
    cmp dl, '0'
    db 0x74, 2 // only print if digit is not zero; jz $+2
    int 0x21
    mov dl, bl
    add dl, '0'
    int 0x21
  }
  asm {
    lea dx, TEXT_SCORE2
    mov ah, 0x9
    int 0x21
  }
  return;
}
