{
    VGADEMO -- VGA graphics demo.  Pixels, lines, circles, ellipses.
    line1 is the original Pascal line procedure.
    line2 is an optimized version with assembly language.
    rectangle is the un-filled rectangle procedure.
    putrow draws a horizontal line using assembly language.
    fillrect1 is the slow filled rectangle procedure.
    fillrect2 is the fast filled rectangle procedure.
    circle1 uses floating point arithmetic.
    circle2 uses integer arithmetic.
    circle3 also uses the circle's symmetry.
    fillcircle1 draws a line across the circle for each pixel.
    fillcircle2 does not draw duplicate lines, so is faster.
    ellipse is the un-filled ellipse procedure.
    fillellipse is the filled ellipse procedure.
}

{$G+}
program vgademo;
uses crt;

{--------------------------------------------------------------------}

procedure setmode(mode:integer); assembler;
asm
    mov ax,mode;
    xor ah,ah;
    int 10h;
end;

{--------------------------------------------------------------------}

procedure cls; assembler;
asm
    mov ax,0A000h;          { ES = video memory }
    mov es,ax;

    xor di,di;              { Set up for clear }
    xor ax,ax;
    mov cx,8000h;

    rep stosw;              { Clear the screen }
end;

{--------------------------------------------------------------------}

procedure putpixel(x, y, color:integer); assembler;
asm
    mov bx,x;               { BX = X, CX = Y }
    mov cx,y;
    cmp bx,320; jae @done;  { Out of range? }
    cmp cx,200; jae @done;

    add bh,cl;              { BX = offset }
    shl cx,6;
    add bx,cx;

    mov ax,0A000h;          { ES = video memory }
    mov es,ax;
    mov ax,color;           { AX = color }
    mov es:[bx],al;         { Plot pixel }
@done:
end;

{--------------------------------------------------------------------}

function getpixel(x, y:integer):integer; assembler;
asm
    mov bx,x;               { BX = X, CX = Y }
    mov cx,y;
    add bh,cl;              { BX = offset }
    shl cx,6;
    add bx,cx;

    mov ax,0A000h;          { ES = video memory }
    mov es,ax;
    mov al,es:[bx];         { Read pixel }
    xor ah,ah;
end;

{--------------------------------------------------------------------}

procedure line1(x1, y1, x2, y2, color:integer);
var
    d, x, y, sx, sy, dx, dy:integer;
begin
    {
        dx, dy = distance, d = deviation,
        sx, sy = sign of distance.
    }

    dx := x2 - x1; dy := y2 - y1;
    if dx > 0 then sx := 1 else sx := -1;
    if dy > 0 then sy := 1 else sy := -1;
    dx := abs(dx); dy := abs(dy);
    x := x1; y := y1;

    if dx > dy then begin
        d := (dy - dx) div 2;       { X is major axis }
        while x <> x2 do begin
            putpixel(x, y, color);  { Put pixel }
            if d >= 0 then begin    { Minor-axis change? }
                dec(d, dx);         { Decrease by 1 }
                inc(y, sy);
            end;
            inc(d, dy);             { Increase by dy/dx }
            inc(x, sx);
        end;
        putpixel(x, y, color);      { Put last pixel }
    end else begin
        d := (dx - dy) div 2;       { Y is major axis }
        while y <> y2 do begin
            putpixel(x, y, color);  { Put pixel }
            if d >= 0 then begin    { Minor-axis change? }
                dec(d, dy);         { Decrease by 1 }
                inc(x, sx);
            end;
            inc(d, dx);             { Increase by dx/dy }
            inc(y, sy);
        end;
        putpixel(x, y, color);      { Put last pixel }
    end;
end;

{--------------------------------------------------------------------}

procedure line2(x1, y1, x2, y2, color:integer);
var
    d, x, y, sx, sy, dx, dy:integer;
begin
    {
        dx, dy = distance, d = deviation,
        sx, sy = sign of distance.
    }

    dx := x2 - x1; dy := y2 - y1;
    if dx > 0 then sx := 1 else sx := -1;
    if dy > 0 then sy := 1 else sy := -1;
    dx := abs(dx); dy := abs(dy);
    x := x1; y := y1;

    asm                             { ES = video memory }
        mov ax,0A000h;
        mov es,ax;
    end;

    if dx > dy then begin
        d := (dy - dx) div 2;       { X is major axis }
        while x <> x2 do begin
            asm
                mov bx,x; mov cx,y; { Plot the pixel }
                cmp bx,320; jae @done;
                cmp cx,200; jae @done;
                add bh,cl; shl cx,6;
                add bx,cx; mov ax,color;
                mov es:[bx],al;
            @done:
            end;
            if d >= 0 then begin    { Minor-axis change? }
                dec(d, dx);         { Decrease by 1 }
                inc(y, sy);
            end;
            inc(d, dy);             { Increase by dy/dx }
            inc(x, sx);
        end;
        asm
            mov bx,x; mov cx,y;     { Plot the last pixel }
            cmp bx,320; jae @done;
            cmp cx,200; jae @done;
            add bh,cl; shl cx,6;
            add bx,cx; mov ax,color;
            mov es:[bx],al;
        @done:
        end;
    end else begin
        d := (dx - dy) div 2;       { Y is major axis }
        while y <> y2 do begin
            asm
                mov bx,x; mov cx,y; { Plot the pixel }
                cmp bx,320; jae @done;
                cmp cx,200; jae @done;
                add bh,cl; shl cx,6;
                add bx,cx; mov ax,color;
                mov es:[bx],al;
            @done:
            end;
            if d >= 0 then begin    { Minor-axis change? }
                dec(d, dy);         { Decrease by 1 }
                inc(x, sx);
            end;
            inc(d, dx);             { Increase by dx/dy }
            inc(y, sy);
        end;
        asm
            mov bx,x; mov cx,y;     { Plot the last pixel }
            cmp bx,320; jae @done;
            cmp cx,200; jae @done;
            add bh,cl; shl cx,6;
            add bx,cx; mov ax,color;
            mov es:[bx],al;
        @done:
        end;
    end;
end;

{--------------------------------------------------------------------}

procedure rectangle(x1, y1, x2, y2, color:integer);
begin
    line2(x1, y1, x2, y1, color);
    line2(x1, y2, x2, y2, color);
    line2(x1, y1, x1, y2, color);
    line2(x2, y1, x2, y1, color);
end;

{--------------------------------------------------------------------}

procedure fillrect1(x1, y1, x2, y2, color:integer);
var
    i:integer;
begin
    if y1 > y2 then begin   { Put Y in order }
        i := y1; y1 := y2; y2 := i;
    end;
    for i := y1 to y2 do line2(x1, i, x2, i, color);
end;

{--------------------------------------------------------------------}

procedure putrow(x1, x2, y, color:integer); assembler;
asm
    mov ax,0A000h;          { ES = video memory }
    mov es,ax;

    mov ax,y;               { Get parameters }
    mov bx,x1;
    mov cx,x2;

    cmp ax,200; jae @done;  { Y out of range? }
    cmp bx,cx; jle @1;      { Put X in order }
    xchg bx,cx;
@1: cmp cx,0; jl @done;     { Not visible? }
    cmp bx,319; jg @done;

    cmp bx,0; jge @2;       { Clip to screen }
    mov bx,0;
@2: cmp cx,319; jle @3;
    mov cx,319;
@3:
    imul di,ax,320;         { DI = offset }
    add di,bx;
    sub cx,bx;              { CX = length }
    inc cx;

    mov ax,color;           { AL, AH = color }
    mov ah,al;

    shr cx,1;               { Store by words }
    rep stosw;
    adc cx,0;               { Store possible odd byte }
    rep stosb;
@done:
end;

{--------------------------------------------------------------------}

procedure fillrect2(x1, y1, x2, y2, color:integer);
var
    i:integer;
begin
    if y1 > y2 then begin   { Put Y in order }
        i := y1; y1 := y2; y2 := i;
    end;

    for i := y1 to y2 do putrow(x1, x2, i, color);
end;

{--------------------------------------------------------------------}

procedure circle1(x, y, r, color:integer);
var
    i:integer;
    ix, iy:real;
begin
    if r < 1 then r := 1;
    ix := 0; iy := r;
    for i := 1 to (r * 44 div 7 {2 Pi}) do begin
        putpixel(trunc(x + ix), trunc(y + iy), color);
        ix := ix + iy / r;    { Step to next pixel }
        iy := iy - ix / r;
    end;
end;

{--------------------------------------------------------------------}

procedure circle2(x, y, r, color:integer);
var
    i, ix, iy:integer;
begin
    if r < 1 then r := 1;
    ix := 0; iy := r * 64;
    for i := 1 to (r * 44 div 7 {2 Pi}) do begin
        putpixel(x + (ix div 64), y + (iy div 64), color);
        ix := ix + iy div r;    { Step to next pixel }
        iy := iy - ix div r;
    end;
end;

{--------------------------------------------------------------------}

procedure circle3(x, y, r, color:integer);
var
    ix, iy, a, b:integer;
begin
    if r < 1 then r := 1;
    ix := 0; iy := r * 64;

    repeat begin
        a := (ix + 32) shr 6; b := (iy + 32) shr 6;

        { plot eight pixels using symmetry }

        putpixel(x + a, y + b, color); putpixel(x - a, y + b, color);
        putpixel(x + a, y - b, color); putpixel(x - a, y - b, color);
        putpixel(x + b, y + a, color); putpixel(x - b, y + a, color);
        putpixel(x + b, y - a, color); putpixel(x - b, y - a, color);

        { step to next pixel }

        ix := ix + iy div r;
        iy := iy - ix div r;
    end until b <= a;
end;

{--------------------------------------------------------------------}

procedure fillcircle1(x, y, r, color:integer);
var
    ix, iy, a, b:integer;
begin
    if r < 1 then r := 1;
    ix := 0; iy := r * 64;

    repeat begin
        a := (ix + 32) shr 6; b := (iy + 32) shr 6;

        { plot four rows using symmetry }

        putrow(x - a, x + a, y + b, color);
        putrow(x - a, x + a, y - b, color);
        putrow(x - b, x + b, y + a, color);
        putrow(x - b, x + b, y - a, color);

        { step to next pixel }

        ix := ix + iy div r;
        iy := iy - ix div r;
    end until b <= a;
end;

{--------------------------------------------------------------------}

procedure fillcircle2(cx, cy, r, color:Integer);
var
    x, y, a, b, na, nb:integer;
begin
    if r < 1 then r := 1;
    x := 0;                     { calculate original pixel position }
    y := r * 64;
    na := (x + 32) shr 6;
    nb := (y + 32) shr 6;

    repeat begin
        a := na; b := nb;       { save old pixel position }

        x := x + (y div r);     { step to next pixel }
        y := y - (x div r);
        na := (x + 32) shr 6;   { calculate new pixel position }
        nb := (y + 32) shr 6;

        { always plot long rows }

        putrow(cx - b, cx + b, cy + a, color);
        putrow(cx - b, cx + b, cy - a, color);

        { plot short rows only if Y changed }

        if b <> nb then begin
            putrow(cx - a, cx + a, cy + b, color);
            putrow(cx - a, cx + a, cy - b, color);
        end;
    end until b <= a;
end;

{--------------------------------------------------------------------}

procedure ellipse(x, y, rx, ry, color:integer);
var
    ix, iy, a, b, c, d:integer;
begin
    if rx < 1 then rx := 1; if ry < 1 then ry := 1;

    if rx > ry then begin
        ix := 0; iy := rx * 64;
        repeat begin
            a := (ix + 32) shr 6; b := (iy + 32) shr 6;
            c := (a * ry) div rx; d := (b * ry) div rx;

            { plot eight pixels using symmetry }
            putpixel(x + a, y + d, color); putpixel(x - a, y + d, color);
            putpixel(x + a, y - d, color); putpixel(x - a, y - d, color);
            putpixel(x + b, y + c, color); putpixel(x - b, y + c, color);
            putpixel(x + b, y - c, color); putpixel(x - b, y - c, color);
            ix := ix + iy div rx;   { step to next pixel }
            iy := iy - ix div rx;
        end until b <= a;
    end else begin
        ix := 0; iy := ry * 64;
        repeat begin
            a := (ix + 32) shr 6; b := (iy + 32) shr 6;
            c := (a * rx) div ry; d := (b * rx) div ry;
    
            { plot eight pixels using symmetry }
            putpixel(x + c, y + b, color); putpixel(x - c, y + b, color);
            putpixel(x + c, y - b, color); putpixel(x - c, y - b, color);
            putpixel(x + d, y + a, color); putpixel(x - d, y + a, color);
            putpixel(x + d, y - a, color); putpixel(x - d, y - a, color);
            ix := ix + iy div ry;   { step to next pixel }
            iy := iy - ix div ry;
        end until b <= a;
    end;
end;

{--------------------------------------------------------------------}

procedure fillellipse(cx, cy, rx, ry, color:Integer);
var
    x, y, a, b, c, d, na, nb, nc, nd:integer;
begin
    if rx < 1 then rx := 1;
    if ry < 1 then ry := 1;

    if rx > ry then begin
        x := 0;  y := rx * 64;         { calculate original pixel position }
        na := 0; nb := (y + 32) shr 6;
        nc := 0; nd := (nb * ry) div rx;
    
        repeat begin
            a := na; b := nb; c := nc; d := nd;  { save old pixel position }
    
            x := x + (y div rx); { step to next pixel & get pixel position }
            y := y - (x div rx);
            na := (x + 32) shr 6;   nb := (y + 32) shr 6;
            nc := (na * ry) div rx; nd := (nb * ry) div rx;
    
            putrow(cx - b, cx + b, cy + c, color); { always plot long rows }
            putrow(cx - b, cx + b, cy - c, color);
            if d <> nd then begin      { plot short rows only if Y changed }
                putrow(cx - a, cx + a, cy + d, color);
                putrow(cx - a, cx + a, cy - d, color);
            end;
        end until b <= a;
    end else begin
        x := 0;  y := ry * 64;         { calculate original pixel position }
        na := 0; nb := (y + 32) shr 6;
        nc := 0; nd := (nb * rx) div ry;

        repeat begin
            a := na; b := nb; c := nc; d := nd;  { save old pixel position }
    
            x := x + (y div ry); { step to next pixel & get pixel position }
            y := y - (x div ry);
            na := (x + 32) shr 6;   nb := (y + 32) shr 6;
            nc := (na * rx) div ry; nd := (nb * rx) div ry;
    
            putrow(cx - d, cx + d, cy + a, color); { always plot long rows }
            putrow(cx - d, cx + d, cy - a, color);
            if b <> nb then begin      { plot short rows only if Y changed }
                putrow(cx - c, cx + c, cy + b, color);
                putrow(cx - c, cx + c, cy - b, color);
            end;
        end until b <= a;
    end;
end;

{--------------------------------------------------------------------}

var
    i:integer;
begin
    setmode($13);

    while not keypressed do begin
        for i := 1 to 100 do
            putpixel(random(320), random(200), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 40 do
            line1(random(320), random(200),
                  random(320), random(200),
                  random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 40 do
            line2(random(320), random(200),
                  random(320), random(200),
                  random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 20 do
            rectangle(random(320), random(200),
                      random(320), random(200),
                      random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 5 do
            fillrect1(random(320), random(200),
                      random(320), random(200),
                      random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 5 do
            fillrect2(random(320), random(200),
                      random(320), random(200),
                      random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 10 do
            circle1(random(320), random(200),
                    random(50), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 10 do
            circle2(random(320), random(200),
                    random(50), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 10 do
            circle3(random(320), random(200),
                    random(50), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 5 do
            fillcircle1(random(320), random(200),
                        random(50), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 5 do
            fillcircle2(random(320), random(200),
                        random(50), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 10 do
            ellipse(random(320), random(200),
                    random(60), random(60), random(256));
    end;

    while keypressed do readkey;
    cls;

    while not keypressed do begin
        for i := 1 to 5 do
            fillellipse(random(320), random(200),
                        random(60), random(60), random(256));
    end;

    while keypressed do readkey;
    cls;

    setmode($03);
end.
