/*+
    Name:	hlwin.c
    Date:	06-Jun-1988
    Author:	Kent J. Quirk
		(c) Copyright 1988 Ziff Communications Co.
    Abstract:	Contains window open/close functions.
    History:	09-Sep-88   kjq     Version 1.00
-*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <graph.h>
#include "hl.h"
#include "hltimes.h"
#include "winmenu.h"

unsigned int scrn_seg = 0xB800;         /* color */
int scrn_width = 160;                   /* bytes */

/**** o p e n _ w i n d o w ****
    Abstract:	Opens a window on the screen, saving the data under it.
    Parameters: The 1-based coordinates of the corners of the window,
		the attribute with which to fill the window.
    Returns:	A pointer to the window object created.
    Comments:	Would be nice to have automatic boxes drawing.
****************************/
WINDOW *open_window(short r1, short c1, short r2, short c2, char attr)
{
    int y, linewid;
    unsigned int buf_off, buf_seg;
    char far *bp;
    char far *blankline;
    struct videoconfig config;
    WINDOW *w;

    _getvideoconfig(&config);
    if ((config.mode == _TEXTMONO) || (config.mode == _HERCMONO))
    {
	scrn_seg = 0xB000;
	if ((attr & 0xF0) == 0)
	    attr = 0x07;
	else
	    attr = 0x70;
    }

    if ((w = malloc(sizeof(WINDOW))) == NULL)
	return(NULL);

    linewid = (c2-c1+1)*2;
    w->bufsize = (r2-r1+1) * linewid;		 /* how big is rectangle? */
    if ((w->scrnbuf = malloc(w->bufsize)) == NULL)  /* alloc a buf for it */
    {
	free(w);
	return(NULL);
    }
    if ((blankline = (char far *)malloc(linewid)) == NULL)
    {
	free(w->scrnbuf);
	free(w);
	return(NULL);
    }

    for (y=0; y<linewid; y+=2)
    {
	blankline[y] = ' ';
	blankline[y+1] = attr;
    }

    w->oldcpos = _gettextposition();	    /* where is cursor now? */
    w->r1 = r1;
    w->c1 = c1;
    w->r2 = r2;
    w->c2 = c2;
    w->attr = attr;
    w->x = 'X'; 			/* bug detector */
    w->cpos.row = w->cpos.col = 1;
    bp = (char far *)(w->scrnbuf);

    for (y=r1-1; y<=r2-1; y++)
    {
	movedata(scrn_seg, y*scrn_width+(c1-1)*2,
	    FP_SEG(bp), FP_OFF(bp), linewid);
	movedata(FP_SEG(blankline), FP_OFF(blankline),
	    scrn_seg, y*scrn_width+(c1-1)*2, linewid);
	bp += linewid;
    }
    free((char near *)blankline);
    return(w);
}

/**** c l o s e _ w i n d o w ****
    Abstract:	Given a window, this deallocates the data structure and
		removes it from the screen, restoring the previous screen
		contents.
    Parameters: The window pointer
    Returns:	Zero.
    Comments:	A check is made for "validity" of the window pointer. This
		prevents spectacular crashes if it's not valid, since if
		it isn't, close_window tries to restore random data to the
		video screen area.
		Unfortunately, we can't print much diagnostic info, because
		we just don't have any available.
****************************/
int close_window(WINDOW *w)
{
    int y, linewid;
    char far *bp;

    if (w->x != 'X')
    {
	_clearscreen(_GCLEARSCREEN);
	printf("Internal error in close_window(%p).\n", w);
	printf("Press any key.\n");
	exit(1);
    }
    linewid = (w->c2 - w->c1 + 1)*2;
    bp = (char far *)w->scrnbuf;

    for (y=w->r1-1; y <= w->r2-1; y++)
    {
	movedata(FP_SEG(bp), FP_OFF(bp),
	    scrn_seg, y*scrn_width + (w->c1 - 1) * 2, linewid);
	bp += linewid;
    }
    _settextposition(w->oldcpos.row, w->oldcpos.col);
    free(w->scrnbuf);
    free(w);
    return(0);
}

/**** d e a c t i v a t e _ w i n d o w ****
    Abstract:	Given a window, this remembers the current attribute and
		cursor position and sets those values.
    Parameters: The window pointer.
    Returns:	nothing
    Comments:	Not useful in an overlaid environment, but tiled windows
		work fine with this.
****************************/
void deactivate_window(WINDOW *w)
{
    w->attr = 0;
    w->attr |= 0x0F & _gettextcolor();
    w->attr |= 0xF0 & ((int)_getbkcolor() << 4);
    w->cpos = _gettextposition();
}

/**** a c t i v a t e _ w i n d o w ****
    Abstract:	Restores the cursor and attributes created by the
		deactivate_window() function.
    Parameters: The window
    Returns:	Nothing
****************************/
void activate_window(WINDOW *w)
{
    _settextwindow(w->r1, w->c1, w->r2, w->c2);
    _settextcolor(w->attr & 0x0F);
    _setbkcolor((long)((w->attr & 0xF0) >> 4));
    _settextposition(w->cpos.row, w->cpos.col);
    _wrapon(_GWRAPOFF);
}

/**** f i l l _ s c r e e n ****
    Abstract:	Given an open file containing a screen image, this copies
		the file data to the screen.
    Parameters: The file pointer
    Returns:	Nothing
****************************/
void fill_screen(FILE *f)
{
    char *buf;
    char far *bp;
    struct videoconfig config;
    int size;
    /* struct SREGS segs; */

    _getvideoconfig(&config);
    size = config.numtextrows * config.numtextcols * 2;
    /* segread(&segs); */
    if ((buf = malloc(size)) == NULL)
	return;
    
    fread(buf, size, 1, f);
    bp = (char far *)buf;
    movedata(FP_SEG(bp), FP_OFF(bp), scrn_seg, 0, size);
    free(buf);
}

/**** b i o s _ g e t c u r ****
    Abstract:	Gets the current cursor position from bios.
    Parameters: Nothing
    Returns:	An rccoord struct with the cursor position set.
****************************/
struct rccoord bios_getcur()
{
    struct rccoord cp;
    union REGS regs;

    regs.h.ah = 3;
    regs.h.bh = 0;
    int86(0x10, &regs, &regs);
    cp.row = regs.h.dh;
    cp.col = regs.h.dl;
    return(cp);
}

/**** b i o s _ s e t c u r ****
    Abstract:	Sets the cursor position using bios.
    Parameters: The row and column cursor positions
    Returns:	The previous cursor position as found by bios_getcur().
****************************/
struct rccoord bios_setcur(short r, short c)
{
    struct rccoord cp;
    union REGS regs;

    cp = bios_getcur();
    regs.h.ah = 2;
    regs.h.bh = 0;
    regs.h.dh = (unsigned char)r;
    regs.h.dl = (unsigned char)c;
    int86(0x10, &regs, &regs);
    return(cp);
}
