;
;	TIMER.ASM
;
;		Low level assembly transmit and receive routines.
;
;	July, 1989
;	Andrew C. Payne
;
;	This module implements:
;
;               timer()	      - returns the current timer state
;		transit(time) - waits until time given, and causes and output
;				transition
;		waittrans(til)  Waits until the time given for a transition,
;				returns the time of the transition
;		waituntil(t)	Waits until time given
;
;	Modifications:
;		090589 - added 'waittrans' (acp)
;

	DOSSEG
	.MODEL  LARGE		; 'C' Large Memory Model

; ----- Equates

TIMER1	EQU	040H		; base I/O Port for timer
TIMER2	EQU	TIMER1+3	; port to latch count

; ----- Externals
	GLOBAL	_TXPort:word,_TXBit:byte
	GLOBAL	_RXPort:word,_RXBit:byte

; ----- Macros

READTIME	MACRO		; reads BIOS timer into AX

	MOV	AL,0		; latch count
	OUT	TIMER2,AL

	IN	AL,TIMER1	; read LSB
	XCHG	AL,AH		; save
	IN	AL,TIMER1	; read MSB
	XCHG	AL,AH		; make order correct
	ENDM

; ----- Code Segment
	.CODE

; timer()
;	Returns current value of Counter #0
;
	PUBLIC _timer
_timer	PROC
	READTIME		; read the current time into AX
	RET
_timer	ENDP

; waituntil(time)
;
;	Waits until the time given.
;
	PUBLIC	_waituntil
_waituntil PROC
	ARG	EndTime:WORD

	PUSH	BP
	MOV	BP,SP		; set up stack frame

; wait till ending time
	MOV	BX,[EndTime]	; get into a register

wuloop:
	READTIME		; get current time
	SUB	AX,BX
	JNS	tloop		; wait until passed

	POP	BP		; restore stack frame
	RET

_waituntil ENDP


;
; transit(at)
;	Causes an output TX data transition to occur at the time given
;
	PUBLIC	_transit
_transit PROC
	ARG	EndTime:WORD

	PUSH	BP
	MOV	BP,SP		; set up stack frame

; wait till ending time
	MOV	BX,[EndTime]	; get into a register

tloop:
	READTIME		; get current time
	SUB	AX,BX
	JNS	tloop		; wait until passed

; make transition
	MOV	DX,[_TXPort]
	IN	AL,DX		; get current port value
	XOR	AL,[_TXBit]	; invert TX data bit
	OUT	DX,AL		; write it out

	READTIME		; read time of transition

	POP	BP		; restore stack frame
	RET

_transit ENDP

;
; waittrans(till)
;	Waits until the time given for a transition, returns the time of the
;	transition.  The time return will equal 'till' if no tranisition
;	occurred.
;
;

	PUBLIC	_waittrans
_waittrans PROC
	ARG	EndTime:WORD

	PUSH	BP
	MOV	BP,SP		; set up stack frame

	MOV	BX,[EndTime]	; get ending time into a register

	MOV	DX,[_RXPort]	; get input port
	IN	AL,DX		; read current value
	MOV	CL,AL		; save

waitt:
	IN	AL,DX		; read current value
	XOR	AL,CL		; compare bits
	AND	AL,[_RXBit]	; mask the RX data bit
	JNZ	waitt1		; got transition, return time

; read current time, check to see if we are done yet or not
	READTIME
	SUB	AX,BX
	JNS	waitt		; loop until time out

; time out, got no transition
	MOV	AX,BX		; return ending time parameter
	JMP	waitt2		; all done

; got a transition, return the current time
waitt1:
	READTIME
waitt2:
	POP	BP
	RET

_waittrans ENDP
	END