/***************************************************************************
 *     DISKTEST vers 1.0, DOS                                              *
 *     Checks EMS to EMS-ramdisk writes                                    *
 *     12/30/89                                                            *
 *     by James W. Birdsall                                                *
 *                                                                         *
 *     compiles under Turbo C 2.0 USING THE COMPACT MODEL!                 *
 *                                                                         *
 *   Usage: DISKTEST drive                                                 *
 *   where drive is a single letter corresponding to a valid disk drive.   *
 *   This program is used to test EMS ramdisks. It attempts to write       *
 *   directly from the EMS page frames to the ramdisk and then checks the  *
 *   file written. Many EMS ramdisks do not check the address of the       *
 *   origin and will conflict with one or more pages.                      *
 *                                                                         *
 ***************************************************************************/


#include <stdio.h>
#include <dos.h>
#include <io.h>
#include <alloc.h>
#include <signal.h>



#define VERSION 1.0
#define COLUMNS 5               /* columns to use printing frame pointers */



char copyright[] = "Copyright (c) 1989 James W. Birdsall. All Rights Reserved";



int checkEMM(void);
void releaseEMS(void);
void restoreEMS(void);
int hardhandler(int, int, int, int);
void breakhandler(void);



int handle;                     /* EMS handle */
int vers;                       /* EMS version */
void *mapbuffer;                /* page map buffer for 4.0 */



/***************************************************************************
 *     FUNCTION MAIN                                                       *
 ***************************************************************************/
main(int argc, char *argv[])
{
   union REGS r;
   struct SREGS s;
   int vers, frames;
   unsigned int *junkbuffer;
   void **framebuffer;
   int t1, t3;
   char filename[] = "c:disktest.tst";
   FILE *testfile;
   unsigned char *walker;
   long t2;
   long trigger1, trigger2;


   /* check for argument */
   if (argc == 1) {
      printf("Disk letter required to execute tests.\n");
      exit(5);
      }
   /* set up test filename using disk letter */
   filename[0] = argv[1][0];
   /* check for EMM */
   if (checkEMM() == 0) {
      printf("EMS manager not present or malfunctioning.\n");
      exit(3);
      }
   /* install hardware-error and ctrl-brk handlers */
   harderr(hardhandler);
   signal(SIGINT, breakhandler);

   printf("DISKTEST vers %.1f by James W. Birdsall\n", VERSION);
   printf("   EMS to disk write tests on disk %c: with ", argv[1][0]);

   /* get version */
   r.h.ah = 0x46;
   int86(0x67, &r, &r);
   if (r.h.ah != 0) {
      printf("\nError 0x%x talking to EMS manager.\n", r.h.ah);
      exit(3);
      }
   if ((r.h.al >> 4) < 3) {
      printf("\nDISKTEST requires LIM EMS version 3.0 or better.\n");
      exit(2);
      }
   printf("LIM EMS version %d.%d\n", r.h.al >> 4, r.h.al & 0x0F);
   /* set up vers: 1 if 4.0 or better, 0 if not */
   if ((r.h.al >> 4) >= 4) {
      vers = 1;
      }
     else {
      vers = 0;
      }

   /* get/calculate frame addresses */
   if (vers == 0) {
      frames = 4;
      /* 3.0/3.2 get page frame address */
      r.h.ah = 0x41;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      /* synthesize buffer in style of 4.0, calculate addresses of */
      /* higher frames */
      if ((junkbuffer = (unsigned int *) farcalloc(frames,
                                      2 * sizeof(unsigned int))) == NULL) {
         printf("Error allocating memory.\n");
         exit(4);
         }
      junkbuffer[0] = r.x.bx;
      junkbuffer[2] = r.x.bx + 0x400;
      junkbuffer[4] = r.x.bx + 0x800;
      junkbuffer[6] = r.x.bx + 0xC00;
      }
     else {
      /* 4.0 get number of mappable pages == number of frames */
      r.x.ax = 0x5801;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      frames = r.x.cx;
      /* allocate buffer */
      if ((junkbuffer = (unsigned int *) farcalloc(frames,
                                      2 * sizeof(unsigned int))) == NULL) {
         printf("Error allocating memory.\n");
         exit(4);
         }
      /* and fill: 4.0 get addresses of mappable pages */
      r.x.ax = 0x5800;
      r.x.di = (unsigned int) FP_OFF(junkbuffer);
      s.es = (unsigned int) FP_SEG(junkbuffer);
      int86x(0x67, &r, &r, &s);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      }

   /* print number of frames */
   printf("%d frames at addresses:\n", frames);

   /* create array of far pointers to frames */
   if ((framebuffer = (void **) farcalloc(frames, sizeof(void *))) == NULL) {
      printf("Error allocating memory.\n");
      exit(4);
      }
   /* and print */
   for(t1 = 0; t1 < frames; t1++) {
      framebuffer[t1] = MK_FP(junkbuffer[t1*2], 0);
      printf("%Fp      ", framebuffer[t1]);
      if (((t1 + 1) % COLUMNS) == 0) {
         printf("\n");
         }
      }
   if (t1 % COLUMNS) {
      printf("\n");
      }
   /* get rid of original buffer */
   free(junkbuffer);

   /* get number of free pages */
   r.h.ah = 0x42;
   int86(0x67, &r, &r);
   if (r.h.ah != 0) {
      printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
      exit(3);
      }
   /* check if enough to fill all frames */
   if (r.x.bx < frames) {
      printf("Insufficient free pages for test.\n");
      exit(5);
      }

   /* allocate as many pages as frames */
   r.h.ah = 0x43;
   r.x.bx = frames;
   int86(0x67, &r, &r);
   if (r.h.ah != 0) {
      printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
      exit(3);
      }
   handle = r.x.dx;
   /* install exit handler to release pages upon exit */
   atexit(releaseEMS);

   /* save page map by version */
   if (vers == 0) {
      /* 3.0/3.2 save page map */
      r.h.ah = 0x47;
      r.x.dx = handle;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      }
     else {
      /* 4.0 get size of page-map information */
      r.x.ax = 0x4E03;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      /* allocate page-map buffer */
      if ((mapbuffer = (void *) farmalloc(r.h.al)) == NULL) {
         printf("Error allocating memory.\n");
         exit(4);
         }
      /* 4.0 save page map */
      r.x.ax = 0x4E00;
      r.x.di = (unsigned int) FP_OFF(mapbuffer);
      s.es = (unsigned int) FP_SEG(mapbuffer);
      int86x(0x67, &r, &r, &s);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      }
   /* install exit handler to restore page map upon exit */
   atexit(restoreEMS);

   /* fill each page with unique pattern */
   printf("Filling page ");
   for(t1 = 0; t1 < frames; t1++) {
      /* map logical page into frame 0 */
      r.h.ah = 0x44;
      r.h.al = 0;
      r.x.bx = t1;
      r.x.dx = handle;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      printf("%d ", t1);
      /* fill with pattern */
      walker = (unsigned char *) framebuffer[0];
      for(t2 = 0; FP_OFF(walker) < 16384; walker++, t2 += (t1 + 1)) {
         *walker = (t2 % 256);
         }
      }
   printf("\n");

   /* remap to fill all frames */
   printf("Pages filled, remapping and checking for accuracy.\n");
   for(t1 = 0; t1 < frames; t1++) {
      /* map logical page into frame with same number */
      r.h.ah = 0x44;
      r.h.al = t1;
      r.x.bx = t1;
      r.x.dx = handle;
      int86(0x67, &r, &r);
      if (r.h.ah != 0) {
         printf("Error 0x%x talking to EMS manager.\n", r.h.ah);
         exit(3);
         }
      }
   /* and check each remapped page */
   printf("Checking page ");
   for(t1 = 0; t1 < frames; t1++) {
      printf("%d ", t1);
      walker = (unsigned char *) framebuffer[t1];
      for(t2 = 0; FP_OFF(walker) < 16384; walker++, t2 += (t1 + 1)) {
         if (*walker != (t2 % 256)) {
            printf("Error encountered in page %d, offset %d!\n", t1,
                                                          FP_OFF(walker));
            exit(6);
            }
         }
      }
   printf("\nAll pages checked out. Starting individual write tests.\n");

   /* write each page/frame to file, then read back and check */
   for(t1 = 0; t1 < frames; t1++) {
      printf("Testing %d: Writing. ", t1);
      /* open test file */
      if ((testfile = fopen(filename, "wb")) == NULL) {
         printf("Error opening file\n");
         exit(7);
         }
      /* write from EMS frame */
      if (fwrite(framebuffer[t1], sizeof(char), 16384, testfile) !=
                                                                 16384) {
         printf("Disk write error!\n");
         exit(7);
         }
      /* close file */
      fclose(testfile);
      printf("Reading. ");
      /* reopen for read */
      if ((testfile = fopen(filename, "rb")) == NULL) {
         printf("Error opening file\n");
         exit(7);
         }
      /* check contents of file and contents of EMS frame (which may */
      /* be being used by RAMdisk!                                   */
      walker = (unsigned char *) framebuffer[t1];
      trigger1 = trigger2 = 0;
      for(t2 = 0; FP_OFF(walker) < 16384; walker++, t2 += (t1 + 1)) {
         if ((t3 = fgetc(testfile)) != (t2 % 256)) {
            if (trigger1 == 0) {
               printf("Pattern violation in file.\n");
               trigger1 = 1;
               }
            }
         if (*walker != t3) {
            if (trigger2 == 0) {
               printf("Pattern violation in memory.");
               trigger2 = 1;
               }
            }
         }
      /* if get through reads OK, print */
      if ((trigger1 == 0) && (trigger2 == 0)) {
         printf("Read OK.");
         }
      printf("\n");
      /* close and erase file */
      fclose(testfile);
      unlink(filename);
      }
   printf("Tests completed.\n");

   exit(0);
} /* end of function main */



/***************************************************************************
 *     FUNCTION CHECKEMM                                                   *
 *   Checks for function EMM. 1 = OK, 0 = error                            *
 ***************************************************************************/
int checkEMM(void)
{
   FILE *tempfile;
   int temphandle;

   /* check if EMM is present */
   if ((tempfile = fopen("EMMXXXX0","rb")) == NULL)
      return 0;
   /* make sure we've found a device and not a file */
   temphandle = fileno(tempfile);
   if ((ioctl(temphandle, 0) & 0x80) != 0x80)
      return 0;
   /* check status of EMM */
   if (ioctl(temphandle, 7) <= 0)
      return 0;
   /* close and return */
   fclose(tempfile);
   return 1;
} /* end of function checkEMM */



/***************************************************************************
 *     FUNCTION RELEASEEMS                                                 *
 *   Exit handler, releases allocated pages upon exit                      *
 ***************************************************************************/
void releaseEMS(void)
{
   union REGS r;

   /* release pages */
   r.h.ah = 0x45;
   r.x.dx = handle;
   int86(0x67, &r, &r);
   return;
} /* end of function releaseEMS */



/***************************************************************************
 *     FUNCTION RESTOREEMS                                                 *
 *   Exit handler, restores page map and frees map buffer upon exit        *
 ***************************************************************************/
void restoreEMS(void)
{
   union REGS r;

   /* restore page map by version */
   if (vers == 0) {
      /* 3.0/3.2 restore page map */
      r.h.ah = 0x48;
      r.x.dx = handle;
      int86(0x67, &r, &r);
      }
     else {
      /* 4.0 restore page map */
      r.x.ax = 0x4E01;
      r.x.si = (unsigned int) mapbuffer;
      int86(0x67, &r, &r);
      /* free page map buffer */
      free(mapbuffer);
      }
   return;
} /* end of function restoreEMS */



/***************************************************************************
 *     FUNCTION HARDHANDLER                                                *
 *   Error handler. Calls restoreEMS() and releaseEMS() upon fatal disk    *
 *   hardware error.                                                       *
 ***************************************************************************/
int hardhandler(int errval, int ax, int bp, int si)
{
   /* clean up EMS situation */
   restoreEMS();
   releaseEMS();

   /* DOS call to print */
   bdosptr(0x09, "\nDISK ERROR!\n$", 0);

   /* return code for ABORT */
   return(2);
} /* end of function hardhandler */



/***************************************************************************
 *     FUNCTION BREAKHANDLER                                               *
 *   Ctrl-break handler. Exits immediately, which automatically calls      *
 *   restoreEMS() and releaseEMS()                                         *
 ***************************************************************************/
void breakhandler(void)
{
   exit(10);
} /* end of function breakhandler */


/* end of file DISKTEST.C */