/**
 * \file test_axifilter.cpp
 * \author Sorin SUCIU <sorin.suciu@gecadtech.com>
 * \date 12-07-2005
 *
 * Copyright &copy; 2005 GeCAD Technologies SRL. All Rights Reserved.
 * \brief a test binary for the external filters library
 */

#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <arpa/inet.h>
#include "lib_filters.h"

#define MAXCONN 5

class MyAxiFilter: public AxigenFilter 
{
public:
	MyAxiFilter(const char *filterName, int descriptor,
				const char *queuePath):AxigenFilter(filterName, descriptor, queuePath) { }
	
	virtual void error(ReturnTypes errorCode);
	virtual int process(const char *filename, const char *rpath, stringArray *rcpts,
						bool allowclean);
	//the warning function is not called in this version
	virtual void warning(ReturnTypes warningCode) { }
	//in this example we have nothing to do in complete
	virtual int complete() { return 0; }

	bool needcl;
};

class ServerSocket
{
public:
	ServerSocket();
	~ServerSocket();
	
	int create(char *addr, int port);
	int accept(ServerSocket &socket);
	
private:
	int descriptor;
	sockaddr_in address;	
};

ServerSocket::ServerSocket() :descriptor(-1)
{
	memset(&address,0,sizeof(address));
}
ServerSocket::~ServerSocket()
{
  	if (descriptor != -1)  ::close(descriptor);
}
int ServerSocket::create(char *addr, int port)
{
 	descriptor = socket (AF_INET, SOCK_STREAM, 0);
	
	if (descriptor == -1) return -1;
	
	int on = 1;
  	if (setsockopt(descriptor, SOL_SOCKET, SO_REUSEADDR,
  				 (const char*) &on, sizeof (on)) == -1) return -2;
  				 
  	address.sin_family = AF_INET;
  	if (0 == inet_pton(AF_INET, addr, &address.sin_addr)) return -3;
  	address.sin_port = htons(port);

  	int res = ::bind (descriptor, (struct sockaddr *) &address, sizeof (address));
  	if (res == -1) return -4;
  	
  	res = ::listen (descriptor, MAXCONN);
  	if (res == -1) return -5;

	return 0;
}
int ServerSocket::accept(ServerSocket& socket )
{
  int len = sizeof (address);
  socket.descriptor = ::accept (descriptor, (sockaddr *) &address, (socklen_t *) &len);

  return socket.descriptor;
}

void MyAxiFilter::error(ReturnTypes errorCode)
{
	const char *message;
	switch (errorCode) {
		case RT_INVALID_STATE: message = "RT_INVALID_STATE"; break;
		case RT_COMPLETE_ERROR: message = "RT_COMPLETE_ERROR"; break;
		case RT_PROCESSING_ERROR: message = "RT_PROCESSING_ERROR"; break;
		case RT_INVALID_ARGUMENTS: message = "RT_INVALID_ARGUMENTS"; break;
		case RT_UNKNOWN_ERROR: message = "RT_UNKNOWN_ERROR"; break;
		case RT_ERROR_QUEUE: message = "RT_ERROR_QUEUE"; break;
		case RT_ERROR_INTERNAL: message = "RT_ERROR_INTERNAL"; break;
		case RT_ERROR_PROTOCOL: message = "RT_ERROR_PROTOCOL"; break;
		case RT_ERROR_SOCKET_WRITE: message = "RT_ERROR_SOCKET_WRITE"; break;
		case RT_ERROR_SOCKET_READ: message = "RT_ERROR_SOCKET_READ"; break;
		case RT_ERROR_CONN_CLOSED: message = "RT_ERROR_CONN_CLOSED"; break;
		default: message = "Unknown return code"; break;
	}
	printf("TESTLIB [Error]: code: %d, msg: %s\n",errorCode,message);		 
}

int MyAxiFilter::process(const char *filename, const char *rpath, stringArray *rcpts,
	 					bool allowclean)
{
	if (!allowclean) {
		needcl = true;
		return 0;
	}
	needcl = false;
	int len = strlen(filename);
	char tmpfilename[len+4];
	sprintf(tmpfilename,"%stmp",filename);
	
	FILE *fin = fopen(filename, "rb");
	if (NULL == fin) {
		printf("TESTLIB [LibProcess]: Unable to open mail file\n");
		return -1;
	}
	
	FILE *fout = fopen(tmpfilename,"wb");
	if (NULL == fout) {
		printf("TESTLIB [LibProcess]: Unable to create temporary mail file\n");
		fclose(fin);
		return -1;
	}
	char buf[8193];
	fwrite("Test-Lib: This is just a test\n",1,30,fout);
	if (ferror(fout)) {
		printf("TESTLIB [LibProcess]: Unable to write temporary mail file\n");
		fclose(fin);
		fclose(fout);
		return -1;
	}
	int rval = 0;
	while (!ferror(fin) && !ferror(fout) && !feof(fin)) {
		rval = fread(buf,1,8192,fin);
		if (!ferror(fin)) {
			fwrite(buf,1,rval,fout);
		}
	}
	if (ferror(fin) || ferror(fout)) {
		printf("TESTLIB [LibProcess]: Unable to write temporary mail file\n");
		fclose(fin);
		fclose(fout);
		return -1;
	}
	fclose(fin);
	fclose(fout);
	rval = remove(filename);
	if (rval < 0) {
		printf("TESTLIB [LibProcess]: Unable to delete mail file\n");
		return -1;
	}
	rval = rename(tmpfilename, filename);
	if (rval < 0) {
		printf("TESTLIB [LibProcess]: Unable to rename temporary mail file\n");
		return -1;
	}
	printf("TESTLIB [LibProcess]: Processing completed\n");
	return 0;
}

int run = 1;

void sigt(int s) {
    run = 0;
}

int main( int argc, int argv[] ) 
{
	MyAxiFilter::init();
	const char *address = "127.0.0.1";
	int port			= 7850;
	const char *quepath = "/tmp/testqueue";
	
	signal(SIGTERM, sigt);
	signal(SIGPIPE, SIG_IGN);
	

	ServerSocket *sock = new ServerSocket;
	int rval = sock->create((char *)address,port);
	if (rval != 0) {
		printf("Could not create and bind socket on address: %s | %d\n",address,rval);
		delete sock;
		return -1;
	}
	while (run) {
		ServerSocket *connSock = new ServerSocket;
		int descriptor = sock->accept(*connSock);
		if (descriptor < 0) {
			printf("Could not create connection: %s\n",address);
			delete connSock;
			run = 0;
			continue;
		}
		MyAxiFilter *axiFilter = new MyAxiFilter("TEST",descriptor,"/tmp/testqueue");
        if (axiFilter->initialize() == -1) {
            printf("The parameters passed to MyAxiFilter constructor were invalid");
            delete axiFilter;
            delete connSock;
            run = 0;
            continue;
        }            
		bool read = true;
		while (read) {
			MyAxiFilter::ReturnTypes res = axiFilter->read();
			MyAxiFilter::virus_spam test;
			test.spamLevel=3;
			int cres;
			switch (res) {
				case MyAxiFilter::RT_READ_TIMEOUT: 
					break;
				case MyAxiFilter::RT_WRITE_TIMEOUT:
					break; 
				case MyAxiFilter::RT_PROCESSING_COMPLETE:
					if (axiFilter->needcl) axiFilter->writeNeedsCleaning();
					else axiFilter->writeCleannedOk("TEST FILTER: Cleanned OK",test);
					break;
				default:
					cres=axiFilter->clean();
					if (cres != MyAxiFilter::RT_CLEAN_COMPLETE) read = false;
			}
		}	
		delete axiFilter;
		delete connSock;
	}
	
	delete sock;
	MyAxiFilter::cleanup();
	return 0;	
}
