/* $ vesa.h 29/06/99 21:24 $
 * VESA VBE functions. Revision 1.0
 *
 * Copyright (C) 1999  Dmitry Uvarov <mit@pc2o205a.pnpi.spb.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef __VESA_H__
#define __VESA_H__

#include "bool.h"
#include "vesadef.def"
/* enable structures packing */
#pragma pack(1)
/***************************************************************************/
/*                      VESA information block                             */
/***************************************************************************/
typedef struct {  // VBE_vbeInfo;
        char    VbeSignature[4];        /* 'VESA' 4 byte signature          */
        word    VbeVersion;             /* VBE version number in BCD        */
        char   *OemStringPtr;           /* Pointer to OEM string            */
        dword   Capabilities;           /* Capabilities of video card       */
        word   *VideoModePtr;           /* Pointer to supported modes list  */
        word    TotalMemory;            /* Number of 64kb memory blocks     */

        /* VBE 2.0+ extended info */
        word    OemSoftwareRev;         /* OEM Software revision number     */
        char   *OemVendorNamePtr;       /* Pointer to Vendor Name string    */
        char   *OemProductNamePtr;      /* Pointer to Product Name string   */
        char   *OemProductRevPtr;       /* Pointer to Product Revision str  */
        byte    reserved[222];          /* Pad to 256 byte block size       */
        byte    OemDATA[256];           /* Pad for OEM data                 */
        } VBE_vbeInfo;

#define vbeVESA20_SIG           '2EBV'  /* Signature to force VBE supply VESA2.0 extended info */
#define vbeVESA_SIG             'ASEV'  /* Default vesa signature */
#define vbeCorePM_SIG           'DIMP'  /* Signature of VBE/Core PM block   */
/************************************/
/* Flags for the Capabilities field */
/************************************/
#define vbe8BitDAC              0x0001  /* DAC width is switchable to 8 bit */
#define vbeNonVGA               0x0002  /* Controller is not-VGA compatible */
#define vbeBlankRAMDAC          0x0004  /* Programmed DAC with blank bit    */
/* VESA 3.0 extended info */
#define vbeHWStereoSync         0x0008  /* Hardware stereo signalling       */
#define vbeEVCStereoSync        0x0010  /* HW stereo sync via EVC connector */

/***************************************************************************/
/*                      VESA mode information block                        */
/***************************************************************************/
typedef struct {   // VBE_modeInfo;
        word    ModeAttributes;         /* Mode attributes                         */
        byte    WinAAttributes;         /* Window A attributes                     */
        byte    WinBAttributes;         /* Window B attributes                     */
        word    WinGranularity;         /* Window granularity in kb                */
        word    WinSize;                /* Window size in kb                       */
        word    WinASegment;            /* Window A start segment                  */
        word    WinBSegment;            /* Window B start segment                  */
        void   *WinFuncPtr;             /* Pointer to window function (bank swich) */
        word    BytesPerScanLine;       /* Bytes per scanline                      */
        word    XResolution;            /* Horizontal resolution                   */
        word    YResolution;            /* Vertical resolution                     */
        byte    XCharSize;              /* Character cell width in pixels          */
        byte    YCharSize;              /* Character cell height in pixels         */
        byte    NumberOfPlanes;         /* Number of memory planes                 */
        byte    BitsPerPixel;           /* Bits per pixel                          */
        byte    NumberOfBanks;          /* Number of banks                         */
        byte    MemoryModel;            /* Memory model type                       */
        byte    BankSize;               /* Bank size in Kb                         */
        byte    NumberOfImagePages;     /* Number of images pages (Max)            */
        byte    res1;                   /* Reserved for page function              */
        byte    RedMaskSize;            /* Size of direct color red mask           */
        byte    RedFieldPosition;       /* Bit posn of lsb of red mask             */
        byte    GreenMaskSize;          /* Size of direct color green mask         */
        byte    GreenFieldPosition;     /* Bit posn of lsb of green mask           */
        byte    BlueMaskSize;           /* Size of direct color blue mask          */
        byte    BlueFieldPosition;      /* Bit posn of lsb of blue mask            */
        byte    RsvdMaskSize;           /* Size of direct color res mask           */
        byte    RsvdFieldPosition;      /* Bit posn of lsb of res mask             */
        byte    DirectColorModeInfo;    /* Direct color mode attributes            */

        /* VBE 2.0 extension information */
        dword   PhysBasePtr;            /* Physical base address for flar linear frame */
        dword   OffScreenMemOffset;     /* Pointer to start of off screen memory       */
        word    OffScreenMemSize;       /* Size of off screen memory in Kb             */

        /* VBE 3.0 extension information */
        word    LinBytesPerScanLine;    /* Bytes per scanline               */
        byte    BnkNumberOfImagePages;  /* Number of images pages (banked)  */
        byte    LinNumberOfImagePages;  /* Number of images pages (linear)  */
        byte    LinRedMaskSize;         /* Size of direct color red mask    */
        byte    LinRedFieldPosition;    /* Bit posn of lsb of red mask      */
        byte    LinGreenMaskSize;       /* Size of direct color green mask  */
        byte    LinGreenFieldPosition;  /* Bit posn of lsb of green mask    */
        byte    LinBlueMaskSize;        /* Size of direct color blue mask   */
        byte    LinBlueFieldPosition;   /* Bit posn of lsb of blue mask     */
        byte    LinRsvdMaskSize;        /* Size of direct color res mask    */
        byte    LinRsvdFieldPosition;   /* Bit posn of lsb of res mask      */
        dword   MaxPixelClock;          /* Maximum pixel clock              */

        char    res2[190];              /* Pad to 256 byte block size       */
        } VBE_modeInfo;
/**********************************/
/* Flags for ModeAttributes field */
/**********************************/
#define vbeModeAvailable        0x0001          /* Video mode is available           */
#define vbeModeTTYOutput        0x0004          /* TTY BIOS output is supported      */
#define vbeModeColorMode        0x0008          /* Mode is a color video mode        */
#define vbeModeGraphMode        0x0010          /* Mode is a graphics mode           */
#define vbeModeNonVGA           0x0020          /* Mode is not VGA compatible        */
#define vbeModeNonBanked        0x0040          /* Banked mode is not supported      */
#define vbeModeLinear           0x0080          /* Linear mode supported             */
/* VESA 3.0 extended flags */
#define vbeModeDoubleScan       0x0100          /* Mode is double scanned            */
#define vbeModeInterlaced       0x0200          /* Supports interlaced timings       */
#define vbeModeTripleBuf        0x0400          /* Mode supports HW triple buffer    */
#define vbeModeStereo           0x0800          /* Mode supports hardware stereo     */
#define vbeModeDualDisp         0x1000          /* Mode supports dual display start  */
/* Additional definitions */
#define vbeModeMemoryModel      0x00C0          /* Flags for supported memory modes (Linear/Banked/LinearOrBanked) */
#define vbeModeBankedOnly       0x0000          /* Mode is banked only               */
#define vbeModeLinearOrBanked   0x0080          /* Supported both linear and banked  */
#define vbeModeLinearOnly       0x00C0          /* Mode is linear only               */

/***************************************************/
/* CRTC information block for refresh rate control */
/***************************************************/
typedef struct {
        word  HorizontalTotal;                /* Horizontal total (pixels)            */
        word  HorizontalSyncStart;            /* Horizontal sync start position       */
        word  HorizontalSyncEnd;              /* Horizontal sync end position         */
        word  VerticalTotal;                  /* Vertical Total (lines)               */
        word  VerticalSyncStart;              /* Vertical sync start position         */
        word  VerticalSyncEnd;                /* Vertical sync end position           */
        byte  Flags;                          /* Initialisation flags for mode        */
        dword PixelClock;                     /* Pixel clock in units of Hz           */
        word  RefreshRate;                    /* Refresh rate in units of 0.01 Hz     */
        } VBE_CRTCInfo;

/************************************************/
/* Definitions for CRTC information block flags */
/************************************************/
#define vbeDoubleScan           0x0001  /* Enable double scanned mode           */
#define vbeInterlaced           0x0002  /* Enable interlaced mode               */
#define vbeHSyncNeg             0x0004  /* Horizontal sync is negative          */
#define vbeVSyncNeg             0x0008  /* Vertical sync is negative            */


/*********************************/
/* Flags for WinAttributes field */
/*********************************/
#define vbeWinRelocatable       0x0001          /* Realocatable window(s) supported  */
#define vbeWinReadable          0x0002          /* Window is readable                */
#define vbeWinWriteable         0x0004          /* Window is writeable               */

/***************************************/
/* Flags for DirectColorModeInfo field */
/**************************************/
#define vbeCRProgrammable       0x0001          /* Color ramp is programmable           */
#define vbeRsvdUsable           0x0002          /* Rsvd field is usable by applications */

/**************************************/
/* Memory models (Memory model field) */
/**************************************/
typedef enum {  // VBE_memModels;
        vbeMemTXT      = 0,             /* Text mode memory model         */
        vbeMemCGA      = 1,             /* CGA style mode                 */
        vbeMemHGC      = 2,             /* Hercules graphics style mode   */
        vbeMemPL       = 3,             /* 16 color VGA style planar mode */
        vbeMemPK       = 4,             /* Packed pixel mode              */
        vbeMemX        = 5,             /* Non-chain 4, 256 color (ModeX) */
        vbeMemRGB      = 6,             /* Direct color RGB mode          */
        vbeMemYUV      = 7,             /* Direct color YUV mode          */
        } VBE_memModels;

/***************************************************************************/
/*                 VBE/Core 3.0 Protected mode block                       */
/***************************************************************************/
/*
 * Note: this structure is not optional and can not be implemented in some
 * VBE/Code 3.0 implementations. Structure can be locatated somewere within
 * BIOS image started at 0xC0000 with size of 32K
 * Refer to VBE/Core documentation for more info
 */
typedef union {
           struct {
              char    Signature[4];     /* must be 'PMID'                             */
              word    EntryPoint;       /* offset of PM entry point within BIOS       */
              word    PMInitialize;     /* offset of PM initialization entry point    */
              word    BIOSDataSel;      /* selector to BIOS data area emulation block */
                                        /* must be 16bit pm selector with limit at least 600h bytes */
              word    A0000Sel;         /* 16bit selectors with limit of 64K (32K for b8000sel)     */
              word    B0000Sel;
              word    B8000Sel;
              word    CodeSegSel;       /* selector to access code as data            */
                                        /* must be 16bit pm selector with limit of 64K and */
                                        /*   read/write permittions.                       */
              byte    InProtectMode;    /* set to 1 when in protected mode            */
              byte    CheckSum;         /* must add up all bytes to 0                 */
           } fields;
           char bytes[20];
    } VBE_corePMBlock;
/***************************************************************************/
/*                    32bit PM interface info block                        */
/***************************************************************************/

typedef struct {  // VBE_pmInfo;
        // All offsets are calculated from beginning of a table
        word    setWindow;                      /* Offset of Set Window call        */
        word    setDisplayStart;                /* Offset of Set Display Start call */
        word    setPalette;                     /* Offset of Set Primary Palette    */
        word    IOPrivInfo;                     /* Offset of I/O priveledge info    */

        /* UniVBE extensions */
        long    extensionSig;                   /* Identify extension (0xFBADFBAD)  */
        long    setWindowLen;                   /* Used to virtualise framebuffer   */
        long    setDisplayStartLen;
        long    setPaletteLen;

        /* ... code and tables located in here */
        } VBE_pmInfo;

#define vbeVBE20_EXT_SIG                0xFBADFBAD  /* Identify extention */

/*************************************************************/
/* Palette entry structure passed to function SetPaletteData */
/*************************************************************/
typedef struct {  // VBE_palette;
        byte    blue;                                   /* Blue component of color     */
        byte    green;                                  /* Green component of color    */
        byte    red;                                    /* Red component of color      */
        byte    alpha;                                  /* Alpha or alignment byte     */
        } VBE_palette;

/* return structures packing to default state */
#pragma pack()
/***************************************************************************/
/*                        Additional flags                                 */
/***************************************************************************/

/********************************************************/
/* Flags for combining with video modes during mode set */
/********************************************************/
#define vbeDontClear    0x8000          /* Dont clear display memory                   */
#define vbeLinearBuffer 0x4000          /* Enable linear framebuffer mode              */
#define vbeModeMask     0x03FF          /* Mask for VBE mode numbers (clear all flags) */
#define vbeSpecialMode  0x81FF          /* special mode. enable access to whole video memory. VESA 2.0 only */
/* VESA 3.0 extended flags */
#define vbeRefreshCtrl  0x0800          /* Use refresh rate control */

/* definitions for VBE_WaitRetrace */
#define vbeWaitVR       0x08            /* wait for vertical retrace */
#define vbeWaitHR       0x01            /* wait for horizontal retrace */
/******************/
/* VESA functions */
/******************/
#define vbeInt                 0x0010    /* VBE interrupt               */
#define vbeGetVBEInfo          0x4F00    /* Get VBE information         */
#define vbeGetModeInfo         0x4F01    /* Return VBE mode information */
#define vbeSetMode             0x4F02    /* Set VBE mode                */
#define vbeGetMode             0x4F03    /* Return current VBE mode     */
#define vbeSaveRestoreState    0x4F04    /* Save Restore State          */
     /* sub-functions */
     #define vbeStGetSize      0x0000    /* Get Svga State buffer size  */
     #define vbeStStore        0x0001    /* Save state                  */
     #define vbeStRestore      0x0002    /* Restore state               */
       /* sub-defs */
       #define  vbeStHardware  0x0001    /* Save the hardware state     */
       #define  vbeStBIOS      0x0002    /* Save the BIOS state         */
       #define  vbeStDAC       0x0004    /* Save the DAC state          */
       #define  vbeStSVGA      0x0008    /* Save the SuperVGA state     */
       #define  vbeStAll       0x000F    /* Save all states             */
#define vbeDispWinControl      0x4F05    /* VBE Display Window control  */
     /* sub-functions */
     #define vbeWindowA        0x0000    /* Window A flag               */
     #define vbeWindowB        0x0001    /* Window B flag               */
     #define vbeSetWindow      0x0000    /* Set Window                  */
     #define vbeGetWindow      0x0100    /* Get Window                  */
#define vbeSetLogicalSize      0x4F06    /* VBE Get/Set Logical Scan Line length */
     /* sub-functinos */
     #define vbeSetInPixels    0x0000    /* Set Scan Line Length in Pixels */
     #define vbeGetLength      0x0001    /* Get Scan Line Length        */
     #define vbeSetInBytes     0x0002    /* Set Scan Line Length in Bytes */
     #define vbeGetMaxLength   0x0003    /* Get maximum Scan Line Length  */
#define vbeDispStartControl    0x4F07    /* VBE Get/Set display start control */
     /* sub-functions */
     #define vbeSetDispStart   0x0000    /* Set display start           */
     #define vbeGetDispStart   0x0001    /* Get display start           */
     #define vbeSetDispStartVR 0x0080    /* Set display start during vertical retrace */
     /* VBE/Core 3.0 specific definitions */
     #define vbeSetDispStartAlt    0x0002   /* Set display start           */
     #define vbeSetDispStartAltVR  0x0082   /* Set display start during vertical retrace */
     #define vbeGetDispStartStatus 0x0004   /* Get last display start control request status */
     #define vbeEnableStereoMode   0x0005   /* Enable hardware stereo mode  */
     #define vbeDisableStereoMode  0x0006   /* Disable hardware stereo mode */
     #define vbeSetStereoDispStart 0x0003   /* Set stereo display start    */
     #define vbeSetStereoDispStartVR 0x0083   /* Set stereo display start during vertical retrace */

/* VESA 2.0 functions */
#define vbeDACControl          0x4F08    /* VBE Get/Set palette format  */
     /* sub-functions */
     #define vbeSetDAC         0x0000    /* Set DAC palette format      */
     #define vbeGetDAC         0x0001    /* Get DAC palette format      */
#define vbePaletteControl      0x4F09    /* VBE Load/Unload palette data */
     /* sub-functions */
     #define vbeSetPalette     0x0000    /* Set palette data            */
     #define vbeGetPalette     0x0001    /* Get palette data            */
     #define vbeSetSecPalette  0x0002    /* Set secondary palette data  */
     #define vbeGetSecPalette  0x0003    /* Get secondary palette data  */
     #define vbeSetPaletteVR   0x0080    /* Set palette data during vertical retrace */
#define vbePMInterface         0x4F0A    /* VBE 2.0 protected mode interface */
/* VBE/Core 3.0 functions */
#define vbeGetClosestClock     0x4F0B    /* Returns closest pixel clock */


/********************/
/* VESA error codes */
/********************/
/* error code in AL */
#define vbeSuccess             0x004F    /* Function supported           */
/* error code in AH */
#define vbeFuncCallOK          0x00      /* VBE function call successful */
#define vbeFuncCallFailed      0x01      /* VBE function call failed     */
#define vbeFuncNotSupported    0x02      /* VBE function call not supported in current configuration */
#define vbeFuncCallInvalid     0x03      /* VBE function call invalid for current mode */

/***************************************************************************/
/*                        Global Variables                                 */
/***************************************************************************/

/***************************************************************************/
/*                        Global functions                                 */
/***************************************************************************/
extern dword  VBEMemory;
extern word   VBEVersion;
extern word   VBE_MMIOSel;
/*********************/
/* VBE API functions */
/*********************/
ibool  VBE_init();
void   VBE_getRmBuf(long *size, word *sel, word *rseg);
ibool  VBE_getVBEInfo(VBE_vbeInfo *vbeInfo);
ibool  VBE_getModeInfo(int mode, VBE_modeInfo *modeInfo);
ibool  VBE_setVideoMode(int);
int    VBE_getVideoMode();
void*  VBE_saveState(word states = vbeStAll);
ibool  VBE_restoreState(void* buf);
ibool  VBE_setBank(int bank, int window = vbeWindowA);
#pragma aux VBE_setBank =                                  \
       " mov eax, 0x4F05"      /* vbeDispWinControl */     \
       " xor bh,bh      "      /* subfunc - set window */  \
       " int 10h        "                                  \
       " and eax,0xFF   "      /* extract result code */   \
       parm nomemory [edx] [ebx] modify nomemory value [eax];
/**********************************************************/
int    VBE_getBank(int window = vbeWindowA);
#pragma aux VBE_getBank =                                  \
       " mov eax, 0x4F05"      /* vbeDispWinControl */     \
       " mov bh,1       "      /* subfunc - get window */  \
       " xor edx,edx    "                                  \
       " int 10h        "                                  \
       parm nomemory [ebx] modify nomemory [eax] value [edx];
/**********************************************************/
ibool  VBE_setPixelsPerScanline(int pixelsPerLine, int *newBytes=NULL, int *newPixels=NULL, int *maxScanLines=NULL);
ibool  VBE_setBytesPerScanline (int  bytesPerLine, int *newBytes=NULL, int *newPixels=NULL, int *maxScanLines=NULL);
ibool  VBE_getScanlineLength(int *bytesPerLine,int *pixelsPerLine, int *maxScanlines=NULL);
ibool  VBE_getMaxScanlineLength(int *maxBytes,int *maxPixels);
ibool  VBE_setDisplayStart(int x, int y, ibool waitVR = FALSE);
#pragma aux VBE_setDisplayStart =                          \
       " mov eax, 0x4F07" /* vbeDispStartControl */        \
       " xor ebx,ebx    " /* bx = 0                   */   \
       " setnz bh       " /* if waitVT=TRUE bx = 0x80 */   \
       " shl ebx,7      "                                  \
       " int 0x10       "                                  \
       " and eax,0xFF   " /* extract result code */        \
       parm nomemory [ecx] [edx] [ebx] modify nomemory value [eax];
/**********************************************************/
ibool  VBE_getDisplayStart(int *x, int *y);
#pragma aux VBE_getDisplayStart =                          \
       " mov eax, 0x4F07" /* vbeDispStartControl */        \
       " mov ebx,1      "                                  \
       " int 0x10       "                                  \
       " and ecx,0xFFFF "                                  \
       " and edx,0xFFFF "                                  \
       " mov [esi],ecx  " /* store x             */        \
       " and eax,0xFF   " /* extract result code */        \
       " mov [edi],edx  " /* store y             */        \
       parm nomemory [esi] [edi] modify nomemory [ecx ebx edx] value [eax];
/**********************************************************/

/* VESA 2.0 extended functions */
ibool  VBE_setDACWidth(int width, int *currwidth = NULL);
int    VBE_getDACWidth();
ibool  VBE_setPalette(int start, int num, VBE_palette *pal, ibool waitVR = FALSE);
ibool  VBE_setSecondaryPalette(int start, int num, VBE_palette *pal);
ibool  VBE_getPalette(int start, int num, VBE_palette *pal);
ibool  VBE_getSecondaryPalette(int start, int num, VBE_palette *pal);
ibool  VBE_getPalette(int start, int num, VBE_palette *pal);
ibool  VBE_getSecondaryPalette(int start, int num, VBE_palette *pal);

/* Functions to return the addresses of the VBE 2.0 32 bit code */
ibool  VBE_getPMInfo(VBE_pmInfo *pmInfo);
ibool  VBE_initPMCode();
void   VBE_freePMCode();
void  *VBE_getSetBank();
void  *VBE_getSetDisplayStart();
void  *VBE_getSetPalette();
/***************************************************************/

/* New VBE/Core 3.0 functions */
ibool  VBE_setVideoModeExt(int mode,VBE_CRTCInfo *crtc);
ibool  VBE_setDisplayStartAlt(dword startAddr,ibool waitVRT = FALSE);
#pragma aux VBE_setDisplayStartAlt =                       \
       " mov eax, 0x4F07" /* vbeDispStartControl */        \
       " test ebx,ebx   "                                  \
       " mov ebx,2      "                                  \
       " jz noVR        "                                  \
       " mov ebx,0x0082 "                                  \
       "   noVR:        "                                  \
       " int 0x10       "                                  \
       " and eax,0xFF   " /* extract result code */        \
       parm nomemory [ecx] [ebx] modify nomemory value [eax];
/**********************************************************/
int    VBE_getDisplayStartStatus();
ibool  VBE_enableStereoMode();
ibool  VBE_disableStereoMode();
ibool  VBE_setStereoDisplayStart(dword leftAddr,dword rightAddr,ibool waitVRT = FALSE);
dword  VBE_getClosestClock(word mode,dword pixelClock);

/* additional pragmas */
void   VBE_waitRetrace(int retrace = vbeWaitVR);
#pragma aux VBE_waitRetrace =      \
       "jecxz nowait"              \
       "mov dx,3dah "              \
       "  wait1:    "/*wait end*/  \
       "in al,dx    "              \
       "test ax,cx  "              \
       "jnz wait1   "              \
       "  wait2:    "/*wait start*/\
       "in al,dx    "              \
       "test ax,cx  "              \
       "jz wait2    "              \
       "  nowait:   "              \
       parm nomemory [ecx] modify exact nomemory [ax dx];

#endif //__VESA_H__