/*******************  start of original comments  ********************/
/*
 * Written by Douglas Thomson (1989/1990)
 *
 * This source code is released into the public domain.
 */

/*
 * Name:    dte - Doug's Text Editor program - window module
 * Purpose: This file contains the code associated with opening and sizing
 *           windows, and also displaying the help window.
 * File:    window.c
 * Author:  Douglas Thomson
 * System:  this file is intended to be system-independent
 * Date:    October 12, 1989
 */
/*********************  end of original comments   ********************/


/*
 * The window routines have been EXTENSIVELY rewritten.  Some routines were
 * changed so only one logical function is carried out, eg. 'initialize_window'.
 * I like the Microsoft way of resizing windows - just press the up and down
 * arrows to adjust the window to desired size.  I also like pressing one key
 * to change windows.  All of which are implemented in TDE.
 *
 * In TDE, version 1.4, I added support for vertical windows.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This modification of Douglas Thomson's code is released into the
 * public domain, Frank Davis.  You may distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "define.h"
#include "tdefunc.h"


/*
 * Name:    initialize_window
 * Purpose: To open a new window
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Returns: OK if window opened successfully
 *          ERROR if anything went wrong
 * Notes:   If this is first window, then set up as normal displayed window;
 *          otherwise, make the present window invisible and open a new
 *          window in the same screen location as the old one.
 *
 * Change:  Use of new function get_next_letter( )
 */
int  initialize_window( void )
{
int  top;
int  bottom;
int  start_col;
int  end_col;
TDE_WIN *wp;        /* used for scanning windows */
TDE_WIN *window;
register file_infos *fp;     /* used for scanning files */
register int rc;
line_list_ptr ll;
line_list_ptr temp_ll;
long n;

   rc = OK;
   window = g_status.current_window;
   fp = g_status.current_file;
   if (window == NULL) {
      /*
       * special case if this is the first window on screen.
       */
      top = start_col = 0;
      bottom  = g_display.nlines;
      end_col = g_display.ncols - 1;
   } else {
      /*
       * else put the new window in same place as current window.
       *  make current window invisible.  new window becomes current window.
       */
      top       = window->top_line - 1;
      bottom    = window->bottom_line;
      start_col = window->start_col;
      end_col   = window->end_col;
   }

   assert( top < bottom );
   assert( start_col < end_col );
   assert( fp != NULL );

   if (create_window( &wp, top, bottom, start_col, end_col, fp ) == ERROR) {
      /*
       * out of memory
       */
      error( WARNING, bottom, main4 );

      /*
       * This is a real nuisance. We had room for the file and the
       *  file structure, but not enough for the window as well.
       * Now we must free all the memory that has already been
       *  allocated.
       */
      if (fp->ref_count == 0) {

         /*
          * remove fp from file pointer list.
          */
         if (fp->prev != NULL)
            fp->prev->next = fp->next;
         else
            g_status.file_list = fp->next;

         if (fp->next != NULL)
            fp->next->prev = fp->prev;

         /*
          * free the undo stack, line pointers, and linked list.
          */

         ll = fp->undo_top;
         while (ll != NULL) {
            temp_ll = ll->next;
            if (ll->line != NULL)
               my_free( ll->line );
            my_free( ll );
            ll = temp_ll;
         }

         ll = fp->line_list;
         while (ll != NULL) {
            temp_ll = ll->next;
            if (ll->line != NULL)
               my_free( ll->line );
            my_free( ll );
            ll = temp_ll;
         }

#if defined( __MSC__ )
         _fheapmin( );
#endif

         free( fp );
         wp = g_status.current_window;
         if (wp != NULL && wp->visible)
            g_status.current_file = wp->file_info;
         else
            g_status.stop = TRUE;
      }
      rc = ERROR;
   }

   if (rc != ERROR) {
      /*
       * set up the new cursor position as appropriate
       */
      wp->ccol = wp->start_col;
      wp->rcol = wp->bcol = 0;

      /*
       * if user entered something like +10 file.doc, make line 10 the
       *  current line. jmh - If the line number is negative, then take
       *  it from the end of the file (so -1 is the last line).
       */
      wp->ll    = fp->line_list->next;
      wp->rline = 1L;
      if (g_status.jump_to < 0)
         g_status.jump_to += fp->length + 1;
      if (g_status.jump_to > 0  &&  g_status.jump_to <= fp->length) {
         ll = wp->ll;
         n = 1;
         for (; n < g_status.jump_to; n++) {
            wp->bin_offset += ll->len;
            ll = ll->next;
         }
         wp->rline = g_status.jump_to;
         wp->ll    = ll;
      } else if (wp->bottom_line > wp->top_line + wp->ruler)
         ++wp->cline;
      wp->visible = TRUE;
      wp->letter = get_next_letter( fp->next_letter++ );
      if (window != NULL)
         window->visible = FALSE;

      /*
       * the new window becomes the current window.
       */
      g_status.current_window = wp;
      g_status.jump_to = 0;
   }
   return( rc );
}


/*
 * Name:    get_next_letter
 * Purpose: To find next letter to use as a window name
 * Date:    August 29, 1993
 * Author:  Byrial Jensen
 * Passed:  next_letter:  The number window to create starting from 0
 *          (Yes, the name is not very good, but I want to keep the
 *          changes as small as possible)
 */
int  get_next_letter( int next_letter )
{
   if (next_letter + 1 > (int)strlen( windowletters ) )
      return( LAST_WINDOWLETTER );
   else
      return( (int) (windowletters[next_letter]) );
}


/*
 * Name:    next_window
 * Purpose: To move to the next visible window.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Start with current window.  If next window exists then go to it
 *           else go to the first (top) window on screen.
 *          When I added vertical windows, finding the "correct" next
 *           window became extremely, unnecessarily, unmanageably complicated.
 *           let's just use a simple procedure to find the first available,
 *           visible, next window.
 */
int  next_window( TDE_WIN *window )
{
register TDE_WIN *wp;
int  change;

   if (window != NULL) {
      change = FALSE;
      /*
       * start with current window and look for first next
       *  visible window
       */
      wp = window->next;
      while (wp != NULL) {
         if (wp->visible) {
            change = TRUE;
            break;
         }
         wp = wp->next;
      }

      /*
       * if we haven't found a visible window yet, go to the beginning of
       *  the list until we find a visible window.
       */
      if (!change) {
         wp = g_status.window_list;
         while (wp != window) {
            if (wp->visible) {
               change = TRUE;
               break;
            }
            wp = wp->next;
         }
      }
      if (change == TRUE) {
         entab_linebuff( );
         un_copy_line( window->ll, window, TRUE );
         g_status.current_window = wp;
         g_status.current_file = wp->file_info;
      }
   }
   return( OK );
}


/*
 * Name:    prev_window
 * Purpose: To move to the previous visible window.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Start with current window.  If previous window exists then go to
 *           it else go to the last (bottom) window on screen.  Opposite of
 *           next_window.
 *          when I added vertical windows, finding the "correct" previous
 *           window became extremely, unnecessarily, unmanageably complicated.
 *           let's just use a simple procedure to find the first available,
 *           visible, previous window.
 */
int  prev_window( TDE_WIN *window )
{
register TDE_WIN *wp;
int  change;

   if (window != NULL) {
      change = FALSE;

      /*
       * start with current window and look for first previous
       *  visible window
       */
      wp = window->prev;
      while (wp != NULL) {
         if (wp->visible) {
            change = TRUE;
            break;
         }
         wp = wp->prev;
      }

      /*
       * if we haven't found a visible window yet, go to the end of
       *  the list and work backwards until we find a visible window.
       */
      if (!change) {
         wp = window->next;
         if (wp != NULL) {
            while (wp->next != NULL)
               wp = wp->next;
            while (wp != window) {
               if (wp->visible) {
                  change = TRUE;
                  break;
               }
               wp = wp->prev;
            }
         }
      }
      if (change == TRUE) {
         entab_linebuff( );
         un_copy_line( window->ll, window, TRUE );
         g_status.current_window = wp;
         g_status.current_file = wp->file_info;
      }
   }
   return( OK );
}


/*
 * Name:    split_horizontal
 * Purpose: To split screen horizontally at the cursor.
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:  window:  pointer to current window
 * Notes:   split the screen horizontally at the cursor position.
 */
int  split_horizontal( TDE_WIN *window )
{
register TDE_WIN *wp;
register TDE_WIN *win;  /* register pointer for window */
TDE_WIN *temp;
file_infos *file;       /* file structure for file belonging to new window */
int  rc;

   rc = OK;
   win = window;
   if ( win != NULL) {

      /*
       * check that there is room for the window
       */
      if (win->bottom_line - win->cline < 2) {
         /*
          * move cursor up first
          */
         error( WARNING, win->bottom_line, win1 );
         rc = ERROR;
      } else {
         file = win->file_info;

         assert( file != NULL );

         if (create_window( &temp, win->cline+1, win->bottom_line,
                            win->start_col, win->end_col, file ) == ERROR) {
            /*
             * out of memory
             */
            error( WARNING, win->bottom_line, main4 );
            rc = ERROR;
         }
         if (rc == OK  &&  temp != NULL) {
            entab_linebuff( );
            un_copy_line( win->ll, win, TRUE );
            wp = temp;
            /*
             * record that the current window has lost some lines from
             *  the bottom for the new window, and adjust its page size
             *  etc accordingly.
             */
            win->bottom_line = win->cline;
            setup_window( win );
            display_current_window( win );

            /*
             * set up the new cursor position as appropriate
             */
            wp->rcol = win->rcol;
            wp->ccol = win->ccol;
            wp->bcol = win->bcol;
            wp->rline = win->rline;
            wp->bin_offset = win->bin_offset;
            wp->ll    = win->ll;
            wp->cline += win->cline - (win->top_line + win->ruler);
            if (wp->cline > wp->bottom_line)
               wp->cline = wp->bottom_line;
            wp->visible = TRUE;
            wp->vertical = win->vertical;
            wp->letter = get_next_letter( file->next_letter++ );
            wp->ruler  = mode.ruler;
            wp->syntax = win->syntax;

            /*
             * the new window becomes the current window.
             */
            g_status.current_window = wp;

            show_window_count( g_status.window_count );
            show_window_header( wp );
            display_current_window( wp );
            if (wp->vertical)
               show_vertical_separator( wp );
            make_ruler( wp );
            show_ruler( wp );
            rc = OK;
         }
      }
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    split_vertical
 * Purpose: To split screen vertically at the cursor.
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:  window:  pointer to current window
 * Notes:   split the screen vertically at the cursor position.
 *
 * Change:  Use of new function: get_next_letter( )
 */
int  split_vertical( TDE_WIN *window )
{
register TDE_WIN *wp;
register TDE_WIN *win;  /* register pointer for window */
TDE_WIN *temp;
file_infos *file;       /* file structure for file belonging to new window */
int  rc;

   rc = OK;
   win = window;
   if (win != NULL) {

      /*
       * check that there is room for the window
       */
      if (win->start_col + 15 > win->ccol) {
         /*
          * move cursor right first
          */
         error( WARNING, win->bottom_line, win2 );
         rc = ERROR;
      } else if (win->end_col - 15 < win->ccol) {
         /*
          * move cursor left first
          */
         error( WARNING, win->bottom_line, win3 );
         rc = ERROR;
      } else {
         file = win->file_info;

         assert( file != NULL );

         if (create_window( &temp, win->top_line-1, win->bottom_line,
                            win->ccol+1, win->end_col, file ) == ERROR) {
            /*
             * out of memory
             */
            error( WARNING, win->bottom_line, main4 );
            rc = ERROR;
         }

         if (rc == OK  &&  temp != NULL) {
            entab_linebuff( );
            un_copy_line( win->ll, win, TRUE );
            wp = temp;

            /*
             * record that the current window has lost some columns from
             *  the window to the left for the new window
             */
            win->ccol = win->end_col = win->ccol - 1;
            win->rcol--;
            win->vertical = TRUE;
            show_window_header( win );
            show_vertical_separator( win );
            display_current_window( win );
            make_ruler( win );
            show_ruler( win );
            show_ruler_pointer( win );

            /*
             * set up the new cursor position as appropriate
             */
            wp->rcol = win->rcol;
            wp->ccol = wp->start_col + win->ccol - win->start_col;
            if (wp->ccol > wp->end_col)
               wp->ccol = wp->end_col;
            wp->bcol  = win->bcol;
            wp->rline = win->rline;
            wp->bin_offset = win->bin_offset;
            wp->ll       = win->ll;
            wp->cline    = win->cline;
            wp->visible  = TRUE;
            wp->vertical = TRUE;
            wp->letter   = get_next_letter( file->next_letter++ );
            wp->ruler    = mode.ruler;
            wp->syntax   = win->syntax;

            /*
             * the new window becomes the current window.
             */
            g_status.current_window = wp;

            check_virtual_col( wp, wp->rcol, wp->ccol );
            wp->file_info->dirty = FALSE;
            show_window_count( g_status.window_count );
            show_window_header( wp );
            display_current_window( wp );
            make_ruler( wp );
            show_ruler( wp );
         }
      }
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    show_vertical_separator
 * Purpose: To separate vertical screens
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_vertical_separator( TDE_WIN *window )
{
int  i;
int  line;
int  col;

   line = window->top_line - 1;
   col  = window->end_col + 1;
   if (col < g_display.ncols - 1) {
      i = window->bottom_line - line;

      assert( i <= g_display.nlines );

      while (i-- >= 0)
         c_output( VERTICAL_CHAR, col, line++, g_display.head_color );
   }
}


/*
 * Name:    size_window
 * Purpose: To change the size of the current and one other window.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Use the Up and Down arrow keys to make the current window
 *           bigger or smaller.  The window above will either grow
 *           or contract accordingly.
 *
 * jmh 980803: the few times I want to use this function, it won't let me. So
 *              now it resizes everything, except the top window, which is now
 *              defined as top_line == 1 && start_col == 0.
 *             Borrowing the close window example:
 *
 *                    Ŀ
 *                              2     
 *                          Ĵ
 *                            3   4  
 *                      1   Ĵ
 *                              5     
 *                          Ĵ
 *                              6     
 *                    
 *
 *             window 1 is the top window and cannot be resized. Window 2
 *             cannot be sized horizontally, since it still works on top line.
 *             Sizing any of windows 2, 3, 5 & 6 vertically will size the other
 *             windows, plus window 1. Sizing window 3 or 4 horizontally will
 *             affect the other, plus 2. Similarly, 5 will affect 3 and 4.
 *             As you've probably guessed, use Left and Right (ie. CharLeft
 *             and CharRight) to move the left edge.
 *             I've also used the mode line to display the message (in the
 *              help color) and removed the cursor.
 *
 */
int  size_window( TDE_WIN *window )
{
int  func;
long c;
int  resize;
int  old_bottom_line;
int  old_top_line;
int  old_right_col;
int  old_left_col;
int  new_bottom_line;
int  new_top_line;
int  new_right_col;
int  new_left_col;
int  lowest_top_line  = MAX_LINES;
int  highest_top_line = 1;
int  lowest_left_col  = MAX_COLS;
int  highest_left_col = 0;
register TDE_WIN *above;
register TDE_WIN *win;
TDE_WIN *vis_win[20];   /* assume only 20 visible windows */
int  vis_flag[20];      /* display flags for those windows */
int  visible = 0;       /* number of visible windows */
int  flag;
int  i;
DISPLAY_BUFF;
char *errmsg = NULL;

/*
 * These are the display flags.
 */
#define TOP_EDGE    1
#define BOTTOM_EDGE 2
#define LEFT_EDGE   4
#define RIGHT_EDGE  8

   win = window;
   if (win->top_line == 1 && win->start_col == 0)
      /*
       * cannot resize top window
       */
      errmsg = win6;
   else {
      /*
       * resizing affects current window and all (visible) windows that
       * share its top and/or left edge.
       */
      for (above = g_status.window_list; above != NULL; above = above->next) {
         if (above->visible) {
            flag = 0;
            if (win->top_line != 1) {
               if (above->top_line == win->top_line) {
                  flag |= TOP_EDGE;
                  if (above->bottom_line - above->ruler < lowest_top_line)
                     lowest_top_line = above->bottom_line - above->ruler;

               } else if (above->bottom_line + 2 == win->top_line) {
                  flag |= BOTTOM_EDGE;
                  if (above->top_line + above->ruler + 2 > highest_top_line)
                     highest_top_line = above->top_line + above->ruler + 2;
               }
            }
            if (win->start_col != 0) {
               if (above->start_col == win->start_col) {
                  flag |= LEFT_EDGE;
                  if (above->end_col - 14 < lowest_left_col)
                     lowest_left_col = above->end_col - 14;

               } else if (above->end_col + 2 == win->start_col) {
                  flag |= RIGHT_EDGE;
                  if (above->start_col + 16 > highest_left_col)
                     highest_left_col = above->start_col + 16;
               }
            }
            if (flag != 0) {
               if (visible == 20) {
                  /*
                   * too many windows
                   */
                  errmsg = win5;
                  break;
               }
               vis_flag[visible]  = flag;
               vis_win[visible++] = above;
            }
         }
      }
   }
   if (errmsg != NULL) {
      error( WARNING, win->bottom_line, errmsg );
      return( ERROR );
   }

   entab_linebuff( );
   un_copy_line( win->ll, win, TRUE );
   save_screen_line( 0, g_display.mode_line, display_buff );
   eol_clear( 0, g_display.mode_line, g_display.help_color );
   i = (g_display.ncols - (int)strlen( win4 )) / 2;
   if (i < 0) {
      i = 0;
      win4[g_display.ncols] = '\0';
   }
   /*
    * use cursor keys to change window size
    */
   s_output( win4, g_display.mode_line, i, g_display.help_color );
   xygoto( -1, -1 );

   old_top_line    = win->top_line;
   old_bottom_line = old_top_line - 2;
   old_left_col    = win->start_col;
   old_right_col   = old_left_col - 2;
   for (;;) {
      /*
       * If user has redefined the ESC and Return keys, make them Rturn
       *  and AbortCommand in this function.
       */
      do {
         c    = getkey( );
         func = getfunc( c );
         if (c == RTURN || func == NextLine || func == BegNextLine)
            func = Rturn;
         else if (c == ESC)
            func = AbortCommand;
      } while (func != Rturn    && func != AbortCommand &&
               func != LineUp   && func != LineDown     &&
               func != CharLeft && func != CharRight);

      if (func == Rturn || func == AbortCommand)
         break;

      resize = FALSE;
      new_top_line = win->top_line;
      new_left_col = win->start_col;
      for (i = 0; i < visible; ++i) {
         above = vis_win[i];
         flag  = vis_flag[i];
         switch (func) {
            /*
             * if LineUp, make current window top line grow and bottom line
             *  of above window shrink.  if window movement covers up current
             *  line of window then we must adjust logical line and real line.
             */
            case LineUp:
               if (new_top_line > highest_top_line) {
                  if (flag & TOP_EDGE) {
                     if (above->rline == (above->cline -
                                          (above->top_line + above->ruler)))
                        --above->cline;
                     --above->top_line;
                     resize = TRUE;
                     if (mode.ruler) {
                        if (above->ruler == FALSE) {
                           if (above->cline == above->top_line)
                              ++above->cline;
                           if (above->cline > above->bottom_line)
                              above->cline = above->bottom_line;
                           above->ruler = TRUE;
                           --lowest_top_line;
                        }
                     }
                  } else if (flag & BOTTOM_EDGE) {
                     if (above->cline == above->bottom_line)
                        --above->cline;
                     --above->bottom_line;
                     resize = TRUE;
                  }
               }
               break;

            /*
             * if LineDown, make current window top line shrink and bottom line
             *  of above window grow.  if window movement covers up current
             *  line of window then we must adjust logical line and real line.
             */
            case LineDown:
               if (new_top_line < lowest_top_line) {
                  if (flag & TOP_EDGE) {
                     if (above->cline == above->top_line + above->ruler)
                        ++above->cline;
                     ++above->top_line;
                     resize = TRUE;
                  } else if (flag & BOTTOM_EDGE) {
                     ++above->bottom_line;
                     resize = TRUE;
                     if (mode.ruler) {
                        if (above->ruler == FALSE) {
                           if (above->cline == above->top_line)
                              ++above->cline;
                           if (above->cline > above->bottom_line)
                              above->cline = above->bottom_line;
                           above->ruler = TRUE;
                           ++highest_top_line;
                        }
                     }
                  }
               }
               break;

            /*
             * if CharLeft, make current window left edge grow and right edge
             * of adjacent window shrink.
             */
            case CharLeft:
               if (new_left_col > highest_left_col) {
                  if (flag & LEFT_EDGE) {
                     --above->start_col;
                     --above->ccol;
                     resize = TRUE;
                  } else if (flag & RIGHT_EDGE) {
                     --above->end_col;
                     check_virtual_col( above, above->rcol, above->ccol );
                     resize = TRUE;
                  }
               }
               break;

            /*
             * if CharRight, make current window left edge shrink and right
             * edge of adjacent window grow.
             */
            case CharRight:
               if (new_left_col < lowest_left_col) {
                  if (flag & LEFT_EDGE) {
                     ++above->start_col;
                     ++above->ccol;
                     check_virtual_col( above, above->rcol, above->ccol );
                     resize = TRUE;
                  } else if (flag & RIGHT_EDGE) {
                     ++above->end_col;
                     resize = TRUE;
                  }
               }
               break;
         }

         /*
          * if we resize a window, then update window size and current and
          *  real lines if needed.
          */
         if (resize == TRUE) {
            setup_window( above );
            show_window_header( above );
            above->ruler = mode.ruler;
            make_ruler( above );
            show_ruler( above );
            show_ruler_pointer( above );
            display_current_window( above );
            if (above->vertical)
               show_vertical_separator( above );
         }
      }
   }
   /*
    * Now all the hidden windows need to be adjusted.
    */
   new_top_line    = win->top_line;
   new_bottom_line = new_top_line - 2;
   new_left_col    = win->start_col;
   new_right_col   = new_left_col - 2;
   for (above=g_status.window_list; above != NULL; above=above->next) {
      if (!above->visible) {
         if (above->bottom_line == old_bottom_line) {
            above->bottom_line = new_bottom_line;
            if (above->cline < new_bottom_line)
               above->cline = new_bottom_line;
            setup_window( above );
         } else if (above->top_line == old_top_line) {
            above->top_line = new_top_line;
            if (above->cline < new_top_line)
               above->cline = new_top_line;
            if ((long)(above->cline - (above->top_line+above->ruler)) >
                                                          above->rline)
               above->cline = (int)above->rline + above->top_line +
                               above->ruler;
            setup_window( above );
         }
         if (above->start_col == old_left_col) {
            above->start_col = new_left_col;
            check_virtual_col( above, above->rcol, above->ccol );
         } else if (above->end_col == old_right_col) {
            above->end_col = new_right_col;
            check_virtual_col( above, above->rcol, above->ccol );
         }
      }
   }
   restore_screen_line( 0, g_display.mode_line, display_buff );
   xygoto( win->ccol, win->cline );
   return( OK );
}


/*
 * Name:    zoom_window
 * Purpose: To blow-up current window.
 * Date:    September 1, 1991
 * Modified: August 11, 1997, Jason Hood - added call to setup_window
 * Passed:  window:  pointer to current window
 * Notes:   Make all windows, visible and hidden, full size.
 */
int  zoom_window( TDE_WIN *window )
{
register TDE_WIN *wp;

   if (window != NULL) {
      entab_linebuff( );
      un_copy_line( window->ll, window, TRUE );
      for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
         if (wp != window && wp->visible)
            wp->visible = FALSE;

         /*
          * can't diff one window, reset the diff
          */
         diff.defined = FALSE;
         if (wp->top_line != 1)
            wp->cline = wp->cline - (wp->top_line+wp->ruler) + 1;
         wp->top_line = 1;
         wp->bottom_line = g_display.nlines;
         wp->end_col   = g_display.ncols - 1;
         wp->start_col = 0;
         wp->vertical  = FALSE;
         check_virtual_col( wp, wp->rcol, wp->ccol );
         make_ruler( wp );
         setup_window( wp );
      }
      redraw_screen( window );
      show_ruler( window );
   }
   return( OK );
}


/*
 * Name:    next_hidden_window
 * Purpose: To display the window that is "behind" current window.
 * Date:    September 1, 1991
 * Passed:  window:  pointer to current window
 */
int  next_hidden_window( TDE_WIN *window )
{
int  poof = FALSE;
register TDE_WIN *wp;

   if (window != NULL) {

      /*
       * look for next hidden window starting with current window.
       */
      wp = window;
      for (wp=window->next; wp != NULL && !poof; ) {
         if (!wp->visible)
            poof = TRUE;
         else
            wp = wp->next;
      }

      /*
       * if we haven't found an invisible window yet, start looking
       *  for a hidden window from the beginning of the window list.
       */
      if (!poof) {
         for (wp=g_status.window_list; wp != NULL && !poof; ) {
            if (!wp->visible)
               poof = TRUE;
            else
               wp = wp->next;
         }
      }

      if (poof) {
         entab_linebuff( );
         un_copy_line( window->ll, window, TRUE );
         wp->cline = window->top_line + window->ruler +
                       (wp->cline - (wp->top_line + wp->ruler));
         wp->top_line = window->top_line;
         wp->bottom_line = window->bottom_line;
         wp->start_col = window->start_col;
         wp->end_col   = window->end_col;
         wp->vertical  = window->vertical;
         if (wp->cline < wp->top_line + wp->ruler)
            wp->cline = wp->top_line + wp->ruler;
         if (wp->cline > wp->bottom_line)
            wp->cline = wp->bottom_line;
         if ((wp->cline - (wp->top_line+wp->ruler)) > wp->rline)
            wp->cline = (int)wp->rline + wp->top_line + wp->ruler;
         check_virtual_col( wp, wp->rcol, wp->ccol );
         wp->visible = TRUE;
         window->visible = FALSE;
         if (diff.defined  &&  (diff.w1 == window  ||  diff.w2 == window))
            diff.defined = FALSE;
         g_status.current_window = wp;
         redraw_current_window( wp );
         make_ruler( wp );
         show_ruler( wp );
      }
   }
   return( OK );
}


/*
 * Name:    prev_hidden_window
 * Purpose: To display the window that is "in front of" current window.
 * Author:  Jason Hood
 * Date:    November 24, 1996
 * Passed:  window:  pointer to current window
 * Notes:   a duplication of next_hidden_window, with the appropriate
 *           modifications
 */
int  prev_hidden_window( TDE_WIN *window )
{
int  poof = FALSE;
register TDE_WIN *wp;

   if (window != NULL) {

      /*
       * look for previous hidden window starting with current window.
       */
      wp = window;
      for (wp=window->prev; wp != NULL && !poof; ) {
         if (!wp->visible)
            poof = TRUE;
         else
            wp = wp->prev;
      }

      /*
       * if we haven't found an invisible window yet, start looking
       *  for a hidden window from the end of the window list.
       */
      if (!poof) {
         wp = window->next;
         if (wp != NULL) {
           while (wp->next != NULL) wp = wp->next;
           for (; wp != window && !poof; ) {
              if (!wp->visible)
                 poof = TRUE;
              else
                 wp = wp->prev;
           }
         }
      }

      if (poof) {
         entab_linebuff( );
         un_copy_line( window->ll, window, TRUE );
         wp->cline = window->top_line + window->ruler +
                       (wp->cline - (wp->top_line + wp->ruler));
         wp->top_line = window->top_line;
         wp->bottom_line = window->bottom_line;
         wp->start_col = window->start_col;
         wp->end_col   = window->end_col;
         wp->vertical  = window->vertical;
         if (wp->cline < wp->top_line + wp->ruler)
            wp->cline = wp->top_line + wp->ruler;
         if (wp->cline > wp->bottom_line)
            wp->cline = wp->bottom_line;
         if ((wp->cline - (wp->top_line+wp->ruler)) > wp->rline)
            wp->cline = (int)wp->rline + wp->top_line + wp->ruler;
         check_virtual_col( wp, wp->rcol, wp->ccol );
         wp->visible = TRUE;
         window->visible = FALSE;
         if (diff.defined  &&  (diff.w1 == window  ||  diff.w2 == window))
            diff.defined = FALSE;
         g_status.current_window = wp;
         redraw_current_window( wp );
         make_ruler( wp );
         show_ruler( wp );
      }
   }
   return( OK );
}


/*
 * Name:    setup_window
 * Purpose: To set the page length and the center line of a window, based
 *           on the top and bottom lines.
 * Date:    June 5, 1991
 * Passed:  window: window to be set up
 */
void setup_window( TDE_WIN *window )
{
   window->page = window->bottom_line - (window->top_line + window->ruler) -
                  g_status.overlap + 1;
   if (window->page < 1)
      window->page = 1;
}


/*
 * Name:    finish
 * Purpose: To remove the current window and terminate the program if no
 *           more windows are left.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Order of deciding which window becomes current window:
 *          1) If any invisible window with same top and bottom line,
 *          and start_col and end_col, then first invisible one becomes
 *          current window.
 *          2) window above if it exists becomes current window
 *          3) window below if it exists becomes current window
 *          4) window right if it exists becomes current window
 *          5) window left  if it exists becomes current window
 *          6) first available invisible window becomes current window.
 *              (Modified January 23, 1998 by jmh.)
 *          When I added vertical windows, this routine became a LOT
 *           more complicated.  To keep things reasonably sane, let's
 *           only close windows that have three common edges, eg.
 *
 *                    Ŀ
 *                              no    
 *                          Ĵ
 *                          yes1 yes1
 *                      no  Ĵ
 *                             yes2   
 *                          Ĵ
 *                             yes2   
 *                    
 *
 *          Windows with 'no' cannot be closed.  Windows with 'yes' can
 *          be combined with windows that have the same yes number.
 *
 * jmh 980809: removed menu to choose RepeatGrep; just press Return instead.
 */
void finish( TDE_WIN *window )
{
register TDE_WIN *wp;   /* for scanning other windows */
register TDE_WIN *win;  /* register pointer for window */
file_infos *file, *fp;  /* for scanning other files */
int  poof;
int  cline;
int  top;
int  bottom;
int  start_col;
int  end_col;
int  max_letter;
int  letter_index;
int  file_change = FALSE;
line_list_ptr ll;
line_list_ptr temp_ll;
long key;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return;

   /*
    * remove all hidden windows that point to same file
    */
   file = win->file_info;
   for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
      if (wp->file_info == file) {
         if (!wp->visible) {
            if (wp->prev == NULL) {
               if (wp->next == NULL)
                  g_status.stop = TRUE;
               else
                  g_status.window_list = wp->next;
            } else
               wp->prev->next = wp->next;
            if (wp->next)
               wp->next->prev = wp->prev;
            --wp->file_info->ref_count;
            free( wp );
            --g_status.window_count;
         }
      }
   }

   if (win->prev == NULL && win->next == NULL)
      g_status.stop = TRUE;

   poof = FALSE;

   if (g_status.stop != TRUE) {
      /*
       * see if there are any invisible windows with same top and bottom,
       *  lines, and start_col and end_col as this window.
       */
      top       = win->top_line;
      bottom    = win->bottom_line;
      start_col = win->start_col;
      end_col   = win->end_col;
      wp = win->prev;
      while (wp != NULL && poof == FALSE) {
         if (wp->top_line == top         &&  wp->bottom_line == bottom  &&
             wp->start_col == start_col  &&  wp->end_col == end_col     &&
                                                              !wp->visible)
            poof = TRUE;
         else
            wp = wp->prev;
      }
      if (poof == FALSE) {
         wp = win->next;
         while (wp != NULL && poof == FALSE) {
            if (wp->top_line == top         &&  wp->bottom_line == bottom  &&
                wp->start_col == start_col  &&  wp->end_col == end_col     &&
                                                                 !wp->visible)
               poof = TRUE;
            else
               wp = wp->next;
         }
      }

      if (poof == FALSE) {
         /*
          * see if there are any windows above
          */
         wp = g_status.window_list;
         while (wp != NULL && poof == FALSE) {
            if (wp->bottom_line+2 == win->top_line  &&
                wp->start_col     == win->start_col &&
                wp->end_col       == win->end_col   && wp->visible) {
               poof = TRUE;
               top  = wp->top_line;
            } else
               wp = wp->next;
         }
         if (poof == FALSE) {
            /*
             * see if there are any windows below
             */
            wp = g_status.window_list;
            while (wp != NULL && poof == FALSE) {
               if (wp->top_line-2 == win->bottom_line  &&
                   wp->start_col  == win->start_col    &&
                   wp->end_col    == win->end_col      && wp->visible) {
                  poof = TRUE;
                  bottom = wp->bottom_line;
               } else
                  wp = wp->next;
            }
         }
         if (poof == FALSE) {
            /*
             * see if there are any windows right
             */
            wp = g_status.window_list;
            while (wp != NULL && poof == FALSE) {
               if (wp->top_line    == win->top_line    &&
                   wp->bottom_line == win->bottom_line &&
                   wp->start_col-2 == win->end_col     && wp->visible) {
                  poof = TRUE;
                  end_col = wp->end_col;
               } else
                  wp = wp->next;
            }
         }
         if (poof == FALSE) {
            /*
             * see if there are any windows left
             */
            wp = g_status.window_list;
            while (wp != NULL && poof == FALSE) {
               if (wp->top_line    == win->top_line    &&
                   wp->bottom_line == win->bottom_line &&
                   wp->end_col+2   == win->start_col   && wp->visible) {
                  poof = TRUE;
                  start_col = wp->start_col;
               } else
                  wp = wp->next;
            }
         }
         if (poof == FALSE) {
            /*
             * see if there are any other invisible windows.
             */
            wp = win->prev;
            while (wp != NULL && poof == FALSE) {
               if (!wp->visible)
                  poof = TRUE;
               else
                  wp = wp->prev;
            }
            if (poof == FALSE) {
               wp = win->next;
               while (wp != NULL && poof == FALSE) {
                  if (!wp->visible)
                     poof = TRUE;
                  else
                     wp = wp->next;
               }
            }
         }
      }
      if (poof) {
         wp->visible = TRUE;
         cline = wp->cline - (wp->top_line+wp->ruler);
         wp->top_line = top;
         wp->bottom_line = bottom;
         wp->cline = (wp->top_line+wp->ruler) + cline;
         if (wp->cline > wp->bottom_line)
            wp->cline = wp->top_line+wp->ruler;
         wp->start_col = start_col;
         wp->end_col   = end_col;
         if (start_col == 0 && end_col == g_display.ncols - 1)
            wp->vertical = FALSE;
         else
            wp->vertical = TRUE;
         check_virtual_col( wp, wp->rcol, wp->ccol );
         setup_window( wp );
         show_window_header( wp );
         if (wp->vertical)
            show_vertical_separator( wp );

         /*
          * The window above, below, or previously invisible becomes the new
          *  current window.
          */
         g_status.current_window = wp;
      }
   }

   if (!poof && g_status.stop != TRUE)
      /*
       * cannot close current window
       */
      error( WARNING, win->bottom_line, win7 );
   else {

      /*
       * free unused file memory if necessary
       */
      if (--file->ref_count == 0) {

         /*
          * if a block is marked, unmark it
          */
         if (file == g_status.marked_file) {
            g_status.marked      = FALSE;
            g_status.marked_file = NULL;
         }

         for (fp=g_status.file_list; fp != NULL; fp=fp->next) {
            if (fp->file_no > file->file_no)
               fp->file_no--;
         }
         file_change = file->file_no;

         /*
          * no window now refers to this file, so remove file from the list
          */
         if (file->prev == NULL)
            g_status.file_list = file->next;
         else
            file->prev->next = file->next;
         if (file->next)
            file->next->prev = file->prev;

         /*
          * free the line pointers, linked list of line pointers, and
          *  file struc.
          */
         ll = file->undo_top;
         while (ll != NULL) {
            temp_ll = ll->next;
            if (ll->line != NULL)
               my_free( ll->line );
            my_free( ll );
            ll = temp_ll;
         }

         ll = file->line_list;
         while (ll != NULL) {
            temp_ll = ll->next;
            if (ll->line != NULL)
               my_free( ll->line );
            my_free( ll );
            ll = temp_ll;
         }

#if defined( __MSC__ )
         _fheapmin( );
#endif

         free( file );
         if (--g_status.file_count) {
            show_file_count( g_status.file_count );
            show_avail_mem( );
         }
      }

      /*
       * remove the current window from the window list
       */
      if (win->prev == NULL)
         g_status.window_list = win->next;
      else
         win->prev->next = win->next;

      if (win->next)
         win->next->prev = win->prev;

      if (diff.defined  &&  (diff.w1 == win  ||  diff.w2 == win))
         diff.defined = FALSE;

      /*
       * free the memory taken by the window structure
       */
      free( win );
      --g_status.window_count;

      if (g_status.stop == FALSE) {
         g_status.current_file = wp->file_info;
         wp->file_info->dirty = LOCAL;
         make_ruler( wp );
         show_ruler( wp );
         show_window_count( g_status.window_count );
         if (file_change) {
            for (wp=g_status.window_list; wp!=NULL; wp=wp->next)
               if (wp->visible)
                  show_window_number_letter( wp );
         } else {
            max_letter = 0;
            for (wp=g_status.window_list; wp!=NULL; wp=wp->next) {
               if (wp->file_info == file) {
                  if (wp->letter == LAST_WINDOWLETTER) {
                     max_letter = strlen( windowletters );
                     break;
                  } else {
                     letter_index = strchr( windowletters, wp->letter )
                                    - windowletters;
                     if (letter_index > max_letter)
                        max_letter = letter_index;
                  }
               }
            }
            if (max_letter < file->next_letter - 1)
               file->next_letter = max_letter + 1;
         }
      }
   }

   if (g_status.stop == TRUE) {
      if (g_status.sas_defined && g_status.sas_arg < g_status.sas_argc) {
         show_avail_mem( );
         for (bottom=0; bottom <= g_display.nlines; bottom++)
            eol_clear( 0, bottom, g_display.text_color );
         bottom  = g_display.nlines;
         set_prompt( win18, bottom );
         key = getkey( );
         top = (key == RTURN) ? RepeatGrep : getfunc( key );
         eol_clear( 0, bottom, g_display.text_color );
         if (top == RepeatGrep) {
            g_status.command = RepeatGrep;
            g_status.window_list = g_status.current_window = NULL;
            if (search_and_seize( g_status.window_list ) != ERROR)
               g_status.stop = FALSE;
         }
      /* This else added by jmh to load the next command line file */
      } else {
         if (!g_status.sas_defined && g_status.arg < g_status.argc) {
            g_status.command = EditNextFile;
            g_status.window_list = g_status.current_window = NULL;
            if (edit_next_file( g_status.window_list ) != ERROR)
               g_status.stop = FALSE;
         }
      }
   }
}


/*
 * Name:    create_window
 * Purpose: To allocate space for a new window structure and set up some
 *           of the relevant fields.
 * Date:    June 5, 1991
 * Passed:  window: pointer to window pointer
 *          top:    the top line of the new window
 *          bottom: the bottom line of the new window
 *          start_col:  starting column of window on screen
 *          end_col:    ending column of window on screen
 *          file:   the file structure to be associated with the new window
 * Returns: OK if window could be created
 *          ERROR if out of memory
 */
int  create_window( TDE_WIN **window, int top, int bottom, int start_col,
                    int end_col, file_infos *file )
{
TDE_WIN *wp;            /* temporary variable - use it instead of **window */
register TDE_WIN *prev;
int  rc;                /* return code */

   rc = OK;
   /*
    * allocate space for new window structure
    */
   if ((*window = (TDE_WIN *)calloc( 1, sizeof(TDE_WIN) )) == NULL) {
      /*
       * out of memory
       */
      error( WARNING, g_display.nlines, main4 );
      rc = ERROR;
   } else {

     /*
      * set up appropriate fields
      */
      wp              = *window;
      wp->file_info   = file;
      wp->top_line    = top+1;
      wp->bottom_line = bottom;
      wp->start_col   = start_col;
      wp->end_col     = end_col;
      wp->bin_offset  = 0;
      wp->ruler       = mode.ruler;
      make_ruler( wp );
      wp->cline       = wp->top_line + wp->ruler;
      if (start_col == 0 && end_col == g_display.ncols-1)
         wp->vertical = FALSE;
      else
         wp->vertical = TRUE;
      wp->prev        = NULL;
      wp->next        = NULL;

      setup_window( wp );

      /*
       * add window into window list
       */
      prev = g_status.current_window;
      if (prev) {
         (*window)->prev = prev;
         if (prev->next)
            prev->next->prev = *window;
         (*window)->next = prev->next;
         prev->next = *window;
      }
      if (g_status.window_list == NULL)
         g_status.window_list = *window;

      /*
       * record that another window is referencing this file
       */
      ++file->ref_count;
      file->dirty = LOCAL;
      ++g_status.window_count;
   }
   return( rc );
}
