/*
 * This file was originally console.c, but I split that into three separate
 * files for each of the systems. These query functions were what was left
 * over - jmh.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993 version 3.1
 * Date:             November 13, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may distribute it freely.
 */


#include "tdestr.h"
#include "common.h"
#include "define.h"
#include "tdefunc.h"
#include <stdarg.h>


/*
 * Name:    getfunc
 * Purpose: get the function assigned to key c
 * Date:    July 11, 1991
 * Passed:  c:  key just pressed
 * Notes:   key codes less than 256 or 0x100 are not assigned a function.
 *           The codes in the range 0-255 are ASCII and extended ASCII chars.
 * jmh 980727: 0 is used to represent the insert_overwrite function for
 *              normal characters, and an unassigned function key or two-key.
 *             If the function is PlayBack, point g_status.key_macro to
 *              the macro structure; otherwise it's undefined.
 * jmh 980826: "keys" between 256+MAX_KEYS and 256+MAX_KEYS+NUM_FUNCS are
 *              actually functions.
 */
int  getfunc( long c )
{
int  func = 0;
TREE *two_key = NULL;
TDE_WIN  *win;
LANGUAGE *language;

   if (c > 256) {
      if (PARENT_KEY( c )) {
         win = g_status.current_window;
         if (win->syntax) {
            language = win->file_info->syntax;
            do {
               two_key = search_tree( c, language->key_tree.right );
               language = language->parent;
            } while ((two_key == NULL || two_key->type.func == 0) &&
                     language != NULL);
         }
         if (two_key == NULL || two_key->type.func == 0)
            two_key = search_tree( c, key_tree.right );
         if (two_key != NULL) {
            if (two_key->type.func < NUM_FUNCS)
               func = two_key->type.func;
            else {
               func = PlayBack;
               g_status.key_macro = two_key->type.macro;
            }
         }
      } else if (c > 256+MAX_KEYS)
         func = (int)c - (256+MAX_KEYS);
      else {
         func = key_func.key[(int)c - 256];
         if (func == PlayBack)
            g_status.key_macro = macro[(int)c - 256];
      }
   }
   return( func );
}


/*
 * Name:    translate_key
 * Purpose: to transform the key to something else
 * Author:  Jason Hood
 * Date:    July 25, 1998
 * Passed:  key:  the key to be translated
 * Returns: the new key;
 *          ERROR if the key should be ignored.
 * Notes:   basically the same as Int15 service 4Fh keyboard intercept.
 * jmh 980726: Translate two-keys directly. Keys are now a long, where normal
 *             ASCII and Extended-ASCII are 0-255, function keys are 256-511,
 *             (pseudo-macros are 8481-65535) and two-keys are 16842753 onwards
 *             (as hi- and lo-word for the two keys, where the hi-key parent
 *             must be a function key).
 */
long translate_key( int key )
{
long new_key = key;
int  func    = 0;
static int two_key = 0;

   if (two_key) {
      show_search_message( CLR_SEARCH );
      new_key = CREATE_TWOKEY( two_key, key );
      two_key = 0;
      func    = getfunc( new_key );

   } else {
      func = getfunc( key );
      if (func == TwoCharKey) {
         /*
          * Next key..
          */
         show_search_message( NEXTKEY );
         two_key = key;
         return( ERROR );
      }
   }

   /*
    * Test for a one-character literal macro.
    */
   if (func == PlayBack && g_status.key_macro->len == 1 &&
                           g_status.key_macro->key.key <= 255) {
      new_key = g_status.key_macro->key.key;
      if (capslock_active())
         new_key = (bj_islower( (int)new_key )) ? bj_toupper( (int)new_key ) :
                                                  bj_tolower( (int)new_key );
#if !defined( __UNIX__ )
   /*
    * Toggle the graphic chararacters. This allows them to be used anywhere
    * (as I feel they should), but at the expense of flexible macro
    * definitions (it wouldn't use the graphic character of the current set,
    * but the recorded character).
    */
   } else if (func == ToggleGraphicChars) {
      toggle_graphic_chars( NULL );     /* Note: dummy argument */
      new_key = ERROR;

   /*
    * Translate numbers and minus to an appropriate graphic character.
    * Use F1 - F5 to select the set:
    *  F1 - single line;
    *  F2 - double line;
    *  F3 - single horizontal, double vertical;
    *  F4 - double horizontal, single vertical.
    *  F5 - solid (24568), "arrows" (79), stipples (013), and small block (-).
    */
   } else if (g_status.graphic_chars > 0) {
      if (new_key >= _F1 && new_key <= _F5) {
         g_status.graphic_chars = key - _F1 + 1;
         show_graphic_chars( );
         new_key = ERROR;
      } else if ((new_key >= '0' && new_key <= '9') || new_key == '-') {
         key     = (key == '-') ? 10 : key - '0';
         new_key = graphic_char[g_status.graphic_chars-1][key];
      }
#endif
   }

   return( new_key );
}


/*
 * Name:    get_name
 * Purpose: To prompt the user and read the string entered in response.
 * Date:    June 5, 1992
 * Passed:  prompt: prompt to offer the user
 *          line:   line to display prompt
 *          name:   default answer
 * Returns: name:   user's answer
 *          OK if user entered something
 *          ERROR if user aborted the command
 * Notes:   with the addition of macros in tde, this function became a little
 *           more complicated.  we have to deal with both executing macros
 *           and macros that are the user uses when entering normal text
 *           at the prompt.  i call these local and global macros.  a global
 *           macro is when this function is called from a running macro.
 *           the running macro can enter text and return from this function
 *           w/o any action from the user.  a local macro is when the user
 *           presses a key inside this function, which happens quite often
 *           when keys are assigned to ASCII and Extended ASCII characters.
 *
 * jmh 980718: If a global macro executes another macro, make it the local
 *              macro. If the global macro finishes before entry is complete
 *              (which can only happen from a config definition) then assume
 *              Rturn rather than AbortCommand.
 *
 * jmh 980727: If a global macro pauses, continue input from the keyboard.
 * jmh 980731: Use NextLine and BegNextLine (ie. Shift+ & Ctrl+Enter) to copy
 *              the word and full-word from the current window.
 * jmh 980813: moved from utils.c and added WordLeft and WordRight functions.
 *             Removed color parameter; always use g_display.message_color.
 */
int  get_name( char *prompt, int line, char *name )
{
int  col;                       /* cursor column for answer */
long c = 0;                     /* character user just typed */
char *cp;                       /* cursor position in answer */
char *answer;                   /* user's answer */
int first = TRUE;               /* first character typed */
register int len;               /* length of answer */
int  plen;                      /* length of prompt */
int  func;                      /* function of key pressed */
int  stop;                      /* flag to stop getting characters */
char *p;                        /* for copying text in answer */
char buffer[MAX_COLS+2];        /* line on which name is being entered */
int  normal;     
MACRO *local_macro = NULL;
int  next = 0;   
int  paused = FALSE;            /* Is a global macro wanting keyboard input? */
int  i;          
DISPLAY_BUFF;
int  color = g_display.message_color;

   /*
    * set up prompt and default
    */
   assert( strlen( prompt ) < (size_t)g_display.ncols );
   assert( strlen( name )   < (size_t)g_display.ncols );

   strcpy( buffer, prompt );
   plen = strlen( prompt );
   answer = buffer + plen;
   strcpy( answer, name );

   /*
    * let user edit default string
    */
   len = strlen( answer );
   col = strlen( buffer );
   cp  = answer + len;
   normal = g_display.text_color;
   save_screen_line( 0, line, display_buff );
   s_output( buffer, line, 0, color );
   eol_clear( col, line, normal );
   for (stop = FALSE; stop == FALSE;) {
      xygoto( col, line );
      if (local_macro == NULL) {
         if (g_status.macro_executing && !paused) {
            if (g_status.macro_next < g_status.cur_macro->len) {
               c    = g_status.cur_macro->key.keys[g_status.macro_next++];
               func = getfunc( c );
               if (func == Pause)
                  paused = TRUE;
            } else {
               c    = RTURN;
               func = Rturn;
            }
         } else {
            c = getkey( );
            /*
             * User may have redefined the Enter and ESC keys.  Make the Enter
             *  key perform a Rturn in this function. Make the ESC key do an
             *  AbortCommand.
             * jmh 980809: test for F1/Help here as well.
             */
            func = (c == RTURN) ? Rturn        :
                   (c == ESC)   ? AbortCommand :
                   (c == _F1)   ? Help         :
                   getfunc( c );

            if (mode.record && !paused) {
               record_key( c, func );
               if (func == Pause)
                  paused = TRUE;
            }
         }
         if (func == PlayBack) {
            local_macro = g_status.key_macro;
            if (local_macro->len == 1) {
               c = local_macro->key.key;
               local_macro = NULL;
            } else {
               next = 0;
               c    = local_macro->key.keys[next++];
            }
            func = getfunc( c );
         }
      } else {
         if (next < local_macro->len)
            c = local_macro->key.keys[next++];
         else {
            local_macro = NULL;
            c = 0x100;
         }
         func = getfunc( c );
      }

      switch (func) {
         case Help :
            if (show_help( ) == ERROR) {
               s_output( buffer, line, 0, color );
               eol_clear( col, line, normal );
               s_output( cp, line, col, color );
            }
            break;
         case ToggleSearchCase :
            toggle_search_case( NULL );      /* Note: dummy argument */
            break;
         case ToggleOverWrite :
            toggle_overwrite( NULL );        /* Note: dummy argument */
            break;
         case NextLine    :
         case BegNextLine :
            if (g_status.current_window != NULL) {
               if (first) {
                  len = 0;
                  cp  = answer;
                  *cp = '\0';
                  col = plen;
                  eol_clear( col, line, normal );
               }
               i = copy_word( g_status.current_window, cp,
                              g_display.ncols - col - 1,
                              func == BegNextLine );
               if (i > 0) {
                  if (mode.insert)
                     len += i;
                  else
                     if (i + (col - plen) > len)
                        len = (col - plen) + i;
                  answer[len] = '\0';
                  s_output( cp, line, col, color );
                  cp  += i;
                  col += i;
               }
            }
            break;
         case Rturn       :
            answer[len] = '\0';
            assert( strlen( answer ) < (size_t)g_display.ncols );
            strcpy( name, answer );
            /*
             * finished
             */
            stop = TRUE;
            break;
         case BackSpace :
            /*
             * delete to left of cursor
             */
            if (cp > answer) {
               for (p=cp-1; p < answer+len; p++) {
                  *p = *(p+1);
               }
               --len;
               --col;
               --cp;
               c_output( ' ', plen+len, line, normal );
               s_output( cp, line, col, color );
               *(answer + len) = '\0';
            }
            break;
         case DeleteChar :
            /*
             * delete char under cursor
             */
            if (*cp) {
               for (p=cp; p < answer+len; p++) {
                  *p = *(p+1);
               }
               --len;
               c_output( ' ', plen+len, line, normal );
               s_output( cp, line, col, color );
               *(answer + len) = '\0';
            }
            break;
         case DeleteLine :
            /*
             * delete current line
             */
            col = plen;
            cp  = answer;
            *cp = '\0';
            len = 0;
            eol_clear( col, line, normal );
            break;
         case AbortCommand :
            stop = TRUE;
            break;
         case CharLeft :
            /*
             * move cursor left
             */
            if (cp > answer) {
               col--;
               cp--;
            }
            break;
         case CharRight :
            /*
             * move cursor right
             */
            if (*cp) {
               col++;
               cp++;
             }
             break;
         case WordLeft :
            /*
             * move cursor to beginning of word
             */
            if (cp > answer) {
               --cp;
               --col;
               while (cp > answer && myiswhitespc( *cp )) {
                  --cp;
                  --col;
               }
               while (cp > answer && !myiswhitespc( *cp )) {
                  --cp;
                  --col;
               }
               if (myiswhitespc( *cp )) {
                  ++cp;
                  ++col;
               }
            }
            break;
         case WordRight :
            /*
             * move cursor to beginning of next word
             */
            while (*cp && !myiswhitespc( *cp )) {
               ++cp;
               ++col;
            }
            while (*cp && myiswhitespc( *cp )) {
              ++cp;
              ++col;
            }
            break;
         case BegOfLine :
            /*
             * move cursor to start of line
             */
            col = plen;
            cp  = answer;
            break;
         case EndOfLine :
            /*
             * move cursor to end of line
             */
            col = plen + len;
            cp  = answer + len;
            break;
         default :
            if (c < 0x100) {
               /*
                * insert character at cursor
                */
               if (first) {
                  /*
                   * delete previous answer
                   */
                  col = plen;
                  cp  = answer;
                  *cp = '\0';
                  len = 0;
                  eol_clear( col, line, normal );
               }

               /*
                * insert new character
                */
               if (col < g_display.ncols-1) {
                  if (*cp == '\0') {
                     ++len;
                     *(answer + len) = '\0';
                  }
                  if (mode.insert && *cp) {
                     for (p=answer+len; p >= cp; p--) {
                        *(p+1) = *p;
                     }
                     ++len;
                     *cp = (char)c;
                     s_output( cp, line, col, color );
                  } else {
                     *cp = (char)c;
                     c_output( (int)c, col, line, color );
                  }
                  ++cp;
                  ++col;
               }
            }
            break;
      }
      if (func != Pause)
         first = FALSE;
   }
   restore_screen_line( 0, line, display_buff );
   return( func == AbortCommand ? ERROR : OK );
}


/*
 * Name:    copy_word
 * Purpose: Copy a word from the current window into a buffer
 * Author:  Jason Hood
 * Date:    July 31, 1998
 * Passed:  window:  current window
 *          buffer:  buffer to store word
 *          len:     length of buffer
 *          full:    non-zero to copy a full-word
 * Returns: number of characters copied.
 * Notes:   uses the mode.insert flag.
 *          If the current window is not on a word, does nothing.
 *          Expects buffer to be zero-terminated, but does not leave
 *           the new buffer zero-terminated.
 */
int  copy_word( TDE_WIN *window, char *buffer, int len, int full )
{
text_ptr line = window->ll->line;
int llen = window->ll->len;
int rcol;
int (*space)( int );    /* Function to determine what a word is */
int end;

   if (llen == EOF)
      return( 0 );

   rcol = window->rcol;
   if (mode.inflate_tabs)
      rcol = entab_adjust_rcol( line, llen, rcol );

   space = (full) ? bj_isspc : myiswhitespc;
   if (space( line[rcol] ))
      return 0;

   end = rcol;

   while (--rcol >= 0 && !space( line[rcol] )) ;
   ++rcol;
   while (++end < llen && !space( line[end] )) ;
   llen = end - rcol;
   end = (mode.insert) ? strlen( buffer ) : 0;
   if (llen > len - end)
      llen = len - end;

   if (mode.insert)
      memmove( buffer + llen, buffer, end );
   my_memcpy( (text_ptr)buffer, line + rcol, llen );

   return( llen );
}


/*
 * Name:    get_response
 * Purpose: to prompt the user and wait for a key to be pressed
 * Date:    August 13, 1998
 * Author:  Jason Hood
 * Passed:  prompt: prompt to offer the user
 *          line:   line to display prompt
 *          flag:   see below
 *          num:    number of responses
 *          let:    first response letter
 *          res:    first response code
 * Returns: the appropriate response for the letter
 * Notes:   prompt is displayed using set_prompt.
 *          If prompt is NULL, no prompt is output, line has no meaning and the
 *           R_PROMPT flag is ignored. The cursor remains unchanged.
 *          If the flag contains R_PROMPT, the response letters are displayed
 *           in brackets, in lower-case, separated by slashes, followed by a
 *           colon and space.
 *          If the flag contains R_MACRO, allow the response to be read from,
 *           or recorded to, a macro.
 *          If the flag contains R_DEFAULT, pressing RTURN/Rturn will accept
 *           the first response (ie. res).
 *          If the flag contains R_ABORT, pressing ESC/AbortCommand will cancel
 *           the query and return ERROR.
 */
int  get_response( char* prompt, int line, int flag,
                   int num, int let1, int res1, ... )
{
long c;                 /* the user's response */
register int rc;        /* return code */
int  let[20];           /* twenty responses should be plenty */
int  res[20];
char buf[MAX_COLS+2];
int  col = 0;
int  i;
va_list ap;
DISPLAY_BUFF;

   assert( num < 20 );

   save_screen_line( 0, line, display_buff );

   let[0] = let1;
   res[0] = res1;
   va_start( ap, res1 );
   for (col = 1; col < num; ++col) {
      let[col] = bj_toupper( va_arg( ap, int ) );
      res[col] = va_arg( ap, int );
   }
   va_end( ap );

   if (prompt != NULL) {
      col = strlen( prompt );
      strcpy( buf, prompt );
      if (flag & R_PROMPT) {
         buf[col++] = ' ';
         buf[col++] = '(';
         for (i = 0; i < num; ++i) {
            buf[col++] = bj_tolower( let[i] );
            buf[col++] = '/';
         }
         buf[col-1] = ')';
         buf[col++] = ':';
         buf[col++] = ' ';
         buf[col++] = '\0';
      }
      set_prompt( buf, line );
   }
   for (;;) {
      c  = (flag & R_MACRO) ? getkey_macro( ) : getkey( );
      rc = 0;
      if (c < 256)
         c = bj_toupper( (int)c );
      else
         rc = (c == RTURN) ? Rturn        :
              (c == ESC)   ? AbortCommand :
              getfunc( c );

       if (rc == Rturn && (flag & R_DEFAULT)) {
         rc = res1;
         break;
      } else if (rc == AbortCommand && (flag & R_ABORT)) {
         rc = ERROR;
         break;
      } else {
         for (i = 0; i < num && let[i] != (int)c; ++i) ;
         if (i < num) {
            rc = res[i];
            break;
         }
      }
   }
   restore_screen_line( 0, line, display_buff );
   return( rc );
}


/*
 * Name:     get_sort_order
 * Purpose:  To prompt the user and get sort direction
 * Date:     June 5, 1992
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:   window
 * Returns:  OK if user entered something
 *           ERROR if user aborted the command
 *
 * jmh 980813: moved from utils.c
 */
int  get_sort_order( TDE_WIN *window )
{
register int c;

   /*
    * sort ascending or descending
    */
   c = get_response( utils4, window->bottom_line, R_ALL,
                     2, L_ASCENDING, ASCENDING, L_DESCENDING, DESCENDING );
   if (c != ERROR) {
      sort.direction = c;
      c = OK;
   }
   return( c );
}


/*
 * Name:     get_direction
 * Purpose:  To prompt the user and get replace string / match pair direction
 * Date:     October 31, 1992
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:   window
 * Returns:  OK if user entered something
 *           ERROR if user aborted the command
 *
 * jmh 980614: renamed from get_replace_direction and added the prompt
 *             parameter so it could be used with my modifications to
 *             match_pair.
 * jmh 980813: moved from utils.c
 */
int  get_direction( TDE_WIN *window, char *prompt )
{
register int c;
char temp[MAX_COLS+2];

   /*
    * forward or backward
    */
   combine_strings( temp, prompt, " ", utils5b );
   c = get_response( temp, window->bottom_line, R_ALL,
                     2, L_FORWARD, FORWARD, L_BACKWARD, BACKWARD );
   return( c );
}
