/*
 * Now that I own both MSC 7.0 and BC 3.1 and have linux, lets rearrange stuff
 * so many compilers can compile TDE.  Several implementation specific
 * functions needed for several environments were gathered into this file.
 *
 * In version 3.2, these functions changed to support unix.
 *
 * Incidentally, there is a difference between a NULL line pointer and
 * a pointer to a line that contains no characters.  For example, calling
 *
 *       line = malloc( 0 );
 *
 *                   or, more precisely in TDE:
 *
 *       line = _fmalloc( 0 );
 *       line = farmalloc( 0 );
 *
 * will return a valid pointer to an item of 0 length in some compilers
 * and a NULL pointer in other compilers.  malloc( 0 ) will return a valid
 * pointer to an object of zero length in MSC.  malloc( 0 ) will return a
 * NULL pointer in BC.  The problem with returning a NULL pointer for
 * malloc( 0 ) is that it's a little harder to tell if the heap is out of
 * memory or if we have a valid NULL pointer.  On the other hand, the good
 * part about returning a NULL pointer for malloc( 0 ) is that extra space
 * is not wasted for an object of 0 length.  In TDE, we will test for 0
 * before calling my_malloc( ) and set an ERROR code if out of memory.
 *
 * Although many PC C compilers have findfirst and findnext functions for
 * finding files, let's write our own to keep a closer watch on
 * critical errors.
 *
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This code is released into the public domain, Frank Davis.
 * You may use and distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"
#include "define.h"


/*
 * Name:    my_malloc
 * Purpose: malloc from the far heap
 * Date:    April 1, 1993
 * Passed:  mem:  pointer to memory to free in far heap
 *          rc:   pointer to return code
 * Notes:   set the return code only if an ERROR occured with malloc.
 *           returning a NULL pointer is not neccessarily an ERROR.
 */
void FAR * my_malloc( size_t size, int *rc )
{
void FAR *mem;

   assert( size < MAX_LINE_LENGTH );

   if (size == 0)

      /*
       * if 0 bytes are requested, return NULL
       */
      mem = NULL;
   else {

#if defined( __MSC__ )
      mem = _fmalloc( size );
#else
      mem = farmalloc( size );
#endif

      /*
       * if malloc failed, return NULL and an ERROR.
       */
      if (mem == NULL)
         *rc = ERROR;
   }
   return( mem );
}


/*
 * Name:    my_free
 * Purpose: free memory from the far heap
 * Date:    April 1, 1993
 * Passed:  mem:  pointer to memory to free in far heap
 */
void my_free( void FAR *mem )
{
   assert( mem != NULL );

#if defined( __MSC__ )
   _ffree( mem );
#else
   farfree( mem );
#endif
}


/*
 * Name:    my_heapavail
 * Purpose: available free memory from the far heap
 * Date:    November 13, 1993
 */
long my_heapavail( void )
{
long avail_mem;

#if defined( __MSC__ )
unsigned paragraphs;

   _dos_allocmem( 0xffff, &paragraphs );
   /*
    * A paragraph is 16 bytes.  Convert paragraphs to bytes by shifting left
    * 4 bits.
    */
   avail_mem = (long)paragraphs << 4;
#else

   /*
    * use the Borland farcoreleft( ) function.
    */
   avail_mem = farcoreleft( );
#endif
   return( avail_mem );
}


/*
 * Name:    my_memcpy
 * Purpose: copy memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 * Notes:   far memory copy in DOS real mode
 */
void my_memcpy( void FAR *dest, void FAR *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      _fmemcpy( dest, src, size );
   }
}


/*
 * Name:    my_memmove
 * Purpose: move memory
 * Date:    November 13, 1993
 * Passed:  dest: pointer to destination
 *          src:  pointer to source
 *          size: number of bytes to copy
 * Notes:   far memory move in DOS real mode - handles mem overlap
 */
void my_memmove( void FAR *dest, void FAR *src, size_t size )
{
   if (size > 0) {
      assert( dest != NULL );
      assert( src  != NULL );
      _fmemmove( dest, src, size );
   }
}


static int find_first( DTA FAR *dta, char FAR *path, int f_attr );
static int find_next(  DTA FAR *dta );
static FTYPE found;

/*
 * Name:    find_first
 * Purpose: find the first file matching a pattern using DOS interrupt
 * Date:    January 6, 1992
 * Passed:  dta:    disk transfer address
 *          path:   path to search for files
 *          f_attr: attributes of files to search for
 * Notes:   return codes for find_first:
 *             0  no error
 *             2  file is invalid or does not exist
 *             3  path is invalid or does not exist
 *            18  no matching directory entry was found
 *            -1  check the critical error flag for critical errors
 */
static int find_first( DTA FAR *dta, char FAR *path, int f_attr )
{
void FAR *old_dta;
void FAR *new_dta;
int  rc;

   new_dta = (void FAR *)dta;

   ASSEMBLE {

/*
; save the old dta
*/
        mov     ah, 0x2f                /* DOS get dta */
        int     0x21                    /* DOS interrupt */
        mov     WORD PTR old_dta, bx    /* save OFFSET of old DTA */
        mov     ax, es
        mov     WORD PTR old_dta+2, ax  /* save SEGMENT of old DTA */

/*
; set the new dta
*/
        push    ds                      /* save ds */
        mov     dx, WORD PTR new_dta    /* get OFFSET of new dta */
        mov     ax, WORD PTR new_dta+2  /* get SEGMENT of new dta */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x1a                /* DOS set dta */
        int     0x21                    /* DOS interrupt */
        pop     ds                      /* get back ds */

/*
; find first matching file
*/
        push    ds                      /* save ds */
        mov     cx, WORD PTR f_attr     /* file attributes to search for */
        mov     dx, WORD PTR path       /* get OFFSET of path */
        mov     ax, WORD PTR path+2     /* get SEGMENT of path */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x4e                /* DOS find first file */
        int     0x21                    /* DOS interrupt */
        pop     ds                      /* get back ds */

/*
; save the return code
*/
        jc      an_error                /* carry is set if an error occured */
        xor     ax, ax                  /* zero out ax, return OK if no error */
   }
an_error:

   ASSEMBLE {
        mov     WORD PTR rc, ax         /* save the return code */

/*
; get back old dta
*/
        push    ds                      /* save ds */
        mov     dx, WORD PTR old_dta    /* get OFFSET of old dta */
        mov     ax, WORD PTR old_dta+2  /* get SEGMENT of old dta */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x1a                /* DOS set dta */
        int     0x21                    /* DOS interrupt */
        pop     ds                      /* get back ds */
   }
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    find_next
 * Purpose: find the next file matching a pattern using DOS interrupt
 * Date:    January 6, 1992
 * Passed:  dta:  disk transfer address
 * Notes:   find_first() MUST be called before calling this function.
 *          return codes for find_next (see DOS tech ref manuals):
 *             0  no error
 *             2  path is invalid or does not exist
 *            18  no matching directory entry was found
 *            -1  check the critical error flag for critical errors
 */
static int find_next( DTA FAR *dta )
{
void FAR *old_dta;
void FAR *new_dta;
int  rc;

   new_dta = (void FAR *)dta;

   ASSEMBLE {

/*
; save the old dta
*/
        mov     ah, 0x2f                /* DOS get dta */
        int     0x21                    /* DOS interrupt */
        mov     WORD PTR old_dta, bx    /* save OFFSET of old DTA */
        mov     ax, es
        mov     WORD PTR old_dta+2, ax  /* save SEGMENT of old DTA */

/*
; set the new dta
*/
        push    ds                      /* save ds */
        mov     dx, WORD PTR new_dta    /* get OFFSET of new dta */
        mov     ax, WORD PTR new_dta+2  /* get SEGMENT of new dta */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x1a                /* DOS set dta */
        int     0x21                    /* DOS interrupt */
        pop     ds                      /* get back ds */

/*
; find next matching file
*/
        mov     ah, 0x4f                /* DOS find first file */
        int     0x21                    /* DOS interrupt */

/*
; save the return code
*/
        jc      an_error                /* carry is set if an error occured */
        xor     ax, ax                  /* zero out ax, return OK if no error */
   }
an_error:

   ASSEMBLE {
        mov     WORD PTR rc, ax         /* save the return code */

/*
; get back old dta
*/
        push    ds                      /* save ds */
        mov     dx, WORD PTR old_dta    /* get OFFSET of old dta */
        mov     ax, WORD PTR old_dta+2  /* get SEGMENT of old dta */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x1a                /* DOS set dta */
        int     0x21                    /* DOS interrupt */
        pop     ds                      /* get back ds */
   }
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    my_findfirst
 * Purpose: find the first file matching a pattern
 * Date:    August 5, 1997
 * Passed:  path: path and pattern to search for files
 *          dta:  file finding info
 *          dirs: TRUE to search for directories and return file sizes
 * Notes:   Returns NULL for no matching files or bad pattern;
 *          otherwise a pointer to a static FTYPE, with fname holding the
 *          filename that was found.
 *          If dirs is TRUE fsize will hold the size of the file, and the
 *          name will have a trailing slash ('/') if it's a directory.
 *          The name is converted to lower-case.
 */
FTYPE * my_findfirst( char *path, FFIND *dta, int dirs )
{
int i;
fattr_t fattr;
char temp[PATH_MAX];

   dta->dirs = dirs;
   /*
    * Separate the path and pattern
    */
   i = strlen( path ) - 1;
   if (i == -1) {
      get_current_directory( dta->stem, 0 );
      strcpy( dta->pattern, "*" );
   }
   else if (path[i] == '/' || path[i] == '\\' || path[i] == ':') {
      strcpy( dta->stem, path );
      strcpy( dta->pattern, "*" );
   } else if (get_fattr( path, &fattr ) == OK && (fattr & SUBDIRECTORY)) {
      strcpy( dta->stem, path );
      strcat( dta->stem, "/" );
      strcpy( dta->pattern, "*" );
   } else {
      for (--i; i >= 0; --i) {
         if (path[i] == '/' || path[i] == ':')
            break;
         /* if it's a backslash, it could be an escape for the pattern */
         if (path[i] == '\\' && strchr( "!^-\\]", path[i+1] ) == NULL)
            break;
      }
      if (i >= 0) {
         strncpy( dta->stem, path, ++i );
         dta->stem[i] = '\0';
         strcpy( dta->pattern, path+i );
      } else {
         get_current_directory( dta->stem, 0 );
         strcpy( dta->pattern, path );
      }
      if (!is_valid_pattern( dta->pattern, &i ))
         return NULL;
   }
   /*
    * Start scanning the directory
    */
   strcpy( temp, dta->stem );
   strcat( temp, "*.*" );
   fattr = NORMAL | READ_ONLY | HIDDEN | SYSTEM | ARCHIVE;
   if (dirs) fattr |= SUBDIRECTORY;
   i = find_first( &dta->find_info, temp, fattr );
   if (i != OK) return NULL; /* empty directory? */
   do {
      if (dirs && (dta->find_info.attrib & SUBDIRECTORY)) break;
      if (wildcard( dta->pattern, dta->find_info.name ) == TRUE) break;
      i = find_next( &dta->find_info );
   } while (i == OK);
   if (i != OK) return NULL; /* nothing matched */

   /* ignore the "." entry (assuming it to be found in this initial scan) */
   if (dirs && dta->find_info.name[0] == '.' && dta->find_info.name[1] == '\0')
      return my_findnext( dta );

   strcpy( found.fname, dta->find_info.name );
   for (i = 0; found.fname[i]; ++i)
      found.fname[i] = bj_tolower( found.fname[i] );
   if (dirs) {
      found.fsize = dta->find_info.size;
      found.fattr = dta->find_info.attrib;
      if (dta->find_info.attrib & SUBDIRECTORY) {
         found.fname[i] = '/';
         found.fname[i+1] = '\0';
      }
   }
   return &found;
}


/*
 * Name:    my_findnext
 * Purpose: find the next file matching a pattern
 * Date:    August 5, 1997
 * Passed:  dta: file finding info
 * Notes:   my_findfirst() MUST be called before calling this function.
 *          Returns NULL if no more matching names;
 *          otherwise same as my_findfirst.
 */
FTYPE * my_findnext( FFIND *dta )
{
int i;

   i = find_next( &dta->find_info );
   if (i != OK) return NULL;
   do {
      if (dta->dirs && (dta->find_info.attrib & SUBDIRECTORY)) break;
      if (wildcard( dta->pattern, dta->find_info.name ) == TRUE) break;
      i = find_next( &dta->find_info );
   } while (i == OK);
   if (i != OK) return NULL; /* all done */
   strcpy( found.fname, dta->find_info.name );
   for (i = 0; found.fname[i]; ++i)
      found.fname[i] = bj_tolower( found.fname[i] );
   if (dta->dirs) {
      found.fsize = dta->find_info.size;
      found.fattr = dta->find_info.attrib;
      if (dta->find_info.attrib & SUBDIRECTORY) {
         found.fname[i] = '/';
         found.fname[i+1] = '\0';
      }
   }
   return &found;
}


/*
 * Name:    hw_fattrib
 * Purpose: To determine the current file attributes.
 * Date:    December 26, 1991
 * Passed:  name: name of file to be checked
 * Returns: use the function in the tdeasm file to get the DOS file
 *          attributes.  get_fattr() returns 0 or OK if no error.
 */
int  hw_fattrib( char *name )
{
register int rc;
fattr_t fattr;

   rc = get_fattr( name, &fattr );
   return( rc == OK ? rc : ERROR );
}


/*
 * Name:    change_mode
 * Purpose: To prompt for file access mode.
 * Date:    January 11, 1992
 * Passed:  name:  name of file
 *          line:  line to display message
 * Returns: OK if file could be changed
 *          ERROR otherwise
 * Notes:   function is used to change file attributes for save_as function.
 */
int  change_mode( char *name, int line )
{
int    result;
fattr_t fattr;
register int rc;

   rc = OK;
   result = get_fattr( name, &fattr );
   if (result != OK)
      rc = ERROR;
   else if (result == OK && fattr & READ_ONLY) {
      /*
       * file is write protected. overwrite anyway?
       */
      if (get_yn( main6, line, R_PROMPT | R_ABORT ) != A_YES)
         rc = ERROR;
      if (rc == OK && set_fattr( name, ARCHIVE ) != OK)
         rc = ERROR;
   }
   return( rc );
}


/*
 * Name:    change_fattr
 * Purpose: To change the file attributes
 * Date:    December 31, 1991
 * Passed:  window:  pointer to current window
 */
int  change_fattr( TDE_WIN *window )
{
file_infos   *file;
TDE_WIN      *wp;
int          prompt_line;
register int ok;
fattr_t      fattr;
char         *s;
int          rc;
char         answer[MAX_COLS+2];

   prompt_line = window->bottom_line;

   answer[0] = '\0';
   /*
    * enter new file attributes
    */
   if ((ok = get_name( utils14, prompt_line, answer )) == OK) {
      if (*answer != '\0') {
         fattr = 0;
         s = answer;

         /*
          * yes, I know lint complains about "ok = *s++".
          * jmh - added ",ok" to correct it
          */
         while (ok = bj_toupper( *(s++) ),ok) {
            switch (ok) {
               case L_DOS_ARCHIVE :
                  fattr |= ARCHIVE;
                  break;
               case L_DOS_SYSTEM :
                  fattr |= SYSTEM;
                  break;
               case L_DOS_HIDDEN :
                  fattr |= HIDDEN;
                  break;
               case L_DOS_READ_ONLY :
                  fattr |= READ_ONLY;
                  break;
               default :
                  break;
            }
         }
         file = window->file_info;
         if (set_fattr( file->file_name, fattr ))
            /*
             * new file attributes not set
             */
            error( WARNING, prompt_line, utils15 );
         else {
            file->file_attrib = fattr;
            for (wp=g_status.window_list; wp!=NULL; wp=wp->next) {
               if (wp->file_info == file && wp->visible)
                  show_window_fname( wp );
            }
         }
      }
      rc = OK;
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_fattr
 * Purpose: To get dos file attributes
 * Date:    December 26, 1991
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: pointer to file attributes
 * Returns: 0 if successfull, non zero if not
 * Notes:   Uses the DOS function to get file attributes.  I really didn't
 *           like the file attribute functions in the C library:  fstat() and
 *           stat() or access() and chmod().
 *           FYI, File Attributes:
 *              0x00 = Normal.  Can be read or written w/o restriction
 *              0x01 = Read-only.  Cannot be opened for write; a file with
 *                     the same name cannot be created.
 *              0x02 = Hidden.  Not found by directory search.
 *              0x04 = System.  Not found by directory search.
 *              0x08 = Volume Label.
 *              0x10 = Directory.
 *              0x20 = Archive.  Set whenever the file is changed, or
 *                     cleared by the Backup command.
 *           Return codes:
 *              0 = No error
 *              1 = AL not 0 or 1
 *              2 = file is invalid or does not exist
 *              3 = path is invalid or does not exist
 *              5 = Access denied
 */
int  get_fattr( char FAR *fname, fattr_t *fattr )
{
int  rc;                /* return code */
int  attr;

   ASSEMBLE {
        push    ds
        mov     dx, WORD PTR fname      /* get OFFSET of filename string */
        mov     ax, WORD PTR fname+2    /* get SEGMENT of filename string */
        mov     ds, ax                  /* put SEGMENT in ds */
        mov     ax, 0x4300              /* function:  get file attributes */
        int     0x21                    /* DOS interrupt */
        pop     ds

        jc      an_error                /* save the error code from get attr */
        xor     ax, ax                  /* if no carry, no error */
        jmp     SHORT get_out           /* lets get out */
   }
an_error:


   ASSEMBLE {
        xor     cx, cx                  /* if error, then zero out cx - attrs */
   }
get_out:

   ASSEMBLE {
        mov     WORD PTR rc, ax         /* ax contains error number on error */
        mov     WORD PTR attr, cx       /* cx contains file attributes */
   }
   *fattr = attr;
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    set_fattr
 * Purpose: To set dos file attributes
 * Date:    December 26, 1991
 * Passed:  fname: ASCIIZ file name.  Null terminated file name
 *          fattr: file attributes
 * Returns: 0 if successfull, non zero if not
 * Notes:   Uses the DOS function to get file attributes.
 *           Return codes:
 *              0 = No error
 *              1 = AL not 0 or 1
 *              2 = file is invalid or does not exist
 *              3 = path is invalid or does not exist
 *              5 = Access denied
 */
int  set_fattr( char FAR *fname, fattr_t fattr )
{
int  rc;                /* return code */

   ASSEMBLE {
        push    ds
        mov     dx, WORD PTR fname      /* get OFFSET of filename string */
        mov     ax, WORD PTR fname+2    /* get SEGMENT of filename string */
        mov     ds, ax                  /* put SEGMENT in ds */
        mov     cx, WORD PTR fattr      /* cx contains file attributes */
        mov     ax, 0x4301              /* function:  get file attributes */
        int     0x21                    /* DOS interrupt */
        pop     ds

        jc      get_out                 /* save the error code from get attr */
        xor     ax, ax                  /* if no carry, no error */
   }
get_out:

   ASSEMBLE {
        mov     WORD PTR rc, ax         /* ax contains error number on error */
   }
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_current_directory
 * Purpose: get current directory
 * Date:    February 13, 1992
 * Passed:  path:  pointer to buffer to store path
 *          drive: drive to get current directory
 * Notes:   use simple DOS interrupt
 *          path is expected to be at least PATH_MAX long
 *          if drive isn't zero, the drive specification is included
 *          In the interests of uniform behaviour, convert backslashes to
 *           slashes (ie. '\' to '/') and use lower-case. Append the
 *           directory with a final slash.
 *
 * jmh 980503: store the drive specifier as well.
 */
int  get_current_directory( char FAR *path, int drive )
{
int  rc;
int  i;

   ASSEMBLE {
        push    si                      /* save register vars if any */
        push    ds                      /* save ds */

        mov     dx, WORD PTR drive      /* dl = drive, 0 = default, 1 = a,... */
        mov     si, WORD PTR path       /* get OFFSET of path */
        mov     ax, WORD PTR path+2     /* get SEGMENT of path */
        mov     ds, ax                  /* put it in ds */
        mov     al, dl                  /* get the drive */
        or      al, al
        jne     drive_let               /* drive specified */
        mov     ah,0x19                 /* get default drive, 0 = a,... */
        int     0x21
        inc     ax                      /* make 1 = a,... */
   }
drive_let:
   ASSEMBLE {
        add     al, 'A' - 1             /* make it an uppercase letter */
        mov     ah, ':'
        mov     WORD PTR [si], ax       /* store the drive specification */
        inc     si
        inc     si
        mov     BYTE PTR [si], '/'
        inc     si
        mov     ah, 0x47                /* function 0x47 == get current dir */
        int     0x21                    /* standard DOS interrupt */
        mov     ax, ERROR               /* return -1 if error */
        jc      error                   /* if carry set, then an error */
        xor     ax, ax                  /* zero out ax, return OK if no error */
   }
error:

   ASSEMBLE {
        pop     ds                      /* get back ds */
        pop     si                      /* get back si */
        mov     WORD PTR rc, ax         /* save return code */
   }
   if (ceh.flag == ERROR)
      rc = ERROR;
   else {
      for (i = 0; path[i]; ++i) {
         if (path[i] == '\\') path[i] = '/';
         else path[i] = bj_tolower( path[i] );
      }
      if (path[i-1] != '/') {
         path[i] = '/';
         path[i+1] = '\0';
      }
   }
   return( rc );
}


/*
 * Name:    set_current_directory
 * Purpose: set current directory
 * Date:    February 13, 1992
 * Passed:  new_path: directory path, which may include drive letter
 * Notes:   use simple DOS interrupt
 */
int  set_current_directory( char FAR *new_path )
{
int  rc;

   ASSEMBLE {
        push    ds                      /* save ds */

        mov     dx, WORD PTR new_path   /* get OFFSET of new_path */
        mov     ax, WORD PTR new_path+2 /* get SEGMENT of new_path */
        mov     ds, ax                  /* put it in ds */
        mov     ah, 0x3b                /* function 0x3b == set current dir */
        int     0x21                    /* standard DOS interrupt */
        mov     ax, ERROR               /* return -1 if error */
        jc      error                   /* if carry set, then an error */
        xor     ax, ax                  /* zero out ax, return OK if no error */
   }
error:

   ASSEMBLE {
        pop     ds                      /* get back ds */
        mov     WORD PTR rc, ax         /* save return code */
   }
   if (ceh.flag == ERROR)
      rc = ERROR;
   return( rc );
}


/*
 * Name:    get_full_path
 * Purpose: retrieve the fully-qualified path name for a file
 * Date:    May 3, 1998
 * Passed:  in_path:  path to be canonicalized
 *          out_path: canonicalized path
 * Notes:   out_path is assumed to be PATH_MAX characters.
 *          out_path is converted to lower-case and backslashes to slashes.
 * 980511:  wrote an assembly version.
 */
void get_full_path( char FAR *in_path, char FAR *out_path )
{
int i;

   ASSEMBLE {
        push    ds
        push    es
        push    si
        push    di
        lds     si, DWORD PTR in_path   /* ds:si = in_path */
        les     di, DWORD PTR out_path  /* es:di = out_path */
        mov     ah, 0x60                /* function 0x60 == truename */
        int     0x21
        jnc     out                     /* no error */

        push    es                      /* Find the length of in_path */
        push    ds                      /* scasb wants es:di, but in_path */
        pop     es                      /* is in ds:si, so swap */
        xchg    si,di
        push    di
        xor     al,al                   /* search for the terminating nul */
        mov     cx,0xffff
        cld
        repne   scasb
        neg     cx
        dec     cx
        pop     di
        pop     es
        xchg    si,di
        rep     movsb                   /* strcpy( out_path, in_path ) */
   }
out:
   ASSEMBLE {
        pop     di
        pop     si
        pop     es
        pop     ds
   }

   for (i = 0; out_path[i]; ++i) {
      if (out_path[i] == '\\') out_path[i] = '/';
      else out_path[i] = bj_tolower( out_path[i] );
   }
}
