/* Copyright (C) Ken R. Lunde, 1991-1992.  All rights reserved. */

/* These routines originally came from Ken R. Lunde's JCONV.C program, */
/* version 2.3, August 12, 1992.  Many thanks to Ken for his donation. */

/* The original code was reformatted (to fit the style of the rest     */
/* of the program).  Because EUC is used as the internal representa-   */
/* tion, code corresponding to conversions not involving EUC is        */
/* deleted.  Also, the original file-based character I/O machanism     */
/* is changed.                                                         */

/* A modified version of the original documentation follows...         */

/*
Program: jconv.c
Version: 2.3
Date:    August 12, 1992
Author:  Ken R. Lunde, Adobe Systems Incorporated
  EMAIL: lunde@mv.us.adobe.com
  MAIL : 1585 Charleston Road, P.O. Box 7900, Mountain View, CA 94039-7900
Type:    A tool for converting the Japanese code of Japanese textfiles.
Code:    ANSI C (portable)

PORTABILITY:
This source code was written so that it would be portable on C compilers which
conform to the ANSI C standard. It has been tested on a variety of compilers.

I used THINK C and GNU C as my development platforms. I left in the Macintosh-
specific lines of code so that it would be easier to enhance/debug this tool
later. For those of you who wish to use this tool on the Macintosh, simply
add the ANSI library to the THINK C project, and then build the application.
Be sure that THINK_C has been defined, though, as the conditional compilation
depends on it. You then have a double-clickable application, which when
launched, will greet you with a Macintosh-style interface.

DISTRIBUTION AND RESTRICTIONS ON USAGE:
 1) Please give this source code away to your friends at no charge.
 2) Please try to compile this source code on various platforms to check for
    portablity, and please report back to me with any results be they good or
    bad. Suggestions are always welcome.
 3) Only use this tool on a copy of a file -- do not use an original. This
    is just common sense.
 4) This source code or a compiled version may be bundled with commercial
    software as long as the author is notified beforehand. The author's name
    should also be mentioned in the credits.
 5) Feel free to use any of the algorithms for your own work. Many of them are
    being used in other tools I have written.
 6) The most current version can be obtained by requesting a copy directly
    from me.

DESCRIPTION:
 1) Supports Shift-JIS, EUC, New-JIS, Old-JIS, and NEC-JIS for both input and
    output.
 2) Automatically detects infile's Japanese code (the ability to force an
    input Japanese code is also supported through a command-line option).
 3) The ability to convert Shift-JIS and EUC half-width katakana into full-
    width equivalents. Note that half-width katakana includes other symbols
    such as a Japanese period, Japanese comma, center dot, etc.
 4) Supports conversion between the same code (i.e., EUC -> EUC, Shift-JIS ->
    Shift-JIS, etc.). This is useful as a filter for converting half-width
    katakana to their full-width equivalents.
 5) If the infile does not contain any Japanese, then its contents are
    echoed to the outfile. It will also try to repair the file as though
    it had stripped escape characters (see #6 below).
 6) The functionality of my other tool called repair-jis.c is included in
    this tool by using the "-r[CODE]" option. This will recover stripped
    escape characters, then convert the file to be in CODE format.
*/

#include "jwp.h"

#include <ctype.h>
#include <errno.h>


#define TOFULLSIZE		TRUE
#define UNGETBUFSIZ 	10

#define ReadChar()      ((stackp > 0) ? UngetBuf[--stackp] : f_charin())
#define WriteChar(x)    f_charout(x)
#define UnreadChar(x)   UngetBuf[stackp++] = (x)



#define FF_EUCSJIS  (FF_NEC + 1)

#define NUL         0
#define NL          10
#define FF          12
#define CR          13
#define ESC         27

#define SJIS1(A)    ((A >= 129 && A <= 159) || (A >= 224 && A <= 239))
#define SJIS2(A)    (A >= 64 && A <= 252)
#define HANKATA(A)  (A >= 161 && A <= 223)
#define ISEUC(A)    (A >= 161 && A <= 254)
#define ISMARU(A)   (A >= 202 && A <= 206)
#define ISNIGORI(A) ((A >= 182 && A <= 196) || (A >= 202 && A <= 206))

void StraightEcho (void);
void HanToZen(int *p1, int *p2, FILEFORMAT incode);
void SJisToJis(int *p1, int *p2);
void JisToSJis(int *p1, int *p2);
void ShiftToEuc(FILEFORMAT incode);
void EucToSeven(FILEFORMAT incode, char *ki, char *ko);
void EucToEuc(FILEFORMAT incode);
void EucToShift(FILEFORMAT incode);
void SevenToEuc(void);
void JisRepair(FILEFORMAT outcode, char *ki, char *ko);
BOOL SkipESCSeq(int ch, int *TwoBytes);
FILEFORMAT DetectCodeType(void);



/* The I/O Routines */

static int (* f_charin)(void);
static void (* f_charout)(int);

static int UngetBuf[UNGETBUFSIZ];
static int stackp;




void SetupIO (int (* in)(void), void (* out)(int))
{
	f_charin = in;
	f_charout = out;
	stackp = 0;
}


static void OutPuts (char *buf)
{
    for (; *buf; buf++) WriteChar(*buf);
}



/* Convert input file ==> EUC */

void FileImport (FILEFORMAT incode)
{
    switch (incode) {
		case FF_UNKNOWN:    StraightEcho(); break;

		case FF_OLDJIS:
        case FF_NEWJIS:
		case FF_NEC:        SevenToEuc(); break;

		case FF_EUC:        EucToEuc(incode); break;

		case FF_SJIS:       ShiftToEuc(incode); break;
	}
}


/* Convert internal EUC ==> Output format */

void FileExport (FILEFORMAT outcode)
{
	switch (outcode) {
        case FF_NEWJIS:     EucToSeven(FF_EUC, "$B", "(J"); break;
        case FF_OLDJIS:     EucToSeven(FF_EUC, "$@", "(J"); break;
		case FF_NEC:        EucToSeven(FF_EUC, "K", "H"); break;

		case FF_UNKNOWN:
		case FF_EUC:        /*EucToEuc(FF_EUC); */ StraightEcho(); break;

		case FF_SJIS:       EucToShift(FF_EUC); break;
	}
}


#ifdef FOOBAR
void main(int argc,char **argv)
{
  *out;
#ifndef THINK_C
  int rc;
#endif
  int tempincode,incode,doing = FALSE,forcedelesc = FALSE;
  int makeoutfile = TRUE,outcode = FF_UNKNOWN,delesc = FALSE;
  int repairjis = FALSE,TOFULLSIZE = FALSE,setincode = FALSE,docheck = FALSE;
  char infilename[100],outfilename[100],extension[10],ki[10],ko[10],toolname[20];

#ifdef THINK_C
  argc = ccommand(&argv);
#endif

  strcpy(toolname,*argv);
  while (--argc > 0 && (*++argv)[0] == '-')
	switch (ToUpperCase(*++argv[0])) {
      case 'C' :
        docheck = TRUE;
        break;
      case 'F' :
		TOFULLSIZE = TRUE;
        break;
      case 'H' :
		dohelp(toolname);
        break;
      case 'I' :
        setincode = TRUE;
        doing = INPUT;
		incode = getcode(extension,ToUpperCase(*++argv[0]),ki,ko,doing);
        break;
      case 'O' :
        doing = OUTPUT;
        outcode = getcode(extension,ToUpperCase(*++argv[0]),ki,ko,doing);
		break;
      case 'R' :
        repairjis = TRUE;
		doing = REPAIR;
        outcode = getcode(extension,ToUpperCase(*++argv[0]),ki,ko,doing);
        break;
      case 'S' :
        delesc = TRUE;
		strcpy(extension,".rem");
        if (ToUpperCase(*++argv[0]) == 'F')
          forcedelesc = TRUE;
        break;
      case 'T' :
		switch (ToUpperCase(*++argv[0])) {
          case 'E' :
            doeuctable();
			break;
          case 'J' :
          case 'N' :
          case 'O' :
            dojistable();
			break;
          case 'S' :
            dosjistable();
            break;
          default :
			dojistable();
            dosjistable();
            doeuctable();
			break;
        }
        exit(0);
        break;
      case 'V' :
		break;
      default :
        fprintf(stderr,"Illegal option \"-%c\"! Try using the \"-h\" option for help.\n",*argv[0]);
        fprintf(stderr,"Usage: %s [-options] [infile] [outfile]\nExiting...\n",toolname);
        exit(1);
		break;
    }
  if (repairjis && delesc) {
	fprintf(stderr,"Error! Both \"-r\" and \"-s\" options cannot be selected! Exiting...\n");
    exit(1);
  }
  if (outcode == FF_UNKNOWN && !repairjis && !delesc) {
    strcpy(ki,DEFAULT_OKI);
	strcpy(ko,DEFAULT_OKO);
    strcpy(extension,DEFAULT_OS);
    outcode = DEFAULT_O;
  }
  if (argc == 0) {
	in = stdin;
    out = stdout;
  }
  else if (argc > 0) {
    if (argc == 1) {
      strcpy(infilename,*argv);
      if (strchr(*argv,PERIOD) != NULL)
        *strrchr(*argv,PERIOD) = '\0';
	  strcpy(outfilename,*argv);
      strcat(outfilename,extension);
      if (!strcmp(infilename,outfilename)) {
        if (strchr(outfilename,PERIOD) != NULL)
          *strrchr(outfilename,PERIOD) = '\0';
		strcat(outfilename,"-");
        strcat(outfilename,extension);
      }
	}
    else if (argc > 1) {
      strcpy(infilename,*argv++);
      if (*argv[0] == '-') {
        out = stdout;
        makeoutfile = FALSE;
      }
      else {
        strcpy(outfilename,*argv);
        if (!strcmp(infilename,outfilename)) {
		  if (strchr(outfilename,PERIOD) != NULL)
            *strrchr(outfilename,PERIOD) = '\0';
          strcat(outfilename,"-");
		  strcat(outfilename,extension);
        }
      }
    }
    if ((in = fopen(infilename,"r")) == NULL) {
      fprintf(stderr,"Cannot open %s! Exiting...\n",infilename);
      exit(1);
    }
    if (!docheck && makeoutfile)
      if ((out = fopen(outfilename,"w")) == NULL) {
		fprintf(stderr,"Cannot open %s! Exiting...\n",outfilename);
        exit(1);
	  }
  }
  if (repairjis) {
    JisRepair(in,out,outcode,ki,ko);
    exit(0);
  }
  if (delesc) {
    RemoveEscape(in,out,forcedelesc);
    exit(0);
  }
  tempincode = incode;
  if (!setincode || docheck) {
    incode = DetectCodeType(in);
	if (docheck) {
	  if (setincode && docheck)
        fprintf(stderr,"NOTE: The selected \"-iCODE\" option was ignored\n");
      fprintf(stderr,"Detected input code: ");
      printcode(incode);
      if (docheck)
        exit(0);
    }
    rewind(in);
  }
  if (setincode)
	incode = tempincode;
  switch (incode) {
	case FF_UNKNOWN :
	  fprintf(stderr,"Unknown input code! Exiting...\n");
      exit(1);
      break;
	case FF_EUCSJIS :
      fprintf(stderr,"Ambiguous (Shift-JIS or EUC) input code!\n");
      fprintf(stderr,"Try using the \"-iCODE\" option to specify either Shift-JIS or EUC.\n");
      fprintf(stderr,"Exiting...\n");
      exit(1);
      break;
    case ASCII :
	  fprintf(stderr,"Since detected input code is ASCII, it may be damaged New- or Old-JIS\n");
      fprintf(stderr,"Trying to repair...\n");
	  JisRepair(in,out,outcode,ki,ko);
	  break;
    case FF_NEWJIS :
    case FF_OLDJIS :
    case FF_NEC :
      switch (outcode) {
        case FF_NEWJIS :
        case FF_OLDJIS :
        case FF_NEC :
          seven2seven(in,out,ki,ko);
          break;
		case FF_EUC :
          SevenToEuc(in,out);
		  break;
		case FF_SJIS :
		  seven2shift(in,out);
		  break;
	  }
	  break;
	case FF_EUC :
	  switch (outcode) {
		case FF_NEWJIS :
		case FF_OLDJIS :
		case FF_NEC :
		  EucToSeven(in,out,incode,ki,ko);
		  break;
		case FF_EUC :
		  EucToEuc(in,out,incode,TOFULLSIZE);
		  break;
		case FF_SJIS :
          EucToShift(in,out,incode,TOFULLSIZE);
          break;
      }
      break;
	case FF_SJIS :
      switch (outcode) {
        case FF_NEWJIS :
        case FF_OLDJIS :
        case FF_NEC :
		  shift2seven(in,out,incode,ki,ko);
		  break;
		case FF_EUC :
          ShiftToEuc(in,out,incode,TOFULLSIZE);
          break;
		case FF_SJIS :
          shift2shift(in,out,incode,TOFULLSIZE);
          break;
      }
      break;
  }
  exit(0);
}
#endif


static void StraightEcho (void)
{
  int p1;
 
  while ((p1 = ReadChar()) != EOF) WriteChar(p1);
}

 
static BOOL SkipESCSeq (int ch, int *TwoBytes)
{
  BOOL temp;

  temp = *TwoBytes;
  if (ch == '$' || ch == '(') ReadChar();

  if (ch == 'K' || ch == '$') *TwoBytes = TRUE;
  else *TwoBytes = FALSE;

  return (temp == *TwoBytes);
}


static void JisRepair (FILEFORMAT outcode, char *ki, char *ko)
{
  int p1, p2, p3;
  BOOL TwoBytes = FALSE;
  unsigned long count = 0;

  while ((p1 = ReadChar()) != EOF) {
    if (TwoBytes) {
      if (p1 == ESC) {
        p2 = ReadChar();
        if (p2 == '(') {
          p3 = ReadChar();
          switch (p3) {
            case 'J' :
            case 'B' :
            case 'H' :
              TwoBytes = FALSE;
              switch (outcode) {
                case FF_NEWJIS :
                case FF_NEC :
                case FF_OLDJIS :
                  WriteChar(ESC);
                  OutPuts(ko);
                  break;
                default :
                  break;
              }
              break;
            default :
              WriteChar(p1);
              WriteChar(p2);
              WriteChar(p3);
              break;
          }
        }
        else if (p2 == 'H') {
          TwoBytes = FALSE;
          switch (outcode) {
            case FF_NEWJIS :
            case FF_NEC :
            case FF_OLDJIS :
              WriteChar(ESC);
              OutPuts(ko);
              break;
            default :
              break;
          }
        }
        else {
          WriteChar(p1);
          WriteChar(p2);
        }
      }
      else if (p1 == '(') {
        p2 = ReadChar();
        switch (p2) {
          case 'J' :
          case 'B' :
          case 'H' :
            TwoBytes = FALSE;
            switch (outcode) {
              case FF_NEWJIS :
              case FF_NEC :
              case FF_OLDJIS :
                WriteChar(ESC);
                OutPuts(ko);
                break;
              default :
                break;
            }
            count++;
            break;
          default :
            switch (outcode) {
              case FF_NEWJIS :
              case FF_NEC :
              case FF_OLDJIS :
                WriteChar(p1);
                WriteChar(p2);
                break;
              case FF_EUC :
                p1 += 128;
                p2 += 128;
                WriteChar(p1);
                WriteChar(p2);
                break;
			  case FF_SJIS :
                JisToSJis(&p1,&p2);
                WriteChar(p1);
                WriteChar(p2);
                break;
            }
            break;
        }
      }
      else if (p1 == NL) {
        switch (outcode) {
          case FF_NEWJIS :
          case FF_NEC :
          case FF_OLDJIS :
            WriteChar(ESC);
            OutPuts(ko);
            WriteChar(p1);
            break;
          default :
            WriteChar(p1);
            break;
        }
        count++;
        TwoBytes = FALSE;
      }
      else if (p1 == FF)
        ;
      else {
        p2 = ReadChar();
        switch (outcode) {
          case FF_NEWJIS :
          case FF_NEC :
          case FF_OLDJIS :
            WriteChar(p1);
            WriteChar(p2);
            break;
          case FF_EUC :
            p1 += 128;
            p2 += 128;
            WriteChar(p1);
            WriteChar(p2);
            break;
		  case FF_SJIS :
            JisToSJis(&p1,&p2);
            WriteChar(p1);
            WriteChar(p2);
            break;
        }
      }
    }
    else {
      if (p1 == ESC) {
        p2 = ReadChar();
        if (p2 == '$') {
          p3 = ReadChar();
          switch (p3) {
            case 'B' :
            case '@' :
              TwoBytes = TRUE;
              switch (outcode) {
                case FF_NEWJIS :
                case FF_NEC :
                case FF_OLDJIS :
                  WriteChar(ESC);
                  OutPuts(ki);
                  break;
                default :
                  break;
              }
              break;
            default :
              WriteChar(p1);
              WriteChar(p2);
              WriteChar(p3);
              break;
          }
        }
        else if (p2 == 'K') {
          TwoBytes = TRUE;
          switch (outcode) {
            case FF_NEWJIS :
            case FF_NEC :
            case FF_OLDJIS :
              WriteChar(ESC);
              OutPuts(ki);
              break;
            default :
              break;
          }
        }
        else {
          WriteChar(p1);
          WriteChar(p2);
        }
      }
      else if (p1 == '$') {
        p2 = ReadChar();
        switch (p2) {
          case 'B' :
          case '@' :
            TwoBytes = TRUE;
            switch (outcode) {
              case FF_NEWJIS :
              case FF_NEC :
              case FF_OLDJIS :
                WriteChar(ESC);
                OutPuts(ki);
                break;
              default :
                break;
            }
            count++;
            break;
          default :
            switch (outcode) {
              case FF_NEWJIS :
              case FF_NEC :
              case FF_OLDJIS :
                WriteChar(p1);
                WriteChar(p2);
                break;
              case FF_EUC :
                WriteChar(p1);
                WriteChar(p2);
                break;
			  case FF_SJIS :
                WriteChar(p1);
                WriteChar(p2);
                break;
            }
            break;
        }
      }
      else if (p1 == FF)
        ;
      else
        WriteChar(p1);
    }
  }
  if (TwoBytes) {
    switch (outcode) {
      case FF_NEWJIS :
      case FF_NEC :
      case FF_OLDJIS :
        WriteChar(ESC);
        OutPuts(ko);
        count++;
        break;
      default :
        break;
    }
  }
}

static void SJisToJis (int *p1, int *p2)
{
    register unsigned char c1 = *p1;
    register unsigned char c2 = *p2;
    register int adjust = c2 < 159;
    register int RowOffset = c1 < 160 ? 112 : 176;
    register int CellOffset = adjust ? (31 + (c2 > 127)) : 126;

    *p1 = ((c1 - RowOffset) << 1) - adjust;
    *p2 -= CellOffset;
}

static void JisToSJis (int *p1, int *p2)
{
    register unsigned char c1 = *p1;
    register unsigned char c2 = *p2;
    register int RowOffset = c1 < 95 ? 112 : 176;
    register int CellOffset = c1 % 2 ? 31 + (c2 > 95) : 126;

    *p1 = ((c1 + 1) >> 1) + RowOffset;
    *p2 = c2 + CellOffset;
}


static void ShiftToEuc (FILEFORMAT incode)
{
  int p1, p2;
  
  while ((p1 = ReadChar()) != EOF) {
    switch (p1) {
      case CR :
        WriteChar(CR);
        break;
      case NL :
        WriteChar(NL);
        break;
      case NUL :
      case FF :
        break;
      default :
		if SJIS1(p1) {
          p2 = ReadChar();
		  if SJIS2(p2) {
            SJisToJis(&p1,&p2);
            p1 += 128;
            p2 += 128;
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else if HANKATA(p1) {
          if (TOFULLSIZE) {
            HanToZen(&p1,&p2,incode);
            SJisToJis(&p1,&p2);
            p1 += 128;
            p2 += 128;
          }
          else {
			p2 = p1;
            p1 = 142;
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else
          WriteChar(p1);
        break;
    }
  }
}
 
static void EucToSeven (FILEFORMAT incode, char *ki, char *ko)
{
  int p1, p2;
  BOOL TwoBytes = FALSE;

  while ((p1 = ReadChar()) != EOF) {
    switch (p1) {
      case NL :
        if (TwoBytes) {
          TwoBytes = FALSE;
          WriteChar(ESC);
          OutPuts(ko);
        }
        WriteChar(p1);
        break;
      case FF :
        break;
      default :
        if ISEUC(p1) {
          p2 = ReadChar();
          if ISEUC(p2) {
            p1 -= 128;
            p2 -= 128;
            if (!TwoBytes) {
              TwoBytes = TRUE;
              WriteChar(ESC);
              OutPuts(ki);
            }
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else if (p1 == 142) {
          p2 = ReadChar();
          if HANKATA(p2) {
            p1 = p2;
            HanToZen(&p1,&p2,incode);
            SJisToJis(&p1,&p2);
            if (!TwoBytes) {
              TwoBytes = TRUE;
              WriteChar(ESC);
              OutPuts(ki);
            }
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else {
          if (TwoBytes) {
            TwoBytes = FALSE;
            WriteChar(ESC);
            OutPuts(ko);
          }
          WriteChar(p1);
        }
        break;
    }
  }
  if (TwoBytes) {
    WriteChar(ESC);
    OutPuts(ko);
  }
}
 
static void EucToShift (FILEFORMAT incode)
{
  int p1, p2;

  while ((p1 = ReadChar()) != EOF) {
    switch (p1) {
      case FF :
        break;
      default :
        if ISEUC(p1) {
          p2 = ReadChar();
          if ISEUC(p2) {
            p1 -= 128;
            p2 -= 128;
            JisToSJis(&p1,&p2);
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else if (p1 == 142) {
          p2 = ReadChar();
          if HANKATA(p2) {
            if (TOFULLSIZE) {
              p1 = p2;
              HanToZen(&p1,&p2,incode);
              WriteChar(p1);
              WriteChar(p2);
            }
            else {
			  p1 = p2;
              WriteChar(p1);
            }
          }
          else {
            WriteChar(p1);
            WriteChar(p2);
          }
        }
        else
          WriteChar(p1);
        break;
    }
  }
}

static void EucToEuc (FILEFORMAT incode)
{
  int p1, p2;

  while ((p1 = ReadChar()) != EOF) {
    switch (p1) {
      case FF :
        break;
      default :
        if ISEUC(p1) {
          p2 = ReadChar();
          if ISEUC(p2) {
            WriteChar(p1);
            WriteChar(p2);
          }
        }
        else if (p1 == 142) {
          p2 = ReadChar();
          if (HANKATA(p2) && TOFULLSIZE) {
            p1 = p2;
            HanToZen(&p1,&p2,incode);
            SJisToJis(&p1,&p2);
            p1 += 128;
            p2 += 128;
          }
          WriteChar(p1);
          WriteChar(p2);
        }
        else
          WriteChar(p1);
        break;
    }
  }
}

  
static void SevenToEuc (void)
{
  int temp,p1, p2,TwoBytes = FALSE;

  while ((p1 = ReadChar()) != EOF) {
    switch (p1) {
      case ESC :
        temp = ReadChar();
        SkipESCSeq(temp,&TwoBytes);
        break;
      case NL :
        TwoBytes = FALSE;
        WriteChar(p1);
        break;
      case FF :
        break;
      default :
        if (TwoBytes) {
          p2 = ReadChar();
          p1 += 128;
          p2 += 128;
          WriteChar(p1);
          WriteChar(p2);
        }
        else
          WriteChar(p1);
        break;
    }
  }
}


FILEFORMAT DetectCodeType (void)
{
  int c = 0;
  FILEFORMAT whatcode = FF_UNKNOWN;

  while ((whatcode == FF_EUCSJIS || whatcode == FF_UNKNOWN) && c != EOF) {
    if ((c = ReadChar()) != EOF) {
      if (c == ESC) {
        c = ReadChar();
        if (c == '$') {
          c = ReadChar();
          if (c == 'B')
            whatcode = FF_NEWJIS;
          else if (c == '@')
            whatcode = FF_OLDJIS;
        }
        else if (c == 'K')
          whatcode = FF_NEC;
      }
      else if ((c >= 129 && c <= 141) || (c >= 143 && c <= 159))
		whatcode = FF_SJIS;
      else if (c == 142) {
        c = ReadChar();
        if ((c >= 64 && c <= 126) || (c >= 128 && c <= 160) || (c >= 224 && c <= 252))
		  whatcode = FF_SJIS;
        else if (c >= 161 && c <= 223)
		  whatcode = FF_EUCSJIS;
      }
      else if (c >= 161 && c <= 223) {
        c = ReadChar();
        if (c >= 240 && c <= 254)
          whatcode = FF_EUC;
        else if (c >= 161 && c <= 223)
		  whatcode = FF_EUCSJIS;
        else if (c >= 224 && c <= 239) {
		  whatcode = FF_EUCSJIS;
		  while (c >= 64 && c != EOF && whatcode == FF_EUCSJIS) {
            if (c >= 129) {
              if (c <= 141 || (c >= 143 && c <= 159))
				whatcode = FF_SJIS;
              else if (c >= 253 && c <= 254)
                whatcode = FF_EUC;
            }
            c = ReadChar();
          }
        }
        else if (c <= 159)
		  whatcode = FF_SJIS;
      }
      else if (c >= 240 && c <= 254)
        whatcode = FF_EUC;
      else if (c >= 224 && c <= 239) {
        c = ReadChar();
        if ((c >= 64 && c <= 126) || (c >= 128 && c <= 160))
		  whatcode = FF_SJIS;
        else if (c >= 253 && c <= 254)
          whatcode = FF_EUC;
        else if (c >= 161 && c <= 252)
		  whatcode = FF_EUCSJIS;
      }
    }
  }

  if (whatcode == FF_EUCSJIS) whatcode = FF_EUC;

  return (whatcode);
}

static void HanToZen (int *p1, int *p2, FILEFORMAT incode)
{
  int junk, maru, nigori;

  maru = nigori = FALSE;
  if (incode == FF_SJIS) {
    *p2 = ReadChar();
    if (*p2 == 222) {
      if (ISNIGORI(*p1) || *p1 == 179)
        nigori = TRUE;
      else
        UnreadChar(*p2);
    }
    else if (*p2 == 223) {
      if ISMARU(*p1)
        maru = TRUE;
      else
        UnreadChar(*p2);
    }
    else
      UnreadChar(*p2);
  }
  else if (incode == FF_EUC) {
    junk = ReadChar();
    if (junk == 142) {
      *p2 = ReadChar();
      if (*p2 == 222) {
        if (ISNIGORI(*p1) || *p1 == 179)
          nigori = TRUE;
        else {
          UnreadChar(*p2);
          UnreadChar(junk);
        }
      }
      else if (*p2 == 223) {
        if ISMARU(*p1)
          maru = TRUE;
        else {
          UnreadChar(*p2);
          UnreadChar(junk);
        }
      }
      else {
        UnreadChar(*p2);
        UnreadChar(junk);
      }
    }
    else
      UnreadChar(junk);
  }
  switch (*p1) {
    case 161 :
      *p1 = 129;
      *p2 = 66;
      break;
    case 162 :
      *p1 = 129;
      *p2 = 117;
      break;
    case 163 :
      *p1 = 129;
      *p2 = 118;
      break;
    case 164 :
      *p1 = 129;
      *p2 = 65;
      break;
    case 165 :
      *p1 = 129;
      *p2 = 69;
      break;
    case 166 :
      *p1 = 131;
      *p2 = 146;
      break;
    case 167 :
      *p1 = 131;
      *p2 = 64;
      break;
    case 168 :
      *p1 = 131;
      *p2 = 66;
      break;
    case 169 :
      *p1 = 131;
      *p2 = 68;
      break;
    case 170 :
      *p1 = 131;
      *p2 = 70;
      break;
    case 171 :
      *p1 = 131;
      *p2 = 72;
      break;
    case 172 :
      *p1 = 131;
      *p2 = 131;
      break;
    case 173 :
      *p1 = 131;
      *p2 = 133;
      break;
    case 174 :
      *p1 = 131;
      *p2 = 135;
      break;
    case 175 :
      *p1 = 131;
      *p2 = 98;
      break;
    case 176 :
      *p1 = 129;
      *p2 = 91;
      break;
    case 177 :
      *p1 = 131;
      *p2 = 65;
      break;
    case 178 :
      *p1 = 131;
      *p2 = 67;
      break;
    case 179 :
      *p1 = 131;
      *p2 = 69;
      break;
    case 180 :
      *p1 = 131;
      *p2 = 71;
      break;
    case 181 :
      *p1 = 131;
      *p2 = 73;
      break;
    case 182 :
      *p1 = 131;
      *p2 = 74;
      break;
    case 183 :
      *p1 = 131;
      *p2 = 76;
      break;
    case 184 :
      *p1 = 131;
      *p2 = 78;
      break;
    case 185 :
      *p1 = 131;
      *p2 = 80;
      break;
    case 186 :
      *p1 = 131;
      *p2 = 82;
      break;
    case 187 :
      *p1 = 131;
      *p2 = 84;
      break;
    case 188 :
      *p1 = 131;
      *p2 = 86;
      break;
    case 189 :
      *p1 = 131;
      *p2 = 88;
      break;
    case 190 :
      *p1 = 131;
      *p2 = 90;
      break;
    case 191 :
      *p1 = 131;
      *p2 = 92;
      break;
    case 192 :
      *p1 = 131;
      *p2 = 94;
      break;
    case 193 :
      *p1 = 131;
      *p2 = 96;
      break;
    case 194 :
      *p1 = 131;
      *p2 = 99;
      break;
    case 195 :
      *p1 = 131;
      *p2 = 101;
      break;
    case 196 :
      *p1 = 131;
      *p2 = 103;
      break;
    case 197 :
      *p1 = 131;
      *p2 = 105;
      break;
    case 198 :
      *p1 = 131;
      *p2 = 106;
      break;
    case 199 :
      *p1 = 131;
      *p2 = 107;
      break;
    case 200 :
      *p1 = 131;
      *p2 = 108;
      break;
    case 201 :
      *p1 = 131;
      *p2 = 109;
      break;
    case 202 :
      *p1 = 131;
      *p2 = 110;
      break;
    case 203 :
      *p1 = 131;
      *p2 = 113;
      break;
    case 204 :
      *p1 = 131;
      *p2 = 116;
      break;
    case 205 :
      *p1 = 131;
      *p2 = 119;
      break;
    case 206 :
      *p1 = 131;
      *p2 = 122;
      break;
    case 207 :
      *p1 = 131;
      *p2 = 125;
      break;
    case 208 :
      *p1 = 131;
      *p2 = 126;
      break;
    case 209 :
      *p1 = 131;
      *p2 = 128;
      break;
    case 210 :
      *p1 = 131;
      *p2 = 129;
      break;
    case 211 :
      *p1 = 131;
      *p2 = 130;
      break;
    case 212 :
      *p1 = 131;
      *p2 = 132;
      break;
    case 213 :
      *p1 = 131;
      *p2 = 134;
      break;
    case 214 :
      *p1 = 131;
      *p2 = 136;
      break;
    case 215 :
      *p1 = 131;
      *p2 = 137;
      break;
    case 216 :
      *p1 = 131;
      *p2 = 138;
      break;
    case 217 :
      *p1 = 131;
      *p2 = 139;
      break;
    case 218 :
      *p1 = 131;
      *p2 = 140;
      break;
    case 219 :
      *p1 = 131;
      *p2 = 141;
      break;
    case 220 :
      *p1 = 131;
      *p2 = 143;
      break;
    case 221 :
      *p1 = 131;
      *p2 = 147;
      break;
    case 222 :
      *p1 = 129;
      *p2 = 74;
      break;
    case 223 :
      *p1 = 129;
      *p2 = 75;
      break;
  }
  if (nigori) {
    if ((*p2 >= 74 && *p2 <= 103) || (*p2 >= 110 && *p2 <= 122))
      (*p2)++;
    else if (*p1 == 131 && *p2 == 69)
      *p2 = 148;
  }
  else if (maru && *p2 >= 110 && *p2 <= 122)
    *p2 += 2;
}

