/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"
#ifdef CTL3D
    #include <ctl3d.h>
#endif CTL3D


#define BOXGAP          (AVGHEIGHT / 4)
#define LISTGAP         1
#define LISTLEADING     2

extern LONG FAR PASCAL FileWinProc (HWND, WORD, WORD, LONG);




void DrawBoundingBox (HWND hwnd, HDC hdc, int control)
{
    RECT rect;
    HWND dlghwnd;
    HBRUSH hbrush;
    HPEN hpen;

    dlghwnd = GetDlgItem(hwnd, control),
    GetClientRect(dlghwnd, &rect);
    rect.left = GetWindowWord(dlghwnd, sizeof(FILEOPTIONS *));
    rect.top = GetWindowWord(dlghwnd, sizeof(FILEOPTIONS *) + sizeof(WORD));

    hbrush = SelectObject(hdc, CreateSolidBrush(GetSysColor(COLOR_WINDOW)));
    hpen = SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME)));

    Rectangle(hdc, rect.left, rect.top, rect.left + rect.right + 2 * BOXGAP,
                   rect.top + rect.bottom + 2 * BOXGAP);

    DeleteObject(SelectObject(hdc, hbrush));
    DeleteObject(SelectObject(hdc, hpen));
}



LONG FAR PASCAL JeditProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	FILEOPTIONS *f;


    f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);


	switch (message) {
        case WM_CREATE: {
            CREATESTRUCT *cp;
			RECT rect;

            f = NewFile (FN_CONTROL, TRUE);
			SetWindowWord(hwnd, 0, (WORD) f);

			GetClientRect (hwnd, &rect);
			f->hwnd = hwnd;
            f->parent = GetParent(hwnd);
			f->width = rect.right;
			f->height = rect.bottom;
            f->linelen = 1000;

            /* Now, we make sure that the window is wide enough */
            cp = (CREATESTRUCT *) lParam;

            SetWindowWord(hwnd, sizeof(FILEOPTIONS *), (WORD) cp->x);
            SetWindowWord(hwnd, sizeof(FILEOPTIONS *) + sizeof(WORD), (WORD) cp->y);
            SetWindowWord(hwnd, sizeof(FILEOPTIONS *) + 2 * sizeof(WORD), NULL);

            MoveWindow(hwnd, cp->x + BOXGAP, cp->y + BOXGAP,
                        rect.right - 2 * BOXGAP,
                        SYSFONT->height + 2 * LINEGAP(f) + 1, FALSE);

            break;
		}

        case WM_SIZE:
            f->width = LOWORD(lParam);
            f->height = HIWORD(lParam);
            break;

        case EM_GETHANDLE:
            f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);
            return ((LONG) f);

        case EM_GETLINE:
            f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);
            return ((LONG) f->paragraph->text);

        case EM_SETMODIFY:
            f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);
            f->type = wParam;
            return ((LONG) f);

        case EM_SETSEL:
            f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);
            TurnOffSelection(f);

            SELPARA1(f) = SELPARA2(f) = f->paragraph;
            SELPOS1(f) = LOWORD(lParam);
            SELPOS2(f) = HIWORD(lParam);
            SELTYPE(f) = SEL_SELECTION;
            FlipHighlight(f);

            if (wParam >= 0) {
                CURCHAR(f) = wParam;
                if (!FindCaret(f, TRUE)) {
                    MoveIntoWindow(f);
                    InvalidateRect(f->hwnd, NULL, TRUE);
                    UpdateWindow(f->hwnd);
                }
                DoCaret(f, CURX(f), CURY(f) - CURLINE(f)->height, TRUE);
            }
            return (0);

        case EM_GETSEL:
            f = (FILEOPTIONS *) GetWindowWord(hwnd, 0);
			if (SELPARA1(f) == NULL) return (0);
            return (MAKELONG(SELPOS1(f), SELPOS2(f)));

        case EM_SETRECT:
            SetWindowWord(hwnd, sizeof(FILEOPTIONS *) + 2 * sizeof(WORD), wParam);
            return (0);

        case WM_GETDLGCODE:
            /* This is an edit control, so we want to process key-strokes * /
            /* by ourselves */

            return (DLGC_WANTARROWS | DLGC_WANTALLKEYS | DLGC_WANTCHARS);

        case WM_KEYDOWN:
            /* Some of the keys have special meanings */

            switch (wParam) {
                case VK_TAB:
                    if (GetKeyState(VK_SHIFT) < 0)
                        SetFocus(GetNextDlgTabItem(GetParent(hwnd), hwnd, TRUE));
                    else
                        SetFocus(GetNextDlgTabItem(GetParent(hwnd), hwnd, FALSE));
                    return (0);

                case VK_UP:
                case VK_DOWN:
                case VK_PRIOR:
                case VK_NEXT:
                    return (0);

                case VK_RETURN:
                    SendMessage(GetParent(hwnd), WM_COMMAND, IDOK, 0L);
                    return (0);

                case VK_ESCAPE:
                    SendMessage(GetParent(hwnd), WM_COMMAND, IDCANCEL, 0L);
                    return (0);
            }
            break;

		case EM_REPLACESEL: {
			KANJI far *cp;
			POSITION p;
			KANJI buf[1] = { 0 };

			cp = (KANJI far *) lParam;

            PARAOF(p) = f->paragraph;
            LINEOF(p) = f->paragraph->lines;
            POSOF(p) = 0;

            TakeCareOfThings(f, TRUE);      /* Get rid of selections */

            ReplaceString(f, p, unitlen(f->paragraph->text), (cp != NULL) ? cp : buf,
                            OP_REFORMAT | OP_MOVETOEND | OP_CHOOSEKANJI);

            SendMessage(hwnd, WM_KEYDOWN, VK_HOME, 0L);     /* Move to beginning */
            InvalidateRect(hwnd, NULL, TRUE);
            return (TRUE);
		}

        case EM_LINELENGTH:
            return (unitlen(f->paragraph->text));

		case WM_DESTROY:
			CloseFile(f);
			FreeMem(f);
            SetWindowWord(hwnd, 0, (WORD) NULL);
            return (0);

        default:
            break;
	}

    if (f == NULL) return (DefWindowProc(hwnd, message, wParam, lParam));

    curfile = f;

	return (CallWindowProc((FARPROC) FileWinProc, hwnd, message, wParam, lParam));
}



static int KanjiCompare(KANJI far *s1, KANJI far *s2, BOOL KataWithHira, BOOL ASCIIFirst, BOOL SymbolsLast)
{
    KANJI c1, c2;

    for (; ; s1++, s2++) {
        c1 = *s1 & 0x7f7f;
        c2 = *s2 & 0x7f7f;
        if (c1 == 0 && c2 == 0) return (0);

        else if (c1 == 0) return (-1);
        else if (c2 == 0) return (1);
        else if (c1 == c2) continue;

        if (ISKANJI(c1)) {
            if (!ISKANJI(c2)) {
                if (ASCIIFirst) return (1); else return (-1);
            }

            if (SymbolsLast) {
                if (HIBYTE(c1) < 0x24) c1 += 0x8000;
                if (HIBYTE(c2) < 0x24) c2 += 0x8000;
            }

            switch (HIBYTE(c1)) {
                case 0x24:
                    switch (HIBYTE(c2)) {
                        case 0x24: return (LOBYTE(c1) - LOBYTE(c2));
                        case 0x25: return (KataWithHira ? (LOBYTE(c1) - LOBYTE(c2)) : -1);
                        default:   return (-1);
                    }
                case 0x25:
                    switch (HIBYTE(c2)) {
                        case 0x24: return (KataWithHira ? (LOBYTE(c1) - LOBYTE(c2)) : 1);
                        case 0x25: return (LOBYTE(c1) - LOBYTE(c2));
                        default:   return (-1);
                    }
                default:
                    return (c1 - c2);
            }
        } else {
            if (ISKANJI(c2)) {
                if (ASCIIFirst) return (-1); else return (1);
            }

            if (SymbolsLast) {
                if (c1 < 'A') c1 += 0x8000;
                if (c2 < 'A') c2 += 0x8000;
            }
            return (c1 - c2);
        }
    }
}



LONG JlistProc (HWND hwnd, WORD message, WORD wParam, LONG lParam, BOOL select,
                KANJI far * (* ConvertToString)(int, LONG, KANJI *))
{
	int i, j, k, r;
	char ch;
	COMPAREITEMSTRUCT *comp;
	DELETEITEMSTRUCT *deletep;
	DRAWITEMSTRUCT *drawp;
	MEASUREITEMSTRUCT *measurep;
	KANJI far *cp;
	RECT rect;
	BYTE far *cbufp;
	KANJI buffer[BUFSIZE];
    HBRUSH hbrush;
    HPEN hpen;

    switch (message) {
        case WM_COMPAREITEM: {
            KANJI buffer2[BUFSIZE];
			KANJI far *cp2;

            comp = (COMPAREITEMSTRUCT *) lParam;

            if (ConvertToString != NULL) {
                cp = ConvertToString(comp->CtlID, comp->itemData1, buffer);
                cp2 = ConvertToString(comp->CtlID, comp->itemData2, buffer2);
            } else {
                cp = (KANJI far *) comp->itemData1;
				cp2 = (KANJI far *) comp->itemData2;
            }

            i = KanjiCompare(cp, cp2, TRUE, FALSE, TRUE);

            if (i < 0) return (-1);
            else if (i > 0) return (1);
            else return (0);
        }

        case WM_DELETEITEM:
            return (TRUE);

		case WM_DRAWITEM:
            drawp = (DRAWITEMSTRUCT *) lParam;

            if (ConvertToString != NULL) {
                cp = ConvertToString(drawp->CtlID, drawp->itemData, buffer);
            } else {
                cp = (KANJI far *) drawp->itemData;
            }

            if (select && drawp->itemAction & ODA_FOCUS) {
                hbrush = SelectObject(drawp->hDC, GetStockObject(NULL_BRUSH));
                if (drawp->itemState & ODS_FOCUS) {
                    if (drawp->itemState & ODS_SELECTED) {
                        SetBkColor(drawp->hDC, GetSysColor(COLOR_HIGHLIGHT));
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_HIGHLIGHTTEXT));
                    } else {
                        SetBkColor(drawp->hDC, GetSysColor(COLOR_WINDOW));
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWTEXT));
                    }
                } else {
                    if (drawp->itemState & ODS_SELECTED) {
                        SetBkColor(drawp->hDC, GetSysColor(COLOR_HIGHLIGHT));
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_HIGHLIGHT));
                    } else {
                        SetBkColor(drawp->hDC, GetSysColor(COLOR_WINDOW));
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOW));
                    }
                }
                hpen = SelectObject(drawp->hDC, hpen);
                Rectangle(drawp->hDC, drawp->rcItem.left, drawp->rcItem.top,
                          drawp->rcItem.right, drawp->rcItem.bottom);

                SelectObject(drawp->hDC, hbrush);
                DeleteObject(SelectObject(drawp->hDC, hpen));
            }

			if ((select && drawp->itemAction & ODA_SELECT) || drawp->itemAction & ODA_DRAWENTIRE) {
                SelectObject(drawp->hDC, NULL_PEN);

				if (drawp->itemState & ODS_SELECTED) {
					SetTextColor(drawp->hDC, GetSysColor(COLOR_HIGHLIGHTTEXT));
					SetBkColor(drawp->hDC, GetSysColor(COLOR_HIGHLIGHT));
					hbrush = CreateSolidBrush(GetSysColor(COLOR_HIGHLIGHT));
                    if (drawp->itemState & ODS_FOCUS) {
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_HIGHLIGHTTEXT));
                        hpen = SelectObject(drawp->hDC, hpen);
                        hbrush = SelectObject(drawp->hDC, hbrush);
                        Rectangle(drawp->hDC, drawp->rcItem.left, drawp->rcItem.top,
                                  drawp->rcItem.right, drawp->rcItem.bottom);
                        DeleteObject(SelectObject(drawp->hDC, hpen));
                        hbrush = SelectObject(drawp->hDC, hbrush);
                    } else {
                        FillRect(drawp->hDC, &(drawp->rcItem), hbrush);
                    }
				} else {
					SetTextColor(drawp->hDC, GetSysColor(COLOR_WINDOWTEXT));
					SetBkColor(drawp->hDC, GetSysColor(COLOR_WINDOW));
					hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
                    if (drawp->itemState & ODS_FOCUS) {
                        hpen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWTEXT));
                        hpen = SelectObject(drawp->hDC, hpen);
                        hbrush = SelectObject(drawp->hDC, hbrush);
                        Rectangle(drawp->hDC, drawp->rcItem.left, drawp->rcItem.top,
                                  drawp->rcItem.right, drawp->rcItem.bottom);
                        DeleteObject(SelectObject(drawp->hDC, hpen));
                        hbrush = SelectObject(drawp->hDC, hbrush);
                    } else {
                        FillRect(drawp->hDC, &(drawp->rcItem), hbrush);
                    }
				}

                DeleteObject(hbrush);

                k = drawp->rcItem.bottom - (BOXGAP / 2);

				for (i = 0, j = drawp->rcItem.left + BOXGAP ; cp[i] != 0; i++) {
                    if (ISKANJI(cp[i])) {
                        r = Jis2Index(cp[i], SYSFONT->holes);
                        if (r < 0) r = Jis2Index(BADKANJI, SYSFONT->holes);
                        r = GetKanjiBitmap(SYSFONT, r, &cbufp);
                        DisplayKanjiBitmap(drawp->hDC, j, k, SYSFONT->width,
                                            SYSFONT->height, r, SRCCOPY, cbufp);

                        j += (SYSFONT->width + LISTLEADING);
                    } else if (cp[i] >= ' ') {
                        r = FontCharWidth(cp[i], 0);        /* System font */

						ch = cp[i];
						TextOut(drawp->hDC, j, k - AVGHEIGHT, &ch, 1);

                        j += r;
                        if (cp[i+1] != 0 && ISKANJI(cp[i+1])) j+= LISTLEADING;
                    }
                }
            }

            return (TRUE);

        case WM_MEASUREITEM:
            measurep = (MEASUREITEMSTRUCT *) lParam;

            GetClientRect(GetDlgItem(hwnd, measurep->CtlID), &rect);

            measurep->itemHeight = SYSFONT->height;
            measurep->itemWidth = rect.right;

            if (measurep->itemHeight < AVGHEIGHT)
                measurep->itemHeight = AVGHEIGHT;

            measurep->itemHeight += BOXGAP;

            return (TRUE);
    }

    return (TRUE);
}



LONG FAR PASCAL JtextProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_CREATE: {
            RECT rect;

            SetWindowWord(hwnd, 0, NULL);
            SetWindowWord(hwnd, sizeof(int), 0);
            SetWindowWord(hwnd, sizeof(KANJI *) + sizeof(int), FALSE);

            GetWindowRect(hwnd, &rect);

            SetWindowPos(hwnd, NULL, 0, 0, rect.right - rect.left, SYSFONT->height,
                         SWP_NOMOVE | SWP_NOZORDER);
            return (0);
        }

        case EM_SETRECT: {
            RECT rect;
            BOOL Box;

            Box = (BOOL) GetWindowWord(hwnd, sizeof(KANJI *) + sizeof(int));
            SetWindowWord(hwnd, sizeof(KANJI *) + sizeof(int), wParam);

            if ((Box && !wParam) || (!Box && wParam)) {
                GetWindowRect(hwnd, &rect);

                if (wParam) {
                    SetWindowPos(hwnd, NULL, 0, 0, rect.right - rect.left,
                                 SYSFONT->height + 2 * BOXGAP,
                                 SWP_NOMOVE | SWP_NOZORDER);
                } else {
                    SetWindowPos(hwnd, NULL, 0, 0, rect.right - rect.left, SYSFONT->height,
                                 SWP_NOMOVE | SWP_NOZORDER);
                }
                InvalidateRect(hwnd, NULL, TRUE);
            }

            return (0);
        }

        case EM_GETLINE:
        case EM_GETHANDLE:
            return ((LONG) GetWindowWord(hwnd, 0));

		case EM_REPLACESEL: {
            int i, len;
            KANJI *kp, far *kp1;

            kp1 = (KANJI far *) lParam;
            if (kp1 == NULL) kp1 = "";

            len = kanjilen(kp1);

            kp = (KANJI *) GetWindowWord(hwnd, 0);

            i = (len + 5) * sizeof(KANJI);

            if (kp == NULL) {
                kp = (KANJI *) MemAlloc(i);
                SetWindowWord(hwnd, sizeof(KANJI *), i);
            } else if (GetWindowWord(hwnd, sizeof(KANJI *)) < i) {
                FreeMem(kp);
                kp = (KANJI *) MemAlloc(i);
                SetWindowWord(hwnd, sizeof(KANJI *), i);
            }
            for (i = 0; kp1[i]; i++) kp[i] = kp1[i];
            kp[i] = 0;

            SetWindowWord(hwnd, 0, kp);
            return (0);
        }

        case WM_PAINT: {
            int i, x, r;
            int offset = 0;
            int ch;
			BYTE far *cbufp;
			KANJI *kp, kch;
			RECT rect;
			HDC hdc;
			PAINTSTRUCT ps;
			extern BOOL Dialogs3D;

			hdc = BeginPaint(hwnd, &ps);

            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));

            if (GetWindowWord(hwnd, sizeof(KANJI *) + sizeof(int))) {
                /* Draw bounding box */

                offset = BOXGAP;

				GetClientRect(hwnd, &rect);
                Rectangle(hdc, 0, 0, rect.right, rect.bottom);
#ifdef CTL3D
			} else if (Dialogs3D) {
				HBRUSH hbrush;

				hbrush = Ctl3dCtlColorEx(WM_CTLCOLOR, hdc, MAKELONG(hwnd, CTLCOLOR_DLG));
				GetClientRect(hwnd, &rect);
                FillRect(hdc, &rect, hbrush);
#endif CTL3D
            } else {
                HBRUSH hbrush;

                hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
				GetClientRect(hwnd, &rect);
                FillRect(hdc, &rect, hbrush);
                DeleteObject(hbrush);
            }

			kp = (KANJI *) GetWindowWord(hwnd, 0);
			if (kp == NULL) {
				EndPaint(hwnd, &ps);
				return (0);
			}

			x = 0;

            for (i = 0; kp[i]; i++) {
                kch = kp[i];

                if (ISKANJI(kch)) {
					if (i > 0 && !ISKANJI(kp[i-1])) x += SYSFONT->leading;

                    r = Jis2Index(kch, SYSFONT->holes);
                    if (r < 0) r = Jis2Index(BADKANJI, SYSFONT->holes);
                    r = GetKanjiBitmap(SYSFONT, r, &cbufp);
                    DisplayKanjiBitmap(hdc, x + offset, SYSFONT->height + offset,
                                        SYSFONT->width, SYSFONT->height,
                                        r, SRCAND, cbufp);
					x += SYSFONT->width + SYSFONT->leading;
                } else {
                    ch = LOBYTE(kch);

                    switch (ch) {
                        case '\t': ch = 0xbb; break;    /* >> symbol */
                        case '\r': ch = 0xb6; break;    /* paragraph symbol */
                        case '\n': ch = 0xab; break;    /* << symbol */
                    }

                    TextOut(hdc, x + offset, SYSFONT->height + offset - global.textmetric.tmHeight, &ch, 1);
                    x += FontCharWidth(ch, 0);          /* System font */
                }
            }
            EndPaint(hwnd, &ps);
            return (0);
		}

        case WM_DESTROY: {
            KANJI *kp;

            kp = (KANJI *) GetWindowWord(hwnd, 0);
            if (kp != NULL) FreeMem(kp);
            return (0);
        }
    }

    return (DefWindowProc(hwnd, message, wParam, lParam));
}
