/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"

#include "idm.h"


typedef struct {
    KANJI far *shortcut;
    KANJI far *phrase;
    BOOL dynamic;
    BOOL hidden;
} GLOSSARY;


static GLOSSARY Glossary[MAXGLOSSARY];

static int NrGlossary = 0;
static BOOL GlossaryChanged = FALSE;
static BOOL sizing = FALSE;

static KANJI far *CurrentShortCut;
static KANJI far *CurrentPhrase;



static KANJI far *ConvertGlossary (int id, LONG lParam, KANJI *buf)
{
    return (Glossary[lParam].shortcut);
}



static KANJI far *ConvertGlossary1 (int id, LONG lParam, KANJI *buf)
{
    return (Glossary[lParam].phrase);
}



static int GlossaryComp (void const *v1, void const *v2)
{
	GLOSSARY *p1, *p2;

	p1 = (GLOSSARY *) v1;
	p2 = (GLOSSARY *) v2;

	return (kanjicmp(p1->shortcut, p2->shortcut));
}



void ReadGlossary (char *filename)
{
	int i, j, len, fd;
	OFSTRUCT of;

	/* Clear the old glossary */

    for (i = 0; i < NrGlossary; i++) {
        if (Glossary[i].shortcut != NULL) FreeBlock(Glossary[i].shortcut);
        if (Glossary[i].phrase != NULL) FreeBlock(Glossary[i].phrase);
        Glossary[i].shortcut = Glossary[i].phrase = NULL;
    }

    fd = OpenFile(filename, &of, OF_READ);
    if (fd < 0) return;

	/* Read the glossary file */

	lseek(fd, 0L, 0);
	read(fd, &NrGlossary, sizeof(int));

	for (i = 0; i < NrGlossary; i++) {
		/* Read the options */
		read(fd, &j, sizeof(int));
        Glossary[i].dynamic = j & 0x0001;
        Glossary[i].hidden = j & 0x0002;

        /* Read the short-cut */
        read(fd, &len, sizeof(int));
        len *= sizeof(KANJI);
        Glossary[i].shortcut = BlockAlloc(len);
        if (Glossary[i].shortcut == NULL) break;
        _lread(fd, (char far *) Glossary[i].shortcut, len);

        /* Read the phrase */
        read(fd, &len, sizeof(int));
        len *= sizeof(KANJI);
        Glossary[i].phrase = BlockAlloc(len);
        if (Glossary[i].shortcut == NULL) break;
        _lread(fd, (char far *) Glossary[i].phrase, len);
    }

    close(fd);

    if (NrGlossary > 0) {
        qsort(Glossary, NrGlossary, sizeof(GLOSSARY), GlossaryComp);
    }
}



void WriteGlossary (char *filename)
{
	int i, j, len, fd;
	OFSTRUCT of;


	if (!GlossaryChanged) return;

	fd = OpenFile(filename, &of, OF_WRITE | OF_CREATE);
	if (fd < 0) {
        ErrorMessage(global.hwnd, "Cannot create glossary file %s!", filename);
		return;
	}

	for (i = j = 0; i < NrGlossary; i++) {
		if (Glossary[i].shortcut != NULL) j++;
	}

	lseek(fd, 0L, 0);
	write(fd, &j, sizeof(int));

	for (i = 0; i < NrGlossary; i++) {
		if (Glossary[i].shortcut == NULL) continue;

		/* Write the options flag */
        j = 0;
        if (Glossary[i].dynamic) j |= 0x0001;
        if (Glossary[i].hidden) j |= 0x0002;
		write(fd, &j, sizeof(int));

		/* Write the short-cut */
		len = kanjilen(Glossary[i].shortcut) + 1;
		write(fd, &len, sizeof(int));
		len *= sizeof(KANJI);
		_lwrite(fd, (char far *) Glossary[i].shortcut, len);

		/* Write the phrase */
		len = kanjilen(Glossary[i].phrase) + 1;
		write(fd, &len, sizeof(int));
		len *= sizeof(KANJI);
		_lwrite(fd, (char far *) Glossary[i].phrase, len);
    }

    close(fd);
}



int SearchGlossary (KANJI far *key)
{
	int len;
    int top, bottom, middle;
    int diff;


    if (NrGlossary <= 0) return (-2);


    /* Now binary search */

	top = 0;
    bottom = NrGlossary - 1;

	len = kanjilen(key);

    if (!kanjicmp(key, Glossary[top].shortcut)) {
        if (Glossary[top].dynamic) return (top);
    } else if (!kanjincmp(key, Glossary[top].shortcut, len)) {
        if (Glossary[top].dynamic) return (-1);
    }

    if (!kanjicmp(key, Glossary[bottom].shortcut)) {
        return (Glossary[bottom].dynamic ? bottom : -2);
    } else if (!kanjincmp(key, Glossary[bottom].shortcut, len)) {
        return (Glossary[bottom].dynamic ? -1 : -2);
    }


	for (;;) {
		middle = (top + bottom) / 2;

        diff = kanjicmp(key, Glossary[middle].shortcut);

        if (diff == 0) {
            return (Glossary[middle].dynamic ? middle : -2);
        }

        if (top >= bottom - 1) {
            for (; bottom < NrGlossary; bottom++) {
				diff = kanjincmp(key, Glossary[bottom].shortcut, len);
                if (!Glossary[bottom].dynamic || diff != 0) continue;
                return (-1);
            }
            return (-2);
        }

        if (diff > 0) top = middle;
		else bottom = middle;
	}
}



KANJI far *GetGlossary (int index)
{
    return (Glossary[index].phrase);
}




BOOL FAR PASCAL EditGlossaryProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
		case WM_INITDIALOG: {
            int i;
			FILEOPTIONS *f;
            KANJI buf[BUFSIZE];

            /* Set the type and mode-change icon */
            SendDlgItemMessage(hwnd, 4201, EM_SETMODIFY, FN_CONTROL | FN_NOKANJI, 0L);
            SendDlgItemMessage(hwnd, 4201, EM_SETRECT, GetDlgItem(hwnd, 4211), 0L);

            SendDlgItemMessage(hwnd, 4202, EM_SETMODIFY, FN_CONTROL, 0L);
            SendDlgItemMessage(hwnd, 4202, EM_SETRECT, GetDlgItem(hwnd, 4211), 0L);

            f = (FILEOPTIONS *) SendDlgItemMessage(hwnd, 4201, EM_GETHANDLE, 0, 0L);
			SendDlgItemMessage(hwnd, 4211, EM_SETHANDLE, f->hwnd, 0L);  /* mode-change icon */

            if (CurrentShortCut != NULL) {
                SendDlgItemMessage(hwnd, 4201, EM_REPLACESEL, 0, (LONG) CurrentShortCut);
                SetFocus(GetDlgItem(hwnd, 4202));
            }

            if (CurrentPhrase != NULL) {
                /* Edit */
                SendDlgItemMessage(hwnd, 4202, EM_REPLACESEL, 0, (LONG) CurrentPhrase);
                i = kanjilen(CurrentPhrase);
                SendDlgItemMessage(hwnd, 4202, EM_SETSEL, i, MAKELONG(0, i-1));
                SetWindowText(GetDlgItem(hwnd, 1), "&Done!");
                SetWindowText(hwnd, "Edit an Existing Glossary Item");
            } else {
                /* Add */
                SetWindowText(GetDlgItem(hwnd, 1), "&Add!");
                SetWindowText(hwnd, "Add a Glossary Item");
            }

			CenterDialogBox(hwnd);
			return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case IDCANCEL:
                    CurrentShortCut = CurrentPhrase = NULL;
                    EndDialog(hwnd, FALSE);
                    return (TRUE);

                case IDOK: {
                    int i;
                    UNIT far *up, far *up1;
                    KANJI ch;

                    up1 = (UNIT far *) SendDlgItemMessage(hwnd, 4202, EM_GETLINE, 0, 0L);
                    if (unitlen(up1) <= 0) {
                        ErrorMessage(hwnd, "Sorry, you must enter a phrase for the short-cut.");
                        SetFocus(GetDlgItem(hwnd, 4202));
                        return (TRUE);
                    }

                    up = (UNIT far *) SendDlgItemMessage(hwnd, 4201, EM_GETLINE, 0, 0L);
                    if (unitlen(up) <= 0) {
                        ErrorMessage(hwnd, "Sorry, you must enter a short-cut for the phrase.");
                        SetFocus(GetDlgItem(hwnd, 4201));
                        return (TRUE);
                    }

					CurrentShortCut = (KANJI far *) BlockAlloc((unitlen(up) + 5) * sizeof(KANJI));
                    for (i = 0; up[i].kanji; i++) CurrentShortCut[i] = up[i].kanji;
                    CurrentShortCut[i] = 0;

                    CurrentPhrase = (KANJI far *) BlockAlloc((unitlen(up1) + 5) * sizeof(KANJI));
                    for (i = 0; up1[i].kanji; i++) CurrentPhrase[i] = up1[i].kanji;
                    CurrentPhrase[i] = 0;

                    EndDialog(hwnd, TRUE);
                    return (TRUE);
                }
            }
            return (TRUE);

        case WM_PAINT: {
            HDC hdc;
			PAINTSTRUCT ps;

			hdc = BeginPaint(hwnd, &ps);

            DrawBoundingBox(hwnd, hdc, 4201);
            DrawBoundingBox(hwnd, hdc, 4202);

            EndPaint(hwnd, &ps);
			return (TRUE);
		}
    }
    return (FALSE);
}



BOOL FAR PASCAL GlossaryProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_INITDIALOG: {
            int i;

            for (i = 0; i < NrGlossary; i++) {
                SendDlgItemMessage(hwnd, 4211, LB_ADDSTRING, 0, (LONG) i);
            }

            if (NrGlossary > 0) {
                SendDlgItemMessage(hwnd, 4211, LB_SETCURSEL, 0, 0L);
                i = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, 0, 0L);
                SendDlgItemMessage(hwnd, 4201, EM_REPLACESEL, 0, (LONG) Glossary[i].shortcut);
                SendDlgItemMessage(hwnd, 4202, EM_REPLACESEL, 0, (LONG) Glossary[i].phrase);
                CheckDlgButton (hwnd, 4231, Glossary[0].dynamic);
                CheckDlgButton (hwnd, 4232, Glossary[0].hidden);
            } else {
                EnableWindow(GetDlgItem(hwnd, 4231), FALSE);
                EnableWindow(GetDlgItem(hwnd, 4232), FALSE);
            }

            SetFocus(GetDlgItem(hwnd, 4211));

            CenterDialogBox(hwnd);
			return (TRUE);
        }

		case WM_KEYDOWN:
            switch (wParam) {
                case VK_ESCAPE: SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                                return (TRUE);
            }
            break;

		case WM_COMMAND: {
            int i, j;

            switch (wParam) {
                case 4211:      /* List Box */
					switch (HIWORD(lParam)) {
                        case LBN_SELCHANGE:
                            i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                            if (NrGlossary <= 0 || i == LB_ERR) {
                                SendDlgItemMessage(hwnd, 4201, EM_REPLACESEL, 0, (LONG) NULL);
                                SendDlgItemMessage(hwnd, 4202, EM_REPLACESEL, 0, (LONG) NULL);
                                EnableWindow(GetDlgItem(hwnd, 4231), FALSE);
                                EnableWindow(GetDlgItem(hwnd, 4232), FALSE);
                                return (TRUE);
							}

							i = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, i, 0L);

							SendDlgItemMessage(hwnd, 4201, EM_REPLACESEL, 0, (LONG) Glossary[i].shortcut);
							SendDlgItemMessage(hwnd, 4202, EM_REPLACESEL, 0, (LONG) Glossary[i].phrase);
							CheckDlgButton (hwnd, 4231, Glossary[i].dynamic);
                            CheckDlgButton (hwnd, 4232, Glossary[i].hidden);
							EnableWindow(GetDlgItem(hwnd, 4231), TRUE);
                            EnableWindow(GetDlgItem(hwnd, 4232), TRUE);
                            InvalidateRect(GetDlgItem(hwnd, 4201), NULL, TRUE);
                            InvalidateRect(GetDlgItem(hwnd, 4202), NULL, TRUE);
							return (TRUE);

						case LBN_DBLCLK:
                            SendMessage(hwnd, WM_COMMAND, 4222, 0L);    /* Edit */
							return (TRUE);
					}
					return (TRUE);

				case 4221: {    /* Add */
                    int len;

                    CurrentShortCut = CurrentPhrase = NULL;
                    if (!DialogBox(hInstance, "EditGlossary", hwnd, EditGlossaryProc)) {
                        CurrentShortCut = CurrentPhrase = NULL;
                        return (TRUE);
                    }

					/* Trim trailing blanks */

					len = kanjilen(CurrentShortCut);

					for (i = len - 1; i >= 0; i--) {
						CurrentShortCut[i] &= 0x7f7f;
						if (CurrentShortCut[i] != 0x2121 && CurrentShortCut[i] > ' ') break;
					}

                    CurrentShortCut[i+1] = 0;

					/* Duplicate? */

					for (i = 0; i < NrGlossary; i++) {
						if (Glossary[i].shortcut == NULL) continue;
                        if (!kanjicmp(CurrentShortCut, Glossary[i].shortcut)) break;
					}

					if (i < NrGlossary) {
                        if (YesNo(hwnd, "The short-cut you want to add already exists.  "
                                        "You should first delete the old glossary item before "
                                        "adding a new one with the same short-cut.\n\n"
                                        "Do you REALLY want to overwrite the already-existing "
                                        "short-cut?") != IDYES) {

                            FreeBlock(CurrentShortCut);
                            FreeBlock(CurrentPhrase);
							CurrentShortCut = CurrentPhrase = NULL;
                            return (TRUE);
                        }

                        FreeBlock((void far *) CurrentShortCut);
                        /* Replace the phrase */
						len = kanjilen(CurrentPhrase);
                        FreeBlock((void far *) Glossary[i].phrase);
                        Glossary[i].phrase = CurrentPhrase;
                        InvalidateRect(GetDlgItem(hwnd, 4221), NULL, TRUE);
                    } else {
						/* Put it in the glossary */

						Glossary[NrGlossary].shortcut = CurrentShortCut;
						Glossary[NrGlossary].phrase = CurrentPhrase;

                        Glossary[NrGlossary].dynamic = TRUE;
						Glossary[NrGlossary].hidden = FALSE;

						SendDlgItemMessage(hwnd, 4211, LB_ADDSTRING, 0, (LONG) NrGlossary);

						i = NrGlossary++;
					}

                    /* Which item is the one just added? */

					len = SendDlgItemMessage(hwnd, 4211, LB_GETCOUNT, 0, 0L);

                    for (j = 0; j < len; j++) {
                        if (SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, j, 0L) == i) break;
                    }

                    if (j < len) SendDlgItemMessage(hwnd, 4211, LB_SETCURSEL, j, 0L);

					SendMessage(hwnd, WM_COMMAND, 4211, MAKELONG(hwnd, LBN_SELCHANGE));

					GlossaryChanged = TRUE;

                    CurrentPhrase = CurrentShortCut = NULL;

                    SetFocus(GetDlgItem(hwnd, 4211));
					return (TRUE);
                }

                case 4222: {    /* Edit */
					int len;

					i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) {
                        MessageBeep(0);
                        return (TRUE);
                    }

                    i = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, i, 0L);

                    CurrentShortCut = Glossary[i].shortcut;
                    CurrentPhrase = Glossary[i].phrase;

                    if (!DialogBox(hInstance, "EditGlossary", hwnd, EditGlossaryProc)) {
                        CurrentShortCut = CurrentPhrase = NULL;
                        return (TRUE);
                    }

					/* Trim trailing blanks */

					len = kanjilen(CurrentShortCut);

                    for (j = len - 1; j >= 0; j--) {
						CurrentShortCut[j] &= 0x7f7f;
						if (CurrentShortCut[j] != 0x2121 && CurrentShortCut[j] > ' ') break;
					}

                    CurrentShortCut[j+1] = 0;

                    if (!kanjicmp(CurrentShortCut, Glossary[i].shortcut)) {
                        FreeBlock(CurrentShortCut);
                        FreeBlock(CurrentPhrase);
                        CurrentShortCut = CurrentPhrase = NULL;
                        return (TRUE);
                    }

                    FreeBlock(Glossary[i].shortcut);
                    Glossary[i].shortcut = CurrentShortCut;
                    FreeBlock(Glossary[i].phrase);
                    Glossary[i].phrase = CurrentPhrase;

                    CurrentShortCut = CurrentPhrase = NULL;
                    return (TRUE);
                }

                case 4223:      /* Delete */
                    if (NrGlossary <= 0) return (TRUE);
                    i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) return (TRUE);

                    if (YesNo(hwnd, "Do you REALLY want to delete this item?") != IDYES)
                        return (TRUE);

                    j = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, i, 0L);

                    SendDlgItemMessage(hwnd, 4211, LB_DELETESTRING, i, 0L);

                    FreeBlock(Glossary[j].shortcut);
                    FreeBlock(Glossary[j].phrase);

                    Glossary[j].shortcut = Glossary[j].phrase = NULL;

                    if (SendDlgItemMessage(hwnd, 4211, LB_GETCOUNT, 0, 0L) > 0) {
						SendDlgItemMessage(hwnd, 4211, LB_SETCURSEL, 0, 0L);
                    }
                    SendMessage(hwnd, WM_COMMAND, 4211, MAKELONG(hwnd, LBN_SELCHANGE));
                    GlossaryChanged = TRUE;

                    SetFocus(GetDlgItem(hwnd, 4211));
                    return (TRUE);

                case 4231:      /* Dynamic button */
                    i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) return (TRUE);

                    j = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, i, 0L);
                    Glossary[j].dynamic = IsDlgButtonChecked(hwnd, 4231);
                    GlossaryChanged = TRUE;
					return (TRUE);

                case 4232:      /* Hidden button */
                    i = SendDlgItemMessage(hwnd, 4211, LB_GETCURSEL, 0, 0L);
                    if (i == LB_ERR) return (TRUE);

                    j = SendDlgItemMessage(hwnd, 4211, LB_GETITEMDATA, i, 0L);
                    Glossary[j].hidden = IsDlgButtonChecked(hwnd, 4232);
                    GlossaryChanged = TRUE;
					return (TRUE);

                case IDOK:
                    /* Compact the entries */

                    for (i = 0; i < NrGlossary; i++) {
                        if (Glossary[i].shortcut == NULL) {
                            for (j = i + 1; j < NrGlossary; j++)
                                Glossary[j-1] = Glossary[j];

                            NrGlossary--;
                            i--;
                        }
					}

                    if (NrGlossary > 0) {
						qsort(Glossary, NrGlossary, sizeof(GLOSSARY), GlossaryComp);
                    }

                    if (global.glisthwnd != NULL)
                        SendMessage(global.glisthwnd, WM_USER, 0, 0L);

                    EndDialog(hwnd, FALSE);
                    return (TRUE);
            }

			return (FALSE);
        }

        case WM_COMPAREITEM:
        case WM_DELETEITEM:
        case WM_DRAWITEM:
        case WM_MEASUREITEM:
            return (JlistProc(hwnd, message, wParam, lParam, TRUE, ConvertGlossary));
    }

    return (FALSE);
}



BOOL FAR PASCAL GlossaryListProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i, j;

    switch (message) {
        case WM_INITDIALOG: {
            RECT rect;

            sizing = FALSE;

            GetWindowRect(global.hwnd, &rect);

            SetWindowPos(hwnd, NULL,
                         rect.left + global.glistsize.left,
                         rect.top + global.glistsize.top,
                         0, 0, SWP_NOSIZE | SWP_NOZORDER);

            SetWindowPos(hwnd, NULL, 0, 0, global.glistsize.right, global.glistsize.bottom, SWP_NOMOVE | SWP_NOZORDER);

            SendMessage(hwnd, WM_USER, 0, 0L);

            return (TRUE);
        }

		case WM_USER:
			SendDlgItemMessage(hwnd, 4201, LB_RESETCONTENT, 0, 0L);
			for (i = 0; i < NrGlossary; i++) {
                if (Glossary[i].hidden) continue;
				SendDlgItemMessage(hwnd, 4201, LB_ADDSTRING, 0, (LONG) i);
			}
			return (TRUE);

        case WM_MOVE: {
			RECT rect1, rect2;

            GetWindowRect(global.hwnd, &rect1);
            GetWindowRect(hwnd, &rect2);

            global.glistsize.left = rect2.left - rect1.left;
            global.glistsize.top = rect2.top - rect1.top;
			return (TRUE);
        }

        case WM_SIZE: {
            RECT rect1, rect2, rect3;

			if (sizing) return (TRUE);

            GetWindowRect(hwnd, &rect1);
            GetClientRect(hwnd, &rect2);

            MoveWindow(GetDlgItem(hwnd, 4201), 0, 0, rect2.right, rect2.bottom, TRUE);
            GetWindowRect(GetDlgItem(hwnd, 4201), &rect3);

            rect3.right += (rect1.right - rect1.left - rect2.right);
            rect3.bottom += (rect1.bottom - rect1.top - rect2.bottom);

            sizing = TRUE;

			global.glistsize.right = i = rect3.right - rect3.left;
			global.glistsize.bottom = j = rect3.bottom - rect3.top;

            SetWindowPos(hwnd, NULL, 0, 0, i, j, SWP_NOMOVE | SWP_NOZORDER);
			sizing = FALSE;

            return (TRUE);
        }

        case WM_SYSCOMMAND:
            if ((wParam & 0xfff0) == SC_CLOSE) {
                SetActiveWindow(global.hwnd);

				EndDialog(hwnd, FALSE);
                SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_GLOSSARY);
                CheckMenuItem(hmenu, IDM_VIEWGLOSSARY, MF_UNCHECKED | MF_BYCOMMAND);
                global.glisthwnd = NULL;
                return (TRUE);
            }
            break;

        case WM_COMMAND:
			switch (wParam) {
				case 4201:
					switch (HIWORD(lParam)) {
						case LBN_SELCHANGE: {
							FILEOPTIONS *f;

							i = SendDlgItemMessage(hwnd, 4201, LB_GETCURSEL, 0, 0L);
							if (i == LB_ERR) return (TRUE);
                            if (global.active == NULL) {
                                MessageBeep(0);
                                SendDlgItemMessage(hwnd, 4201, LB_SETCURSEL, LB_ERR, 0L);
								return (TRUE);
							}

							i = SendDlgItemMessage(hwnd, 4201, LB_GETITEMDATA, i, 0L);

                            f = global.active;

                            TurnOffSelection(f);
                            UndoAddTyping(f, f->current, kanjilen(Glossary[i].phrase), FALSE);
							InsertString(f, f->current, Glossary[i].phrase,
								OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND);

                            SendDlgItemMessage(hwnd, 4201, LB_SETCURSEL, LB_ERR, 0L);
							SetActiveWindow(global.hwnd);

                            switch (global.glistbehaviour) {
                                case 0: break;
                                case 1: EndDialog(hwnd, FALSE);
                                        CheckMenuItem(hmenu, IDM_VIEWGLOSSARY, MF_UNCHECKED | MF_BYCOMMAND);
                                        global.glisthwnd = NULL;
                                        SendMessage(global.tbhwnd, BM_SETSTATE, FALSE, ID_GLOSSARY);
										break;
                            }
							return (TRUE);
						}
					}
					return (TRUE);
			}
			return (FALSE);

        case WM_KEYDOWN:
        case WM_CHAR:
            switch (wParam) {
                case VK_ESCAPE: SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
								return (TRUE);
            }
            break;

		case WM_COMPAREITEM:
		case WM_DELETEITEM:
        case WM_DRAWITEM:
        case WM_MEASUREITEM:
            return (JlistProc(hwnd, message, wParam, lParam, TRUE, ConvertGlossary1));
    }

    return (FALSE);
}
