/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"

#include "idm.h"


#define HIGH_BIT_ESCAPE     0

#define MAXUNDOLEVELS       32000


static struct {
    char *name;
} SupportedVersions[] = {
	{ "B1" },
	{ NULL }
};

static char filename[MAXFILENAMELEN];
static char filespec[MAXFILENAMELEN];
static OFSTRUCT of;
static FILE *Fp;
static FILEOPTIONS *filep;
static int FormatChoice;
static BOOL ConvertLineBreaks, AlignKanjis, OutputLineBreaks;
static int LineLength;
static long int FileLength, StatusSteps, CharRead;
static HWND statushwnd, statushwnd1;

static FILEFORMAT OriginalFormat;
static char *OverwriteMessage;

static BOOL HasText, InWord;
static int LineNumber;
static int BlankSpaces, CharPosition;
static int LastIndent, CurrentIndent, FirstIndent;
static int LastChar;
static BOOL LastTextIsKanji;
static BOOL SkipHighBitWarnings;
static int LevelsSaved = 0;

static POSITION p;


char *FileExtensions[] = {
    ".*",
    ".JWP",
    ".TPL",
    ".EUC",
    ".SJS",
    ".JIS",
    ".OLD",
    ".NEC",
    NULL
};

char *FileFormatNames[] = {
	"Auto-Detect",
	"Normal",
    "Template",
	"EUC",
	"Shift JIS",
    "New JIS",
	"Old JIS",
    "NEC JIS",
    NULL
};

BOOL FAR PASCAL FileOpenDlgProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL FileSaveAsDlgProc (HWND, WORD, WORD, LONG);
extern BOOL FAR PASCAL FileFindProc (HWND, WORD, WORD, LONG);



static void MoveStatusBar (long int percent)
{
    long int len;
    char     buffer[10];
    HWND     hwnd;
    RECT     rect;
    HDC      hdc;

    if (statushwnd == NULL) return;

    if (percent < 0L) return;
    if (percent > 100L) percent = 100L;

    sprintf(buffer, "%ld%%", percent);
    SetDlgItemText(statushwnd, 4204, buffer);

    hwnd = GetDlgItem(statushwnd, 4205);
    GetClientRect(hwnd, &rect);
    len = (rect.right - 4) * percent / 100L;

    hdc = GetDC(hwnd);
    SelectObject(hdc, GetStockObject(GRAY_BRUSH));
    Rectangle(hdc, 2, 2, len + 2, rect.bottom - 2);
    ReleaseDC(hwnd, hdc);
}



void NewParagraph (FILEOPTIONS *f, POSITION *pos, unsigned int n)
{
    POSITION p = *pos;
    PARAGRAPH far *pp;
    ONELINE far *lp;


    if (PARAOF(p) == NULL) {
        PARAOF(p) = f->paragraph = f->eof = pp = StructAlloc(PARAGRAPH);
        pp->prev = pp->next = NULL;
    } else {
        PARAOF(p)->next = f->eof = pp = StructAlloc(PARAGRAPH);
        pp->prev = PARAOF(p);
        pp->next = NULL;
        PARAOF(p) = pp;
    }

	pp->leftindent = pp->firstindent = pp->rightindent = 0;

    if (pp->prev != NULL) {
        pp->spacing = pp->prev->spacing;
        pp->spacemulti = pp->prev->spacemulti;
    } else {
        pp->spacing = f->basefont->spacing;
        pp->spacemulti = 100;
    }
	n *= TEXTBLOCKSIZE;
    pp->text = (UNIT far *) BlockAlloc(n * sizeof(UNIT));
    pp->textsize = n;
    pp->text[0].kanji = '\0';

    LINEOF(p) = lp = pp->lines = pp->lastline = StructAlloc(ONELINE);
    lp->next = lp->prev = NULL;
    lp->position = 0;
    lp->height = BASEFONT->height;
    POSOF(p) = lp->length = 0;

    f->nr_lines++;

    LineNumber = 0;
    *pos = p;
}



static void WriteNormalFile (FILE *fp, FILEOPTIONS *f, int saveundo)
{
    int i, j, k;
	UNIT far *cp;
	PARAGRAPH far *pp;
	FILEHEADER header;
	FILEPARAHEADER pheader;
    UNDOBUF far *up;

	memset(&header, 0, sizeof(FILEHEADER));
	memset(&pheader, 0, sizeof(FILEPARAHEADER));

	header.magic = MAGIC;
	strcpy(header.version, JWPFILEVERSION);
	header.landscape = f->pagesetup.landscape ? 1 : 0;
    header.paragraphs = 0;

    header.lrheader = f->lrheader;
    header.nofirstpage = f->nofirstpage;

	for (i = 0; i < 4; i++) header.margins[i] = (float) f->pagesetup.margins[i];

	for (i = 0; i < NRSUMMARIES && f->summary[i] == NULL; i++);
    if (i < NRSUMMARIES) header.summary = TRUE;

    for (j = 0; j < 4; j++) {
        for (i = 0; i < NRHEADERS && f->header[j][i] == NULL; i++);
        if (i < NRHEADERS) {
            header.headers = TRUE;
            break;
        }
    }


    /* Undo's */

    if (saveundo != 0) {
        if (saveundo < 0) saveundo = f->undolevels;

        for (i = 0, up = f->undo; up != NULL; up = up->next, i++);
        f->undolevels = i;

        if (saveundo > f->undolevels) saveundo = f->undolevels;
        header.undo = saveundo;
    } else {
        header.undo = 0;
    }


    /* How many paragraphs? */

    for (pp = f->paragraph; pp != NULL; pp = pp->next, header.paragraphs++);

    fwrite (&header, sizeof(FILEHEADER), 1, fp);


    /* Summaries */

    if (header.summary) {
        KANJI buffer[MAXLINELEN];

        for (i = 0; i < NRSUMMARIES; i++) {
            if (f->summary[i] == NULL) {
                j = 0;
                fwrite(&j, sizeof(int), 1, fp);
                continue;
            }

			j = kanjilen(f->summary[i]) + 1;
            if (j >= MAXLINELEN) j = MAXLINELEN - 1;
            fwrite(&j, sizeof(int), 1, fp);
			kanjicpy(buffer, f->summary[i]);
            fwrite(buffer, sizeof(KANJI), j, fp);
        }
    }

    /* Headers */

    if (header.headers) {
        KANJI buffer[MAXLINELEN];

        for (k = 0; k < 4; k++) {
            for (i = 0; i < NRHEADERS; i++) {
                if (f->header[k][i] == NULL) {
                    j = 0;
                    fwrite(&j, sizeof(int), 1, fp);
                    continue;
                }

                j = kanjilen(f->header[k][i]) + 1;
                if (j >= MAXLINELEN) j = MAXLINELEN - 1;
                fwrite(&j, sizeof(int), 1, fp);
                kanjicpy(buffer, f->header[k][i]);
                fwrite(buffer, sizeof(KANJI), j, fp);
            }
        }
    }

    /* Undo's */

    if (saveundo > 0) {
        unsigned int len;
        PARAGRAPH far *pp;
        char buffer[MAXLINELEN];

        for (i = 0, up = f->undo; i < saveundo && up != NULL; i++, up = up->next) {
            _fmemcpy(buffer, up, sizeof(UNDOBUF));
            fwrite(buffer, sizeof(UNDOBUF), 1, fp);

            switch (up->action) {
                case U_INSERT: break;
                case U_DELETE:
				case U_PARAFORMAT:
                    for (pp = (PARAGRAPH far *) up->data; pp != NULL; pp = pp->next) {
                        _fmemcpy(buffer, pp, sizeof(PARAGRAPH));
                        fwrite(buffer, sizeof(PARAGRAPH), 1, fp);
                        if (pp->text == NULL) {
                            len = 0;
                            fwrite(&len, sizeof(int), 1, fp);
                        } else {
                            len = unitlen(pp->text) + 1;
                            fwrite(&len, sizeof(int), 1, fp);
                            _fmemcpy(buffer, pp->text, len * sizeof(UNIT));
                            fwrite(buffer, sizeof(UNIT), len, fp);
                        }
                    }
                    break;
            }
        }
    }

    /* Output the file */

	for (CharRead = 0L, pp = f->paragraph; pp != NULL; pp = pp->next) {

        pheader.textsize = unitlen(pp->text) + 1;
        pheader.spacemulti = pp->spacemulti;
        pheader.firstindent = pp->firstindent;
        pheader.leftindent = pp->leftindent;
        pheader.rightindent = pp->rightindent;
        fwrite(&pheader, sizeof(FILEPARAHEADER), 1, fp);

        for (cp = pp->text; cp->kanji; cp++, CharRead++) {
			if (CharRead % StatusSteps == 0)
				MoveStatusBar(CharRead / StatusSteps);

            if (ISKANJI(cp->kanji)) {
                fputc(HIBYTE(cp->kanji) | 0x0080, fp);
                fputc(LOBYTE(cp->kanji) | 0x0080, fp);
            } else if (cp->kanji & 0x0080) {
                fputc(HIGH_BIT_ESCAPE, fp);
                fputc(cp->kanji & 0x007f, fp);
			} else {
                fputc(cp->kanji & 0x007f, fp);
            }
		}

        fputc('\n', fp);
	}

	MoveStatusBar(100L);
}



static BOOL ReadNormalFile (FILE *fp, FILEOPTIONS *f, char * filename, HWND hwnd, BOOL template)
{
	unsigned int i, j, n;
	int ch;
	int len;
    int version;
    FILEHEADER header;
    FILEPARAHEADER pheader;
    POSITION p;

    fread (&header, sizeof(FILEHEADER), 1, fp);

	if (header.magic != MAGIC) {
        ErrorMessage(global.hwnd, "%s is not a normal " PROGNAME " %s!",
                        filename, template ? "template" : "file");
        return (FALSE);
    }

    /* Verify the version */

	if (!strcmp(header.version, JWPFILEVERSION)) {
        version = -1;
    } else {
        for (i = 0; SupportedVersions[i].name != NULL; i++)
            if (!strcmp(header.version, SupportedVersions[i].name)) break;

        if (SupportedVersions[i].name == NULL) {
            ErrorMessage(hwnd, "%s is a %s created by an earlier version of "
                                PROGNAME ".  Unfortunately, it cannot be read by "
                                "the current version.  We are so very sorry...",
                                filename, template ? "template" : "file");
            return (FALSE);
        } else {
            if (YesNo(hwnd, "%s is a %s created by an earlier version of " PROGNAME ".  %s",
                            filename, template ? "template" : "file",
                            template ? "Do you still want to use it?" :
                                       "It can be read but will be overwritten "
                                       "by the new format when you save it.  "
                                       "Do you still want to open it?") != IDYES)
                return (FALSE);

            version = i;
        }
    }

	for (i = 0; i < 4; i++) f->pagesetup.margins[i] = (double) header.margins[i];
	f->pagesetup.landscape = header.landscape;
    f->lrheader = header.lrheader;
    f->nofirstpage = header.nofirstpage;
	f->linelen = CalcLineLength(f);


    /* Any summary? */

	for (i = 0; i < NRSUMMARIES; i++) f->summary[i] = NULL;

    if (header.summary) {
        KANJI buffer[MAXLINELEN];

        for (i = 0; i < NRSUMMARIES; i++) {
            fread(&n, sizeof(int), 1, fp);
            if (n > 0) {
                fread((char *) buffer, sizeof(KANJI), n, fp);
				f->summary[i] = (KANJI far *) BlockAlloc((n + 2) * sizeof(KANJI));
				kanjicpy(f->summary[i], buffer);
            }
        }
    }

    /* Any headers? */

    for (j = 0; j < 4; j++) for (i = 0; i < NRHEADERS; i++) f->header[j][i] = NULL;

    if (header.headers) {
        KANJI buffer[MAXLINELEN];

        for (j = 0; j < 4; j++) {
            for (i = 0; i < NRHEADERS; i++) {
                fread(&n, sizeof(int), 1, fp);
                if (n > 0) {
                    fread((char *) buffer, sizeof(KANJI), n, fp);
                    f->header[j][i] = (KANJI far *) BlockAlloc((n + 2) * sizeof(KANJI));
                    kanjicpy(f->header[j][i], buffer);
                }
            }
        }
    }


    /* Any undo chains? */

    f->undo = f->undotail = NULL;
    f->undolevels = 0;

    if (!template && header.undo) {
        unsigned int x, len;
        UNIT ch;
        UNDOBUF far *up = NULL, far *up1;
        PARAGRAPH far *pp, far *pp1;
        char buffer[MAXLINELEN];

        for (i = 0; i < header.undo; i++) {
            fread(buffer, sizeof(UNDOBUF), 1, fp);

            if (up == NULL) {
                up1 = up = StructAlloc(UNDOBUF);
                _fmemcpy(up1, buffer, sizeof(UNDOBUF));
                up1->next = up1->prev = NULL;
            } else {
                up1->next = StructAlloc(UNDOBUF);
                _fmemcpy(up1->next, buffer, sizeof(UNDOBUF));
                up1->next->prev = up1;
                up1 = up1->next;
                up1->next = NULL;
            }

            f->undolevels++;

            switch (up1->action) {
                case U_INSERT: up1->data = NULL; break;
                case U_DELETE:
				case U_PARAFORMAT:
                    pp = NULL;
                    for (;;) {
                        fread(buffer, sizeof(PARAGRAPH), 1, fp);

                        if (pp == NULL) {
                            pp1 = pp = StructAlloc(PARAGRAPH);
                            _fmemcpy(pp1, buffer, sizeof(PARAGRAPH));
                            pp1->next = pp1->prev = NULL;
                        } else {
                            pp1->next = StructAlloc(PARAGRAPH);
                            _fmemcpy(pp1->next, buffer, sizeof(PARAGRAPH));
                            pp1->next->prev = pp1;
                            pp1 = pp1->next;
                            pp1->next = NULL;
                        }

                        fread(&len, sizeof(unsigned int), 1, fp);
                        if (len > 0) {
                            pp1->text = (UNIT far *) BlockAlloc(len + 5);
                            for (x = 0; x < len; x++) {
                                fread(&ch, sizeof(UNIT), 1, fp);
                                pp1->text[x] = ch;
                            }
                            pp1->text[len].kanji = 0;
                        } else {
                            pp1->text = NULL;
                        }

                        if (pp1->next == NULL) break;
                    }

                    up1->data = (void far *) pp;
                    break;
            }
        }

        f->undo = up;
        f->undotail = up1;

        TrimUndoChain(f);
    }

    /* Read the file */

    PARAOF(p) = NULL;
    LINEOF(p) = NULL;
    POSOF(p) = 0;

    CharRead = 0L;

    for (i = 0; i < header.paragraphs; i++) {
        OLDFILEPARAHEADER100 old1;

        switch (version) {
            default:
            case -1: fread(&pheader, sizeof(FILEPARAHEADER), 1, fp); break;
            case 0:  fread(&old1, sizeof(OLDFILEPARAHEADER100), 1, fp);
                     pheader.textsize = old1.textsize;
                     pheader.firstindent = old1.firstindent;
                     pheader.leftindent = old1.leftindent;
                     pheader.rightindent = old1.rightindent;
                     pheader.spacemulti = 100;
                     break;
        }

        len = pheader.textsize;
		n = (len / TEXTBLOCKSIZE) + 1;

		NewParagraph(f, &p, n);

        PARAOF(p)->spacemulti = pheader.spacemulti;
        PARAOF(p)->spacing = (f->basefont->height + f->basefont->spacing) * pheader.spacemulti / 100;
        PARAOF(p)->spacing -= f->basefont->height;
        PARAOF(p)->firstindent = pheader.firstindent;
        PARAOF(p)->leftindent = pheader.leftindent;
        PARAOF(p)->rightindent = pheader.rightindent;

		for (j = 0; j < len; j++) {
			ch = fgetc(fp);

			if (ch & 0x0080) {      /* Kanji */
				PARAOF(p)->text[j].kanji = ((ch & 0x007f) << 8) | (fgetc(fp) & 0x007f);
				CharRead += 2L;
            } else if (ch == HIGH_BIT_ESCAPE) {
                PARAOF(p)->text[j].kanji = fgetc(fp) | 0x0080;
                CharRead += 2L;
			} else {
				PARAOF(p)->text[j].kanji = ch;
				CharRead++;
			}
            f->nr_bytes++;
            if (StatusSteps > 0 && CharRead % StatusSteps == 0)
				MoveStatusBar(CharRead / StatusSteps);
		}
        PARAOF(p)->text[len-1].kanji = 0;
        LINEOF(p)->length = len-1;
	}

    MoveStatusBar(100L);
	return (TRUE);
}



static int OpenCharIn (void)
{
    long int percent;
    int ch;

    ch = fgetc(Fp);

    if (statushwnd == NULL) return (ch);

    CharRead++;

    if (CharRead % StatusSteps == 0 || ch == EOF) {
        percent = CharRead / StatusSteps;
        if (ch == EOF || CharRead >= FileLength) percent = 100L;

        MoveStatusBar(percent);
    }

    return (ch);
}


static void OpenCharOut (int ch)
{
    int i, j;
    KANJI k;
    KANJI buf[10];


    if (ch == '\r') return;         /* Ignore */

    /* Leading spaces */

    if (ch <= ' ' && ch != '\n') {
        if (!HasText) {
            CurrentIndent++;
        } else {
            BlankSpaces++;
        }
        return;
    }

    if (ch == '\n') {
        CurrentIndent = 0;
        LineNumber++;

        if (!ConvertLineBreaks) {
            NewParagraph(filep, &p, 1);
            FirstIndent = LastIndent = 0;
            LineNumber = 0;
        } else if (!HasText) {             /* Blank line */
            NewParagraph(filep, &p, 1);
            FirstIndent = LastIndent = -1;
            LineNumber = 2;
        }

        BlankSpaces = 0;

        HasText = FALSE;
        return;
    }

    /* Check indents */

    if (!HasText) {
        switch (LineNumber) {
            case 0:
                /* No need to check the first line */
                FirstIndent = LastIndent = CurrentIndent;
				PARAOF(p)->firstindent = PARAOF(p)->leftindent = FirstIndent / 2;
                break;

            case 1:
                /* The second line */
                LastIndent = CurrentIndent;
                PARAOF(p)->leftindent = LastIndent / 2;
                break;

            default:
                /* The rest */
                if (LastIndent != CurrentIndent) {
                    NewParagraph(filep, &p, 1);

                    FirstIndent = LastIndent = CurrentIndent;
					PARAOF(p)->firstindent = PARAOF(p)->leftindent = FirstIndent / 2;
                }
                break;
        }
    }


    /* Take care of the blank spaces */

    if (BlankSpaces > 0) {
        j = (BlankSpaces + 1) / 2;

        if (!HasText) {
            k = 0x2121;
        } else if (!LastTextIsKanji && (ch & 0x80)) {
            if (AlignKanjis) {
                j = 1;
                k = '\t';
            } else {
                j = BlankSpaces;
                k = ' ';
            }
        } else if (LastTextIsKanji && !(ch & 0x80)) {
            j = BlankSpaces;
            k = ' ';
        } else if (!LastTextIsKanji && !(ch & 0x80)) {
            j = BlankSpaces;
            k = ' ';
        } else {
            k = 0x2121;
        }

        buf[0] = k;
        buf[1] = 0;
        for (i = 0; i < j; i++, POSOF(p)++) InsertString (filep, p, buf, 0);

        BlankSpaces = 0;
    }


    /* Insert the character */

    i = 0;

    if (LastChar != 0) {
        if (!HasText && LineNumber > 0 && !LastTextIsKanji) buf[i++] = ' ';
        if (ch & 0x80) {
            buf[i++] = (LastChar << 8) | (ch & 0x7f);
            LastChar = 0;
            LastTextIsKanji = TRUE;
        } else {
            buf[i++] = LastChar;
            buf[i++] = ch;
            LastChar = 0;
            LastTextIsKanji = FALSE;
        }
    } else if (ch & 0x80) {
        LastChar = (ch & 0x7f);
        return;
    } else {
        if (!HasText && LineNumber > 0) buf[i++] = ' ';
        buf[i++] = ch;
        LastTextIsKanji = FALSE;
    }

    buf[i] = 0;

    InsertString (filep, p, buf, 0);
    HasText = TRUE;
    BlankSpaces = 0;

    POSOF(p) += i;
}



BOOL DoFileOpen (char *pathname)
{
    int         fd;
	FILEFORMAT  incode;
	HCURSOR     hCursor;
    FILEOPTIONS *ToClose = NULL;
    char        buffer[256];


    /* Is this the first file opened? */

    if (fileoptions != NULL && fileoptions->next == NULL && !fileoptions->changed) {
        GetWindowText(fileoptions->parent, buffer, 256);
        if (!strcmp(buffer, "Untitled #1")) ToClose = fileoptions;
    }


    if (pathname == NULL) {
        if (!DialogBox(hInstance, "FileOpen", global.hwnd, FileOpenDlgProc)) {
			return (FALSE);
        }
	} else {
		strcpy(filename, pathname);
        FormatChoice = FF_UNKNOWN;
    }

    hCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);

    incode = (FILEFORMAT) FormatChoice;

    LastIndent = CurrentIndent = FirstIndent = LastChar = 0;
	LineNumber = BlankSpaces = 0;
    HasText = FALSE;
    LastTextIsKanji = TRUE;

	fd = OpenFile(filename, &of, OF_READ);
	if (fd < 0) return (FALSE);

    /* Normal format? */

    if (incode == FF_UNKNOWN) {
        unsigned long int magic;

        lseek(fd, 0L, 0);
		read(fd, &magic, sizeof(unsigned long int));
        lseek(fd, 0L, 0);

		if (magic == MAGIC) incode = FF_NORMAL;
    }


    Fp = fdopen(fd, "r");
    if (Fp == NULL) {
        close(fd);
        return (FALSE);
    }


    SetupIO(OpenCharIn, OpenCharOut);

    fseek(Fp, 0L, 2);
    FileLength = ftell(Fp);
    StatusSteps = (FileLength / 100L) + 1L;
    CharRead = 0L;

	fseek(Fp, 0L, 0);


    /* Open the status dialog box */

    statushwnd1 = CreateDialog(hInstance, "FileStatus", 0, NULL);
    SetDlgItemText(statushwnd1, 4201, "Reading File:");
    sprintf(buffer, "%s  (%0.2fK)", filename, (double) FileLength / 1024.0);
    SetDlgItemText(statushwnd1, 4202, buffer);
    SetDlgItemText(statushwnd1, 4203, "Determining File Format...");
    SetDlgItemText(statushwnd1, 4204, "0%");

    statushwnd = NULL;
    if (incode == FF_UNKNOWN) {
        incode = DetectCodeType();
        if (incode == FF_UNKNOWN) {
            if (YesNo(statushwnd1, "The file %s is not in any of the "
                      "supported Japanese text formats.  Open it anyway?", filename) != IDYES) {
                fclose(Fp);
                DestroyWindow(statushwnd1);
                statushwnd1 = NULL;
                return (FALSE);
            }
        }
        fseek(Fp, 0L, 0);
    }

    sprintf(buffer, "File Format: %s", (incode == FF_UNKNOWN) ? "Unknown" : FileFormatNames[incode]);
    statushwnd = statushwnd1;
    SetDlgItemText(statushwnd, 4203, buffer);


    filep = NewFile(FN_NORMAL, (incode != FF_NORMAL && incode != FF_TEMPLATE));
    if (filep == NULL) {
        fclose(Fp);
        ErrorMessage(global.hwnd, "Cannot open any more file window!");
        DestroyWindow(statushwnd);
        statushwnd = NULL;
        return (FALSE);
    }

    if (incode == FF_NORMAL || incode == FF_TEMPLATE) {
		strcpy(filep->filename, of.szPathName);

        if (!ReadNormalFile(Fp, filep, filep->filename, statushwnd, FALSE)) {
            FILEOPTIONS *f1;

            fclose(Fp);
            DestroyWindow(statushwnd);
            statushwnd = NULL;

            ShowCursor(FALSE);
            SetCursor(hCursor);

            CloseFile(filep);

            /* Unlink it */

            if (fileoptions == filep) {
                fileoptions = filep->next;
            } else {
                for (f1 = fileoptions; f1 != NULL && f1->next != filep; f1 = f1->next);
                if (f1 == NULL) {
                    ErrorMessage(global.hwnd, "Funny: cannot find FILEOPTIONS record to deallocate");
                } else {
                    f1->next = filep->next;
                }
            }
            return (FALSE);
        }
    } else {
        filep->filename[0] = '\0';

        PARAOF(p) = filep->paragraph;
        LINEOF(p) = filep->paragraph->lines;
        POSOF(p) = 0;

        FileImport(incode);
	}

    fclose(Fp);



    /* Reformat */

    SetDlgItemText(statushwnd1, 4201, "Reformatting...");

    TOPPARA(filep) = CURPARA(filep) = filep->paragraph;
    TOPLINE(filep) = CURLINE(filep) = filep->paragraph->lines;
    CURCHAR(filep) = 0;

	ReformatFile(filep);

    DestroyWindow(statushwnd);
    statushwnd = NULL;


    if (incode != FF_UNKNOWN && incode != FF_NORMAL && incode != FF_TEMPLATE) {
        filep->fileformat = incode;
        sprintf(buffer, "%s (%s)", filename, FileFormatNames[incode]);
    } else {
        strcpy(buffer, filename);
    }
    strcpy(filep->filename, filename);
    ShowFileWindow(buffer, filep);


    if (ToClose != NULL) {
        ToClose->changed = FALSE;
        SendMessage(global.clienthwnd, WM_MDIDESTROY, ToClose->parent, 0L);
    }

    ShowCursor(FALSE);
    SetCursor(hCursor);

	filep->changed = FALSE;

	PutQuickFile(of.szPathName);
    UpdateQuickFiles();

    return (TRUE);
}



FILEOPTIONS *CreateUsingTemplate (char far *tpl, int Untitled)
{
	int fd;
	FILE *Fp;
	OFSTRUCT of;
    FILEOPTIONS *f, *ToClose;
    char buffer[MAXLINELEN];

    /* Is this the first file opened? */

    if (fileoptions != NULL && fileoptions->next == NULL && !fileoptions->changed) {
        GetWindowText(fileoptions->parent, buffer, 256);
        if (!strcmp(buffer, "Untitled #1")) ToClose = fileoptions;
    }


    fd = OpenFile(tpl, &of, OF_READ);
    if (fd < 0) return (NULL);

    Fp = fdopen(fd, "r");
    if (Fp == NULL) {
        close(fd);
        return (NULL);
    }

	fseek(Fp, 0L, 0);

    f = NewFile(FN_NORMAL, FALSE);
    if (f == NULL) {
        fclose(Fp);
        ErrorMessage(global.hwnd, "Cannot open any more file window!");
        return (NULL);
    }

    statushwnd = NULL;
    StatusSteps = 0L;

    if (!ReadNormalFile(Fp, f, "This", global.hwnd, TRUE)) {
        FILEOPTIONS *f1;

        fclose(Fp);
		CloseFile(f);

        /* Unlink it */

        if (fileoptions == f) {
            fileoptions = f->next;
        } else {
            for (f1 = fileoptions; f1 != NULL && f1->next != f; f1 = f1->next);
            if (f1 == NULL) {
                ErrorMessage(global.hwnd, "Funny: cannot find FILEOPTIONS record to deallocate");
            } else {
                f1->next = f->next;
            }
		}
        return (NULL);
    }

    close(fd);

    TOPPARA(f) = CURPARA(f) = f->paragraph;
    TOPLINE(f) = CURLINE(f) = f->paragraph->lines;
    CURCHAR(f) = 0;

    ReformatFile(f);

    f->fileformat = FF_NORMAL;
    sprintf(buffer, "Untitled #%d", Untitled);
    ShowFileWindow(buffer, f);

    f->changed = FALSE;

    if (ToClose != NULL) {
        ToClose->changed = FALSE;
        SendMessage(global.clienthwnd, WM_MDIDESTROY, ToClose->parent, 0L);
    }
    return (f);
}



static int SaveCharIn (void)
{
    int i;
	KANJI ch, temp;
    long int percent;
    HWND hwnd;

	if (LastChar != 0) {
        ch = LastChar;
		LastChar = 0;
        CharPosition++;
		return (ch);
    } else if (PARAOF(p) == NULL || LINEOF(p) == NULL) {
        ch = EOF;
        goto StatusBar;     /* Put the bar to 100% */
    } if (OutputLineBreaks && !HasText && CharPosition <
            (LineNumber ? PARAOF(p)->leftindent : PARAOF(p)->firstindent) * 2) {
        CharPosition++;
		return (' ');
	} else if (OutputLineBreaks && CharPosition >= LineLength - PARAOF(p)->rightindent) {
		HasText = InWord = FALSE;
        CharPosition = 0;
        LineNumber++;
        return ('\n');
    }


    for (;;) {
        ch = (CHAROF(p, POSOF(p)) & 0x7fff);
        if (ISKANJI(ch)) ch &= 0x7f7f;

        if (!ch) {
			POSOF(p) = 0;
            PARAOF(p) = PARAOF(p)->next;
			if (PARAOF(p) != NULL) LINEOF(p) = PARAOF(p)->lines;
            else LINEOF(p) = NULL;

            HasText = InWord = FALSE;
            CharPosition = LineNumber = 0;
            return ('\n');
		}

        if (!HasText && (ch <= ' ' || ch == 0x2121)) {
			POSOF(p)++;
            CharRead++;
			continue;
		}

		if (ISKANJI(ch)) {
			if (ch == 0x2121) {
				ch = LastChar = ' ';
			} else {
                LastChar = (LOBYTE(ch) | 0x80);
				ch = (HIBYTE(ch) | 0x80);
			}
            InWord = FALSE;
        } else if (ch <= ' ') {
            ch = ' ';
			InWord = FALSE;
        } else {
            if (!SkipHighBitWarnings && ch & 0x80) {
                if (statushwnd == NULL) hwnd = global.hwnd; else hwnd = statushwnd;

                if (YesNo(hwnd, "This file contains "
                                "the extended ASCII character '%c'.  "
                                "Unfortunately, the file format you have "
                                "chosen does not support extended ASCII "
                                "characters.  "
                                "The high-bit of this character will "
                                "be lost in the saved file, and it will "
                                "become '%c'.\n\n"
                                "Do you want to continue reporting "
                                "other extended ASCII characters?",
                                ch, ch & 0x7f) == IDNO)
                    SkipHighBitWarnings = TRUE;
            }

			ch &= 0x7f7f;

            if (strchr("-", ch) != NULL) {
                InWord = FALSE;
            } else {
                if (!InWord) {          /* Search for word break */
                    for (i = 1; ; i++) {
                        temp = CHAROF(p, POSOF(p)+i);
                        if (!temp) break;
                        if (IsSmallKana(temp)) break;
                        if (temp <= ' ' || strchr("-", temp) != NULL) break;

                        if (OutputLineBreaks && CharPosition + i >= LineLength - PARAOF(p)->rightindent) {
							HasText = InWord = FALSE;
							CharPosition = 0;
                            LineNumber++;
                            return ('\n');
                        }
                    }
                }
                InWord = TRUE;
            }
        }

		break;
    }

	POSOF(p)++;
    CharPosition++;
    HasText = TRUE;

    if (statushwnd == NULL) return (ch);

    CharRead++;

StatusBar:

	if (CharRead % StatusSteps == 0 || ch == EOF) {
        percent = CharRead / StatusSteps;
		if (ch == EOF || CharRead >= FileLength) percent = 100L;

        MoveStatusBar(percent);
    }
	return (ch);
}



static void SaveCharOut (int ch)
{
    fputc(ch, Fp);
}



BOOL FAR PASCAL FileSaveFormatProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    switch (message) {
        case WM_INITDIALOG: {
            char buffer[100];

            sprintf(buffer, OverwriteMessage, filename, FileFormatNames[OriginalFormat]);
            SetDlgItemText(hwnd, 4211, buffer);
            SetDlgItemText(hwnd, 4201, FileFormatNames[OriginalFormat]);
            CenterDialogBox(hwnd);
            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case IDOK:      OriginalFormat = FF_NORMAL;
                                EndDialog(hwnd, TRUE);
                                return (TRUE);

                case IDCANCEL:  OriginalFormat = FF_UNKNOWN;
                                EndDialog(hwnd, TRUE);
                                return (TRUE);

                case 4201:      EndDialog(hwnd, TRUE);
                                return (TRUE);

                default:        break;
            }
            break;
    }

    return (FALSE);
}



BOOL DoFileSaveAs (BOOL Dialog)
{
	int         fd;
    int         i, olddrive;
	FILEFORMAT  outcode;
	HCURSOR     hCursor;
    char        buffer[MAXLINELEN];
	char        olddir[MAXLINELEN];

    if (global.active == NULL) return (FALSE);

    filep = global.active;

    if (Dialog || !filep->filename[0]) {
        if (!DialogBox(hInstance, "FileSaveAs", global.hwnd, FileSaveAsDlgProc)) {
			return (FALSE);
        }
        outcode = (FILEFORMAT) FormatChoice + FF_NORMAL;

        if (outcode == FF_NORMAL)
            SendMessage(global.hwnd, WM_COMMAND, IDM_FILESUMMARY, 0L);
    } else if (filep->fileformat != FF_NORMAL && filep->fileformat != FF_TEMPLATE) {
        BOOL same;

        strcpy(filename, filep->filename);

        /* Change the extension to .JWP */

        for (i = strlen(filename) - 1; i >= 0; i--) {
            if (filename[i] == '\\' || filename[i] == '/' ||
				filename[i] == '.' || filename[i] == ':') break;
        }

        if (i < 0 || filename[i] != '.') {
            strcat(filename, FileExtensions[1]);    /* Normal extension */
            same = FALSE;
        } else if (!strncpy(filename + i, FileExtensions[1], 4)) {
            same = TRUE;
        } else {
            strcpy(filename + i, FileExtensions[1]);
            same = FALSE;
        }


        /* Overwrite with normal format? */

        OriginalFormat = filep->fileformat;

        if (!same) OverwriteMessage = "Save file as %s (Normal Format)?\nOr keep it %s?";
        else OverwriteMessage = "Overwrite %s with Normal Format?\nOr keep it %s?";

        MessageBeep(0);

        DialogBox(hInstance, "FileSaveFormat", global.hwnd, FileSaveFormatProc);

        if (OriginalFormat == FF_UNKNOWN) return (FALSE);

        if (OriginalFormat != FF_NORMAL) strcpy(filename, filep->filename);

        outcode = OriginalFormat;
        FormatChoice = outcode - FF_NORMAL;

        if (outcode == FF_NORMAL)
            SendMessage(global.hwnd, WM_COMMAND, IDM_FILESUMMARY, 0L);
    } else {
        strcpy(filename, filep->filename);
        outcode = filep->fileformat;
        FormatChoice = outcode - FF_NORMAL;
    }

    /* Template? */

    if (outcode == FF_TEMPLATE) {
        if (filep->summary[0] == NULL) {    /* No title */
            ErrorMessage(global.hwnd, "You must provide a Title for this template.");
            return (FALSE);
        }

        /* Switch to the JWP directory */

		getcwd(olddir, MAXLINELEN);
		if (global.jwppath[0] - 'A' != getdisk()) {
			olddrive = getdisk();
			setdisk(global.jwppath[0] - 'A');
        } else {
            olddrive = -1;
        }

		_fstrcpy(buffer, global.jwppath);
        i = strlen(buffer);
        if (i > 3) buffer[i-1] = '\0';

		if (chdir(buffer) < 0) {
			ErrorMessage(global.hwnd, "Cannot switch to the " PROGNAME " directory!");
			if (olddrive >= 0) setdisk(olddrive);
            return (FALSE);
        }
    }


    /* Now save the file */

    hCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);

	LineNumber = LastChar = CharPosition = 0;
	HasText = InWord = FALSE;

	fd = OpenFile(filename, &of, OF_WRITE | OF_CREATE);
	if (fd < 0) return (FALSE);

    if (outcode == FF_NORMAL || outcode == FF_TEMPLATE) {
        Fp = fdopen(fd, "wb");
    } else {
        Fp = fdopen(fd, "wt");
    }
	if (Fp == NULL) {
        close(fd);
        return (FALSE);
    }

	SetupIO(SaveCharIn, SaveCharOut);

    FileLength = 0;
    StatusSteps = (FileLength / 100L) + 1L;
    CharRead = 0L;

	fseek(Fp, 0L, 0);

    /* Open the status dialog box */

    statushwnd = CreateDialog(hInstance, "FileStatus", 0, NULL);
    SetDlgItemText(statushwnd, 4201, "Writing File:");
	SetDlgItemText(statushwnd, 4202, filename);
    sprintf(buffer, "File Format: %s", FileFormatNames[outcode]);
    SetDlgItemText(statushwnd, 4203, buffer);
    SetDlgItemText(statushwnd, 4204, "0%");


    CharRead = 0L;
    FileLength = filep->nr_bytes;

    StatusSteps = (FileLength / 100L) + 1L;


    if (outcode == FF_NORMAL) {
        WriteNormalFile(Fp, filep, global.saveundolevels);
    } else if (outcode == FF_TEMPLATE) {
        WriteNormalFile(Fp, filep, 0);
	} else {
        PARAOF(p) = filep->paragraph;
		LINEOF(p) = filep->paragraph->lines;
        POSOF(p) = 0;

        SkipHighBitWarnings = FALSE;
        FileExport(outcode);
    }

    fclose(Fp);

    strcpy(filep->filename, filename);
    filep->fileformat = outcode;

    if (outcode != FF_NORMAL && outcode != FF_TEMPLATE) {
        sprintf(buffer, "%s (%s)", filep->filename, FileFormatNames[outcode]);
    } else {
        strcpy(buffer, filep->filename);
    }
    SetWindowText(filep->parent, buffer);

    DestroyWindow(statushwnd);
    statushwnd = NULL;

    if (outcode == FF_TEMPLATE) {
		if (olddrive >= 0) setdisk(olddrive);
		chdir(olddir);
    }

    ShowCursor(FALSE);
    SetCursor(hCursor);

	filep->changed = FALSE;

	PutQuickFile(of.szPathName);
    UpdateQuickFiles();

	return (TRUE);
}



BOOL FAR PASCAL FileOpenDlgProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	int i, j;


	switch (message) {

	case WM_INITDIALOG:
		strcpy(filename,"*.*");
        strcpy(filespec,"*.*");

        SendDlgItemMessage(hwnd, 4201, EM_LIMITTEXT, MAXFILENAMELEN, 0L);
		SendDlgItemMessage(hwnd, 4202, EM_LIMITTEXT, MAXFILENAMELEN, 0L);

        DlgDirList(hwnd, filespec, 4211, 4202, 0x0000);
        DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);

        SetDlgItemText(hwnd, 4201, filespec);

		for (i = 0; FileFormatNames[i] != NULL; i++)
            SendDlgItemMessage(hwnd, 4221, CB_INSERTSTRING, -1, (LONG) ((LPSTR) FileFormatNames[i]));
        SendDlgItemMessage(hwnd, 4221, CB_SETCURSEL, 0, 0L);

        CheckDlgButton (hwnd, 4222, FALSE);
        CheckDlgButton (hwnd, 4223, TRUE);

        CenterDialogBox(hwnd);

		return (TRUE);

    case WM_CHAR:
        switch (wParam) {
            case '\x1b': SendMessage(hwnd, WM_COMMAND, IDCANCEL, 0L);
                         return (TRUE);
        }
        break;

	case WM_COMMAND:
		switch (wParam) {
			case 4211:
                switch (HIWORD(lParam)) {
					case LBN_SELCHANGE:
					case LBN_DBLCLK:
                        DlgDirSelect(hwnd, filename, 4211);
                        SetDlgItemText(hwnd, 4201, filename);
						if (HIWORD(lParam) == LBN_DBLCLK)
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                        return (TRUE);
                }
                break;

			case 4212: {
				char directory[MAXFILENAMELEN];

				switch (HIWORD(lParam)) {
					case LBN_DBLCLK:
                        DlgDirSelect(hwnd, directory, 4212);

						strcat(directory, "*.*");

                        DlgDirList(hwnd, directory, 4212, 4202, 0xc010);
                        DlgDirList(hwnd, filespec, 4211, 4202, 0x0000);

                        SetDlgItemText(hwnd, 4201, filespec);
						return (TRUE);
				}
				break;
			}

			case 4201:
				if (HIWORD(lParam) == EN_CHANGE)
                    EnableWindow(GetDlgItem(hwnd, IDOK), (BOOL) SendMessage(LOWORD(lParam), WM_GETTEXTLENGTH, 0, 0L));
				return (TRUE);

			case 4221:
                if (HIWORD(lParam) == LBN_SELCHANGE) {
                    i = (WORD) SendMessage(GetDlgItem(hwnd, 4221), CB_GETCURSEL, 0, 0L);
					strcpy(filespec,"*");
                    strcat(filespec, FileExtensions[i]);
                    SetDlgItemText(hwnd, 4201, filespec);
                    SendMessage(hwnd, WM_COMMAND, IDOK, 0L);

                    return (TRUE);
                }
                break;

            case 4231:      /* File Find */
                i = DialogBox(hInstance, "FileFind", hwnd, FileFindProc);

                if (i) EndDialog(hwnd, TRUE);
                return (TRUE);

			case IDOK:
                GetDlgItemText(hwnd, 4201, filename, 80);

				i = strlen(filename);
				j = filename[i-1];

				if (j == '\\' || j == ':') strcat(filename, "*.*");

				if (strchr(filename, '*') != NULL || strchr(filename, '?') != NULL) {
					if (DlgDirList(hwnd, filename, 4211, 4202, 0x0000)) {
                        DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);
						strcpy(filespec, filename);
                        SetDlgItemText(hwnd, 4201, filespec);
					} else {
						MessageBeep(0);
					}

					return (TRUE);
				}

				strcat(strcat(filename, "\\"), "*.*");

                if (DlgDirList(hwnd, filename, 4211, 4202, 0x0000)) {
                    DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);
					strcpy(filespec, filename);
                    SetDlgItemText(hwnd, 4201, filespec);
					return (TRUE);
				}

                filename[i] = '\0';

				if (OpenFile(filename, &of, OF_READ | OF_EXIST) == -1) {
                    strcat(filename, FileExtensions[1]);    /* Normal */
                    if (OpenFile(filename, &of, OF_READ | OF_EXIST) == -1) {
						MessageBeep(0);
						return (TRUE);
					}
				}

				OemToAnsi(filename, filename);
                AnsiUpper(filename);

                FormatChoice = (WORD) SendDlgItemMessage(hwnd, 4221, CB_GETCURSEL, 0, 0L);
                ConvertLineBreaks = IsDlgButtonChecked(hwnd, 4222);
				AlignKanjis = IsDlgButtonChecked(hwnd, 4223);

				EndDialog(hwnd, TRUE);
				return (TRUE);

            case IDCANCEL:
				EndDialog(hwnd, FALSE);
				return (TRUE);
		}
	}

	return (FALSE);
}



BOOL FAR PASCAL FileSaveAsDlgProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int fd;
	int i, j;
	char buffer[10];


	switch (message) {

	case WM_INITDIALOG:
        sprintf(filename,"*%s",FileExtensions[FF_NORMAL]);
        strcpy(filespec,filename);

        SendDlgItemMessage(hwnd, 4201, EM_LIMITTEXT, MAXFILENAMELEN, 0L);
        SendDlgItemMessage(hwnd, 4202, EM_LIMITTEXT, MAXFILENAMELEN, 0L);

		DlgDirList(hwnd, filespec, 4211, 4202, 0x0000);
        DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);

        SetDlgItemText(hwnd, 4201, filespec);

        for (i = FF_NORMAL; FileFormatNames[i] != NULL; i++)    /* Skips Auto-Detect */
			SendDlgItemMessage(hwnd, 4221, CB_INSERTSTRING, -1, (LONG) ((LPSTR) FileFormatNames[i]));

        /* Select the Normal format */
        SendDlgItemMessage(hwnd, 4221, CB_SETCURSEL, 0, 0L);

        CheckDlgButton (hwnd, 4222, 0);
        EnableWindow(GetDlgItem(hwnd, 4222), FALSE);
		SetDlgItemText(hwnd, 4223, "");
        EnableWindow(GetDlgItem(hwnd, 4223), FALSE);

        if (global.saveundolevels == 0) {
            CheckDlgButton (hwnd, 4224, 0);
            EnableWindow(GetDlgItem(hwnd, 4225), FALSE);
        } else {
            CheckDlgButton (hwnd, 4224, 1);
            EnableWindow(GetDlgItem(hwnd, 4225), TRUE);
        }

        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "All");
        SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 0, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "1");
        if (global.saveundolevels >= 1) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 1, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "5");
        if (global.saveundolevels >= 5) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 2, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "10");
        if (global.saveundolevels >= 10) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 3, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "50");
        if (global.saveundolevels >= 50) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 4, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "100");
        if (global.saveundolevels >= 100) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 5, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "500");
        if (global.saveundolevels >= 500) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 6, 0L);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) "1000");
        if (global.saveundolevels >= 1000) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 7, 0L);
        sprintf(buffer, "%d", MAXUNDOLEVELS);
        SendDlgItemMessage(hwnd, 4225, CB_INSERTSTRING, -1, (LONG) buffer);
        if (global.saveundolevels >= MAXUNDOLEVELS) SendDlgItemMessage(hwnd, 4225, CB_SETCURSEL, 8, 0L);

        CenterDialogBox(hwnd);

		return (TRUE);

    case WM_CHAR:
        switch (wParam) {
			case '\x1b': SendMessage(hwnd, WM_COMMAND, IDCANCEL, 0L);
						 return (TRUE);
		}
		break;

	case WM_COMMAND:
		switch (wParam) {
			case 4211:
				switch (HIWORD(lParam)) {
                    case LBN_SELCHANGE:
					case LBN_DBLCLK:
                        DlgDirSelect(hwnd, filename, 4211);
                        SetDlgItemText(hwnd, 4201, filename);
						if (HIWORD(lParam) == LBN_DBLCLK)
                            SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
						return (TRUE);
                }
                break;

			case 4212: {
				char directory[MAXFILENAMELEN];

                switch (HIWORD(lParam)) {
					case LBN_DBLCLK:
                        DlgDirSelect(hwnd, directory, 4212);

						strcat(directory, "*.*");

                        DlgDirList(hwnd, directory, 4212, 4202, 0xc010);
                        DlgDirList(hwnd, filespec, 4211, 4202, 0x0000);

                        SetDlgItemText(hwnd, 4201, filespec);
						return (TRUE);
				}
				break;
			}

			case 4201:
				if (HIWORD(lParam) == EN_CHANGE)
					EnableWindow(GetDlgItem(hwnd, IDOK), (BOOL) SendMessage(LOWORD(lParam), WM_GETTEXTLENGTH, 0, 0L));
				return (TRUE);

			case 4221:
                if (HIWORD(lParam) == LBN_SELCHANGE) {
                    i = (WORD) SendMessage(GetDlgItem(hwnd, 4221), CB_GETCURSEL, 0, 0L);
                    if (i > (FF_TEMPLATE - FF_NORMAL)) {       /* Not Normal? */
                        EnableWindow(GetDlgItem(hwnd, 4222), TRUE);
                        EnableWindow(GetDlgItem(hwnd, 4223), IsDlgButtonChecked(hwnd, 4222));
                        EnableWindow(GetDlgItem(hwnd, 4224), FALSE);
                        EnableWindow(GetDlgItem(hwnd, 4225), FALSE);
                    } else {
						EnableWindow(GetDlgItem(hwnd, 4222), FALSE);
                        EnableWindow(GetDlgItem(hwnd, 4223), FALSE);
                        EnableWindow(GetDlgItem(hwnd, 4224), TRUE);
                        EnableWindow(GetDlgItem(hwnd, 4225), IsDlgButtonChecked(hwnd, 4224));
                    }
                    strcpy(filespec,"*");
                    strcat(filespec, FileExtensions[i + 1]);
                    SetDlgItemText(hwnd, 4201, filespec);
                    SendMessage(hwnd, WM_COMMAND, IDOK, 0L);
                    return (TRUE);
                }
                break;

			case 4222:
				if (IsDlgButtonChecked(hwnd, 4222)) {
					sprintf(buffer, "%d", global.active->linelen);
                    EnableWindow(GetDlgItem(hwnd, 4223), TRUE);
                    SetDlgItemText(hwnd, 4223, buffer);
				} else {
                    EnableWindow(GetDlgItem(hwnd, 4223), FALSE);
                    SetDlgItemText(hwnd, 4223, "");
				}
				return (TRUE);

            case 4224:
                if (IsDlgButtonChecked(hwnd, 4224)) {
                    EnableWindow(GetDlgItem(hwnd, 4225), TRUE);
				} else {
                    EnableWindow(GetDlgItem(hwnd, 4225), FALSE);
				}
				return (TRUE);

            case IDOK:
				GetDlgItemText(hwnd, 4201, filename, 80);

				i = strlen(filename);
				j = filename[i-1];

				if (j == '\\' || j == ':') strcat(filename, "*.*");

				if (strchr(filename, '*') != NULL || strchr(filename, '?') != NULL) {
                    if (DlgDirList(hwnd, filename, 4211, 4202, 0x0000)) {
                        DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);
                        strcpy(filespec, filename);
                        SetDlgItemText(hwnd, 4201, filespec);
					} else {
						MessageBeep(0);
					}

					return (TRUE);
				}

				strcat(strcat(filename, "\\"), "*.*");

                if (DlgDirList(hwnd, filename, 4211, 4202, 0x0000)) {
                    DlgDirList(hwnd, "*.*", 4212, 4202, 0xc010);
					strcpy(filespec, filename);
                    SetDlgItemText(hwnd, 4201, filespec);
					return (TRUE);
				}

                filename[i] = '\0';

                FormatChoice = (WORD) SendDlgItemMessage(hwnd, 4221, CB_GETCURSEL, 0, 0L);

                strcat(filename, FileExtensions[FormatChoice+1]);
                AnsiUpper(filename);

                if (OpenFile(filename, &of, OF_READ | OF_EXIST) != -1) {
                    switch (Confirmation (hwnd, "File %s already exists.\nOverwrite it?", filename)) {
                        case IDCANCEL: EndDialog(hwnd, FALSE); return(TRUE);
						case IDNO:     return (TRUE);
						case IDYES:    break;
					}
                } else if ((fd = OpenFile(filename, &of, OF_WRITE | OF_CREATE)) == -1) {
                    filename[i] = '\0';
                    if (OpenFile(filename, &of, OF_READ | OF_EXIST) != -1) {
                        switch (Confirmation (hwnd, "File %s already exists.\nOverwrite it?", filename)) {
                            case IDCANCEL: EndDialog(hwnd, FALSE); return(TRUE);
                            case IDNO:     return (TRUE);
                            case IDYES:    break;
                        }
                    } else if ((fd = OpenFile(filename, &of, OF_WRITE | OF_CREATE)) == -1) {
						MessageBeep(0);
						return (TRUE);
                    }
                }

                close(fd);

				OemToAnsi(filename, filename);

                OutputLineBreaks = IsDlgButtonChecked(hwnd, 4222);
				if (OutputLineBreaks) {
                    GetDlgItemText(hwnd, 4223, buffer, 3);
                    LineLength = atoi(buffer);
					if (LineLength <= 0) {
                        ErrorMessage(global.hwnd, "Bad Line Length: %s", buffer);
						sprintf(buffer, "%d", global.active->linelen);
                        SetDlgItemText(hwnd, 4223, buffer);
						return (TRUE);
					}
                    LineLength *= 2;
                }

                if (IsDlgButtonChecked(hwnd, 4224)) {
                    i = (WORD) SendMessage(GetDlgItem(hwnd, 4225), CB_GETCURSEL, 0, 0L);
                    if (i != LB_ERR) {
                        switch (i) {
                            case 0: global.saveundolevels = -1; break;
                            case 1: global.saveundolevels = 1; break;
                            case 2: global.saveundolevels = 5; break;
                            case 3: global.saveundolevels = 10; break;
                            case 4: global.saveundolevels = 50; break;
                            case 5: global.saveundolevels = 100; break;
                            case 6: global.saveundolevels = 500; break;
                            case 7: global.saveundolevels = 1000; break;
                            case 8: global.saveundolevels = MAXUNDOLEVELS; break;
                        }
                    }
                } else {
                    global.saveundolevels = 0;
                }


				EndDialog(hwnd, TRUE);
				return (TRUE);

            case IDCANCEL:
				EndDialog(hwnd, FALSE);
				return (TRUE);
		}
	}

	return (FALSE);
}
