/* Power Render Light Interface */

#ifndef _PRLIGHT_
#define _PRLIGHT_           /* Mark this header already included */

#ifdef __cplusplus
 extern "C" {
#endif

#define NORMAL_CONST 1024               /* Accuracy of normals */

#define DIRECTIONAL_LIGHT 0
#define POINT_LIGHT 1

extern PR_LIGHTLIST scenelights;
extern PR_DWORD PR_AmbientLight;

extern PR_REAL PR_AmbientRed;
extern PR_REAL PR_AmbientGreen;
extern PR_REAL PR_AmbientBlue;


void PR_AddLightsToScene (PR_LIGHTLIST *userlights);
/* -----------------------------------------------------------------------
   Function:   PR_AddLightsToScene
   Purpose:    Adds a list of lights to the ones in the scene 
   Parameters: userlights - Pointer to the light list
   Return:     None
   ----------------------------------------------------------------------- */


void PR_AllocLights (PR_LIGHTLIST *lights, PR_DWORD numlights);
/* -----------------------------------------------------------------------
   Function:   PR_AllocLights
   Purpose:    Allocate space for the given number of lights. 
   Parameters: lights - Pointer to the light list
               numlights - Number of lights to allocate
   Return:     None
   ----------------------------------------------------------------------- */


void PR_InitializeLights (PR_LIGHTLIST *lights);
/* -----------------------------------------------------------------------
   Function:   PR_InitializeLights
   Purpose:    Sets every light in a list to some default values
   Parameters: lights - Pointer to the light list
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PR_CalcFaceLighting (PR_FACE *face);
/* -----------------------------------------------------------------------
   Function:   PR_CalcFaceLighting
   Purpose:    Calculates the lighting value for a given face 
   Parameters: face - Pointer to the face
   Return:     Amount of light hitting the face (0-1023)
   ----------------------------------------------------------------------- */


void PR_CalcNormal (PR_VERTEX *wp1, PR_VERTEX *wp2, PR_VERTEX *wp3,
                    PR_WORD *nx,  PR_WORD *ny, PR_WORD *nz);
/* -----------------------------------------------------------------------
   Function:   PR_CalcNormal
   Purpose:    Calculates a normal vector given three vertices 
   Parameters: wp1 - Pointer to the first vertex
               wp2 - Pointer to the second vertex
               wp3 - Pointer to the third vertex
               nx,ny,nz - Pointer to the resulting normal components
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PR_CalcVertexLighting (PR_VERTEX *vert, PR_VERTEX_DATA *vertdata);
/* -----------------------------------------------------------------------
   Function:   PR_CalcVertexLighting
   Purpose:    Calculates the lighting value for a given vertex 
   Parameters: vert - Pointer to the vertex
               vertdata - Pointer to the vertex data
   Return:     Amount of light hitting the vertex (0-1023)
   ----------------------------------------------------------------------- */


PR_DWORD PR_DotProduct (PR_DWORD x1, PR_DWORD y1, PR_DWORD z1, PR_DWORD x2,
                        PR_DWORD y2, PR_DWORD z2);
/* -----------------------------------------------------------------------
   Function:   PR_DotProduct
   Purpose:    Compute the dot product between two 3D vectors 
   Parameters: x1,y1,z1 - First vector
               x2,y2,z2 - Second vector
   Return:     Dot product of two vectors, in fixed point
   ----------------------------------------------------------------------- */


PR_REAL PR_DotProductReal (PR_REAL x1, PR_REAL y1, PR_REAL z1,
                            PR_REAL x2, PR_REAL y2, PR_REAL z2);
/* -----------------------------------------------------------------------
   Function:   PR_DotProductReal
   Purpose:    Compute the dot product between two 3D vectors 
   Parameters: x1,y1,z1 - First vector
               x2,y2,z2 - Second vector
   Return:     Dot product of two vectors, in floating point
   ----------------------------------------------------------------------- */


PR_REAL PR_GetAmbientLight (void);
/* -----------------------------------------------------------------------
   Function:   PR_GetAmbientLight
   Purpose:    Returns the amount of ambient light
   Parameters: None
   Return:     Amount of ambient light in floating point
   ----------------------------------------------------------------------- */


void PR_GetLightColor (PR_LIGHTLIST *lights, PR_DWORD light,
                       PR_REAL *r, PR_REAL *g, PR_REAL *b);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightColor
   Purpose:    Get the color of a light 
   Parameters: lights - Pointer to the light list
               light - light number
               r,g,b - Pointer to the RGB components
   Return:     None
   ----------------------------------------------------------------------- */


PR_REAL PR_GetLightFalloff (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightFalloff
   Purpose:    Get the falloff value for a light 
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     Falloff value for point lights
   ----------------------------------------------------------------------- */


void PR_GetLightPosition (PR_LIGHTLIST *lights, PR_DWORD light, 
                          PR_REAL *x, PR_REAL *y, PR_REAL *z);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightPosition
   Purpose:    Get the position of a light 
   Parameters: lights - Pointer to the light list
               light - light number
               x,y,z - Pointer to the world coordinates
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PR_GetLightState (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightState
   Purpose:    Returns the status of the given light (1 = on, 0 = off) 
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     0 or 1
   ----------------------------------------------------------------------- */


PR_REAL PR_GetLightStrength (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightStrength
   Purpose:    Gets the strength value for a light 
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     Strength value for the light
   ----------------------------------------------------------------------- */


PR_DWORD PR_GetLightType (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_GetLightType
   Purpose:    Returns the type of the light
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     DIRECTIONAL_LIGHT or POINT_LIGHT
   ----------------------------------------------------------------------- */


void PR_SetAmbientLight (PR_REAL light);
/* -----------------------------------------------------------------------
   Function:   PR_SetAmbientLight
   Purpose:    Sets the amount of ambient light 
   Parameters: light - Amount of ambient light
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightColor (PR_LIGHTLIST *lights, PR_DWORD light, 
                       PR_REAL r, PR_REAL g, PR_REAL b);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightColor
   Purpose:    Set the color of a light
   Parameters: lights - Pointer to the light list
               light - light number
               r,b,g - RGB components
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightFalloff (PR_LIGHTLIST *lights, PR_DWORD light, PR_REAL falloff);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightFalloff
   Purpose:    Set the falloff value for a light 
   Parameters: lights - Pointer to the light list
               light - light number
               falloff - Falloff value for point lights
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightOff (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightOff
   Purpose:    Turns the specified light off 
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightOn (PR_LIGHTLIST *lights, PR_DWORD light);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightOn
   Purpose:    Turns the specified light on 
   Parameters: lights - Pointer to the light list
               light - light number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightPosition (PR_LIGHTLIST *lights, PR_DWORD light, 
                PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightPosition
   Purpose:    Set the position of a light
   Parameters: lights - Pointer to the light list
               light - light number
               x,y,z - Location of light
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetLightStrength (PR_LIGHTLIST *lights, PR_DWORD light, PR_REAL strength);
/* -----------------------------------------------------------------------
   Function:
   Purpose:    Set the strength value for a light
   Parameters: lights - Pointer to the light list
               light - light number
               strength - Strength of light
   Return:
   ----------------------------------------------------------------------- */


void PR_SetLightType (PR_LIGHTLIST *lights, PR_DWORD light, PR_DWORD type);
/* -----------------------------------------------------------------------
   Function:   PR_SetLightType
   Purpose:    Sets the type of the light 
   Parameters: lights - Pointer to the light list
               light - light number
               type -  Either POINT_LIGHT or DIRECTIONAL_LIGHT
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetSegmentLighting (PR_ENTITY *entity, PR_SEGMENT *seg,
                            PR_ORIENTATION *seg_orient);
/* -----------------------------------------------------------------------
   Function:   PR_SetSegmentLighting
   Purpose:    Computes the normal of the light with respect to a segment 
   Parameters: entity - Pointer to the entity containing the segment
               seg -    Pointer to the segment
               seg_orient - Pointer to the segment orientation info
   Return:     None
   ----------------------------------------------------------------------- */


void PR_TransformLights (PR_LIGHTLIST *lights);
/* -----------------------------------------------------------------------
   Function:   PR_TransformLights
   Purpose:    Transforms the lights to the viewport and adds them to the
               scene
   Parameters: lights - Pointer to the light list
   Return:     None
   ----------------------------------------------------------------------- */

#ifdef __cplusplus
}
#endif

#endif



