/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * CTIME.C                                                                 *
 *                                                                         *
 * Jonathan Senning                               September, November 1987 *
 *                                                                         *
 * This is a collection of utility routines used to access the battery     *
 * operated clock on the AT&T 6300 PC.  They use routines and structures   *
 * which are unique to TURBO C.                                            *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <stdio.h>
#include <dos.h>
#include "ctime.h"

#define BASE_YEAR 1984

static char read_port(int);	/* local function prototype */


/*--------------------------------------------------------------------------*
 * function reset_clock()                                                   *
 * This function resets the AT&T 6300 battery operated clock.  This is      *
 * accoplished by taking the clock out of test mode and then stopping and   *
 * restarting the clock.                                                    *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void reset_clock()
{
    outportb(0x70, (char) 0);     /* take clock out of test mode */
    outportb(0x7E, (char) 0);     /* turn clock off */
    outportb(0x7E, (char) 0xFF);  /* restart clock */
}


/*--------------------------------------------------------------------------*
 * function getcdate()                                                      *
 * This function gets the date from the real-time clock on an AT&T 6300.    *
 * The date is returned in the date structure defined in dos.h              *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void getcdate(dblk)
struct date *dblk;
{
    dblk->da_year = (int) (read_port(0x7F) & 0x07) + BASE_YEAR;
    dblk->da_day  = (char) 10 * read_port(0x79) + read_port(0x78);
    dblk->da_mon  = (char) 10 * read_port(0x7C) + read_port(0x7B);
}


/*--------------------------------------------------------------------------*
 * function setcdate()                                                      *
 * This function sets the date for the real-time clock on an AT&T 6300.     *
 * The date is in the date structure defined in dos.h                       *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void setcdate(dblk)
struct date *dblk;
{
    int i;

    outportb(0x70, (char) 0);     /* take clock out of test mode */
    outportb(0x7E, (char) 0);     /* turn clock off */

    outportb(0x79, (char) (dblk->da_day / 10) & 0x0F);
    outportb(0x78, (char) (dblk->da_day % 10) & 0x0F);
    outportb(0x7C, (char) (dblk->da_mon / 10) & 0x0F);
    outportb(0x7B, (char) (dblk->da_mon % 10) & 0x0F);
    outportb(0x7F, (char) ((dblk->da_year % 8) & 0x0F) | 0x08);
    for (i = 0; i < 3; i++) {
        read_port(0x7F);
    }

    outportb(0x7E, (char) 0xFF);  /* restart clock */
}


/*--------------------------------------------------------------------------*
 * function showdate()                                                      *
 * This function displays the date in standard American format.  The date   *
 * is in the date structure defined in dos.h                                *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void showdate(dblk)
struct date *dblk;
{
    printf("%02d-%02d-%4d\n", dblk->da_mon, dblk->da_day, dblk->da_year);
}


/*--------------------------------------------------------------------------*
 * function getctime()                                                      *
 * This function gets the time from the real-time clock on an AT&T 6300.    *
 * The time is returned in the time structure defined in dos.h              *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void getctime(tblk)
struct time *tblk;
{
    tblk->ti_min  = (unsigned char) (10 * read_port(0x75) + read_port(0x74));
    tblk->ti_hour = (unsigned char) (10 * read_port(0x77) + read_port(0x76));
    tblk->ti_hund = (unsigned char) (10 * read_port(0x71));
    tblk->ti_sec  = (unsigned char) (10 * read_port(0x73) + read_port(0x72));
}


/*--------------------------------------------------------------------------*
 * function setctime()                                                      *
 * This function sets the time for the real-time clock on an AT&T 6300.     *
 * The time is in the time structure defined in dos.h                       *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void setctime(tblk)
struct time *tblk;
{
    int i;

    outportb(0x70, (char) 0);     /* take clock out of test mode */
    outportb(0x7E, (char) 0);     /* turn clock off */

    outportb(0x77, (char) (tblk->ti_hour / 10) & 0x0F);
    outportb(0x76, (char) (tblk->ti_hour % 10) & 0x0F);
    outportb(0x75, (char) (tblk->ti_min / 10) & 0x0F);
    outportb(0x74, (char) (tblk->ti_min % 10) & 0x0F);

    outportb(0x7F, (char) ((read_port(0x7F) & 0x0F) | 0x08));
    for (i = 0; i < 3; i++) {
        read_port(0x7F);
    }

    outportb(0x7E, (char) 0xFF);  /* restart clock */
}


/*--------------------------------------------------------------------------*
 * function showtime()                                                      *
 * This function displays the time in standard format.  The time is in the  *
 * time structure defined in dos.h                                          *
 * UNIQUE TO TURBO C                                                        *
 *--------------------------------------------------------------------------*/
void showtime(tblk)
struct time *tblk;
{
    printf("%02d:%02d:%02d.%02d\n",
            tblk->ti_hour, tblk->ti_min, tblk->ti_sec, tblk->ti_hund);
}


/*--------------------------------------------------------------------------*
 * function read_port()                                                     *
 * This function reads one of the AT&T 6300 real-time clock ports.  It con- *
 * tinues to read the port until it gets a valid return.                    *
 *--------------------------------------------------------------------------*/
static char read_port(port_num)
int port_num;
{
    char contents;

    while ((contents = (inportb(port_num) & 0x0F)) == 0x0F);
    return(contents);
}
