/*==========================================================================
 *
 *  !BEST004.C                                        Sunday, April 17, 1994
 *
 *  .INI file routines
 *  supplementary source file 4 for The BESTLibrary
 *
 *  Authored independently by George Vanous
 *
 *==========================================================================*/


/* ------------------------------------------------------------------------ */
/* ----------------------------  INCLUDE FILES  --------------------------- */

#include <dir.h>
#include <dos.h>
#include <alloc.h>
#include <stdlib.h>
#include "!bestlib.h"                  // include !BESTLIB.H in compilation

/* ------------------------------------------------------------------------ */
/* ------------------------------  CONSTANTS  ----------------------------- */

/* used in procedure "ini_get_error" */
#define BOOLEAN  0
#define COLOR    1
#define NUMBER   2
#define POSITION 3
#define ROWS     4
#define STATE    5
#define STRING   6

/* ------------------------------------------------------------------------ */
/* ---------------------------  ERROR MESSAGES  --------------------------- */

#define ERR03 "\nI cannot open the initialization file -- I need this file to continue.\n\n"
#define ERR04 "\nIt seems the initialization file is incorrect\n\nI do not know this color:  %s\n\nThis color is located here:\n%s\n\nThe colors I know are:  BLACK  BLUE  GREEN  CYAN  RED  MAGENTA  BROWN  LIGHTGREYDARKGREY  LIGHTBLUE  LIGHTGREEN  LIGHTCYAN  LIGHTRED  LIGHTMAGENT  YELLOW  WHITE\n"
#define ERR05 "\nI believe the initialization file is incorrect\n\nThere is an invalid text string located here:\n%s\n\nThe offensive string is:  %s\n\nA valid string must be enclosed within quotation marks:  \"This string is OK!\"\n\nPlease remedy this problem.\n\n"
#define ERR06 "\nI have found something incorrect in the initialization file\n\nThere is a section header missing from section:  %s\n\nI can not find this header:  %s\n\nPlease type it in.\n\n"
#define ERR07 "\nThere is a problem with the initialization file\n\nI have located an open section\n\nI can not find this section terminator:  %s\n\nPlease input it.\n\n"
#define ERR08 "\nI think something needs to be added to the initialization file\n\nI am missing an important option from this section:  %s\n\nCan you please insert this option:  %s\n\n"
#define ERR09 "\nI cannot seem to understand one part of the initialization file\n\nI think an equality symbol, =, is missing from this section:\n%s\n\nI can not locate the = symbol for this option:  %s\n\nCan you please input it for me?\n\n"
#define ERR10 "\nMaybe you can help me understand the initialization file\n\nThe problem lies in this section:\n%s\n\nThe parameter I am having trouble with is:  %s\n\nI think it can only be TRUE or FALSE -- can you please correct this problem?\n\n"
#define ERR11 "\nThere seems to be an invalid number in the initialization file\n\nThe offensive number lies in this section:\n%s\n\nThe number I do not recognize is:  %s\n\nPlease try to correct this problem.\n\n"
#define ERR12 "\nI am having difficulty understanding the initialization file\n\nMy question comes from this section:\n%s\n\nI do not know what is meant by this parameter:  %s\n\nI expected it to be either ON, OFF, or UNMODIFY -- can you help me?\n\n"
#define ERR13 "\nThe initialization file must be in error\n\nI do not understand this screen location: %s\n\nIt is located in this section:\n%s\n\nI only understand these two screen positions:  TOP  BOTTOM\n\nPlease correct this problem and try again.\n\n"
#define ERR14 "\nI think there is something that needs correcting in the initialization file\n\nI can only display 25 or 50 rows per screen, but in the section:\n%s\n\nI am asked to display %s rows.\n\nPlease input one of these values:  25, 50, or UNMODIFY.  Thank you.\n\n"
#define ERR15 "\nI cannot find the file %s -- it was requested in the initialization file\n\nThis is the section that asked for the file:\n%s\n\nEither this file is not in the current directory or it is misspelled.\n\n"

/* ------------------------------------------------------------------------ */
/* -------------------------  FUNCTION PROTOTYPES  ------------------------ */

void ini_get_error(char *section, char *str, word *ptr, byte type);
char *ini_verify_format(char *section, char *section_title,
                        char *option, word *ptr);

/* ------------------------------------------------------------------------ */


/*----------------------------------------------------------------------------
 * Translate the boolean word "TRUE" or "FALSE" into boolean TRUE or FALSE.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = TRUE if found string "TRUE"
 * = FALSE if found string "FALSE"
 */
boolean ini_get_boolean(char *section, char *section_title, char *option)
{
  shortint value;                      // return value
  char *str;                           // parameter
  word  ptr[2];                        // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  if ((value = con_str_to_bool(str)) == -1)
    ini_get_error(section, str, ptr, BOOLEAN);

  return( (boolean)value );
}

/*----------------------------------------------------------------------------
 * Translate a color name into its numerical equivalent.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = numerical equivalent of color name
 */
byte ini_get_color(char *section, char *section_title, char *option)
{
  shortint value;                      // return value
  char *str;                           // parameter
  word  ptr[2];                        // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  if ((value = con_str_to_color(str)) == -1)
    ini_get_error(section, str, ptr, COLOR);

  return( (byte)value );
}

/*----------------------------------------------------------------------------
 * Display the offending parameter in the initialization file and exit to DOS
 *   with the appropriate ERRORLEVEL.
 *
 * "section" - section text
 * "str"     - parameter in question
 * "ptr"     - two offset pointers
 * "type"    - type of error
 */
void ini_get_error(char *section, char *str, word *ptr, byte type)
{
  char error[1000];                    // ample space for error message

  // isolate the section containing the error
  *(section+ptr[0] + scan_byte(section+ptr[0], LF, NULL, FORWARD)-1) = NULL;

  // display the correct error message
  switch (type) {
    case BOOLEAN  : sprintf(error, ERR10, section+ptr[1], str            ); break;
    case COLOR    : sprintf(error, ERR04, str,            section+ptr[1] ); break;
    case NUMBER   : sprintf(error, ERR11, section+ptr[1], str            ); break;
    case POSITION : sprintf(error, ERR13, str,            section+ptr[1] ); break;
    case ROWS     : sprintf(error, ERR14, section+ptr[1], str            ); break;
    case STATE    : sprintf(error, ERR12, section+ptr[1], str            ); break;
    case STRING   : sprintf(error, ERR05, section+ptr[1], str            );
  }
  fprintf(stderr, error);
  beep(); exit(3);
}

/*----------------------------------------------------------------------------
 * Translate an ASCII representation of a number into its numerical equivalent.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = numerical representation of number string
 */
int ini_get_number(char *section, char *section_title, char *option)
{
  char *str;                           // parameter
  word  ptr[2];                        // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  if (!is_number_dec(str))
    ini_get_error(section, str, ptr, NUMBER);

  return( atoi(str) );
}

/*----------------------------------------------------------------------------
 * Translate the screen position "TOP" or "BOTTOM" to the screen row number
 *   (taking into account a possible 50-line text mode).
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = screen row number
 */
int ini_get_position(char *section, char *section_title, char *option)
{
  char *str;                           // parameter
  word  ptr[2];                        // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  str_case_up(str);
  if (!str_cmp(str, "TOP"))
    return( 0 );
  if (str_cmp(str, "BOTTOM"))
    ini_get_error(section, str, ptr, POSITION);

  return( txt_rows() );
}

/*----------------------------------------------------------------------------
 * Translate the number of rows per screen page "25", "50", or "UNMODIFY",
 *   into the numerical equivalent.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = rows per screen page
 */
int ini_get_rows(char *section, char *section_title, char *option)
{
  char *str;                           // parameter
  word num,                            // generic word
       ptr[2];                         // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  if ((num = atoi(str)) == 25 || num == 50)
    return( num-1 );

  str_case_up(str);
  if (str_cmp(str, "UNMODIFY"))
    ini_get_error(section, str, ptr, ROWS);

  return( txt_rows() );
}

/*----------------------------------------------------------------------------
 * Translate the keyboard state "ON", "OFF", or "UNMODIFY" into the numerical
 *   equivalent.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = 0 - OFF
 * = 1 - ON
 * = 2 - UNMODIFY
 */
byte ini_get_state(char *section, char *section_title, char *option)
{
  char *str;                           // parameter
  word ptr[2];                         // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  str_case_up(str);
  if (!str_cmp(str, "OFF"))
    return( 0 );
  if (!str_cmp(str, "ON"))
    return( 1 );
  if (str_cmp(str, "UNMODIFY"))
    ini_get_error(section, str, ptr, STATE);
  return( 2 );
}

/*----------------------------------------------------------------------------
 * Translate a text string surrounded by quotes into a text string not
 *   surrounded by quotes.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = pointer to quote-less string
 */
char *ini_get_text(char *section, char *section_title, char *option)
{
  char *str,                           // parameter
       *str2;                          // temporary string space
  word index,                          // generic character index
       ptr[2];                         // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  // verify the text is surrounded by quotes
  if (*str != '"' || str_right(str, 1) != '"')
    ini_get_error(section, str, ptr, STRING);

  // remove the quotes and keep just the text
  index = str_len(str)-2;              // remove two quotes
  str2 = malloc(index+1);              // allocate memory for text + NULL
  mem_copy(str2, str+1, index);        // make a copy of the text
  str2[index] = NULL;                  // terminate string
  return( str2 );
}

/*----------------------------------------------------------------------------
 * Translate the response "YES" or "NO" into the boolean TRUE or FALSE.
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 *
 * RETURNS:
 * = TRUE  - YES
 * = FALSE - NO
 */
boolean ini_get_yesno(char *section, char *section_title, char *option)
{
  char *str;                           // parameter
  word ptr[2];                         // two generic offset pointers

  str = ini_verify_format(section, section_title, option, ptr);

  str_case_up(str);
  if (!str_cmp(str, "YES"))
    return( TRUE );
  if (str_cmp(str, "NO"))
    ini_get_error(section, str, ptr, BOOLEAN);
  return( FALSE );
}

/*----------------------------------------------------------------------------
 * Read in the relevent information of a section.
 *
 * "str1" - string marking the beginning of the section
 * "str2" - string marking the ending of the section
 * "f"    - file handle of file to read from
 *
 * RETURNS:
 * = pointer to buffer containing relevant section information
 */
char *ini_read_section(char *str1, char *str2, FILE *f)
{
  char *section;     // relevent section information (can double as error msg)
  word  size = 1000;                   // default size of text input buffer

  if (!fil_skip_to(str1, f)) {
    section = (char *) malloc(1000);   // ample error message space
    sprintf(section, ERR06, str1, str1);
    fprintf(stderr, section);
    free(section);
    beep(); exit(4);
  }

  if ((section = fil_read_to_strip(str2, &size, f)) == NULL) {
    section = (char *) malloc(1000);   // ample error message space
    sprintf(section, ERR07, str2);
    fprintf(stderr, section);
    free(section);
    beep(); exit(5);
  }

  return( section );                   // return relevent section information
}

/*----------------------------------------------------------------------------
 * Verify format of the initialization file.
 * Checks if the - section is present
 *               - option is present
 *               - an equals '=' is present
 *
 * "section"       - section text
 * "section_title" - section title to read from
 * "option"        - option to read from
 * "ptr"           - two offset pointers
 *
 * RETURNS:
 * = pointer to next parameter
 */
char *ini_verify_format(char *section, char *section_title,
                        char *option, word *ptr)
{
  char *param;                         // parameter (can double as error msg)
  word  index1,                        // generic buffer index #1
        index2;                        // generic buffer index #2

  // verify the desired section is present
  if (!(ptr[0] = scan_str(section, section_title, NULL, FORWARD))) {
    section[scan_byte(section, LF, NULL, FORWARD)-1] = NULL;
    param = (char *) malloc(1000);     // ample error message space
    sprintf(param, ERR06, section, section_title);     // a section is missing
    fprintf(stderr, param);
    free(param);
    beep(); exit(4);
  }

  // get length of this section
  index1 = scan_byte(section+ptr[0], '[', NULL, FORWARD);

  // skip past section header
  index2 = scan_byte(section+ptr[0], LF, index1, FORWARD);
  ptr[1] = ptr[0]-1;                   // keep an index to section beginning
  ptr[0] += index2;

  // verify the desired option is present
  if (!(index2 = scan_str(section+ptr[0], option, index1-index2, FORWARD))) {
    param = (char *) malloc(1000);     // ample error message space
    sprintf(param, ERR08, section_title, option);      // an option is missing
    fprintf(stderr, param);
    free(param);
    beep(); exit(6);
  }

  // get to the offset of the second letter of that option
  ptr[0] += index2;

  // search for an equal sign
  index2 = scan_byte(section+ptr[0], '=', NULL, FORWARD);

  // make sure the equal sign follows the option in question
  if (index2 != str_len(option)) {
    section[ptr[1]+index1] = NULL;
    param = (char *) malloc(1000);     // ample error message space
    sprintf(param, ERR09, section+ptr[1], option);     // '=' (equals) missing
    fprintf(stderr, param);
    free(param);
    beep(); exit(7);
  }

  // get to the offset to the parameter (just beyond the equal sign)
  ptr[0] += index2;
  param = str_next_str(section+ptr[0]);// read in next parameter
  return( param );                     // return next parameter
}

/*==============================  END-OF-FILE  =============================*/
