/*
   Module:  parse.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides the function needed to parse the lines from
            an ADI file, breaking the line into a command and its arguments.

   Functions Provided:

        parse

   Functions Required:

        none
*/



#include <stdlib.h>
#include "adicodes.h"
#include "retcodes.h"



/*
   Function: parse
   Purpose:  Parse a line from an ADI file, converting it to a command and
             its arguments.

   Pre: line is a pointer to a line from an ADI file.
        cmd is a pointer to storage for the ADI command.
        arg1 is a pointer to storage for the first argument to the command
        (if it has an argument)
        arg2 is a pointer to storage for the second argument to the command
        (if it has two arguments)

   Post: An attempt is made to decode line.
         The ADI command found in line is copied to cmd.
         If the command has an argument it is copied to arg1.
         If the command has two arguments, the second is copied to arg2.
         If an error occurs, NOCMD is copied to cmd.
         If arg1 and/or arg2 are not required for a command, their contents
         are undefined.
*/

#define BASE 10  /* base to use for string to num conversions */

void parse (line, cmd, arg1, arg2)
   char *line;
   int *cmd; 
   unsigned *arg1, *arg2;
{

   char *endptr;
   *cmd = *line++ - '0';

   switch(*cmd) {
      case END_PLOT:
      case PEN_CHANGE:
      case ABORT_PLOT:
         /* commands with no arg */
         if (*line != (char)NULL)
            *cmd=BADFMT;
         break;

      case BEGIN_PLOT:
      case NEW_PEN:
      case SET_SPEED:
      case LINE_TYPE:
         /* commands with 1 arg */
         if (*line++ != ',') {
            *cmd = BADFMT;
            break;
         }
         *arg1 = (unsigned) strtol (line, &endptr, BASE);
         if (*endptr != (char)NULL)
            *cmd = BADFMT;
         break;

      case MOVE:
      case DRAW:
         /* commands with 2 args */
         if (*line++ != ',') {
            *cmd = BADFMT;
            break;
         }
         *arg1 = (unsigned) strtol (line, &endptr, BASE);
         if (*endptr != ',') {
            *cmd = BADFMT;
            break;
         }
         line = endptr+1;
         *arg2 = (unsigned) strtol (line, &endptr, BASE);
         if (*endptr != (char)NULL)
            *cmd = BADFMT;
         break;

      default:
         *cmd = BADFMT;
         break;
   }
}

