/*
   Module:  hp7221.h
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides function prototypes for the functions 
            used to assemble the codes that are to be sent to the HP 7221 
            plotter.  Each function assembles a char array of codes that are 
            used to represent the desired plotter functions.  The resulting 
            arrays are then output via the putarr function that is provided 
            by the generic ADI module.

            DEV_WELCOME is also define here.  This defines arguements for a
            printf statement that will be called during program startup.

   Functions provided:

        dev_beginplot
        dev_endplot
        dev_move
        dev_draw
        dev_newpen      (automatic pen selection)
        dev_setspeed
        dev_linetype
        dev_penchange   (pen up for manual pen change)
        dev_abortplot

        Note: When a dev_penchange is called, the main driver is expected
        to pause while the operator manually changes the pen.  If this is
        not possible (ie the driver is not a direct feed to the plotter),
        dev_penchange should not be called.

*/

/* the welcome message for the main program */
#define DEV_WELCOME "ADI7221 1.0b.  University of Rochester, Psychology Dept.\n\n"


/*
   Function: dev_beginplot
   Purpose:  Produces the code used to initialize the plotter for a new
             plot.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.

   Post: The char array of codes necessary to initialize the plotter is 
         output via putarr().  The array contains the HP_BEGINPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_beginplot (PLTFILE *pltfp);



/*
   Function: dev_endplot
   Purpose:  Produces the code used to signal the plotter that the current
             plot is complete.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progess.

   Post: The char array of codes necessary to signal the end of a plot is 
         output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE, and the 
         array begins with the HP_END_DRAW code. The array always contains 
         the HP_ENDPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_endplot (PLTFILE *pltfp);



/*
   Function: dev_move
   Purpose:  Produces the code used to cause the plotter to perform a pen 
             up move.

   Pre: xcoord contains the X coordinate of the ending location of the move.
        ycoord contains the Y coordinate of the ending location of the move.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates if a draw is currently in
        progress.

   Post: The char array of codes necessary to perform a pen up move is 
         output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes: the HP_MOVE code, the MBP formatted 
         representation of xcoord and ycoord, and the HP_END_MOVE code.
         If xcoord or ycoord are out of range, BADFMT is returned.
         If putarr returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_move (unsigned xcoord, unsigned ycoord, PLTFILE *pltfp);



/*
   Function: dev_draw
   Purpose:  Produces the code used to cause the plotter to perform a pen
             down draw.

   Pre: xcoord contains the X coordinate of the ending location of the draw.
        ycoord contains the Y coordinate of the ending location of the draw.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether or not a draw is 
        currently in progress.

   Post: The char array of codes necessary to perform a pen down draw is 
         output via putarr().
         If the global variable 'drawing' is initially TRUE, the array
         contains only the MBP formatted representation of xcoord and ycoord.
         Otherwise, 'drawing' is set TRUE, and the array contains the
         HP_DRAW code, followed by the MBP formatted representation of
         xcoord and ycoord.
         If xcoord or ycoord are out of range, BADFMT is returned. 
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_draw (unsigned xcoord, unsigned ycoord, PLTFILE *pltfp);



/*
   Function: dev_newpen
   Purpose:  Produces the code used to cause the plotter to automatically
             select a new pen.

   Pre: pennum contains the number of the pen to be selected.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to perform an automatic pen 
         select is output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes the HP_NEWPEN code followed by the SBN
         formatted representation of pennum.
         If pennum is out of range, BADFMT is returned. 
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_newpen (unsigned pennum, PLTFILE *pltfp);



/*
   Function: dev_setspeed
   Purpose:  Produces the code used to cause the plotter to select a new
             pen speed.

   Pre: speed contains the code for the new speed.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to cause the plotter to select 
         a new speed is output via putarr().
         If speed is out of range, BADFMT is returned. 
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes the HP_SETSPEED code followed by the SBN
         fomatted representation of speed.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_setspeed (unsigned speed, PLTFILE *pltfp);



/*
   Function: dev_linetype
   Purpose:  Produces the code needed to cause the plotter to beging using a
             different line type for subesequent draw commands.

             Currently supported line types are:

                (0)  ----------------------------------- (solid)
                (1)  -- -- -- -- -- -- -- -- -- -- -- -- (dashed)
                (2)  - - - - - - - - - - - - - - - - - - (hidden)
                (3)  ---- - ---- - ---- - ---- - ---- -  (center)
                (4)  ----- - - ----- - - ----- - - ----- (phantom)
                (5)  ................................... (dot)
                (6)  -- . -- . -- . -- . -- . -- . -- .  (dashdot)
                (7)  -- -- . -- -- . -- -- . -- -- . --  (border)
                (8)  -- .. -- .. -- .. -- .. -- .. -- .. (divide)
        
   Pre: linecode contains the code for the desired line type.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to cause the plotter to begin 
         using a different line type is output via putarr().
         If linecode is out of range, BADFMT is returned. 
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.
         The array always contains the HP_LINETYPE code followed by the
         SBN and MBN representations of the code required to set the desired 
         line type.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.

         If the module is compiled with VARIABLE_DASHES defined, patterns
         in the lines will be varied so that an integral number of patterns
         will be included in each line segment.  If VARIABLE_DASHES is not
         defined, fixed length patterns will be used without regard for
         the completeness of each pattern. (ie short line segments may not
         include a complete pattern)
*/

int dev_linetype (unsigned linecode, PLTFILE *pltfp);



/*
   Function: dev_penchange
   Purpose:  Produces the code needed to raise the pen so that the main
             program can pause and allow the operator to manually change
             pens.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes needed to cause the plotter to raise the 
         pen is output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always contains the HP_PENCHANGE code.
         If putarr() returns an error, BADIO is returned.
         Otherwise, TRUE is returned.
*/

int dev_penchange (PLTFILE *pltfp);



/*
   Function: dev_abortplot
   Purpose:  Produces the code needed to cause the plotter to abort all
             actions related to the current plot.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.

   Post: The char array of codes needed to cause the plotter to abort the 
         current plot is output via putarr().
         The array contains the HP_ABORTPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise, TRUE is returned.
*/

int dev_abortplot (PLTFILE *pltfp);
