#include "pch.h"
#include "bbcVideo.h"
#include "bbcComputer.h"
#include "bbcTeletextFont.h"
#include "bbcSystemVIA.h"
#ifdef bbcDEBUG_VIDEO
#include <stdio.h>
#endif
#ifdef bbcDEBUG_VIDEO
#include "bbcUserVIA.h"
#endif

//Pointless logging.
//#define bbcVIDEO_LOG_DLQ

//Whether to update registers BEFORE or AFTER the scanline...
//I was hoping this would fix Revs (mode change being 1 line out), but it didn't!
//To be investigated!
//#define bbcVIDEO_UPDATE_REGISTERS_BEFORE

//If defined, vsync is negative -- vsync start is -CA1, vsync end is +CA1
//Otherwise, +CA1 then -CA1.
//(Nothing seems to work properly if defined -- to be investigated!)
//#define VSYNC_NEGATIVE

//using t65::byte;

#ifdef bbcDEBUG_VIDEO
static void VlogDebug(const char *fmt,...) {
	va_list v;
	va_start(v,fmt);
	vfprintf(bbcVideo::log_frames_h_,fmt,v);
	va_end(v);
}
#define LOGFRAME(X)\
	if(bbcVideo::log_frames_h_) {\
		VlogDebug X;\
	}
#else
#define LOGFRAME(X)
#endif

std::vector<t65::byte> bbcVideo::buffer_vector;
t65::byte *bbcVideo::buffer=0;
int bbcVideo::buffer_width=640;
int bbcVideo::buffer_height=272;
int bbcVideo::buffer_size;//This is set up during Init.

#define END_OF_BUFFER (&buffer[bbcVideo::buffer_size])


int bbcVideo::frame_count_;
bool bbcVideo::buffer_dirty;
std::string bbcVideo::reg_names_[NUM_REGS];
t65::byte bbcVideo::cpu_current_[NUM_REGS];//used by CPU
t65::byte bbcVideo::video_current_[NUM_REGS];//used to render
int bbcVideo::last_flush_at_;
bbcVideo::DlEntry bbcVideo::dlq_[dlq_num_entries];
int bbcVideo::dlq_read_;
int bbcVideo::dlq_write_;
//void *bbcVideo::dest_;
t65::word bbcVideo::read_address_;
t65::qword bbcVideo::expand_1bit_slow_[16];
t65::dword bbcVideo::expand_1bit_fast_[16];
t65::byte bbcVideo::rearrange_2bit_[256];
t65::byte bbcVideo::rearrange_4bit_[256];

unsigned bbcVideo::reg_6845_select_;
int bbcVideo::cur_line_;
int bbcVideo::cur_subline_;
//int bbcVideo::vsync_left_;
int bbcVideo::cur_scanline_;
int bbcVideo::num_blank_scanlines_;
int bbcVideo::num_total_scanlines_;
t65::word bbcVideo::framebuffer_base_;
t65::word bbcVideo::wrap_size_;
bool bbcVideo::palette_dirty_slow_;
bool bbcVideo::palette_dirty_fast_;
bool bbcVideo::palette_dirty_3bit_;
bool bbcVideo::is_teletext_frame_;
t65::byte bbcVideo::palette_3bit_[16];
int bbcVideo::hblank_leftaligned_;
int bbcVideo::last_double_height_line_;
const t65::byte *bbcVideo::video_ram_;
#ifdef bbcDEBUG_VIDEO
bool bbcVideo::show_all_;
int bbcVideo::num_log_frames_left_;
FILE *bbcVideo::log_frames_h_;
bool bbcVideo::show_t1_timeouts_;
unsigned bbcVideo::show_all_offset_;
#endif
t65::byte bbcVideo::pixel_fixed_bits_;

//////////////////////////////////////////////////////////////////////////
// Bits in the ULA control register
enum {
	ULA_MASTERCURSORSIZE=128,
	ULA_CURSORWIDTH=64|32,
	ULA_6845FASTCLOCK=16,
	ULA_CHARSPERLINE=8|4,
	ULA_TELETEXT=2,
	ULA_FLASHCOLOUR=1,

	ULA_CURSORWIDTH_SHIFT=5,
	ULA_CHARSPERLINE_SHIFT=2,	
};

//////////////////////////////////////////////////////////////////////////
// Accessors for the register set
#define CRTC(N) (video_current_[CRTC_BASE+(N)])
#define PALETTE(N) (video_current_[ULA_PALETTE+(N)])
#define ULACTRL() (video_current_[ULA_CTRL])

//////////////////////////////////////////////////////////////////////////
// Definitions for read/write/bitness of each CRTC register
#define MASK(N) ((1<<(N))-1)
#define RW(N) bbcVideo::WRITABLE|bbcVideo::READABLE|MASK(N)
#define WO(N) bbcVideo::WRITABLE|MASK(N)

const int bbcVideo::reg_6845_flags_[18]={
	WO(8),//R0
	WO(8),//R1
	WO(8),//R2
	WO(8),//R3
	WO(7),//R4
	WO(5),//R5
	WO(7),//R6
	WO(7),//R7
	WO(8),//R8
	WO(5),//R9
	WO(7),//R10
	WO(5),//R11
	RW(6),//R12
	RW(8),//R13
	RW(6),//R14
	RW(8),//R15
	WO(6),//R16
	WO(8),//R17
};

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////
// bbcVideo::Init
//
// Initialises everything.
void bbcVideo::Init() {
	dlq_read_=0;
	dlq_write_=0;
	framebuffer_base_=0;
//	vsync_left_=-1;
	wrap_size_=0x5000;//arbitrary
	palette_dirty_slow_=true;
	palette_dirty_fast_=true;
	palette_dirty_3bit_=true;
	frame_count_=0;

	buffer_size=buffer_width*buffer_height;
	buffer_vector.clear();
	buffer_vector.resize(buffer_size);
	buffer=&buffer_vector[0];

	memset(buffer,0,buffer_size);
	buffer_dirty=true;
//	dest_=buffer_;
	read_address_=0x3000;

	last_flush_at_=bbcComputer::cycles;

	cur_line_=0;
	cur_subline_=0;
	cur_scanline_=0;
	num_blank_scanlines_=0;
	num_total_scanlines_=312;

	SetupTables();

	reg_6845_select_=0;
	framebuffer_base_=0;

	is_teletext_frame_=false;
	last_double_height_line_=-1;

	for(unsigned i=0;i<NUM_REGS;++i) {
		static char tmpbuf[100];
		std::string s;
		if(i==DUMMY) {
			s="dummy";
		}
		if(i==ULA_CTRL) {
			s+="ula ctrl";
		}
		if(i>=ULA_PALETTE&&i<ULA_PALETTE+16) {
			_snprintf(tmpbuf,sizeof tmpbuf,"ula pal #%d",i-ULA_PALETTE);
			s+=tmpbuf;
		}
		if(i>=CRTC_BASE&&i<CRTC_BASE+18) {
			_snprintf(tmpbuf,sizeof tmpbuf,"crtc r%d",i-CRTC_BASE);
			s+=tmpbuf;
		}
		reg_names_[i]=s;
	}

#ifdef bbcDEBUG_VIDEO
	show_all_=false;
	num_log_frames_left_=0;
	log_frames_h_=0;
	show_t1_timeouts_=false;
	show_all_offset_=0;
#endif

	pixel_fixed_bits_=0;
}

//////////////////////////////////////////////////////////////////////////
// Shutdown
//
// Frees any allocated memory.
void bbcVideo::Shutdown() {
	delete[] buffer;
	buffer=0;
}

//////////////////////////////////////////////////////////////////////////
// Sets the bits that will be fixed for any values written to the video
// buffer.
void bbcVideo::SetPixelFixedBits(t65::byte fixed_bits) {
	BASSERT((fixed_bits&7)==0);
	if(fixed_bits!=pixel_fixed_bits_) {
		pixel_fixed_bits_=fixed_bits;
		palette_dirty_3bit_=true;

		//Update the buffer right now. Otherwise, an attempt could be
		//made to present a bitmap that contains out of range colour
		//values -- a problem in window mode as all pixels are guaranteed
		//to be 0-7 inclusive only.
		for(int i=0;i<buffer_size;++i) {
			buffer[i]&=7;
			buffer[i]|=pixel_fixed_bits_;
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// SetupTables
//
// Sets up some tables to swizzle mode 1/2/5/8 data about.
void bbcVideo::SetupTables() {
	//Swizzles memory bits so that with >> and & can get ULA palette indices.
	//ABABABAB ends up as AAAABBBB
	static const t65::byte tbl4bit[8]={0x80,0x08,0x40,0x04,0x20,0x02,0x10,0x01,};
	//ABCDABCD ends up as ABABCDCD
	static const t65::byte tbl2bit[8]={0x80,0x40,0x08,0x04,0x20,0x10,0x02,0x01,};
	unsigned i;
	for(i=0;i<256;++i) {
		t65::byte *b2=&rearrange_2bit_[i];
		t65::byte *b4=&rearrange_4bit_[i];

		*b2=*b4=0;
		for(unsigned j=0;j<8;++j) {
			bool set=!!(i&(1<<(7-j)));
			if(set) {
				*b2|=tbl2bit[j];
				*b4|=tbl4bit[j];
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// Rebuild(Fast|Slow)ClockPaletteTables
//
// Rebuilds palette tables for decoding modes 0/3/4/6
//
// Each table is just for half a byte to increase coherency of memory accesses.

#define RPT_FAST_DO(N) \
	do {\
		if(i&(1<<(N))) {\
			expand_1bit_fast_[i]|=palette_3bit_[8]<<((3-(N))*8);\
		} else {\
			expand_1bit_fast_[i]|=palette_3bit_[0]<<((3-(N))*8);\
		}\
	} while(false)

//TODO I suspect some of these qword casts aren't needed!!
#define RPT_SLOW_DO(N) \
	do {\
		if(i&(1<<(N))) {\
			expand_1bit_slow_[i]|=t65::qword((palette_3bit_[8]<<8)|palette_3bit_[8])<<((t65::qword(3)-(N))*t65::qword(16));\
		} else {\
			expand_1bit_slow_[i]|=t65::qword((palette_3bit_[0]<<8)|palette_3bit_[0])<<((t65::qword(3)-(N))*t65::qword(16));\
		}\
	} while(false)

void bbcVideo::RebuildFastClockPaletteTables() {
	unsigned i;
#ifdef bbcDEBUG_VIDEO
	if(show_all_) {
		for(i=0;i<16;++i) {
			t65::qword v;
			t65::qword bits=0x07;
			for(unsigned j=0;j<8;++j) {
				if(i&(1<<j)) {
					v|=bits<<t65::qword((7-j)*8);
				}
			}
			expand_1bit_fast_[i]=v;
		}
	} else
#endif
	{
		for(i=0;i<16;++i) {
			expand_1bit_fast_[i]=0;
			RPT_FAST_DO(3);
			RPT_FAST_DO(2);
			RPT_FAST_DO(1);
			RPT_FAST_DO(0);
		}
	}
	palette_dirty_fast_=false;
}

void bbcVideo::RebuildSlowClockPaletteTables() {
	unsigned i;
#ifdef bbcDEBUG_VIDEO
	if(show_all_) {
		for(i=0;i<16;++i) {
			t65::qword v=0;
			t65::qword bits=0x0707;
			for(unsigned j=0;j<4;++j) {
				if(i&(1<<j)) {
					v|=bits<<t65::qword((3-j)*16);
				}
			}
			BASSERT(i==0||v);
			expand_1bit_slow_[i]=v;
		}
	} else
#endif
	{
		for(i=0;i<16;++i) {
			expand_1bit_slow_[i]=0;
			RPT_SLOW_DO(3);
			RPT_SLOW_DO(2);
			RPT_SLOW_DO(1);
			RPT_SLOW_DO(0);
		}
	}
	palette_dirty_slow_=false;
}

//////////////////////////////////////////////////////////////////////////
// Reset
//
// Resets the video system. Fairly arbitrarily, it starts in mode 0.
//
void bbcVideo::Reset() {
	//See aug p462 -- this is mode 0
	CRTC(0)=0x7f;
	CRTC(1)=0x50;
	CRTC(2)=0x62;
	CRTC(3)=0x08|(0x02<<4);
	CRTC(4)=0x26;
	CRTC(5)=0x00;
	CRTC(6)=0x20;
	CRTC(7)=0x22;
	CRTC(8)=0x01;
	CRTC(9)=0x07;
	CRTC(10)=0x67;
	CRTC(11)=0x08;
	t65::word start=0x3000;
	start>>=3;
	CRTC(12)=start>>8;
	CRTC(13)=start&0xFF;
	CRTC(14)=0;
	CRTC(15)=0;
	CRTC(16)=0;
	CRTC(17)=0;
#ifdef bbcVIDEO_READ_ADDRESS
	read_address_=0x3000;
#endif
	for(unsigned i=0;i<16;++i) {
		video_current_[ULA_PALETTE+i]=i&8?7:0;
	}

	video_current_[ULA_CTRL]=0x9C;

	hblank_leftaligned_=98;

//	CopyCurrentToStarting();
	memcpy(cpu_current_,video_current_,sizeof video_current_);
}

//////////////////////////////////////////////////////////////////////////
// Write6845
//
// Writes a value to the currently select 6845 reg
//
void bbcVideo::Write6845(t65::byte offset,t65::byte val) {
	//	twprintf("bbcVideo::Write6845: offset=%d val=%d cycles=%d\n",offset,val,cycles);
	if(!(offset&1)) {
		reg_6845_select_=val&31;
	} else if(reg_6845_select_<18&&(reg_6845_flags_[reg_6845_select_]&WRITABLE)) {
		Write(CRTC_BASE+reg_6845_select_,
			val&reg_6845_flags_[reg_6845_select_]&0xFF);
	}
}

//////////////////////////////////////////////////////////////////////////
// WriteUla
//
// Writes a value to the ULA control register
//
void bbcVideo::WriteUla(t65::byte,t65::byte val) {
	Write(ULA_CTRL,val);
}

//////////////////////////////////////////////////////////////////////////
// WritePalette
//
// Writes a value to the palette register
//
void bbcVideo::WritePalette(t65::byte,t65::byte val) {
	Write(ULA_PALETTE+(val>>4),(val&0xf)^7);
}

//////////////////////////////////////////////////////////////////////////
// Read6845
//
// Reads a byte from the 6845
//
//TODO offset is ignored, it always returns the data register
t65::byte bbcVideo::Read6845(t65::byte offset) {
	if(reg_6845_select_<18&&(reg_6845_flags_[reg_6845_select_]&READABLE)) {
		return cpu_current_[CRTC_BASE+reg_6845_select_];
	}
	return 0;
}

//////////////////////////////////////////////////////////////////////////
// Write
//
// Writes a byte to a video system register. The reg is one of the bbcVideo::Register
// enumerants. The write is added to the display list queue, if there's room,
// or more display is produced if there's no room left.
//
void bbcVideo::Write(unsigned reg,t65::byte value) {
	int next_write=DlqNext(dlq_write_);
	if(next_write==dlq_read_) {
		Update(bbcComputer::cycles);//Does not modify dlq_write_
	}
	BASSERT(reg<NUM_REGS);
	DlEntry *ent=&dlq_[dlq_write_];
	ent->when=bbcComputer::cycles;
	ent->reg=reg;
	ent->val=value;
	cpu_current_[ent->reg]=value;
#ifdef bbcDEBUG_VIDEO
	if(log_frames_h_) {
		fprintf(log_frames_h_,"%d: Write: cycles=%d CPU wrote 0x%02X to %s, dlq index %d\n",
			bbcComputer::cycles,ent->when,ent->val,reg_names_[ent->reg].c_str(),dlq_write_);
	}
#endif
	dlq_write_=next_write;
}

//////////////////////////////////////////////////////////////////////////
// Debugging crap
static int last_vsync_cycles=0;
static int vsync_low_cycles=0;

//////////////////////////////////////////////////////////////////////////
// Subline divisors
//
// Depending on the setting of R8, there might be .5x as many scanlines on
// screen as the CRTC thinks there are, or something. Who knows. This table
// is indexed using the bottom 2 bits of R8, and sorts it all out.
//
static const int subline_divisors[4]={
	1,//Normal sync mode
	1,//Normal sync mode
	1,//Interlace sync mode
	2,//Interlace sync and video
};

//////////////////////////////////////////////////////////////////////////
// Macro hackery

//Width in PC pixels of a CRTC column 
#define CRTC_COLUMN_WIDTH() (ULACTRL()&ULA_TELETEXT?12:(ULACTRL()&ULA_6845FASTCLOCK?8:16))

//Current subline divisor
#define SUBLINE_DIVISOR() (subline_divisors[CRTC(8)&3])

//Scanlines in one character row
#define SCANLINES_PER_ROW() (CRTC(9)/SUBLINE_DIVISOR()+1)

//true if 6845 fast clock in effect
#define IS_FASTCLOCK() (!!(video_current_[ULA_CTRL]&ULA_6845FASTCLOCK))

//How far to bob the screen across
#define HORIZ_OFFSET() (std::_cpp_max(((hblank_leftaligned_-CRTC(2)*(IS_FASTCLOCK()?1:2))*CRTC_COLUMN_WIDTH()/2),0))

//////////////////////////////////////////////////////////////////////////
// DrawCursor
//
// Adds a cursor to the screen based on the current settings. This is added
// as a postprocess effect, and uses the current cursor settings in effect
// like RIGHT NOW. This is heinously incorrect, but looks fine.
//
void bbcVideo::DrawCursor() {
	//Bits 6 and 7 if both set disable cursor
	if((CRTC(8)&0xC0)==0xC0) {
		return;
	} else if(CRTC(1)==0) {
		//Screen is too small
		return;
	}

	//Cursor may be flashing or disabled
	switch((CRTC(10)>>5)&3) {
	case 0:
		//Non-blink
		break;
	case 1:
		//Non-display
		return;
	case 2:
		//1/16 field rate blink
		if(frame_count_&0x10) {
			return;
		}
		break;
	case 3:
		//1/32 field rate blink
		if(frame_count_&0x20) {
			return;
		}
		break;
	}
	
	//Rather dodgy calculation, should be OK for most stuff though.
	t65::word fb_base=framebuffer_base_;
	t65::Word cursor_addr;
	cursor_addr.l=CRTC(15);
	cursor_addr.h=CRTC(14);
	if(is_teletext_frame_) {
		cursor_addr.h^=0x20;
		cursor_addr.h+=0x74;
	} else {
		//cursor_addr.w<<=3;
		fb_base>>=3;
	}
	//Make it relative to screen start
	if(cursor_addr.w<fb_base) {
		cursor_addr.w-=fb_base;
		cursor_addr.w+=is_teletext_frame_?1024:wrap_size_>>3;
		return;
	} else {
		cursor_addr.w-=fb_base;
	}
	//Turn it back into character positions
	if(!is_teletext_frame_) {
//		cursor_addr.w>>=3;
	}
	int x=cursor_addr.w%CRTC(1)*CRTC_COLUMN_WIDTH()+HORIZ_OFFSET();
	int y=cursor_addr.w/CRTC(1)*SCANLINES_PER_ROW();
//	y+=last_start_scanline_;

	int start_line=(CRTC(10)&31)/SUBLINE_DIVISOR();
	int end_line=CRTC(11)/SUBLINE_DIVISOR();

	if(y+start_line>=0&&y+end_line<buffer_height) {
		//index using (ULACTRL()>>5)&3 -- see AUG p379
		static const int cursor_widths[4]={
			1,//0/3/4/6
			0,//undefined
			2,//1/5/7
			4,//2
		};
		int w=cursor_widths[(ULACTRL()&ULA_CURSORWIDTH)>>ULA_CURSORWIDTH_SHIFT];
		if(!(ULACTRL()&ULA_MASTERCURSORSIZE)) {
			w/=2;
		}
		w*=CRTC_COLUMN_WIDTH();
		if(w>0&&x>=0&&x<=buffer_width-w) {
//			for(int i=CRTC(10)&31;i<=CRTC(11);i+=SUBLINE_DIVISOR()) {
//				t65::byte *dest=&buffer_[x+(y+i/SUBLINE_DIVISOR())*buffer_width];
//				for(int j=0;j<w;++j) {
//					dest[j]^=7;
//				}
//			}
			for(int i=start_line;i<=end_line;++i) {
				t65::byte *dest=&buffer[x+(y+i)*buffer_width];
				for(int j=0;j<w;++j) {
					dest[j]^=7;
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// Update
//
// Does a video system update. 'cycles' is the current 2MHz cycle count.
// Renders more video, or doesn't, and does its thing.
//
void bbcVideo::Update(int cycles) {
	BASSERT(video_ram_);
	int ent=dlq_read_;
	int cycles_per_scanline;
	do {
		int c=last_flush_at_;

		//If less than one line to render, don't bother (yet)
		cycles_per_scanline=CRTC(0)+1;
		if(!(video_current_[ULA_CTRL]&ULA_6845FASTCLOCK)||(video_current_[ULA_CTRL]&ULA_TELETEXT)) {
			cycles_per_scanline*=2;
		}
		if(cycles-last_flush_at_<cycles_per_scanline) {
			break;
		}

		//Do this line
//		if(num_blank_scanlines_>0) {
//			--num_blank_scanlines_;
//			++cur_scanline_;
//			//and do nothing.
//		} else {

#ifdef VIDEO_UPDATE_REGISTERS_BEFORE
		ent=DlqRun(c+cycles_per_scanline,ent);
#endif
		LOGFRAME(("%-9d: %-3d: Render: ",bbcComputer::cycles,cur_scanline_));
#ifdef bbcDEBUG_VIDEO
		//Don't clip against ndisp if showall
		if(cur_scanline_>=0&&cur_scanline_<buffer_height&&
			(show_all_||cur_line_<CRTC(6)))
#else
		if(cur_scanline_>=0&&cur_scanline_<buffer_height&&cur_line_<CRTC(6))
#endif
		{
			//int pitch=cycles_per_scanline*5;
			//128 is the correct amount for one scanline
			//this value is therefore a 1:7 fixed point number
			//so (cycles_per_scanline*pitch)>>7
			int pitch=(cycles_per_scanline*bbcVideo::buffer_width)>>7;
			t65::byte *dest=&buffer[cur_scanline_*pitch];
			//adjust based on hblank position. doesn't work very well but it
			//makes mode 7 appear in the right place. [3/24/2003]
			//normal posn is 98 (taking into account clock speed)
			//There's nothing special about '98', it's just the value used by OS1.20
			//and so the screen is centred when using a standard mode.
			dest+=HORIZ_OFFSET();

			if(dest>=buffer&&dest+buffer_width<END_OF_BUFFER) {
				//BASSERT(dest+buffer_width<END_OF_BUFFER);//seems that was the bug then!
#ifdef bbcDEBUG_VIDEO
				//**** NOTE DANGLING ELSE ****
				int timer_min=cycles_per_scanline*2;
				if(show_t1_timeouts_&&
					(bbc_system_via.t1c_<timer_min||bbc_user_via.t1c_<timer_min))
				{
					int colour=0;
					if(bbc_system_via.t1c_<timer_min) {
						colour|=1;
						LOGFRAME(("[SysVIA-T1] "));
					}
					if(bbc_user_via.t1c_<timer_min) {
						colour|=2;
						LOGFRAME(("[UserVIA-T1]"));
					}
					memset(dest,colour,pitch);
				} else
#endif
#ifdef bbcDEBUG_VIDEO
				if(!show_all_&&num_blank_scanlines_>0)
#else
				if(num_blank_scanlines_>0)
#endif
				{
					//If R8 bits 4+5 both set, display is disabled.
#ifdef bbcDEBUG_VIDEO
					if(num_blank_scanlines_>0) {
						LOGFRAME(("Blank scanline (num_blank_scanlines=%d)",num_blank_scanlines_));
					} else {
						LOGFRAME(("Blank scanline (display disabled)"));
					}
#endif
					memset(dest,0,pitch);
				} else if(is_teletext_frame_) {
					t65::word src=framebuffer_base_+cur_line_*CRTC(1);
					if((CRTC(8)&0x30)==0x30) {
						LOGFRAME(("Blank scanline (display disabled)"));
						memset(dest,0,pitch);
					} else {
						LOGFRAME(("Teletext scanline @ 0x%04X",src));
						RenderTeletextScanline(dest,read_address_);
					}
				} else {
					//t65::word src=framebuffer_base_+cur_line_*CRTC(1)*8+cur_subline_;
					if(cur_subline_>=8) {
						LOGFRAME(("Blank scanline (subline %d)",cur_subline_));
						memset(dest,0,pitch);
					} else {
						if((CRTC(8)&0x30)==0x30) {
							LOGFRAME(("Blank scanline (display disabled)"));
							memset(dest,0,pitch);
						} else if(video_current_[ULA_CTRL]&ULA_6845FASTCLOCK) {
							LOGFRAME(("Fast 6845 scanline, ULA=&%02X (%d)",video_current_[ULA_CTRL],video_current_[ULA_CTRL]));
							RenderFastClockScanline(dest,read_address_+cur_subline_);
						} else {
							LOGFRAME(("Slow 6845 scanline, ULA=&%02X (%d)",video_current_[ULA_CTRL],video_current_[ULA_CTRL]));
							RenderSlowClockScanline(dest,read_address_+cur_subline_);
						}
					}
				}
			}
		}
#ifdef bbcDEBUG_VIDEO
		else {
			LOGFRAME(("Off screen, clipped"));
		}
#endif
		LOGFRAME(("\n           %-3d:",cur_scanline_));

		if(num_blank_scanlines_>0) {
			--num_blank_scanlines_;
		} else {
			cur_subline_+=SUBLINE_DIVISOR();
			if(cur_subline_>CRTC(9)) {
				cur_subline_=0;
				++cur_line_;
				if(is_teletext_frame_) {
					read_address_+=CRTC(1);
					if(read_address_&0x8000) {
						read_address_-=1024;
					}
				} else {
					read_address_+=CRTC(1)*8;
					if(read_address_&0x8000) {
						read_address_-=wrap_size_;
					}
				}
			}
		}

		//Update timing counts for that lot
		//TODO this was previously below the vsync bit and before the end-of-frame bit.
		c+=cycles_per_scanline;
		last_flush_at_=c;
		++cur_scanline_;
		
		//This won't be correct if R7==R4
		if(cur_line_==CRTC(7)) {
			if(cur_subline_==0) {
				DrawCursor();
				cur_scanline_=-32;

#ifdef VSYNC_NEGATIVE
				bbc_system_via.CA1(false);
#else
				bbc_system_via.CA1(true);
#endif
				LOGFRAME((" [Vsync+]"));
				vsync_low_cycles=cycles;
				buffer_dirty=true;
#ifdef bbcDEBUG_VIDEO
				if(log_frames_h_) {
					LOGFRAME((" [Frame %d done]",frame_count_));
					--num_log_frames_left_;
					if(num_log_frames_left_==0) {
						if(bbcComputer::trace) {
							bbcComputer::trace->WriteTrace(log_frames_h_,0,0);
							delete bbcComputer::trace;
							bbcComputer::trace=0;
						}
						fclose(log_frames_h_);
						log_frames_h_=0;
					}
				}
				if(show_all_&&(frame_count_%25==0)) {
					++show_all_offset_;
					palette_dirty_3bit_=true;
				}
#endif
				++frame_count_;
			} else {
				int vsynclen=(CRTC(3)>>4);//+1;
				int rowheight=CRTC(9);
				if(cur_subline_/SUBLINE_DIVISOR()==vsynclen||
					(vsynclen>rowheight&&cur_subline_==rowheight))
				{
#ifdef VSYNC_NEGATIVE
					bbc_system_via.CA1(true);
#else
					bbc_system_via.CA1(false);
#endif
					LOGFRAME((" [Vsync-]"));
					{
						//static char tmp[20];
						//printf("%d\n",cycles-last_vsync_cycles);
						last_vsync_cycles=cycles;
					}
				}
			}
		}

		//If past the bottom, go back to the top.
		if(cur_line_>CRTC(4)) {
			//Reset the scanline counter to 0 if the mode changes.
//			bool new_is_teletext_frame=!!(video_current_[ULA_CTRL]&ULA_TELETEXT);
//			if(new_is_teletext_frame!=is_teletext_frame_) {
//				cur_scanline_=0;
//			}
//			is_teletext_frame_=new_is_teletext_frame;
			LOGFRAME((" [line=%d, R4=%d]",cur_line_,CRTC(4)));
			is_teletext_frame_=!!(video_current_[ULA_CTRL]&ULA_TELETEXT);
			if(is_teletext_frame_) {
				t65::Word tmp;
				tmp.h=CRTC(12);
				tmp.h^=0x20;
				tmp.h+=0x74;
				tmp.l=CRTC(13);
				framebuffer_base_=tmp.w;

				//Default R2 setting in Mode 7 is 51. That's *2 for 102. 109-102 is 7, product
				//of which & column width is 84, which is not far shy of where the normal
				//Mode 7 screen (480x250) needs to be to be centred.
				hblank_leftaligned_=109;

				last_double_height_line_=-1;
			} else {
				framebuffer_base_=(CRTC(12)<<8|CRTC(13))<<3;
				hblank_leftaligned_=98;
			}
			cur_subline_=0;
			cur_line_=0;
			//In an interlaced PAL frame, there are 312.5 scanlines.
			//No interlace here, so there's just a timing fudge.
			if(CRTC(8)&1) {
				last_flush_at_+=cycles_per_scanline/2;
			}
			//There are extra blank scanlines according to R5.
			num_blank_scanlines_=CRTC(5);
			read_address_=framebuffer_base_;
		}
		LOGFRAME(("\n"));
#ifndef VIDEO_UPDATE_REGISTERS_BEFORE
		ent=DlqRun(last_flush_at_,ent);
#endif
	} while(cycles-last_flush_at_>=cycles_per_scanline);
	dlq_read_=ent;
//	CopyCurrentToStarting();
	bbcComputer::SetNextStop(last_flush_at_+cycles_per_scanline);
}

//////////////////////////////////////////////////////////////////////////
// DlqRun
//
// Runs the display list to the point where 'stop_cycles' cycles had executed,
// starting from entry 'ent'. Updates dirty flags as relevant.
//
int bbcVideo::DlqRun(int stop_cycles,int ent) {
	int startent=ent;
	//Copy and paste from non-rupture Update
	for(;ent!=dlq_write_;ent=DlqNext(ent)) {
		const DlEntry *dle=&dlq_[ent];
		if(stop_cycles<dle->when) {
			break;
		}
		video_current_[dle->reg]=dle->val;
		LOGFRAME(("%-9d: WWW: dle->when=%u 0x%02X (%d) written to %s\n",
			bbcComputer::cycles,dle->when,dle->val,dle->val,reg_names_[dle->reg].c_str()));
		if(dle->reg>=ULA_BEGIN&&dle->reg<ULA_END) {
//			palette_dirty_slow_=true;
//			palette_dirty_fast_=true;
			palette_dirty_3bit_=true;
		}
	}
	LOGFRAME(("         : WWW: queue ran, started at %d finished at %d\n",startent,ent));
	return ent;
}

//#define BITOF(M) ((bbcComputer::ram_[addr]&(M))?7:0)

//////////////////////////////////////////////////////////////////////////
// RenderTeletextScanline
//
// renders a teletext scanline.
//
// TODO I think double height is a bit wrong, see instructions screen for
// Frak!.
enum Mode7TextSettings {
	M7_GFX=1,
	M7_SEP=2,
	M7_GFXMASK=M7_GFX|M7_SEP,
	M7_DH=4,
};

void bbcVideo::RenderTeletextScanline(t65::byte *dest,t65::word src) {
	if(cur_subline_&1) {
		return;
	}
//	int scanline=CUR_SCANLINE();
//	byte *dest=DEST_ADDRESS();
	int subline=cur_subline_/SUBLINE_DIVISOR();
	if(subline>=10) {
		memset(dest,0,buffer_width);
	} else {
//		word src=framebuffer_base_+CRTC(1)*cur_line_;
		unsigned cpl=std::_cpp_min(t65::byte(buffer_width/12),CRTC(1));//cap at 53, that's 636 pixels
		t65::byte fg=7,bg=0;
		t65::byte new_fg=7,new_bg=0;
		bool hold_graphics=false;
		unsigned textsettings=0;
		static const t65::byte *textfonts[4]={
			bbcTeletextFont::normal_chars_src,//normal
			bbcTeletextFont::graphics_chars_src,//graphics
			bbcTeletextFont::normal_chars_src,//normal separated
			bbcTeletextFont::sep_graphics_chars_src,//graphics separated
		};
		//Last byte seen. Used for hold graphics mode.
		t65::byte last_b=0;
		bool was_double_height=false;
		bool was_bottom_half=false;
		for(unsigned i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=1024;
			}
			t65::byte b=video_ram_[src]&0x7f;//bbcComputer::ram_[src]&0x7F;
			switch(b) {
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
				//Text mode colour
				new_fg=b;
				textsettings&=~M7_GFX;
				break;
			case 140&0x7F:
				//Single height
				textsettings&=~M7_DH;
			case 141&0x7F:
				//Double height
				textsettings|=M7_DH;
				was_double_height=true;
				break;
			case 145&0x7F:
			case 146&0x7F:
			case 147&0x7F:
			case 148&0x7F:
			case 149&0x7F:
			case 150&0x7F:
			case 151&0x7F:
				//Graphics mode colour
				new_fg=b-(144&0x7f);
				textsettings|=M7_GFX;
				break;
			case 157&0x7f:
				//Set bg colour
				new_bg=fg;
				break;
			case 156&0x7f:
				//Reset bg colour
				new_bg=0;
				break;
			case 154&0x7f:
				//Separated graphics
				textsettings|=M7_SEP;
				break;
			case 153&0x7f:
				//Normal graphics
				textsettings&=~M7_SEP;
				break;
			case 158&0x7f:
				//Hold graphics
				hold_graphics=true;
				break;
			case 159&0x7f:
				//Release graphics
				hold_graphics=false;
				break;
			}
			const t65::byte *ch=&textfonts[textsettings&M7_GFXMASK][0];
			if(b>=32) {
				ch+=(b-32)*60;
			} else {
				if(hold_graphics) {
					if(last_b>=32) {
						ch+=(last_b-32)*60;
					}
				} else {
					fg=new_fg;
					bg=new_bg;
				}
			}
			last_b=b;
			if(!(textsettings&M7_DH)) {
				ch+=subline*6;
			} else {
				//Double height
				ch+=subline/2*6;
				if(last_double_height_line_>=0&&last_double_height_line_==cur_line_-1) {
					//Bottom half
					ch+=5*6;
					was_bottom_half=true;
				}
			}
			for(unsigned j=0;j<6;++j) {
				dest[j*2+1]=dest[j*2]=(ch[j]&fg)|(~ch[j]&bg)|pixel_fixed_bits_;
			}

			fg=new_fg;
			bg=new_bg;
			dest+=12;
			++src;
		}
		if(was_double_height&&!was_bottom_half) {
			last_double_height_line_=cur_line_;
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// Rebuild3bitPalette
//
// Rebuilds the current palette_3bit_ table. Indexed by ULA palette index,
// this contains a _3_ bit value which takes into account the current flash
// bit.
//
// Note to self: stop dicking about with this function! It works and
// you will only cock it up!
void bbcVideo::Rebuild3bitPalette() {
#ifdef bbcDEBUG_VIDEO
	if(show_all_) {
		palette_3bit_[0]=0;
		palette_3bit_[1]=1;
		palette_3bit_[2]=2;
		palette_3bit_[3]=3;
		palette_3bit_[4]=4;
		palette_3bit_[5]=5;
		palette_3bit_[6]=6;
		palette_3bit_[7]=7;
		palette_3bit_[8]=7;
		palette_3bit_[9]=6;
		palette_3bit_[10]=5;
		palette_3bit_[11]=4;
		palette_3bit_[12]=3;
		palette_3bit_[13]=2;
		palette_3bit_[14]=1;
		palette_3bit_[15]=0;
	} else
#endif
	{
		if(!(video_current_[ULA_CTRL]&1)) {
			for(unsigned i=0;i<16;++i) {
				palette_3bit_[i]=PALETTE(i)&7;
				palette_3bit_[i]|=pixel_fixed_bits_;
			}
		} else {
			for(unsigned i=0;i<16;++i) {
				palette_3bit_[i]=PALETTE(i);
				if(palette_3bit_[i]&8) {
					//Flashing colour
					palette_3bit_[i]^=15;//remove bit 3 invert bits 210
				}
				palette_3bit_[i]|=pixel_fixed_bits_;
			}
		}
	}
	palette_dirty_fast_=true;
	palette_dirty_slow_=true;
}

//////////////////////////////////////////////////////////////////////////
// Render(Fast|Slow)ClockScanline
//
// Renders a bitmap mode scanline for the appropriate 6845 clock speed.
void bbcVideo::RenderFastClockScanline(t65::byte *dest,t65::word src) {
	if(palette_dirty_3bit_) {
		Rebuild3bitPalette();
		palette_dirty_3bit_=false;
	}
	unsigned cpl=std::_cpp_min(CRTC(1),t65::byte(buffer_width/8));//cap at 80, that's 640 pixels
#ifdef bbcDEBUG_VIDEO
	if(show_all_) {
		cpl=buffer_width/8;
	}
#endif
	unsigned i;
	switch((video_current_[ULA_CTRL]&ULA_CHARSPERLINE)>>ULA_CHARSPERLINE_SHIFT) {
	case 0://10 (broken)
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
		}
		break;
	case 1://20 (mode 2)
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			t65::byte knobbled=rearrange_4bit_[video_ram_[src]];
			t65::byte first=palette_3bit_[knobbled>>4];
			t65::byte second=palette_3bit_[knobbled&15];
			dest[3]=dest[2]=dest[1]=dest[0]=first;
			dest[7]=dest[6]=dest[5]=dest[4]=second;
			src+=8;
			dest+=8;
		}
		break;
	case 2://40 (mode 1)
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			t65::byte knobbled=rearrange_2bit_[video_ram_[src]];
			//Higher nybble: A-A----- and -B-B---- (1010 == 10)
			dest[1]=dest[0]=palette_3bit_[(knobbled>>4)&10];
			dest[3]=dest[2]=palette_3bit_[(knobbled>>3)&10];

			//Lower nybble: ----C-C- and -----D-D
			dest[5]=dest[4]=palette_3bit_[knobbled&10];
			dest[7]=dest[6]=palette_3bit_[(knobbled<<1)&10];
			
			src+=8;
			dest+=8;
		}
		break;
	case 3://80 (mode 0/3)
		if(palette_dirty_fast_) {
			RebuildFastClockPaletteTables();
		}
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			t65::byte b=video_ram_[src];
			*reinterpret_cast<t65::dword *>(dest)=expand_1bit_fast_[b>>4];
			*reinterpret_cast<t65::dword *>(dest+4)=expand_1bit_fast_[b&0xf];

			src+=8;
			dest+=8;
		}
		break;
	}
}

void bbcVideo::RenderSlowClockScanline(t65::byte *dest,t65::word src) {
	if(palette_dirty_3bit_) {
		Rebuild3bitPalette();
		palette_dirty_3bit_=false;
	}
	unsigned cpl=std::_cpp_min(CRTC(1),t65::byte(buffer_width/16));//cap at 40, that's 640 pixels
#ifdef bbcDEBUG_VIDEO
	if(show_all_) {
		cpl=buffer_width/16;
	}
#endif
	unsigned i;
	switch((video_current_[ULA_CTRL]&ULA_CHARSPERLINE)>>ULA_CHARSPERLINE_SHIFT) {
	case 0://10 (mode '8')
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			t65::byte knobbled=rearrange_4bit_[video_ram_[src]];
			t65::byte first=palette_3bit_[knobbled>>4];
			t65::byte second=palette_3bit_[knobbled&15];
			dest[7]=dest[6]=dest[5]=dest[4]=dest[3]=dest[2]=dest[1]=dest[0]=first;
			dest[15]=dest[14]=dest[13]=dest[12]=dest[11]=dest[10]=dest[9]=dest[8]=second;
			src+=8;
			dest+=16;
		}
		break;
	case 1://20 (mode 5)
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			t65::byte knobbled=rearrange_2bit_[video_ram_[src]];
			//Higher nybble: A-A----- and -B-B---- (1010 == 10)
			dest[3]=dest[2]=dest[1]=dest[0]=palette_3bit_[(knobbled>>4)&10];
			dest[7]=dest[6]=dest[5]=dest[4]=palette_3bit_[(knobbled>>3)&10];
			
			//Lower nybble: ----C-C- and -----D-D
			dest[11]=dest[10]=dest[9]=dest[8]=palette_3bit_[knobbled&10];
			dest[15]=dest[14]=dest[13]=dest[12]=palette_3bit_[(knobbled<<1)&10];
			
			src+=8;
			dest+=16;
		}
		break;
	case 2://40 (mode 4/6)
		if(palette_dirty_slow_) {
			RebuildSlowClockPaletteTables();
		}
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}

			t65::byte b=video_ram_[src];
			*reinterpret_cast<t65::qword *>(dest)=expand_1bit_slow_[b>>4];
			*reinterpret_cast<t65::qword *>(dest+8)=expand_1bit_slow_[b&0xf];
			
			src+=8;
			dest+=16;
		}
		break;
	case 3://80 (broken)
		for(i=0;i<cpl;++i) {
			if(src&0x8000) {
				src-=wrap_size_;
			}
			src+=8;
		}
		break;
	}
}

//////////////////////////////////////////////////////////////////////////
// DebugDump
//
// Hmm.
void bbcVideo::DebugDump(FILE *h) {
	fprintf(h,"Video system state:\n");
	fprintf(h,"===================\n");
	fprintf(h,"\n");
	unsigned i;
	for(i=0;i<NUM_REGS;++i) {
		fprintf(h,"%s: 0x%02X (%d)\n",reg_names_[i].c_str(),
			video_current_[i],video_current_[i]);
	}
	fprintf(h,"?&355=%d\n",bbcComputer::ram_[0x355]);
}

//////////////////////////////////////////////////////////////////////////
// Sets the dimensions of the screen buffer.
// Returns false if the dimensions are inappropriate.
bool bbcVideo::SetBufferDimensions(int width,int height) {
	if(width<640||width>800) {
		return false;
	}
	if(height<256||height>300) {
		return false;
	}
	buffer_width=width;
	buffer_height=height;
	return true;
}

#ifdef bbcDEBUG_VIDEO
void bbcVideo::SetShowAll(bool new_show_all) {
	show_all_=new_show_all;
	palette_dirty_3bit_=true;
	show_all_offset_=0;
}

void bbcVideo::LogFrames(const char *filename,int num_frames,bool with_disassembly) {
	fprintf(stderr,"bbcVideo::LogFrames(\"%s\",%d): ",filename,num_frames);
	if(log_frames_h_) {
		fprintf(stderr,"no, %d frames left\n",num_log_frames_left_);
	} else {
		log_frames_h_=fopen(filename,"wt");
		if(log_frames_h_) {
			fprintf(stderr,"yes\n");
			num_log_frames_left_=num_frames;
			if(with_disassembly&&!bbcComputer::trace) {
				bbcComputer::trace=new bbcDebugTrace;
			}
		} else {
			fprintf(stderr,"no, open failed\n");
		}
	}
}
#endif

//////////////////////////////////////////////////////////////////////////
//  0         1         2         3         4         5         6
//  0123456789012345678901234567890123456789012345678901234567890123456789
//00 0 H TOTAL   XX YYY 11 C'R END   XX YYY PAL  00 = 00
// 1 1 CHRS/LINE XX YYY 12 START MSB XX YYY
// 2 2 HSYNC POS XX YYY 13 START LSB XX YYY
// 3 3           XX YYY 14 C'R MSB   XX YYY
// 4   HSYNC LEN XX YYY 15 C'R LSB   XX YYY
// 5   VSYNC LEN XX YYY  
// 6 4 VERT TOT  XX YYY SCREEN BASE   &XXXX
// 7 5 VERT ADJ  XX YYY WRAP SIZE     &XXXX
// 8 6 VERT DISP XX YYY 
// 9 7 VSYNC POS XX YYY 
//10 8           XX YYY 
// 1   INTERLACE XX YYY 
// 2   DISP DEL  XX YYY 
// 3   CURS DEL  XX YYY
// 4 9 LINES/CHR XX YYY 
// 510           XX YYY
// 6   C'R START XX YYY
// 7   C'R BLINK XX YYY
// 8   C'R TYPE  XX YYY

