#ifndef IOP6502_H_
#define IOP6502_H_

//References:
//
//[0]	64doc
//		(1.8 1994/06/03 19:50:04 jopi Exp $)
//[1]	6502 Undocumented Opcodes, Based on the Atari 8-bit 6502C (CO14806)
//		Freddy Offenga (F.Offenga@student.kun.nl)
//

#include "bbcComputer.h"
#include <t65Generic6502InstructionSet.h>
#include <t65Generic6502.h>
#include <t65Sim6502.h>

//hmm wish I'd done this AGES ago.
#define START_INSTR(N) struct Instr##N {static const char *Mnemonic() {return #N "*";}
#define END_INSTR() };

#define DOUBLE_INSTR(M,I0,I1)\
	struct Instr##M:\
	public DoubleInstr<Instr##I0,Instr##I1>\
	{\
		static const char *Mnemonic() {\
			return #M "*";\
		}\
	}

template<class T>
struct bbcSY6502A:
public t65::Generic6502<T>
{
	//////////////////////////////////////////////////////////////////////////
	// Type class for undocumented instructions that are equivalent to a RMW
	// instr followed by a read one.
	template<class DoubleInstr,class AddrMode>
	struct RmwThenRead {
		typedef AddrMode AddrModeType;
		typedef DoubleInstr InstrType;

		static FINLINE void Execute(MachineType &state) {
			t65::Word addr=AddrMode::WriteModeEA(state);
			t65::byte val=AddrMode::ReadOperand(state,addr);
			AddrMode::WriteOperand(state,addr,val);
			val=DoubleInstr::InstrRmw::Execute(state,val);
			AddrMode::WriteOperand(state,addr,val);
			//According to "6502 Undocumented opcodes" (F.Offenga@student.kun.nl,
			//Version 3.1, 6/28/1999) the read instruction doesn't take any extra time
			//compared to just doing the RMW one.
			//
			//Weird.
			DoubleInstr::InstrRead::Execute(state,val);
		}
	};

	//Instruction class for use with the above.
	template<class I1,class I2>
	struct DoubleInstr {
		typedef I1 InstrRmw;
		typedef I2 InstrRead;
	};

	//Instructions that work like that
	DOUBLE_INSTR(ISB,INC,SBC);
	DOUBLE_INSTR(RLA,ROL,AND);
	DOUBLE_INSTR(RRA,ROL,ADC);
	DOUBLE_INSTR(SLO,ASL,ORA);
	DOUBLE_INSTR(SRE,LSR,EOR);

	//////////////////////////////////////////////////////////////////////////
	//AND byte with accumulator. If result is negative then carry is
	//set. Status flags: N,Z,C
	START_INSTR(AAC)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			InstrAND::Execute(state,val);
			state.cpustate.p|=state.cpustate.a>>7;
		}
	END_INSTR()
		
	//////////////////////////////////////////////////////////////////////////
	//AND X register with accumulator and store result in memory.
	//Status flags: N,Z
	START_INSTR(AAX)
		static FINLINE t65::byte Execute(MachineType &state) {
			t65::byte v=state.cpustate.x&state.cpustate.a;
			SetNZ(state,v);
			return v;
		}
	END_INSTR()
		
	//////////////////////////////////////////////////////////////////////////
	//AND byte with accumulator, then shift right one bit in accumu-
	//lator. Status flags: N,Z,C
	START_INSTR(ASR)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			InstrAND::Execute(state,val);
			state.cpustate.a=InstrLSR::Execute(state,state.cpustate.a);
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	//STZ -- store zero
	//
	//allegedly...
	//AND Y register with the high byte of the target address of the
	//argument + 1. Store the result in memory.
	//
	//I've always seen it given as STZ for the Beeb, though...
	START_INSTR(STZ)
		static FINLINE t65::byte Execute(MachineType &) {
			return 0;
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// LAX -- load A and X simultaneously, setting NZ.
	START_INSTR(LAX)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			state.cpustate.x=state.cpustate.a=val;
			SetNZ(state,val);
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// DCP -- subtract 1 from memory without borrow
	// Only affects C flag -- can this REALLY be right?
	// TODO check. I assume C works like SBC, so set based on val!=0.
	START_INSTR(DCP)
		static FINLINE t65::byte Execute(MachineType &state,t65::byte val) {
			state.cpustate.p&=~t65::C_MASK;
			state.cpustate.p|=val!=0;//NB true/false 1/0 value used
			return val-1;
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// ARR -- AND followed by ROR A, with some wierd flag settings! 64doc
	// reckons it has some ADC aspects and is thus affected by the D flag.
	// I don't bother with that...
	//
	// (V ends up as per bit 6 of result. C set if (bit 5) xor (bit 6).)
	START_INSTR(ARR)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			state.cpustate.a=InstrROR::Execute(state,state.cpustate.a&val);
			//ROR set N+Z
			state.cpustate.p&=~(t65::C_MASK|t65::V_MASK);
			state.cpustate.p|=state.cpustate.p&t65::V_MASK;
			state.cpustate.p|=((state.cpustate.a^(state.cpustate.a<<1))>>6)&1;
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// XAS	store (X&msb(EA))
	// SXA	store (X&msb(EA))
	// AXA	64doc says store (A&X&(msb(EA)+1))
	//		[1] says store (A&X&7) (also notes other instructions previously
	//		believed to &7 actually &(msb(EA)+1), this one got away?)
	// 
	// These are not easily done right now, because instructions don't have
	// access to the current effective address.
	// 
	// Since 9C ends up STZ, I'm assuming that this high byte of EA thing is
	// specific to cheap and crappy computers, and the quality workmanship
	// used for the Beeb means that it ends up 0. (STZ, 9C, [1] says
	// Y&msb(EA)+1))
	//
	// So, pending further testing these are ending up as STZ!

	//////////////////////////////////////////////////////////////////////////
	// AXS [1]/SBX 64doc -- X=(A&X)-M
	//
	// Sets N,Z,C like CMP, 64doc reckons a hideous amalgam of CMP and DEX!
	// (But without the DE bit.)
	START_INSTR(SBX)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			Compare(state,state.cpustate.a&state.cpustate.x,val);
			state.cpustate.x=(state.cpustate.a&state.cpustate.x)-val;
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// LAR -- A=X=S=S&M, SetNZ(A)
	START_INSTR(LAR)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			state.cpustate.a=state.cpustate.x=state.cpustate.s.l=state.cpustate.s.l&val;
			SetNZ(state,state.cpustate.a);
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// Unusual Instructions
	//
	// 8B (XAA [1]/ANE [0])
	// AB (ATX [1]/LXA [0])
	//
	// 64doc says A=(A|0xEE)&X&V for both (always 0xEE on 2MHz C128)
	// [1] says ATX is X=A=A&V,SetNZ(X); XAA is unknown.
	// 
	// Hmm... pending further testing will leave AB as per [1] and 8B as per [0].
	START_INSTR(ANE)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			state.cpustate.a=(state.cpustate.a|0xEE)&state.cpustate.x&val;
			SetNZ(state,state.cpustate.a);//I assume it does this!
		}
	END_INSTR()

	START_INSTR(ATX)
		static FINLINE void Execute(MachineType &state,t65::byte val) {
			state.cpustate.x=state.cpustate.a=state.cpustate.a&val;
			SetNZ(state,state.cpustate.x);
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// New CLI and PLP -- set next stop straight away if there are pending
	// interrupts and interrupts were just enabled.
	template<class PInstr>
	struct IrqEnablingInstr:
	public PInstr
	{
		static FINLINE void Execute(MachineType &state) {
			t65::byte old=state.cpustate.p&t65::I_MASK;
			PInstr::Execute(state);
			if(old&&!(state.cpustate.p&t65::I_MASK)) {
				//Interrupts were just enabled
				state.SetNextStop(state.cycles);
			}
		}
	};

	typedef IrqEnablingInstr<InstrCLI> MyInstrCLI;
	typedef IrqEnablingInstr<InstrPLP> MyInstrPLP;

	struct MyInstrRTI {
		static const char *Mnemonic() {
			return "RTI";
		}

		template<class AddrMode>
		static FINLINE void Execute(MachineType &state,const AddrMode &) {
			t65::byte old=state.cpustate.p&t65::I_MASK;
			state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cycles;//2
			state.ReadByte(state.cpustate.s,catReadStack);
			++state.cycles;//3
			state.cpustate.p=Pop(state);
			++state.cycles;//4
			state.cpustate.pc.l=Pop(state);
			++state.cycles;//5
			state.cpustate.pc.h=Pop(state);
			++state.cycles;//6
			if(old&&!(state.cpustate.p&t65::I_MASK)) {
				//Interrupts were just enabled
				state.SetNextStop(state.cycles);
			}
		}
		
		// removed specialization [18/4/2003]
		/*
		template<class AddrMode>
		static FINLINE void Execute(MachineType &state,const AddrMode &) {
			SoftBreak();
		}
		
		template<>
		TSP_STATIC TSP_FINLINE void Execute<ModeIMP>(MachineType &state,const ModeIMP &) {
			t65::byte old=state.cpustate.p&t65::I_MASK;
			state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cycles;//2
			state.ReadByte(state.cpustate.s,catReadStack);
			++state.cycles;//3
			state.cpustate.p=Pop(state);
			++state.cycles;//4
			state.cpustate.pc.l=Pop(state);
			++state.cycles;//5
			state.cpustate.pc.h=Pop(state);
			++state.cycles;//6
			if(!old&&(state.cpustate.p&t65::I_MASK)&&state.irq_flags_) {
				//Interrupts were just enabled, and there are some pending.
				state.SetNextStop(state.cycles);
			}
		}
		*/
	};
};
/*
#ifdef _MSC_VER
#include "VcDotNetInstructionSet.h"
typedef VcDotNetInstructionSet bbcIoSY6502AInstructionSetBase;
#else
*/
//typedef t65::Generic6502InstructionSet<IOP6502> bbcIoSY6502AInstructionSetBase;
//#endif

template<class T>
struct bbcSY6502AInstructionSet:
public t65::Generic6502InstructionSet<T>
{
	//Special instructions for special IRQ handling
	typedef P::Implied<P::MyInstrPLP,P::ModeIMP> Opcode28;
	typedef P::Special<P::MyInstrRTI,P::ModeIMP> Opcode40;
	typedef P::Implied<P::MyInstrCLI,P::ModeIMP> Opcode58;

	//Crazy undocumented opcodes

	//ISB -- INC-SBC
	typedef P::RmwThenRead<P::InstrISB,P::ModeZPG> OpcodeE7;
	typedef P::RmwThenRead<P::InstrISB,P::ModeZPX> OpcodeF7;
	typedef P::RmwThenRead<P::InstrISB,P::ModeABS> OpcodeEF;
	typedef P::RmwThenRead<P::InstrISB,P::ModeABX> OpcodeFF;
	typedef P::RmwThenRead<P::InstrISB,P::ModeABY> OpcodeFB;
	typedef P::RmwThenRead<P::InstrISB,P::ModeINX> OpcodeE3;
	typedef P::RmwThenRead<P::InstrISB,P::ModeINY> OpcodeF3;

	//RLA -- ROL memory then ORA memory
	typedef P::RmwThenRead<P::InstrRLA,P::ModeZPG> Opcode27;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeZPX> Opcode37;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeABS> Opcode2F;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeABX> Opcode3F;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeABY> Opcode3B;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeINX> Opcode23;
	typedef P::RmwThenRead<P::InstrRLA,P::ModeINY> Opcode33;

	//RRA -- ROR memory then ADC memory
	typedef P::RmwThenRead<P::InstrRRA,P::ModeZPG> Opcode67;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeZPX> Opcode77;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeABS> Opcode6F;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeABX> Opcode7F;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeABY> Opcode7B;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeINX> Opcode63;
	typedef P::RmwThenRead<P::InstrRRA,P::ModeINY> Opcode73;

	//SBC -- Alternative form
	typedef P::Read<P::InstrSBC,P::ModeIMM> OpcodeEB;

	//SLO -- ASL memory then ORA memory
	typedef P::RmwThenRead<P::InstrSLO,P::ModeZPG> Opcode07;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeZPX> Opcode17;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeABS> Opcode0F;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeABX> Opcode1F;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeABY> Opcode1B;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeINX> Opcode03;
	typedef P::RmwThenRead<P::InstrSLO,P::ModeINY> Opcode13;

	//SRE -- LSR memory then EOR memory
	typedef P::RmwThenRead<P::InstrSRE,P::ModeZPG> Opcode47;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeZPX> Opcode57;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeABS> Opcode4F;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeABX> Opcode5F;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeABY> Opcode5B;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeINX> Opcode43;
	typedef P::RmwThenRead<P::InstrSRE,P::ModeINY> Opcode53;

	//NOP -- extra ones with funny modes!
	typedef P::Read<P::InstrNOP,P::ModeZPG> Opcode04;
	typedef P::Read<P::InstrNOP,P::ModeZPX> Opcode14;
	typedef P::Read<P::InstrNOP,P::ModeZPX> Opcode34;
	typedef P::Read<P::InstrNOP,P::ModeZPG> Opcode44;
	typedef P::Read<P::InstrNOP,P::ModeZPX> Opcode54;
	typedef P::Read<P::InstrNOP,P::ModeZPG> Opcode64;
	typedef P::Read<P::InstrNOP,P::ModeZPX> Opcode74;
	typedef P::Read<P::InstrNOP,P::ModeIMM> Opcode80;
	typedef P::Read<P::InstrNOP,P::ModeIMM> Opcode82;
	typedef P::Read<P::InstrNOP,P::ModeIMM> Opcode89;
	typedef P::Read<P::InstrNOP,P::ModeIMM> OpcodeC2;
	typedef P::Read<P::InstrNOP,P::ModeZPX> OpcodeD4;
	typedef P::Read<P::InstrNOP,P::ModeIMM> OpcodeE2;
	typedef P::Read<P::InstrNOP,P::ModeZPX> OpcodeF4;
	typedef P::Read<P::InstrNOP,P::ModeABS> Opcode0C;
	typedef P::Read<P::InstrNOP,P::ModeABX> Opcode1C;
	typedef P::Read<P::InstrNOP,P::ModeABX> Opcode3C;
	typedef P::Read<P::InstrNOP,P::ModeABX> Opcode5C;
	typedef P::Read<P::InstrNOP,P::ModeABX> Opcode7C;
	typedef P::Read<P::InstrNOP,P::ModeABX> OpcodeDC;
	typedef P::Read<P::InstrNOP,P::ModeABX> OpcodeFC;

	//NOP -- aliases for the normal one.
	typedef OpcodeEA Opcode1A;
	typedef OpcodeEA Opcode3A;
	typedef OpcodeEA Opcode5A;
	typedef OpcodeEA Opcode7A;
	typedef OpcodeEA OpcodeDA;
	typedef OpcodeEA OpcodeFA;
	
	//AAC
	typedef P::Read<P::InstrAAC,P::ModeIMM> Opcode0B;
	typedef P::Read<P::InstrAAC,P::ModeIMM> Opcode2B;

	//AAX
	typedef P::Write<P::InstrAAX,P::ModeZPG> Opcode87;
	typedef P::Write<P::InstrAAX,P::ModeZPY> Opcode97;
	typedef P::Write<P::InstrAAX,P::ModeINX> Opcode83;
	typedef P::Write<P::InstrAAX,P::ModeABS> Opcode8F;

	//ASR
	typedef P::Read<P::InstrASR,P::ModeIMM> Opcode4B;

	//STZ
	typedef P::Write<P::InstrSTZ,P::ModeABS> Opcode9C;

	//LAX
	typedef P::Read<P::InstrLAX,P::ModeINX> OpcodeA3;
	typedef P::Read<P::InstrLAX,P::ModeZPG> OpcodeA7;
	typedef P::Read<P::InstrLAX,P::ModeABS> OpcodeAF;
	typedef P::Read<P::InstrLAX,P::ModeINY> OpcodeB3;
	typedef P::Read<P::InstrLAX,P::ModeZPY> OpcodeB7;
	typedef P::Read<P::InstrLAX,P::ModeABY> OpcodeBF;

	//DCP
	typedef P::RMW<P::InstrDCP,P::ModeINX> OpcodeC3;
	typedef P::RMW<P::InstrDCP,P::ModeZPG> OpcodeC7;
	typedef P::RMW<P::InstrDCP,P::ModeABS> OpcodeCF;
	typedef P::RMW<P::InstrDCP,P::ModeINY> OpcodeD3;
	typedef P::RMW<P::InstrDCP,P::ModeZPX> OpcodeD7;
	typedef P::RMW<P::InstrDCP,P::ModeABY> OpcodeDB;
	typedef P::RMW<P::InstrDCP,P::ModeABX> OpcodeDF;

	//ARR
	typedef P::Read<P::InstrARR,P::ModeIMM> Opcode6B;

	//XAS, SXA, AXA -- STZ on the Beeb I guess!
	typedef P::Write<P::InstrSTZ,P::ModeINY> Opcode93;//AXA INY
	typedef P::Write<P::InstrSTZ,P::ModeABY> Opcode9B;//XAS ABY
	typedef P::Write<P::InstrSTZ,P::ModeABY> Opcode9E;//SXA ABY
	typedef P::Write<P::InstrSTZ,P::ModeABY> Opcode9F;//AXA ABY

	//SBX
	typedef P::Read<P::InstrSBX,P::ModeIMM> OpcodeCB;

	//LAR
	typedef P::Read<P::InstrLAR,P::ModeIMM> OpcodeBB;

	//ANE/ATX/XAA/wierd
	typedef P::Read<P::InstrANE,P::ModeIMM> Opcode8B;
	typedef P::Read<P::InstrATX,P::ModeIMM> OpcodeAB;
};

#endif
