#ifndef BBCMICRO_H_
#define BBCMICRO_H_

#include "bbcCommon.h"
#include "bbcRomSlot.h"
#include "bbcKeys.h"
#include "bbcDebugPanel.h"
//#include "bbc1770Interface.h"
#include "bbcModel.h"
#include "bbcDebugTrace.h"

struct bbc1770Interface;

//State for a BBC computer. This is good for:
//
//Model A			not done
//Model B			ok
//Model B+,			not done
//Model B+128		not done
//Master 128K		not done

struct bbcComputer {
	struct MmioFnInit {
		t65::word start;
		unsigned count;//count==0 for last entry
		bbcReadMmioFn read_fn;
		bbcWriteMmioFn write_fn;
	};
	
	//////////////////////////////////////////////////////////////////////////
	// Interface to rest of emulator
	static void Init(const bbc1770Interface *disc_interface,int video_width,
		int video_height,bbcModel *model);
	static void InstallOSHacks();
	static void Update();
	static void SetOsRomContents(const t65::byte *contents,unsigned size_contents);
	static void SetRomContents(int slot,const t65::byte *contents,unsigned size_contents);
	static void SetRomWriteable(int slot,bool writeable);
//	static void SetHasRomBoard(bool has_rom_board);
	static void DebugDumpMemory();
	static void ResetSoft();//press break
	static void ResetHard();//power on reset
	static void Run6502();//runs 6502 for 0 or more cycles

	//////////////////////////////////////////////////////////////////////////
	// debug
/*
#ifdef bbcDEBUG_PANELS
	static void InitDebugPanel(bbcDebugPanel *panel);
	static void UpdateDebugPanel(bbcDebugPanel *panel);
#endif
*/
#ifdef bbcDEBUG_TRACE
	static bbcDebugTrace *trace;
	static void AddIrqTrace();
#endif
	static int DisassembleBytes(const t65::byte *bytes,t65::Word bytes_addr,
		char *opcode_buf,char *operand_buf);
	
	//////////////////////////////////////////////////////////////////////////
	// Interface to 6502 bits
	static void WriteMmio(t65::Word addr,t65::byte val);
	static t65::byte ReadMmio(t65::Word addr);

	//SetNextStop
	//
	//Set the point (absolute or relative) at which the 6502 should next stop
	//to do some updating type stuff.
	static void SetNextStop(int cycles);
	static void SetNextStopDelta(int delta);

	//SetNextIrqStop
	//SetNextIrqStopDelta
	//
	//Set the point (absolute or relative) at which the 6502 should next stop
	//to receive an IRQ.
	static void SetNextIrqStop(int cycles);
	static void SetNextIrqStopDelta(int delta);

	//ResetNMI
	//SetNMI
	//
	//Set or clear the NMI flag. Don't set irq_flags_ directly, because this
	//won't NMI immediately if interrupts are disabled.
	static void ResetNMI();
	static void SetNMI();

	static int next_stop;//cycle count at which to next stop to do some updating.
	static unsigned irq_flags_;//Interrupts

	//////////////////////////////////////////////////////////////////////////
	// 6502 simulator requirements
	typedef t65::State65xx StateType;
	static StateType cpustate;
	static int cycles;
	
	//memory access categories
	struct WriteNoHW {};
	struct ReadNoHW {};

	struct WriteWithHW {};
	struct ReadWithHW {};
	
	//grim B+ crap
	struct WriteWithHwAndShadowRam {};
	struct ReadWithHwAndShadowRam {};

	template<typename Policy>
	static FINLINE t65::byte ReadByte(t65::Word,const Policy *const=0) {
	}

	template<typename Policy>
	static FINLINE void WriteByte(t65::Word,t65::byte,const Policy *const) {
	}

	template<>
	TSP_STATIC TSP_FINLINE t65::byte ReadByte(t65::Word addr,const ReadWithHW *const) {
		if(t65::byte(addr.h-0xfc)>=3) {
			return read_pages_[addr.h][addr.l];
		} else {
			return ReadMmio(addr);
		}
	}

	template<>
	TSP_STATIC TSP_FINLINE t65::byte ReadByte(t65::Word addr,const ReadNoHW *const) {
		return read_pages_[addr.h][addr.l];
	}
	
	template<>
	TSP_STATIC TSP_FINLINE void WriteByte(t65::Word addr,t65::byte val,
		const WriteWithHW *const)
	{
		if(t65::byte(addr.h-0xfc)>=3) {
			write_pages_[addr.h][addr.l]=val;
		} else {
			WriteMmio(addr,val);
		}
	}

	template<>
	TSP_STATIC TSP_FINLINE void WriteByte(t65::Word addr,t65::byte val,const WriteNoHW *const) {
		write_pages_[addr.h][addr.l]=val;
	}

	template<>
	TSP_STATIC TSP_FINLINE void WriteByte(t65::Word addr,t65::byte val,
		const WriteWithHwAndShadowRam *const)
	{
		if(t65::byte(addr.h-0xfc)>=3) {
			pc_write_pages_[cpustate.pc.h][addr.h][addr.l]=val;
		} else {
			WriteMmio(addr,val);
		}
	}

	template<>
	TSP_STATIC TSP_FINLINE t65::byte ReadByte(t65::Word addr,
		const ReadWithHwAndShadowRam *const)
	{
		if(t65::byte(addr.h-0xfc)>=3) {
			return pc_read_pages_[cpustate.pc.h][addr.h][addr.l];
		} else {
			return ReadMmio(addr);
		}
	}

	static void SyncClocks();//int *cycle_counter);
	//This is the RAM for everything
	//Certain models will use less of it.
	static t65::byte ram_[128*1024];//TODO MAke private

	//B romsel
	static t65::byte ReadRomSelB(t65::byte offset);
	static void WriteRomSelB(t65::byte offset,t65::byte value);
	
	//B+ romsel
	static t65::byte ReadRomSelBPlus(t65::byte offset);
	static void WriteRomSelBPlus(t65::byte offset,t65::byte value);
private:
	static bbcModel *model_;
//	static t65::byte PhysicalRomSlot(t65::byte logical_rom_slot);
	static bbcRomSlot rom_slots_[16];
	static bbcRomSlot os_rom_;
	static t65::byte romsel_;
	static bool vdusel_;//bit 7 of FE34, B+ only.
	static bool paged_ram_sel_;//bit 7 of FE30, B+ only.
//	static t65::byte romsel_mask_,romsel_fixed_bits_;//TODO Remove, overly complex.

	//////////////////////////////////////////////////////////////////////////
	// read_pages_
	// write_pages_
	// These are used for B+ and BBC B. They generally stay fixed. The Master
	// pages will be elsewhere since they get changed a bit more (and in a more
	// straightforward manner).
	//
	//These are always the read_pages for a default Beeb.
	//That is, <blah>_pages[X] points to the Xth page in normal RAM.
	//No shadow or anything like that!
	static t65::byte *read_pages_[256];
	static t65::byte *write_pages_[256];

	//These are used for shadow RAM mode (VDUSEL set)
	static t65::byte *shadow_read_pages_[256];
	static t65::byte *shadow_write_pages_[256];

	//According to MSB of PC (and VDUSEL of course) each entry here points
	//either to X_pages_ or shadow_X_pages_.
	static t65::byte **pc_read_pages_[256];
	static t65::byte **pc_write_pages_[256];

	//////////////////////////////////////////////////////////////////////////
	// Mmemomry mapped I/O functions
	//
	// These handle reads and writes to the area 0xFC00-0xFEFF inclusive.
	static bbcWriteMmioFn mmio_write_fns_[0x300];
	static bbcReadMmioFn mmio_read_fns_[0x300];
	
	static const MmioFnInit mmio_fn_inits_[];

	static void ResetMmioFns();
	static void AddMmioFns(const MmioFnInit *inits);
	static void SetMmioFn(t65::word addr,bbcReadMmioFn read_fn,
		bbcWriteMmioFn write_fn);
};
/*
inline t65::byte bbcComputer::PhysicalRomSlot(t65::byte logical_rom_slot) {
	return (logical_rom_slot&romsel_mask_)|romsel_fixed_bits_;
}
*/
inline void bbcComputer::SetRomWriteable(int slot,bool writeable) {
	rom_slots_[slot].SetWriteable(writeable);
}

inline void bbcComputer::SetOsRomContents(const t65::byte *contents,unsigned size_contents) {
	os_rom_.SetWriteable(false);
	os_rom_.SetContents(contents,size_contents);
}

//this synchronizes the 1mhz and 2mhz clocks.
inline void bbcComputer::SyncClocks() {
	if(cycles&1) {
		cycles+=2;
	} else {
		++cycles;
	}
}

inline void bbcComputer::SetNextStop(int new_next_stop) {
	if(int(new_next_stop-next_stop)<0) {//cycles<next_stop) {
		next_stop=new_next_stop;
	}
}

inline void bbcComputer::SetNextStopDelta(int delta) {
	if(int(cycles+delta-next_stop)<0) {//cycles+delta<next_stop) {
		next_stop=cycles+delta;
	}
}

inline void bbcComputer::SetNextIrqStop(int new_next_stop) {
	if(!(cpustate.p&t65::I_MASK)&&int(new_next_stop-next_stop)<0) {//cycles<next_stop) {
		next_stop=new_next_stop;
	}
}

inline void bbcComputer::SetNextIrqStopDelta(int delta) {
	if(!(cpustate.p&t65::I_MASK)&&int(cycles+delta-next_stop)<0) {//cycles+delta<next_stop) {
		next_stop=cycles+delta;
	}
}

inline void bbcComputer::ResetNMI() {
	irq_flags_&=~IRQ_NMI;
}

inline void bbcComputer::SetNMI() {
	irq_flags_|=IRQ_NMI;
	next_stop=cycles;//stop now
}

FINLINE void bbcComputer::Run6502() {
	model_->Run();
}

inline int bbcComputer::DisassembleBytes(const t65::byte *bytes,t65::Word bytes_addr,
	char *opcode_buf,char *operand_buf)
{
	return model_->DisassembleBytes(bytes,bytes_addr,opcode_buf,operand_buf);
}

#endif
