/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// ECOLINK.C
//
// Acorn ecolink add on card library routines            
//                                                       
// These are all based on the INT2F routines implemented by the TSR program ecolinkl.com from the original driver
// set as supplied with the card
//
// (C) Mark Usher
// marku@magnet.at
// release 1.0  30.06.1999                                                       
*/

#include <go32.h>
#include <dpmi.h>
#include <sys/farptr.h>
#include <sys/types.h>
#include <sys/movedata.h>

#include <conio.h>
#include <dos.h>
#include <stdio.h>

#include "initcard.h"

#include <string.h>
#include <malloc.h>

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Function prototypes
*/


/*
// COMMANDS
*/
byte OpenRxCBH(byte Port, unsigned short Station, byte RPCFlag, unsigned short RPCNo, byte Timeout);
byte OpenTxCBH(	byte ControlByte, byte Port, unsigned short Station,
				byte Retries, unsigned short Data_Length, unsigned short RemoteAddress, char *pData);
byte PollCBH(byte CBH, struct ReplyBlock_PollCB *ReplyBlock);
unsigned short KillCBH(byte CBH, byte Kill_type);
unsigned short ReadCBH(byte CBH, unsigned short StartByte, unsigned short NoOfBytes, char *buffer);

unsigned short GetMachineType(byte *machine);
unsigned short SetMachineType(byte *machine);
unsigned short GetStationNumber(byte *station);
unsigned short SetStationNumber(byte *station);
unsigned short GetTxPauseValue(unsigned short *TXPause);
unsigned short SetTxPauseValue(unsigned short *TXPause);
byte EEPROM(byte command, byte address, byte *data);

unsigned short AddEventHandler(long function, long end_of_function);
unsigned short DeleteEventHandler(long function);
unsigned short SetDefaultEventAction(byte NewAction);
unsigned short SetIndividualAction(byte CBH, byte NewAction);
unsigned short TestClearEventStatus(byte *CBH, byte Action);
unsigned short EventScheduler(byte status);

/*
// Function prototypes
*/
void        ReturnCodeError(unsigned short error_code);
void        EnterCriticalSection(byte ID);
void        LeaveCriticalSection(byte ID);
byte        GetHandle(unsigned long ListOffset);
void        AddHandleToList(unsigned long ListOffset, byte Handle);
void        end_AddHandleToList();
void        RemoveHandleFromList(unsigned long ListOffset, byte Handle, byte pos);
void        end_RemoveHandleFromList();
inline int  GetAddressOfBuffer(byte Handle);
byte        GetFirstCB(unsigned long ListOffset);
byte        FindBufferInList(unsigned long ListOffset, byte Handle);
bool        validHandle(byte CBH);
bool        ScanListForStatus(unsigned long ListOffset, byte pos, byte value);


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 1
// Open RxCBH
//
// This command opens a RxCB. If the port number is zero then the RxCB is open for reception on all ports.
// If both bytes of the station number are zero then the RxCB is open for reception from all stations.
// The Rx primatives must be capable of receiving any length of broadcast on station id 0FEFFH.
//
// If the RPC flag is set, then the CB is open to receive Remote Procedure Calls. If bit6 is set then only
// the RPC number held in bytes 5,6 will be received by this command block. When open for RPC the port number
// is used to define which sort of RPC is allowed.
//
// If the timeout byte is zero then the CB will never timeout. If the timeout byte is non-zero then an event
// will automatically be generated after timeout * 500ms. This will not kill the CB and if data is received
// after the timeout another event my be generated unless the CB is killed.
//
// Multiple CBs open in the same port and station ID should be filled in the order in which they were opened.
// 
// return code is from the standard set
*/
byte OpenRxCBH(byte Port,
               unsigned short Station,
               byte RPCFlag,
               unsigned short RPCNo,
               byte Timeout)
{
    byte CBH;
    int Address;
    struct RXCBHeader *pRxBuffer;

    EnterCriticalSection(1);

    CBH = GetHandle(UNUSED_LIST);               /* Get a handle from the unused list */

    if (CBH == 0)                               /* Check the handles validity */
    {
       LeaveCriticalSection(1);
       return RC_NO_HANDLES_LEFT;
    }


    AddHandleToList(OPEN_RX_LIST, CBH);			/* Add the handle to the open Rx list */

    pRxBuffer = (struct RXCBHeader *)malloc(sizeof(struct RXCBHeader));
    pRxBuffer->ControlByte  = 0x7F;				/* Control Byte Set all apart from the top bit */
    pRxBuffer->Port         = Port;				/* Port */
    pRxBuffer->Station      = Station;			/* Station */
    pRxBuffer->Length       = MAX_PACKET_LENGTH;/* Econet max packet length */
    pRxBuffer->Event        = DefaultEventAction;/* The event will be stored by default on completion */
    pRxBuffer->Status       = 0x00;				/* Status */
    pRxBuffer->KillFlag     = 0x00;				/* Kill flag */
    pRxBuffer->CtrlByteCopy = 0x7F;			    /* Control byte copy */	
    pRxBuffer->RPCFlag      = RPCFlag;          /* Remote procedure call flag */
    pRxBuffer->RPCNumber    = RPCNo;			/* Remote procedure call number */
    pRxBuffer->Timeout      = Timeout;			/* timeout period */
    pRxBuffer->E            = 0x00;
    pRxBuffer->F            = 0x00;
    pRxBuffer->_10          = 0x00;
    pRxBuffer->_11          = 0x00;

    Address = GetAddressOfBuffer(CBH);			/* get the address of the handle's memory block */
    
    movedata(_my_ds(),							/* Download the buffer to the ecolink card */							
             (int)pRxBuffer,
             card_address_selector,
             Address,
             0x11);
	
    free(pRxBuffer);							/* free allocated memory */
    LeaveCriticalSection(1);
    return CBH;
}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 2
// Open TxCBH
//
// This command opens a TxCB.  The data to transmitted is pointed to by the data pointer in the control block.
// 
// If the port number is zero then an immediate operation will be transmitted. The operation is passed in the 
// control byte. If the station number is 0FFFFH
// then the packet will be broadcast; note that while this interface allows broadcast packets which are not 8
// bytes long, any BBC computers on the network will stomp on a broadcast packet of greater than 9 bytes.
// If the station id is 0FEFFH a local broadcast will take place which may be of any length.
//
// The length field should be the same value as the Length of tx data block in the control block; unless the
// length field has its top bit set, in which case no data will be transfered but the length field will be set
// to the low 15 bits, and the Length of tx data block should be set to zero.  This is needed for peek type
// immediate ops.  The returned data can be read using the read command.
//
// Zero length Tx packets are allowed.
// 
// return code is from the standard set
*/
byte OpenTxCBH(	byte ControlByte, byte Port, unsigned short Station,
				byte RetryCount, unsigned short Data_Length, unsigned short RemoteAddress,
				char *pData)
{
    byte CBH;
    byte NoOfOpenCBs;

    bool Found;
    byte TXhandle;
    unsigned short TXstation;
    int TXbuffer;
    int Address;
    int i;

    struct TXCBHeader *pTxBuffer;

    byte NoOpenCBs;
    byte next;

    if ((Data_Length > MAX_PACKET_LENGTH) && (Port !=0))	/* check the length of the data passed */
       return RC_DATA_SIZE_LARGER_THAN_BUFFER;				/* Exceeds the maximum allowed */

    EnterCriticalSection(2);

    CBH = GetHandle(UNUSED_LIST);               /* Get a handle from the unused list */

    if (CBH == 0)                               /* Check the handles validity */
    {
       LeaveCriticalSection(2);
       return RC_NO_HANDLES_LEFT;
    }

    /* The top bit of the control byte (which is read by the POLL CB command
       is automatically set by this command, and is reset when transmission is
       completed (or when retries have been exhausted.
	*/
    ControlByte = ControlByte | 0x80;

    pTxBuffer = (struct TXCBHeader *)malloc(sizeof(struct TXCBHeader));
    pTxBuffer->ControlByte  = ControlByte;		/* Control byte */
    pTxBuffer->Port         = Port;				/* Port */
    pTxBuffer->Station      = Station;			/* Station */
    pTxBuffer->Length       = Data_Length;		/* Data length to be transmitted */
    pTxBuffer->Event        = DefaultEventAction;/* The event will be stored by default on completion */
    pTxBuffer->Status       = 0x00;				/* Status */
    pTxBuffer->KillFlag     = 0x00;				/* Kill flag */
    pTxBuffer->CtrlByteCopy = ControlByte;		/* Control byte copy */
    pTxBuffer->RetryCount   = RetryCount;		/* retry count */

    if (Port == 0)								/* it's an immediate op */
		pTxBuffer->BufferOffset = RemoteAddress;/* remote address of data to be returned */
	else
		pTxBuffer->BufferOffset = 0x00;			/* otherwise not needed */

	pTxBuffer->Timeout      = 0x00;				/* timeout */
    pTxBuffer->E            = 0x00;

    Address = GetAddressOfBuffer(CBH);			/* get the address of the handle's memory block */

    movedata(_my_ds(),							/* Download the tx data to the ecolink card */
             (int)pData,						
             card_address_selector,
             Address + 0x11,
             Data_Length);

	/* When a Transmit is allocated for a station, if there already exists a tx operation for the station
	   then a chain must be built and this handle must be added to the end of the chain.

	   Search for the station in the list of open TX control blocks 
	*/

    NoOfOpenCBs = _farpeekb(card_address_selector, OPEN_TX_LIST);   /* Number of open TX CB's  */

    Found = FALSE;

    if (NoOfOpenCBs != 0)			/* if there are any handles open then search through them */
    {
       /* Loop through the TxCB buffers to find the last one with a matching station number 
          This is indicated where station number is the same and byte (+0x10) Next == 0 
	   */
       for (i=1; i<=NoOfOpenCBs; i++)
       {
          TXhandle = _farpeekb(card_address_selector, OPEN_TX_LIST + i );   /* Handle of open TX CB  */
          TXbuffer =  GetAddressOfBuffer(TXhandle);                         /* buffer address of open TX CB */
          TXstation = _farpeekw(card_address_selector, TXbuffer + 2);       /* station in the open TX CB */

          if (TXstation == Station)                                         /* same as the station for the new TX CB ? */
          {
             Found = TRUE;										            /* A CB for the same station has been found */
             i = NoOfOpenCBs;												/* i = position */ 

             next = _farpeekb(card_address_selector, TXbuffer + 0x10);		/* Get this handles forward pointer */


             /* If the forward pointer has another CB in it then follow the chain
                to find the last CB
			 */
             while (next !=0)
             {
                 TXbuffer = GetAddressOfBuffer(next);
                 TXhandle = next;
                 next = _farpeekb(card_address_selector, TXbuffer + 0x10);
             }

             /* Update the pointers for the current last control block in the chain to the handle of the CBH being added */
             _farpokeb(card_address_selector, TXbuffer + 0x10, CBH );

			 /* add these details to the CBH that is about to be intialised */
             pTxBuffer->PreviousTx = TXhandle;    /* pointer to previous entry in the list */
             pTxBuffer->NextTx     = 0x00;        /* pointer to next entry in the list */

             /* now that all details are known, copy the buffer header to the card  */
             movedata(_my_ds(),
                      (int)pTxBuffer,
                      card_address_selector,
                      Address,
                      0x11);

             /* and add the CBH to the last position in the chain */
             AddHandleToList(OPEN_TXCHAIN_LIST, CBH);

          }
       }
    }


	/* if there was no chain found then the previous and next pointers will be initialised to 0 */
    if (Found == FALSE)
    {
        pTxBuffer->PreviousTx = 0x00;        /* pointer to previous entry in the list */
        pTxBuffer->NextTx     = 0x00;        /* pointer to next entry in the list */

        /* now that all details are known, copy the buffer header to the card  */
        movedata(_my_ds(),
                 (int)pTxBuffer,
                 card_address_selector,
                 Address,
                 0x11);

        /* and add the CBH to the Tx list  */
        AddHandleToList(OPEN_TX_LIST, CBH);

        /* Set the tx opened flag as this is a CBH for a new destination */
        _farpokeb(card_address_selector, TXCB_OPENED_FLAG, 0x80);
    }

    free(pTxBuffer);						/* free allocated memory */
    LeaveCriticalSection(2);
    return CBH;

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 3
// Poll CB
//
// This command tests the state of a control block and returns
// port, statid and length info if they can be ascertained.
//
// There are two wildcard handles that can be passed to this function.
// 0xFF - TxWildcard will return the 1st TxCB with success - 0xFE in the status byte
// 0xFE - RxWildcard will return the 1st RxCB with success - 0xFD in the status byte
// 
// return code is from the standard set
*/
byte PollCBH(byte CBH, struct ReplyBlock_PollCB *ReplyBlock)
{
    bool rc;
    int Address;

    /* check the handles range */
    switch (CBH)
    {
    case 0x01:    case 0x02:    case 0x03:    case 0x04:
    case 0x05:    case 0x06:    case 0x07:    case 0x08:
    case 0x09:    case 0x0A:    case 0x0B:    case 0x0C:
    case 0x0D:    case 0x0E:    case 0x0F:    case 0x10:
    case 0x11:    case 0x12:    case 0x13:    case 0x14:
    case 0x15:    case 0x16:
         /* handle is valid so check if it is in the unused list */
        rc = FindBufferInList(UNUSED_LIST, CBH);
        if (rc != 0)
            return RC_HANDLE_NOT_IN_USE;

        break;
    case RxWILDCARD:
         /* find the first Rx block in the completed list with it's status byte set
            to 0xFD RxCB successful                                           
		 */
         CBH = ScanListForStatus(COMPLETED_LIST, 0x07, RxCB_SUCCESSFUL);
         break;
    case TxWILDCARD:
         /* find the first Tx block in the completed list with it's status byte set
            to 0xFF TxCB successful                                           
		 */
         CBH = ScanListForStatus(COMPLETED_LIST, 0x07, TxCB_SUCCESSFUL);
         break;
    default:
         return RC_BAD_HANDLE;
    }

    if (CBH==0)
        return RC_NO_MATCH_FOR_WILDCARD;

    Address = GetAddressOfBuffer(CBH);			/* get the address of the handle's memory block */

    ReplyBlock->CBH         = CBH;
    ReplyBlock->ControlByte = _farpeekb(card_address_selector, Address);        /* Control byte  */
    ReplyBlock->Port        = _farpeekb(card_address_selector, Address + 1);    /* port byte     */
    ReplyBlock->Station     = _farpeekw(card_address_selector, Address + 2);	/* Station byte  */
    ReplyBlock->Length      = _farpeekw(card_address_selector, Address + 4);    /* Length byte   */

    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 4
// Kill CBH
//
// This command kills a control block and returns it to the free chain.  It will return 6 as a return code if the kill
// condition fails.  If the kill condition is not 'always' and the block is currently being either received or
// transmitted then it will return 7 as a return code; under these circumstances the operation should be retried.
// 
// return code is from the standard set
*/
unsigned short KillCBH(byte CBH, byte Kill_type)
{
    int Address;
    int rc;
    byte pos;
    byte CtrlByteCopy;

    if (validHandle(CBH)==FALSE)			    /* Check to see if the handle is valid */
       return RC_BAD_HANDLE;

    Address = GetAddressOfBuffer(CBH);			/* get the address of the handle's memory block */

    EnterCriticalSection(02);
    EnterCriticalSection(01);

    pos = FindBufferInList(UNUSED_LIST, CBH);	/* get the position in the list now */
    CtrlByteCopy = _farpeekb(card_address_selector, Address + 0x09);	/* get the copy of the control byte */

	/* depending on the type of kill operation */
    switch (Kill_type) 
	{
    case KILL_ALL:
        _farpokeb(card_address_selector, Address + 0x08 , 0x80 );
        rc = RC_COMMAND_COMPLETED_OK;
        break;
    case KILL_TOP_BIT_ZERO:						/* kill if the control byte copy top bit is 0 ie an Rx handle */
        if (( CtrlByteCopy & 0x80) == 0)
        {
            _farpokeb(card_address_selector, Address + 0x08 , 0x80 );
            rc = RC_COMMAND_COMPLETED_OK;
        }
        else
            rc = RC_KILL_CONDITION_FAILED;
        break;
    case KILL_TOP_BIT_SET:
        if (( CtrlByteCopy & 0x80) == 0x080)	/* kill if the control byte copy top bit is 1 ie an Tx handle */
        {
            _farpokeb(card_address_selector, Address + 0x08 , 0x80 );
            rc = RC_COMMAND_COMPLETED_OK;
        }
        else
            rc = RC_KILL_CONDITION_FAILED;
       break;
    default:
         rc = RC_KILL_CONDITION_FAILED;
    }

	/* if the handle wasn't in the unused list */
    if (pos == 0)									
    {
        RemoveHandleFromList(COMPLETED_LIST, CBH, 0);
        AddHandleToList(        UNUSED_LIST, CBH);
    }

    LeaveCriticalSection(01);
    LeaveCriticalSection(02);

	/* make sure that the handle is removed from the event lists */
    RemoveHandleFromList(ENABLED_LIST, CBH, 0);
    RemoveHandleFromList( STORED_LIST, CBH, 0);

    return rc;
}

/* THERE IS NO NET COMMAND 5 */

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 6
// Read CBH
//
// This command reads received data from an RxCB or the data to be transmitted from a TxCB.  If the start byte is past
// the end of the data then no data will be returned.  If the number of bytes field is zero or greater than the number
// of bytes in the RxCB then all the data from the start byte to the end of the RxCB will be returned.
//
// The action of reading a CB does not kill it.  It may be read many times or not at all.  The contents of TxCBs may
// also be read.
// 
// return code is from the standard set
*/
unsigned short ReadCBH(byte CBH, unsigned short StartByte, unsigned short NoOfBytes, char *buffer)
{
    int Address;
    unsigned short ByteCount;

    if (validHandle(CBH)==FALSE)								    /* Check to see if the handle is valid */
       return RC_BAD_HANDLE;

    Address = GetAddressOfBuffer(CBH);								/* get the address of the handle's memory block */
    ByteCount = _farpeekw(card_address_selector, Address + 04);		/* number of bytes that have been received */

    if ((NoOfBytes > ByteCount) ||	(NoOfBytes == 0))				/* sanity check */		
          return RC_BAD_ARGUMENT_IN_COMMAND;

    movedata(card_address_selector,									/* retreive the received data from the card */
             Address + 0x11 + StartByte,
             _my_ds(),
             (int)buffer,
             NoOfBytes);

    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// NET COMMAND 7
// This has been split up into its individual procedures
//
// GetMachineType
// SetMachineType
// GetStationNumber
// SetStationNumber
// GetTxPauseValue
// SetTxPauseValue
// EEPROM (Read/Write non-volatile byte)
*/

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// GetMachineType
// 
// return code is from the standard set
*/
unsigned short GetMachineType(byte *machine)
{
    *machine = _farpeekb(card_address_selector, MACHINE_TYPE);	/* get the machine type from the card */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SetMachineType
// 
// return code is from the standard set
*/
unsigned short SetMachineType(byte *machine)
{

    byte test;

	if (*machine > MC_LAST_MACHINE )							/* Machine types 00 to 0x0D are valid */
       return RC_BAD_ARGUMENT_IN_COMMAND;
	
    _farpokeb(card_address_selector, MACHINE_TYPE, *machine);	/* write the type code to the board */

    test = _farpeekb(card_address_selector, MACHINE_TYPE);		/* check to see that it has been sucessfully written */
    if (test != *machine)
       return RC_RETRYABLE_BOARD_ERROR;

    *machine = test;											/* return the machine type read back from the card */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// GetStationNumber
// Stored in byte 0 of the EEPROM
// 
// return code is from the standard set
*/
unsigned short GetStationNumber(byte *station)
{
    EEPROM(EEPROM_READ, EEPROM_STATION, station);				/* get the station number from the card */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SetStationNumber
// Stored in byte 0 of the EEPROM
// 
// return code is from the standard set
*/
unsigned short SetStationNumber(byte *station)
{
	if ((*station == 00) || (*station==0xFF))					/* Station numbers 00 and 0xFF are invalid */
       return RC_BAD_ARGUMENT_IN_COMMAND;

    EEPROM(EEPROM_WRITE, EEPROM_STATION, station);				/* write the station number to the card */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// GetTxPauseValue
// This is measured in units of 5ms
// 
// return code is from the standard set
*/
unsigned short GetTxPauseValue(unsigned short *TXPause)
{
    *TXPause = _farpeekb(card_address_selector, TX_PAUSE_VALUE);	/* get the pause value from the card */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SetTxPauseValue
// This is measured in units of 5ms
// The initial value is 20
// 
// return code is from the standard set
*/
unsigned short SetTxPauseValue(unsigned short *TXPause)
{
    byte test;

    _farpokeb(card_address_selector, TX_PAUSE_VALUE, *TXPause);			/* set the TxPause value */
	_farpokeb(card_address_selector, TIMER_VALUE, (*TXPause / 0x0B));	/* calculate and set the new timer value */

	test = _farpeekb(card_address_selector, TX_PAUSE_VALUE);			/* check TxPause has been written sucessfully */
    if (test != *TXPause)
       return RC_RETRYABLE_BOARD_ERROR;

    *TXPause = test;													/* return the value read back from the card */
    return RC_COMMAND_COMPLETED_OK;

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// EEPROM
// This replaces Get/Set non volatile byte.
// There are a total of 0x7F locations.
// commands 00 = do nothing
//          01 = read  EPROM
//          02 = write EPROM
//
// return codes
//          EEPROM_SUCCESS				0x00
//          EPROM_READ_WRITE_ERROR		0xFE
//          EEPROM_INVALID_COMMAND      0xFF
*/
byte EEPROM(byte command, byte address, byte *data)
{
    byte temp;

    if (address > NON_VOLATILE_BYTES)								/* check the range of the byte to be written */
       return RC_BAD_ARGUMENT_IN_COMMAND;

    asm ("sti");
    _farpokeb(card_address_selector, EEPROM_ADDRESS, address);		/* address to the card */
    _farpokeb(card_address_selector, EEPROM_DATA, *data);			/* data to the card */
    _farpokeb(card_address_selector, EEPROM_COMMAND, command);		/* send command to the card */
	
	/* read back the command byte until it is 0 indicating the operation has completed */ 
	do
    {
        temp = _farpeekb(card_address_selector, EEPROM_COMMAND);
    }
    while (temp !=0);

    *data =_farpeekb(card_address_selector, EEPROM_DATA);			/* return the data as in the card */

    return _farpeekb(card_address_selector, EEPROM_RC);				/* return the operations return code */
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 3
// Add event handler
//
// The function being passed must be passed with it's end_of_function marker so that it can be locked.
//
// A maximum of MAX_EVENT_HANDLERS event handlers are allowed
// return code is from the standard set
*/
unsigned short AddEventHandler(long function, long end_of_function)
{
    _go32_dpmi_lock_code(function, (long)end_of_function - (long)function);

    if (NoOfEventHandlers == MAX_EVENT_HANDLERS)				/* sanity check */
       return RC_COMMAND_ABORTED;

    EventHandlers[NoOfEventHandlers++] = function;				/* add to the array of functions and increase the number
	                                                               of entries
	                                                            */
    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 4
// Delete event handler
//
// 
// return code is from the standard set
*/
unsigned short DeleteEventHandler(long function)
{
    int pos;
    pos=0;

    /* Find the position of the function in the Event Handler array */
    while ((EventHandlers[pos] != function) && (pos <= NoOfEventHandlers))
    {
      pos++;
    }

    /* If the number of functions in the array has been exceeded then the function has not ben found */
    if (pos > NoOfEventHandlers)				/* Not found */
       return RC_BAD_ARGUMENT_IN_COMMAND;

    if (pos == NoOfEventHandlers)				/* It was the last in the list */
	{
		EventHandlers[pos] = 0;
    }
	else
    {
        /* move all the functions down 1 place in the array */
        while (pos < NoOfEventHandlers)
        {
            EventHandlers[pos] = EventHandlers[pos+1];
            pos++;
        }
    }

    NoOfEventHandlers --;					/* decrease the number of entries */
    return RC_COMMAND_COMPLETED_OK;

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 5
// Set default event action
//
// NewAction = 0    Disabled
// NewAction = 1    Enabled
// NewAction = 2    Stored (Default)
//
// If events are disabled then all incoming events are ignored.
// If enabled then an incomind event calls the event handler chain.
// If stored then an incoming event is stored until cleared or enabled.
//
// It is recommended that this command should never be used and that events should be
// enabled individually using command 6.
// 
// return code is from the standard set
*/
unsigned short SetDefaultEventAction(byte NewAction)
{

    if (NewAction > STORED)					/* Sanity check */
    {
        return RC_BAD_ARGUMENT_IN_COMMAND;
    }
    else
    {
        DefaultEventAction = NewAction;
        return RC_COMMAND_COMPLETED_OK;
    }

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 6
// Set individual event action
//
//
// return code is from the standard set
*/
unsigned short SetIndividualAction(byte CBH, byte NewAction)
{
    int Address;

    if (validHandle(CBH)==FALSE)								    /* Check to see if the handle is valid */
       return RC_BAD_HANDLE;

    Address = GetAddressOfBuffer(CBH);								/* get the address of the handle's memory block */

    switch (NewAction) {
    case DISABLE:													/* remove from both enabled and stored */ 
         RemoveHandleFromList(ENABLED_LIST, CBH, 0);                   
         RemoveHandleFromList( STORED_LIST, CBH, 0);
         break;
    case ENABLE:
         RemoveHandleFromList( STORED_LIST, CBH, 0);				/* remove from stored */
         AddHandleToList(     ENABLED_LIST, CBH);					/* add to enabled */
         break;	
    case STORED:
         RemoveHandleFromList(ENABLED_LIST, CBH, 0);				/* remove from enabled */
         AddHandleToList(      STORED_LIST,CBH);					/* add to stored */
         break;
    default:
         return RC_BAD_ARGUMENT_IN_COMMAND;
    }

    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 7
// Test / Clear event status
//
// If event number = 0xFF then the first event in the event queue will be returned.
// Action = 0  the event will be tested
// Action = 1  the event will be tested and cleared
//
// On exit EventNumber contains the event number if it is pending, otherwise 0
//
// return code is from the standard set
*/
unsigned short TestClearEventStatus(byte *CBH, byte Action)
{
    int Address;

    if (*CBH == GET_FIRST_EVENT)									/* if wildcard */
       *CBH = GetFirstCB(STORED_LIST);								/* then get the first CBH from the stored list */

    if (validHandle(*CBH)==FALSE)								    /* Check to see if the handle is valid */
       return RC_BAD_HANDLE;

    Address = GetAddressOfBuffer(*CBH);								/* get the address of the handle's memory block */

    switch (Action) {
    case TEST_EVENT:
         break;
    case TEST_AND_CLEAR_EVENT:
         RemoveHandleFromList(STORED_LIST, *CBH, 0);				/* remove the event from the stored list */
         *CBH = 0;
         break;
    default:
         return RC_BAD_ARGUMENT_IN_COMMAND;
         break;
    }

    if ((_farpeekb(card_address_selector, Address + 0x07)) == RxCB_TIMED_OUT)	/* test the events status */
         return RC_TIMEOUT_ERROR;

    return RC_COMMAND_COMPLETED_OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// COMMAND 8
// Enable / Disable Event scheduler
//
// ENABLE_EVENT_SCHEDULER
// the scheduler is enabled and all events happen normally
//
// DISABLE_EVENT_SCHEDULER
// the scheduler is disabled and all events are queued until re-enabled.
//
// NOTE: A queued event is not killed but will be sent when the scheduler is re-enabled
//
// On exit the previous state is returned.
*/
unsigned short EventScheduler(byte status)
{
    byte old_state;

    if ((status != ENABLE_EVENT_SCHEDULER) || (status != DISABLE_EVENT_SCHEDULER))
		return RC_BAD_ARGUMENT_IN_COMMAND;

    old_state = EventSchedulerStatus;             /* save the current state    */
    EventSchedulerStatus = status;                /* set the new state         */
    return old_state;                             /* return the previous state */
    
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Function to display the relevant error message when an error code is returned
*/
void ReturnCodeError(unsigned short error_code)
{

	if (error_code >= RC_BAD_COMMAND_IN_AL)
	{
		switch (error_code)
		{
		case RC_BAD_COMMAND_IN_AL:
			printf("Error: Bad command in AL.\n");
		    break;
		case RC_TIMEOUT_ERROR:
			printf("Error: Timeout.\n");
			break;
		default:
			printf("Error: Other error %d\n",(error_code & 0xFF));
		}
	return;
	}

	if (error_code >= RC_FATAL_BOARD_ERROR)
	{
		printf("Error: Fatal board error %d\n",(error_code & 0xFF));
		return;
	}
    
	if (error_code >= RC_RETRYABLE_BOARD_ERROR)
	{
		printf("Error: Retryable board error %d\n",(error_code & 0xFF));
		return;
	}
    
	switch (error_code)
	{
	case RC_COMMAND_COMPLETED_OK:
		printf("Info: Command completed OK.\n");
	    break;
	case RC_BAD_COMMAND_IN_CONTROL_BLOCK:
		printf("Error: Bad command in control block.\n");
		break;
	case RC_NO_HANDLES_LEFT:
		printf("Error: No handles left.\n");
		break;
	case RC_BAD_HANDLE:
		printf("Error: Bad handle.\n");
		break;
	case RC_BAD_ARGUMENT_IN_COMMAND:
		printf("Error: Bad argument in command.\n");
		break;
	case RC_DATA_SIZE_LARGER_THAN_BUFFER:
		printf("Error: Data size larger than buffer.\n");
		break;
 	case RC_KILL_CONDITION_FAILED:
		printf("Error: Kill condition failed.\n");
		break;
	case RC_CONTROL_BLOCK_IN_USE:
		printf("Error: Control block in use.\n");
		break;
	case RC_COMMAND_ABORTED:
		printf("Error: Command aborted.\n");
		break;
    case RC_HANDLE_NOT_IN_USE:
		printf("Error : Handle not in use.\n");
		break;
    case RC_NO_MATCH_FOR_WILDCARD:
		printf("Error : No match found for wilcard poll.\n");
		break;
	default:
		printf("Error: Unknown error %d\n",error_code);
	}

	return;
}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// EnterCriticalSection
*/
void EnterCriticalSection(byte SectionID)
{
    byte x;

    do
    {

        _farpokeb(card_address_selector, LOC_0x0001, SectionID );      /* write SectionID to card */
        SetLatch(_6512_IRQ, LATCH_ON);                                 /* Generate IRQ to 65C12 */

        do
		{
            x = _farpeekb(card_address_selector, LOC_0x0001);
        }
        while (x != 0);                                                /* wait until the location is 00 */

        x = _farpeekb(card_address_selector, LOC_0x0001 + SectionID);
    }
    while (x != 1);

}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// LeaveCriticalSection
*/
void LeaveCriticalSection(byte SectionID)
{
    byte x;

    _farpokeb(card_address_selector, LOC_0x0001, SectionID | 0x80 );      /* write SectionID with top bit set to card */
    SetLatch(_6512_IRQ, LATCH_ON);                                        /* Generate IRQ to 65C12 */

    do
	{
        x = _farpeekb(card_address_selector, LOC_0x0001);
    }
    while (x != 0);                                                /* wait until the location is 00 */
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Get Handle
// retrive a free handle from the list
*/
byte GetHandle(unsigned long ListOffset)
{
    byte index;
    byte Handle;
    byte count;
    byte dummy;

    index = _farpeekb(card_address_selector, ListOffset);   /* Index = ?ListOffset  */

    if (index == 0)             /* are there any handles available */
       return (0);              /* no */

    _farpokeb(card_address_selector, ListOffset, index-1 );          /* decrease the amount of handles still available */
    Handle = _farpeekb(card_address_selector, ListOffset+1);         /* get the first handle */

    count = 1;
    while (count<index)
    {
         dummy = _farpeekb(card_address_selector, ListOffset+count+1);   /* move the remaing handles */
         _farpokeb(card_address_selector, ListOffset+count, dummy );      /* up in the list */
         count ++;
    }
    _farpokeb(card_address_selector, ListOffset+count, 0);

    return Handle;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// AddHandleToList
// add a handle to the end of a list
*/
void AddHandleToList(unsigned long ListOffset, byte Handle)
{
    byte index;

    index = _farpeekb(card_address_selector, ListOffset);          /* get the Index */
    index++;                                                       /* increment the index  */
    _farpokeb(card_address_selector, ListOffset, index );          /* write the new index  */
    _farpokeb(card_address_selector, ListOffset+index, Handle);    /* write the handle to the list */

}

void end_AddHandleToList() {/* dummy function to calculate length of previous function */}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// RemoveHandleFromList
// remove a handle from a list
*/
void RemoveHandleFromList(unsigned long ListOffset, byte Handle, byte pos)
{
    byte index;
    byte nextCBH;
    int i;

    index = _farpeekb(card_address_selector, ListOffset);
    if (index==0)			/* if there are no handles in the list return */
       return;

    /* if pos = 0 then find the position in the list */
    if (pos==0)
        pos = FindBufferInList(ListOffset, Handle);

    if (pos == 0)       /* not found */
       return;

    /* set the new index -1 */
    _farpokeb(card_address_selector, ListOffset, index-1);

    /* shuffle the list from the position of the CBH to be removed */
    for (i=pos; i<=index; i++)
    {
        nextCBH = _farpeekb(card_address_selector, ListOffset+i+1);
        _farpokeb(card_address_selector, ListOffset + i, nextCBH);
    }

    return;
}
void end_RemoveHandleFromList() {/* dummy function to calculate length of previous function */}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// GetAddressOfBuffer
// get the address of the control block handle's buffer
*/
inline int GetAddressOfBuffer(byte Handle)
{
    return (0xB0 + ((Handle - 1) * 0x0511));
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// GetFirstCB
// get the first handle in a list
*/
byte GetFirstCB(unsigned long ListOffset)
{
    byte index;

    index = _farpeekb(card_address_selector, ListOffset);		/* get the index */
    if (index==0)												/* are there entries in the list ? */
       return 0;

    return _farpeekb(card_address_selector, ListOffset + 1);	/* return the first one */

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// validHandle
*/
bool validHandle(byte CBH)
{
    if ((CBH >= 1) & (CBH<=22))
       return TRUE;
    else
       return FALSE;

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// FindBufferInList
*/
byte FindBufferInList(unsigned long ListOffset, byte CBH)
{
    byte index;
    int i;

    index = _farpeekb(card_address_selector, ListOffset);   /* retrieve the index of the list  */

    if (index == 0)             /* are there any handles available */
       return (0);              /* no */

    for (i=1; i<=index; i++)
    {
        if (_farpeekb(card_address_selector,   ListOffset + i) == CBH)
           return i;
    }

	return 0;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// ScanListForStatus
*/
bool ScanListForStatus(unsigned long ListOffset, byte pos, byte value)
{

    byte index;
    byte dummy;
    bool Found;
    int count;
    int Address;

    index = _farpeekb(card_address_selector, ListOffset);   /* retrieve the index of the list  */

    if (index == 0)                                         /* are there any handles available */
       return (FALSE);                                      /* no */

    count = 1;
    Found = FALSE;
    while ((count<=index) AND (Found == FALSE))
    {
         dummy = _farpeekb(card_address_selector, ListOffset+count);   /* move the remaing handles */
         Address = GetAddressOfBuffer(dummy);
         dummy = _farpeekb(card_address_selector, Address+ pos);   /* get the byte needed */
         if (dummy == value)                                       /* byte match */
             Found = TRUE;

         count ++;
    }

    return dummy;

}

