/*
This file is part of VoidHawk.

VoidHawk is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

VoidHawk is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with VoidHawk.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cpcrslib.h"
#include <cpcwyzlib.h>

#include "sprites/sprites.h"

#include "utils/cpc.h"
#include "utils/utils.h"
#include "utils/vector2d.h"

#include "music/dataSonido.h"

#include "sprites/gameover.h"
#include "objects/puntos.h"

#include "objects/star.h"

#include "objects/nave.h"
#include "objects/rayo.h"
#include "objects/kamikaze.h"
#include "objects/carrier.h"
#include "objects/sweeper.h"

#define TECLA_IZQ       0
#define TECLA_DER       1
#define TECLA_ARR       2
#define TECLA_ABA       3
#define TECLA_DIS       4
#define TECLA_CON       5

#define MAX_KAMIKAZES   4
#define MAX_SWEEPERS    3

int DRONE_POSICIONES[4][4];

Nave nave;
Rayo rayo;
Kamikaze kamikazes[MAX_KAMIKAZES];
Carrier carrier;
Sweeper sweepers[MAX_SWEEPERS];


#define STARS_NUM 10

_tStar aStars[STARS_NUM];
_tStar *pStar = NULL;


// Funciones de sonido:
void setEfectosSonido() {

    cpc_WyzInitPlayer(0, 0, EFFECT_TABLE, 0);       //initialize data
    cpc_WyzSetPlayerOn();
}

void ReproducirEfectoDisparo() {

    cpc_WyzStartEffect(0,1);
}

void ReproducirEfectoExplosion() {

    cpc_WyzStartEffect(0,0);
}

void InicializarNave()
{
    nave.vida = NAVE_VIDA;
    nave.pintar = 0;
    nave.posicion.x = 0;
    nave.posicion.y = 9800;
	nave.velocidad.x = 0;
	while(nave.posicion.x < 1200)
	{
		nave.velocidad.x += 1;
        nave.velocidad.x = clamp(nave.velocidad.x, 0, 10);
		nave.posicion.x += nave.velocidad.x;
		PintarNave();
		nave.pintar = 1;
	}
    nave.oldPos.x = nave.posicion.x;
    nave.oldPos.y = nave.posicion.y;
    nave.velocidad.x = 0;
    nave.velocidad.y = 0;
}

void InicializarKamikazes()
{
    int i;
    for (i = 0; i < MAX_KAMIKAZES; i++)
    {
        kamikazes[i].vida = KAMIKAZE_VIDA;
        kamikazes[i].pintar = 1;
        kamikazes[i].posicion.x = 7500;
        kamikazes[i].posicion.y = 5000+ (i * 2500);
        kamikazes[i].oldPos.x = kamikazes[i].posicion.x;
        kamikazes[i].oldPos.y = kamikazes[i].posicion.y;
        kamikazes[i].velocidad.x = 0;
        kamikazes[i].velocidad.y = 0;
    }
}

void InicializarSweepers()
{
    int i, j;
    for (i = 0; i < MAX_SWEEPERS; i++)
    {
        sweepers[i].vida = SWEEPER_VIDA;
        sweepers[i].pintar = 0;
        sweepers[i].posicion.x = SWEEPER_MIN_X;
        sweepers[i].posicion.y = SWEEPER_MIN_Y;
        sweepers[i].oldPos.x = SWEEPER_MIN_X;
        sweepers[i].oldPos.y = SWEEPER_MIN_Y;
        sweepers[i].velocidad.x = 0;
        sweepers[i].velocidad.y = 0;

        sweepers[i].flanger.vivo = 0;
        sweepers[i].flanger.disparado = 0;
        sweepers[i].flanger.pintar = 0;
        sweepers[i].flanger.posicion.x = SWEEPER_MIN_X + rand() % SWEEPER_MAX_X;
        sweepers[i].flanger.posicion.y = SWEEPER_MIN_Y + rand() % SWEEPER_MAX_Y;
        sweepers[i].flanger.oldPos.x = sweepers[i].flanger.posicion.x;
        sweepers[i].flanger.oldPos.y = sweepers[i].flanger.posicion.y;

        sweepers[i].actual = 0;
        for (j = 0; j < SWEEPER_MAX_CAMINO; j++)
        {
            sweepers[i].camino[j].x = SWEEPER_MIN_X + rand() % SWEEPER_MAX_X;
            sweepers[i].camino[j].y = SWEEPER_MIN_Y + rand() % SWEEPER_MAX_Y;
        }
    }
}

void InicializarDrones()
{
    int i;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
    {
        carrier.drones[i].estado = 0;
        carrier.drones[i].vida = DRONE_VIDA;
        carrier.drones[i].pintar = 0;
        carrier.drones[i].objetivo.x = 0;
        carrier.drones[i].objetivo.y = 0;
        carrier.drones[i].posicion.x = carrier.posicion.x - 500;
        carrier.drones[i].posicion.y = carrier.posicion.y + DRONE_POSICIONES[3][i];
        carrier.drones[i].oldPos.x = carrier.drones[i].posicion.x;
        carrier.drones[i].oldPos.y = carrier.drones[i].posicion.y;
        carrier.drones[i].velocidad.x = 0;
        carrier.drones[i].velocidad.y = 0;

        carrier.drones[i].rayo.vivo = 0;
        carrier.drones[i].rayo.pintar = 0;
        carrier.drones[i].rayo.posicion.x = 0;
        carrier.drones[i].rayo.posicion.y = 0;
        carrier.drones[i].rayo.oldPos.x = 0;
        carrier.drones[i].rayo.oldPos.y = 0;
        carrier.drones[i].rayo.velocidad.x = 0;
        carrier.drones[i].rayo.velocidad.y = 0;
    }

    carrier.dronesVivos = CARRIER_MAX_DRONES;
}

void InicializarCarrier()
{
    carrier.estado = 0;
    carrier.direccion = 1;
    carrier.escudo = 0;
    carrier.vida = CARRIER_VIDA;
    carrier.pintar = 0;
    carrier.posicion.x = CARRIER_POS_X;
    carrier.posicion.y = CARRIER_POS_Y;
    carrier.oldPos.x = CARRIER_POS_X;
    carrier.oldPos.y = CARRIER_POS_Y;
    carrier.velocidad.x = 0;
    carrier.velocidad.y = 1;

    carrier.phaser.vivo = 0;
    carrier.phaser.pintar = 0;
    carrier.phaser.posicion.x = 0;
    carrier.phaser.posicion.y = 0;
    carrier.phaser.oldPos.x = 0;
    carrier.phaser.oldPos.y = 0;
    carrier.phaser.velocidad.x = 0;
    carrier.phaser.velocidad.y = 0;

    InicializarDrones();
}

void BorrarKamikaze(int i)
{
    cpc_PutSp(negro20x10, KAMIKAZE_ALTO, KAMIKAZE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(kamikazes[i].posicion.x), MetrosToPixels(kamikazes[i].posicion.y)));
    kamikazes[i].vida = -1;
    kamikazes[i].pintar = 0;
    kamikazes[i].posicion.x = 0;
    kamikazes[i].posicion.y = 0;
}

void BorrarCarrier()
{
    cpc_PutSp(negro20x10, CARRIER_ALTO, CARRIER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(carrier.posicion.x), MetrosToPixels(carrier.posicion.y)));
    carrier.vida = -1;
    carrier.pintar = 0;
    carrier.posicion.x = 0;
    carrier.posicion.y = 0;

    if (carrier.phaser.vivo == 1) {
        cpc_PutSp(negro20x10, PHASER_ALTO, PHASER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.phaser.posicion.x), MetrosToPixels(carrier.phaser.posicion.y)));
    }
}

void BorrarSweeper(int i)
{
    cpc_PutSp(negro20x10, SWEEPER_ALTO, SWEEPER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(sweepers[i].posicion.x), MetrosToPixels(sweepers[i].posicion.y)));
    sweepers[i].vida = -1;
    sweepers[i].pintar = 0;
    sweepers[i].posicion.x = 0;
    sweepers[i].posicion.y = 0;

    if (sweepers[i].flanger.vivo == 1) {
        cpc_PutSp(negro20x10, FLANGER_ALTO, FLANGER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(sweepers[i].flanger.posicion.x), MetrosToPixels(sweepers[i].flanger.posicion.y)));
    }
}

void CambiarDeFase()
{
    int nuevaFase = rand() % 10;
    nave.muertos = 0;
    if (nuevaFase >= 0 && nuevaFase <= 3)
    {
        nave.fase = 0;
        InicializarKamikazes();
    }
    else if (nuevaFase >= 4 && nuevaFase <= 6)
    {
        nave.fase = 1;
        InicializarSweepers();
    }
    else
    {
        nave.fase = 2;
        InicializarCarrier();
    }
}

void CambiarBordeVida()
{
    if(nave.vida >= 21 && nave.vida <= 30)
        cpc_SetBorder(9);
    else if(nave.vida >= 11 && nave.vida <= 20)
        cpc_SetBorder(24);
    else
        cpc_SetBorder(3);
}

void ActuarNave()
{
    Vector2D_copy(&nave.oldPos, &nave.posicion);

    if (nave.velocidad.x > 0)
    {
        nave.velocidad.x -= NAVE_ROZAMIENTO_X;
        nave.velocidad.x = clamp(nave.velocidad.x, 0, NAVE_VEL_MAX);
    }
    else if (nave.velocidad.x < 0)
    {
        nave.velocidad.x += NAVE_ROZAMIENTO_X;
        nave.velocidad.x = clamp(nave.velocidad.x, -NAVE_VEL_MAX, 0);
    }

    if (nave.velocidad.y > 0)
    {
        nave.velocidad.y -= NAVE_ROZAMIENTO_Y;
        nave.velocidad.y = clamp(nave.velocidad.y, 0, NAVE_VEL_MAX_Y);
    }
    else if (nave.velocidad.y < 0)
    {
        nave.velocidad.y += NAVE_ROZAMIENTO_Y;
        nave.velocidad.y = clamp(nave.velocidad.y, -NAVE_VEL_MAX_Y, 0);
    }

    Vector2D_add(&nave.posicion, &nave.posicion, &nave.velocidad);
    Vector2D_clamp(&nave.posicion, 0, NAVE_MAX_X, 0, NAVE_MAX_Y);

    nave.pintar = 1;
}

void PintarNave()
{
    if (nave.vida > 0 && nave.pintar == 1)
    {
		cpc_PutSp(negro20x10, NAVE_ALTO, NAVE_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(nave.oldPos.x), MetrosToPixels(nave.oldPos.y)));

		cpc_PutSp(naveSP, NAVE_ALTO, NAVE_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(nave.posicion.x), MetrosToPixels(nave.posicion.y)));

        nave.pintar = 0;
    }
}

void InicializarRayo()
{
    rayo.vivo = 0;
    rayo.pintar = 0;
    rayo.posicion.x = 0;
    rayo.posicion.y = 0;
    rayo.oldPos.x = 0;
    rayo.oldPos.y = 0;
    rayo.velocidad.x = 0;
    rayo.velocidad.y = 0;
}

// Comprueba la colisión del rayo con los enemigos
// Solo una colisión a la vez
void ComprobarColisiones()
{
    char c = 0;
    int i;
    int muertos = 0;

    // Colision Rayo-Kamikaze
    for (i = 0; i < MAX_KAMIKAZES && c == 0; i++)
    {
        if (Colision(&rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO,
                     &kamikazes[i].posicion, KAMIKAZE_ANCHO_MUNDO, KAMIKAZE_ALTO_MUNDO) == 1 &&
            kamikazes[i].vida > 0)
        {
            c = 1;
            kamikazes[i].vida -= RAYO_DANYO;
            ReproducirEfectoExplosion();
            if (kamikazes[i].vida <= 0)
            {
                nave.puntos += KAMIKAZE_VIDA;
                kamikazes[i].pintar = 0;
                BorrarKamikaze(i);
                nave.muertos++;
            }
        }
    }

    // Colision Rayo-Carrier
    if (Colision(&rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO,
                 &carrier.posicion, CARRIER_ANCHO_MUNDO, CARRIER_ALTO_MUNDO) == 1 &&
        carrier.vida > 0)
    {
        c = 1;
        carrier.vida -= RAYO_DANYO;
        ReproducirEfectoExplosion();
        if (carrier.vida <= 0)
        {
            nave.puntos += CARRIER_VIDA + (DRONE_VIDA * CARRIER_MAX_DRONES);
            carrier.pintar = 0;
            BorrarCarrier();
            nave.muertos++;
        }
    }

    // Colision Rayo-Sweeper
    for (i = 0; i < MAX_SWEEPERS && c == 0; i++)
    {
        if (Colision(&rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO,
                     &sweepers[i].posicion, SWEEPER_ANCHO_MUNDO, SWEEPER_ALTO_MUNDO) == 1 &&
            sweepers[i].vida > 0)
        {
            c = 1;
            sweepers[i].vida -= RAYO_DANYO;
            ReproducirEfectoExplosion();
            if (sweepers[i].vida <= 0)
            {
                nave.puntos += SWEEPER_VIDA;
                sweepers[i].pintar = 0;
                BorrarSweeper(i);
                nave.muertos++;
            }
        }
    }
    if (c == 1)
        BorrarRayo();
}

void ActuarRayo()
{
    if (rayo.vivo == 1)
    {
        // Guardar la última posición, para poder borrar el rastro
        Vector2D_copy(&rayo.oldPos, &rayo.posicion);

        if (rayo.posicion.x < RAYO_MAX_X)
        {
            ComprobarColisiones();

            rayo.velocidad.x += RAYO_ACELERACION;
            rayo.velocidad.x = clamp(rayo.velocidad.x, 0, RAYO_VEL_MAX);

            // Variar nuestra posición en función de la velocidad
            Vector2D_add(&rayo.posicion, &rayo.posicion, &rayo.velocidad);

            // Mantener la posición en el rango permitido
            Vector2D_clamp(&rayo.posicion, 0, RAYO_MAX_X, 0, RAYO_MAX_Y);

            rayo.pintar = 1;
        }
        else
        {
            BorrarRayo();
        }
    }
}

void BorrarRayo()
{
    cpc_PutSp(negro20x10, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(rayo.oldPos.x), MetrosToPixels(rayo.oldPos.y)));
    rayo.posicion.x = 0;
    rayo.posicion.y = 0;
    rayo.vivo = 0;
    rayo.pintar = 0;
}

void PintarRayo()
{
    if (rayo.vivo > 0 && rayo.pintar == 1)
    {
        cpc_PutSp(negro20x10, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(rayo.oldPos.x), MetrosToPixels(rayo.oldPos.y)));

        cpc_PutSp(rayoSP, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(rayo.posicion.x), MetrosToPixels(rayo.posicion.y)));

        rayo.pintar = 0;
    }
}

void ActuarKamikaze(int i)
{
    Vector2D steering;

    // Guardar la última posición, para poder borrar el rastro
    Vector2D_copy(&kamikazes[i].oldPos, &kamikazes[i].posicion);

    // Colisión Kamikaze-Nave
    if (Colision(&kamikazes[i].posicion, KAMIKAZE_ANCHO_MUNDO, KAMIKAZE_ALTO_MUNDO,
                 &nave.posicion, NAVE_ANCHO_MUNDO, NAVE_ALTO_MUNDO) == 1)
    {
        nave.vida -= KAMIKAZE_DANYO;
        CambiarBordeVida();
        BorrarKamikaze(i);
        nave.muertos++;
    }

    // Calcular la fuerza del steering
    pursuit(&steering, &kamikazes[i].posicion, &kamikazes[i].velocidad,
            &nave.posicion, &nave.velocidad,
            KAMIKAZE_VEL_MAX, KAMIKAZE_PURSUIT_FORCE, KAMIKAZE_PURSUIT_T);

    // Añadir a la velocidad, la fuerza del steering
    Vector2D_add(&kamikazes[i].velocidad, &kamikazes[i].velocidad, &steering);

    // Truncar la nueva velocidad para que no se exceda
    //Vector2D_truncate(&kamikazes[i].velocidad, KAMIKAZE_VEL_MAX);

    // Variar nuestra posición en función de la velocidad
    Vector2D_add(&kamikazes[i].posicion, &kamikazes[i].posicion, &kamikazes[i].velocidad);

    // Mantener la posición en el rango permitido
    Vector2D_clamp(&sweepers[i].posicion, KAMIKAZE_MIN_X, KAMIKAZE_MAX_X, KAMIKAZE_MIN_Y, KAMIKAZE_MAX_Y);

    kamikazes[i].pintar = 1;
}

void ActuarKamikazes()
{
    int i;
    for (i = 0; i < MAX_KAMIKAZES; i++)
    {
        if (kamikazes[i].vida > 0)
        {
            ActuarKamikaze(i);
        }
    }
}

void PintarKamikaze(int i)
{
    cpc_PutSp(negro20x10, KAMIKAZE_ALTO, KAMIKAZE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(kamikazes[i].oldPos.x), MetrosToPixels(kamikazes[i].oldPos.y)));

    cpc_PutSp(kamikazeSP, KAMIKAZE_ALTO, KAMIKAZE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(kamikazes[i].posicion.x), MetrosToPixels(kamikazes[i].posicion.y)));

    kamikazes[i].pintar = 0;
}

void PintarKamikazes()
{
    int i;
    for (i = 0; i < MAX_KAMIKAZES; i++)
    {
        if (kamikazes[i].pintar == 1 && kamikazes[i].vida > 0)
        {
            PintarKamikaze(i);
        }
    }
}

void BorrarPhaser()
{
    cpc_PutSp(negro20x10, PHASER_ALTO, PHASER_ANCHO, cpc_GetScrAddress(
          MetrosToPixels(carrier.phaser.posicion.x), MetrosToPixels(carrier.phaser.posicion.y)));
    carrier.phaser.posicion.x = 0;
    carrier.phaser.posicion.y = 0;
    carrier.phaser.vivo = 0;
    carrier.phaser.pintar = 0;
}

void ActuarPhaser()
{
    if (carrier.phaser.vivo == 1)
    {
        if (carrier.phaser.posicion.x > 0)
        {
            // Colision Phaser-Nave
            if (Colision(&carrier.phaser.posicion, PHASER_ANCHO_MUNDO, PHASER_ALTO_MUNDO,
                         &nave.posicion, NAVE_ANCHO_MUNDO, NAVE_ALTO_MUNDO) == 1)
            {
                ReproducirEfectoExplosion();

                nave.vida -= PHASER_DANYO;
                //CambiarBordeVida();
                BorrarPhaser();
            }
            // Colision Phaser-Rayo
            else if (Colision(&carrier.phaser.posicion, PHASER_ANCHO_MUNDO, PHASER_ALTO_MUNDO,
                              &rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO) == 1)
            {
                BorrarRayo();
            }

            // Guardar la última posición, para poder borrar el rastro
            Vector2D_copy(&carrier.phaser.oldPos, &carrier.phaser.posicion);

            carrier.phaser.velocidad.x -= PHASER_ACELERACION;
            if (carrier.dronesVivos == 4)
                carrier.phaser.velocidad.x = clamp(carrier.phaser.velocidad.x, -PHASER_VEL_MAX, 0);
            else
                carrier.phaser.velocidad.x = clamp(carrier.phaser.velocidad.x, -PHASER_VEL_MAX_FINAL, 0);

            // Variar nuestra posición en función de la velocidad
            carrier.phaser.posicion.x += carrier.phaser.velocidad.x;

            // Mantener la posición en el rango permitido
            carrier.phaser.posicion.x = clamp(carrier.phaser.posicion.x , 0, PHASER_MAX_X);

            carrier.phaser.pintar = 1;
        }
        else
        {
            BorrarPhaser();
            if (carrier.dronesVivos == 4)
                carrier.estado = 3;
            else
                carrier.estado = 0;
        }
    }
}

void ReordenarDrones()
{
    int i;
    int j = 0;

    carrier.velocidad.y = 0;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
	{
		if (carrier.drones[i].vida > 0)
		{
			carrier.drones[i].objetivo.x = carrier.drones[i].posicion.x;
			carrier.drones[i].objetivo.y = carrier.posicion.y + DRONE_POSICIONES[carrier.dronesVivos - 1][j];
			carrier.drones[i].estado = 1;
            carrier.drones[i].velocidad.x = 0;
            carrier.drones[i].velocidad.y = 0;
			j++;
		}
	}
}

void BorrarRayoDron(int i)
{
    cpc_PutSp(negro20x10, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(carrier.drones[i].rayo.oldPos.x), MetrosToPixels(carrier.drones[i].rayo.oldPos.y)));
    carrier.drones[i].rayo.posicion.x = 0;
    carrier.drones[i].rayo.posicion.y = 0;
    carrier.drones[i].rayo.vivo = 0;
    carrier.drones[i].rayo.pintar = 0;
}

void BorrarDrone(int i)
{
    carrier.drones[i].estado = -1;
    carrier.drones[i].vida = -1;
    carrier.dronesVivos--;

    cpc_PutSp(negro20x10, DRONE_ALTO, DRONE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(carrier.drones[i].posicion.x), MetrosToPixels(carrier.drones[i].posicion.y)));

    // Borrar rayo también si el drone muere
    if (carrier.drones[i].rayo.vivo == 1)
        BorrarRayoDron(i); // Igual el rayo deberia continuar aunque el drone muera ????
}

void ActuarRayoDron(int i)
{
    if (carrier.drones[i].rayo.vivo == 1)
    {
        // Guardar la última posición, para poder borrar el rastro
        Vector2D_copy(&carrier.drones[i].rayo.oldPos, &carrier.drones[i].rayo.posicion);

        if (carrier.drones[i].rayo.posicion.x > 0)
        {
            // Colisión Rayo Dron-Nave
            if (Colision(&carrier.drones[i].rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO,
                         &nave.posicion, NAVE_ANCHO_MUNDO, NAVE_ALTO_MUNDO) == 1)
            {
                ReproducirEfectoExplosion();

                nave.vida -= DRONE_DANYO;
                CambiarBordeVida();
                BorrarRayoDron(i);
            }

            carrier.drones[i].rayo.velocidad.x -= RAYO_ACELERACION;
            carrier.drones[i].rayo.velocidad.x = clamp(carrier.drones[i].rayo.velocidad.x, -RAYO_VEL_MAX, 0);

            // Variar nuestra posición en función de la velocidad
            Vector2D_add(&carrier.drones[i].rayo.posicion, &carrier.drones[i].rayo.posicion, &carrier.drones[i].rayo.velocidad);

            // Mantener la posición en el rango permitido
            Vector2D_clamp(&rayo.posicion, 0, RAYO_MAX_X, 0, RAYO_MAX_Y);

            carrier.drones[i].rayo.pintar = 1;
        }
        else
        {
            BorrarRayoDron(i);
        }
    }
}

void ActuarDrone(int i)
{
    if (carrier.drones[i].vida > 0)
    {
        Vector2D steering;

        // Colisión Dron-Rayo
        if (Colision(&carrier.drones[i].posicion, DRONE_ANCHO_MUNDO, DRONE_ALTO_MUNDO,
                     &rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO) == 1 &&
              carrier.drones[i].vida > 0)
        {
            if (carrier.drones[i].estado == 0 && carrier.estado == 0 || carrier.drones[i].estado == 2)
                carrier.drones[i].vida -= RAYO_DANYO;
            BorrarRayo();
            ReproducirEfectoExplosion();
            if (carrier.drones[i].vida <= 0)
            {
                // Reordenar en caso de que no sea un kamikaze el que muere
                if (carrier.drones[i].estado != 2)
                {
                    BorrarDrone(i);
                    if (carrier.dronesVivos > 0)
                        ReordenarDrones();
                }
                else
                    BorrarDrone(i);
            }
        }
        // Colisión Dron-Nave (kamikaze)
        else if (Colision(&carrier.drones[i].posicion, DRONE_ANCHO_MUNDO, DRONE_ALTO_MUNDO,
                          &nave.posicion, NAVE_ANCHO_MUNDO, NAVE_ALTO_MUNDO) == 1)
        {
            ReproducirEfectoExplosion();

            nave.vida -= DRONE_DANYO;
            CambiarBordeVida();
            BorrarDrone(i);
        }

        // Guardar la última posición, para poder borrar el rastro
        Vector2D_copy(&carrier.drones[i].oldPos, &carrier.drones[i].posicion);

        if (carrier.drones[i].estado == 0)
        {
            carrier.drones[i].velocidad.y = carrier.velocidad.y;
        }
        else if (carrier.drones[i].estado == 1)
        {
            seek(&steering, &carrier.drones[i].posicion, &carrier.drones[i].velocidad,
                 &carrier.drones[i].objetivo, DRONE_VEL_MAX, DRONE_SEEK_FORCE);

            if (steering.x == 0 && steering.y == 0)
            {
                carrier.drones[i].estado = 0;
                carrier.drones[i].velocidad.x = 0;
                carrier.drones[i].velocidad.y = 0;
                carrier.drones[i].posicion.x = carrier.drones[i].objetivo.x;
                carrier.drones[i].posicion.y = carrier.drones[i].objetivo.y;
            }
            else
                // Añadir a la velocidad, la fuerza del steering
                Vector2D_add(&carrier.drones[i].velocidad, &carrier.drones[i].velocidad, &steering);
        }
        else if (carrier.drones[i].estado == 2)
        {
            // Si el dron kamikaze nos sobrepasa muere
            if (carrier.drones[i].posicion.x <= nave.posicion.x)
            {
                BorrarDrone(i);

                // Recuperar escudo y la vida si es necesario
                if (carrier.dronesVivos == 0 && carrier.escudo == 1)
                {
                    InicializarDrones();
                    carrier.escudo = 0;
                    carrier.vida = CARRIER_VIDA;
                    return;
                }
            }

            seek(&steering, &carrier.drones[i].posicion, &carrier.drones[i].velocidad,
                 &nave.posicion, DRONE_VEL_MAX_KAMIKAZE, DRONE_SEEK_FORCE);

            // Añadir a la velocidad, la fuerza del steering
            Vector2D_add(&carrier.drones[i].velocidad, &carrier.drones[i].velocidad, &steering);
        }

        Vector2D_add(&carrier.drones[i].posicion, &carrier.drones[i].posicion, &carrier.drones[i].velocidad);

        carrier.drones[i].pintar = 1;

        ActuarRayoDron(i);
    }
}

char ComprobarEstadoDrones()
{
    int i;
    int cant = 0;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
        if (carrier.drones[i].vida > 0)
            if (carrier.drones[i].estado == 0)
                cant++;
    return (cant == carrier.dronesVivos && cant > 0);
}

char ComprobarNave()
{
    int objetivo;
    int carrierY;

    if (carrier.dronesVivos > 0)
        objetivo = nave.posicion.y + (nave.velocidad.y * 8);
    else
        objetivo = nave.posicion.y + (nave.velocidad.y * 2);

    if (carrier.estado == 0 && abs(carrier.velocidad.y) > 110 &&
        carrier.posicion.y <= objetivo && carrier.posicion.y + 500 >= objetivo)
    {
        if (carrier.dronesVivos == 4 && carrier.escudo == 0)
            return 1; // Enviar drones a posición de ataque
        else if (carrier.dronesVivos <= 0)
            return 2; // Disparar Phaser
        else if (carrier.dronesVivos == 2 || carrier.dronesVivos == 3)
            return 3; // Disparar los drones cuando son 2 o 3
        else
            return 4; // Queda un dron -> dron kamikaze
    }
    return 0;
}

void PosicionAtaque()
{
    carrier.estado = 1;
    carrier.velocidad.y = 0;

    carrier.drones[1].objetivo.x = carrier.drones[0].posicion.x - 600;
	carrier.drones[1].objetivo.y = carrier.drones[0].posicion.y;
    carrier.drones[1].estado = 1;
    carrier.drones[2].objetivo.x = carrier.drones[3].posicion.x - 600;
	carrier.drones[2].objetivo.y = carrier.drones[3].posicion.y;
    carrier.drones[2].estado = 1;
}

void RecuperarPosicion()
{
    carrier.estado = 4;

    carrier.drones[1].objetivo.x = carrier.drones[0].posicion.x;
    carrier.drones[1].objetivo.y = carrier.posicion.y;
    carrier.drones[1].estado = 1;
    carrier.drones[2].objetivo.x = carrier.drones[3].posicion.x;
    carrier.drones[2].objetivo.y = carrier.posicion.y + 1200;
    carrier.drones[2].estado = 1;
}

void DispararPhaser()
{
    ReproducirEfectoExplosion();
    carrier.estado = 2;
    carrier.phaser.vivo = 1;
    carrier.phaser.posicion.x = carrier.posicion.x - 500;
    carrier.phaser.posicion.y = carrier.posicion.y + 500;
}

void DronesDisparar()
{
    int i;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
    {
        if (carrier.drones[i].vida > 0 && carrier.drones[i].rayo.vivo == 0)
        {
            ReproducirEfectoExplosion();
            carrier.drones[i].rayo.vivo = 1;
            carrier.drones[i].rayo.posicion.x = carrier.drones[i].posicion.x - 400;
			carrier.drones[i].rayo.posicion.y = carrier.drones[i].posicion.y;
        }
    }
}

void DronesKamikaze()
{
    int i;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
        if (carrier.drones[i].vida > 0)
            carrier.drones[i].estado = 2;
}

void ActuarCarrier()
{
    if (carrier.vida > 0)
    {
        int i;
        int estado;

        // Guardar la última posición, para poder borrar el rastro
        Vector2D_copy(&carrier.oldPos, &carrier.posicion);

        // Reaccionar al estado de la Nave
        estado = ComprobarNave();
        if (estado == 1)        // Enviar drones a posición de ataque
            PosicionAtaque();
        else if (estado == 2)
            DispararPhaser();   // Disparar Phaser
        else if (estado == 3)
            DronesDisparar();   // Disparar los drones cuando son 2 o 3
        else if (estado == 4)
            DronesKamikaze();   // Queda un dron -> dron kamikaze

        // Escudo kamikaze si esta algun Drone vivo y se le quita vida al Carrier
        if (carrier.escudo == 0 && carrier.vida < CARRIER_VIDA && carrier.dronesVivos > 0)
        {
            DronesKamikaze();
            carrier.escudo = 1;
            carrier.estado = 0;
            carrier.velocidad.y = carrier.direccion;
        }

        if (carrier.velocidad.y > 0)
            carrier.velocidad.y += CARRIER_ACELERACION;
        else if (carrier.velocidad.y < 0)
            carrier.velocidad.y -= CARRIER_ACELERACION;
        else // Se ha parado por reorganización de Drones
        {
            if (ComprobarEstadoDrones() == 1 && carrier.escudo == 0)
            {
                if (carrier.estado == 0)
                    carrier.velocidad.y = carrier.direccion;
                else if (carrier.estado == 1)
                    DispararPhaser();
                else if (carrier.estado == 3)
                    RecuperarPosicion();
                else if (carrier.estado == 4)
                    carrier.estado = 0;
            }
        }

        carrier.velocidad.y = clamp(carrier.velocidad.y, -CARRIER_VEL_MAX, CARRIER_VEL_MAX);

        carrier.posicion.y += carrier.velocidad.y;

        if (carrier.posicion.y >= CARRIER_MAX_Y)
        {
            carrier.direccion = -1;
            carrier.posicion.y = CARRIER_MAX_Y;
            carrier.velocidad.y = -carrier.velocidad.y;
        }
        else if (carrier.posicion.y <= CARRIER_MIN_Y)
        {
            carrier.direccion = 1;
            carrier.posicion.y = CARRIER_MIN_Y;
            carrier.velocidad.y = -carrier.velocidad.y;
        }

        carrier.pintar = 1;

        for (i = 0; i < CARRIER_MAX_DRONES; i++)
            ActuarDrone(i);

        ActuarPhaser();
    }
}

void PintarPhaser()
{
    if (carrier.phaser.pintar == 1 && carrier.phaser.vivo == 1)
    {
        cpc_PutSp(negro20x10, PHASER_ALTO, PHASER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.phaser.oldPos.x), MetrosToPixels(carrier.phaser.oldPos.y)));

        cpc_PutSp(phaserSP, PHASER_ALTO, PHASER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.phaser.posicion.x), MetrosToPixels(carrier.phaser.posicion.y)));

        carrier.phaser.pintar = 0;
    }
}

void PintarDrone(int i)
{
    cpc_PutSp(negro20x10, DRONE_ALTO, DRONE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(carrier.drones[i].oldPos.x), MetrosToPixels(carrier.drones[i].oldPos.y)));

    cpc_PutSp(droneSP, DRONE_ALTO, DRONE_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(carrier.drones[i].posicion.x), MetrosToPixels(carrier.drones[i].posicion.y)));

    carrier.drones[i].pintar = 0;
}

void PintarRayoDrone(int i)
{
    if (carrier.drones[i].rayo.vivo == 1 )
    {
        cpc_PutSp(negro20x10, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.drones[i].rayo.oldPos.x), MetrosToPixels(carrier.drones[i].rayo.oldPos.y)));

        cpc_PutSp(rayoSP, RAYO_ALTO, RAYO_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.drones[i].rayo.posicion.x), MetrosToPixels(carrier.drones[i].rayo.posicion.y)));

        carrier.drones[i].pintar = 0;
    }
}

void PintarDrones()
{
    int i;
    for (i = 0; i < CARRIER_MAX_DRONES; i++)
    {
        if (carrier.drones[i].pintar == 1 && carrier.drones[i].vida > 0)
        {
            PintarDrone(i);
            PintarRayoDrone(i);
        }
    }
}

void PintarCarrier()
{
    if (carrier.pintar == 1 && carrier.vida > 0)
    {
        cpc_PutSp(negro20x10, CARRIER_ALTO, CARRIER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.oldPos.x), MetrosToPixels(carrier.oldPos.y)));

        cpc_PutSp(carrierSP, CARRIER_ALTO, CARRIER_ANCHO, cpc_GetScrAddress(
                  MetrosToPixels(carrier.posicion.x), MetrosToPixels(carrier.posicion.y)));

        carrier.pintar = 0;

        PintarDrones();
        PintarPhaser();
    }
}

void ActuarFlanger(int i)
{
    if (sweepers[i].flanger.vivo == 1)
    {
        // Colision Flanger-Nave
        if (Colision(&sweepers[i].flanger.posicion, FLANGER_ANCHO_MUNDO, FLANGER_ALTO_MUNDO,
                     &nave.posicion, NAVE_ANCHO_MUNDO, NAVE_ALTO_MUNDO) == 1)
        {
            ReproducirEfectoExplosion();
            sweepers[i].flanger.disparado = 0;
            sweepers[i].flanger.vivo = 0;
            sweepers[i].flanger.pintar = 0;
            nave.vida -= FLANGER_DANYO;
            CambiarBordeVida();
            cpc_PutSp(negro20x10, FLANGER_ALTO, FLANGER_ANCHO, cpc_GetScrAddress(
                      MetrosToPixels(sweepers[i].flanger.oldPos.x), MetrosToPixels(sweepers[i].flanger.oldPos.y)));
        }
        // Colision Flanger-Rayo
        else if (Colision(&sweepers[i].flanger.posicion, FLANGER_ANCHO_MUNDO, FLANGER_ALTO_MUNDO,
                     &rayo.posicion, RAYO_ANCHO_MUNDO, RAYO_ALTO_MUNDO) == 1)
        {
            BorrarRayo();
        }

        if (sweepers[i].flanger.disparado == 1)
        {
            Vector2D_copy(&sweepers[i].flanger.oldPos, &sweepers[i].flanger.posicion);
            sweepers[i].flanger.posicion.x -= FLANGER_VEL_X;
            sweepers[i].flanger.pintar = 1;
        }

        if (sweepers[i].flanger.posicion.x <= 0)
        {
            sweepers[i].flanger.disparado = 0;
            sweepers[i].flanger.vivo = 0;
            sweepers[i].flanger.pintar = 0;
            cpc_PutSp(negro20x10, FLANGER_ALTO, FLANGER_ANCHO, cpc_GetScrAddress(
                      MetrosToPixels(sweepers[i].flanger.oldPos.x), MetrosToPixels(sweepers[i].flanger.oldPos.y)));
        }
    }
}

char ComprobarSweeperMasCercano(Vector2D* ahead, int i)
{
    int j;
    int mostThreatening = -1;
    int distance = 32000;
    int distAhead = 0;
    for (j = 0; j < MAX_SWEEPERS; j++)
    {
        // No nos tenemos en cuenta a nosotros mismos
        if (i != j)
        {
            // Si el punto futuro (ahead), colisiona con algun sweeper
            if (ColisionPunto(ahead, &sweepers[j].posicion, SWEEPER_ANCHO_MUNDO, SWEEPER_ALTO_MUNDO) == 1)
            {
                distAhead = Vector2d_length(&sweepers[i].posicion, &sweepers[j].posicion);
                // Comprobamos si la distancia entre los sweepers es la más pequeña
                if (distAhead < distance)
                {
                    mostThreatening = j;
                    distance = distAhead;
                }
            }
        }
    }
    return mostThreatening;
}

Vector2D avoidance;
Vector2D ahead;

void ActuarSweeper(int i)
{
    Vector2D steering;

    char mostThreatening = 0;
    avoidance.x = 0;
    avoidance.y = 0;

    ahead.x = sweepers[i].posicion.x + (sweepers[i].velocidad.x * SWEEPER_MAX_SEE_AHEAD);
    ahead.y = sweepers[i].posicion.y + (sweepers[i].velocidad.y * SWEEPER_MAX_SEE_AHEAD);

    // Si hay que evitar un sweeper cercano
    mostThreatening = ComprobarSweeperMasCercano(&ahead, i);
    if (mostThreatening != -1)
    {
        Vector2D_sub(&avoidance, &ahead, &sweepers[mostThreatening].posicion);
    }

    // Guardar la última posición, para poder borrar el rastro
    Vector2D_copy(&sweepers[i].oldPos, &sweepers[i].posicion);

    // Calcular la fuerza del steering
    seek(&steering, &sweepers[i].posicion, &sweepers[i].velocidad,
         &sweepers[i].camino[sweepers[i].actual], SWEEPER_VEL_MAX, SWEEPER_SEEK_FORCE);

    // Añadir al steering, la fuerza del avoidance (ahead)
    Vector2D_add(&steering, &steering, &avoidance);

    // Añadir a la velocidad, la fuerza del steering
	Vector2D_add(&sweepers[i].velocidad, &sweepers[i].velocidad, &steering);

    // Variar nuestra posición en función de la velocidad
    Vector2D_add(&sweepers[i].posicion, &sweepers[i].posicion, &sweepers[i].velocidad);

    // Mantener la posición en el rango permitido
    Vector2D_clamp(&sweepers[i].posicion, KAMIKAZE_MIN_X, KAMIKAZE_MAX_X, KAMIKAZE_MIN_Y, KAMIKAZE_MAX_Y);

    sweepers[i].pintar = 1;

    // Pasar al proximo camino
    if (Vector2D_near(&sweepers[i].posicion, &sweepers[i].camino[sweepers[i].actual], SWEEPER_NEAR_X, SWEEPER_NEAR_Y))
	{
        sweepers[i].camino[sweepers[i].actual].x = SWEEPER_MIN_X + rand() % SWEEPER_MAX_X;
        sweepers[i].camino[sweepers[i].actual].y = SWEEPER_MIN_Y + rand() % SWEEPER_MAX_Y;
        sweepers[i].actual++;
        sweepers[i].actual %= SWEEPER_MAX_CAMINO;
    }

    // Decidir cuando disparar a la nave
    if (sweepers[i].flanger.disparado == 0 &&
        sweepers[i].posicion.y < nave.posicion.y + 80 &&
		sweepers[i].posicion.y > nave.posicion.y - 80 &&
		nave.posicion.x < sweepers[i].posicion.x - 4000)
    {
        ReproducirEfectoExplosion();
        sweepers[i].flanger.posicion.x = sweepers[i].posicion.x - 2000;
        sweepers[i].flanger.posicion.y = sweepers[i].posicion.y;
        sweepers[i].flanger.disparado = 1;
        sweepers[i].flanger.vivo = 1;
    }

    // Empujar a la nave si esta muy cerca
    if (Vector2D_near(&nave.posicion, &sweepers[i].posicion, 3000, 20000))
    {
        //Vector2D_sub(&nave.posicion, &nave.posicion, &nave.velocidad);
        nave.posicion.x -= abs(nave.velocidad.x);
    }

    ActuarFlanger(i);
}

void ActuarSweepers()
{
    int i;
    for (i = 0; i < MAX_SWEEPERS; i++)
    {
        if (sweepers[i].vida > 0)
        {
            ActuarSweeper(i);
        }
    }
}

void PintarFlanger(int i)
{
    cpc_PutSp(negro20x10, FLANGER_ALTO, FLANGER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(sweepers[i].flanger.oldPos.x), MetrosToPixels(sweepers[i].flanger.oldPos.y)));

    cpc_PutSp(flangerSP, FLANGER_ALTO, FLANGER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(sweepers[i].flanger.posicion.x), MetrosToPixels(sweepers[i].flanger.posicion.y)));

    sweepers[i].flanger.pintar = 0;
}

void PintarSweeper(int i)
{
    cpc_PutSp(negro20x10, SWEEPER_ALTO, SWEEPER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(sweepers[i].oldPos.x), MetrosToPixels(sweepers[i].oldPos.y)));

    cpc_PutSp(sweeperSP, SWEEPER_ALTO, SWEEPER_ANCHO, cpc_GetScrAddress(
              MetrosToPixels(sweepers[i].posicion.x), MetrosToPixels(sweepers[i].posicion.y)));

    sweepers[i].pintar = 0;
}

void PintarSweepers()
{
    int i;
    for (i = 0; i < MAX_SWEEPERS; i++)
    {
        if (sweepers[i].pintar == 1 && sweepers[i].vida > 0)
        {
            PintarSweeper(i);
            if (sweepers[i].flanger.pintar == 1 && sweepers[i].flanger.vivo == 1)
            {
                PintarFlanger(i);
            }
        }
    }
}

void InicializarEstrellas()
{
    int nStar;
    memset(aStars, 0, sizeof(aStars));
    for(nStar = 0; nStar < STARS_NUM; nStar++)
    {
        aStars[nStar].nX = rand() % 160;
        aStars[nStar].nY = rand() % 200;
        aStars[nStar].nStarType = rand() % 3;
        aStars[nStar].pLineAddress = GetLineAddress(aStars[nStar].nY);
        aStars[nStar].pCurrentAddress = aStars[nStar].pLineAddress;
    }
}

void PintarEstrellas()
{
    int nStar;
    // Esto se ocupa del fondo
    for(nStar = 0; nStar < STARS_NUM; nStar++)
    {
        pStar = &aStars[nStar];
        //delete star
        *pStar->pCurrentAddress = 0;

        //move star
        switch(pStar->nStarType)
        {
            case 0: //slow star
                pStar->nX -= 1;
                break;
            case 1: //medium star
                pStar->nX -= 2;
                break;
            case 2: //fast star
                pStar->nX -= 3;
                break;
        }

        if(pStar->nX <= 0)
        {
            pStar->nX = 160;
            pStar->nY = rand() % 100;
            pStar->nStarType = rand() % 3;
            continue;
        }

        //paint star
        pStar->pCurrentAddress = pStar->pLineAddress + (pStar->nX / 2);
        *pStar->pCurrentAddress = GetMode0PixelColorByte(pStar->nStarType + 1, pStar->nX % 2);
    }
}

void EscribirHistoria()
{
    cpc_SetInkGphStr(0,0);
    cpc_SetInkGphStr(4,128);
    cpc_SetInkGphStr(3,128);
    cpc_SetInkGphStr(2,128);
    cpc_SetInkGphStr(1,128);

    // Borro la pantalla de bienvenida
    cpc_ClrScr();

    // Escribo la intro de la historia
    cpc_PrintGphStrXY("BIENVENIDO;COMANDANTE<",5,10);
    cpc_PrintGphStrXY("COMO;BIEN;SABES;LOS;MECHAS",5,32);
    cpc_PrintGphStrXY("LIDERADOS;POR;LA;SUPERMENTE;OMNIUS",5,44);
    cpc_PrintGphStrXY("HAN;SECUESTRADO;LA;NAVE;EN;LA;QUE",5,56);
    cpc_PrintGphStrXY("VIAJABA;LA;PRINCESA;ZELDA;EN;MISION",5,68);
    cpc_PrintGphStrXY("HUMANITARIA<;UN;ATAQUE;FRONTAL;CONTRA",5,80);
    cpc_PrintGphStrXY("LAS;FUERZAS;DE;ONMIUS;ES;UNA;MISION",5,92);
    cpc_PrintGphStrXY("SUICIDA;POR;LO;QUE;EL;ALTO;MANDO;HA",5,104);
    cpc_PrintGphStrXY("DECIDIDO;ENVIARTE;A;TI;Y;A;TU;NAVE",5,116);
    cpc_PrintGphStrXY("LA;VOIDHAWK;PARA;QUE;TE;INFILTRES;EN",5,128);
    cpc_PrintGphStrXY("EL;VECTOR;EN;EL;QUE;DESAPARECIO;LA",5,140);
    cpc_PrintGphStrXY("NAVE;DE;LA;PRINCESA<",5,152);
    cpc_PrintGphStrXY("PULSA;UNA;TECLA;PARA;CONTINUAR",5,172);

    while (1) // Cuando se presione una tecla continua
        if(cpc_AnyKeyPressed() == 1)
            break;
    cpc_ClrScr();
}

/*void MostrarInstrucciones()
{
    // Escribo la intro de la historia
    cpc_PrintGphStrXY("NO;HACE;FALTA;QUE;TE;LO",5,20);
    cpc_PrintGphStrXY("RECORDEMOS;PERO;POR;SI;ACASO",5,32);
    cpc_PrintGphStrXY("PARA;MODIFICAR;EL;RUMBO;DE;TU;NAVE",5,44);
    cpc_PrintGphStrXY("USA;LAS;FLECHAS;DE;DIRECCION<",5,56);
    cpc_PrintGphStrXY("PARA;DISPARAR;LOS;RAYOS;USA;EL",5,68);
    cpc_PrintGphStrXY("ESPACIO<",5,80);
    cpc_PrintGphStrXY("BUENA;SUERTE;CON;TU;MISION;COMANDANTE",5,104);
    cpc_PrintGphStrXY("NO;SABEMOS;LO;QUE;HAY;FUERA",5,116);
    cpc_PrintGphStrXY("PULSA;UNA;TECLA;PARA;EMPEZAR",5,172);

    while (1) // Cuando se presione una tecla continua
        if(cpc_AnyKeyPressed() == 1)
            break;
    cpc_ClrScr();
}*/

Puntos p1;
Puntos p2;
Puntos p3;
Puntos p4;
Puntos p5;
Puntos *pu[5];

void InicializarPuntuacion()
{
    p1.nombre = "IMP";
    p1.puntos = 3000;
    pu[0] = &p1;
    p2.nombre = "EST";
    p2.puntos = 460;
    pu[1] = &p2;
    p3.nombre = "FRA";
    p3.puntos = 320;
    pu[2] = &p3;
    p4.nombre = "DAN";
    p4.puntos = 240;
    pu[3] = &p4;
    p5.nombre = "JOA";
    p5.puntos = 170;
    pu[4] = &p5;
}

void main()
{
    int i = 0;

    char d = 0;
    char maxP = 5;
    char ignorarUltimo = 0;
    char msg[9];

    // Redefinimos las teclas
    cpc_AssignKey (TECLA_IZQ, 0x4101);		// TECLA IZQUIERDA
    cpc_AssignKey (TECLA_DER, 0x4002);		// TECLA DERECHA
    cpc_AssignKey (TECLA_ARR, 0x4001);		// TECLA ARRIBA
    cpc_AssignKey (TECLA_ABA, 0x4004);		// TECLA ABAJO
    cpc_AssignKey (TECLA_DIS, 0x4580);		// TECLA ESPACIO
    cpc_AssignKey (TECLA_CON, 0x4740);      // TECLA CONTINUAR


    SetColours();                           // Poner la paleta
    cpc_SetModo(0);                         // 16 colores
    cpc_SetBorder(0);                       // Borde

    DRONE_POSICIONES[0][0] =   600;
    DRONE_POSICIONES[0][1] =    -1;
    DRONE_POSICIONES[0][2] =    -1;
    DRONE_POSICIONES[0][3] =    -1;
    DRONE_POSICIONES[1][0] =     0;
    DRONE_POSICIONES[1][1] =  1200;
    DRONE_POSICIONES[1][2] =    -1;
    DRONE_POSICIONES[1][3] =    -1;
    DRONE_POSICIONES[2][0] =  -600;
    DRONE_POSICIONES[2][1] =   600;
    DRONE_POSICIONES[2][2] =  1800;
    DRONE_POSICIONES[2][3] =    -1;
    DRONE_POSICIONES[3][0] = -1200;
    DRONE_POSICIONES[3][1] =     0;
    DRONE_POSICIONES[3][2] =  1200;
    DRONE_POSICIONES[3][3] =  2400;

    InicializarPuntuacion();

    // Inicializar efectos de sonido:
    setEfectosSonido();

    EscribirHistoria();
    //MostrarInstrucciones();

    while (1)
    {
        nave.puntos = 0;
        nave.muertos = 0;
        nave.fase = 0;

        InicializarEstrellas();
        InicializarNave();
        InicializarRayo();
        InicializarKamikazes();

        cpc_SetBorder(9);

        while(nave.vida > 0)
        {
            if (cpc_TestKey(TECLA_IZQ) == 1)
            {
                if (nave.velocidad.x > 0)
                    nave.velocidad.x = -nave.velocidad.x;
                nave.velocidad.x -= NAVE_ACELERACION;
            }
            if (cpc_TestKey(TECLA_DER) == 1)
            {
                if (nave.velocidad.x < 0)
                    nave.velocidad.x = -nave.velocidad.x;
                nave.velocidad.x += NAVE_ACELERACION;
            }

            if (cpc_TestKey(TECLA_ARR) == 1)
            {
                if (nave.velocidad.y > 0)
                    nave.velocidad.y = -nave.velocidad.y;
                nave.velocidad.y -= NAVE_ACELERACION_Y;
            }
            if (cpc_TestKey(TECLA_ABA) == 1)
            {
                if (nave.velocidad.y < 0)
                    nave.velocidad.y = -nave.velocidad.y;
                nave.velocidad.y += NAVE_ACELERACION_Y;
            }
            if (cpc_TestKey(TECLA_DIS) == 1)
            {
                if (rayo.vivo == 0)
                {
                    ReproducirEfectoDisparo();

                    rayo.posicion.x = nave.posicion.x + 500;
                    rayo.posicion.y = nave.posicion.y + 100;
                    rayo.vivo = 1;
                }
            }

            ActuarNave();
            ActuarRayo();
            if (nave.fase == 0)
                ActuarKamikazes();
            else if (nave.fase == 1)
                ActuarSweepers();
            else if (nave.fase == 2)
                ActuarCarrier();

            PintarEstrellas();
            PintarNave();
            PintarRayo();
            if (nave.fase == 0)
                PintarKamikazes();
            else if (nave.fase == 1)
                PintarSweepers();
            else if (nave.fase == 2)
                PintarCarrier();

            if (nave.fase == 0 && nave.muertos == MAX_KAMIKAZES)
                CambiarDeFase();
            else if (nave.fase == 1 && nave.muertos == MAX_SWEEPERS)
                CambiarDeFase();
            else if (nave.fase == 2 && nave.muertos == 1)
                CambiarDeFase();
        }
        cpc_SetBorder(0); // Borde negro
        // cuando mueres llegas aqui
        cpc_ClrScr();

        cpc_PutSp(gameoverSP, 80, 80, cpc_GetScrAddress(0, 2));

        cpc_PrintGphStrXY("HIGH;SCORES", 28, 102);

        d = 0;
        for(i = 1; i < maxP; i++)
        {
            if(pu[i]->puntos <= nave.puntos &&
                pu[i-1]->puntos > nave.puntos)
            {
                cpc_PrintGphStrXY("YOU", 26, 114 + (i * 12));
                sprintf(msg, "%d", nave.puntos);
                cpc_PrintGphStrXY(msg, 45, 114 + (i * 12));
                d = 12;
                ignorarUltimo = 1;
            }

            if(i == maxP-1 && ignorarUltimo == 1)
                continue;

            cpc_PrintGphStrXY(pu[i]->nombre, 26, 114 + (i * 12 + d));
            sprintf(msg, "%d", pu[i]->puntos);
            cpc_PrintGphStrXY(msg, 45, 114 + (i * 12 + d));
        }
        ignorarUltimo = 0;
        cpc_PrintGphStrXY("PULSA;C;PARA;TRANSFERIR;TU;COPIA",5,188);
        while(1) // Cuando se presione una tecla empezara el juego
        {
            if(cpc_TestKey(TECLA_CON) == 1)
                break;
        }

        EscribirHistoria();
    }
}
