/*
 * Binpatch.
 *
 * This program is designed to produce a patched binary file based on the
 * old image and a patch file. In addition it can split a single image into
 * multiple resulting files.
 *
 * The program works as follows.
 *
 * To produce a patch file the supplier executes the command:
 *
 *       binpatch oldimage patchfile newimage [additional new images]
 *
 * This produces a 'patchfile' that can be used to regenerate the new images
 * but only if the user has a copy of the old image:
 *
 *       binpatch oldimage patchfile
 *
 * Binpatch will create the new files with _exactly_ the same names as those
 * given in the command to produce the patch file.
 *
 * This is designed to operate in situations where the new image is very
 * similar to the old image but this isn't required, it is even possible
 * to generate a patchfile that'll generate an image from an unrelated file.
 *
 * (C) 1996 Robert de Bath,   rdebath@cix.compulink.co.uk
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <malloc.h>

FILE * pfd, * ifd, * ofd;
long getnum();
void writenum(long val);

char namebuf[128];

main(argc, argv)
int argc;
char ** argv;
{
   int ar;
   if( argc < 3 )
   {
      fprintf(stderr, "Usage, decode: %s Image_file patch_file\n", argv[0]);
      fprintf(stderr, "Usage, encode: %s Image_file patch_file original1 [more originals]\n", argv[0]);
      exit(9);
   }
   if( argc == 3 ) { read_file(argv[1], argv[2]); exit(0); }

   read_image(argv[1]);

   ofd = fopen(argv[2], "w");
   if( ofd == 0 )
   {
      fprintf(stderr, "Cannot open output file %s\n", argv[2]);
      exit(9);
   }

   for(ar=3; ar<argc; ar++)
      make_patch(argv[ar]);

   close(ofd);
   exit(0);
}

char * image;
int imagelen;

read_image(imagename)
char * imagename;
{
   int fd;
   struct stat st;

   fd = open(imagename, 0);
   if( fstat(fd, &st) < 0 )
   {
      fprintf(stderr, "Cannot stat file %s\n", imagename);
      exit(9);
   }
   imagelen=st.st_size;
   if( imagelen!=st.st_size || (image = malloc(imagelen)) == 0 )
   {
      fprintf(stderr, "Out of memory\n");
      exit(7);
   }
   if( read(fd, image, imagelen) != imagelen )
   {
      fprintf(stderr, "Read error reading %s\n", imagename);
      exit(7);
   }
   close(fd);
}

make_patch(orig_name)
char * orig_name;
{
   char * pattern;
   int patternlen;
   int maxlen, where, match;
   int scan1, scan2;
   int count=0, sum;
   int lastmatch = 0;
   int cnt = 0;

   char * p;

   int fd;
   struct stat st;

   fd = open(orig_name, 0);
   if( fstat(fd, &st) < 0 )
   {
      fprintf(stderr, "Cannot stat file %s\n", orig_name);
      exit(9);
   }
   pattern = malloc(patternlen=st.st_size);
   read(fd, pattern, patternlen);
   close(fd);

   for(p=orig_name; *p; p++) writenum((long) *p);
   writenum(0L);
   writenum(imagelen);

   for(scan1=0; scan1<patternlen; )
   {
      maxlen=0;
      for(scan2=0; scan2<imagelen; scan2++)
      {
         if( pattern[scan1] == image[scan2] )
	 {
	    match=1;
	    while(scan1+match < patternlen &&
	          scan2+match < imagelen &&
		  pattern[scan1+match] == image[scan2+match])
	    {
	       match++;
	    }

	    if( match > maxlen ) { maxlen=match; where=scan2; }
	    else if( match == maxlen && where <= lastmatch)
	       { maxlen=match; where=scan2; }
	 }
      }
      if( maxlen )
         lastmatch = where;
      else
      {
	 if( imagelen )
            writenum(-1L ^ ((pattern[scan1]^image[scan1%imagelen])&0xFF));
	 else
            writenum(-1L ^ (pattern[scan1]&0xFF));
	 scan1++;
	 count++;
	 continue;
      }
      writenum(where);
      writenum(maxlen);
      scan1+=maxlen;
      count++;
   }

   for(sum=scan1=0; scan1<patternlen; scan1++)
   {
      sum += (pattern[scan1] & 0xFF);
      sum = (sum & 0xFFFFFF) + (sum>>24);
   }

   fprintf(stderr, "Patch for %s generated, rom is %d chunks sum is %d\n",
                    orig_name, count, sum);

   writenum(sum);
   writenum(-1L);

   if( patternlen/count < 8 )
      fprintf(stderr, "Hmm, the run lengths are rather small, are you sure you have the right files?\n");
}

read_file(image_file, patch_file)
char *image_file, *patch_file;
{
   long sum;
   long posn, len;
   long off;
   long count;
   int i,ch;

   pfd = fopen(image_file, "rb");
   if( pfd == 0 )
   {
      fprintf(stderr, "Cannot open file %s\n", image_file);
      exit(9);
   }

   ifd = fopen(patch_file, "rb");
   if( ifd == 0 )
   {
      fprintf(stderr, "Cannot open file %s\n", patch_file);
      exit(9);
   }

   for(;;)
   {
      for(i=0; (ch=getnum()) > 0; ) namebuf[i++] = ch;
      namebuf[i] = 0;

      if( *namebuf == 0 ) break;
      imagelen = getnum();

      ofd = fopen(namebuf, "wb");
      if( ofd == 0 )
      {
	 fprintf(stderr, "Cannot open output file %s\n", namebuf);
	 exit(9);
      }

      count = sum = 0;
      for(;;)
      {
	 posn = getnum();
	 if( posn < 0 )
	 {
	    int ch = (posn ^ -1);

	    if( imagelen )
	    {
	       if( fseek(pfd, count%imagelen, 0) < 0 ) perror("seek");
	       ch ^= fgetc(pfd);
	    }

	    putc(ch, ofd);
	    sum += (ch & 0xFF);
	    sum = (sum & 0xFFFFFF) + (sum>>24);
	    count++;
	    continue;
	 }
	 len = getnum();
	 if( len <= 0 ) break;

	 if( fseek(pfd, posn, 0) < 0 ) perror("seek");

	 for(off=0; off<len; off++)
	 {
	    int ch = getc(pfd);
	    if(ch==EOF) break;
	    putc(ch, ofd);
	    sum += (ch & 0xFF);
	    sum = (sum & 0xFFFFFF) + (sum>>24);
	    count++;
	 }
      }
      fclose(ofd);

      if( sum == posn )
	 printf("Calculated checksum matches file '%s'\n", namebuf);
      else
	 printf("Checksum failed, image '%s' is probably wrong\n", namebuf);
   }
   fclose(ifd);
   fclose(pfd);
}

/* This functions read and write long integers to a file.
 * The encoding scheme reduces the amount of data to be written to the
 * file considerably by making the values that are normally the most common
 * (near zero) into the shortest.
 * 
 * The first byte is encoded like this:
 *
 *    0XXXXXXX   0..127					+ no more
 *    10XXXXXX   0..16383				+ 1 byte
 *    110XXXXX   0..2097151				+ 2 bytes
 *    1110XXXX   0..0xFFFFFFF 				+ 3 bytes
 *    1111XXXX   0..0xFFFFFFFFF				+ 4 bytes (except FF)
 *    11111111   Minus, negates following number	+ number
 */

long getnum()
{
   long val = 0;
   int sgn = 1;
   int ch = getc(ifd);
   if( ch == EOF ) return -1;
   if( ch == 0xFF ) { sgn = -1; ch = getc(ifd); }
   switch(ch>>4)
   {
   case 15:
      val = (ch&0x0F);
      if(0) {
   case 14:
         val = (ch&0x0F);
      }
      else val = (val<<8) + getc(ifd); if(0) {
   case 12: case 13:
         val = (ch&0x1F);
      }
      else val = (val<<8) + getc(ifd); if(0) {
   case 8: case 9: case 10: case 11:
         val = (ch&0x3F);
      }
      else val = (val<<8) + getc(ifd); if(0) {
   case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
	 val = ch;
      }
      else val = (val<<8) + getc(ifd);
   }
   if( sgn < 0 ) val = -val;  /* Ignore -0 => -2^31 */
   return val;
}

void
writenum(val)
long val;
{
   if( val < 0 )
   {
      putc(0xFF, ofd);
      val = -val;
   }
   if( val < 0x80 )
      ;
   else if( val < 0x4000 ) putc(0x80|(int)(val>>8), ofd);
   else
   {
      if( val < 0x200000L ) putc(0xC0|(int)(val>>16), ofd);
      else
      {
         if( val < 0x10000000L ) putc(0xE0|(int)(val>>24), ofd);
	 else
	 {
	    putc(0xF0, ofd);
	    putc((int)(val>>24), ofd);
	 }
	 putc((int)(val>>16), ofd);
      }
      putc((int)(val>>8), ofd);
   }
   putc((int)val, ofd);
}
