/* >>>>>>>>>>>>>>>>>>> The BBC Tape Utility <<<<<<<<<<<<<<<<<<<<<<<<<
 *
 * Copyright (c) Ole Stauning 1996.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the copyright holder
 * not be used in advertising or publicity pertaining to distribution
 * of the software without specific, written prior permission. The
 * copyright holder makes no representations about the suitability of
 * this software for any purpose. It is provided "as is" without express
 * or implied warranty.
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  So if you fry your soundcard, do not blame it on me :-)
 *
 *  Please send comments, questions, bug-reports and flames to 
 *
 *                        os@imm.dtu.dk
 */

#include <math.h>
#include <malloc.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <strings.h>
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>
#include <sys/soundcard.h>

/* >>>>> SPECIFIC BBC FILE FORMAT STUFF <<<<<< */

#define DEFAULT_AUDIO_DEV "/dev/dsp"
#define DEFAULT_SAMPLE_FREQ 44100
#define DEFAULT_BAUD_RATE 1200
#define DEFAULT_BLOCK_SIZE 256
#define DEFAULT_START_BITS 0
#define DEFAULT_NUMBER_OF_START_BITS 1
#define DEFAULT_STOP_BITS 1
#define DEFAULT_NUMBER_OF_STOP_BITS 1
#define DEFAULT_START_STOP_BITS "0/1"

#define SIGNAL_AMP 100
#define SIGNAL_ZERO 128
#define SIGNAL_PHASE -1
#define SYNC_BYTE 0x2a
#define SYNC_BUGF 0xaa
#define DATA_BITS 8
#define HEADER_SPACE 6000
#define BLOCK_SPACE 1200
#define SYNC_LENGTH 500

#define NO_ERROR 0
#define BAD_AUDIO 1
#define WRONG_BLOCK 2
#define END_OF_AUDIO 3 

int BAUD_RATE=DEFAULT_BAUD_RATE,
    SAMPLE_FREQ=DEFAULT_SAMPLE_FREQ,
    BLOCK_SIZE=DEFAULT_BLOCK_SIZE,
    START_BITS=DEFAULT_START_BITS,
    NUMBER_OF_START_BITS=DEFAULT_NUMBER_OF_START_BITS,
    STOP_BITS=DEFAULT_STOP_BITS,
    NUMBER_OF_STOP_BITS=DEFAULT_NUMBER_OF_STOP_BITS;
    
u_char TAPE_FILENAME[255]; /* length should be <= 10, but anyway, it might be usefull??*/
u_long LOAD_ADDRESS,EXEC_ADDRESS;

/* >>>>> END OF SPECIFIC BBC FILE FORMAT STUFF <<<<<< */

u_char AUDIO_DEV[30]=DEFAULT_AUDIO_DEV;
u_char WAV_FILE[255];
    
int USE_WAV_FILE=0;
u_char *command;

/* the save wave table */
u_char *one,*zero;
int length;

/* some signal specific variables. Used in loading */
sigset_t dsp_mask,dsp_oldmask;
sigset_t load_mask,load_oldmask; 
int dsp_int=0,load_int=0; 

int synccount=0;

/* input audio buffers */
#define AUDIO_BUFFERS 2
u_char *audiobufs[AUDIO_BUFFERS],*audiobuf;
int *audiobufs_length;
int abuf_size,abuf_index,abuf_number;

/* The crc variable */

u_short crc;

/* Definitions for Micro$oft WAVE format */

#define RIFF		0x46464952	
#define WAVE		0x45564157
#define FMT		0x20746D66
#define DATA		0x61746164
#define PCM_CODE	1
#define WAVE_MONO	1
#define WAVE_STEREO	2
     
typedef struct _waveheader {
  u_long	main_chunk;	/* 'RIFF' */
  u_long	length;		/* filelen */
  u_long	chunk_type;	/* 'WAVE' */

  u_long	sub_chunk;	/* 'fmt ' */
  u_long	sc_len;		/* length of sub_chunk, =16 */
  u_short	format;		/* should be 1 for PCM-code */
  u_short	modus;		/* 1 Mono, 2 Stereo */
  u_long	sample_fq;	/* frequence of sample */
  u_long	byte_p_sec;
  u_short	byte_p_spl;	/* samplesize; 1 or 2 bytes */
  u_short	bit_p_spl;	/* 8, 12 or 16 bit */ 

  u_long	data_chunk;	/* 'data' */
  u_long	data_length;	/* samplecount */
} WaveHeader;  

void write_wav_header(int fd){ /* write a WAVE-header */
  WaveHeader wh;

  wh.main_chunk = RIFF;
  wh.length     = 0 + sizeof(WaveHeader) - 8; 
  wh.chunk_type = WAVE;
  wh.sub_chunk  = FMT;
  wh.sc_len     = 16;
  wh.format     = PCM_CODE;
  wh.modus      = 1; /* mono */
  wh.sample_fq  = SAMPLE_FREQ ;
  wh.byte_p_spl = 1; /* 1 byte sample size */
  wh.byte_p_sec = wh.sample_fq * wh.modus * wh.byte_p_spl;
  wh.bit_p_spl  = 8; /* 8 byte sample size */
  wh.data_chunk = DATA;
  wh.data_length= 0x7fffffff; /* Only stupid WINDOZE programs use this */
  write (fd, &wh, sizeof(WaveHeader));
}

void test_wavefile(int audio){ /* test a WAVE-header */
WaveHeader *wp;
     
  if ((wp=malloc(sizeof(WaveHeader)))==NULL){
    fprintf(stderr,"%s: Unable to allocate wave header buffer for input\n",command);
    exit(-1);
  }
  read(audio,(u_char *)wp, sizeof(WaveHeader));
  
  if (wp->main_chunk != RIFF || wp->chunk_type != WAVE ||
    wp->sub_chunk != FMT || wp->data_chunk != DATA) {
      fprintf (stderr, "%s: This is not a WAV format file, trying raw.\n", command);
  } else {
    if (wp->format != PCM_CODE) {
      fprintf (stderr, "%s: can't decode not PCM-coded WAVE-files\n", command);
      exit (-1);
    }
    if (wp->modus > 1) {
      fprintf (stderr, "%s: can only decode mono WAVE-files\n", command);
      exit (-1);
    }
    if (wp->bit_p_spl!=8) {
      fprintf (stderr, "%s: can only decode 8 bit WAVE-files\n", command);
      exit (-1);
    }
    SAMPLE_FREQ = wp->sample_fq;   
  }

  free(wp);
}

/* end of definitions for Micro$oft WAVE format */
         
int audio_open(int mode){
int audio,i,dummy;

 if ((audio = open(AUDIO_DEV,mode,0))==-1){
    perror(AUDIO_DEV);
    exit(-1);
  }
  dummy=8;
  ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &dummy);
  if (8 != dummy){
    fprintf(stderr, "%s: Unable to set the sample size to 8 bit\n",command);
    exit(-1);
  }
  dummy=0;
  if (ioctl(audio, SNDCTL_DSP_STEREO, &dummy)==-1){
    fprintf(stderr, "%s: Unable to set mono\n",command);
    exit(-1);
  }
  dummy=SAMPLE_FREQ;
  if (ioctl(audio, SNDCTL_DSP_SPEED, &dummy) == -1){
    fprintf(stderr, "%s: Unable to set audio speed\n",command);
    exit(-1);
  }
  ioctl(audio, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
  if (abuf_size < 4096 || abuf_size > 65536) {
    if (abuf_size == -1)
      perror (AUDIO_DEV);
    else
      fprintf (stderr, "%s: Invalid audio buffers size %d\n",command,abuf_size);
    exit (-1);
  } 
  return audio;
} 

void clear_crc(){
  crc=0;
}

u_short read_crc(){
  return crc;
}

void byte2crc(u_char b){
int i;

  crc^=b<<8;
  for(i=0;i<8;i++) 
    crc = (crc&0x8000) ? ( (crc ^ 0x810) << 1 ) | 1 : crc << 1;   
}

void make_wave_table(){
 int i;
  length=SAMPLE_FREQ/BAUD_RATE;
  if ((zero=malloc(length))==NULL || (one=malloc(length))==NULL){
    fprintf(stderr,"%s: Unable to allocate audio buffer for output\n",command);
    exit(-1);
  }
  for(i=0;i<length;i++) /* zero wave table */
    zero[i]=SIGNAL_ZERO+SIGNAL_AMP*SIGNAL_PHASE*sin((2.0*M_PI*i)/length);
  for(i=0;i<length;i++) /* one wave table  */
    one[i]=zero[(2*i)%length];                                          
}

void delete_wave_table(){
  free(zero);free(one);
}

void save_byte(int audio, u_char b){
int i;
u_char *onezero;

  byte2crc(b); /* update the crc value */

  for(i=0;i<NUMBER_OF_START_BITS;i++){ /* first write the start bits */
    onezero=(START_BITS&(1<<i)) ? one : zero;
    if (write(audio,onezero,length)!=length){
      perror(AUDIO_DEV);
      exit(-1);
    }
  }
  for(i=0;i<DATA_BITS;i++){ /* then write the data */
    onezero=(b&(1<<i)) ? one : zero;
    if (write(audio,onezero,length)!=length){
      perror(AUDIO_DEV);
      exit(-1);
    }
  }
  for(i=0;i<NUMBER_OF_STOP_BITS;i++){ /* finally write the stop bits */
    onezero=(STOP_BITS&(1<<i)) ? one : zero;
    if (write(audio,onezero,length)!=length){
      perror(AUDIO_DEV);
      exit(-1);
    }
  }
}

void save_long(int audio,u_long ul){
int i;
  for(i=0;i<4;i++){
    save_byte(audio,ul&0xff);
    ul>>=8;
  }
}    

void save_short(int audio,u_short us){
int i;
  for(i=0;i<2;i++){
    save_byte(audio,us&0xff);
    us>>=8;
  }
}    

void make_sync(int audio,int sync_length){ /* create the tone */ 
int i; 
  
  for(i=0;i<sync_length;i++) 
    if (write(audio,one,length)!=length){
      perror(AUDIO_DEV);
      exit(-1);
    }
}
          
void save(int source, int audio){
int i;
u_char *data,flags;
u_short block=0,block_length,CRC;

  if ((data=malloc(BLOCK_SIZE))==NULL){
    fprintf(stderr,"%s: Unable to allocate data buffer\n",command);
    exit(-1);
  }
  
  make_wave_table(); /* initialize the sound tables */
    
  make_sync(audio,HEADER_SPACE); /* create the file header tone */ 
     
  while((block_length=read(source,data,BLOCK_SIZE))>0){
 
    /* write the header first: */ 
  
    save_byte(audio,(u_char)0x2a); /* sync byte */
    
    clear_crc(); /* refresh the crc */
    
    for(i=0;TAPE_FILENAME[i]!='\0';i++)
      save_byte(audio,TAPE_FILENAME[i]);    
    save_byte(audio,(u_char)0x00);
    fprintf(stderr,"%s ",TAPE_FILENAME);
    save_long(audio,LOAD_ADDRESS);
    fprintf(stderr,"%8.8x ",LOAD_ADDRESS);
    save_long(audio,EXEC_ADDRESS);
    fprintf(stderr,"%8.8x ",EXEC_ADDRESS);
    save_short(audio,block);
    fprintf(stderr,"%2.2x ",block++);
    save_short(audio,block_length);
    fprintf(stderr,"%4.4x ",block_length);
    flags=(block_length!=BLOCK_SIZE)<<7;
    save_byte(audio,flags);
    fprintf(stderr,"%2.2x \n",flags);
    save_long(audio,0);
    
    CRC=read_crc();
    CRC=((CRC&0xff00)>>8)|((CRC&0x00ff)<<8); /* Why this? is it more safe */
    save_short(audio,CRC);
    
    clear_crc(); /* refresh the crc */
    
    /* now flush the data */
    
    for(i=0;i<block_length;i++)
      save_byte(audio,data[i]);
     
    /* and write data CRC */
    
    CRC=read_crc();
    CRC=((CRC&0xff00)>>8)|((CRC&0x00ff)<<8); /* Why this? is it more safe */
    save_short(audio,CRC);
     
    make_sync(audio,BLOCK_SPACE); /* block sync tone */
    
  }  
  delete_wave_table(); /* clean up before we leave */ 
}     

void dsp_handler(int dummy){ /* hey, someone is calling dsp child */
  dsp_int=1;
} 

void dsp_pause(){
  sigprocmask(SIG_BLOCK, &dsp_mask, &dsp_oldmask);
  while(!dsp_int)sigsuspend(&dsp_oldmask); /* sleep until new buffer from parent */ 
  dsp_int=0;
  sigprocmask(SIG_UNBLOCK, &dsp_mask, NULL);
}

void load_handler(int dummy){ /* hey, someone is calling loader parent */
  load_int=1;
} 

void load_pause(){
  sigprocmask(SIG_BLOCK, &load_mask, &load_oldmask);
  while(!load_int)sigsuspend(&load_oldmask);  
  load_int=0;
  sigprocmask(SIG_UNBLOCK, &load_mask, NULL);
}

int re_sample(int* s, double interval, int n){
int i,old_abuf_index=abuf_index;

  for(i=0;i<=n;i++){
    abuf_index=old_abuf_index+rint((interval*i)/n);
    
    if (abuf_index>=audiobufs_length[abuf_number]) {  /* Buffer shift */    

      if (dsp_int && !USE_WAV_FILE) 
        fprintf(stderr,"%s: WARNING signal processing too slow.\n",command);
      dsp_pause(); /* await a new workload from parent */
      kill(getppid(),SIGUSR2); /* ok got it */
    
      old_abuf_index-=audiobufs_length[abuf_number];
      abuf_index%=audiobufs_length[abuf_number];
      abuf_number=(abuf_number+1)%AUDIO_BUFFERS;
      if (audiobufs_length[abuf_number]<=0) { 
        fprintf(stderr,"end of audio reached \n");
        return END_OF_AUDIO; /* end of audio */
      }
    }
    s[i]=(int)audiobufs[abuf_number][abuf_index];
  }
  return NO_ERROR;
}

int load_byte(u_char *b){
static int number=0,mask[2],i,bit,idx,status;
double track=0;
int error=0,s[5],d;

  status=idx=0;
  (*b)=0;
       
  while(status<3 && !error){
   
    for(i=0;i<2;i++) { /* sample the next two half bauds */

      error=re_sample(s,((double)SAMPLE_FREQ)/(2*BAUD_RATE)+track,4); 
       
      mask[i]= ((3*s[1]>s[0]+s[2]+s[4])<<1 | (3*s[3]>s[0]+s[2]+s[4])) ^ ((SIGNAL_PHASE==-1)?0x3:0x0);
   
      switch (mask[i]) { /* the calculation of track is done in doubles to get nice rounding. */
                         /* the value of track has actually a physical meaning. If you really */
                         /* want to know then email me.                                       */

        case 0x03:/* 11 -> bit=0 */
          if ((d=s[4]+s[0])>0)
            track=((double)s[4]-s[0])*SIGNAL_PHASE*SAMPLE_FREQ/(d*8.*BAUD_RATE);  
          break;
              
        case 0x02:/* 10 -> bit=1 */
          if ((d=s[2]+s[4])>0)
            track=((double)s[2]-s[4])*SIGNAL_PHASE*SAMPLE_FREQ/(d*8.*BAUD_RATE);  
          break;
	        
        case 0x01:/* 01 -> bit=1 */
          if ((d=s[4]+s[2])>0)
            track=((double)s[4]-s[2])*SIGNAL_PHASE*SAMPLE_FREQ/(d*8.*BAUD_RATE);  
	  break;
	
        case 0x00:/* 00 -> bit=0 */
          if ((d=s[0]+s[4])>0)
            track=((double)s[0]-s[4])*SIGNAL_PHASE*SAMPLE_FREQ/(d*8.*BAUD_RATE);  
          break;
	
        default:
          track=1; /* Just to avoid getting stuck in a perfect reverse phase */
      }         
    }

    switch (mask[0]<<2 | mask[1]) { 
      case 0x0c:/* 1100 -> bit=0 */
      case 0x03:
        bit=0;
        if (synccount<SYNC_LENGTH) synccount=0; 
        break;
              
      case 0x0a:/* 1010 -> bit=1 */
      case 0x05:
        bit=1;
	if (synccount<SYNC_LENGTH && ++synccount+1==SYNC_LENGTH) { /* This is also the sync tone */
          fprintf(stderr,"Sync tone detected\n");
          synccount=SYNC_LENGTH;
        }
        break;
	    
      case 0x02:/* 0010 -> bit=0 */
      case 0x0d:
        bit=0;
	if (synccount<SYNC_LENGTH) synccount=0;
        break;
	    
      case 0x0b:/* 1011 -> bit=1 */
      case 0x04:
        bit=1;
	if (synccount<SYNC_LENGTH) synccount=0;
	break;
	
      default:
        if (synccount==SYNC_LENGTH) { /* Arghh! all is lost */
          fprintf(stderr,"%s: unknown signal\n",command); 
        }
        synccount=0;  /* no sync */
        bit=2;   /* Its an X-bit, uhhh */
    }

    if (synccount==SYNC_LENGTH) { /* we are on the right track */
      switch (status) {
        case 0 : /* waiting for startbits */
          if ((bit<<idx)==(START_BITS&(1<<idx))) {
            idx++;
            if (idx==NUMBER_OF_START_BITS) {
              status=1;idx=0; /* now read the byte */
            }
          } else 
              if (idx>0) { 
                fprintf(stderr,"%s: startbit not correct\n",command);
                error=BAD_AUDIO;
              }
          break;
          
        case 1 : /* loading data bits     */
          (*b)|=(bit<<idx++);
          if (idx==DATA_BITS) {
            byte2crc(*b); /* include the new byte in the crc value */
            status=2;idx=0; /* now expect some stopbits */
          }
          break;
          
        case 2 : /* loading stopbits     */
          if ((bit<<idx)==(STOP_BITS&(1<<idx))) {
            idx++;
            if (idx==NUMBER_OF_STOP_BITS) status=3; /* The byte is ready for delivery */
          } else { 
              fprintf(stderr,"%s: stopbit not correct\n",command);
              error=BAD_AUDIO;
            }
      }
    } 
  }             
  return error;       
}
  
int load_long(u_long* ul){
int i,error=0;
u_char b;
  (*ul)=0;
  for(i=0;i<4 && !error;i++){
    error=load_byte(&b);
    (*ul)|=b<<(i*8);
  }
  return error;
}   

int load_short(u_short* us){
int i,error=0;
u_char b;
  (*us)=0;
  for(i=0;i<2 && !error;i++){
    error=load_byte(&b);
    (*us)|=b<<(i*8);
  }
  return error;
}   

void load(int data,int audio){
int shmid,i;
pid_t dsp_child;
u_char *incoming,byte,flags;
int word,error;
u_long extra;
u_short block_length,block,expected_block,CCRC,CRC;

  /* First allocate some shared memory */

  if (abuf_size==0) abuf_size=65536;  /* we are reading from file (i hope) */ 
  if ((shmid=shmget(IPC_PRIVATE,abuf_size*AUDIO_BUFFERS,IPC_CREAT|IPC_EXCL|384))<0){
    fprintf(stderr,"%s: Unable to allocate shared memory\n",command);
    exit (-1);
  }
  if ((audiobuf=shmat(shmid,0,0))<0){
    fprintf(stderr,"%s: Unable to attach shared memory to data segment\n",command);
    exit (-1);
  }
  if ((shmid=shmget(IPC_PRIVATE,sizeof(int)*AUDIO_BUFFERS,IPC_CREAT|IPC_EXCL|384))<0){
    fprintf(stderr,"%s: Unable to allocate shared memory\n",command);
    exit (-1);
  }
  if ((audiobufs_length=(int*)shmat(shmid,0,0))<0){
    fprintf(stderr,"%s: Unable to attach shared memory to data segment\n",command);
    exit (-1);
  }
  
  for(i=0;i<AUDIO_BUFFERS;i++) /* audio buffer array */
    audiobufs[i]=audiobuf+i*abuf_size; 

  *audiobuf=0;
 
  if (dsp_child=fork()) {                /* i am the parent, i will just load audio to  */
    struct sigaction load_action;        /* shared memory and let my child do the work. */  
   
    sigemptyset(&load_mask);sigaddset(&load_mask,SIGUSR2);    
    load_action.sa_handler=load_handler;
    load_action.sa_mask=load_mask;
    load_action.sa_flags=0;
    sigaction(SIGUSR2, &load_action, NULL);  
      
    while(!(*audiobuf)); /* do some busy wait until child has set up signalling */  

    i=0;
    
    while((audiobufs_length[i]=read(audio,audiobufs[i],abuf_size)) > 0) {   
      kill(dsp_child,SIGUSR1);           /* Hey dsp child! you have some more work!     */
      load_pause();                      /* wait for signal from hard working child     */
      i=(i+1)%AUDIO_BUFFERS;
    }
    kill(dsp_child,SIGUSR1);
    wait(NULL); 
  } else {                               /* i am the signal proccessing child           */
    struct sigaction dsp_action;       
     
    sigemptyset(&dsp_mask);sigaddset(&dsp_mask,SIGUSR1);
    
    dsp_action.sa_handler=dsp_handler;
    dsp_action.sa_mask=dsp_mask;
    dsp_action.sa_flags=0;
    sigaction(SIGUSR1, &dsp_action, NULL);  
     
    *audiobuf=1; /* ok! ready, signalling set up. */
     
    dsp_pause(); /* await the first workload from parent */
    kill(getppid(),SIGUSR2); /* ok got your workload, thanx */
       
    abuf_index=abuf_number=0;   
       
    flags=0;error=0;expected_block=0;
    
    while(!(flags&0x80)&& !error){   
    
      error=load_byte(&byte); /* sync byte */
      
      if (!error && block==0 && byte==SYNC_BUGF) {
        fprintf(stderr,"%s: sync bug fix byte %2x recieved\n",command,byte);
        error=load_byte(&byte); /* sync byte */
      }
      
      if (!error && byte!=SYNC_BYTE) {
        fprintf(stderr,"%s: unknown magic number %2x\n",command,byte);
        error=1;        
      }
      
      clear_crc(); /* start from a fresh crc value */
            
      i=0;byte=1;
      while(!error && byte>0) {
        error=load_byte(&byte); 
        TAPE_FILENAME[i++]=byte;
      }
      if (!error) fprintf(stderr,"%s ",TAPE_FILENAME);
      if (!error) error=load_long(&LOAD_ADDRESS);
      if (!error) fprintf(stderr,"%8.8x ",LOAD_ADDRESS);
      if (!error) error=load_long(&EXEC_ADDRESS);
      if (!error) fprintf(stderr,"%8.8x ",EXEC_ADDRESS);
      if (!error) error=load_short(&block);
      if (!error) fprintf(stderr,"%2.2x ",block);
      if (!error) error=load_short(&block_length);
      if (!error) fprintf(stderr,"%4.4x ",block_length);
      if (!error) error=load_byte(&flags);
      if (!error) fprintf(stderr,"%2.2x \n",flags);
      if (!error) error=load_long(&extra);  
     
      CCRC=read_crc();
      if (!error) error=load_short(&CRC); 
      CRC=((CRC&0xff00)>>8)|((CRC&0x00ff)<<8);/*  Why this? is it more safe */
      if (!error && CRC!=CCRC) {
        fprintf(stderr,"%s: CRC check failed for header\n",command);
        error=BAD_AUDIO;         
      }
      
      if (!error && expected_block!=block)
        fprintf(stderr,"%s: WARNING expecting block %2.2x.\n",command,expected_block);
                
      if (!error) {
        if ((incoming=malloc(block_length))==NULL){
          fprintf(stderr,"%s: Unable to allocate data buffer for input\n",command);
          exit(-1);
        }
  
        clear_crc(); /* start from a fresh crc value */
     
        /* now load the data */
    
        for(i=0;i<block_length && !error;i++) {
          error=load_byte(&byte);
          incoming[i]=byte;
        }
     
        /* and now the CRC stuff */
    
        CCRC=read_crc();
        if (!error) error=load_short(&CRC);
        CRC=((CRC&0xff00)>>8)|((CRC&0x00ff)<<8);/*  Why this? is it more safe */
        if (!error && CRC!=CCRC) {
          fprintf(stderr,"%s: CRC check failed for data block\n",command);
          error=BAD_AUDIO;        
        }
        
        if (!error && expected_block==block) 
          if (write(data,incoming,block_length)!=block_length){
            fprintf(stderr,"%s: error writing data\n",command);
            exit(-1);
          }
  
        free(incoming); 
      }
      if (!error && expected_block==block) expected_block++; /* just perfect */
      else if (error!=END_OF_AUDIO) { /* we just try again */
        error=NO_ERROR;
        synccount=0;
      }
    }
    kill(getppid(),SIGINT);
    shmdt((char*)audiobuf);shmdt((char*)audiobufs_length); /* am i right?? */    
  } 
}

void usage(){
 fprintf(stderr,"Usage:\n");
 fprintf(stderr,"bbc-load [options] [filename]\n", command);
 fprintf(stderr,"bbc-save [options] tape-filename load-address exec-address [filename]\n", command);
 fprintf(stderr,"options             | Description              | default value\n");
 fprintf(stderr,"--------------------------------------------------------------\n");
 fprintf(stderr,"-d device           | use another device file. | %s\n", DEFAULT_AUDIO_DEV);
 fprintf(stderr,"-f freq Hz          | sample frequency.        | %d\n", DEFAULT_SAMPLE_FREQ);
 fprintf(stderr,"-r baud-rate        | baud rate of file.       | %d\n", DEFAULT_BAUD_RATE);
 fprintf(stderr,"-s block-size       | block size of file.      | %d\n", DEFAULT_BLOCK_SIZE);
 fprintf(stderr,"-b start/stop bits  | start and stop bits.     | %s\n", DEFAULT_START_STOP_BITS);
 fprintf(stderr,"-w wav-file         | use wav file.            | \n"); 
 fprintf(stderr,"NOTE: '-' can is used as a wav-filename to specify in-/output\n");
 fprintf(stderr,"      from/to stdin/stdout. This utility is meant to be used with\n");
 fprintf(stderr,"      programs such as sox and vrec to filter samples realtime.\n");
 fprintf(stderr,"example: bbc-save tapefile 0xe00 0x8023 beebfile\n");
 fprintf(stderr,"example: bbc-load beebfile\n");
 exit (-1);
}

int getoptions(int argc, char *argv[]){
char c;
int i;

  while ((c = getopt(argc, argv, "d:f:r:s:b:w:")) != EOF){
    switch (c) {
      case 'd': /* change device name */
	strcpy(AUDIO_DEV,optarg);
	break;
      case 'f': /* change sample frequency */
	SAMPLE_FREQ=atoi(optarg);
	break;
      case 'r': /* change baud rate */
	BAUD_RATE=atoi(optarg);
	break;
      case 's': /* change block size */
	BLOCK_SIZE=atoi(optarg);
	break;
      case 'b': /* change start- and stop- bits */
	NUMBER_OF_STOP_BITS=0;
	NUMBER_OF_START_BITS=0;
	STOP_BITS=START_BITS=0;
	
        for(i=0;optarg[i]!='/';i++) 
          if (optarg[i]=='1') 
            START_BITS |= 1<<i;
          else 
            if (optarg[i]!='0') usage();
          
        if ((NUMBER_OF_START_BITS=i)==0){
          fprintf(stderr,"%s: you must have startbits\n",command);
          exit(-1);
        }        
        for(i++;optarg[i]!='\0';i++) 
          if (optarg[i]=='1') 
            STOP_BITS |= 1<<(i-NUMBER_OF_START_BITS-1);
          else 
            if (optarg[i]!='0') usage();
            
        if ((NUMBER_OF_STOP_BITS=i-NUMBER_OF_START_BITS-1)==0){
          fprintf(stderr,"%s: you must have stopbits\n",command);
          exit(-1);
        }        
        
        if (START_BITS&1){
          fprintf(stderr,"%s: first startbit has to be 0\n",command);
          exit(-1);
        }
       
	break;
      case 'w': /* load/save wav file */
        USE_WAV_FILE=1;
	strcpy(WAV_FILE,optarg);
	break;
      default:
        usage();
    }
  }
  return optind;
}

void bbc_load(int argc, char *argv[]){
int optind,data,audio;

  optind=getoptions(argc,argv);

  if (USE_WAV_FILE) {
    if (strstr(WAV_FILE,"-")) /* the wav file is coming from stdin */
      audio=0;
    else
      if ((audio = open(WAV_FILE, O_RDONLY, 0))==-1){ /* the wav file is coming from disk */
        perror(WAV_FILE);
        exit(-1);
      }
    test_wavefile(audio);
  }
  else
    audio=audio_open(O_RDONLY); /* the wav file is coming from dsp */ 

  if (optind > argc - 1) /* write data to stdout */
    data=1;
  else
    if ((data = open( argv[optind], O_WRONLY | O_CREAT, 0666))==-1) { /* write data to file */
      perror(argv[optind]);
      exit(-1);
    }
  load(data,audio);
  if (data!=1) close(data);
  if (audio!=0) close(audio);
}

void bbc_save(int argc, char *argv[]){
int idx,data,audio;

  idx=getoptions(argc,argv);
    
  if (idx > argc - 1) usage(); /* no tape filename specified */
  strcpy(TAPE_FILENAME,argv[idx++]);
  if (idx > argc - 1) usage(); /* no load address specified */
  LOAD_ADDRESS=strtol(argv[idx++],NULL,0);
  if (idx > argc - 1) usage(); /* no exec address specified */
  EXEC_ADDRESS=strtol(argv[idx++],NULL,0);
  
  if (USE_WAV_FILE) { 
    if (strstr(WAV_FILE,"-")) /* the wav file sent to stdout */
      audio=1;
    else
      if ((audio = open(WAV_FILE, O_WRONLY | O_CREAT, 0666))== -1) { /* well, lets waste some HD :) */
        perror(WAV_FILE);
        exit(-1);
      }
    write_wav_header(audio);
  }
  else audio=audio_open(O_WRONLY); /* the wav file sent to dsp */

  if (idx > argc - 1) /* read data from stdin */
    data=0;
  else
    if ((data = open( argv[idx], O_RDONLY, 0))==-1) { /* read data from file */
      perror(argv[idx]);
      exit(-1);
    }
  save(data,audio);
  if (data!=0) close(data);
  if (audio!=1) close(audio);
}

int main(int argc, char *argv[]){

  command = argv[0];
  if (strstr (command, "bbc-load")) bbc_load(argc,argv);
  else if (strstr (command, "bbc-save")) bbc_save(argc,argv);
  else {
    fprintf (stderr,"Error: command should be named either bbc-load or bbc-save\n");
    exit (1);
  }
  return 1;
}
  
  
 
