/*
 *	Zox3D demo version 1.5 by Robert Schmidt <robert@stud.unit.no>
 *	(C) Copyright 1993 of Ztiff Zox Softwear
 *
 *	None of the source code modules of this project is free to use
 *  commercially.
 *
 *	See ZOX3D.DOC for an introductory text, disclaimers and more.
 *
 */

/*
 *	Things to do:
 *
 *	Find a way to make the aspect ratio *always* square (scaleHeight).
 *	(Semi-)transparent walls - monochrome glass walls.
 *	Light walls + generic light walls (moving doors).
 *	Make moving objects/bitmaps.
 *	Figure out a convenient palette/palette managment scheme.
 *
 */

#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <stdlib.h>
#include <assert.h>

#include "misc.hpp"
#include "vga.hpp"
#include "ilbm.hpp"			// 320x200x256 LBM file decoding library
#include "keyboard.hpp"
#include "resource.hpp"
#include "ztimer.h"
#include "trace.hpp"
#include "map.hpp"
#include "div0.h"			// division overflow handler
#include "fixed.h"			// fixed point math library
#include "player.hpp"
#include "wslice.hpp"
#include "mouse.hpp"
#include "joystick.hpp"

#include <alloc.h>
#include <io.h>
#include <fcntl.h>


// Here follows some important variables which define the 3D viewing
// window, and some important distances and conversion factors.

int windowWidth;			// width of the 3D view window
int windowHeight;			// height of the 3D view window
Fixed screenToWorld;			// conversion factor screen-to-world
Fixed eyeScreenDist;			// distance from eye to screen, world coords
Fixed scaleHeight;				// factor to scale wall height to screen

Fixed pixelX, pixelY;			// screen pixel position relative to guy

Fixed floorDist, ceilingDist;

Fixed iScreenX;					// 1/2 the window size in world coords

int windowChanged;				// flag, 1 if window size modified
char *windowTopLeft;				// pointer to the top left corner of
								// the 3D window (in video memory)

// Accelerator tables.  These store useful values from a frame to another,
// and reused if the player hasn't turned.

Fixed *tPixelX=0, *tPixelY=0, *tdhX=0, *tdvY=0;
char *tDoTrace=0;

// Trace state variables.

int doHorz, doVert;				// Flags: 1 if horizontal/vertical tracing
								// should be performed.  They are usually
								// both 1.
int column;						// The current screen column being traced.


// Fancy & special stuff

BitmapX *winBitmap;
char *winLogoData;
char *skyData, *skyBase, *floorData;

// initSky() allocates and loads the 64 Kb sky bitmap.  Because it is
// exactly 64Kb, the ResourceTable would barf on it, so we handle it
// as a special case.

void initSky()
	{
	skyBase = (char *)farmalloc(65536+15);
	assert(skyBase);
	if (FP_OFF(skyBase) != 0)
		{
		skyBase = (char*)MK_FP(FP_SEG(skyBase)+1, 0);
		}
	int skyh = open("clouds.dat", O_RDONLY | O_BINARY);
	if (skyh == -1 || read(skyh, skyBase, 65534) == -1
				   || read(skyh, skyBase+65534, 2) == -1
				   || close(skyh))
		{
		perror("clouds.dat");
		exit(0);
		}
	skyData = skyBase;
	}

// adjustToWindow() adjusts the distances and temporary buffers/tables
// according to the current screen size, selected by screenWidth and
// screenHeight.

void adjustToWindow()
	{
	// Make sure the window dimensions are no larger than the screen,
	// and no smaller than 1/32 of the screen.

	bound(windowWidth, screenWidth>>5, screenWidth);
	bound(windowHeight, screenHeight>>5, screenHeight);

	// Find the conversion factor from the window coordinate system to
	// world coordinates.

	screenToWorld = BLOCKSIZE_WORLD/(windowWidth<<1);

	// Find a decent distance from the eye to the screen.  I chose 3/4
	// of the windowWidth, and that seems to be quite OK.

	eyeScreenDist = 3*windowWidth*screenToWorld >> 2;

	// scaleHeight is the factor to scale from wall height to screen coords.

	FixDiv(BLOCKSIZE_WORLD, screenToWorld, scaleHeight);

	// iScreenX is the position of the left window edge relative to the
	// middle window column, in world coords.

	iScreenX = - long(windowWidth>>1) * screenToWorld;

	// windowTopLeft points to the top left corner of the window on the
	// screen.  This formula ensures that the window is centered on the
	// screen.

	windowTopLeft = vga
		+ lineOffset[screenHeight-windowHeight>>1]
		+ ((screenWidth - windowWidth) >> 3);

	// Reallocate the tables according to the new dimensions.
	// This might look like a speed hog, but consider the fact that
	// the player won't spend most of his time resizing the window.

	delete[] tempBuf;	tempBuf = new char[windowHeight];
	delete[] tPixelX;	tPixelX = new Fixed[windowWidth];
	delete[] tPixelY;	tPixelY = new Fixed[windowWidth];
	delete[] tdhX;		tdhX = new Fixed[windowWidth];
	delete[] tdvY;		tdvY = new Fixed[windowWidth];
	delete[] tDoTrace;	tDoTrace = new char[windowWidth];
	delete[] tFactorX;	tFactorX = new Fixed[windowWidth];
	delete[] tFactorY;	tFactorY = new Fixed[windowWidth];
	delete[] tRatio;	tRatio = new Fixed[windowHeight];

	windowChanged = 0;
	}


// Main program

int main(int argc, char **argv)
	{
	// Display banner and show input device status.

	cout << "Zox3D demo version 1.5, by Robert Schmidt <robert@stud.unit.no>" << endl;
	cout << "(C) Copyright 1993 of Ztiff Zox Softwear" << endl << endl;

	cout << "Mouse is ";
	if (!mouse.isInstalled())
		cout << "NOT ";
	cout << "ready." << endl;

	cout << "Joystick A is ";
	if (!joystickA.isPresent())
		cout << "NOT ready.";
	else
		cout << "ready.  To calibrate and use, press J when in demo.";
	cout << endl << endl;
	cout << "If an assertion occurs, you probably don't have enough free memory.";
	cout << endl << endl;
	Gamecard::reset(FALSE);				// disable gamecard
	joystickA.calibrate();				// disable joystick

	// Present a menu of available unchained 256-color modes.

	int mode = argc>1 ? atoi(argv[1]) : selectTweakMode();

	// Load the Ztiff Zox logo bitmap, show it, then deallocate its memory.

	IffBitMap *i = new IffBitMap("zz.lbm");
	i->showMode13h();
	delete i;

	// Ready the map array.

	initMap();

	// Load the resources (bitmaps) described in a text file.

	ifstream *resFile = new ifstream("resource.dat");
	resTab = new ResourceTable(*resFile);
	assert(resTab);
	delete resFile;

	// Load and prepare the cloudy sky image for scrolling, and get a direct
	// pointer to the floor bitmap data.  (Because the floor has only one
	// bitmap which is being tiled, we can avoid accessing the resource
	// table each time we want to draw a piece of floor.)

	initSky();
	floorData = (*resTab)[8].getData();

	// Let the user view the pretty logo for a second more, then set the
	// selected video mode and adjust the screen- and window-related 
	// variables to the resolution of the mode.

	delay(1000);
	setTweakMode(mode);
	windowWidth = screenWidth>>0;
	windowHeight = screenHeight>>0;
	adjustToWindow();

	// Display the word presents in the middle of the screen, and wait a 
	// second more.

	char *presents = "present";
	gprintf((screenWidth-8*strlen(presents))/2, screenHeight/2-4, LIGHTCYAN,
		presents);
	delay(1000);

	// Load and set the default palette

	IffBitMap *pal = new IffBitMap("pal.lbm");
	assert(pal);
	pal->cmap.out(ColorMap::C256X3);
	delete pal;

	// Position the player on the map.

	initPlayer();

	// Select which pages to be used.  We use 0 and 1, as all modes are
	// guaranteed to have those two pages.

	inactivePage = 0 * pageSize;
	activePage = 1 * pageSize;

	// Initialize a timer to measure the frame rate.

	LZTimer timer;				// make a timer object to measure frame rate

#ifdef POLL_KEYS
	// Disable interrupt 09h chaining if Zox3D was compiled with
	// a keyboard interrupt handler.  (This is to avoid the beeps from the
	// BIOS when its buffer fills up with unread key scan codes.)

	keyboard.setChainState(FALSE);
#endif

	// This main loop ends when the player presses ESC to quit.

	while (!playerQuit)
		{
		// Restart the timer.
		timer.restart();

		SHOWMAP(guyX, guyY, 15);

		// blockGuyX, blockGuyY contains which map square is occupied by
		// the player.

		Fixed blockGuyX = guyX & BLOCKSIZE_MASK;
		Fixed blockGuyY = guyY & BLOCKSIZE_MASK;

		// If the player has changed the size of the window, readjust to
		// the new dimensions after clearing the video memory.

		if (windowChanged)
			{
			setPlane(0x0f);
			memset(vga, 0, 0xffff);
			adjustToWindow();
			prevAngle = guyAngle-1;	// force recalc. of angle dependant vars
			prevElev = guyElev-1;	// force recalc. of height dependant vars
			}

		if (guyAngle != prevAngle)
			{
			// The player's angle has changed - precalculate some useful
			// values and tables.

			cosGuy = FixCos(guyAngle);
			sinGuy = FixSin(guyAngle);

			// (iPixelX,iPixelY) is the vector from the eye to the center
			// of the screen, in world coordinates.

			Fixed iPixelX, iPixelY;
			FixMul(eyeScreenDist, cosGuy, iPixelX);
			FixMul(eyeScreenDist, sinGuy, iPixelY);

			// (preFactorX,preFactorY) are constants in the following loop.

			Fixed preFactorX, preFactorY;
			FixMul(cosGuy, eyeScreenDist, preFactorX);
			FixMul(sinGuy, eyeScreenDist, preFactorY);

			// Now calculate some tables indexed by the screen column.
			// Start at the left (iScreenX is 1/2 the width of the screen
			//  in world coords).

			Fixed screenX = iScreenX;
			column = 0;
			while (column < windowWidth)
				{

				// tFactorX[column] and tFactorY[column] are used when
				// drawing the floor and ceiling.  See WSLICE.CPP.

				Fixed factor = preFactorX;
				FixMulAdd(sinGuy, screenX, factor);
				tFactorX[column] = factor;
				factor = preFactorY;
				FixMulAdd(-cosGuy, screenX, factor);
				tFactorY[column] = factor;

				// tPixelX[column] and tPixelY[column] is the vector from
				// the center of the screen to the current column, in
				// world coords.

				FixMul(screenX, sinGuy, pixelX);
				tPixelX[column] = (pixelX += iPixelX);
				FixMul(-screenX, cosGuy, pixelY)
				tPixelY[column] = (pixelY += iPixelY);

				// tdhX[column] contains the distance to step in the x
				// direction when tracing from one horizontal wall to the
				// next, for this column.
				// tdvY[column] is ditto for y and vertical walls.
				//
				// Keep in mind that dhY and dvX are constants (in absolute
				// values), so there's no need to precalculate them.  In
				// fact: dhY = SIGN(dvY), dvX = SIGN(dhX).
				//
				// If a division by zero fault occurs for either horizontal
				// or vertical walls, we flag that tracing should not occur 
				// along those walls in tDoTrace[column].

				div0Clear();
				FixDiv(pixelX, FixAbs(pixelY), dhX);
				tdhX[column] = dhX;
				doHorz = !div0Error();
				FixDiv(pixelY, FixAbs(pixelX), dvY)
				tdvY[column] = dvY;
				doVert = !div0Error();
				tDoTrace[column] = doHorz | doVert<<1;

				// Now look at the next screen column.

				screenX += screenToWorld;
				++column;
				}
			prevAngle = guyAngle;
			}

		if (guyElev != prevElev)
			{

			// Player's elevation has changed, so recalculate some tables
			// and variables dependant on this.

			prevElev = guyElev;

			// guyElevScreen is the player's elevation in screen coords,
			// which is used when calculating the wall heights. (WSLICE.CPP)

			FixMul(scaleHeight, guyElev, guyElevScreen);
			FixDiv(guyElevScreen, BLOCKSIZE_WORLD, guyElevScreen);

			if (features & (DRAW_SKY | DRAW_FLOOR))
				{

				// The player has enabled floor/ceiling, so calculate the
				// depth ratio for each y on screen.

				int count;
				Fixed screenY, *skyRatio, *floorRatio;

				// guyElev is the distance from the top of the walls to the
				// eye.  To get distance from floor to eye, do:

				Fixed floorDist = BLOCKSIZE_WORLD - guyElev;

				// Now find a suitable distance to the sky.  Since it must
				// appear to be high above the maze, add 8 times the wall
				// height to guyElev.

				Fixed skyDist = (BLOCKSIZE_WORLD<<3) + guyElev;

				// This loop calculates the depth ratio for each y.

				count = windowHeight >> 1;
				screenY = screenToWorld;
				floorRatio = tRatio + count;
				skyRatio = floorRatio - 1;

				while (count--)
					{
					FixDiv( floorDist, screenY, *(floorRatio++) );
					FixDiv( skyDist, screenY, *(skyRatio--) );
					screenY += screenToWorld;
					}
				}
			}

		// Trace a ray from eye through each vertical pixel column in
		// the view window.

		// Start tracing at the left of the window:

		column = 0;
		vgaBuf = windowTopLeft + activePage; // + (column >> 2)

		while (column < windowWidth)
			{
			// Calculate the position of the screen pixel relative
			// to the eye, in world coordinates.  Here we use the
			// precalculated data from above.

			pixelX = tPixelX[column];
			pixelY = tPixelY[column];
			dhX = tdhX[column];
			dvY = tdvY[column];
			doHorz = tDoTrace[column] & 1;
			doVert = tDoTrace[column] & 2;

			SHOWMAP(guyX+pixelX, guyY+pixelY, 12);

			// We know dhX and dvY, so find dhY and dvX.  If either
			// horizontal or vertical walls are not to be traced,
			// adjust the delta variables accordingly.

			// We also calculate (hX,hY) and (vX,vY), which are the ray's
			// starting positions on the horizontal and vertical walls
			// (respectively), directly behind the player.

			if (!doHorz)
				{
				if (pixelX < 0)
					{
					dhX = -MAXFIX;
					dvX = -1;
					}
				else
					{
					dhX = MAXFIX;
					dvX = 1;
					}
				}

			if (doVert)
				{
				vX = FIX2INT(guyX);
				FixMul(blockGuyX, dvY, vY)

				if (dhX < 0)
					{
					++vX;
					vY += guyY - dvY;
					dvX = -1;
					}
				else
					{
					vY = guyY - vY;
					dvX = 1;
					}
				}
			else
				{
				if (pixelY < 0)
					{
					dvY = -MAXFIX;
					dhY = -1;
					}
				else
					{
					dvY = MAXFIX;
					dhY = 1;
					}
				}

			if (doHorz)
				{
				hY = FIX2INT(guyY);
				FixMul(blockGuyY, dhX, hX)

				if (dvY < 0)
					{
					++hY;
					hX += guyX - dhX;
					dhY = -1;
					}
				else
					{
					hX = guyX - hX;
					dhY = 1;
					}
				}

			// Now, (hX,hY) is the first point on a horizontal wall
			// *behind* the player, on the line between his eye and the
			// current screen column.  Ditto for (vX,vY) and vertical walls.

			// Determine dominant direction.  The dominant direction
			// let us decide whether we mainly step along horizontal or
			// vertical walls.

			hDominant = FixAbs(dhX) > FixAbs(dvY);

			// Now get ready for the real tracing thing.

			// Initially, we trace directly to the screen, but if we during
			// the tracing hit a transparent bitmap or a mirror, we will
			// switch over to a temporary buffer to minimize the number of
			// accesses to the video memory.  This is handled in TRACE.CPP.

			directTrace = 1;
			activeBuf = vgaBuf;
			bufStepY = screenWidthBytes;
			setPlane(1<<(column&3));

			// mirrorDepth is the number of mirrors encountered during the
			// tracing.  Note that because of recursion, the same mirror
			// may count several times.

			mirrorDepth = 0;

			// traceLevel is the depth of tracing recursion.  The next depth
			// is entered if a mirror or transparent wall is hit.

			traceLevel = -1;

			// (virtualGuyX,virtualGuyY) is the player's position as seen
			// from the current ray tracing point, taking into account
			// all the mirrors encountered.

			virtualGuyX = guyX;
			virtualGuyY = guyY;

			// mirroredX or mirroredY is toggled each time the ray is mirrored
			// in the x and y direction, respectively.

			mirroredX = mirroredY = 0;

			// This kludge resets the floor's bottom y and the ceiling's
			// top y to the bottom and top of the screen, respectively.

			WallSlice::resetHither();

			// And here we go, selecting an appropriate tracing function
			// in accordance to which direction is dominant, horizontal
			// or vertical.  See TRACE.CPP for next episode.

			if (hDominant)
				traceHorz(FixAbs(INT2FIX(vX)-guyX) >= FixAbs(hX-guyX));
			else
				traceVert(FixAbs(INT2FIX(hY)-guyY) >= FixAbs(vY-guyY));

			// ...and we're back again.  Advance to the next column on the
			// screen, taking care of the weirdness of unchained 256-color
			// mode x-coordinates.

			if ((++column & 0x03) == 0)
				vgaBuf++;
			}


		// Draw help message.

		if (features & SHOW_HELP)
			{
			gprintf(8, 8, YELLOW, "Zox3D demo version 1.5");
			gprintf(8, 16, RED, "Robert Schmidt, 1993");
			gprintf(8, 32, LIGHTMAGENTA, "ESC: quit demo");
			gprintf(8, 40, LIGHTMAGENTA, "  H: toggle this help");
			gprintf(8, 48, LIGHTMAGENTA, "  F: toggle floor");
			gprintf(8, 56, LIGHTMAGENTA, "  S: toggle sky");
			gprintf(8, 72, LIGHTMAGENTA, "  J: calibrate joystick");
			gprintf(8, 64, LIGHTMAGENTA, ",/.: change window size");
			gprintf(8, 82, LIGHTMAGENTA, "Use %c,%c,%c and %c, or your",
				24,25,26,27);
			gprintf(8, 90, LIGHTMAGENTA, "mouse to move around!");
			gprintf(8, 98, LIGHTMAGENTA, "Keypad -/+: move up/down");
			}
		else
			gprintf(screenWidth-86, screenHeight-15, YELLOW,
				"H for help");

		// Process player commands.

		processPlayer();

		// Scroll the arrow wall bitmap, by simply moving the pointer one
		// row back in the bitmap.  If we have reached the start of the first
		// copy of the bitmap, jump ahead to the start of the second again.

		if (winBitmap->getData() == winLogoData)
			winBitmap->getData() =
				winLogoData + winBitmap->getSize() - winBitmap->getHeight();
		else
			winBitmap->getData() -= winBitmap->getHeight();

		// Move the sky.  The sky is 256x256 pixels, so adding 257 will
		// effectively move it diagonally.  If all scrolling was this easy!

		skyData += 257;

		// Flip the "flash" bitmap around, by toggling bit 7 in the bitmap 
		// number.  See MAP.HPP and MAP.CPP for a description of this bit.

		map[6][4].horz.bmap ^= 0x80;

		// Set the visible page to the page that has just been drawn.

		setStartAdress(activePage);

		// Calculate frames per second and print on screen.

		timer.stop();
		gprintf(8,screenHeight-15,LIGHTRED,"%05.2f fps",
			float(timer.resolution())/timer.count());

		// Swap the active and inactive page addresses.

		swap(activePage, inactivePage);

		// Clear the area in which to print the frame rate, in case the
		// 3D window is shrinked.  The screen is never cleared, so we
		// don't want the fps rates to be superimposed on the previous
		// figure, which would make it unreadable.

		box(7,screenHeight-16,74,10,BLACK);
		}

	// We're leaving, so clean up properly.

	delete resTab;
	farfree(skyBase);
	delete[] tempBuf;
	delete[] tempBuf;
	delete[] tPixelX;
	delete[] tPixelY;
	delete[] tdhX;
	delete[] tdvY;
	delete[] tDoTrace;
	delete[] tFactorX;
	delete[] tFactorY;
	delete[] tRatio;

	// Set text mode 80x25.

	setMode(3);

	return 0;
	}
