;=====================================================================
; Fixed point routines.  Based on code by Michael Abrash.


MUL_ROUNDING_ON EQU     0       ;1 for rounding on multiplies,
				; 0 for no rounding. Not rounding is faster,
				; rounding is more accurate and generally a
				; good idea

DIV_ROUNDING_ON EQU     0       ;1 for rounding on divides,
				; 0 for no rounding. Not rounding is faster,
				; rounding is more accurate, but because
				; division is only performed to project to
				; the screen, rounding quotients generally
				; isn't necessary

ALIGNMENT       EQU     2	; 4 does not work - no idea why not

	IDEAL
	MODEL LARGE, C
	P386

	CODESEG

	PUBLIC	C	FixSqrt, UFixSqrt, FixSqr, UFixSqr
	PUBLIC	C	FixMul, FixDivRound, FixCos, FixSin



;=====================================================================
; FixSqrt calculates the square root of a fixed point value.
; It hasn't been tested much, but I think it works.  The algorithm
; is too clever for me.  See ZOX3D.DOC for credits to those who
; helped me with this (while I thought I needed it - it is never
; used in Zox3D!).

		ALIGN   ALIGNMENT
PROC	FixSqrt 	X:DWORD
	mov		ecx, [X]
	mov		ebx, 40000000h	; b = 0x40...
	xor		eax, eax	; q = 0
@@1:
	or		ebx, ebx	; while(b>=256)
	jz		short @@2
	mov		edx, eax	; t = q
	add		edx, ebx	; t = q+b
	cmp		ecx, edx	; if (r>=t)
	jb		short @@3
	sub		ecx, edx	; r -= t
	mov		eax, edx	; q = t
	add		eax, ebx	; q = t+b
@@3:
	shl		ecx, 1		; r *= 2
	shr		ebx, 1		; b /= 2
	jmp		@@1
@@2:
	mov		edx, eax
	shr		eax, 8
	shr		edx, 24
	ret
ENDP	FixSqrt


;=====================================================================
; UFixSqrt calculates the square root of an unsigned fixed point value.
; It hasn't been tested much, and I do NOT think that it works.

		ALIGN	ALIGNMENT
PROC	UFixSqrt	  X:DWORD
	mov		ecx, [X]
	mov		ebx, 80000000h	; b = 0x40...
	xor		eax, eax	; q = 0
@@1:
	or		ebx, ebx	; while(b>=256)
	jz		short @@2
	mov		edx, eax	; t = q
	add		edx, ebx	; t = q+b
	cmp		ecx, edx	; if (r>=t)
	jb		short @@3
	sub		ecx, edx	; r -= t
	mov		eax, edx	; q = t
	add		eax, ebx	; q = t+b
@@3:
	shl		ecx, 1		; r *= 2
	shr		ebx, 1		; b /= 2
	jmp		@@1
@@2:
	mov		edx, eax
	shr		eax, 8
	shr		edx, 24
	ret
ENDP	UFixSqrt


;=====================================================================
; FixSqr finds the square of a fixed point value, by multiplying it with
; itself.


	ALIGN   ALIGNMENT
PROC	FixSqr  X:DWORD
	mov     eax,[X]
	imul    eax			; multiply
if MUL_ROUNDING_ON
	add     eax,8000h       	; round by adding 2^(-17)
	adc     edx,0           	; whole part of result is in DX
endif ;MUL_ROUNDING_ON
	shr     eax,16          	; Put the fractional part in AX
	ret
ENDP	FixSqr



;=====================================================================
; UFixSqr finds the square of an unsigned fixed point value, by
; multiplying it with itself.


	ALIGN   ALIGNMENT
PROC	UFixSqr  X:DWORD
	mov     eax,[X]
	mul     eax			; multiply
if MUL_ROUNDING_ON
	add     eax,8000h       	; round by adding 2^(-17)
	adc     edx,0           	; whole part of result is in DX
endif ;MUL_ROUNDING_ON
	shr     eax,16          	; put the fractional part in AX
	ret
ENDP	UFixSqr




;=====================================================================
; FixMul multiplies two fixed-point values together.

	ALIGN   ALIGNMENT
PROC	FixMul  M1:DWORD, M2:DWORD
	mov     eax,[M1]
	imul    [dword ptr M2] 		; multiply
if MUL_ROUNDING_ON
	add     eax,8000h       	; round by adding 2^(-17)
	adc     edx,0           	; whole part of result is in DX
endif ;MUL_ROUNDING_ON
	shr     eax,16          	; put the fractional part in AX
	ret
ENDP	FixMul





;=====================================================================
; FixDiv divides one fixed-point value by another.

	ALIGN   ALIGNMENT
PROC	FixDivRound  Dividend:DWORD, Divisor:DWORD

if DIV_ROUNDING_ON
	sub     cx,cx           	; assume positive result
	mov     eax,[Dividend]
	and     eax,eax         	; positive dividend?
	jns     FDP1            	; yes
	inc     cx              	; mark it's a negative dividend
	neg     eax             	; make the dividend positive

FDP1:   sub     edx,edx         	; make it a 64-bit dividend, then shift
					;  left 16 bits so that result will be
					;  in EAX
	rol     eax,16          	; put fractional part of dividend in
					;  high word of EAX
	mov     dx,ax           	; put whole part of dividend in DX
	sub     ax,ax           	; clear low word of EAX
	mov     ebx,[DWORD PTR Divisor]
	and     ebx,ebx         	; positive divisor?
	jns     FDP2            	; yes
	dec     cx              	; mark it's a negative divisor
	neg     ebx             	; make divisor positive

FDP2:   div     ebx             	; divide
	shr     ebx,1           	; divisor/2, minus 1 if the divisor is
	adc     ebx,0           	;  even
	dec     ebx
	cmp     ebx,edx         	; set Carry if the remainder is at least
	adc     eax,0           	;  half as large as the divisor, then
					;  use that to round up if necessary
	and     cx,cx           	; should the result be made negative?
	jz      FDP3            	;  no
	neg     eax             	;  yes, negate it

FDP3:
else ;!DIV_ROUNDING_ON
	mov     edx,[Dividend]
	sub     eax,eax
	shrd    eax,edx,16      	; position so that result ends up
	sar     edx,16          	;  in EAX
	mov	ebx, [DWORD PTR Divisor]
	idiv    ebx
endif ;DIV_ROUNDING_ON
	shld    edx,eax,16      	; whole part of result in DX;
					;  fractional part is already in AX
	ret
ENDP	FixDivRound


;=====================================================================
; FixSin and FixCos returns the sine and cosine of an angle, respectively.
;
; The angle parameter is an unsigned word, but we the functions are
; designed so that we map the word onto the actual resolution of the
; trigonometry tables.  Thus 0 is always the minimal angle, and 0xffff
; is always the maximal angle.  This way, no checks for overflow is
; neccessary when angles are manipulated.
;
; Currently the tables contain 1024 sine and 1024 cosine fixed point
; values, so only the 10 upper bits of the angle parameter is used.


	ALIGN   ALIGNMENT

LABEL	Tables	DWORD

	INCLUDE "trigtab.inc" 		; include the trigonometry tables
					;  for sin & cos produced by
					;  TRIGTAB.CPP.

	ALIGN   ALIGNMENT

PROC	FixCos	Angle: WORD
	mov	bx, [Angle]		; get the angle word
	and	bx, 0ffc0h		; mask away the lower bits
	shr	bx, 6 - 2		; shift it into position for lookup
	mov	eax, [Tables+bx]	; get the fixed-point cosine value
	shld	edx, eax, 16		; we have to return dx:ax, so shift
					;  eax into dx
	ret
ENDP	FixCos

PROC	FixSin	Angle: WORD		; (same comments as for FixCos)
	mov	bx, [Angle]
	and	bx, 0ffc0h
	shr	bx, 6 - 2
	mov	eax, [Tables+4*1024+bx]
	shld	edx, eax, 16
	ret
ENDP	FixSin


	END
