#include "pch.h"
#include <imagehlp.h>
#include <malloc.h>
#include <crtdbg.h>

//inserts no-mans-land before/after each allocation, then cehcks
//on deallocation to ensure lack of overwrites.
//#define twDEBUG_CHECK_GAPS

//get full stack trace rather than just line and file.
//#define twDEBUG_STACK_TRACE

//use the debug heap
//#define twDEBUG_USE_TW_HEAP

//log heap activity
//#define twDEBUG_ALLOC_LOG

#if (defined UNICODE)||(defined _UNICODE)
#error "Ascii only, sorry, I can't be arsed"
//excessive bracketing due to removal of all _T() macros.
#endif

#if (defined twDEBUG_USE_TW_HEAP)&&(defined twIMAGEHLP_OK)&&(defined _DEBUG)
static unsigned tw_alloc_count=0;

#ifdef twDEBUG_ALLOC_LOG
static FILE *alloc_log=0;

static void twLog(const char *fmt,...) {
	if(alloc_log) {
		va_list v;
		va_start(v,fmt);
		vfprintf(alloc_log,fmt,v);
		va_end(v);
		fflush(alloc_log);
	}
}
#endif

//////////////////////////////////////////////////////////////////////////
static const unsigned BUFFERSIZE=512;
static bool tw_debug_initialised=false;

int twDebugStart(const char *szUserSymbolPath) {
	char szSymbolPath[BUFFERSIZE];
	DWORD dwSymOptions=SymGetOptions(); 
	dwSymOptions|=SYMOPT_LOAD_LINES; 
	dwSymOptions&=~SYMOPT_UNDNAME;
	SymSetOptions(dwSymOptions);
	
	char szPath[BUFFERSIZE];
	
	if (szUserSymbolPath) {
		strcpy(szSymbolPath,szUserSymbolPath);
		strcat(szSymbolPath,";.");
	} else {
		strcpy(szSymbolPath,".");
	}
	
	// environment variable _NT_SYMBOL_PATH
	
	if (GetEnvironmentVariableA("_NT_SYMBOL_PATH",szPath,BUFFERSIZE)){
		strcat(szSymbolPath,";");
		strcat(szSymbolPath,szPath);
	}
	
	// environment variable _NT_ALTERNATE_SYMBOL_PATH
	
	if (GetEnvironmentVariableA("_NT_ALTERNATE_SYMBOL_PATH",szPath,BUFFERSIZE)){
		strcat(szSymbolPath,";");
		strcat(szSymbolPath,szPath);
	}
	
	// environment variable SYSTEMROOT
	
	if (GetEnvironmentVariableA("SYSTEMROOT",szPath,BUFFERSIZE)){
		strcat(szSymbolPath,";");
		strcat(szSymbolPath,szPath);
		strcat(szSymbolPath,";");
		
		// SYSTEMROOT\System32
		
		strcat(szSymbolPath,szPath);
		strcat(szSymbolPath,"\\System32");
	}
	
	if(GetModuleFileNameA(GetModuleHandle(0),szPath,BUFFERSIZE)){
		char *slash=strrchr(szPath,'\\');
		if(!slash) {
			slash=strrchr(szPath,'/');
		}
		if(slash) {
			*slash=0;
			strcat(szSymbolPath,";");
			strcat(szSymbolPath,szPath);
		}
	}

	int df_old=_CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
	df_old|=_CRTDBG_LEAK_CHECK_DF;
//	df_old|=_CRTDBG_CHECK_ALWAYS_DF;
	_CrtSetDbgFlag(df_old);

#ifdef twDEBUG_ALLOC_LOG
	alloc_log=fopen("allocs.txt","wt");
#endif
	
	return SymInitialize( GetCurrentProcess(), szSymbolPath, TRUE);
}

void twDebugStop() {
	SymCleanup(GetCurrentProcess());
}

void __cdecl DumpClientDumpMemoryLeaks(void *, size_t);

void twDebugEnsureInitialised() {
	if(!tw_debug_initialised) {
		twDebugStart(0);
		tw_debug_initialised=true;
		//make the client_block allocation (didn't work, dump client reset by an
		//atexit function!)
//		_malloc_dbg(16,_CLIENT_BLOCK,"ignore me",-1);
//		_CrtSetDumpClient(&DumpClientDumpMemoryLeaks);
		wxLog::GetActiveTarget();
	}
}

bool twDebugGetModuleNameFromAddress(unsigned address,char *module_name,unsigned module_name_size) {
	IMAGEHLP_MODULE moduleInfo;
	
	ZeroMemory(&moduleInfo,sizeof(moduleInfo));
	moduleInfo.SizeOfStruct=sizeof(moduleInfo);
	if(SymGetModuleInfo(GetCurrentProcess(),(DWORD)address,&moduleInfo)) {
		_snprintf(module_name,module_name_size,"%s",moduleInfo.ModuleName);
	} else {
		_snprintf(module_name,module_name_size,"?");
	}
	return true;
}

// Get function prototype and parameter info from ip address and stack address
bool twDebugGetFunctionInfoFromAddresses(ULONG fnAddress,ULONG stackAddress,char *lpszSymbol) {
	bool ret=false;
	DWORD dwDisp = 0;
	DWORD dwSymSize = 10000;
	char lpszUnDSymbol[BUFFERSIZE]="?";
	CHAR lpszNonUnicodeUnDSymbol[BUFFERSIZE]="?";
	char *lpszParamSep = NULL;
	const char *lpszParsed = lpszUnDSymbol;
	PIMAGEHLP_SYMBOL pSym =(PIMAGEHLP_SYMBOL)GlobalAlloc( GMEM_FIXED, dwSymSize );
	
	::ZeroMemory( pSym, dwSymSize );
	pSym->SizeOfStruct = dwSymSize;
	pSym->MaxNameLength = dwSymSize - sizeof(IMAGEHLP_SYMBOL);
	
	// Set the default to unknown
	strcpy( lpszSymbol, "?" );
	
	// Get symbol info for IP
	if ( SymGetSymFromAddr( GetCurrentProcess(), (ULONG)fnAddress, &dwDisp, pSym ) )
	{
		// Make the symbol readable for humans
		UnDecorateSymbolName( pSym->Name, lpszNonUnicodeUnDSymbol, BUFFERSIZE, 
			UNDNAME_COMPLETE | 
			UNDNAME_NO_THISTYPE |
			UNDNAME_NO_SPECIAL_SYMS |
			UNDNAME_NO_MEMBER_TYPE |
			UNDNAME_NO_MS_KEYWORDS |
			UNDNAME_NO_ACCESS_SPECIFIERS );
		
		// Symbol information is ANSI string
		strcpy( lpszNonUnicodeUnDSymbol, lpszUnDSymbol );
		
		// I am just smarter than the symbol file :)
		if ( strcmp(lpszUnDSymbol, ("_WinMain@16")) == 0 ) {
			strcpy(lpszUnDSymbol, ("WinMain(HINSTANCE,HINSTANCE,LPCTSTR,int)"));
		} else if ( strcmp(lpszUnDSymbol, ("_main")) == 0 ) {
			strcpy(lpszUnDSymbol, ("main(int,TCHAR * *)"));
		} else if ( strcmp(lpszUnDSymbol, ("_mainCRTStartup")) == 0 ) {
			strcpy(lpszUnDSymbol, ("mainCRTStartup()"));
		} else if ( strcmp(lpszUnDSymbol, ("_wmain")) == 0 ) {
			strcpy(lpszUnDSymbol, ("wmain(int,TCHAR * *,TCHAR * *)"));
		} else if ( strcmp(lpszUnDSymbol, ("_wmainCRTStartup")) == 0 ) {
			strcpy(lpszUnDSymbol, ("wmainCRTStartup()"));
		}
						
		lpszSymbol[0] = ('\0');
		
		// Let's go through the stack, and modify the function prototype, and insert the actual
		// parameter values from the stack
		if ( strstr( lpszUnDSymbol, ("(void)") ) == NULL && strstr( lpszUnDSymbol, ("()") ) == NULL)
		{
			ULONG index = 0;
			for( ; ; index++ )
			{
				lpszParamSep = strchr( lpszParsed, (',') );
				if ( lpszParamSep == NULL )
					break;
				
				*lpszParamSep = ('\0');
				
				strcat( lpszSymbol, lpszParsed );
				sprintf( lpszSymbol + strlen(lpszSymbol), ("=0x%08X,"), *((ULONG*)(stackAddress) + 2 + index) );
				
				lpszParsed = lpszParamSep + 1;
			}
			
			lpszParamSep = strchr( lpszParsed, (')') );
			if ( lpszParamSep != NULL )
			{
				*lpszParamSep = ('\0');
				
				strcat( lpszSymbol, lpszParsed );
				sprintf( lpszSymbol + strlen(lpszSymbol), ("=0x%08X)"), *((ULONG*)(stackAddress) + 2 + index) );
				
				lpszParsed = lpszParamSep + 1;
			}
		}
		
		strcat( lpszSymbol, lpszParsed );
		
		ret = TRUE;
	}
	
	GlobalFree( pSym );
	
	return ret;
}

bool twDebugGetLineAndFileFromAddress(DWORD addr,char *file,int &line) {
	IMAGEHLP_LINE line_info;
	DWORD disp;
	if(!SymGetLineFromAddr(GetCurrentProcess(),addr,&disp,&line_info)) {
		strcpy(file,"?");
		line=-1;
		return false;
	}
	line=line_info.LineNumber;
	strcpy(file,line_info.FileName);
	return true;
}

bool twDebugGetSourceInfoFromAddress(UINT address,LPTSTR lpszSourceInfo)
{
	bool ret=false;
	IMAGEHLP_LINE  lineInfo;
	DWORD          dwDisp;
	TCHAR          lpszFileName[BUFFERSIZE] = ("");
	TCHAR          lpModuleInfo[BUFFERSIZE] = ("");
	
	strcpy( lpszSourceInfo, ("?(?)") );
	
	::ZeroMemory( &lineInfo, sizeof( lineInfo ) );
	lineInfo.SizeOfStruct = sizeof( lineInfo );
	
	if ( SymGetLineFromAddr( GetCurrentProcess(), address, &dwDisp, &lineInfo ) )
	{
		// Got it. Let's use "sourcefile(linenumber)" format
		//PCSTR2LPTSTR( lineInfo.FileName, lpszFileName );
		//_stprintf( lpszSourceInfo, ("%s(%d)"), lpszFileName, lineInfo.LineNumber );
		sprintf(lpszSourceInfo,"%s(%d)",lineInfo.FileName,lineInfo.LineNumber);
		ret = true;
	}
	else
	{
		// There is no source file information. :(
		// Let's use the "modulename!address" format
		twDebugGetModuleNameFromAddress( address, lpModuleInfo ,BUFFERSIZE);
		
		if ( lpModuleInfo[0] == ('?') || lpModuleInfo[0] == ('\0')) {
			sprintf(lpszSourceInfo,"0x%08X",address);
		} else {
			sprintf(lpszSourceInfo,"%s!0x%08X",lpModuleInfo,address);
		}
//			// There is no modulename information. :((
//			// Let's use the "address" format
//			_stprintf( lpszSourceInfo, ("0x%08X"), lpModuleInfo, address );
//		else
//			_stprintf( lpszSourceInfo, _T("%s!0x%08X"), lpModuleInfo, address );
		
		ret = false;
	}
	
	return ret;
}

static void twDebugShortenStlSymbol(char *str) {
	char tmp[BUFFERSIZE];
	static const char *szTruncateSymbols[]={
		"map",
		"vector",
		"string",
		"allocator",
		"deque",
		"basic_string",
	};
	static const char uNumTruncateSymbols=sizeof szTruncateSymbols/sizeof szTruncateSymbols[0];
	
	//std:: (5 chars)
	//std::_T< (8 chars, smallest symbol, >8 because template parameters follow)
	if(strnicmp(str,"std::",5)==0&&strlen(str)>8) {
		if(char *lt=strchr(&str[5],'<')) {
			bool truncate=false;
			
			//Copy \1 from std\:\:(.*)\< to tmp
			unsigned n=lt-(str+5);
			strncpy(tmp,&str[5],n);
			tmp[n]=0;
			
			//System symbol?
			if(tmp[0]=='_'&&isupper(tmp[1])) {
				truncate=true;
			} else {
				for(unsigned i=0;i<uNumTruncateSymbols;++i) {
					if(stricmp(tmp,szTruncateSymbols[i])==0) {
						truncate=true;
						break;
					}
				}
			}
			if(truncate) {
				strcpy(lt,"<...>");
			}
		}
	}
}

static int StrCountCopy(char *szDest,const char *szSrc,int iDestSize) {
	char *szDestPtr=szDest;
	int iCount=0;
	while (*szSrc) {
		if (iCount>=iDestSize-1) {
			*szDestPtr=0;
			return iCount;
		}
		*szDestPtr++=*szSrc++;
		iCount++;
	}
	*szDestPtr=0;
	return iCount;
}

void twDebugGetStackTrace(char *szBuff,int iBufferSize)//,FormatEnum eFormat)
{
	int i;
	HANDLE hThread=GetCurrentThread();
	LPCTSTR lpszMessage="";
	STACKFRAME     callStack;
	BOOL           bResult;
	CONTEXT        context;
	char           szTempBuff[BUFFERSIZE];
	HANDLE         hProcess = GetCurrentProcess();

	*szBuff=0;
	char *szBuffPtr=szBuff;
	const char *szSep;
	
	szSep="\n";
	*szBuffPtr++='\n';
	
	::ZeroMemory( &context, sizeof(context) );
	context.ContextFlags = CONTEXT_FULL;

	if ( !GetThreadContext( hThread, &context ) ) {
	   return;
	}
	
	::ZeroMemory( &callStack, sizeof(callStack) );
	callStack.AddrPC.Offset    = context.Eip;
	callStack.AddrStack.Offset = context.Esp;
	callStack.AddrFrame.Offset = context.Ebp;
	callStack.AddrPC.Mode      = AddrModeFlat;
	callStack.AddrStack.Mode   = AddrModeFlat;
	callStack.AddrFrame.Mode   = AddrModeFlat;

	for (i=0 ; ; i++) 
	{
		bResult = StackWalk(
			IMAGE_FILE_MACHINE_I386,
			hProcess,
			hThread,
	      &callStack,
			NULL, 
			NULL,
			SymFunctionTableAccess,
			SymGetModuleBase,
			NULL);

		if (i==0)
		{
		   continue;
		}

		if( !bResult || callStack.AddrFrame.Offset == 0 )
		{
			break;
		}

		twDebugGetSourceInfoFromAddress(callStack.AddrPC.Offset,szTempBuff);
//		szBuffPtr+=StrCountCopy(szBuffPtr,GetFileName(szTempBuff),iBufferSize-(szBuffPtr-szBuff));
		szBuffPtr+=StrCountCopy(szBuffPtr,szTempBuff,iBufferSize-(szBuffPtr-szBuff));
		szBuffPtr+=StrCountCopy(szBuffPtr,":",iBufferSize-(szBuffPtr-szBuff));

		//Determine whether this is a standard library symbol. If it is, see if it begins with
		//'_' followed by a capital letter. Identifiers of this kind are reserved for the system
		//so the template parameters (anything following the first '<') can be removed; they aren't
		//particularly interesting and take up _loads_ of space.
		twDebugGetFunctionInfoFromAddresses(callStack.AddrPC.Offset,callStack.AddrFrame.Offset,szTempBuff);
		
		twDebugShortenStlSymbol(szTempBuff);

		szBuffPtr+=StrCountCopy(szBuffPtr,szTempBuff,iBufferSize-(szBuffPtr-szBuff));
		szBuffPtr+=StrCountCopy(szBuffPtr,szSep,iBufferSize-(szBuffPtr-szBuff));
		// See if the buffer is full.

		if ((szBuffPtr-szBuff)==iBufferSize-1)
		{
			break;
		}

	}
	if ((szBuffPtr-szBuff)==iBufferSize-1)
	{
		szBuffPtr[-3]='.';
		szBuffPtr[-2]='.';
		szBuffPtr[-1]='.';
	}
	*szBuffPtr=0;
}

//////////////////////////////////////////////////////////////////////////
// Allocation functions

struct twAllocation {
	//How much the user requested
	unsigned char *user_base;
	unsigned user_size;

	//front gap is &user_base[-gap_size]
	//back gap is &user_base[user_size]
	int gap_size;

	//base + size of the allocation as the RTL sees it
	unsigned char *rtl_base;
	unsigned rtl_size;

	//where it was allocated
#ifdef twDEBUG_STACK_TRACE
	char stack_trace[2048];
#else
	char file[MAX_PATH];
	int line;
#endif
//	const char *file;
//	int line;

	void PrintName() const;
	int CheckGap(unsigned char *start) const;
	bool AreGapsValid() const;
	unsigned char *FrontGap() const;
	unsigned char *BackGap() const;
};

void twAllocation::PrintName() const {
#ifdef twDEBUG_STACK_TRACE
	OutputDebugStringA(this->stack_trace);
#else
	static char numbuf[20];
	_snprintf(numbuf,sizeof numbuf,"(%d): ",this->line);
	OutputDebugStringA(this->file);
	OutputDebugStringA(numbuf);
#endif
}

unsigned char *twAllocation::FrontGap() const {
	return this->user_base-this->gap_size;
}

unsigned char *twAllocation::BackGap() const {
	return this->user_base+this->user_size;
}

int twAllocation::CheckGap(unsigned char *start) const {
	for(int i=0;i<gap_size;++i) {
		if(start[i]!=i) {
			return i;
		}
	}
	return -1;
}

bool twAllocation::AreGapsValid() const {
	bool ret=true;
	int fg=this->CheckGap(this->user_base-gap_size);
	static char buf[100];
	if(fg>=0) {
		this->PrintName();
		_snprintf(buf,sizeof buf,"front gap: start-%d: ==0x%02X, !=0x%02X\n",
			this->gap_size-fg,this->FrontGap()[fg],fg);
		OutputDebugStringA(buf);
		ret=false;
	}
	int bg=this->CheckGap(this->user_base+this->user_size);
	if(bg>=0) {
		this->PrintName();
		_snprintf(buf,sizeof buf,"front gap: end+%d: ==0x%02X, !=0x%02X\n",
			bg,this->BackGap()[bg],bg);
		OutputDebugStringA(buf);
		ret=false;
	}
	return ret;
}

static const unsigned max_num_allocations=10000;
static twAllocation allocations[max_num_allocations];
static unsigned num_allocations=0;

void twVerifyAllocations() {
#ifdef twDEBUG_CHECK_GAPS
	for(unsigned i=0;i<num_allocations;++i) {
		wxASSERT(allocations[i].AreGapsValid());
	}
#endif
}

static void *twAllocate(unsigned size,unsigned caller) {//,const char *file,int line) {
	twDebugEnsureInitialised();
	twVerifyAllocations();
	wxASSERT(num_allocations<max_num_allocations);//too many
	twAllocation *a=&allocations[num_allocations++];

	//fill in debug info.
	if(!tw_track_heap_allocations) {
#ifdef twDEBUG_STACK_TRACE
		_snprintf(a->stack_trace,sizeof a->stack_trace,"%u",tw_alloc_count);
#else
		strcpy(a->file,"");
		a->line=int(tw_alloc_count);
#endif
	} else {
#ifdef twDEBUG_STACK_TRACE
		twDebugGetStackTrace(a->stack_trace,sizeof a->stack_trace);
#else
		twDebugGetLineAndFileFromAddress(caller,a->file,a->line);
#endif
	}
	++tw_alloc_count;

	//Decide on size
	//This is a PC, therefore infinite memory, so super size gaps.
#ifdef twDEBUG_CHECK_GAPS
	a->gap_size=128;
#else
	a->gap_size=0;
#endif
	a->rtl_size=size+a->gap_size*2;
#ifdef _DEBUG
# ifdef twDEBUG_STACK_TRACE
	a->rtl_base=static_cast<unsigned char *>(
		_malloc_dbg(a->rtl_size,_NORMAL_BLOCK,a->stack_trace,tw_alloc_count));
# else
	a->rtl_base=static_cast<unsigned char *>(
		_malloc_dbg(a->rtl_size,_NORMAL_BLOCK,a->file,a->line));
# endif
#else
	a->rtl_base=static_cast<unsigned char *>(malloc(a->rtl_size));
#endif
	wxASSERT(a->rtl_base);//out of memory

	a->user_base=a->rtl_base+a->gap_size;
	a->user_size=size;

	//	a->file=file;
//	a->line=line;

	//sort out the gaps
	for(int i=0;i<a->gap_size;++i) {
		a->FrontGap()[i]=i;
		a->BackGap()[i]=i;
	}

#ifdef twDEBUG_ALLOC_LOG
	twLog("alloc: usr: %u bytes @ 0x%p rtl: %u bytes @ 0x%p\n",
		a->user_size,a->user_base,a->rtl_size,a->rtl_base);
#ifdef twDEBUG_STACK_TRACE
	twLog("from %s\n",a->stack_trace);
#else
	twLog("%s(%d) <- from\n",a->file,a->line);
#endif
#endif
	return a->user_base;
}

static void twDeallocate(void *ptr,unsigned caller) {
	twDebugEnsureInitialised();
	twVerifyAllocations();
	if(ptr) {
		unsigned i;
		for(i=0;i<num_allocations;++i) {
			if(allocations[i].user_base==ptr) {
				break;
			}
		}
		wxASSERT(i<num_allocations);//that pointer isn't valid
		twAllocation *a=&allocations[i];
		wxASSERT(a->AreGapsValid());//oops, overran start/end of allocation!
#ifdef twDEBUG_ALLOC_LOG
		twLog("free : usr: %u bytes @ 0x%p rtl: %u bytes @ 0x%p\n",
			a->user_size,a->user_base,a->rtl_size,a->rtl_base);
#ifdef twDEBUG_STACK_TRACE
		static char stack_trace[2048];
		twDebugGetStackTrace(stack_trace,sizeof stack_trace);
		twLog("from %s\n",stack_trace);
#else
		const char *file;
		int line;
		twDebugGetLineAndFileFromAddress(caller,file,line);
		twLog("%s(%d) <- from\n",file,line);
#endif
#endif
		//Seems ok, get rid of that one.
		//_free_dbg(a->rtl_base,_NORMAL_BLOCK);
		_free_dbg(a->rtl_base,_NORMAL_BLOCK);
		--num_allocations;
		*a=allocations[num_allocations];
	}
}

void __cdecl DumpClientDumpMemoryLeaks(void *, size_t) {
	const unsigned max_n=4096;
	char buf[max_n];

	for(unsigned i=0;i<num_allocations;++i) {
		twAllocation *a=&allocations[i];
		int n=0;

#ifndef twDEBUG_STACK_TRACE
		n+=_snprintf(&buf[n],max_n-n,"%s(%u): ",a->file,a->line);
#endif
		n+=_snprintf(&buf[n],max_n-n,"user: %u @ 0x%p rtl: %u @ 0x%p\n",
			a->user_size,a->user_base,a->rtl_base,a->rtl_size);
#ifdef twDEBUG_STACK_TRACE
		n+=_snprintf(&buf[n],max_n-n,"%s\n",a->stack_trace);
#endif
		OutputDebugString(buf);
	}
}

//////////////////////////////////////////////////////////////////////////
// Allocators

void *__cdecl operator new(unsigned sz) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	void *p=twAllocate(sz,caller);//,file,line);//_malloc_dbg(sz,_NORMAL_BLOCK,file,line);
	return p;
}

void __cdecl operator delete(void *ptr) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	twDeallocate(ptr,caller);
}

#if _MSC_VER>0x1200
//VC++6 (0x1200) doesn't seem to distinguish between the 2 forms.
void *__cdecl operator new[](unsigned sz) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	void *p=twAllocate(sz,caller);//,file,line);//_malloc_dbg(sz,_NORMAL_BLOCK,file,line);
	return p;
}

void *__cdecl operator delete[](unsigned sz) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	twDeallocate(ptr,caller);
}

#endif//_MSC_VER>0x1200

#if (defined twDEBUG_STACK_TRACE)&&(defined twDEBUG_USE_TW_HEAP)

#pragma comment(linker,"/force:multiple")

void *realloc(void *ptr,size_t size) 
{
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}

	unsigned i;
	for(i=0;i<num_allocations;++i) {
		if(allocations[i].user_base==ptr) {
			break;
		}
	}
	wxASSERT(i<num_allocations);//that pointer isn't valid
	twAllocation *a=&allocations[i];
	wxASSERT(a->AreGapsValid());//oops, overran start/end of allocation!

	//Make a new one
	void *new_ptr=0;
	if(size!=a->user_size&&size>0) {
		new_ptr=twAllocate(size,caller);
		if(ptr) {
			if(size<a->user_size) {
				memmove(new_ptr,a->user_base,size);
			} else {
				memmove(new_ptr,a->user_base,a->user_size);
			}
		}
	}

	//Remove old one.
	if(ptr) {
		twDeallocate(ptr,caller);
	}

	return new_ptr;
}

void *malloc(size_t n) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	return twAllocate(n,caller);
}
	
void free(void *ptr) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	twDeallocate(ptr,caller);
}

void *calloc(size_t num,size_t size) {
	unsigned caller;
	__asm {
		mov eax,[ebp+4];
		mov caller,eax;
	}
	size_t total_size=num*size;
	void *ptr=twAllocate(total_size,caller);
	if(ptr) {
		memset(ptr,0,total_size);
	}
	return ptr;
}

#endif//twDEBUG_STACK_TRACE twDEBUG_USE_TW_HEAP

#endif//twDEBUG_USE_TW_HEAP
